/**
* @file endpoint.cpp
* @brief Definition of Endpoint class
* @date Feb. 16, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#include <time.h>
#include "m3glib.h"
#include "logger.h"
#include "endpoint.h"
#include "endpointmgr.h"
#include "locker.h"
#include "port_connect.h"
#include "appevents.h"

#include <gcip.h>
#include <gclib.h>
#include <gcisdn.h>
#include <srllib.h>

#include "dm3cc_parm.h"



const unsigned int Endpoint::DCI_MODE_DEFAULT(0x0);
const unsigned int Endpoint::DCI_MODE_INBAND(0x1);
const unsigned int Endpoint::DCI_MODE_H245_OLC(0x2);

const char szParms[][60] = {
   "M3G_E_PRM_RELAY_DIGIT_TO_MEDIA_DEV",
   "M3G_E_PRM_RELAY_DIGIT_TO_H245UII",
   "M3G_E_PRM_RELAY_FASTUPDATE_TO_MEDIA_DEV",
   "M3G_E_PRM_RELAY_FASTUPDATE_TO_H245",
   "M3G_E_PRM_RELAY_TEMPORALSPATIALTRADEOFF_TO_MEDIA_DEV",
   "M3G_E_PRM_RELAY_TEMPORALSPATIALTRADEOFF_TO_H245",
   "M3G_E_PRM_RELAY_VIDEOFREEZE_TO_MEDIA_DEV",
   "M3G_E_PRM_RELAY_VIDEOFREEZE_TO_H245",
   "M3G_E_PRM_SKEWINDICATION",
   "M3G_E_PRM_AUDIOVISUALSYNC",
   "M3G_E_PRM_H245_TERMINAL_TYPE",
   "M3G_E_PRM_MAX_CCSRL_SEGMENT",
   "M3G_E_PRM_RETRANSMIT_ON_IDLE",
   "M3G_E_PRM_AMR_PAYLOAD_FORMAT",
   "M3G_E_PRM_BOARD_DEBUG",                                 
   "M3G_E_PRM_ALL_INSTANCES_DEBUG",                        
   "M3G_E_PRM_INSTANCE_DEBUG",                            
   "M3G_E_PRM_STACK_DEBUG_LEVEL",                        
   "M3G_E_PRM_STACK_DEBUG_MODULES",                     
   "M3G_E_PRM_MPEG4_TX_DCI",
   "M3G_E_PRM_MPEG4_RX_DCI",
   "M3G_E_PRM_TX_SKEW_ADJUSTMENT",
   "M3G_E_PRM_RX_SKEW_ADJUSTMENT",
   "M3G_E_PRM_VIDEO_BIT_RATE",
   "M3G_E_PRM_VIDEO_FRAME_RATE",
   "M3G_E_PRM_EARLY_MES",
   "M3G_E_PRM_AUTO_VFU_PERIOD",
   "M3G_E_PRM_H223_SYNC_TIMER"
   "M3G_E_PRM_TX_VENDORID", 
   "M3G_E_PRM_H264_TX_DCI",
   "M3G_E_PRM_IFRAME_NOTIFY_CONTROL_MASK",
};

int UnsignedCharHexArrayToString(unsigned char* origStr, int strLen, char* destStr);

//*****************************************************************************
// Function: Endpoint::Endpoint(int numEP, int numPeer, int numLoopback, const char *dtiName, const char *m3gName, const char *mapID)
// Description: Initializing constructor
// Return:  Endpoint*
// Parameters: int numEP 
//             int numPeer 
//             int numLoopback 
//             const char *dtiName 
//             const char *m3gName 
//             const char *mapID 
//*****************************************************************************
Endpoint::Endpoint(int numEP, int numPeer, int numLoopback, const char *dtiName, const char *m3gName, const char *mapID) :
   m_Index(numEP),
   m_PeerIndex(numPeer),
   m_LoopbackIndex(numLoopback),
   m_NetworkHandle(0),
   m_NetworkDeviceHandle(0),
   m_ControlHandle(0),
   m_AudioHandle(0),
   m_VideoHandle(0),
   m_IsManagedBySigProtocol(false),
   m_RxAudioLCN(0),
   m_TxAudioLCN(0),
   m_RxVideoLCN(0),
   m_TxVideoLCN(0),
   m_DigitToSend('\0'),
   m_pCurrentState(NULL),
   m_ReinitDone(false),
   m_pSigProtocolEndpoint(NULL),
   m_TraceLevel(0),
   m_DciOctetStrSize(0),
   m_DciMode(0),
   m_NbupMediaStarted(false),
   m_NbupIPMPortConnected(false),
   m_Nbup3GPortConnected(false)
{
   SetEndpointType(M3G_ENDPOINT);
   strncpy(m_transportName, dtiName, DEV_NAME_SIZE-1);
   strncpy(m_m3gName, m3gName, DEV_NAME_SIZE-1);
   strncpy(m_mapID, mapID, DEV_NAME_SIZE-1);
   strncpy(m_AudTxCapStr, "none", MAX_MEDIA_STR);
   strncpy(m_VidTxCapStr, "none", MAX_MEDIA_STR);
   strncpy(m_CurrentStateStr, "OPENING", MAX_CURRENT_STATE_STR);
   strcpy(m_LastMessageBuff, "");
   strcpy(m_callInfo, "");
   strcpy(m_TraceFileName, "");

   memset(&m_PreferredH264Params, 0, sizeof(LOCAL_H264_PARAMS));
   memset(&m_LocalH223Caps, 0, sizeof(M3G_CAPS_LIST));
   memset(&m_LocalAudioCaps, 0, sizeof(M3G_CAPS_LIST));
   memset(&m_LocalVideoCaps, 0, sizeof(M3G_CAPS_LIST));
   memset(&m_AudioRxPortList, 0, sizeof(DM_PORT_INFO_LIST));
   memset(&m_AudioTxPortList, 0, sizeof(DM_PORT_INFO_LIST));
   memset(&m_VideoRxPortList, 0, sizeof(DM_PORT_INFO_LIST));
   memset(&m_VideoTxPortList, 0, sizeof(DM_PORT_INFO_LIST));
   memset(&m_DciOctetStr, 0, OCTET_STRING_SIZE);

   m_pOpeningState = new OpeningState(this);
   m_pConnectingState = new ConnectingState(this);
   m_pH245InactiveState = new H245InactiveState(this);
   m_pH245StartingState = new H245StartingState(this);
   m_pOpeningLCsState = new OpeningLCsState(this);
   m_pMediaActiveState = new MediaActiveState(this);
   m_pClosingLCsState = new ClosingLCsState(this);
   m_pRxMediaStoppedState = new RxMediaStoppedState(this);
}

//*****************************************************************************
// Function: Endpoint::~Endpoint()
// Description: Destructor
// Return:  none
// Parameters: none 
//*****************************************************************************
Endpoint::~Endpoint()
{
   delete m_pOpeningState;
   delete m_pConnectingState;
   delete m_pH245InactiveState;
   delete m_pH245StartingState;
   delete m_pOpeningLCsState;
   delete m_pMediaActiveState;
   delete m_pClosingLCsState;
   delete m_pRxMediaStoppedState;
}


//*****************************************************************************
// Function: void Endpoint::SetPreferredMediaCaps(int numCaps, MEDIA_CAP *pPreferredMediaList)
// Description: Set the prefered media capabilities
// Return: void 
// Parameters: int numCaps 
//             MEDIA_CAP *pPreferredMediaList 
//*****************************************************************************
void Endpoint::SetPreferredMediaCaps(int numCaps, MEDIA_CAP *pPreferredMediaList)
{
   m_NumPreferredMediaCaps = (numCaps < MAX_MEDIA_CAPS) ? numCaps : MAX_MEDIA_CAPS;
   memcpy(&m_PreferredMediaCapsList, pPreferredMediaList, m_NumPreferredMediaCaps * sizeof(MEDIA_CAP));
}


//*****************************************************************************
// Function: void Endpoint::ChangeState(EPState::E_EPSTATE e_NewState)
// Description: Change the object state
// Return: void 
// Parameters: EPState::E_EPSTATE e_NewState 
//*****************************************************************************
void Endpoint::ChangeState(EPState::E_EPSTATE e_NewState)
{
   EPState *pPrevState = m_pCurrentState;
   char prevStateStr[MAX_CURRENT_STATE_STR];
   strcpy(prevStateStr, m_pCurrentState->GetStateStr());


   switch ( e_NewState )
   {
      case EPState::CONNECTING_STATE:
         m_pCurrentState = m_pConnectingState;
         break;

      case EPState::H245_INACTIVE_STATE:
         m_pCurrentState = m_pH245InactiveState;
         break;

      case EPState::H245_STARTING_STATE:
         m_pCurrentState = m_pH245StartingState;
         break;

      case EPState::OPENING_LCS_STATE:
         m_pCurrentState = m_pOpeningLCsState;
         // retain media channel status for MPCs opened via MONA procedures
         m_pCurrentState->SetStatus(pPrevState->GetChannelStatus());
         break;

      case EPState::MEDIA_ACTIVE_STATE:
         m_pCurrentState = m_pMediaActiveState;
         break;

      case EPState::RX_MEDIA_STOPPED_STATE:
         m_pCurrentState = m_pRxMediaStoppedState;
         // retain media channel of muted Rx media
         m_pCurrentState->SetStatus(pPrevState->GetChannelStatus());
         break;

      case EPState::CLOSING_LCS_STATE:
         m_pCurrentState = m_pClosingLCsState;
         break;

      default:
         LOG_ERROR(0,"Endpoint: Unexpected state: %d\n",static_cast<int>(e_NewState));
         break;
   }

   // reset the previous state for the next call
   pPrevState->Reset();

   // cache the state string in the endpoint
   strcpy(m_CurrentStateStr,m_pCurrentState->GetStateStr());
   LOG_ENTRY(m_ControlHandle,"Endpoint[%d] State transition: %s ---> %s\n",GetIndex(), 
             prevStateStr, m_CurrentStateStr);
}


//*****************************************************************************
// Function: void Endpoint::ProcessEvent(long evtType, void *pEvtData, long evtLen, long evtDev)
// Description: Process an event
// Return: none
// Parameters: long evtType 
//             void *pEvtData 
//             long evtLen 
//             long evtDev 
//*****************************************************************************
void Endpoint::ProcessEvent(long evtType, void *pEvtData, long  evtLen, long  evtDev)
{
   m_StateLock.Lock();
   LOG_ENTRY( evtDev,"Endpoint::ProcessesEvent(%04X %d)\n",evtType, evtDev);
   // dispatch event information to appropriate state:
   m_pCurrentState->ProcessEvent(evtType, pEvtData, evtLen, evtDev);
   m_StateLock.Unlock();
}

//*****************************************************************************
// Function: void Endpoint::ProcessUserPrompt(EPState::E_USER_PROMPT_TYPE eUserPrompt)
// Description: Process a user interface request
// Return: void 
// Parameters: EPState::E_USER_PROMPT_TYPE eUserPrompt 
//*****************************************************************************
void Endpoint::ProcessUserPrompt(EPState::E_USER_PROMPT_TYPE eUserPrompt)
{
   m_StateLock.Lock();
   m_pCurrentState->ProcessUserPrompt(eUserPrompt);
   m_StateLock.Unlock();
}

//*****************************************************************************
// Function: void Endpoint::ProcessSetNetworkHndEvent(SRL_DEVICE_HANDLE hnd)
// Description: Process receiving a GC open for ISDN/SS7
// Return: void
// Parameters: SRL_DEVICE_HANDLE
//*****************************************************************************
void Endpoint::ProcessSetNetworkHndEvent(SRL_DEVICE_HANDLE hnd)
{
  LOG_ENTRY( 0,"Endpoint::ProcessSetNetworkHndEvent Endpoint[%d] networkhnd=%d\n",GetIndex(), hnd);
  m_NetworkHandle = hnd;
  ProcessEvent(GCEV_OPENEX, NULL, 0, hnd);
}

//*****************************************************************************
// Function: EPState::E_EPSTATE Endpoint::GetState()
// Description: Return the state
// Return: EPState::E_EPSTATE 
// Parameters: none 
//*****************************************************************************
EPState::E_EPSTATE Endpoint::GetState()
{
   m_StateLock.Lock();
   EPState::E_EPSTATE state = EPState::NULL_STATE;
   if (m_pCurrentState)
     state = m_pCurrentState->GetState();
   m_StateLock.Unlock();
   return state;
}


//*****************************************************************************
// Function: void Endpoint::OpenSubDevs()
// Description: Open all devices associated with the 3G endpoint
// Return: void 
// Parameters: none 
//*****************************************************************************
void Endpoint::OpenSubDevs()
{
   // initialize the endpoint state to opening:
   m_pCurrentState = m_pOpeningState;
   LOG_ENTRY (0, "Opening state set\n");

   // Transport for 3G endpoint is either clear channel dti device, ipm device (NBUP)
   // Idti (ISDN) Sdti (SS7) or NONE = internal loopback mode
   if (IsClearChannel())
   {
      LOG_ENTRY(0, "%s is connected via clear channel on %s\n", m_m3gName, m_transportName);
      char gcDeviceName[32];
      sprintf( gcDeviceName, ":N_%s:P_isdn", m_transportName);
      if ( gc_OpenEx(&m_NetworkHandle,gcDeviceName,EV_ASYNC,this) != GC_SUCCESS )
      {
         LOG_ERROR(0, "Error calling gc_OpenEx(%s) for clear channel transport\n",m_transportName);
         LogGlobalCallError();
      }
      else
      {
         LOG_ENTRY(m_NetworkHandle, "Successful gc_Openx() completion for clear channel transport\n");
      }
   }
   else if (IsNbup())
   {
      LOG_ENTRY(0, "%s is connected via Nbup on %s\n", m_m3gName, m_transportName);
      if ( (m_NetworkHandle = ipm_Open(m_transportName, NULL, EV_ASYNC)) == -1 )
      {
         LOG_ERROR(0, "Error calling ipm_Open(%s) for Nbup transport\n", m_transportName);
      }
      else
      {
         LOG_ENTRY(m_NetworkHandle, "Successful ipm_Open() completion for Nbup transport\n");
      }
   }
   else if (IsSigProtocolManaged())
   {
        LOG_ENTRY(0, "%s is controlled by signalling via %s\n", m_m3gName, m_transportName);
        m_IsManagedBySigProtocol = true;
   }

   m_ControlHandle = m3g_Open(m_m3gName, NULL, this);
   if ( 0 > m_ControlHandle )
   {
      LOG_ERROR(0, "m3g_Open(%s) failure\n",m_m3gName);
   }
   else
   {
      LOG_ENTRY(m_ControlHandle, "Successful m3g_Open() completion\n");
   }

   char mediaName[30];
   sprintf(mediaName, "%s:AUDIO1",m_m3gName);
   m_AudioHandle = m3g_Open(mediaName, NULL, this);
   if ( 0 > m_AudioHandle )
   {
      LOG_ERROR(0, "m3g_Open(%s) failure\n",mediaName);
   }
   else
   {
      LOG_ENTRY(m_AudioHandle, "Successful m3g_Open() completion\n");
   }

   // reuse media name buffer for video port
   sprintf(mediaName, "%s:VIDEO1",m_m3gName);
   m_VideoHandle = m3g_Open(mediaName, NULL, this);
   if ( 0 > m_VideoHandle )
   {
      LOG_ERROR(0, "m3g_Open(%s) failure\n",mediaName);
   }
   else
   {
      LOG_ENTRY(m_VideoHandle, "Successful m3g_Open() completion\n");
   }

   // can now set the peer endpoint pointer
   m_pPeerEndpoint = EndpointMngr::Instance()->GetEPFromIdx(m_PeerIndex);
   m_pLoopbackEndpoint = EndpointMngr::Instance()->GetEPFromIdx(m_LoopbackIndex);
}


//*****************************************************************************
// Function: void Endpoint::CloseSubDevs()
// Description: Close all devices associated with the 3G endpoint
// Return: void 
// Parameters: none 
//*****************************************************************************
void Endpoint::CloseSubDevs()
{
   //usleep(500000);

   // Transport for 3G endpoint is either clear channel dti device, ipm device (NBUP)
   // or NONE = internal loopback mode
   if (IsClearChannel() || IsSigProtocolManaged())
   {
      ResetNetworkConfig();
      if ( gc_Close(m_NetworkHandle) < 0 )
      {
         LOG_ERROR(m_NetworkHandle, "Error calling gc_Close() for clear channel, ISDN or SS7 transport\n");
         LogGlobalCallError();
      }
      else
      {
        LOG_ENTRY(m_NetworkHandle, "Successful gc_Close() for clear channel, ISDN or SS7 transport\n");
      }
   }
   else if (IsNbup())
   {
       // NBUP used for transport
       if ( ipm_Close(m_NetworkHandle, NULL) < 0 ) {
         LOG_ERROR(m_NetworkHandle, "Error calling ipm_Close(%s) for Nbup transport\n", m_transportName);
      }
      else
      {
         LOG_ENTRY(m_NetworkHandle, "Successful ipm_Close() completion for Nbup transport\n");
      }
   }

   // stop 3G tracing if active
   StopTracing();

   LOG_ENTRY(m_ControlHandle, "m3g_Close()\n");
   if ( 0 > m3g_Close(m_ControlHandle) )
   {
      LOG_ERROR(m_ControlHandle, "m3g_Close() failure\n");
   }

   LOG_ENTRY(m_AudioHandle, "m3g_Close()\n");
   if ( 0 > m3g_Close(m_AudioHandle) )
   {
      LOG_ERROR(m_AudioHandle, "m3g_Close() failure\n");
   }

   LOG_ENTRY(m_VideoHandle, "m3g_Close()\n");
   if ( 0 > m3g_Close(m_VideoHandle) )
   {
      LOG_ERROR(m_VideoHandle, "m3g_Close() failure\n");
   }
}

//*****************************************************************************
// Function: void Endpoint::RecordPortInfoList(long evtDev, long evtType, DM_PORT_INFO_LIST *pPortInfoList)
// Description: Save the port information
// Return: void 
// Parameters: long evtDev, long evtType
//             DM_PORT_INFO_LIST *pPortInfoList 
//*****************************************************************************
void Endpoint::RecordPortInfoList(long evtDev, long evtType, DM_PORT_INFO_LIST* pPortInfoList)
{
   static const char* portMediaTypeStr[] = 
   {
      "NONE",
      "AUDIO",
      "VIDEO",
      "H223/NBUP"
   };  
 
   LOG_ENTRY(evtDev,"DM_PORT_INFO_LIST:\n");
   LOG_ENTRY(evtDev,"\tunVersion:%d\n", pPortInfoList->unVersion);
   LOG_ENTRY(evtDev,"\tunCount:%d\n", pPortInfoList->unCount);
   for ( unsigned int i=0; i< pPortInfoList->unCount; i++ )
   {
      LOG_ENTRY(evtDev,"\tport_info[%d]:\n",i);
      LOG_ENTRY(evtDev,"\t\tunVersion: %d\n",pPortInfoList->port_info[i].unVersion);

      char tmpBuff[128] = {0};
      char tokenBuff[10] = {0};
      for ( int j=0; j<16; j++ )
      {
         sprintf(tokenBuff, "0x%x ", int(pPortInfoList->port_info[i].device_ID[j]));
         strcat(tmpBuff, tokenBuff);
      }
      LOG_ENTRY(evtDev,"\t\tdevice_ID: %s\n", tmpBuff);
      LOG_ENTRY(evtDev,"\t\tport_ID: %s\n",reinterpret_cast<char *>(pPortInfoList->port_info[i].port_ID));
      LOG_ENTRY(evtDev,"\t\tport_media_type: %d (%s)\n", 
                            pPortInfoList->port_info[i].port_media_type,
                            portMediaTypeStr[pPortInfoList->port_info[i].port_media_type]);
   }
}


//*****************************************************************************
// Function: bool Endpoint::IsEndpointAllPortsRcvd()
// Description: Answer if port information has been received for all, ports
// Return: bool 
// Parameters: none 
//*****************************************************************************
bool Endpoint::IsEndpointAllPortsRcvd()
{
   m_StateLock.Lock();
   bool isAllPortsRcvd;
   // only valid in connecting state:
   isAllPortsRcvd = (EPState::CONNECTING_STATE == GetState()) && 
                    (true == m_pCurrentState->IsEndpointAllPortsRcvd());
   m_StateLock.Unlock();
   return isAllPortsRcvd;
}


//*****************************************************************************
// Function: void Endpoint::ConnectToLoopback()
// Description: Connect to a loopback endpoint
// Return: void 
// Parameters: none 
//*****************************************************************************
void Endpoint::ConnectToLoopback()
{
   DM_PORT_CONNECT_INFO_LIST portConnectInfoList = {0};
   LOG_ENTRY(m_VideoHandle, "Doing dev_PortConnect() for loopback peer\n");
   portConnectInfoList.unVersion = DM_PORT_CONNECT_INFO_LIST_VERSION_0;
   // all 3G endpoints have only one Tx and Rx port:
   portConnectInfoList.unCount = 1;
   portConnectInfoList.port_connect_info[0].unVersion = DM_PORT_CONNECT_INFO_VERSION_0;

   // connect local audio Tx port to peer's audio rx port:
   if(GetAudTranscodeEnabled())
   {
      LOG_ENTRY(m_AudioHandle,"Mode:DMFL_TRANSCODE_ON");
      portConnectInfoList.port_connect_info[0].unFlags = DMFL_TRANSCODE_ON;
   }
   else
   {
      LOG_ENTRY(m_AudioHandle,"Mode:DMFL_TRANSCODE_NATIVE");
      portConnectInfoList.port_connect_info[0].unFlags = DMFL_TRANSCODE_NATIVE;
   }
   portConnectInfoList.port_connect_info[0].port_info_tx = GetAudioTxPortInfo();
   portConnectInfoList.port_connect_info[0].port_info_rx = m_pLoopbackEndpoint->GetAudioRxPortInfo();
   if ( DEV_SUCCESS != dev_PortConnect(m_AudioHandle, &portConnectInfoList, this) )
   {
      LOG_ERROR(m_AudioHandle,"dev_PortConnect() failure\n");
   }
   else
   {
      LOG_ENTRY(m_AudioHandle,"Successful dev_PortConnect() completion\n");
   }

   // connect local video Tx port to peer's video rx port:
   if(GetVidTranscodeEnabled())
   {
      LOG_ENTRY(m_VideoHandle,"Mode:DMFL_TRANSCODE_ON");
      portConnectInfoList.port_connect_info[0].unFlags = DMFL_TRANSCODE_ON;
   }
   else
   {
      LOG_ENTRY(m_VideoHandle,"Mode:DMFL_TRANSCODE_NATIVE");
      portConnectInfoList.port_connect_info[0].unFlags = DMFL_TRANSCODE_NATIVE;
   }
   portConnectInfoList.port_connect_info[0].port_info_tx = GetVideoTxPortInfo();
   portConnectInfoList.port_connect_info[0].port_info_rx = m_pLoopbackEndpoint->GetVideoRxPortInfo();
   if ( DEV_SUCCESS != dev_PortConnect(m_VideoHandle, &portConnectInfoList, this) )
   {
      LOG_ERROR(m_VideoHandle,"dev_PortConnect() failure\n");
   }
   else
   {
      LOG_ENTRY(m_VideoHandle,"Successful dev_PortConnect() completion\n");
   }
}

//*****************************************************************************
// Function: void Endpoint::ConnectToPeer(DM_PORT_INFO &a_audioPortRxInfo, DM_PORT_INFO &a_videoPortRxInfo)
// Description: Connect to a peer endpoint
// Return: void 
// Parameters: DM_PORT_INFO &a_audioPortRxInfo 
//             DM_PORT_INFO &a_videoPortRxInfo 
//*****************************************************************************
void Endpoint::ConnectToPeer(DM_PORT_INFO& a_audioPortRxInfo, DM_PORT_INFO& a_videoPortRxInfo)
{
   DM_PORT_CONNECT_INFO_LIST portConnectInfoList;

   LOG_ENTRY(m_VideoHandle, "Doing dev_PortConnect() for peer\n");
   // First call connects the Tx of 3G device to Rx of IPM device (Audio)
   EndpointMngr::Instance()->InitPortConnectInfoList(&portConnectInfoList, EndpointMngr::AUDIO_PORT, GetAudTranscodeEnabled());
   memcpy(&(portConnectInfoList.port_connect_info[0].port_info_tx), 
          &(GetAudioTxPortInfo()),
          sizeof(DM_PORT_INFO));
   memcpy(&(portConnectInfoList.port_connect_info[0].port_info_rx), 
          &a_audioPortRxInfo,
          sizeof(DM_PORT_INFO));

   if ( DEV_SUCCESS != dev_PortConnect(GetAudioHandle(), &portConnectInfoList, this) )
   {
      LOG_ERROR(GetAudioHandle(), 
                "dev_PortConnect() failure in M3G endpoint - %s\n", 
                ATDV_ERRMSGP(GetAudioHandle()));
   }
   else
   {
      LOG_ENTRY(GetAudioHandle(),"Successful dev_PortConnect() call in M3G endpoint\n");
   }

   // This call connects the Tx of 3G device to Rx of IPM device (Video)
   EndpointMngr::Instance()->InitPortConnectInfoList(&portConnectInfoList, EndpointMngr::VIDEO_PORT, GetVidTranscodeEnabled());
   memcpy(&(portConnectInfoList.port_connect_info[0].port_info_tx), 
          &(GetVideoTxPortInfo()),
          sizeof(DM_PORT_INFO));
   memcpy(&(portConnectInfoList.port_connect_info[0].port_info_rx), 
          &(a_videoPortRxInfo),
          sizeof(DM_PORT_INFO));

   if ( DEV_SUCCESS != dev_PortConnect(GetVideoHandle(), &portConnectInfoList, this) )
   {
      LOG_ERROR(GetVideoHandle(),
                "dev_PortConnect() failure in M3G endpoint - %s\n",
                ATDV_ERRMSGP(GetVideoHandle()));
   }
   else
   {
      LOG_ENTRY(GetVideoHandle(),"Successful dev_PortConnect() call in M3G endpoint\n");
   }

}

//*****************************************************************************
// Function: void Endpoint::DisconnectFromPeer()
// Description: Disconnect from the peer endpoint
// Return: void 
// Parameters: none 
//*****************************************************************************
void Endpoint::DisconnectFromPeer()
{
    DM_PORT_INFO* pAudPortInfo = NULL;
    DM_PORT_INFO* pVidPortInfo = NULL;
    DM_PORT_CONNECT_INFO_LIST portConnectInfoList;

#ifdef USE_RTSP
    RTSPEndpoint *pRTSPEndpoint = NULL;
#endif    
    SIPEndpoint *pSIPEndpoint = NULL;
    MMEndpoint *pMMEndpoint = NULL;
    Endpoint *pM3GEndpoint = NULL;

    if (m_pBridgedCall)
      {
	 pSIPEndpoint = EndpointMngr::Instance()->GetSIPEPFromIdx( GetBridgedCall()->SipIndex());
         pAudPortInfo = &pSIPEndpoint->GetAudioRxPortInfo();
         pVidPortInfo = &pSIPEndpoint->GetVideoRxPortInfo();
      }
      else if (m_pBridgedMedia)
      {
	 pMMEndpoint = EndpointMngr::Instance()->GetMMEPFromIdx( GetBridgedMedia()->MMIndex());
         pAudPortInfo = &pMMEndpoint->GetAudioRxPortInfo();
         pVidPortInfo = &pMMEndpoint->GetVideoRxPortInfo();
      }
      else if (m_pBridgedHairp)
      {
	 pM3GEndpoint = EndpointMngr::Instance()->GetEPFromIdx( GetBridgedHairp()->Index());
         pAudPortInfo = &pM3GEndpoint->GetAudioRxPortInfo();
         pVidPortInfo = &pM3GEndpoint->GetVideoRxPortInfo();
      }
#ifdef USE_RTSP 
      else if (m_pBridgedRtsp)
      {
         LOG_ENTRY(0, "Bridged RTSP index is %d\n", GetBridgedRtsp()->RTSPIndex());
         pRTSPEndpoint = EndpointMngr::Instance()->GetRTSPEPFromIdx( GetBridgedRtsp()->RTSPIndex());
         pAudPortInfo = &pRTSPEndpoint->GetAudioRxPortInfo();
         pVidPortInfo = &pRTSPEndpoint->GetVideoRxPortInfo();
      }
#endif


   // First call disconnects the Tx of 3G device to Rx of IPM device (Audio)
   EndpointMngr::Instance()->InitPortConnectInfoList(&portConnectInfoList, EndpointMngr::AUDIO_PORT,GetAudTranscodeEnabled());
   memcpy(&(portConnectInfoList.port_connect_info[0].port_info_tx), 
          &(GetAudioTxPortInfo()),
          sizeof(DM_PORT_INFO));
   memcpy(&(portConnectInfoList.port_connect_info[0].port_info_rx), 
          pAudPortInfo,
          sizeof(DM_PORT_INFO));

   if ( DEV_SUCCESS != dev_PortDisconnect(GetAudioHandle(), &portConnectInfoList, this) )
   {
      LOG_ERROR(GetAudioHandle(), 
                "dev_PortDisconnect() failure in M3G endpoint - %s\n", 
                ATDV_ERRMSGP(GetAudioHandle()));
   }
   else
   {
      LOG_ENTRY(GetAudioHandle(),"Successful dev_PortDisconnect() call in M3G endpoint\n");
   }

   // This call disconnects the Tx of 3G device to Rx of IPM device (Video)
   EndpointMngr::Instance()->InitPortConnectInfoList(&portConnectInfoList, EndpointMngr::VIDEO_PORT,GetVidTranscodeEnabled());
   memcpy(&(portConnectInfoList.port_connect_info[0].port_info_tx), 
          &(GetVideoTxPortInfo()),
          sizeof(DM_PORT_INFO));
   memcpy(&(portConnectInfoList.port_connect_info[0].port_info_rx), 
          pVidPortInfo,
          sizeof(DM_PORT_INFO));

   if ( DEV_SUCCESS != dev_PortDisconnect(GetVideoHandle(), &portConnectInfoList, this) )
   {
      LOG_ERROR(GetVideoHandle(),
                "dev_PortDisconnect() failure in M3G endpoint - %s\n",
                ATDV_ERRMSGP(GetVideoHandle()));
   }
   else
   {
      LOG_ENTRY(GetVideoHandle(),"Successful dev_PortDisconnect() call in M3G endpoint\n");
   }
}

//*****************************************************************************
// Function: void Endpoint::ConnectTSAndGetPorts()
// Description: Connect to the TDM or loopback and get port information
// Return: void 
// Parameters: none 
//*****************************************************************************
void Endpoint::ConnectTSAndGetPorts()
{
   if ( false == GetIsLoopbackNotNetwork() )
   {
      // Not loopback - need to distinguish between DTI clear channel or ISDN/SS7
      // and Nbup transport over IP
      if (IsClearChannel() || IsSigProtocolManaged())
      {
        if ( gc_GetNetworkH(m_NetworkHandle, &m_NetworkDeviceHandle) != GC_SUCCESS )
        {
           LOG_ERROR(m_NetworkHandle, "Error calling gc_GetNetworkH()"); 
           LogGlobalCallError();
        }
        else
        {
           LOG_ENTRY(m_NetworkHandle, "Successful gc_GetNetwork() completion\n"); 
        } 

        // connect 3G mux control device to network DS0 timeslot (dti):
        if ( DEV_SUCCESS != dev_Connect(m_ControlHandle, 
                                        m_NetworkDeviceHandle,
                                        DM_FULLDUP, EV_ASYNC) )
        {
           LOG_ERROR(m_NetworkHandle,"dev_Connect() failure on 3G-DTI\n");
        }
        else
        {
           LOG_ENTRY(m_NetworkHandle, "Successful dev_Connect() completion on 3G-DTI\n");
        }
     }
   }
   else if ( (m_Index) > (m_LoopbackIndex) )// loopback:
   {
      // connect 3G mux control device to peer 3G mux control device:
      if ( DEV_SUCCESS != dev_Connect(m_ControlHandle,
                                      m_pLoopbackEndpoint->m_ControlHandle, 
                                      DM_FULLDUP, EV_ASYNC) )
      {
         LOG_ERROR(m_NetworkHandle,"dev_Connect() failure on 3G-3G\n");
      }
      else
      {
         LOG_ENTRY(m_NetworkHandle, "Successful dev_Connect() completion on 3G-3G\n");
      }
   }

   // retrieve audio Tx and Rx PStream ports:
   if ( DEV_SUCCESS != dev_GetTransmitPortInfo(m_AudioHandle, this) )
   {
      LOG_ERROR(m_AudioHandle,"dev_GetTransmitPortInfo() failure\n");
   }
   else
   {
      LOG_ENTRY(m_AudioHandle,"Successful dev_GetTransmitPortInfo() completion\n");
   }

   if ( DEV_SUCCESS != dev_GetReceivePortInfo(m_AudioHandle, this) )
   {
      LOG_ERROR(m_AudioHandle,"dev_GetReceivePortInfo() failure\n");
   }
   else
   {
      LOG_ENTRY(m_AudioHandle,"Successful dev_GetReceivePortInfo() completion\n");
   }

   // retrieve video Tx and Rx PStream ports:
   if ( DEV_SUCCESS != dev_GetTransmitPortInfo(m_VideoHandle, this) )
   {
      LOG_ERROR(m_VideoHandle,"dev_GetTransmitPortInfo() failure\n");
   }
   else
   {
      LOG_ENTRY(m_VideoHandle,"Successful dev_GetTransmitPortInfo() completion\n");
   }

   if ( DEV_SUCCESS != dev_GetReceivePortInfo(m_VideoHandle, this) )
   {
      LOG_ERROR(m_VideoHandle,"dev_GetReceivePortInfo() failure\n");
   }
   else
   {
      LOG_ENTRY(m_VideoHandle,"Successful dev_GetReceivePortInfo() completion\n");
   }

   if (IsNbup())
     {
       // Looking for port info on 3G control (Nbup) and IPM Nbup 
       GetNbupPortInfo ();
     }
}


//*****************************************************************************
// Function: void Endpoint::DisconnectTS()
// Description: Disconnect from the TDM timeslot
// Return: void 
// Parameters: none 
//*****************************************************************************
void Endpoint::DisconnectTS()
{
   if ( false == GetIsLoopbackNotNetwork() )
   {

      // disconnect 3G mux control device from network DS0 timeslot (dti) 
      // or peer 3G endpoint:
      if ( DEV_SUCCESS != dev_Disconnect(m_ControlHandle, EV_ASYNC) )
      {
         LOG_ERROR(m_ControlHandle,"dev_Disconnect() failure\n");
      }
      else
      {
         LOG_ENTRY(m_ControlHandle, "Successful dev_Disconnect() completion\n");
      }


      if ( DEV_SUCCESS != dev_Disconnect(m_NetworkDeviceHandle, EV_ASYNC) )
      {
         LOG_ERROR(m_NetworkHandle,"dev_Disconnect() failure\n");
      }
      else
      {
         LOG_ENTRY(m_NetworkHandle, "Successful dev_Disconnect() completion\n");
      }
   }
}

//*****************************************************************************
// Function: void Endpoint::Reinitialize()
// Description: Reset M3G devices 
// Return: void 
// Parameters: none 
//*****************************************************************************
void Endpoint::Reinitialize()
{
   if (M3G_ERROR == m3g_Reset(m_ControlHandle))
   {
      LOG_ERROR(m_ControlHandle,"m3g_Reset() failure\n");
   }
   else
   {
      LOG_ENTRY(m_ControlHandle,"Successful m3g_Reset() completion\n");
      m_ReinitDone = true;
   }
}

//*****************************************************************************
// Function: void Endpoint::GetLocalCaps()
// Description: Get the local capabilities
// Return: void 
// Parameters: none 
//*****************************************************************************
void Endpoint::GetLocalCaps()
{
   if (M3G_ERROR == m3g_GetLocalCaps(m_ControlHandle))
   {
      LOG_ERROR(m_ControlHandle,"m3g_GetLocalCaps() failure\n");
   }
   else
   {
      LOG_ENTRY(m_ControlHandle,"Successful m3g_GetLocalCaps() completion\n");
   }

   if (M3G_ERROR == m3g_GetLocalCaps(m_AudioHandle))
   {
      LOG_ERROR(m_AudioHandle,"m3g_GetLocalCaps() failure\n");
   }
   else
   {
      LOG_ENTRY(m_AudioHandle,"Successful m3g_GetLocalCaps() completion\n");
   }

   if (M3G_ERROR == m3g_GetLocalCaps(m_VideoHandle))
   {
      LOG_ERROR(m_VideoHandle,"m3g_GetLocalCaps() failure\n");
   }
   else
   {
      LOG_ENTRY(m_VideoHandle,"Successful m3g_GetLocalCaps() completion\n");
   }
}

//*****************************************************************************
// Function: void Endpoint::RecordCapsList(long evtDev, M3G_CAPS_LIST *pCapsList)
// Description: Save the capabilities list
// Return: void 
// Parameters: long evtDev 
//             M3G_CAPS_LIST *pCapsList 
//*****************************************************************************
void Endpoint::RecordCapsList(long  evtDev, M3G_CAPS_LIST *pCapsList)
{
   static const char* capTypeStr[] =
   {
      "M3G_E_H223_CAPABILITY",
      "M3G_E_AUDIO_CAPABILITY",
      "M3G_E_VIDEO_CAPABILITY"
   };

   LOG_ENTRY(evtDev,"\t  CapsList.version:0x%x\n", pCapsList->version);
   LOG_ENTRY(evtDev,"\t  CapsList.numCaps:%d\n",pCapsList->numCaps);
   if ((0 <= pCapsList->capabilityType) && (3 > pCapsList->capabilityType))
      LOG_ENTRY(evtDev,"\t  CapsList.capabilityType:%s\n", capTypeStr[pCapsList->capabilityType]);
   else
      LOG_ERROR(evtDev,"\t  Unknown CapsList.capabilityType:%d\n", pCapsList->capabilityType);

   for (int i=0; i < pCapsList->numCaps; i++)
   {
      LOG_ENTRY(evtDev,"\t  capability[%d]:\n", i);
       switch(pCapsList->capabilityType)
      {
         case M3G_E_H223_CAPABILITY:
            RecordH223Cap(evtDev, &pCapsList->capability[i].h223Capability);
            break;

         case M3G_E_AUDIO_CAPABILITY:
            RecordAudioCap(evtDev, &pCapsList->capability[i].audioCapability);
            break;

         case M3G_E_VIDEO_CAPABILITY:
            RecordVideoCap(evtDev, &pCapsList->capability[i].videoCapability);
            break;
      }
   }
}

//*****************************************************************************
// Function: void Endpoint::RecordH223Cap(long evtDev, M3G_H223_CAPABILITY *pH223Cap)
// Description: Save H223 capabilities
// Return: void 
// Parameters: long evtDev 
//             M3G_H223_CAPABILITY *pH223Cap 
//*****************************************************************************
void Endpoint::RecordH223Cap(long  evtDev, M3G_H223_CAPABILITY *pH223Cap)
{
   const char* tStr = NULL;

   LOG_ENTRY(evtDev,"\t\th223Capability.version:0x%x\n",pH223Cap->version);
   LOG_ENTRY(evtDev,"\t\th223Capability.adaptationLayerMedia:%d\n", pH223Cap->adaptationLayerMedia);
   LOG_ENTRY(evtDev,"\t\th223Capability.ALxM_AnnexC_Media:%d\n", pH223Cap->ALxM_AnnexC_Media);
   LOG_ENTRY(evtDev,"\t\th223Capability.maxAL2SDUSize:%d\n", pH223Cap->maxAL2SDUSize);
   LOG_ENTRY(evtDev,"\t\th223Capability.maxAL3SDUSize:%d\n", pH223Cap->maxAL3SDUSize);
   tStr = BOOL_STRINGIFY(pH223Cap->frameH223AnnexA);
   LOG_ENTRY(evtDev,"\t\th223Capability.frameH223AnnexA:%s\n",tStr);
   tStr = BOOL_STRINGIFY(pH223Cap->frameH223DoubleFlag);
   LOG_ENTRY(evtDev,"\t\th223Capability.frameH223DoubleFlag:%s\n",tStr);
   tStr = BOOL_STRINGIFY(pH223Cap->frameAnnexB);
   LOG_ENTRY(evtDev,"\t\th223Capability.frameAnnexB:%s\n", tStr);
   tStr = BOOL_STRINGIFY(pH223Cap->frameAnnexBWithHead);
   LOG_ENTRY(evtDev,"\t\th223Capability.frameAnnexBWithHead:%s\n", tStr);
   LOG_ENTRY(evtDev,"\t\th223Capability.maxAL1MPDUSize:%d\n", pH223Cap->maxAL1MPDUSize);
   LOG_ENTRY(evtDev,"\t\th223Capability.maxAL2MPDUSize:%d\n", pH223Cap->maxAL2MPDUSize);
   LOG_ENTRY(evtDev,"\t\th223Capability.maxAL3MPDUSize:%d\n", pH223Cap->maxAL3MPDUSize);
   tStr = BOOL_STRINGIFY(pH223Cap->rsCodeCapability);
   LOG_ENTRY(evtDev,"\t\th223Capability.rsCodeCapability:%s\n", tStr);
   LOG_ENTRY(evtDev,"\t\th223Capability.mobileOpXmitCap:%d\n", int(pH223Cap->mobileOpXmitCap));
   LOG_ENTRY(evtDev,"\t\th223Capability.bitRate:%d\n", pH223Cap->bitRate);
}

//*****************************************************************************
// Function: void Endpoint::RecordAudioCap(long evtDev, M3G_AUDIO_CAPABILITY *pAudioCap)
// Description: Save audio capabilities
// Return: void 
// Parameters: long evtDev 
//             M3G_AUDIO_CAPABILITY *pAudioCap 
//*****************************************************************************
void Endpoint::RecordAudioCap(long  evtDev, M3G_AUDIO_CAPABILITY *pAudioCap)
{
   static const char *directionStr[] =
   {
      "M3G_E_IDLE",
      "M3G_E_TX",
      "M3G_E_RX",
      "M3G_E_TXRX"
   };

   static const char *audioCoderStr[] =
   {
      "M3G_E_G7231",
      "M3G_E_GSM_AMR_NB"
   };

   LOG_ENTRY(evtDev,"\t\taudioCapability.version:0x%x\n", pAudioCap->version);
   LOG_ENTRY(evtDev,"\t\taudioCapability.tableEntryNumber:%d\n", pAudioCap->tableEntryNumber);
   if ((0 <= pAudioCap->direction) && (4 > pAudioCap->direction))
   {
      LOG_ENTRY(evtDev,"\t\taudioCapability.direction:%s\n", directionStr[pAudioCap->direction]);
   }
   else
   {
      LOG_ERROR(evtDev,"\t\tInvalid audioCapability.direction:%d\n", pAudioCap->direction);
   }

   if ((0 <= pAudioCap->coderType) && (2 > pAudioCap->coderType))
   {
      LOG_ENTRY(evtDev,"\t\taudioCapability.coderType:%s\n", audioCoderStr[pAudioCap->coderType]);
   }
   else
   {
      LOG_ERROR(evtDev,"\t\tInvalid audioCapability.direction:%d\n", pAudioCap->direction);
   }

   LOG_ENTRY(evtDev,"\t\taudioCapability.maxFramesPerSDU:%d\n", int(pAudioCap->maxFramesPerSDU));
   LOG_ENTRY(evtDev,"\t\taudioCapability.options:\n");
   if (M3G_E_G7231 == pAudioCap->coderType)
   {
      const char *tStr = BOOL_STRINGIFY(pAudioCap->options.g7231.silenceSup);
      LOG_ENTRY(evtDev,"\t\t\toptions.g7231.silenceSup:%s\n", tStr);
   }
   else if (M3G_E_GSM_AMR_NB == pAudioCap->coderType)
   {
      const char *tStr = BOOL_STRINGIFY(pAudioCap->options.amr.amrComfortNoise);
      LOG_ENTRY(evtDev,"\t\t\toptions.amr.amrComfortNoise:%s\n", tStr);
   }
}

//*****************************************************************************
// Function: void Endpoint::RecordVideoCap(long evtDev, M3G_VIDEO_CAPABILITY *pVideoCap)
// Description: Save video capabilities
// Return: void 
// Parameters: long evtDev 
//             M3G_VIDEO_CAPABILITY *pVideoCap 
//*****************************************************************************
void Endpoint::RecordVideoCap(long  evtDev, M3G_VIDEO_CAPABILITY *pVideoCap)
{
   static const char *directionStr[] =
   {
      "M3G_E_IDLE",
      "M3G_E_TX",
      "M3G_E_RX",
      "M3G_E_TXRX"
   };

   static const char *videoCoderStr[] =
   {
      "M3G_E_H263",
      "M3G_E_MPEG4", 
      "M3G_E_H264"
   };

   LOG_ENTRY(evtDev,"\t\tvideoCapability.version:0x%x\n", pVideoCap->version);
   LOG_ENTRY(evtDev,"\t\tvideoCapability.tableEntryNumber:%d\n", pVideoCap->tableEntryNumber);
   if ((0 <= pVideoCap->direction) && (4 > pVideoCap->direction))
   {
      LOG_ENTRY(evtDev,"\t\tvideoCapability.direction:%s\n", directionStr[pVideoCap->direction]);
   }
   else
   {
      LOG_ERROR(evtDev,"\t\tInvalid videoCapability.direction:%d\n", pVideoCap->direction);
   }
   if ((0 <= pVideoCap->coderType) && (3 > pVideoCap->coderType))
   {
      LOG_ENTRY(evtDev,"\t\tvideoCapability.coderType:%s\n", videoCoderStr[pVideoCap->coderType]);
   }
   else
   {
      LOG_ERROR(evtDev,"\t\tInvalid coderType:%s\n", videoCoderStr[pVideoCap->coderType]);
   }
   LOG_ENTRY(evtDev,"\t\tvideoCapability.maxBitRate:%d\n", pVideoCap->maxBitRate);
   LOG_ENTRY(evtDev,"\t\tvideoCapability.options:\n");
   const char *tStr = NULL;

   if (M3G_E_H263 == pVideoCap->coderType)
   {
      LOG_ENTRY(evtDev,"\t\t\toptions.h263.bppMaxKb:%d\n", pVideoCap->options.h263.bppMaxKb);
      LOG_ENTRY(evtDev,"\t\t\toptions.h263.sqcifMPI:%d\n", int(pVideoCap->options.h263.sqcifMPI));
      LOG_ENTRY(evtDev,"\t\t\toptions.h263.qcifMPI:%d\n", int(pVideoCap->options.h263.qcifMPI));
      tStr = BOOL_STRINGIFY(pVideoCap->options.h263.unrestrictedVector);
      LOG_ENTRY(evtDev,"\t\t\toptions.h263.unrestrictedVector:%s\n", tStr);
      tStr = BOOL_STRINGIFY(pVideoCap->options.h263.arithmeticCoding);
      LOG_ENTRY(evtDev,"\t\t\toptions.h263.arithmeticCoding:%s\n", tStr);
      tStr = BOOL_STRINGIFY(pVideoCap->options.h263.advancedPrediction);
      LOG_ENTRY(evtDev,"\t\t\toptions.h263.advancedPrediction:%s\n", tStr);
      tStr = BOOL_STRINGIFY(pVideoCap->options.h263.pbFrames);
      LOG_ENTRY(evtDev,"\t\t\toptions.h263.pbFrames:%s\n", tStr);
      tStr = BOOL_STRINGIFY(pVideoCap->options.h263.temporalSpatialTradeoffCap);
      LOG_ENTRY(evtDev,"\t\t\toptions.h263.temporalSpatialTradeoffCap:%s\n", tStr);
      tStr = BOOL_STRINGIFY(pVideoCap->options.h263.errorCompensation);
      LOG_ENTRY(evtDev,"\t\t\toptions.h263.errorCompensation:%s\n", tStr);
   }
   else if (M3G_E_MPEG4 == pVideoCap->coderType)
   {
      LOG_ENTRY(evtDev,"\t\t\toptions.mpeg4.profileAndLevel:%d\n", int(pVideoCap->options.mpeg4.profileAndLevel));
      LOG_ENTRY(evtDev,"\t\t\toptions.mpeg4.object:%d\n", int(pVideoCap->options.mpeg4.object));
      LOG_ENTRY(evtDev,"\t\t\toptions.mpeg4.decoderConfigLength:%d\n", int(pVideoCap->options.mpeg4.decoderConfigLength));
#if 1    // JS 1/8/09
      if (pVideoCap->options.mpeg4.decoderConfigLength > 0) {
         char* temp = new char[2*pVideoCap->options.mpeg4.decoderConfigLength+1];
         UnsignedCharHexArrayToString(pVideoCap->options.mpeg4.decoderConfigInfo, pVideoCap->options.mpeg4.decoderConfigLength, temp);
         temp[2*pVideoCap->options.mpeg4.decoderConfigLength] = '\0';
         LOG_ENTRY(evtDev,"\t\t\toptions.mpeg4.decoderConfigInfo: '%s'\n", temp);
         m_MobileDCIRxdStr.assign(temp);
         m_MobileDCIRxdOctetStrLen = pVideoCap->options.mpeg4.decoderConfigLength;
         memcpy(m_MobileDCIRxdOctetStr, pVideoCap->options.mpeg4.decoderConfigInfo, m_MobileDCIRxdOctetStrLen);
         delete [] temp;
      }
#else
      LOG_ENTRY(evtDev,"\t\t\toptions.mpeg4.decoderConfigInfo[0-2]:0x%x 0x%x 0x%x...\n",
                int(pVideoCap->options.mpeg4.decoderConfigInfo[0]),
                int(pVideoCap->options.mpeg4.decoderConfigInfo[1]),
                int(pVideoCap->options.mpeg4.decoderConfigInfo[2]));
#endif
      tStr = BOOL_STRINGIFY(pVideoCap->options.mpeg4.visualBackChannel);
      LOG_ENTRY(evtDev,"\t\t\toptions.mpeg4.visualBackChannel:%s\n", tStr);
   }
   else if (M3G_E_H264 == pVideoCap->coderType)
   {
      LOG_ENTRY(evtDev,"\t\tRecordVideoCap - \n");
      LOG_ENTRY(evtDev,"\t\t\toptions.h264.profile:%d\n", int(pVideoCap->options.h264.profile));
      LOG_ENTRY(evtDev,"\t\t\toptions.h264.level:%d\n", int(pVideoCap->options.h264.level));
      LOG_ENTRY(evtDev,"\t\t\toptions.h264.decoderConfigLength:%d\n", int(pVideoCap->options.h264.decoderConfigLength));

      if (pVideoCap->options.h264.decoderConfigLength > 0) {
         char* temp = new char[2*pVideoCap->options.h264.decoderConfigLength+1];
         UnsignedCharHexArrayToString(pVideoCap->options.h264.decoderConfigInfo, pVideoCap->options.h264.decoderConfigLength, temp);
         temp[2*pVideoCap->options.h264.decoderConfigLength] = '\0';
         LOG_ENTRY(evtDev,"\t\t\toptions.h264.decoderConfigInfo: '%s'\n", temp);
         m_MobileDCIRxdStr.assign(temp);
         m_MobileDCIRxdOctetStrLen = pVideoCap->options.h264.decoderConfigLength;
         memcpy(m_MobileDCIRxdOctetStr, pVideoCap->options.h264.decoderConfigInfo, m_MobileDCIRxdOctetStrLen);
         delete [] temp;
      }


      LOG_ENTRY(evtDev,"\t\t\toptions.h264.h264SignalingMask:%d\n", int(pVideoCap->options.h264.h264SignalingMask));

      LOG_ENTRY(evtDev,"\t\t\toptions.h264.profileIOP:%d\n", int(pVideoCap->options.h264.profileIOP));

   }
}


void Endpoint::RecordLCMuxParams(long  evtDev, M3G_H223_LC_PARAMS* pLCMux)
{
   LOG_ENTRY(evtDev,"\t  M3G_H223_LC_PARAMS:\n");
   LOG_ENTRY(evtDev,"\t\t  version:0x%x\n", pLCMux->version);
   LOG_ENTRY(evtDev,"\t\t  adaptationLayerType:%d\n", pLCMux->adaptationLayerType);
   LOG_ENTRY(evtDev,"\t\t  segmentable:%d\n", pLCMux->segmentable);
   LOG_ENTRY(evtDev,"\t\t  AL3_ControlFieldSize:%u\n", static_cast<unsigned int>(pLCMux->AL3_ControlFieldSize));
   LOG_ENTRY(evtDev,"\t\t  AL3_SendBufferSize:%d\n", pLCMux->AL3_SendBufferSize);
   LOG_ENTRY(evtDev,"\t\t  ALxM_HeaderFormat:%d\n", pLCMux->ALxM_HeaderFormat);
   LOG_ENTRY(evtDev,"\t\t  ALxM_ALPDUInterleaving:%d\n", pLCMux->ALxM_ALPDUInterleaving);
   LOG_ENTRY(evtDev,"\t\t  ALxM_CRCType:%d\n", pLCMux->ALxM_CRCType);
   LOG_ENTRY(evtDev,"\t\t  ALxM_ARQType:%d\n", pLCMux->ALxM_ARQType);
   LOG_ENTRY(evtDev,"\t\t  ALxM_ARQMaxNumRetrans:%u\n", static_cast<unsigned int>(pLCMux->ALxM_ARQMaxNumRetrans));
   LOG_ENTRY(evtDev,"\t\t  ALxMARQSendBufferSize:%d\n", pLCMux->ALxM_ARQSendBufferSize);
   LOG_ENTRY(evtDev,"\t\t  ALxM_RCPCCodeRate:%u\n", static_cast<unsigned int>(pLCMux->ALxM_RCPCCodeRate));
}


//*****************************************************************************
// Function: void Endpoint::SetTCS()
// Description: Set terminal capabilities
// Return: void 
// Parameters: none 
//*****************************************************************************
void Endpoint::SetTCS()
{
   M3G_SIMULTANEOUS_CAP_SET simultaneousCapList = {0};
   simultaneousCapList.version = M3G_LIBRARY_VERSION;

   // configure the local media caps per the configured preferences:
   M3G_CAPS_LIST    preferredH223Caps = m_LocalH223Caps;
   LOCAL_H223_CAPS  configBufH223Caps = m_PreferredH223Caps;
   preferredH223Caps.capability[0].h223Capability.adaptationLayerMedia = configBufH223Caps.AL;
   preferredH223Caps.capability[0].h223Capability.maxAL2SDUSize = configBufH223Caps.AL2Size;
   preferredH223Caps.capability[0].h223Capability.maxAL3SDUSize = configBufH223Caps.AL3Size;
   preferredH223Caps.capability[0].h223Capability.frameH223AnnexA = static_cast<M3G_BOOL>(configBufH223Caps.isAnnexA);
   preferredH223Caps.capability[0].h223Capability.frameH223DoubleFlag = static_cast<M3G_BOOL>(configBufH223Caps.isADblFlag);
   preferredH223Caps.capability[0].h223Capability.frameAnnexB = static_cast<M3G_BOOL>(configBufH223Caps.isAnnexB);
   preferredH223Caps.capability[0].h223Capability.frameAnnexBWithHead = static_cast<M3G_BOOL>(configBufH223Caps.isBHdr);
   preferredH223Caps.capability[0].h223Capability.rsCodeCapability = static_cast<M3G_BOOL>(configBufH223Caps.isRSCode);
   preferredH223Caps.capability[0].h223Capability.mobileOpXmitCap = static_cast<M3G_BOOL>(configBufH223Caps.isMblOpXmt);
   preferredH223Caps.capability[0].h223Capability.bitRate = configBufH223Caps.bitRate;
   simultaneousCapList.pH223Capabilities = &preferredH223Caps;

   // configure the local audio caps per the configured preferences:
   M3G_CAPS_LIST    preferredAudioCaps = {0};
   preferredAudioCaps.version = M3G_LIBRARY_VERSION;
   preferredAudioCaps.numCaps = 0;
   preferredAudioCaps.capabilityType = M3G_E_AUDIO_CAPABILITY;

   M3G_CAPS_LIST    defaultAudioCaps = m_LocalAudioCaps;
 
   for ( int prefIdx=0; prefIdx < m_NumPreferredMediaCaps; prefIdx++ )
   {
      for ( int dfltIdx=0; dfltIdx < defaultAudioCaps.numCaps; dfltIdx++ )
      {
         M3G_E_AUDIO_TYPE preferredAudioType;
         if (G723 == m_PreferredMediaCapsList[prefIdx])
         {
            preferredAudioType = M3G_E_G7231;
         }
         else if (AMR == m_PreferredMediaCapsList[prefIdx])
         {
            preferredAudioType = M3G_E_GSM_AMR_NB;
         }
         else
         {
            continue;
         }
         // if audio cap type is preferred, add it to the preferred list
         if (preferredAudioType == defaultAudioCaps.capability[dfltIdx].audioCapability.coderType)
         {
            preferredAudioCaps.capability[preferredAudioCaps.numCaps].audioCapability = 
            defaultAudioCaps.capability[dfltIdx].audioCapability;
            preferredAudioCaps.numCaps++;
         }
      }
   }
   simultaneousCapList.pAudioCapabilities = &preferredAudioCaps;
   LOG_ENTRY(m_AudioHandle,"Set Preferred Audio Caps:\n");
   RecordCapsList(m_AudioHandle, simultaneousCapList.pAudioCapabilities);

   // configure the local video caps per the configured preferences:
   M3G_CAPS_LIST    preferredVideoCaps = {0};
   preferredVideoCaps.version = M3G_LIBRARY_VERSION;
   preferredVideoCaps.numCaps = 0;
   preferredVideoCaps.capabilityType = M3G_E_VIDEO_CAPABILITY;

   M3G_CAPS_LIST    defaultVideoCaps = m_LocalVideoCaps;
   for ( int prefIdx=0; prefIdx < m_NumPreferredMediaCaps; prefIdx++ )
   {
      for ( int dfltIdx=0; dfltIdx < defaultVideoCaps.numCaps; dfltIdx++ )
      {
         M3G_E_VIDEO_TYPE  preferredVideoType;
         if (H263 == m_PreferredMediaCapsList[prefIdx])
         {
            preferredVideoType = M3G_E_H263;
         }
         else if (MPEG4 == m_PreferredMediaCapsList[prefIdx])
         {
            preferredVideoType = M3G_E_MPEG4;
         }
         else if (H264 == m_PreferredMediaCapsList[prefIdx])
         {
            preferredVideoType = M3G_E_H264;
         }
         else
         {
            continue;
         }
         // if video cap type is preferred, add it to the preferred list
         if (preferredVideoType == defaultVideoCaps.capability[dfltIdx].videoCapability.coderType)
         {
            if (defaultVideoCaps.capability[dfltIdx].videoCapability.coderType == M3G_E_H264)
            {
                LOG_ENTRY(m_VideoHandle,"SetTCS - H264 Video Cap\n");

                LOCAL_H264_PARAMS          configBufH264Caps = m_PreferredH264Params;
                LOG_ENTRY(m_VideoHandle,"Configured acceptRedundantSlices: <%d>\n", configBufH264Caps.acceptRedundantSlices);
                LOG_ENTRY(m_VideoHandle,"Configured profileIOP: <%d>\n", configBufH264Caps.profileIOP);
                LOG_ENTRY(m_VideoHandle,"Configured nalAlignedMode: <%d>\n", configBufH264Caps.nalAlignedMode);

                if (configBufH264Caps.acceptRedundantSlices == M3G_TRUE)
                  defaultVideoCaps.capability[dfltIdx].videoCapability.options.h264.h264SignalingMask |= M3G_H264_ACCEPT_REDUNDANT_SLICES;
                if (configBufH264Caps.nalAlignedMode == M3G_TRUE)
                  defaultVideoCaps.capability[dfltIdx].videoCapability.options.h264.h264SignalingMask |= M3G_H264_NAL_ALIGNED_MODE;
                                                                                         
                defaultVideoCaps.capability[dfltIdx].videoCapability.options.h264.profileIOP = configBufH264Caps.profileIOP;
            }
            preferredVideoCaps.capability[preferredVideoCaps.numCaps].videoCapability = 
                        defaultVideoCaps.capability[dfltIdx].videoCapability;
            preferredVideoCaps.numCaps++;
         }
      }
   }
   simultaneousCapList.pVideoCapabilities = &preferredVideoCaps;
   LOG_ENTRY(m_VideoHandle,"Set Preferred Video Caps:\n");
   RecordCapsList(m_VideoHandle, simultaneousCapList.pVideoCapabilities);


   if (M3G_ERROR == m3g_SetTCS(m_ControlHandle, 1, &simultaneousCapList))
   {
      LOG_ERROR(m_ControlHandle,"m3g_SetTCS() failure - %s\n", ATDV_ERRMSGP(m_ControlHandle));
   }
   else
   {
      LOG_ENTRY(m_ControlHandle,"Successful m3g_SetTCS() completion \n");
   }
}

//*****************************************************************************
// Function: void Endpoint::StartH245()
// Description: Start H345 session
// Return: void 
// Parameters: none 
//*****************************************************************************
void Endpoint::StartH245()
{
   Notify(APP_H245_START_REQUEST);
   if ( M3G_ERROR == m3g_StartH245(m_ControlHandle, 
                                   &m_PreferredH223Defaults) )
   {
      LOG_ERROR(m_ControlHandle,"m3g_StartH245() failure\n");
   }
   else
   {
      LOG_ENTRY(m_ControlHandle,"Successful m3g_StartH245() completion\n");
   }
}

//*****************************************************************************
// Function: void Endpoint::GetMatchedCaps()
// Description: Get matched capabilities
// Return: void 
// Parameters: none 
//*****************************************************************************
void Endpoint::GetMatchedCaps()
{
   M3G_CAPS_LIST matchedH223CapsList = {0};
   if (M3G_ERROR == m3g_GetMatchedCaps(m_ControlHandle, &matchedH223CapsList))
   {
      LOG_ERROR(m_ControlHandle,"m3g_GetMatchedCaps() failure\n");
   }
   else
   {
      LOG_ENTRY(m_ControlHandle,"Successful m3g_GetMatchedCaps() completion\n");
   }
   RecordCapsList(m_ControlHandle, &matchedH223CapsList);

   M3G_CAPS_LIST matchedAudioCapsList = {0};
   if (M3G_ERROR == m3g_GetMatchedCaps(m_AudioHandle, &matchedAudioCapsList))
   {
      LOG_ERROR(m_AudioHandle,"m3g_GetMatchedCaps() failure\n");
   }
   else
   {
      LOG_ENTRY(m_AudioHandle,"Successful m3g_GetMatchedCaps() completion\n");
   }
   RecordCapsList(m_AudioHandle, &matchedAudioCapsList);

   M3G_CAPS_LIST matchedVideoCapsList = {0};
   if (M3G_ERROR == m3g_GetMatchedCaps(m_VideoHandle, &matchedVideoCapsList))
   {
      LOG_ERROR(m_VideoHandle,"m3g_GetMatchedCaps() failure\n");
   }
   else
   {
      LOG_ENTRY(m_VideoHandle,"Successful m3g_GetMatchedCaps() completion\n\n");
   }
   RecordCapsList(m_VideoHandle, &matchedVideoCapsList);

   M3G_AUDIO_CAPABILITY* pAudioCap = NULL;
   M3G_VIDEO_CAPABILITY* pVideoCap = NULL;

   // ensure the Tx audio MPC was not already established via MONA
   if (0 == GetAudioTxLCN())
   {
      // simply choose first (most) preferred TX audio and video capability to open:
      for (int i=0; i< matchedAudioCapsList.numCaps; i++)
      {
         if (M3G_E_TX == matchedAudioCapsList.capability[i].audioCapability.direction)
         {
            pAudioCap = &matchedAudioCapsList.capability[i].audioCapability;
            //Set the Selected audio codertype
            if(pAudioCap->coderType == M3G_E_G7231)
               SetSelectedAudioCoder(AUD_CODER_G723);
            else
               SetSelectedAudioCoder(AUD_CODER_AMR);
            LOG_ENTRY(m_ControlHandle,"Endpoint - 3GSetAudioTo%s\n",GetSelAudCoderStr());
            break;
         }
      }
   }

   // ensure the Tx video MPC was not already established via MONA
   if (0 == GetVideoTxLCN())
   {
      // simply choose first (most) preferred TX audio and video capability to open:
      for (int i=0; i < matchedVideoCapsList.numCaps; i++)
      {
         if (M3G_E_TX == matchedVideoCapsList.capability[i].videoCapability.direction)
         {
            pVideoCap = &matchedVideoCapsList.capability[i].videoCapability;
            //Set the Selected video codertype
            if(pVideoCap->coderType == M3G_E_MPEG4)
            {
               SetSelectedVideoCoder(VID_CODER_MPEG4);
               LOG_ENTRY(m_ControlHandle,"Endpoint - 3GSetVideoTo%s\n",GetSelVidCoderStr());
            }
            else if(pVideoCap->coderType == M3G_E_H264)
            {
               SetSelectedVideoCoder(VID_CODER_H264);
               LOG_ENTRY(m_ControlHandle,"Endpoint - 3GSetVideoTo%s\n",GetSelVidCoderStr());
            }
            else
               SetSelectedVideoCoder(VID_CODER_H263);
            // Set the preferred H.263 resolution to QCIF
            if (M3G_E_H263 == pVideoCap->coderType)
            {
               // ensure only one format is chosen:
               if (0 != pVideoCap->options.h263.qcifMPI)
               {
                LOG_ENTRY(m_ControlHandle,"Endpoint - 3GSetVideoTo%s and size=QCIF\n",GetSelVidCoderStr());
                  pVideoCap->options.h263.sqcifMPI = 0;  // priority given to qcif for now
               }
              else	
                LOG_ENTRY(m_ControlHandle,"Endpoint - 3GSetVideoTo%s and size=SQCIF\n",GetSelVidCoderStr());
               // else only sqcif should be non-zero, so acceptable as is
            }
            break;
         }
      }
   }

   if ((NULL != pAudioCap) || (NULL != pVideoCap))
   {
      M3G_E_ADAPTATION_LYR_TYPE audioAL = GetMatchingAudioAL(&matchedH223CapsList);
      M3G_E_ADAPTATION_LYR_TYPE videoAL = GetMatchingVideoAL(&matchedH223CapsList);
      OpenLCs(pAudioCap, audioAL, pVideoCap, videoAL);
   }
   // if no matches and no MONMA MPCs established, log this as an error
   else if ((0 != GetVideoTxLCN()) && (0 != GetAudioTxLCN()))
   {
      LOG_ERROR(m_AudioHandle,"No MPCs established or matching TX audio or video capabilities found");
   }
}

//*****************************************************************************
// Function: void Endpoint::OpenLCs(M3G_AUDIO_CAPABILITY* pAudioCap, M3G_E_ADAPTATION_LYR_TYPE audioAL, 
// Description: Open the logical channel
// Return: void 
// Parameters: M3G_AUDIO_CAPABILITY *pAudioCap 
//             M3G_E_ADAPTATION_LYR_TYPE audioAL
//             M3G_VIDEO_CAPABILITY *pVideoCap
//             M3G_E_ADAPTATION_LYR_TYPE videoAL
//*****************************************************************************
void Endpoint::OpenLCs(M3G_AUDIO_CAPABILITY* pAudioCap, M3G_E_ADAPTATION_LYR_TYPE audioAL, 
					   M3G_VIDEO_CAPABILITY* pVideoCap, M3G_E_ADAPTATION_LYR_TYPE videoAL)
{
   // first configure and send audio OLC:
   M3G_H223_LC_PARAMS h223OLCParams; 
   INIT_M3G_H223_LC_PARAMS(&h223OLCParams);

   h223OLCParams.adaptationLayerType = audioAL;
   h223OLCParams.segmentable = static_cast<M3G_BOOL>(m_PreferredH223LCParams.isAudioSegmentable);
   h223OLCParams.AL3_ControlFieldSize = m_PreferredH223LCParams.AL3ControlFieldSize;
   h223OLCParams.AL3_SendBufferSize = m_PreferredH223LCParams.AL3SendBufferSize;

   // if opening an audio OLC
   if (NULL != pAudioCap)
   {
      LOG_ENTRY(m_ControlHandle,"Sending Audio OLC:\n");
      // log the audio OLC capabilities
      RecordLCMuxParams(m_ControlHandle, &h223OLCParams);
      RecordAudioCap(m_ControlHandle, pAudioCap);

      SetTxAudioCapType(pAudioCap->coderType);

      if (M3G_ERROR == m3g_OpenLC(m_ControlHandle,
                                  &h223OLCParams,
                                  M3G_E_AUDIO_CAPABILITY,
                                  reinterpret_cast<M3G_CAPABILITY*>(pAudioCap)))
      {
         LOG_ERROR(m_ControlHandle,"m3g_OpenLC(audioLC) failure\n");
      }
      else
      {
         LOG_ENTRY(m_ControlHandle,"Successful m3g_OpenLC(audioLC) completion\n");
      }
   }

   // if opening a video OLC
   if (NULL != pVideoCap)
   {
      // configure and send video OLC:
      h223OLCParams.adaptationLayerType = videoAL;
      h223OLCParams.segmentable = static_cast<M3G_BOOL>(m_PreferredH223LCParams.isVideoSegmentable);

      SetTxVideoCapType(pVideoCap->coderType);

      // add the MPEG-4 decoderConfigurationInformation if necessary
      if ((M3G_E_MPEG4 == pVideoCap->coderType) &&
          (0 < m_DciOctetStrSize) && 
          (m_DciMode & DCI_MODE_H245_OLC))
      {
          pVideoCap->options.mpeg4.decoderConfigLength = m_DciOctetStrSize;
          memcpy(&pVideoCap->options.mpeg4.decoderConfigInfo[0], m_DciOctetStr, m_DciOctetStrSize);
      }
      // add the H264 decoderConfigurationInformation and nalAlignedMode if necessary
      if (M3G_E_H264 == pVideoCap->coderType) 
      {
          if((0 < m_DciOctetStrSize) && 
              (m_DciMode & DCI_MODE_H245_OLC))
          {
              LOG_ENTRY(m_ControlHandle,"Add OLC H264 DCI \n");
              pVideoCap->options.h264.decoderConfigLength = m_DciOctetStrSize;
              memcpy(&pVideoCap->options.h264.decoderConfigInfo[0], m_DciOctetStr, m_DciOctetStrSize);
          }
      }

      // log the a/v OLC capabilities
      LOG_ENTRY(m_ControlHandle,"Sending Video OLC:\n");
      RecordLCMuxParams(m_ControlHandle, &h223OLCParams);
      RecordVideoCap(m_ControlHandle, pVideoCap);

      if (M3G_ERROR == m3g_OpenLC(m_ControlHandle,
                                  &h223OLCParams,
                                  M3G_E_VIDEO_CAPABILITY,
                                  reinterpret_cast<M3G_CAPABILITY*>(pVideoCap)))
      {
         LOG_ERROR(m_ControlHandle,"m3g_OpenLC(videoLC) failure\n");
      }
      else
      {
         LOG_ENTRY(m_ControlHandle,"Successful m3g_OpenLC(videoLC) completion\n");
      }
   }
}


//*****************************************************************************
// Function: M3G_E_ADAPTATION_LYR_TYPE Endpoint::GetMatchingAudioAL(M3G_CAPS_LIST *pMatchedH223CapsList)
// Description: Get Matching Audio Adaption Layer
// Return: M3G_E_ADAPTATION_LYR_TYPE 
// Parameters: M3G_CAPS_LIST *pMatchedH223CapsList 
//*****************************************************************************
M3G_E_ADAPTATION_LYR_TYPE Endpoint::GetMatchingAudioAL(M3G_CAPS_LIST *pMatchedH223CapsList)
{
   M3G_E_ADAPTATION_LYR_TYPE preferredAudioAL = static_cast<M3G_E_ADAPTATION_LYR_TYPE>(m_PreferredH223LCParams.audioAL);
   bool isMatchFound = false;

   if (pMatchedH223CapsList->numCaps != 1)
   {
      LOG_ERROR(m_ControlHandle,"No compatible H.223 capabilities\n");
      // though non-compliant, attempt to open logical channel using preferred audio AL anyway
      return preferredAudioAL;
   }

   // if preferred audio AL is supported by remote use it:
   switch (preferredAudioAL)
   {
      case M3G_E_AL3:
         isMatchFound = (M3G_AUDIO_AL3 == (pMatchedH223CapsList->capability[0].h223Capability.adaptationLayerMedia & M3G_AUDIO_AL3));
         break;

      case M3G_E_AL2_WITH_SEQ_NUMS:      // fall thru:
      case M3G_E_AL2_WITHOUT_SEQ_NUMS:
         isMatchFound = (M3G_AUDIO_AL2 == (pMatchedH223CapsList->capability[0].h223Capability.adaptationLayerMedia & M3G_AUDIO_AL2));
         break;

      case M3G_E_AL1_FRAMED:            // fall thru:
      case M3G_E_AL1_UNFRAMED:
         isMatchFound = (M3G_AUDIO_AL1 == (pMatchedH223CapsList->capability[0].h223Capability.adaptationLayerMedia & M3G_AUDIO_AL1));
         break;

      default:
         LOG_ERROR(m_ControlHandle,"Invalid audio AL specified:%d\n",static_cast<int>(preferredAudioAL));
         break;
   }

   if (true == isMatchFound)
   {
      return preferredAudioAL;
   }
   else
   {
      // otherwise, simply use the most error resilient AL common among both local and remote, ignoring AL1 for now
      // which is typically used for H.245 control channel or data only:
      LOG_ENTRY(m_ControlHandle,"Specified audio AL not common among both local and remote.  Using common AL capability instead\n");
      return ((M3G_AUDIO_AL3 == (pMatchedH223CapsList->capability[0].h223Capability.adaptationLayerMedia & M3G_AUDIO_AL3)) ?
               M3G_E_AL3 : M3G_E_AL2_WITH_SEQ_NUMS);
   }
}

//*****************************************************************************
// Function: M3G_E_ADAPTATION_LYR_TYPE Endpoint::GetMatchingVideoAL(M3G_CAPS_LIST *pMatchedH223CapsList)
// Description: Get Matching Video Adaption Layer
// Return: M3G_E_ADAPTATION_LYR_TYPE 
// Parameters: M3G_CAPS_LIST *pMatchedH223CapsList 
//*****************************************************************************
M3G_E_ADAPTATION_LYR_TYPE Endpoint::GetMatchingVideoAL(M3G_CAPS_LIST *pMatchedH223CapsList)
{
   M3G_E_ADAPTATION_LYR_TYPE preferredVideoAL = static_cast<M3G_E_ADAPTATION_LYR_TYPE>(m_PreferredH223LCParams.videoAL);
   bool isMatchFound = false;
   if (pMatchedH223CapsList->numCaps != 1)
   {
      LOG_ERROR(m_ControlHandle,"No compatible H.223 capabilities\n");
      // though non-compliant, attempt to open logical channel using preferred video AL anyway
      return preferredVideoAL;
   }

   // if preferred video AL is supported by remote use it:
   switch (preferredVideoAL)
   {
      case M3G_E_AL3:
         isMatchFound = (M3G_VIDEO_AL3 == (pMatchedH223CapsList->capability[0].h223Capability.adaptationLayerMedia & M3G_VIDEO_AL3));
         break;

      case M3G_E_AL2_WITH_SEQ_NUMS:      // fall thru:
      case M3G_E_AL2_WITHOUT_SEQ_NUMS:
         isMatchFound = (M3G_VIDEO_AL2 == (pMatchedH223CapsList->capability[0].h223Capability.adaptationLayerMedia & M3G_VIDEO_AL2));
         break;

      case M3G_E_AL1_FRAMED:            // fall thru:
      case M3G_E_AL1_UNFRAMED:
         isMatchFound = (M3G_VIDEO_AL1 == (pMatchedH223CapsList->capability[0].h223Capability.adaptationLayerMedia & M3G_VIDEO_AL1));
         break;

      default:
         LOG_ERROR(m_ControlHandle,"Invalid video AL specified:%d\n",static_cast<int>(preferredVideoAL));
         break;
   }
   if (true == isMatchFound)
   {
      return preferredVideoAL;
   }
   else
   {
      // otherwise, simply use the most error resilient AL common among both local and remote, ignoring AL1 for now
      // which is typically used for H.245 control channel or data only:
      LOG_ENTRY(m_ControlHandle,"Specified video AL not common among both local and remote.  Using common AL instead\n");
      return ((M3G_VIDEO_AL3 == (pMatchedH223CapsList->capability[0].h223Capability.adaptationLayerMedia & M3G_VIDEO_AL3)) ?
               M3G_E_AL3 : M3G_E_AL2_WITH_SEQ_NUMS);
   }
}

//*****************************************************************************
// Function: void Endpoint::AckOLCReq(M3G_REMOTE_OLC_REQ *pOLCReq)
// Description: ACK an open Logical channel requestr
// Return: void 
// Parameters: M3G_REMOTE_OLC_REQ *pOLCReq 
//*****************************************************************************
void Endpoint::AckOLCReq(M3G_REMOTE_OLC_REQ *pOLCReq)
{
   if ( M3G_ERROR == m3g_RespondToOLC(m_ControlHandle, pOLCReq->logicalChannelNumber, M3G_E_OLCACK) )
   {
      LOG_ERROR(m_ControlHandle,"m3g_RespondToOLC() failure\n");
   }
   else
   {
      LOG_ENTRY(m_ControlHandle,"Successful m3g_RespondToOLC() completion\n");
   }
}

//*****************************************************************************
// Function: void Endpoint::StartAudio
// Description: Start audio streaming in specified direction
// Return: void
// Parameters: direction - direction to stream audio
//*****************************************************************************
void Endpoint::StartAudio(M3G_E_DIRECTION direction)
{
   // record direction which is being initiated
   switch (direction)
   {
      case M3G_E_TX :
         m_pCurrentState->SetStatus(EPState::AUDIO_TX_STARTING);
         break;

      case M3G_E_RX :
         m_pCurrentState->SetStatus(EPState::AUDIO_RX_STARTING);
         break;

#ifdef FDX_ONLY
// FDX starting:
      // wait until both LCs or MPCs are established before initiating media
      case M3G_E_TXRX:
         if (M3G_ERROR == m3g_StartMedia(m_AudioHandle))
         {
            LOG_ERROR(m_AudioHandle,"m3g_StartMedia() failure\n");
         }
         else
         {
            LOG_ENTRY(m_AudioHandle,"Successful m3g_StartMedia() completion\n");
         }
      break;

      default:  
         break;
   }
#else
// HDX starting:
         // only cache uni-directional start to differentiate
      default:  
         break;
   }

   // initiate audio as soon as either uni-directional LC or MPC is established
   if (M3G_ERROR == m3g_ModifyMedia(m_AudioHandle, direction))
   {
      LOG_ERROR(m_AudioHandle,"m3g_ModifyMedia() failure\n");
   }
   else
   {
      LOG_ENTRY(m_AudioHandle,"Successful m3g_ModifyMedia() completion\n");
   }
#endif
}


//*****************************************************************************
// Function: void Endpoint::StartVideo
// Description: Start video streaming in specified direction
// Return: void
// Parameters: direction - direction to stream video
//*****************************************************************************
void Endpoint::StartVideo(M3G_E_DIRECTION direction)
{
    // record direction which is being initiated
   switch (direction)
   {
      case M3G_E_TX :
         m_pCurrentState->SetStatus(EPState::VIDEO_TX_STARTING);
         // if video is MPEG-4, check to see if the DCI needs to be configured inband:
         if ((M3G_E_MPEG4 == GetTxVideoCapType()) && (m_DciMode & DCI_MODE_INBAND))
         {
            SetInbandDCITxTo3G();
         }
         // if video is H264, check to see if the DCI needs to be configured inband:
         else if ((M3G_E_H264 == GetTxVideoCapType()) && (m_DciMode & DCI_MODE_INBAND))
         {
            SetInbandH264DCITxTo3G();
         }
         else 
             LOG_ENTRY(m_VideoHandle,"StartVideo: Not Setting Inband DCI\n");
         break;

      case M3G_E_RX :
          m_pCurrentState->SetStatus(EPState::VIDEO_RX_STARTING);
         if (M3G_E_MPEG4 == GetTxVideoCapType() && GetVidTranscodeEnabled()) 
           //SetInbandDCIRvdFrom3G();  // Not required

         break;

#ifdef FDX_ONLY
      case M3G_E_TXRX:
         // if video is H264, check to see if the DCI needs to be configured inband:
         if ((M3G_E_H264 == GetTxVideoCapType()) && (m_DciMode & DCI_MODE_INBAND))
         {
            LOG_ENTRY(m_VideoHandle,"StartVideo M3G_E_TXRX SetInbandH264DCITxTo3G\n");
            SetInbandH264DCITxTo3G();
         }
         if (M3G_ERROR == m3g_StartMedia(m_VideoHandle))
         {
            LOG_ERROR(m_VideoHandle,"m3g_StartMedia() failure\n");
         }
         else
         {
            LOG_ENTRY(m_VideoHandle,"Successful m3g_StartMedia() completion\n");
         }
      break;
#endif

      // only cache uni-directional start to differentiate
      default:  
         break;
   }

#ifndef FDX_ONLY

    if (direction == M3G_E_TXRX)
    {
         if ((M3G_E_H264 == GetTxVideoCapType()) && (m_DciMode & DCI_MODE_INBAND))
         {
            LOG_ENTRY(m_VideoHandle,"StartVideo M3G_E_TXRX SetInbandH264DCITxTo3G\n");
            SetInbandH264DCITxTo3G();
         }        
    }

   if (M3G_ERROR == m3g_ModifyMedia(m_VideoHandle, direction))
   {
      LOG_ERROR(m_VideoHandle,"m3g_ModifyMedia() failure");
   }
   else
   {
     LOG_ENTRY(m_VideoHandle,"Successful m3g_ModifyMedia() completion\n");
   }
#endif
}


//*****************************************************************************
// Function: void Endpoint::StopRxMedia()
// Description: Stop receiving media streaming from the 3G network
// Return: void 
// Parameters: none 
//*****************************************************************************
void Endpoint::StopRxMedia()
{
   // disable Rx by reseting a/v to Tx-only
   m_pCurrentState->ClearStatus((EPState::AUDIO_CHANS_STARTED | EPState::VIDEO_CHANS_STARTED));
   StartAudio(M3G_E_TX);
   StartVideo(M3G_E_TX);
}


//*****************************************************************************
// Function: void Endpoint::RestartFDXMedia()
// Description: Stop receiving media streaming from the 3G network
// Return: void 
// Parameters: none 
//*****************************************************************************
void Endpoint::RestartFDXMedia()
{
   // re-enable Rx by restarting TxRx
   StartAudio(M3G_E_TXRX);
   StartVideo(M3G_E_TXRX);
}


//*****************************************************************************
// Function: void Endpoint::StopMedia()
// Description: Stop media streaming
// Return: void 
// Parameters: none 
//*****************************************************************************
void Endpoint::StopMedia()
{
   StopAudio();
   StopVideo();
}


void Endpoint::StopAudio()
{
   if (M3G_ERROR == m3g_StopMedia(m_AudioHandle))
   {
      LOG_ERROR(m_AudioHandle,"m3g_StopMedia(AUDIO) failure\n");
   }
   else
   {
      LOG_ENTRY(m_AudioHandle,"Successful m3g_StopMedia(AUDIO) completion\n");
   }
}

void Endpoint::StopVideo()
{
   if (M3G_ERROR == m3g_StopMedia(m_VideoHandle))
   {
      LOG_ERROR(m_VideoHandle,"m3g_StopMedia(VIDEO) failure\n");
   }
   else
   {
      LOG_ENTRY(m_VideoHandle,"Successful m3g_StopMedia(VIDEO) completion\n");
   }
}

//*****************************************************************************
// Function: void Endpoint::CloseLCs()
// Description: Close the logical channels
// Return: void 
// Parameters: none 
//*****************************************************************************
void Endpoint::CloseLCs()
{
   // assume we will close our audio and video Tx lcns and the remote should do likewise
   if (0 != GetAudioTxLCN())
   {
      if (M3G_ERROR == m3g_CloseLC(m_ControlHandle, GetAudioTxLCN(), M3G_E_REQ_CHAN_CLOSE_NORMAL))
      {
         LOG_ERROR(m_ControlHandle,"m3g_CloseLC() failure\n");
      }
      else
      {
         LOG_ENTRY(m_ControlHandle,"Successful m3g_CloseLC() completion\n");
         strncpy(m_AudTxCapStr, "none", MAX_MEDIA_STR);
      }
   }

   if (0 != GetVideoTxLCN())
   {
      if (M3G_ERROR == m3g_CloseLC(m_ControlHandle, GetVideoTxLCN(), M3G_E_REQ_CHAN_CLOSE_NORMAL))
      {
         LOG_ERROR(m_ControlHandle,"m3g_CloseLC() failure\n");
      }
      else
      {
         LOG_ENTRY(m_ControlHandle,"Successful m3g_CloseLC() completion\n");
         strncpy(m_VidTxCapStr, "none", MAX_MEDIA_STR);
      }
   }
   // if all LCNs closed already by remote, stop H.245 session
   if ( (0 == GetAudioTxLCN()) && 
        (0 == GetAudioRxLCN()) &&
        (0 == GetVideoTxLCN()) &&
        (0 == GetVideoRxLCN()) )
   {
      LOG_ENTRY(GetControlHandle(), "Calling Notify(APP_M3G_ENDPOINT_DISCONNECT) from CloseLCs\n");
      Notify(APP_M3G_ENDPOINT_DISCONNECT);
      Stop_H245();
   }
}

//*****************************************************************************
// Function: void Endpoint::Stop_H245()
// Description: Stop the H245 session
// Return: void 
// Parameters: none 
//*****************************************************************************
void Endpoint::Stop_H245()
{
   if (M3G_ERROR == m3g_StopH245(m_ControlHandle))
   {
      LOG_ERROR(m_ControlHandle,"m3g_StopH245() failure\n");
   }
   else
   {
      LOG_ENTRY(m_ControlHandle,"Successful m3g_StopH245() completion\n");
   }
}

//*****************************************************************************
// Function: void Endpoint::SetLastMessageString(char *pMessageStr)
// Description: Save a string as the last message
// Return: void 
// Parameters: char *pMessageStr 
//*****************************************************************************
void Endpoint::SetLastMessageString(const char *pMessageStr)
{
   int length = (MAX_LAST_MSG_STRING > strlen(pMessageStr)) ? strlen(pMessageStr) : MAX_LAST_MSG_STRING-1;
   memset(m_LastMessageBuff, 0, sizeof(char)*MAX_LAST_MSG_STRING);
   strncpy(&m_LastMessageBuff[0], pMessageStr, length);
   m_LastMessageBuff[length] = '\0';
}


//*****************************************************************************
// Function: char* Endpoint::GetStateString()
// Description: Return a formatted string of the current endpoint state
// Return: char* 
// Parameters: none 
//*****************************************************************************
char * Endpoint::GetStateString()
{
   memset(m_StateBuffer, 0, sizeof(char)*MAX_STATE_STRING);
   sprintf(m_StateBuffer, "M3G%2d: %15s  %5d  %5d  %5s%1s %5d  %5d  %5s%1s  %6s",
           m_Index,
           m_CurrentStateStr,
           GetAudioTxLCN(),
           GetAudioRxLCN(),
           m_AudTxCapStr,
           (GetAudTranscodeEnabled()?"*":" "),
           GetVideoTxLCN(),
           GetVideoRxLCN(),
           m_VidTxCapStr,
           (GetVidTranscodeEnabled()?"*":" "),
           m_LastMessageBuff);
   return m_StateBuffer;
}

//*****************************************************************************
// Function: void Endpoint::SetParameters()
// Description: Set the M3G device parameters
// Return: void 
// Parameters: none 
//*****************************************************************************
void Endpoint::SetParameters()
{
   int numParameters(0);
   M3G_PARM_INFO *parameterTbl;

   EndpointMngr::Instance()->GetParameters(&numParameters, &parameterTbl);

   m_numParmsSet = 0;
   for ( int parmIndex=0; parmIndex<numParameters ; parmIndex++ )
   {

      if ( M3G_ERROR == m3g_SetParm(m_ControlHandle, &parameterTbl[parmIndex]) )
      {
         LOG_ERROR(m_ControlHandle, "m3g_SetParm() failure\n");
      }
      else
      {
         LOG_ENTRY(m_ControlHandle, "Successful m3g_SetParm(type = %s, value = 0x%X) completion\n",
                   szParms[parameterTbl[parmIndex].parameterType], parameterTbl[parmIndex].parmValue);
      } 
   }
}

//*****************************************************************************
// Function: void Endpoint::SetNetworkConfig()
// Description: Set the TDM networtk configuration
// Return: void 
// Parameters: none 
//*****************************************************************************
void Endpoint::SetNetworkConfig()
{
   if ( GetIsLoopbackNotNetwork())
   {
      LOG_ENTRY(m_ControlHandle,"Generate simulated GCEV_SETCONFIGDATA event (set loopback)\n");
      ProcessEvent( GCEV_SETCONFIGDATA, NULL, 0, GetNetworkHandle());
   }
   else
   {
#ifdef HMP //HMP3.1
      LOG_ENTRY(m_ControlHandle,"Generate GCEV_SETCONFIGDATA simulated event (set non-loopback)\n");
      ProcessEvent( GCEV_SETCONFIGDATA, NULL, 0, GetNetworkHandle());
#else //MMP
    if (IsNbup())
    {
        LOG_ENTRY(m_ControlHandle,"Generate GCEV_SETCONFIGDATA simulated event (set non-loopback)\n");
        ProcessEvent( GCEV_SETCONFIGDATA, NULL, 0, GetNetworkHandle());
    }
    else
    {
      long request_id;
      GC_PARM_BLKP target_datap=NULL;
      gc_util_insert_parm_val(&target_datap,
                              CCSET_DM3FW_PARM,
                              CCPARM_TRANSPARENTMODE, 
                              sizeof(char),
                              CCDM3FW_PARMTRANSPARENTMODE_ENABLE);

      if ( gc_SetConfigData(GCTGT_CCLIB_CHAN,
                            m_NetworkHandle,
                            target_datap,
                            10,
                            GCUPDATE_IMMEDIATE,
                            &request_id,
                            EV_ASYNC) != GC_SUCCESS )
      {
         LOG_ERROR(m_NetworkHandle, "Error calling gc_SetConfigData\n");
         LogGlobalCallError();
      }
      else
      {
         LOG_ENTRY(m_NetworkHandle, "Successful gc_SetConfigData() completion\n");
      }

      gc_util_delete_parm_blk(target_datap);
    }
#endif
   }
}

//*****************************************************************************
// Function: void Endpoint::ResetNetworkConfig()
// Description: Reset the TDM network configuration
// Return: void 
// Parameters: none 
//*****************************************************************************
void Endpoint::ResetNetworkConfig()
{
   if ( GetIsLoopbackNotNetwork())
   {
      LOG_ENTRY(m_ControlHandle,"Generate simulated GCEV_SETCONFIGDATA event (reset loopback)\n");
      ProcessEvent( GCEV_SETCONFIGDATA, NULL, 0, GetNetworkHandle());
   }
   else
   {
#ifdef HMP //HMP3.1
      LOG_ENTRY(m_ControlHandle,"Generate simulated GCEV_SETCONFIGDATA event (reset non-loopback)\n");
      ProcessEvent( GCEV_SETCONFIGDATA, NULL, 0, GetNetworkHandle());
#else //MMP
      long request_id;
      GC_PARM_BLKP target_datap=NULL;
      gc_util_insert_parm_val(&target_datap,
                              CCSET_DM3FW_PARM,
                              CCPARM_TRANSPARENTMODE, 
                              sizeof(char),
                              CCDM3FW_PARMTRANSPARENTMODE_DISABLE);

      if ( gc_SetConfigData(GCTGT_CCLIB_CHAN,
                            m_NetworkHandle,
                            target_datap,
                            10,
                            GCUPDATE_IMMEDIATE,
                            &request_id,
                            EV_ASYNC) != GC_SUCCESS )
      {
         LOG_ERROR(m_NetworkHandle, "Error calling gc_SetConfigData\n");
         LogGlobalCallError();
      }
      else
      {
         LOG_ENTRY(m_NetworkHandle, "Successful gc_SetConfigData() completion\n");
      }

      gc_util_delete_parm_blk(target_datap);
#endif
   }
}

//*****************************************************************************
// Function: void Endpoint::LogGlobalCallError()
// Description: Log a global call error
// Return: void 
// Parameters: none 
//*****************************************************************************
void Endpoint::LogGlobalCallError()
{
   GC_INFO gcInfo;

   if ( gc_ErrorInfo(&gcInfo) != GC_SUCCESS )
   {
      LOG_ERROR(m_NetworkHandle, "gc_ErrorInfo failed\n");
   }
   else
   {
      LOG_ERROR(m_NetworkHandle, "gcValue  : 0x%x\n", gcInfo.gcValue);
      LOG_ERROR(m_NetworkHandle, "gcMsg    : %s\n", gcInfo.gcMsg);
      LOG_ERROR(m_NetworkHandle, "ccLibId  : %d\n", gcInfo.ccLibId);
      LOG_ERROR(m_NetworkHandle, "ccLibName: %s\n", gcInfo.ccLibName);
      LOG_ERROR(m_NetworkHandle, "ccValue  : 0x%x\n", gcInfo.ccValue);
      LOG_ERROR(m_NetworkHandle, "ccMsg    : %s\n", gcInfo.ccMsg);
      LOG_ERROR(m_NetworkHandle, "info     : %s\n", gcInfo.additionalInfo);
   }
}


//*****************************************************************************
// Function: void Endpoint::SendUII()
// Description: Send a UUI message
// Return: void 
// Parameters: none 
//*****************************************************************************
void Endpoint::SendUII()
{
   M3G_H245_UII h245UII;
   INIT_M3G_H245_UII(&h245UII);

   h245UII.numDigits = 1;
   h245UII.digitBuffer[0] = m_DigitToSend;

   if (M3G_ERROR == m3g_SendH245UII(m_ControlHandle, &h245UII))
   {
      LOG_ERROR(m_ControlHandle,"m3g_SendH245UII() failure\n");
   }
   else
   {
      LOG_ENTRY(m_ControlHandle,"Successful m3g_SendH245UII() completion\n");
   }
}


void Endpoint::SendFastVideoUpdate()
{
   
   M3G_H245_MISC_CMD h245MiscCmd;
   INIT_M3G_H245_MISC_CMD(&h245MiscCmd);

   h245MiscCmd.logicalChannelNumber = GetVideoRxLCN();
   h245MiscCmd.h245MiscCmdType = M3G_E_FAST_UPDATE_PICTURE;
   h245MiscCmd.h245MiscCmdParams.noParams = NULL;
   
   LOG_ENTRY(m_ControlHandle,"SendFastVideoUpdate()\n");

   if (M3G_ERROR == m3g_SendH245MiscCmd(m_ControlHandle, &h245MiscCmd))
   {
      LOG_ERROR(m_ControlHandle,"m3g_SendH245MiscCmd() failure");
   }
   else
   {
      LOG_ENTRY(m_ControlHandle,"Successful m3g_SendH245MiscCmd() completion\n");
   }
}

//*****************************************************************************
// Function: void Endpoint::Shutdown
// Description: Dispatch shutdown request to the current state
// Return: void 
// Parameters: none 
//*****************************************************************************
void Endpoint::Shutdown()
{
   LOG_ENTRY(m_ControlHandle,"Endpoint[%d] Shutting down in state %s\n",GetIndex(), 
		m_CurrentStateStr);
   if ( m_pCurrentState ) m_pCurrentState->Shutdown();
}


//*****************************************************************************
// Function: void Endpoint::Abort
// Description: Abort whatever is going on with an m3g_Reset to all devices
// Return: void 
// Parameters: none 
//*****************************************************************************
void Endpoint::Abort()
{
   LOG_ENTRY(m_ControlHandle,"Endpoint[%d] aborting\n",GetIndex());
   Reinitialize(); 
}


//*****************************************************************************
// Function: void Endpoint::Notify
// Description: 3G endpoint needs a Notify that works with a Bridged Hairpin.  No other
// 	endpoints are managed by a hairpin object
// Return: void 
// Parameters: event 
//*****************************************************************************
   void Endpoint::Notify(int event)
   {
      if (m_pBridgedCall)
      {
         METAEVENT metaevent;
         metaevent.evttype =  event;
         metaevent.evtdev = 0;
         m_pBridgedCall->ProcessEvent(metaevent);
      }
      else if (m_pBridgedMedia)
      {
         METAEVENT metaevent;
         metaevent.evttype =  event;
         metaevent.evtdev = 0;
         m_pBridgedMedia->ProcessEvent(metaevent);
      }
      else if (m_pBridgedHairp)
      {
         static int epindex;
         METAEVENT metaevent;
         metaevent.evttype =  event;
         // Bridged hairpins are a little different - they have 2 3G endpoints so we have
         // to distinguish between them.  Use the endpoint index.
         epindex = m_Index;
         metaevent.evtdatap =  &epindex;
         m_pBridgedHairp->ProcessEvent(metaevent);
      }
#ifdef USE_RTSP
      else if (m_pBridgedRtsp)
      {
         static int epindex;
         METAEVENT metaevent;
         metaevent.evttype =  event;
         metaevent.evtdev = 0;
         epindex = m_Index;
         metaevent.evtdatap =  &epindex;
         m_pBridgedRtsp->ProcessEvent(metaevent);
      }
#endif
   }


//*****************************************************************************
// Function: bool Endpoint::Is SigProtocolManaged
// Description: Is internal ISDN or SS7 gateway being used?
// Return: true/false 
// Parameters: none 
//*****************************************************************************
bool Endpoint::IsSigProtocolManaged()
{
   // Have stream device named IdtiBxTy, idtiBxTy
   if (('I' == m_transportName[0] || 'i' == m_transportName[0]) && 'd' == m_transportName[1])
      return true;
   // Have stream device named SdtiBxTy, sdtiBxTy
   if (('S' == m_transportName[0] || 's' == m_transportName[0]) && 'd' == m_transportName[1])
      return true;
   return false;
}

//*****************************************************************************
// Function: void Endpoint::EnableNbupEvents
// Description: Enable needed Nbup-related events on the IPM device used for
//		Nbup
// Return: void 
// Parameters: none 
//*****************************************************************************
void Endpoint::EnableNbupEvents ()
{
  const int nEvents = 4;
  eIPM_EVENT events[nEvents] =
    { EVT_INIT_RECEIVED, EVT_PROCEDURE_DONE, EVT_SEND_FAILED, EVT_NACK_SENT };

  if (ipm_EnableEvents (m_NetworkHandle, events, nEvents, EV_ASYNC) == -1)
   LOG_ERROR(m_NetworkHandle, "Endpoint[%d] - ipm_EnableEvents for Nbup failed\n", GetIndex());
  else
   LOG_ENTRY(m_NetworkHandle, "Endpoint[%d] - successful ipm_EnableEvents for Nbup completion\n", GetIndex());
}

void Endpoint::StartNbupMedia ()
{
  // Set up remote RTP Media Info struct
  IPM_MEDIA_INFO nbupMediaInfo;

  memset(&nbupMediaInfo, 0, sizeof(IPM_MEDIA_INFO));
  nbupMediaInfo.unCount = 2;
  nbupMediaInfo.MediaData[0].eMediaType = MEDIATYPE_NBUP_REMOTE_RTP_INFO;

  // Use RTP port and IP address from config file
  nbupMediaInfo.MediaData[0].mediaInfo.PortInfo.unPortId = m_remoteNbupPort;
  strcpy (nbupMediaInfo.MediaData[0].mediaInfo.PortInfo.cIPAddress, m_remoteNbupIPAddr);

  // Set up NBUP info
  nbupMediaInfo.MediaData[1].eMediaType = MEDIATYPE_NBUP_PROFILE_INFO;
  INIT_IPM_NBUP_PROFILE_INFO (&(nbupMediaInfo.MediaData[1].mediaInfo.NBUPProfileInfo));
  nbupMediaInfo.MediaData[1].mediaInfo.NBUPProfileInfo.eProfileType = NBUP_PROFILE_3G324M;
  nbupMediaInfo.MediaData[1].mediaInfo.NBUPProfileInfo.ucTxPLT = 111;
  nbupMediaInfo.MediaData[1].mediaInfo.NBUPProfileInfo.ucRxPLT = 111;
  LOG_ENTRY(m_NetworkHandle, "Endpoint[%d] - Remote Nbup address - %s        Remote Nbup port - %d\n",
            GetIndex(),
            nbupMediaInfo.MediaData[0].mediaInfo.PortInfo.cIPAddress,
            nbupMediaInfo.MediaData[0].mediaInfo.PortInfo.unPortId);
  // This is to cover a bug where two IPMEV_ENABLE events are delivered, resulting in two ipm_StartMedias
  if (!m_NbupMediaStarted)  {
    if (ipm_StartMedia (m_NetworkHandle, &nbupMediaInfo, DATA_IP_TDM_BIDIRECTIONAL, EV_ASYNC) == -1)
       LOG_ERROR(m_NetworkHandle, "Endpoint[%d] - ipm_StartMedia for Nbup failed\n", GetIndex());
     else  {
       LOG_ENTRY(m_NetworkHandle, "Endpoint[%d] - successful ipm_StartMedia for Nbup completion\n", GetIndex());
       m_NbupMediaStarted = true;
     }
  }
}

//*****************************************************************************
// Function: void Endpoint::
// Description: Do an ipm_Stop on Nbup media stream
// Return: void 
// Parameters: none 
//*****************************************************************************
void Endpoint::StopNbupMedia ()
{
  if (m_NbupMediaStarted)  {
   if (ipm_Stop(m_NetworkHandle, STOP_ALL, EV_SYNC) == -1)
       LOG_ERROR(m_NetworkHandle, "Endpoint[%d] - ipm_Stop for Nbup failed\n", GetIndex());
     else  {
       LOG_ENTRY(m_NetworkHandle, "Endpoint[%d] - successful ipm_Stop for Nbup completion\n", GetIndex());
       m_NbupMediaStarted = false;
     }
  }
}

//*****************************************************************************
// Function: void Endpoint::NbupInitSend
// Description: As part of NBUP connection establishment, send an NBUP init message
// Return: void 
// Parameters: none 
//*****************************************************************************
void Endpoint::NbupInitSend()
{
  // Initiate an NBUP session
  IPM_INIT_SEND initSend;
  IPM_NBUP_INIT_SEND nbupSend;

  // Initialize the main structure
  INIT_IPM_INIT_SEND (&initSend);
  initSend.eProtocol = RTP_PROTOCOL_NBUP;
  initSend.data.pNBUP = &nbupSend;

  // Initialize the NBUP specific structure
  INIT_IPM_NBUP_INIT_SEND (&nbupSend, 1, 1);

  // Init function handles memory allocation
  nbupSend.pRFCIs[0].ucID = (unsigned char) 0x44;
  nbupSend.pRFCIs[0].pSubFlows[0].eFlowSize = NBUP_FLOW_SIZE_320_BITS;
  // Send init message
  if (ipm_InitSend (m_NetworkHandle, &initSend) == -1)
     LOG_ERROR(m_NetworkHandle, "Endpoint[%d] - ipm_InitSend for Nbup failed\n", GetIndex());
  else  
     LOG_ENTRY(m_NetworkHandle, "Endpoint[%d] - successful ipm_InitSend for Nbup completion\n", GetIndex());

  // De-allocate memory allocated
  FREE_IPM_NBUP_INIT_SEND (&nbupSend);
}

//*****************************************************************************
// Function: void Endpoint::NbupInitresponseSend
// Description: As part of Nbup connection establishment, send an NBUP init response
// Return: void 
// Parameters: none 
//*****************************************************************************
void Endpoint::NbupInitResponseSend()
{
  IPM_INIT_RESPONSE initResp;
  IPM_NBUP_INIT_RESPONSE nbupResp;

  // Initialize the main structure
  INIT_IPM_INIT_RESPONSE (&initResp);
  initResp.eProtocol = RTP_PROTOCOL_NBUP;
  initResp.data.pNBUP = &nbupResp;

  // Initialize the NBUP specific structure with an ACK response
  INIT_IPM_NBUP_INIT_RESPONSE (&nbupResp);
  nbupResp.eResponse = NBUP_INIT_RESPONSE_ACK;

  if (ipm_InitResponseSend (m_NetworkHandle, &initResp) == -1)
     LOG_ERROR(m_NetworkHandle, "Endpoint[%d] - ipm_InitResponseSend for Nbup failed\n", GetIndex());
  else  
     LOG_ENTRY(m_NetworkHandle, "Endpoint[%d] - successful ipm_InitResponseSend for Nbup completion\n", GetIndex());
}

//*****************************************************************************
// Function: void Endpoint::GetNbupPortInfo
// Description: Get p-stream port info for IPM device used for Nbup
// Return: void 
// Parameters: none 
//*****************************************************************************
void Endpoint::GetNbupPortInfo ()
{
   // IPM device used for Nbup transport
   if ( dev_GetTransmitPortInfo(m_NetworkHandle, this) != DEV_SUCCESS )
   {
      LOG_ERROR(m_NetworkHandle,"dev_GetTransmitPortInfo() failure\n");
   }
   else
   {
      LOG_ENTRY(m_NetworkHandle,"Successful dev_GetTransmitPortInfo() call\n");
   }

   if ( dev_GetReceivePortInfo(m_NetworkHandle, this) != DEV_SUCCESS )
   {
      LOG_ERROR(m_NetworkHandle,"dev_GetReceivePortInfo() failure\n");
   }
   else
   {
      LOG_ENTRY(m_NetworkHandle,"Successful dev_GetReceivePortInfo() call\n");
   }

   // 3G Tx and Rx ports for control/h223/Nbup
   if ( DEV_SUCCESS != dev_GetTransmitPortInfo(m_ControlHandle, this) )
   {
      LOG_ERROR(m_ControlHandle,"dev_GetTransmitPortInfo() failure\n");
   }
   else
   {
      LOG_ENTRY(m_ControlHandle,"Successful dev_GetTransmitPortInfo() completion\n");
   }

   if ( DEV_SUCCESS != dev_GetReceivePortInfo(m_ControlHandle, this) )
   {
      LOG_ERROR(m_ControlHandle,"dev_GetReceivePortInfo() failure\n");
   }
   else
   {
      LOG_ENTRY(m_ControlHandle,"Successful dev_GetReceivePortInfo() completion\n");
   }
}

//*****************************************************************************
// Function: void Endpoint::ConnectIPMNbupTo3GNbup
// Description: Set up for and do dev_PortConnect on IPM-3G connections
// Return: void 
// Parameters: Device handle, Tx and Rx port info
//*****************************************************************************
void Endpoint::ConnectIPMNbupTo3GNbup(SRL_DEVICE_HANDLE deviceHandle, DM_PORT_INFO &TxPortInfo, 
                                      DM_PORT_INFO &RxPortInfo)
{
   DM_PORT_CONNECT_INFO_LIST portConnectInfoList;

   EndpointMngr::Instance()->InitPortConnectInfoList(&portConnectInfoList, EndpointMngr::NBUP_PORT, false);
   memcpy(&(portConnectInfoList.port_connect_info[0].port_info_tx),
          &TxPortInfo,
          sizeof(DM_PORT_INFO));
   memcpy(&(portConnectInfoList.port_connect_info[0].port_info_rx),
          &RxPortInfo,
          sizeof(DM_PORT_INFO));

   if ( DEV_SUCCESS != dev_PortConnect(deviceHandle, &portConnectInfoList, this) )
   {
      LOG_ERROR(deviceHandle,
                "dev_PortConnect() failure for Nbup - %s\n",
                ATDV_ERRMSGP(deviceHandle));
      if (GetNbupHandle() == deviceHandle)
        m_NbupIPMPortConnected = false;
      else if (GetControlHandle() == deviceHandle)
        m_Nbup3GPortConnected = false;
   }
   else
   {
      LOG_ENTRY(deviceHandle,"Successful dev_PortConnect() call for Nbup\n");
      if (GetNbupHandle() == deviceHandle)
        m_NbupIPMPortConnected = true;
      else if (GetControlHandle() == deviceHandle)
        m_Nbup3GPortConnected = true;
   }
}

//*****************************************************************************
// Function: void Endpoint::DisconnectIPMNbupFrom3GNbup
// Description: Set up for and do dev_PortDisconnect on IPM-3G connections
// Return: void 
// Parameters: Device handle, Tx and Rx port info
//*****************************************************************************
void Endpoint::DisconnectIPMNbupFrom3GNbup(SRL_DEVICE_HANDLE deviceHandle, DM_PORT_INFO &TxPortInfo, 
                                           DM_PORT_INFO &RxPortInfo)
{
   DM_PORT_CONNECT_INFO_LIST portConnectInfoList;

   EndpointMngr::Instance()->InitPortConnectInfoList(&portConnectInfoList,  EndpointMngr::NBUP_PORT, false);
   memcpy(&(portConnectInfoList.port_connect_info[0].port_info_tx),
          &TxPortInfo,
          sizeof(DM_PORT_INFO));
   memcpy(&(portConnectInfoList.port_connect_info[0].port_info_rx),
          &RxPortInfo,
          sizeof(DM_PORT_INFO));

   if ( DEV_SUCCESS != dev_PortDisconnect(deviceHandle, &portConnectInfoList, this) )
   {
      LOG_ERROR(deviceHandle,
                "dev_PortDisconnect() failure for Nbup - %s\n",
                ATDV_ERRMSGP(deviceHandle));
   }
   else
   {
      LOG_ENTRY(deviceHandle,"Successful dev_PortDisconnect() call for Nbup\n");

      if (GetNbupHandle() == deviceHandle)
        m_NbupIPMPortConnected = false;
      else if (GetControlHandle() == deviceHandle)
        m_Nbup3GPortConnected = false;

      // Port disconnect is intended to be called only on system shutdown.  At this point,
      // the event processing loop has been exited.  So, we pick ip the disconnect-related
      // event here, essentially making dev_PortDisconnect a sync call.
      sr_waitevt(1000);
      int evtType = sr_getevttype();
      int evtDev = sr_getevtdev();
      switch (evtType) {
        case DMEV_PORT_DISCONNECT:
          LOG_ENTRY(evtDev, "dev_PortDisconnect terminatated with DMEV_PORT_DISCONNECT\n");
        break;
        case DMEV_PORT_DISCONNECT_FAIL:
          LOG_ERROR(evtDev, "dev_PortDisconnect terminatated with DMEV_PORT_DISCONNECT_FAIL\n");
        break;
        default:
          LOG_ERROR(evtDev, "dev_PortDisconnect terminatated with unknown event 0x%x\n", evtType);
        break;
      }
   }
}

//*****************************************************************************
// Function: void Endpoint::RecordIPMNbupPortInfo
// Description: Save Nbup-related port infor for dev_PortConnect
// Return: void 
// Parameters: device handle, event type and port information 
//*****************************************************************************
void Endpoint::RecordIPMNbupPortInfo(long evtDev, long evtType, DM_PORT_INFO *portInfo)
{
      switch ( portInfo->port_media_type )  {
         case DM_PORT_MEDIA_TYPE_NBUP:
            if ( evtType == DMEV_GET_TX_PORT_INFO )
            {
               LOG_ENTRY(evtDev,"\t\tSaving Tx IPM Nbup port info\n");
               memcpy(&m_IPMNbupTxPortInfo, portInfo, sizeof(DM_PORT_INFO));
            }
            else
            {
               LOG_ENTRY(evtDev,"\t\tSaving Rx IPM Nbup port info\n");
               memcpy(&m_IPMNbupRxPortInfo, portInfo, sizeof(DM_PORT_INFO));
            }
            break;
         default:
            LOG_ERROR(evtDev,"\t\tInvalid DM_PORT_MEDIA_TYPE port info need DM_PORT_MEDIA_TYPE_NBUP\n");
            break;
       }
}


void Endpoint::StartTracing()
{
   if (0 < m_TraceLevel)
   {
      M3G_TRACE_INFO traceInfo;
      INIT_M3G_TRACE_INFO(&traceInfo);
      traceInfo.bitmask = m_TraceLevel;
      traceInfo.logfile = m_TraceFileName;

      if (M3G_ERROR == m3g_StartTrace(m_ControlHandle, &traceInfo))
      {
         LOG_ERROR(m_ControlHandle, "m3g_StartTrace() failure\n");
      }
      else
      {
         LOG_ENTRY(m_ControlHandle, "Successful m3g_StartTrace() completion\n");
      }
   }
}


void Endpoint::StopTracing()
{
   if (0 < m_TraceLevel)
   {
      if (M3G_ERROR == m3g_StopTrace(m_ControlHandle))
      {
         LOG_ERROR(m_ControlHandle, "m3g_StopTrace() failure\n");
      }
      else
      {
         LOG_ENTRY(m_ControlHandle, "Successful m3g_StopTrace() completion\n");
      }
   }
}

void Endpoint::SetInbandDCITxTo3G()
{
   M3G_PARM_INFO inbandDCIParm;
   INIT_M3G_PARM_INFO(&inbandDCIParm);

   // configure MPEG-4 decoderConfigurationInformation transmitted to 3G network
   inbandDCIParm.parameterType = M3G_E_PRM_MPEG4_TX_DCI;
   inbandDCIParm.parmValue.octetString.length = m_DciOctetStrSize;
   // m_DciOctetStrSize datatype of UCHAR precludes buffer overflow
   memcpy(&inbandDCIParm.parmValue.octetString.octet[0], m_DciOctetStr, m_DciOctetStrSize);

   if (M3G_ERROR == m3g_SetParm(m_ControlHandle, &inbandDCIParm))
   {
      LOG_ERROR(m_ControlHandle, "m3g_SetParm() failure\n");
   }
   else
   {
      LOG_ENTRY(m_ControlHandle, "Successful m3g_SetParm(inbandDCI) completion\n");
   }
}

void Endpoint::SetInbandH264DCITxTo3G()
{
   M3G_PARM_INFO inbandDCIParm;
   INIT_M3G_PARM_INFO(&inbandDCIParm);

   // configure H264 decoderConfigurationInformation transmitted to 3G network
   inbandDCIParm.parameterType = M3G_E_PRM_H264_TX_DCI;
   inbandDCIParm.parmValue.octetString.length = m_DciOctetStrSize;
   // m_DciOctetStrSize datatype of UCHAR precludes buffer overflow
   memcpy(&inbandDCIParm.parmValue.octetString.octet[0], m_DciOctetStr, m_DciOctetStrSize);

   if (M3G_ERROR == m3g_SetParm(m_ControlHandle, &inbandDCIParm))
   {
      LOG_ERROR(m_ControlHandle, "m3g_SetParm() failure\n");
   }
   else
   {
      LOG_ENTRY(m_ControlHandle, "Successful m3g_SetParm(inbandDCI) completion\n");
   }
}

void Endpoint::SetInbandDCIRvdFrom3G()
{
   M3G_PARM_INFO inbandDCIParm;
   INIT_M3G_PARM_INFO(&inbandDCIParm);

   if (m_MobileDCIRxdOctetStrLen) {
   // configure MPEG-4 decoderConfigurationInformation transmitted to 3G network
   inbandDCIParm.parameterType = M3G_E_PRM_MPEG4_RX_DCI;
   inbandDCIParm.parmValue.octetString.length = m_MobileDCIRxdOctetStrLen;
   // m_DciOctetStrSize datatype of UCHAR precludes buffer overflow
   memcpy(&inbandDCIParm.parmValue.octetString.octet[0],m_MobileDCIRxdOctetStr, m_MobileDCIRxdOctetStrLen);

   if (M3G_ERROR == m3g_SetParm(m_ControlHandle, &inbandDCIParm))
   {
      LOG_ERROR(m_ControlHandle, "m3g_SetParm() failure\n");
   }
   else
   {
      LOG_ENTRY(m_ControlHandle, "Successful m3g_SetParm(m3g to device inbandDCI) completion for setting \n");
      LOG_ENTRY(m_ControlHandle, "   DCI set is '%s'\n", m_MobileDCIRxdStr.c_str());
   }
   }
   else 
       LOG_ERROR(m_ControlHandle, "Can not set m3g to device inbandDCI, DCI string length is 0\n");
}



#ifdef MONA
void Endpoint::RecordStatistics(long evtDev, M3G_CALL_STATISTICS* pStats)
{
   LOG_ENTRY(evtDev,"\t  callDuration:%d\n", pStats->callDuration);
   LOG_ENTRY(evtDev,"\t  rxMediaCrcErs:%d\n", pStats->rxMediaCrcErrs);
   LOG_ENTRY(evtDev,"\t  rxAudioPacketErrs:%d\n", pStats->rxAudioPacketErrs);
   LOG_ENTRY(evtDev,"\t  rxVideoPacketErrs:%d\n", pStats->rxVideoPacketErrs);
   LOG_ENTRY(evtDev,"\t  rxTotalBytes:%d\n", pStats->rxTotalBytes);
   LOG_ENTRY(evtDev,"\t  rxMediaBytes:%d\n", pStats->rxMediaBytes);
   LOG_ENTRY(evtDev,"\t  rxAudioBytes:%d\n", pStats->rxAudioBytes);
   LOG_ENTRY(evtDev,"\t  rxVideoBytes:%d\n", pStats->rxVideoBytes);
   LOG_ENTRY(evtDev,"\t  rxStuffingBytes:%d\n", pStats->rxStuffingBytes);
   LOG_ENTRY(evtDev,"\t  rxMediaPackets:%d\n", pStats->rxMediaPackets);
   LOG_ENTRY(evtDev,"\t  rxAudioPackets:%d\n", pStats->rxAudioPackets);
   LOG_ENTRY(evtDev,"\t  rxVideoPackets:%d\n", pStats->rxVideoPackets);
   LOG_ENTRY(evtDev,"\t  rxMuxPdus:%d\n", pStats->rxMuxPdus);
   LOG_ENTRY(evtDev,"\t  rxMuxPduBytes:%d\n", pStats->rxMuxPduBytes);
   LOG_ENTRY(evtDev,"\t  txTotalBytes:%d\n", pStats->txTotalBytes);
   LOG_ENTRY(evtDev,"\t  txMediaBytes:%d\n", pStats->txMediaBytes);
   LOG_ENTRY(evtDev,"\t  txAudioBytes:%d\n", pStats->txAudioBytes);
   LOG_ENTRY(evtDev,"\t  txVideoBytes:%d\n", pStats->txVideoBytes);
   LOG_ENTRY(evtDev,"\t  txStuffingBytes:%d\n", pStats->txStuffingBytes);
   LOG_ENTRY(evtDev,"\t  txMediaPackets:%d\n", pStats->txMediaPackets);
   LOG_ENTRY(evtDev,"\t  txAudioPackets:%d\n", pStats->txAudioPackets);
   LOG_ENTRY(evtDev,"\t  txVideoPackets:%d\n", pStats->txVideoPackets);
   LOG_ENTRY(evtDev,"\t  txMuxPdus:%d\n", pStats->txMuxPdus);
   LOG_ENTRY(evtDev,"\t  txMuxPduBytes:%d\n", pStats->txMuxPduBytes);
}
#endif


void Endpoint::SetTxVideoCapType(M3G_E_VIDEO_TYPE txVidCapType)
{ 
   m_TxVideoCapType = txVidCapType;
   if (M3G_E_H263 == txVidCapType)
   {
      strncpy(m_VidTxCapStr, "h263", MAX_MEDIA_STR);
   }
   else if (M3G_E_H264 == txVidCapType)
   {
      strncpy(m_VidTxCapStr, "h264", MAX_MEDIA_STR);
   }
   else
   {
      strncpy(m_VidTxCapStr, "mpeg4", MAX_MEDIA_STR);
   }
}


void Endpoint::SetTxAudioCapType(M3G_E_AUDIO_TYPE txAudCapType)
{ 
   if (M3G_E_GSM_AMR_NB == txAudCapType)
   {
      strncpy(m_AudTxCapStr, "amr", MAX_MEDIA_STR);
   }
   else
   {
      strncpy(m_AudTxCapStr, "g723", MAX_MEDIA_STR);
   }
}

bool Endpoint::IsVidCoderPreferred(E_SEL_VID_CODER VidCoder)
{
   for (int i=0; i < m_NumPreferredMediaCaps; i++)
   {
    if ((VidCoder == VID_CODER_H263) && (H263 == m_PreferredMediaCapsList[i]))
    {
      LOG_ENTRY(m_ControlHandle,"H263 match return true\n");
      return true;
    }
    if ((VidCoder == VID_CODER_MPEG4) &&(MPEG4 == m_PreferredMediaCapsList[i]))
    {
      LOG_ENTRY(m_ControlHandle,"MPEG4 match return true\n");
      return true;
    }
    if ((VidCoder == VID_CODER_H264) &&(H264 == m_PreferredMediaCapsList[i]))
    {
      LOG_ENTRY(m_ControlHandle,"H264 match return true\n");
      return true;
    }
   }
   return false;   
}

bool Endpoint::IsAudCoderPreferred(E_SEL_AUD_CODER AudCoder)
{
for (int i=0; i < m_NumPreferredMediaCaps; i++)
   {
    if ((AudCoder == AUD_CODER_AMR) && (AMR == m_PreferredMediaCapsList[i]))
    {
      LOG_ENTRY(m_ControlHandle,"AMR match return true\n");
      return true;
    }
    if ((AudCoder == AUD_CODER_G723) && (G723 == m_PreferredMediaCapsList[i]))
    {
      LOG_ENTRY(m_ControlHandle,"G723 match return true\n");
      return true;
    }
   }
   return false;
}

const char* BaseEndpoint::GetSelVidCoderStr()
{
   const char *szSelectedVideoCoder[] =
   {
       "none",
       "h263",
       "mpeg4",
       "h263+", 
       "h264"
   };
   return (m_SelectedVideoCoder < VID_CODER_LAST) ? 
      szSelectedVideoCoder[m_SelectedVideoCoder] : "none";
}

const char* BaseEndpoint::GetSelVidResStr()
{
   const char *szSelectedVideoRes[] =
   {
       "none",
       "sqcif",
       "qcif",
       "cif"
   };
   return (m_SelectedVideoResolution < VID_RES_LAST) ? 
      szSelectedVideoRes[m_SelectedVideoResolution] : "none";
}

const char* BaseEndpoint::GetSelVidFpsStr()
{
   switch(m_SelectedVideoFramesPerSec)
   {
     case VIDEO_FRAMESPERSEC_6:
        return "6fps";
     case (static_cast<eVIDEO_FRAMESPERSEC>(0x70032)):
        return "7.5fps";
     case VIDEO_FRAMESPERSEC_10:
        return "10fps";
     case VIDEO_FRAMESPERSEC_15:
        return "15fps";
     case VIDEO_FRAMESPERSEC_25:
        return "25fps";
     case VIDEO_FRAMESPERSEC_30:
        return "30fps";
     default:
        return"--fps";
   }   
}

const char* BaseEndpoint::GetSelAudCoderStr()
{
   const char *szSelectedAudioCoder[] =
   {
       "none",
       "amr",
       "g723",
       "g711u",
       "g711a",
       "g729",
       "pcm"
   };
   return (m_SelectedAudioCoder < AUD_CODER_LAST) ?
      szSelectedAudioCoder[m_SelectedAudioCoder] : "none";
}


#ifdef ISDN_CALL_OUT
void Endpoint::InitiateISDNCall()
{
   GetSigProtocolEndpoint()->MakeCall();
   GetSigProtocolEndpoint()->ChangeState(ISDNEPState::CALLING_STATE);
}
#endif

int UnsignedCharHexArrayToString(unsigned char* origStr, int strLen, char* destStr)
{
    unsigned char temp;
    int i = 0;
    int j = 0;
    for (i = 0; i < strLen; i++) {
        temp = origStr[i] >> 4;
        if (temp <= 9)
            destStr[j] = temp + '0';
        else if (temp >= 0xa && temp <= 0xf)
            destStr[j] = temp + 'a' - 0xa;
        else
            destStr[j] = '-';

        temp = origStr[i] & 0xf;
        j++;

        if (temp <= 9)
            destStr[j] = temp + '0';
        else if (temp >= 0xa && temp <= 0xf)
            destStr[j] = temp + 'a' - 0xa;
        else
            destStr[j] = '-';
        j++;
    }
    return 0;
}
