/**
* @file endpointmgr.h
* @brief Declaration of EndpointMgr class
* @date Feb. 16, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#ifndef  _ENDPOINTMGR_H
#define  _ENDPOINTMGR_H

#include <map>
#include "locker.h"
#include "endpoint.h"
#include "sipendpoint.h"
#include "mmendpoint.h"
#include "isdnendpoint.h"
#include "m3glib.h"
#include "gcip.h"
#include "bridgedcall.h"
#include "bridgedhairp.h"
#include "config.h"
#ifdef USE_RTSP
#   include "bridgedrtsp.h"
#   include "rtspendpoint.h"
#endif
#include "mmconfig.h"

using namespace std;

#define MAX_PARAMETERS 27
#define MAX_EVENTS     8
#define FNAME_SIZE     80
#define DIALOGIC_QCIF_DCI_STR "000001b008000001b58913000001020000012200c488ba9850584121443f"

#include <map>

typedef std::map<char, CMMConfig*> MMCONFIG_MAP;

class Logger;

//*****************************************************************************
// Class: ::EndpointMngr
// Description: Class that allocates, configures, controls, and deallocates 3G
//              and SIP endpoints
//*****************************************************************************
class EndpointMngr
{
public:
   static EndpointMngr *Instance();
   static void Terminate();

   EndpointMngr();
   ~EndpointMngr();
   void SetSipInfoDtmfEnabled(bool enabled) { m_IsSipInfoDtmfEnabled = enabled;}
   bool GetSipInfoDtmfEnabled() { return m_IsSipInfoDtmfEnabled;}
   int AddEndpoint(int numEP, 
                   const char *epType,
                   const char *controlName,
                   const char *streamName,
                   int numPeer,
                   int numLoopbackPeer,
#ifdef ISDN_CALL_OUT
                   const char *mapID,
                   bool AudTranscodeFlag,
                   bool VidTranscodeFlag,
                   const char *dialString);
   void InitiateISDNCall(unsigned int epIndex);
#else
                   const char *mapID,
                   bool isAudXcodeEnabled,
                   bool isVidXcodeEnabled);
#endif
   int InitBoardTracing(unsigned int traceLevel, const char *fileName);
   int InitEPDCI(unsigned int startEP, unsigned int endEP, unsigned char *pDciOctetStr, unsigned char octetStrSize,
                 unsigned int dciMode);
   void InitVendorId(M3G_VENDORID_INFO *pVendorId) {m_VendorId = *pVendorId;}
   void InitEPs();
   Endpoint* GetEPFromIdx(int idx);
   Endpoint* GetEPFromM3GHandle(SRL_DEVICE_HANDLE devH);
   Endpoint* GetEPFromNetworkHandle(SRL_DEVICE_HANDLE devH);
   Endpoint* GetEPFromMapID(char *mapID);
   int GetEPIndexFromMapID(char *mapID);
   SIPEndpoint* GetSIPEPFromIdx(int idx);
   SIPEndpoint* GetSIPEPFromIPMHandle(SRL_DEVICE_HANDLE devH);
   SIPEndpoint* GetSIPEPFromGCHandle(LINEDEV devH);
   MMEndpoint *GetMMEPFromHandle(SRL_DEVICE_HANDLE devH);
   MMEndpoint* GetMMEPFromIdx(int idx);
   void RequestVFU(unsigned int m3gEPIndex);
   int AddParameter(unsigned int parmId, int intVal);

   int GetNumParameters() { return m_NumParameters;}
   void GetParameters( int* numParameters, M3G_PARM_INFO** parameterTbl);
   int AddEvent(int evtId, bool isEnableEvt);
   int InitH223Session(unsigned int startEP, unsigned int endEP, M3G_H223_SESSION *pH223Session);
   int InitH223Caps(unsigned int startEP, unsigned int endEP, LOCAL_H223_CAPS *pLocalH223Caps);
   int InitMediaCaps(unsigned int startEP, unsigned int endEP, int numCaps, MEDIA_CAP *preferredMediaList);
   int InitH223LCParams(unsigned int startEP, unsigned int endEP, LOCAL_H223_LC_PARAMS *pLocalH223LCParams);
   int InitH264Params(unsigned int startEP, unsigned int endEP, LOCAL_H264_PARAMS *pLocalH264Params);
   int InitEPTracing(unsigned int startEP, unsigned int endEP, unsigned int traceLevel, const char *fileName);
#ifdef USE_RTSP
   RTSPEndpoint* GetRTSPEPFromHandle(SRL_DEVICE_HANDLE devH);
   RTSPEndpoint* GetRTSPEPFromIdx(int idx);
#endif

   int SetSIPInfo (unsigned short localSIPPort, const char *originSIPAddr, 
                   const char *destSIPAddr );
   int InitSipVideoCoderList(E_SEL_VID_CODER *preferredSipVideoList);
   int InitSipAudioCoderList(E_SEL_AUD_CODER *preferredSipAudioList);
   E_SEL_VID_CODER GetPreferredSipVideoCoder() {return m_PreferredSipVideoCoderList[0];}
   E_SEL_AUD_CODER GetPreferredSipAudioCoder() {return m_PreferredSipAudioCoderList[0];}
   int SetCmdInfo (const char *connectCmdIPAddr, unsigned short connectCmdRcvPort, unsigned short connectCmdSndPort); 
int SetMMInfo (char dtmf, const char *mmOperation, E_SEL_MM_FILE_TYPE mmFileType, E_SEL_AUD_CODER mmAudCoder,
                             E_SEL_VID_CODER mmVidCoder,E_SEL_VID_RES mmVidRes,
                             eVIDEO_FRAMESPERSEC mmFramesPerSec,eVIDEO_BITRATE mmBitRate,
                             const char *mmPlayAudioFileName, const char *mmPlayVideoFileName,
                             const char *mmRecordAudioFileName, const char *mmRecordVideoFileName);

   int SetNbupInfo (unsigned short associated3GEndpoint, char mbupType, const char *remoteIPAddr,
                    unsigned short remoteNbupPort);
   char *GetConnectCmdIPAddr() {return m_connectCmdIPAddr;}
   unsigned short GetConnectCmdRcvPort() {return m_connectCmdRcvPort;}
   unsigned short GetConnectCmdSndPort() {return m_connectCmdSndPort;}

//char *GetOriginIPAddr() { return m_originIPADDR;}
//char *GetDestIPAddr() { return m_destIPADDR;}
   char *GetOriginSIPAddr() { return m_originSIPADDR;}
   char *GetDestSIPAddr() { return m_destSIPADDR;}
   BridgedCall* GetCallFromIdx(int idx);
   BridgedMedia* GetMediaFromIdx(int idx);
   BridgedHairp* GetHairpFromIdx(int idx);
   BridgedRtsp* GetBridgedRtspFromIdx(int idx);
   bool IsForBoard(void *pIsThisAThis) { return(this == pIsThisAThis);}
   void ProcessEvent(void *pUsrHdl, long  evtType, void *pEvtData, long  evtLen, long  evtDev);
   void SetLogger(Logger *pLogger) { m_pLogger = pLogger;}
   Logger* GetLogger() { return m_pLogger;}
   void GetEPList(int &numEPs, int*&pEPTbl) { numEPs = m_NumEPs; pEPTbl = &m_EPIndexTbl[0];}
   void GetSIPEPList(int &numEPs, int*&pEPTbl) { numEPs = m_NumSIPEPs; pEPTbl = &m_SIPEPIndexTbl[0];}
   void GetMMEPList(int &numEPs, int*&pEPTbl) { numEPs = m_NumMMEPs; pEPTbl = &m_MMEPIndexTbl[0];}
   void GetCallList(int &num, int*&pTbl) { num = m_NumCalls; pTbl = &m_CallIndexTbl[0];}
   void GetMediaList(int &num, int*&pTbl) { num = m_NumMedias; pTbl = &m_MediaIndexTbl[0];}
   void GetHairpList(int &num, int*&pTbl) { num = m_NumHairps; pTbl = &m_HairpIndexTbl[0];}
   void GetRTSPEPList(int &num, int*&pTbl) { num = m_NumRTSPEPs; pTbl = &m_RTSPEPIndexTbl[0];}
   void GetBridgedRtspList(int &num, int*&pTbl) { num = m_NumBridgedRtsp; pTbl = &m_BridgedRtspIndexTbl[0];}
   void GetEPCounts(int &numEPs, int &numSIPEPs, int &numMMEPs, int &numRTSPEPs) {
          numEPs = m_NumEPs; numSIPEPs = m_NumSIPEPs; numMMEPs = m_NumMMEPs; numRTSPEPs = m_NumRTSPEPs;}
   void Check3GConnectionTimers();
   void ProcessUserConnectRequest(int endpointIdx, const char *callInfo);
   void ProcessUserDisconnectRequest(int endpointIdx);
   void CloseM3GBoard();
   void Shutdown();
   void Abort();
   void Cleanup();
   void ResetTermExit();
   void StopM3G();
   bool AllEndpointsAreIdle();
   int ValidateConfiguration();
   const char * GetMMOperation(char dtmf);
   char * GetRTSPOperation() {return m_RtspOperation;}

   const char * GetMMPlayAudioFileName(char dtmf);
   const char * GetMMPlayVideoFileName(char dtmf);
   const char * GetMMRecordAudioFileName(char dtmf);
   const char * GetMMRecordVideoFileName(char dtmf);
   E_SEL_AUD_CODER GetMMAudioCoder(char dtmf);
   E_SEL_VID_CODER GetMMVideoCoder(char dtmf);
   E_SEL_VID_RES GetMMVideoResolution(char dtmf);
   E_SEL_MM_FILE_TYPE GetMMFileType(char dtmf);
   eVIDEO_FRAMESPERSEC GetMMVideoFramesPerSec(char dtmf);
   eVIDEO_BITRATE GetMMVideoBitRate(char dtmf);
   int InitGlobalCall ();
   void StopGlobalCall();

   void OnBoardOpenComplete();
   void OnBoardResetComplete();
   void OnGcSipBoardOpenComplete();

   void InitPortConnectInfoList(DM_PORT_CONNECT_INFO_LIST *pciList, unsigned int unPortType, bool TranscodeFlag);

   static const int AUDIO_PORT;
   static const int VIDEO_PORT;
   static const int NBUP_PORT;
private:
   typedef struct 
   {
      int  event;
      bool isEnableEvt;
   } EVENT_INFO;

   // board level tracing
   void StartBoardTracing();
   void StopBoardTracing();
   void SetVendorId();

   static EndpointMngr*     m_pInstance;
   map<int, Endpoint*>     m_EPContainer;
   map<int, SIPEndpoint*>  m_SIPEPContainer;
   map<int, MMEndpoint*>   m_MMEPContainer;
   map<int, BridgedCall*>  m_CallContainer;
   map<int, BridgedMedia*>  m_MediaContainer;
   map<int, BridgedHairp*>  m_HairpContainer;
   map<int, ISDNEndpoint*>  m_ISDNEPContainer;
#ifdef USE_RTSP
   map<int, RTSPEndpoint*>  m_RTSPEPContainer;
   map<int, BridgedRtsp*>   m_BridgedRtspContainer;
#endif
   Logger                 *m_pLogger;
   bool                    m_IsLoggingEnabled;
   bool                    m_IsSipInfoDtmfEnabled;
   M3G_PARM_INFO           m_ParameterTbl[MAX_PARAMETERS];
   unsigned int  m_NumParameters;
   unsigned int            m_EnableEventMask;
   unsigned int            m_DisableEventMask;
   SRL_DEVICE_HANDLE       m_BoardHandle;
   LINEDEV                 m_GcSipBoardHandle;

   // list of EPs needed for Terminal class:
   int           *m_EPIndexTbl;
   int           m_NumEPs;
   int           *m_SIPEPIndexTbl;
   int           m_NumSIPEPs;
   int           *m_MMEPIndexTbl;
   int           m_NumMMEPs;
   int           *m_CallIndexTbl;
   int           m_NumCalls;
   int           *m_MediaIndexTbl;
   int           m_NumMedias;
   int           *m_HairpIndexTbl;
   int           m_NumHairps;
   int           *m_ISDNEPIndexTbl;
   int           m_NumISDNEPs;
   int           *m_RTSPEPIndexTbl;
   int           m_NumRTSPEPs;
   int           *m_BridgedRtspIndexTbl;
   int           m_NumBridgedRtsp;

   int           m_maxSipTimeSlot;

   // instance mutex
   static CLocker m_InstanceLock;

   // GlobalCall-related
   IPCCLIB_START_DATA   m_libData;
   IP_VIRTBOARD         m_virtBoard;
   CCLIB_START_STRUCT   m_ccLibStart;
   GC_START_STRUCT      m_gcLibStart;

   // IP and SIP addresses for GC and IPM use
   char m_originIPADDR[SIP_ADDR_SIZE];
   char m_destIPADDR[SIP_ADDR_SIZE];
   char m_originSIPADDR[SIP_ADDR_SIZE];
   char m_destSIPADDR[SIP_ADDR_SIZE];
   unsigned short m_localSIPPort;

   // SIP Preferred Coder lists
   E_SEL_VID_CODER m_PreferredSipVideoCoderList[VID_CODER_LAST];
   E_SEL_AUD_CODER m_PreferredSipAudioCoderList[AUD_CODER_LAST];

   // IP address and ports for connection to ISDN gateway
   char m_connectCmdIPAddr[SIP_ADDR_SIZE];
   unsigned short m_connectCmdRcvPort;
   unsigned short m_connectCmdSndPort; 

   // MM endpoint operationt (PLAY/RECORD) and files to do it from/to
   char m_mmOperation[10];
   char m_mmPlayAudioFileName[FNAME_SIZE];
   char m_mmPlayVideoFileName[FNAME_SIZE];
   char m_mmRecordAudioFileName[FNAME_SIZE];
   char m_mmRecordVideoFileName[FNAME_SIZE];

   // MM endpoint Audio/Video coder settings
   E_SEL_AUD_CODER m_mmAudCoder;
   E_SEL_VID_CODER m_mmVidCoder;
   E_SEL_VID_RES   m_mmVidRes;
   eVIDEO_FRAMESPERSEC m_mmVidFramesPerSec;
   eVIDEO_BITRATE m_mmVidBitRate;

   char          m_BrdTraceFileName[MAX_TRACE_FILE_NAME];
   unsigned int  m_BrdTraceLevel;
   M3G_VENDORID_INFO m_VendorId;

   // RTSP endpoint operations
   char m_RtspOperation[10];

   // MM dtmf to operation mappings
   
   MMCONFIG_MAP m_mmOperations;

};

#endif


