/*
* @file h245starting_state.cpp
* @brief Definition of the H245StartingState class
* @date Feb. 16, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#include "h245starting_state.h"
#include "endpointmgr.h"
#include "logger.h"
#include "appevents.h"


// static constant members:
const int H245StartingState::H245_STARTED      = 0x01;
const int H245StartingState::FRAMING_EST       = 0x02;
const int H245StartingState::MSD_CMPLT         = 0x04;
const int H245StartingState::REMOTE_TCS_RCVD   = 0x08;
const int H245StartingState::LOCAL_TCS_ACKD    = 0x10;
const int H245StartingState::H245_READY        = 0x1F;


//*****************************************************************************
// Function: H245StartingState::H245StartingState(Endpoint *pEndpoint)
// Description: Initializing constructor
// Return:  H245StartingState*
// Parameters: Endpoint *pEndpoint 
//*****************************************************************************
H245StartingState::H245StartingState(Endpoint* pEndpoint) :
   EPState(pEndpoint)
{
}

//*****************************************************************************
// Function: H245StartingState::~H245StartingState()
// Description: Destructor
// Return:  none
// Parameters: none 
//*****************************************************************************
H245StartingState::~H245StartingState()
{
}

//*****************************************************************************
// Function: void H245StartingState::ProcessEvent(long evtType, void *pEvtData, long evtLen, long evtDev)
// Description: Process an event
// Return: none
// Parameters: long evtType 
//             void *pEvtData 
//             long evtLen 
//             long evtDev 
//*****************************************************************************
void H245StartingState::ProcessEvent(long  evtType, void *pEvtData, long  evtLen, long  evtDev)
{
   switch ( evtType )
   {
      case M3GEV_START_H245_CMPLT:
         SetStatus(H245_STARTED);
         break;

      case M3GEV_START_H245_FAIL:
         LOG_ERROR(evtDev, "Start H245 failure\n");
         Shutdown();
         break;

      case M3GEV_FRAMING_ESTABLISHED:
         SetStatus(FRAMING_EST);
         CheckH245ReadyToStartChans();
         break;

      case M3GEV_FRAMING_LOST:
         LOG_ERROR(evtDev, "H.223 framing failure\n");
         StopSession();
         break;

      case M3GEV_ENDSESSION_RCVD:
         LOG_ERROR(evtDev, "H245 EndSession rcvd\n");
         StopSession();
         break;

      case M3GEV_MSD_ESTABLISHED:
         {
            SetStatus(MSD_CMPLT);
            const char* msdStr[] =
            {
               "M3G_E_H245_MASTER",
               "M3G_E_H245_SLAVE",
               "M3G_E_H245_IDENTICAL_MSD_NUMBERS"
            };
            M3G_E_H245_MSD_RESULT msdResult = *(reinterpret_cast<M3G_E_H245_MSD_RESULT*>(pEvtData));
            LOG_ENTRY(evtDev,"\t  MSD result:%s\n", msdStr[msdResult]);

            CheckH245ReadyToStartChans();
            break;
         }

      case M3GEV_REMOTE_TCS_RCVD:
         SetStatus(REMOTE_TCS_RCVD);
         CheckH245ReadyToStartChans();
         break;

      case M3GEV_LOCAL_TCS_ACKD:
         SetStatus(LOCAL_TCS_ACKD);
         CheckH245ReadyToStartChans();
         break;

      case M3GEV_H245_MES_EVT:
         {
            M3G_MES* pMesInd = reinterpret_cast<M3G_MES *>(pEvtData);
            LOG_ENTRY(evtDev,"\t  MESEvtType: %d\n", pMesInd->MESEvtType);
            LOG_ENTRY(evtDev,"\t  MESRejectCause: %d\n", pMesInd->MESRejectCause);
            break;
         }

      case M3GEV_H245_MSD_EVT:
         {
            M3G_E_MSD_EVT_TYPE* pMsdInd = reinterpret_cast<M3G_E_MSD_EVT_TYPE *>(pEvtData);
            LOG_ENTRY(evtDev,"\t  M3G_E_MSD_EVT_TYPE: %d\n", int(*pMsdInd));
            break;
         }

      case M3GEV_MSD_FAILED:
         {
            LOG_ERROR(evtDev, "MSD failure.\n");
            StopSession();
            break;
         }

      case M3GEV_RESET_CMPLT:
         ProcessResetComplete();
        break;

      case M3GEV_REMOTE_VENDORID_RCVD:
         {
            M3G_VENDORID_INFO* pRmtVId = reinterpret_cast<M3G_VENDORID_INFO *>(pEvtData);
            LOG_ENTRY(evtDev,"\t  version:0x%x\n", pRmtVId->version);
            if (M3G_E_OID_TYPE == pRmtVId->vendor.oidType)
            {
               LOG_ENTRY(evtDev,"\t  vendor:\n");
               LOG_ENTRY(evtDev,"\t\t oid.length: %d\n", pRmtVId->vendor.oidValue.oid.length);
               LOG_ENTRY(evtDev,"\t\t oid.objectId[]: %x %x %x...\n", 
                         pRmtVId->vendor.oidValue.oid.objectId[0],
                         pRmtVId->vendor.oidValue.oid.objectId[1],
                         pRmtVId->vendor.oidValue.oid.objectId[2]);
            }
            else if (M3G_E_H221_ID_TYPE == pRmtVId->vendor.oidType)
            {
               LOG_ENTRY(evtDev,"\t  vendor:\n");
               LOG_ENTRY(evtDev,"\t\t t35CountryCode: %x\n", 
                         pRmtVId->vendor.oidValue.h221NonStd.t35CountryCode);
               LOG_ENTRY(evtDev,"\t\t t35Extension: %x\n", 
                         pRmtVId->vendor.oidValue.h221NonStd.t35Extension);
               LOG_ENTRY(evtDev,"\t\t manufacturerCode: %x\n", 
                         pRmtVId->vendor.oidValue.h221NonStd.manufacturerCode);
            }
            LOG_ENTRY(evtDev,"\t productNumber[%d]: %x %x %x...\n",
                      pRmtVId->productNumber.length,
                      pRmtVId->productNumber.octet[0],
                      pRmtVId->productNumber.octet[1],
                      pRmtVId->productNumber.octet[2]);
            LOG_ENTRY(evtDev,"\t versionNumber[%d]: %x %x %x...\n",
                      pRmtVId->versionNumber.length,
                      pRmtVId->versionNumber.octet[0],
                      pRmtVId->versionNumber.octet[1],
                      pRmtVId->versionNumber.octet[2]);
            break;
         }

#ifdef MONA
      case M3GEV_SEND_MONA_PREF_MSG:
      case M3GEV_MONA_PREF_MSG_RCVD:
         {
            M3G_MONA_TXRX_MPC_SUPPORT* pMPC = reinterpret_cast<M3G_MONA_TXRX_MPC_SUPPORT*>(pEvtData); 
            LOG_ENTRY(evtDev,"\t rxMPC:0x%x txMPC:0x%x\n", pMPC->rxMPCMask, pMPC->txMPCMask);
            break;
         }

      case M3GEV_TX_MPC_ESTABLISHED:
      case M3GEV_RX_MPC_ESTABLISHED:
         {
            // SKS Next line fixes problem of setting correct states and hence application calls stop media once call disconnected
            m_pEndpoint->Notify(APP_H245_STARTED);

            M3G_MONA_MPC* pMPC = reinterpret_cast<M3G_MONA_MPC*>(pEvtData);
            M3G_LOGICAL_CHANNEL_NUMBER lcn = pMPC->logicalChannelNumber;
            M3G_E_DIRECTION direction = (M3GEV_TX_MPC_ESTABLISHED == evtType) ? M3G_E_TX : M3G_E_RX;

            LOG_ENTRY(evtDev,"\t  version:0x%x\n", pMPC->version);
            LOG_ENTRY(evtDev,"\t  LCN:%d\n",lcn);
            m_pEndpoint->RecordLCMuxParams(evtDev, &pMPC->h223MultiplexParams);

            const char* tStr = (M3G_E_AUDIO_CAPABILITY == pMPC->capabilityType) ? " AUDIO" : " VIDEO";
            LOG_ENTRY(evtDev,"\t  type:%s\n",tStr);

            if (M3G_E_AUDIO_CAPABILITY == pMPC->capabilityType)
            {
               m_pEndpoint->RecordAudioCap(evtDev, &pMPC->mediaCapability.audioCapability);
               ActivateAudioChannel(direction, lcn);

               if (M3GEV_TX_MPC_ESTABLISHED == evtType)
               {
                  m_pEndpoint->SetTxAudioCapType(pMPC->mediaCapability.audioCapability.coderType);
               }
               else if (M3GEV_RX_MPC_ESTABLISHED == evtType)
               {
                  if (pMPC->mediaCapability.audioCapability.coderType == M3G_E_GSM_AMR_NB)
                     LOG_ENTRY(evtDev,"MONA MPC RX 3GSetAudioToamr\n");
               }
            }
            else 
            {
               m_pEndpoint->RecordVideoCap(evtDev, &pMPC->mediaCapability.videoCapability);
               ActivateVideoChannel(direction, lcn);

               if (M3GEV_TX_MPC_ESTABLISHED == evtType)
               {
                  m_pEndpoint->SetTxVideoCapType(pMPC->mediaCapability.videoCapability.coderType);
               }
               else if (M3GEV_RX_MPC_ESTABLISHED == evtType)
               {
                  if (pMPC->mediaCapability.videoCapability.coderType == M3G_E_MPEG4)
                     LOG_ENTRY(evtDev,"MONA MPC RX 3GSetVideoTompeg4\n");
                  else if (pMPC->mediaCapability.videoCapability.coderType == M3G_E_H264)
                     LOG_ENTRY(evtDev,"MONA MPC RX 3GSetVideoToh264\n");
                  else if (pMPC->mediaCapability.videoCapability.coderType == M3G_E_H263)
                     LOG_ENTRY(evtDev,"MONA MPC RX 3GSetVideoToh263\n");
               }
            }
            break;
         }

      case M3GEV_CALL_STATISTICS:
         m_pEndpoint->RecordStatistics(evtDev, reinterpret_cast<M3G_CALL_STATISTICS*>(pEvtData));
         break;
#endif

      // HDX media mode:
      case M3GEV_MODIFY_MEDIA_CMPLT:
         {
            const char *dirStr[] = 
            {
               "M3G_E_IDLE",
               "M3G_E_TX",
               "M3G_E_RX",
               "M3G_E_TXRX"
            };

            M3G_E_DIRECTION direction = *(reinterpret_cast<M3G_E_DIRECTION*>(pEvtData));
            LOG_ENTRY(evtDev,"\t  dir:%s\n",dirStr[direction]);

            // if audio transaction
            if (evtDev == m_pEndpoint->GetAudioHandle())
            {
               switch (direction)
               {
                  case M3G_E_TX:
                     SetStatus(AUDIO_TX_ACTIVE);
                     break;

                  case M3G_E_RX:
                     SetStatus(AUDIO_RX_ACTIVE);
                     break;

                  case M3G_E_TXRX:
                     SetStatus(AUDIO_TX_ACTIVE | AUDIO_RX_ACTIVE);
                     break;

                  default:   // should never happen
                     break;
               }
            }
            else // else video transaction
            {
               switch (direction)
               {
                  case M3G_E_TX:
                     SetStatus(VIDEO_TX_ACTIVE);
                     break;

                  case M3G_E_RX:
                     SetStatus(VIDEO_RX_ACTIVE);
                     break;

                  case M3G_E_TXRX:
                     SetStatus(VIDEO_TX_ACTIVE | VIDEO_RX_ACTIVE);
                     break;

                  default:   // should never happen
                     break;
               }
            }
            // if all media is active, transition to active state
            CheckAllChannelsActive();
            break;
         }

      case M3GEV_MODIFY_MEDIA_FAIL:
         LOG_ERROR(evtDev, "Start media failure\n");
         //TODO:  Determine error handling, i.e. Shutdown();?
        break;

      // FDX only media mode:
      case M3GEV_START_MEDIA_CMPLT:
         // if audio transaction
         if (evtDev == m_pEndpoint->GetAudioHandle())
         {
             SetStatus(AUDIO_TX_ACTIVE | AUDIO_RX_ACTIVE);
         }
         else // else video transaction
         {
             SetStatus(VIDEO_TX_ACTIVE | VIDEO_RX_ACTIVE);
         }
         // if all media is active, transition to active state
         CheckAllChannelsActive();
        break;

      case M3GEV_START_MEDIA_FAIL:
         LOG_ERROR(evtDev, "Start media failure\n");
         //TODO:  Determine error handling, i.e. Shutdown();?
        break;

      default:
         LOG_ERROR(evtDev,"H245StartingState: Unexpected event type: 0x%x\n", evtType);
         break;
   }
}

//*****************************************************************************
// Function: void H245StartingState::ProcessUserPrompt(EPState::E_USER_PROMPT_TYPE eUserPrompt)
// Description: Process a request from the user interface
// Return: void 
// Parameters: EPState::E_USER_PROMPT_TYPE eUserPrompt 
//*****************************************************************************
void  H245StartingState::ProcessUserPrompt(EPState::E_USER_PROMPT_TYPE eUserPrompt)
{
   if ( EPState::USER_DISCONNECT_PROMPT == eUserPrompt )
   {
      StopSession();
   }
   else
   {
      LOG_ERROR(0,"Unexpected user prompt on endpoint[%d] in H245_STARTING state: %d\n",
                m_pEndpoint->GetIndex(), static_cast<int>(eUserPrompt));
   }
}

//*****************************************************************************
// Function: void H245StartingState::Shutdown()
// Description: Process a shutdown request
// Return: void 
// Parameters: none 
//*****************************************************************************
void H245StartingState::Shutdown()
{
  LOG_ENTRY(m_pEndpoint->GetControlHandle(), "Shutdown called in %s state.  No Op\n", GetStateStr());
}


//*****************************************************************************
// Function: void H245StartingState::StopSession()
// Description: Stop the H245 session
// Return: void 
// Parameters: none 
//*****************************************************************************
void H245StartingState::StopSession()
{
   LOG_ENTRY(m_pEndpoint->GetControlHandle(), "Calling Stop_H245 from H245 Starting State\n");
   m_pEndpoint->Stop_H245();
   m_pEndpoint->ChangeState(CLOSING_LCS_STATE);
}


//*****************************************************************************
// Function: void H245StartingState::CheckH245ReadyToStartChans()
// Description: Change state id H245 session has been started
// Return: void 
// Parameters: none 
//*****************************************************************************
void H245StartingState::CheckH245ReadyToStartChans()
{
   LOG_ENTRY(0,"Endpoint[%d] H.245 session starting.  CheckH245ReadyToStartChans: %02X\n",
             m_pEndpoint->GetIndex(), m_StatusBitmask);
   if (IsStatusSet(H245_READY))
   {
      m_pEndpoint->Notify(APP_H245_STARTED);
      m_pEndpoint->ChangeState(OPENING_LCS_STATE);
      m_pEndpoint->GetMatchedCaps();
   }
}

//*****************************************************************************
// Function: void H245StartingState::ProcessResetComplete()
// Description: M3G reset done; set state to inactive to trigger shutdown
// Return: void
// Parameters: none
//*****************************************************************************
void H245StartingState::ProcessResetComplete()
{
      LOG_ENTRY(0,"Endpoint[%d] M3g reset complete.  Triggering shutdown.\n",
                m_pEndpoint->GetIndex());
      LOG_ENTRY(0,"Endpoint[%d] Notify(APP_M3G_ENDPOINT_DISCONNECTED)\n",
                m_pEndpoint->GetIndex());
      m_pEndpoint->Notify(APP_M3G_ENDPOINT_DISCONNECTED);
      m_pEndpoint->ChangeState(H245_INACTIVE_STATE); 
}
