/**
*
* DIALOGIC CONFIDENTIAL
* Copyright � 2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#include <time.h>
#include "gcip.h"
#include "logger.h"
#include "isdnendpoint.h"
#include "endpointmgr.h"
#include "locker.h"
#include "appevents.h"
#include "baseendpoint.h"
#include "gcisdn.h"

//*****************************************************************************
// Function: ISDNEndpoint::ISDNEndpoint(int numEP, const char *ISDNName, const char *IPMName, const char *mapID)
// Description: Initializing constructor
// Return:  ISDNEndpoint*
// Parameters: int numEP 
//             const char *ISDNName 
//             const char *IPMName 
//             const char *mapID 
//*****************************************************************************
#ifdef ISDN_CALL_OUT
ISDNEndpoint::ISDNEndpoint(int numEP, const char *ISDNName, Endpoint *pEndpoint, T_ENDPOINT type, const char *dialString):
#else
ISDNEndpoint::ISDNEndpoint(int numEP, const char *ISDNName, Endpoint *pEndpoint, T_ENDPOINT type):
#endif
   m_Index(numEP),
   m_pCurrentState(NULL),
   m_Endpoint(pEndpoint),   
   m_GCISDNHandle(-1)
{
   char tmpName[64];
   strcpy(tmpName, ISDNName);
   tmpName[0] = 'd';
   tmpName[1] = 't';
   tmpName[2] = 'i';
   SetEndpointType(type);
   if (type == SS7_ENDPOINT)
	   sprintf(m_ISDNName, ":N_%s:P_SS7", tmpName);
   else
	   sprintf(m_ISDNName, ":N_%s:P_ISDN", tmpName);
   strncpy(m_CurrentStateStr, "OPENING        ", MAX_CURRENT_STATE_STR);
   strcpy(m_LastMessageBuff, "");
   m_pISDNOpeningState = new ISDNOpeningState(this);
   m_pISDNWaitCallState = new ISDNWaitCallState(this);
   m_pISDNAcceptedState = new ISDNAcceptedState(this);
   m_pISDNConnectedState = new ISDNConnectedState(this);
   m_pISDNDroppingState = new ISDNDroppingState(this);
   m_pISDNCallingState = new ISDNCallingState(this);
   m_pISDNProceedingState = new ISDNProceedingState(this);
   m_pISDNReleasingState = new ISDNReleasingState(this);
#ifdef ISDN_CALL_OUT
   strncpy(m_DialString, dialString, LINE_BUFFER_SIZE-1);
#endif
}


//*****************************************************************************
// Function: ISDNEndpoint::~ISDNEndpoint()
// Description: Destructor
// Return:  none
// Parameters: none 
//*****************************************************************************
ISDNEndpoint::~ISDNEndpoint()
{
  delete m_pISDNOpeningState;
  delete m_pISDNWaitCallState;
  delete m_pISDNAcceptedState;
  delete m_pISDNConnectedState;
  delete m_pISDNDroppingState;
  delete m_pISDNReleasingState;
}

//*****************************************************************************
// Function: void ISDNEndpoint::ChangeState(ISDNEPState::E_ISDNEPSTATE e_NewState)
// Description: Change the object state
// Return: void 
// Parameters: ISDNEPState::E_ISDNEPSTATE e_NewState 
//*****************************************************************************
void ISDNEndpoint::ChangeState(ISDNEPState::E_ISDNEPSTATE e_NewState)
{
   char oldStateStr[MAX_CURRENT_STATE_STR];
   strcpy(oldStateStr, m_pCurrentState->GetStateStr());

   // reset the current state for the next call:
   m_pCurrentState->Reset();

   switch ( e_NewState )
   {
      case ISDNEPState::OPENING_STATE:
         m_pCurrentState = m_pISDNOpeningState;
         break;

      case ISDNEPState::WAITCALL_STATE:
         m_pCurrentState = m_pISDNWaitCallState;
         break;

      case ISDNEPState::ACCEPTED_STATE:
         m_pCurrentState = m_pISDNAcceptedState;
         break;

      case ISDNEPState::CONNECTED_STATE:
         m_pCurrentState = m_pISDNConnectedState;
         break;

      case ISDNEPState::CALLING_STATE:
         m_pCurrentState = m_pISDNCallingState;
         break;

      case ISDNEPState::PROCEEDING_STATE:
         m_pCurrentState = m_pISDNProceedingState;
         break;

      case ISDNEPState::DROPPING_STATE:
         m_pCurrentState = m_pISDNDroppingState;
         break;

      case ISDNEPState::RELEASING_STATE:
         m_pCurrentState = m_pISDNReleasingState;
         break;

      case ISDNEPState::ALARM_STATE:
         m_pCurrentState = m_pISDNAlarmState;
         break;

       default:
         LOG_ERROR(0,"Unexpected state: %d\n",static_cast<int>(e_NewState));
         break;
   }

   // cache the state string in the endpoint
   strcpy(m_CurrentStateStr,m_pCurrentState->GetStateStr());
   LOG_ENTRY(m_GCISDNHandle, "ISDNEndpoint[%d] State transition: %s ---> %s\n",
		GetIndex(), oldStateStr, m_CurrentStateStr);
}


//*****************************************************************************
// Function: bool ISDNEndpoint::ProcessEvent(METAEVENT metaevent)
// Description: Process an event
// Return: bool 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void ISDNEndpoint::ProcessEvent(METAEVENT metaevent)
{
   m_StateLock.Lock();
   // dispatch event information to appropriate state:
   m_pCurrentState->ProcessEvent(metaevent);
   m_StateLock.Unlock();
}




//*****************************************************************************
// Function: void ISDNEndpoint::WaitCall()
// Description: 
// Return: bool succeeded or not
// Parameters: none 
//*****************************************************************************
bool ISDNEndpoint::WaitCall()
{
  LOG_ENTRY(m_GCISDNHandle, "ISDNEndpoint[%d] WaitCall\n",GetIndex());
  if (gc_WaitCall(GetGCISDNHandle(), NULL, 0, -1, EV_ASYNC) != GC_SUCCESS)
  {
    LogGlobalCallError();
    return false;
  }
  return true;
}

//*****************************************************************************
// Function: void ISDNEndpoint::WaitCall()
// Description: 
// Return: bool succeeded or not
// Parameters: none 
//*****************************************************************************
bool ISDNEndpoint::ResetLineDev()
{
  LOG_ENTRY(m_GCISDNHandle, "ISDNEndpoint[%d] ResetLineDev\n",GetIndex());
  if (gc_ResetLineDev(GetGCISDNHandle(), EV_ASYNC) != GC_SUCCESS)
  {
    LogGlobalCallError();
    return false;
  }
  return true;
}
//*****************************************************************************
// Function: void ISDNEndpoint::SetNetWorkHandle()
// Description: 
// Return: none
// Parameters: none
//*****************************************************************************
void ISDNEndpoint::SetNetWorkHandle()
{
  m_Endpoint->ProcessSetNetworkHndEvent(m_GCISDNHandle);
}
//*****************************************************************************
// Function: void ISDNEndpoint::SendConnected()
// Description: 
// Return: none
// Parameters: connected or not connected
//*****************************************************************************
void ISDNEndpoint::SendConnected(bool connected)
{
  EPState::E_USER_PROMPT_TYPE eUserPrompt;
  LOG_ENTRY(m_GCISDNHandle, "ISDNEndpoint::SendConnected %d\n", (int)connected);
  connected ? eUserPrompt = EPState::USER_CONNECT_PROMPT : eUserPrompt = EPState::USER_DISCONNECT_PROMPT; 
  m_Endpoint->ProcessUserPrompt(eUserPrompt);
}

//*****************************************************************************
// Function: void ISDNEndpoint::DropCall()
// Description: 
// Return: bool succeeded or not
// Parameters: none 
//*****************************************************************************
bool ISDNEndpoint::DropCall()
{
  if ( gc_DropCall(m_CRN, GC_NORMAL_CLEARING, EV_ASYNC) != GC_SUCCESS )
  {
    LOG_ERROR(m_GCISDNHandle, "Error calling gc_DropCall\n");
    LogGlobalCallError();
    return false;
  }
  LOG_ENTRY(m_GCISDNHandle, "Call dropped (gc_DropCall)\n");
  return true;
}
//*****************************************************************************
// Function: void ISDNEndpoint::ReleaseCall()
// Description: 
// Return: bool succeeded or not
// Parameters: none 
//*****************************************************************************
bool ISDNEndpoint::ReleaseCall()
{
  if ( gc_ReleaseCallEx(m_CRN, EV_ASYNC)!= GC_SUCCESS )
  {
    LOG_ERROR(m_GCISDNHandle, "Error calling gc_ReleaseCallEx\n");
    LogGlobalCallError();
    return false;
  }
  LOG_ENTRY(m_GCISDNHandle, "Call dropped (gc_ReleaseCallEx)\n");
  return true;
}


//*****************************************************************************
// Function: void ISDNEndpoint::AnswerCall()
// Description: 
// Return: bool succeeded or not
// Parameters: none 
//*****************************************************************************
bool ISDNEndpoint::AnswerCall(CRN crn)
{
  LOG_ENTRY(m_GCISDNHandle, "ISDNEndpoint[%d] AnswerCall\n",GetIndex());
  m_CRN = crn;
  if (gc_AnswerCall(m_CRN, 0, EV_ASYNC) != GC_SUCCESS)
  {
    LogGlobalCallError();
    return false;
  }
  return true;
}


//*****************************************************************************
// Function: ISDNEPState::E_ISDNEPSTATE ISDNEndpoint::GetState()
// Description: Return the state
// Return: ISDNEPState::E_ISDNEPSTATE 
// Parameters: none 
//*****************************************************************************
ISDNEPState::E_ISDNEPSTATE ISDNEndpoint::GetState()
{
   m_StateLock.Lock();
   ISDNEPState::E_ISDNEPSTATE state = m_pCurrentState->GetState();
   m_StateLock.Unlock();
   return state;
}

//*****************************************************************************
// Function: void ISDNEndpoint::OpenSubDevs()
// Description: Open all associated devices
// Return: void 
// Parameters: none 
//*****************************************************************************
void ISDNEndpoint::OpenSubDevs()
{
   // initialize the endpoint state to opening
   m_pCurrentState = m_pISDNOpeningState;

   // Open the GC ISDN device
   LOG_ENTRY(0, "Opening %s\n", m_ISDNName);
//if(true) // JH  - use to fake a bad open
   if ( gc_OpenEx(&m_GCISDNHandle, (char*)m_ISDNName, EV_ASYNC, (void *)this) != GC_SUCCESS )
   {
      LOG_ERROR(0, "gc_OpenEx(%s) failure\n",m_ISDNName);
      LogGlobalCallError();
   }
   else
   {
      LOG_ENTRY(0, "%s successfully opened\n", m_ISDNName);
   }
}

//*****************************************************************************
// Function: void ISDNEndpoint::LogGlobalCallError()
// Description: Log GC API error
// Return: void 
// Parameters: none 
//*****************************************************************************
void ISDNEndpoint::LogGlobalCallError()
{
   GC_INFO gcInfo;

   if ( gc_ErrorInfo(&gcInfo) != GC_SUCCESS )
   {
      LOG_ERROR(GetGCISDNHandle(), "gc_ErrorInfo failed\n");
   }
   else
   {
      LOG_ERROR(m_GCISDNHandle, "gcValue  : 0x%x\n", gcInfo.gcValue);
      LOG_ERROR(m_GCISDNHandle, "gcMsg    : %s\n", gcInfo.gcMsg);
      LOG_ERROR(m_GCISDNHandle, "ccLibId  : %d\n", gcInfo.ccLibId);
      LOG_ERROR(m_GCISDNHandle, "ccLibName: %s\n", gcInfo.ccLibName);
      LOG_ERROR(m_GCISDNHandle, "ccValue  : 0x%x\n", gcInfo.ccValue);
      LOG_ERROR(m_GCISDNHandle, "ccMsg    : %s\n", gcInfo.ccMsg);
      LOG_ERROR(m_GCISDNHandle, "info     : %s\n", gcInfo.additionalInfo);
   }
}


//*****************************************************************************
// Function: void ISDNEndpoint::CloseSubDevs()
// Description: Close devices assoicated wioth this endpoint
// Return: void 
// Parameters: none 
//*****************************************************************************
void ISDNEndpoint::CloseSubDevs()
{
  if (m_GCISDNHandle == -1) {
    LOG_ENTRY(m_GCISDNHandle, "GC close not needed on device %s\n", m_ISDNName);
  }
  else {
   LOG_ENTRY(m_GCISDNHandle, "Closing GC device %s\n", m_ISDNName);
   if ( gc_Close(m_GCISDNHandle) < 0 )
   {
      LOG_ERROR(m_GCISDNHandle, "gc_Close(%s) failure\n", m_ISDNName);
      LogGlobalCallError();
   }
    else {
      m_GCISDNHandle = -1;
    }
  }

}

//*****************************************************************************
// Function: void ISDNEndpoint::SetLastMessageString(char *pMessageStr)
// Description: Save the last message string
// Return: void 
// Parameters: char *pMessageStr 
//*****************************************************************************
void ISDNEndpoint::SetLastMessageString(char *pMessageStr)
{
   int length = (MAX_LAST_MSG_STRING > strlen(pMessageStr)) ? strlen(pMessageStr) : MAX_LAST_MSG_STRING-1;
   strncpy(&m_LastMessageBuff[0], pMessageStr, length);
   m_LastMessageBuff[length] = '\0';
}


//*****************************************************************************
// Function: char* ISDNEndpoint::GetStateString()
// Description: Format and return a status string
// Return: char* 
// Parameters: none 
//*****************************************************************************
char * ISDNEndpoint::GetStateString()
{
//   cout << "EP:    STATE:    AUDIO:         Tx    VIDEO:        Tx        MISC." << endl;
//   cout << "                 TxLCN: RxLCN:  CAP:  TxLCN: RxLCN: CAP:    CMD/IND:" << endl;
   sprintf(m_StateBuffer, "ISDN%2d: %15s %3d   %3d    %5s   %3d    %3d  %5s      %6s",
           m_Index,
           m_CurrentStateStr,
           0, //GetAudioTxLCN(),
           0, //GetAudioRxLCN(),
           "n/a", //m_AudTxCapStr,
           0, //GetVideoTxLCN(),
           0, //GetVideoRxLCN(),
           "n/a", //m_VidTxCapStr,
           m_LastMessageBuff);
   return m_StateBuffer;
}



//*****************************************************************************
// Function: char* ISDNEndpoint::MakeCall()
// Description: Make and outbound ISDN call using GlobalCall 
// Return: char* 
// Parameters: none 
//*****************************************************************************
void ISDNEndpoint::MakeCall()
{
   char szDialString[] = "8005551212";

#ifdef ISDN_CALL_OUT
   LOG_ENTRY(m_GCISDNHandle, "Making outbound ISDN call to %s\n", m_DialString);
#else
   LOG_ENTRY(m_GCISDNHandle, "Making outbound ISDN/SS7 call (gcMakeCall) to 8005551212\n");
#endif

   SetBearerCap();
#ifdef ISDN_CALL_OUT
   if ( gc_MakeCall(m_GCISDNHandle, &m_CRN, (char *)m_DialString, (GC_MAKECALL_BLKP) NULL,
#else
   if ( gc_MakeCall(m_GCISDNHandle, &m_CRN, szDialString, (GC_MAKECALL_BLKP) NULL,
#endif 
      0, EV_ASYNC ) != GC_SUCCESS )
   {
      LogGlobalCallError();
   }
}



void ISDNEndpoint::SetBearerCap()
{
   GC_IE_BLK   ie;
   IE_BLK      ieblk; 

   ie.gclib = NULL;    // GlobalCall specific portion
   ie.cclib = &ieblk;  // CCLIB specific portion

   ieblk.length  = 10;             // increase length if using both LLC and BC
   ieblk.data[0] = (char)0x04;     // octet 1 IE type i.e. 0x04 = BC / 0x7c = LLC
   ieblk.data[1] = (char)0x03;     // octet 2 length of the IE
   ieblk.data[2] = (char)0x88;     // octet 3 0x80 = speech / 0x88=unrestricted data
   ieblk.data[3] = (char)0x90;     // octet 4 0x90 = 64 kbps packet mode
   ieblk.data[4] = (char)0xA6;     // octet 5 0xA2 = uLaw / 0xA6 = video

   // if not able to place call with LLC setting then try using BC

   ieblk.data[5] = (char)0x7c;     // octet 1 IE type i.e. 0x04 = BC / 0x7c = LLC
   ieblk.data[6] = (char)0x03;     // octet 2 length of the IE
   ieblk.data[7] = (char)0x88;     // octet 3 0x80 = speech  / 0x88=unrestricted data
   ieblk.data[8] = (char)0x90;     // octet 4 0x90 = 64 kbps packet mode
   ieblk.data[9] = (char)0xA6;     // octet 5 0xA2 = uLaw / 0xA6 = video

   if (gc_SetInfoElem(m_GCISDNHandle, &ie) != GC_SUCCESS)
   {
      LOG_ERROR(m_GCISDNHandle, "Setting bearer caps via gc_SetInfoElem( ) failure.\n");
      LogGlobalCallError();
   }
   else
   {
      LOG_ENTRY(m_GCISDNHandle, "Bearer caps successfully set\n");
   }
}
