/**
* @file opening_state.cpp
* @brief Definition of of OpeningState class
* @date Feb. 16, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#include "opening_state.h"
#include "endpointmgr.h"
#include "logger.h"

extern bool g_UserPromptQuit;

// static constant members:
const int OpeningState::NETWORK_OPENED      = 0x0001;
const int OpeningState::NETWORK_UNBLOCKED   = 0x0002;
const int OpeningState::NETWORK_CONFIG_RDY  = 0x0003;
const int OpeningState::NETWORK_CONFIG_SENT = 0x0004;
const int OpeningState::NETWORK_CONFIG_DONE = 0x0008;
const int OpeningState::NETWORK_BIT_MASK    = 0x000F;
const int OpeningState::CONTROL_PORT_OPENED = 0x0010;
const int OpeningState::AUDIO_PORT_OPENED   = 0x0020;
const int OpeningState::VIDEO_PORT_OPENED   = 0x0040;
const int OpeningState::PARMS_SET           = 0x0080;
const int OpeningState::FULLY_OPENED        = 0x00FF;
const int OpeningState::CONTROL_CAPS_RCVD   = 0x0100;
const int OpeningState::AUDIO_CAPS_RCVD     = 0x0200;
const int OpeningState::VIDEO_CAPS_RCVD     = 0x0400;
const int OpeningState::ALL_CAPS_RCVD       = 0x0700;

//*****************************************************************************
// Function: OpeningState::OpeningState(Endpoint *pEndpoint)
// Description: Initializing constructor
// Return:  OpeningState*
// Parameters: Endpoint *pEndpoint 
//*****************************************************************************
OpeningState::OpeningState(Endpoint* pEndpoint) :
   EPState(pEndpoint),
m_NumParametersSet(0)
{
}

//*****************************************************************************
// Function: OpeningState::~OpeningState()
// Description: Destructor
// Return:  none
// Parameters: none 
//*****************************************************************************
OpeningState::~OpeningState()
{
}

//*****************************************************************************
// Function: void OpeningState::ProcessEvent(long evtType, void *pEvtData, long evtLen, long evtDev)
// Description: Process event
// Return: none
// Parameters: long evtType 
//             void *pEvtData 
//             long evtLen 
//             long evtDev 
//*****************************************************************************
void OpeningState::ProcessEvent(long  evtType, void *pEvtData, long  evtLen, long  evtDev)
{
   switch ( evtType )
   {     
      case IPMEV_OPEN:
         // IPM devices in 3G endpoint are used for Nbup
         LOG_ENTRY(evtDev, "IPMEV_OPEN received\n");
         SetStatus(NETWORK_OPENED | NETWORK_UNBLOCKED);
         ProcessDeviceOpen(evtDev);
        break;

      case GCEV_OPENEX:
         LOG_ENTRY(evtDev, "GCEV_OPENEX received\n");
         SetStatus(NETWORK_OPENED | NETWORK_UNBLOCKED);
         LOG_ENTRY(0,"Endpoint[%d] ProcessEvent NETWORK_OPENED|NETWORK_UNBLOCKED (%ld) : %4X\n", m_pEndpoint->GetIndex(), evtDev, m_StatusBitmask);
         ProcessDeviceOpen(evtDev);
         break;

      case GCEV_OPENEX_FAIL:
         LOG_ERROR(evtDev, "GCEV_OPENEX received - GC open failure\n");
         Shutdown();
         break;

      case GCEV_UNBLOCKED:
         LOG_ENTRY(evtDev, "GCEV_UNBLOCKED received\n");
         SetStatus(NETWORK_UNBLOCKED);
         break;

      case GCEV_SETCONFIGDATA:
         LOG_ENTRY(evtDev, "GCEV_SETCONFIGDATA received\n");
         SetStatus(NETWORK_CONFIG_DONE);
         ProcessDeviceOpen(evtDev);
         break;

      case M3GEV_OPEN_CMPLT:
         ProcessDeviceOpen(evtDev);
         break;

      case M3GEV_OPEN_FAIL:
         LOG_ERROR(evtDev, "open failure\n");
         Shutdown();
         break;

      case M3GEV_SET_PARM_CMPLT:
         m_NumParametersSet++;
         if ( m_NumParametersSet == EndpointMngr::Instance()->GetNumParameters() )
         {
            SetStatus(PARMS_SET);
         }
         ProcessIfFullyOpened();
         break;

      case M3GEV_GET_LOCAL_CAPS_CMPLT:
         {
            M3G_CAPS_LIST* pCapsList = reinterpret_cast<M3G_CAPS_LIST*>(pEvtData);

            if (evtDev == m_pEndpoint->GetControlHandle())
            {
               m_pEndpoint->SetLocalH223Caps(pCapsList);
                SetStatus(CONTROL_CAPS_RCVD);
            }
            else if (evtDev == m_pEndpoint->GetAudioHandle())
            {
               m_pEndpoint->SetLocalAudioCaps(pCapsList);
                SetStatus(AUDIO_CAPS_RCVD);
            }
            else // assume video
            {
               m_pEndpoint->SetLocalVideoCaps(pCapsList);
                SetStatus(VIDEO_CAPS_RCVD);
            }

            m_pEndpoint->RecordCapsList(evtDev, pCapsList);

            // if have local H223, audio and video caps, set TCS:
            if (true == IsAllCapsRecvd())
            {
               m_pEndpoint->SetTCS();
            }
            break;
         }

      case M3GEV_GET_LOCAL_CAPS_FAIL:
         LOG_ERROR(evtDev, "local caps failure\n");
         Shutdown();
         break;

      case M3GEV_SET_TCS_CMPLT:
         {
            m_pEndpoint->ConnectTSAndGetPorts();
            m_pEndpoint->ChangeState(CONNECTING_STATE);
            break;
         }

      case M3GEV_SET_TCS_FAIL:
         LOG_ERROR(evtDev, "M3GEV_SET_TCS_FAIL received\n");
         Shutdown();
         break;

      case M3GEV_RESET_CMPLT:
         ProcessResetComplete();
         break;

      case M3GEV_START_TRACE_CMPLT:
         break;  // no-op

      case M3GEV_START_TRACE_FAIL:
         // log the failure but continue as it is not fatal
         LOG_ERROR(evtDev, "failed to start 3G tracing\n");
         break;

      default:
         LOG_ERROR(evtDev,"OpeningState: Unexpected event type: 0x%x\n",evtType);
         break;
   }
}

//*****************************************************************************
// Function: void OpeningState::ProcessUserPrompt(EPState::E_USER_PROMPT_TYPE eUserPrompt)
// Description: Process request from user interface
// Return: void 
// Parameters: EPState::E_USER_PROMPT_TYPE eUserPrompt 
//*****************************************************************************
void  OpeningState::ProcessUserPrompt(EPState::E_USER_PROMPT_TYPE eUserPrompt)
{
   LOG_ERROR(0,"Unexpected user prompt on endpoint[%d] in OPENING state: %d\n",
             m_pEndpoint->GetIndex(), static_cast<int>(eUserPrompt));
}

//*****************************************************************************
// Function: void OpeningState::Shutdown()
// Description: Process shutdown request
// Return: void 
// Parameters: none 
//*****************************************************************************
void OpeningState::Shutdown()
{
  LOG_ENTRY(m_pEndpoint->GetControlHandle(), "Shutdown called in %s state.  Exiting.\n", GetStateStr());
  EndpointMngr::Instance()->ResetTermExit();
}

//*****************************************************************************
// Function: void OpeningState::ProcessDeviceOpen(long evtDev)
// Description: Process device open completion event
// Return: void 
// Parameters: long evtDev 
//*****************************************************************************
void OpeningState::ProcessDeviceOpen(long evtDev)
{
   LOG_ENTRY(m_pEndpoint->GetControlHandle(), "ProcessDeviceOpen - status bit mask 0x%x\n", m_StatusBitmask);
   if ( m_pEndpoint->GetIsLoopbackNotNetwork() )
   {
      SetStatus((NETWORK_OPENED|NETWORK_UNBLOCKED));
   }

   if (evtDev == m_pEndpoint->GetControlHandle())
   {
      SetStatus(CONTROL_PORT_OPENED);
      m_pEndpoint->SetParameters();
   }
   else if (evtDev == m_pEndpoint->GetAudioHandle())
   {
      SetStatus(AUDIO_PORT_OPENED);
   }
   else if ( evtDev == m_pEndpoint->GetVideoHandle() )
   {
      SetStatus(VIDEO_PORT_OPENED);
   }

   if ( true == IsNetworkReadyForConfig() )
   {
      SetStatus(NETWORK_CONFIG_SENT); 
      m_pEndpoint->SetNetworkConfig();
   }
   LOG_ENTRY(evtDev,"Endpoint[%d] ProcessDeviceOpen(%ld) : %4X\n", m_pEndpoint->GetIndex(), evtDev, m_StatusBitmask);
   ProcessIfFullyOpened();
}

//*****************************************************************************
// Function: void OpeningState::ProcessIfFullyOpened()
// Description: do M3G reset if devices are fully opened
// Return: void 
// Parameters: none 
//*****************************************************************************
void OpeningState::ProcessIfFullyOpened()
{
   LOG_ENTRY(m_pEndpoint->GetControlHandle(), "ProcessIfFullyOpened - status bit mask 0x%x\n", m_StatusBitmask);
   if (IsEndpointFullyOpened())
   {
      if (!m_pEndpoint->ReinitDone())
      {
         LOG_ENTRY(0,"Endpoint[%d] sub-devices all open.  Resetting M3G devices\n",
                   m_pEndpoint->GetIndex());
         m_pEndpoint->Reinitialize();
      }
   }
}

//*****************************************************************************
// Function: void OpeningState::ProcessResetComplete()
// Description: M3G reset done; can now get local capabilities 
// Return: void 
// Parameters: none 
//*****************************************************************************
void OpeningState::ProcessResetComplete()
{
      if (g_UserPromptQuit)
      {
        LOG_ENTRY(0,"Endpoint[%d] M3g reset complete as part of shutdown.\n",
                  m_pEndpoint->GetIndex());
        Shutdown();
      }
      else  
      {
         // start tracing
         m_pEndpoint->StartTracing();
        LOG_ENTRY(0,"Endpoint[%d] M3g reset complete.  Querying local capabilities.\n",
                  m_pEndpoint->GetIndex());
        m_pEndpoint->GetLocalCaps();
      }
}


