/**
* @file mediaactive_state.cpp
* @brief Definition of MediaActiveState class
* @date Feb. 16, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#include "rxmediastopped_state.h"
#include "endpointmgr.h"
#include "logger.h"
#include "appevents.h"


// static constant members:
const int RxMediaStoppedState::AUDIO_STOPPED  = 0x01;
const int RxMediaStoppedState::VIDEO_STOPPED  = 0x02;
const int RxMediaStoppedState::MEDIA_STOPPED  = 0x03;
const int RxMediaStoppedState::MEDIA_STOPPING = 0x10;


//*****************************************************************************
// Function: RxMediaStoppedState::RxMediaStoppedState(Endpoint *pEndpoint)
// Description: Initializing constructor
// Return:  RxMediaStoppedState*
// Parameters: Endpoint *pEndpoint 
//*****************************************************************************
RxMediaStoppedState::RxMediaStoppedState(Endpoint* pEndpoint) :
   EPState(pEndpoint)
{
}

//*****************************************************************************
// Function: RxMediaStoppedState::~RxMediaStoppedState()
// Description: Destructor
// Return:  none
// Parameters: none 
//*****************************************************************************
RxMediaStoppedState::~RxMediaStoppedState()
{
}


//*****************************************************************************
// Function: void RxMediaStoppedState::ProcessEvent(long evtType, void *pEvtData, long evtLen, long evtDev)
// Description: Process an event
// Return: none
// Parameters: long evtType 
//             void *pEvtData 
//             long evtLen 
//             long evtDev 
//*****************************************************************************
void RxMediaStoppedState::ProcessEvent(long  evtType, void *pEvtData, long  evtLen, long  evtDev)
{
   switch ( evtType )
   {
      // FDX only media mode:
      case M3GEV_MODIFY_MEDIA_CMPLT:
         {
            const char *dirStr[] = 
            {
               "M3G_E_IDLE",
               "M3G_E_TX",
               "M3G_E_RX",
               "M3G_E_TXRX"
            };

            M3G_E_DIRECTION direction = *(reinterpret_cast<M3G_E_DIRECTION*>(pEvtData));
            LOG_ENTRY(evtDev,"\t  dir:%s\n",dirStr[direction]);

            // if audio transaction
            if (evtDev == m_pEndpoint->GetAudioHandle())
            {
               switch (direction)
               {
                  case M3G_E_TX:
                     SetStatus(AUDIO_TX_ACTIVE);
                     break;

                  case M3G_E_RX:
                     LOG_ERROR(evtDev, "Unexpected RX audio change\n");
                     SetStatus(AUDIO_RX_ACTIVE);
                     break;

                  case M3G_E_TXRX:
                     SetStatus(AUDIO_TX_ACTIVE | AUDIO_RX_ACTIVE);
                     break;

                  default:   // should never happen
                     break;
               }
            }
            else // else video transaction
            {
               switch (direction)
               {
                  case M3G_E_TX:
                     SetStatus(VIDEO_TX_ACTIVE);
                     break;

                  case M3G_E_RX:
                     LOG_ERROR(evtDev, "Unexpected RX video change\n");
                     SetStatus(VIDEO_RX_ACTIVE);
                     break;

                  case M3G_E_TXRX:
                     SetStatus(VIDEO_TX_ACTIVE | VIDEO_RX_ACTIVE);
                     break;

                  default:   // should never happen
                     break;
               }
            }

            // if all media is active, transition to active state
            CheckAllChannelsActive();
            break;
         }

      case M3GEV_MODIFY_MEDIA_FAIL:
         LOG_ERROR(evtDev, "Modify media failure\n");
         break;

      case M3GEV_STOP_MEDIA_CMPLT:
         {
            if (evtDev == m_pEndpoint->GetAudioHandle())
            {
               SetStatus(AUDIO_STOPPED);
            }
            else
            {
               SetStatus(VIDEO_STOPPED);
            }
            // check for multimedia (a/v) both stopped and if so close LCs and transition:
            CheckMediaStopped();
            break;
         }

      case M3GEV_SEND_H245_UII_CMPLT:
         m_pEndpoint->SetLastMessageString("S[UII]");
         break;

      case M3GEV_SEND_H245_MISC_CMD_CMPLT:
         // only support sending fastVideoUpdate in this demo:
         m_pEndpoint->SetLastMessageString("S[FVU]");
         break;

      case M3GEV_REMOTE_CLOSE_LC_RCVD:
         {
            M3G_REMOTE_CLOSED_LC* pCLCInd = reinterpret_cast<M3G_REMOTE_CLOSED_LC*>(pEvtData);
            M3G_LOGICAL_CHANNEL_NUMBER lcn = pCLCInd->logicalChannelNumber;
            LOG_ENTRY(evtDev,"\t  version:%d\n", pCLCInd->version);
            LOG_ENTRY(evtDev,"\t  LCN:%d\n", lcn);
            LOG_ENTRY(evtDev,"\t  reason:%d\n", pCLCInd->reason);
            if ( lcn == m_pEndpoint->GetAudioRxLCN() )
            {
               m_pEndpoint->SetAudioRxLCN(0);
            }
            else if ( lcn == m_pEndpoint->GetAudioTxLCN() )
            {
               m_pEndpoint->SetAudioTxLCN(0);
            }
            else if ( lcn == m_pEndpoint->GetVideoRxLCN() )
            {
               m_pEndpoint->SetVideoRxLCN(0);
            }
            else if ( lcn == m_pEndpoint->GetVideoTxLCN() )
            {
               m_pEndpoint->SetVideoTxLCN(0);
            }
            break;
            // remain in this state assuming remote will soon send H.245 EndSession (handled below)
         }
         break;

      case M3GEV_FRAMING_LOST:
         LOG_ERROR(evtDev, "H.223 framing failure\n");
         StopMedia();
         break;

      case M3GEV_ENDSESSION_RCVD:
         LOG_ERROR(evtDev, "H.245 EndSession rcvd\n");
         StopMedia();
         break;

      case M3GEV_RESET_CMPLT:
         LOG_ENTRY(evtDev, "M3GEV_RESET_CMPLT received\n");
         ProcessResetComplete();
        break;

      default:
         LOG_ERROR(evtDev,"RxMediaStoppedState: Unexpected event type: 0x%x\n", evtType);
         break;
   }
}

//*****************************************************************************
// Function: void RxMediaStoppedState::ProcessUserPrompt(EPState::E_USER_PROMPT_TYPE eUserPrompt)
// Description: Process request from user interface
// Return: void 
// Parameters: EPState::E_USER_PROMPT_TYPE eUserPrompt 
//*****************************************************************************
void  RxMediaStoppedState::ProcessUserPrompt(EPState::E_USER_PROMPT_TYPE eUserPrompt)
{
   if ( EPState::USER_DISCONNECT_PROMPT == eUserPrompt )
   {
      LOG_ENTRY(0,"Endpoint[%d] user disconnect from RxMediaStoppedState.  Stopping media\n",
                m_pEndpoint->GetIndex());
      m_pEndpoint->Notify(APP_M3G_ENDPOINT_DISCONNECT);
      LOG_ENTRY(0,"Endpoint[%d] Notify(APP_M3G_ENDPOINT_DISCONNECT)\n",
                m_pEndpoint->GetIndex());
      StopMedia();
   }
   else
   {
      LOG_ERROR(0,"Unexpected user prompt on endpoint[%d] in RX_MEDIA_STOPPED state: %d\n",
                m_pEndpoint->GetIndex(), static_cast<int>(eUserPrompt));
   }
}


//*****************************************************************************
// Function: void RxMediaStoppedState::Shutdown()
// Description: Process shutdown request
// Return: void 
// Parameters: none 
//*****************************************************************************
void RxMediaStoppedState::Shutdown()
{
  LOG_ENTRY(0,"Endpoint[%d] Shutdown ordered in %s  Stopping media\n",
   m_pEndpoint->GetIndex(), GetStateStr());
   // Stopping M3G media triggers the shutdown process from this state
   m_pEndpoint->StopMedia();
}

//*****************************************************************************
// Function: void RxMediaStoppedState::StopMedia()
// Description: Stop media stream
// Return: void 
// Parameters: none 
//*****************************************************************************
void RxMediaStoppedState::StopMedia()
{
   if (false == IsStatusSet(MEDIA_STOPPING))
   {
      SetStatus(MEDIA_STOPPING);
      m_pEndpoint->StopMedia();
   }
}


//*****************************************************************************
// Function: void RxMediaStoppedState::CheckMediaStopped()
// Description: Change state if media stream is stopped
// Return: void 
// Parameters: none 
//*****************************************************************************
void RxMediaStoppedState::CheckMediaStopped()
{
   if (IsStatusSet(MEDIA_STOPPED))
   {
      m_pEndpoint->ChangeState(CLOSING_LCS_STATE);
      m_pEndpoint->CloseLCs();
   }
}

//*****************************************************************************
// Function: void ClosingiLCsState::ProcessResetComplete()
// Description: M3G reset done; set state to inactive to trigger shutdown
// Return: void
// Parameters: none
//*****************************************************************************
void RxMediaStoppedState::ProcessResetComplete()
{
      LOG_ENTRY(0,"Endpoint[%d] M3g reset complete.  Triggering shutdown.\n",
                m_pEndpoint->GetIndex());
      LOG_ENTRY(0,"Endpoint[%d] Notify(APP_M3G_ENDPOINT_DISCONNECTED)\n",
                m_pEndpoint->GetIndex());
      m_pEndpoint->Notify(APP_M3G_ENDPOINT_DISCONNECTED);
      m_pEndpoint->ChangeState(H245_INACTIVE_STATE); 
}
