/**
* @file sipopening_state.cpp
* @brief Definition of SIPOpeningState class
* @date Feb. 20, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#include "ipmlib.h"
#include "sipopening_state.h"
#include "sipendpoint.h"
#include "endpointmgr.h"
#include "logger.h"

// Temp here
#include "appevents.h"

// static constant members:
const int SIPOpeningState::GC_OPENED      = 0x01;
const int SIPOpeningState::IPM_OPENED     = 0x02;
const int SIPOpeningState::GC_UNBLOCKED     = 0x04;
const int SIPOpeningState::FULLY_OPENED   = 0x07;


//*****************************************************************************
// Function: SIPOpeningState::SIPOpeningState(SIPEndpoint *pSIPEndpoint)
// Description: Initializing constructor
// Return:  SIPOpeningState*
// Parameters: SIPEndpoint *pSIPEndpoint 
//*****************************************************************************
SIPOpeningState::SIPOpeningState(SIPEndpoint* pSIPEndpoint) :
   SIPEPState(pSIPEndpoint)
{
}

//*****************************************************************************
// Function: SIPOpeningState::~SIPOpeningState()
// Description: Destructor
// Return:  none
// Parameters: none 
//*****************************************************************************
SIPOpeningState::~SIPOpeningState()
{
}

//*****************************************************************************
// Function: void SIPOpeningState::ProcessEvent(METAEVENT metaevent)
// Description: Process an event
// Return: none 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void SIPOpeningState::ProcessEvent(METAEVENT metaevent)
{
   switch ( metaevent.evttype )
   {
      case GCEV_OPENEX:
         LOG_ENTRY(metaevent.evtdev, "SIPOpening state: GCEV_OPENEX received\n");
         SetStatus(GC_OPENED);
         ProcessDeviceOpen(metaevent);
         break;
      case GCEV_OPENEX_FAIL:
         LOG_ERROR(metaevent.evtdev, "SIPOpening state: GCEV_OPENEX_FAIL received - GC open failure\n");
         Shutdown();
         break;
      case GCEV_UNBLOCKED:
         LOG_ENTRY(metaevent.evtdev, "SIPOpening state: GCEV_UNBLOCKED received\n");
         SetStatus(GC_UNBLOCKED);
         ProcessDeviceOpen(metaevent);
         break;
      case IPMEV_OPEN:
         LOG_ENTRY(metaevent.evtdev, "SIPOpening state: IPMEV_OPEN received\n");
         SetStatus(IPM_OPENED);
         ProcessDeviceOpen(metaevent);
         break;
      default:
         LOG_ERROR(metaevent.evtdev,"SIPOpening state: Unexpected event type: 0x%x\n",
		   metaevent.evttype);
         break;
   }
}

//*****************************************************************************
// Function: void SIPOpeningState::ProcessUserPrompt(SIPEPState::E_USER_PROMPT_TYPE eUserPrompt)
// Description: Procerss request from user interface
// Return: void 
// Parameters: SIPEPState::E_USER_PROMPT_TYPE eUserPrompt 
//*****************************************************************************
void  SIPOpeningState::ProcessUserPrompt(SIPEPState::E_USER_PROMPT_TYPE eUserPrompt)
{
   LOG_ERROR(0,"Unexpected user prompt on SIP endpoint in OPENING state: %d\n",
             static_cast<int>(eUserPrompt));
}

//*****************************************************************************
// Function: void SIPOpeningState::Shutdown()
// Description: Process shutdown request
// Return: void 
// Parameters: none 
//*****************************************************************************
void SIPOpeningState::Shutdown()
{
  LOG_ENTRY(m_pSIPEndpoint->GetGCSIPHandle(), "Shutdown called in %s state.  No Op\n", GetStateStr());
}

//*****************************************************************************
// Function: void SIPOpeningState::ProcessDeviceOpen(METAEVENT metaevent)
// Description: Process device open completion event
// Return: void 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void SIPOpeningState::ProcessDeviceOpen(METAEVENT metaevent)
{
   LOG_ENTRY(metaevent.evtdev,"ProcessDeviceOpen\n");

   if ( true == IsEndpointFullyOpened() )
   {
      // Half of the SIP endpoints will be set to receive inbound calls and the other
      // half reserved for outbound.
      // Going with even numbered devices (iptB1T2, iptB1T4, etc.) for inbound
      // and odd (iptB1T1, iptB1T3, etc.) for outbound
      if (m_pSIPEndpoint->isSIPInboundDevice())
         m_pSIPEndpoint->WaitForCall();
      m_pSIPEndpoint->ChangeState(RESERVINGMEDIA_STATE);
      m_pSIPEndpoint->GetIPMPortInfo();
      LOG_ENTRY(metaevent.evtdev,"SIPEndpoint -  GC and IPM devices both open, GC Unblocked received, waiting for call on even numbered devices\n");
   }
}

