/**
* @file sipreservingmedia_state.cpp
* @brief Definition of SIPReservingMediaState class
* @date Feb. 20, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#include "ipmlib.h"
#include "sipreservingmedia_state.h"
#include "sipendpoint.h"
#include "endpointmgr.h"
#include "logger.h"

#include "appevents.h"

const int SIPReservingMediaState::TX_PORTS_RCVD = 0x01;
const int SIPReservingMediaState::RX_PORTS_RCVD = 0x02;
const int SIPReservingMediaState::ALL_PORTS_RCVD = 0x03;
const int SIPReservingMediaState::CODEC_RESERVED = 0x04;
const int SIPReservingMediaState::ALL_EVENTS_RCVD = 0x07;

//*****************************************************************************
// Function: SIPReservingMediaState::SIPReservingMediaState(SIPEndpoint *pSIPEndpoint)
// Description: Initializingf constructor
// Return:  SIPReservingMediaState*
// Parameters: SIPEndpoint *pSIPEndpoint 
//*****************************************************************************
SIPReservingMediaState::SIPReservingMediaState(SIPEndpoint* pSIPEndpoint) :
   SIPEPState(pSIPEndpoint)
{
}

//*****************************************************************************
// Function: SIPReservingMediaState::~SIPReservingMediaState()
// Description: Destructor
// Return:  none
// Parameters: none 
//*****************************************************************************
SIPReservingMediaState::~SIPReservingMediaState()
{
}

//*****************************************************************************
// Function: void SIPReservingMediaState::ProcessEvent(METAEVENT metaevent)
// Description: Process an event
// Return: none 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void SIPReservingMediaState::ProcessEvent(METAEVENT metaevent)
{
   switch ( metaevent.evttype )
   {
      // Try getting port info here.  Should be an OK state to do that in?
      case DMEV_GET_TX_PORT_INFO:
         ProcessTxPortInfo(metaevent);
         break;

      case DMEV_GET_RX_PORT_INFO:
         ProcessRxPortInfo(metaevent);
         break;

      case DMEV_RESERVE_RESOURCE :
         LOG_ENTRY(metaevent.evtdev, "SipReservingMediaState: DMEV_RESERVE_RESOURCE received\n");
         ProcessResourceReserved(metaevent);
         break;

      default:
	LOG_ERROR(metaevent.evtdev,"SipReservingMediaState: Unexpected event type: 0x%x\n",
		  metaevent.evttype);
         break;
   }
}

//*****************************************************************************
// Function: void SIPReservingMediaState::Shutdown()
// Description: Process shutdown event
// Return: void 
// Parameters: none 
//*****************************************************************************
void SIPReservingMediaState::Shutdown()
{
  LOG_ENTRY(m_pSIPEndpoint->GetGCSIPHandle(), "Shutdown called in %s state.  No Op\n", GetStateStr());
  //m_pSIPEndpoint->CloseSubDevs();
}

//*****************************************************************************
// Function: void SIPReservingMediaState::ProcessResourceReserved(METAEVENT metaevent)
// Description: Process resource reservation complete event
// Return: void 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void SIPReservingMediaState::ProcessResourceReserved(METAEVENT metaevent)
{
   SetStatus(CODEC_RESERVED);

   if ( AllEventsReceived() )
   {
      ChangeToNextState(metaevent);
   }
}

void SIPReservingMediaState::ProcessRxPortInfo(METAEVENT metaevent)
{
   LOG_ENTRY(metaevent.evtdev, "SipReservingMediaState: DMEV_GET_RX_PORT_INFO received\n");
   DM_PORT_INFO_LIST* pRxPortInfoList = reinterpret_cast<DM_PORT_INFO_LIST*>(metaevent.evtdatap);

   m_pSIPEndpoint->RecordPortInfoList(metaevent, pRxPortInfoList);

   // Save RxPortInfoList
   m_pSIPEndpoint->SetRxPortInfoList(pRxPortInfoList);

   SetStatus(RX_PORTS_RCVD);

   if ( AllPortsReceived() )
   {
      m_pSIPEndpoint->ReserveResources();
   }
}


//*****************************************************************************
// Function: void SIPReservingMediaState::ProcessTxPortInfo(METAEVENT metaevent)
// Description: Process Tx port info received
// Return: void 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void SIPReservingMediaState::ProcessTxPortInfo(METAEVENT metaevent)
{
   LOG_ENTRY(metaevent.evtdev, "SipReservingMediaState: DMEV_GET_TX_PORT_INFO received\n");
   DM_PORT_INFO_LIST* pTxPortInfoList = reinterpret_cast<DM_PORT_INFO_LIST*>(metaevent.evtdatap);

   m_pSIPEndpoint->RecordPortInfoList(metaevent, pTxPortInfoList);

   // Save TxPortInfoList
   m_pSIPEndpoint->SetTxPortInfoList(pTxPortInfoList);


   // Set both masks just to test
   SetStatus(TX_PORTS_RCVD);

   if ( AllPortsReceived() )
   {
      m_pSIPEndpoint->ReserveResources(); ;
   }
}

//*****************************************************************************
// Function: void SIPReservingMediaState::ChangeToNextState(METAEVENT metaevent)
// Description: Change to next state
// Return: void 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void SIPReservingMediaState::ChangeToNextState(METAEVENT metaevent)
{
   // Advance to next state - completing dev_PortConnects between 3G and IPM devices
   LOG_ENTRY(metaevent.evtdev,"SIP endpoint transitioning to PORTCONNECTING state\n");
   m_pSIPEndpoint->ChangeState(PORTCONNECTING_STATE);
   m_pSIPEndpoint->Notify(APP_SIP_ENDPOINT_OPENED);  
}


