/**
* @file terminal.cpp
* @brief Definition of Terminal class
* @date Feb. 20, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#include <sys/timeb.h>
#include "terminal.h"
#include "endpointmgr.h"
#include "endpoint.h"
#include <ncurses.h>

//*****************************************************************************
// Function: Terminal::Terminal()
// Description: Constructor
// Return:  Terminal*
// Parameters: none 
//*****************************************************************************
Terminal::Terminal() :
   m_NumRows(0),
   m_NumColumns(0),
   m_NumEPs(0),
   m_NumSIPEPs(0),
   m_NumMMEPs(0),
   m_NumCalls(0),
   m_NumMedias(0),
   m_NumHairps(0),
   m_NumRTSPEPs(0),
   m_NumBridgedRtsp(0),
   m_PromptType(MAIN_MANUAL_PROMPT)
{
   initscr();
   // Configure stdin for user prompts:
   SetGetchProperties();

   getmaxyx(stdscr, m_NumRows, m_NumColumns);
}

//*****************************************************************************
// Function: Terminal::~Terminal()
// Description: Destructor
// Return:  none
// Parameters: none 
//*****************************************************************************
Terminal::~Terminal()
{
   endwin();
}

//*****************************************************************************
// Function: void Terminal::SetGetchProperties()
// Description: Set getch properties for the terminal
// Return: void 
// Parameters: none 
//*****************************************************************************
void Terminal::SetGetchProperties()
{
   // disable echoing of getch() for the user prompts not to impact the window:
   noecho();
   // disable buffering of getch() for the user prompts:
   cbreak();
   // set getch() to non-blocking mode 
   nodelay(stdscr,TRUE);
   refresh();
}

//*****************************************************************************
// Function: void Terminal::RefreshWindow()
// Description: Refresh terminal windowing`
// Return: void 
// Parameters: none 
//*****************************************************************************
void Terminal::RefreshWindow()
{
   // check to see when the endpoint list is set by the endpoint manager:
   if ( 0 == m_NumEPs )
   {
      EndpointMngr::Instance()->GetEPList(m_NumEPs, m_pEPList);
   }

   if ( 0 == m_NumSIPEPs )
   {
      EndpointMngr::Instance()->GetSIPEPList(m_NumSIPEPs, m_pSIPEPList);
   }

   if ( 0 == m_NumMMEPs )
   {
      EndpointMngr::Instance()->GetMMEPList(m_NumMMEPs, m_pMMEPList);
   }

   if ( 0 == m_NumCalls )
   {
      EndpointMngr::Instance()->GetCallList(m_NumCalls, m_pCallList);
   }

   if ( 0 == m_NumMedias )
   {
      EndpointMngr::Instance()->GetMediaList(m_NumMedias, m_pMediaList);
   }

   if ( 0 == m_NumHairps )
   {
      EndpointMngr::Instance()->GetHairpList(m_NumHairps, m_pHairpList);
   }

   if ( 0 == m_NumRTSPEPs )
   {
      EndpointMngr::Instance()->GetRTSPEPList(m_NumRTSPEPs, m_pRTSPEPList);
   }

   if ( 0 == m_NumBridgedRtsp )
   {
      EndpointMngr::Instance()->GetBridgedRtspList(m_NumBridgedRtsp, m_pBridgedRtspList);
   }

   mvprintw(0,0, "===========================================================================");
   mvprintw(1,0, "                         3G-324M/SIP/MM GATEWAY DEMO                       ");
   mvprintw(2,0, "===========================================================================");
   mvprintw(3,0, "       STATE:                  AUDIO:               VIDEO:           MISC. ");
   mvprintw(4,0, "                        TxLCN: RxLCN:  CAP:  TxLCN: RxLCN:  CAP:   CMD/IND:");
   mvprintw(5,0, "===========================================================================");

   int i;
   for ( i=0;i<m_NumEPs;i++ )
   {
      if ( m_NumRows > (i+7) )
      {
         Endpoint* pEndpoint = EndpointMngr::Instance()->GetEPFromIdx(m_pEPList[i]);
         char *pStateStr = pEndpoint->GetStateString();
         mvprintw(6+i,0, "%s", pStateStr);
      }
   }

   for ( i=0;i<m_NumSIPEPs;i++ )
   {
      if ( m_NumRows > (i+7+m_NumEPs) )
      {
         SIPEndpoint* pSIPEndpoint = EndpointMngr::Instance()->GetSIPEPFromIdx(m_pSIPEPList[i]);
         if ( pSIPEndpoint )
         {
            char *pStateStr = pSIPEndpoint->GetStateString();
            mvprintw(6+i+m_NumEPs,0, "%s", pStateStr);         
         }
         else
         {
            mvprintw(6+i+m_NumEPs,0, "SIP endpoint not found: [%d]=%d", i, m_pSIPEPList[i]);         
         }
      }
   }

   for ( i=0;i<m_NumMMEPs;i++ )
   {
      if ( m_NumRows > (i+7+m_NumEPs+m_NumSIPEPs) )
      {
         MMEndpoint* pMMEndpoint = EndpointMngr::Instance()->GetMMEPFromIdx(m_pMMEPList[i]);
         if ( pMMEndpoint )
         {
            char *pStateStr = pMMEndpoint->GetStateString();
            mvprintw(6+i+m_NumEPs+m_NumSIPEPs,0, "%s", pStateStr);
         }
         else
         {
            mvprintw(6+i+m_NumEPs+m_NumSIPEPs,0, "MM endpoint not found: [%d]=%d", i, m_pMMEPList[i]);         
         }
      }
   }
#ifdef USE_RTSP   
   for ( i=0;i<m_NumRTSPEPs;i++ )
   {
      if ( m_NumRows > (i+7+m_NumEPs+m_NumSIPEPs+m_NumMMEPs) )
      {
         RTSPEndpoint* pRTSPEndpoint = EndpointMngr::Instance()->GetRTSPEPFromIdx(m_pRTSPEPList[i]);
         if ( pRTSPEndpoint )
         {
            char *pStateStr = pRTSPEndpoint->GetStateString();
            mvprintw(6+i+m_NumEPs+m_NumSIPEPs+m_NumMMEPs,0, "%s", pStateStr);
         }
         else
         {
            mvprintw(6+i+m_NumEPs+m_NumSIPEPs+m_NumMMEPs,0, "RTSP endpoint not found: [%d]=%d", i, m_pRTSPEPList[i]);         
         }
      }
   }
#endif

   for ( i=0;i<m_NumCalls;i++ )
   {
      if ( m_NumRows > (i+7+m_NumEPs+m_NumSIPEPs+m_NumMMEPs+m_NumRTSPEPs) )
      {
         BridgedCall* pBridgedCall = EndpointMngr::Instance()->GetCallFromIdx(m_pCallList[i]);
         if ( pBridgedCall )
         {
            char *pStateStr = pBridgedCall->GetStateString();
            mvprintw(6+i+m_NumEPs+m_NumSIPEPs+m_NumMMEPs+m_NumRTSPEPs,0, "%s", pStateStr);         
         }
         else
         {
            mvprintw(6+i+m_NumEPs+m_NumSIPEPs+m_NumMMEPs+m_NumRTSPEPs,0, "Bridged Call not found: [%d]=%d", i, m_pCallList[i]);         
         }
      }
   }

   for ( i=0;i<m_NumMedias;i++ )
   {
      if ( m_NumRows > (i+7+m_NumEPs+m_NumSIPEPs+m_NumMMEPs+m_NumRTSPEPs+m_NumCalls) )
      {
         BridgedMedia* pBridgedMedia = EndpointMngr::Instance()->GetMediaFromIdx(m_pMediaList[i]);
         if ( pBridgedMedia )
         {
            char *pStateStr = pBridgedMedia->GetStateString();
            mvprintw(6+i+m_NumEPs+m_NumSIPEPs+m_NumMMEPs+m_NumRTSPEPs+m_NumCalls,0, "%s", pStateStr);         
         }
         else
         {
            mvprintw(6+i+m_NumEPs+m_NumSIPEPs+m_NumMMEPs+m_NumRTSPEPs+m_NumCalls,0, "Bridged Media not found: [%d]=%d", i, m_pMediaList[i]);         
         }
      }
   }

   for ( i=0;i<m_NumHairps;i++ )
   {
      if ( m_NumRows > (i+7+m_NumEPs+m_NumSIPEPs+m_NumMMEPs+m_NumRTSPEPs+m_NumCalls+m_NumMedias) )
      {
         BridgedHairp* pBridgedHairp = EndpointMngr::Instance()->GetHairpFromIdx(m_pHairpList[i]);
         if ( pBridgedHairp )
         {
            char *pStateStr = pBridgedHairp->GetStateString();
            mvprintw(6+i+m_NumEPs+m_NumSIPEPs+m_NumMMEPs+m_NumRTSPEPs+m_NumCalls+m_NumMedias,0, "%s", pStateStr);         
         }
         else
         {
            mvprintw(6+i+m_NumEPs+m_NumSIPEPs+m_NumMMEPs+m_NumRTSPEPs+m_NumCalls+m_NumMedias,0, "Bridged Hairpin not found: [%d]=%d", 
                     i, m_pHairpList[i]);         
         }
      }
   }
#ifdef USE_RTSP
   for ( i=0;i<m_NumBridgedRtsp;i++ )
   {
      if ( m_NumRows > (i+7+m_NumEPs+m_NumSIPEPs+m_NumMMEPs+m_NumRTSPEPs+m_NumCalls+m_NumMedias+m_NumHairps) )
      {
         BridgedRtsp* pBridgedRtsp = EndpointMngr::Instance()->GetBridgedRtspFromIdx(m_pBridgedRtspList[i]);
         if ( pBridgedRtsp )
         {
            char *pStateStr = pBridgedRtsp->GetStateString();
            mvprintw(6+i+m_NumEPs+m_NumSIPEPs+m_NumMMEPs+m_NumRTSPEPs+m_NumCalls+m_NumMedias+m_NumHairps,0, "%s", pStateStr);         
         }
         else
         {
            mvprintw(6+i+m_NumEPs+m_NumSIPEPs+m_NumMMEPs+m_NumRTSPEPs+m_NumCalls+m_NumMedias+m_NumHairps,0, "Bridged RTSP not found: [%d]=%d", 
                     i, m_pBridgedRtspList[i]);         
         }
      }
   }
#endif

   DisplayPrompt();
}

//*****************************************************************************
// Function: void Terminal::DisplayPrompt()
// Description: Display terminal prompt
// Return: void 
// Parameters: none 
//*****************************************************************************
void Terminal::DisplayPrompt()
{
   // display user prompt:
   int promptRow = ((m_NumEPs+m_NumSIPEPs+m_NumMMEPs+m_NumRTSPEPs+m_NumCalls+m_NumMedias+m_NumBridgedRtsp+7) < (m_NumRows-1)) 
                        ? (m_NumEPs+m_NumSIPEPs+m_NumMMEPs+m_NumRTSPEPs+m_NumCalls+m_NumMedias+m_NumBridgedRtsp+m_NumHairps+7) : (m_NumRows-1);
   mvprintw(promptRow,0, "%s", GetPromptStr());
   refresh();
}

//*****************************************************************************
// Function: void Terminal::UpdatePrompt(E_PROMPT_TYPE promptType)
// Description: Update terminal prompt
// Return: void 
// Parameters: E_PROMPT_TYPE promptType 
//*****************************************************************************
void Terminal::UpdatePrompt(E_PROMPT_TYPE promptType) 
{
   m_PromptType = promptType;
   DisplayPrompt();
}

//*****************************************************************************
// Function: const char* Terminal::GetPromptStr()
// Description: Return the current prompt string
// Return: const char* 
// Parameters: none 
//*****************************************************************************
const char *Terminal::GetPromptStr()
{
   static const char * promptStr[] =
   {
      "PROMPT: (Q)uit                                 ",
      "Shutting down gateway...                       "
   };

   return promptStr[m_PromptType];
}

//*****************************************************************************
// Function: void Terminal::ResizeWindow()
// Description: Resise terminal window
// Return: void 
// Parameters: none 
//*****************************************************************************
void Terminal::ResizeWindow()
{
   getmaxyx(stdscr, m_NumRows, m_NumColumns);
}

