/**********@@@SOFT@@@WARE@@@COPY@@@RIGHT@@@**********************************
* DIALOGIC CONFIDENTIAL
*
* Copyright (C) 2006-2007 Dialogic Corporation. All Rights Reserved.
* The source code contained or described herein and all documents related
* to the source code ("Material") are owned by Dialogic Corporation or its
* suppliers or licensors. Title to the Material remains with Dialogic Corporation
* or its suppliers and licensors. The Material contains trade secrets and
* proprietary and confidential information of Dialogic or its suppliers and
* licensors. The Material is protected by worldwide copyright and trade secret
* laws and treaty provisions. No part of the Material may be used, copied,
* reproduced, modified, published, uploaded, posted, transmitted, distributed,
* or disclosed in any way without Dialogic's prior express written permission.
*
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*
***********************************@@@SOFT@@@WARE@@@COPY@@@RIGHT@@@**********/
//***********************************************************************
//***********************************************************************
// SrlHandle.cpp: implementation of the CSrlHandle class.
//
//////////////////////////////////////////////////////////////////////
#include <fcntl.h>

#include "pdl.h"


//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////
//*****************************************************************************
// Purpose	: 
//    Constructor
// Parameters:	
//    [in] Device type (DEV_IPT or DEV_DTI)
//    [in] Configuration parameters (from configuration file)
//    [in] Optional name. When name is 0, use any available from specified type
// Returns:	
//    none
//*****************************************************************************
CSrlDevice::CSrlDevice(DLG_DEVICE_TYPE devtype,
                       PCommonParams pCommonParams,
                       const char *name )
            : CanLog(glb_pAppLog) {
// Default
   m_pPrivateLog = 0;
   m_srl_state = 0;

   m_srl_handle = INV_SRL_HANDLE;
   m_dx_srl_handle = INV_SRL_HANDLE;
   m_cnf_party_srl_handle = INV_SRL_HANDLE;
   m_ntwk_srl_handle = INV_SRL_HANDLE;

   m_dx_name = 0;
   file_index = 0;
   m_type = devtype;
   m_pCommonParams = pCommonParams;
   if ( glb_pDetectCfg->Reserve(m_type,&name) ) {
        SetName(name);
        SetSrlState(SRLSTATE_RESERVED);
   }
   m_pStateMachine = 0;

   memset (&m_iott,0,sizeof(m_iott));
   m_iott.io_fhandle = INV_FILE_HANDLE;  // invalid handle
   SetDfltTimer(NTWK_TMO);
   m_pCnfBoard = 0;
   separate_ntwk_open = false;	// device opened via gc_Open
 return;
} // End of Constructor()
//*****************************************************************************
// Purpose	: 
//    Destructor
// Parameters:	
//    none
// Returns:	
//    none
//*****************************************************************************
CSrlDevice::~CSrlDevice(){   
   if ( IsSrlState(SRLSTATE_RESERVED, true) ) {
        glb_pDetectCfg->UnReserve(m_type, GetName());
   }
   delete m_pStateMachine;

   // Delete private log
   ClosePrivateLog();
} // End of Destructor()
//----------------------------------------------------------------

//*****************************************************************************
// Purpose	: 
//    Open voice device associated with this object
// Parameters:	
//    none
// Returns:	
//    bool (true = success)
//*****************************************************************************
bool CSrlDevice::open_dx(){
  if (!IS_VALID_HANDLE(m_dx_srl_handle) ) {
      if (glb_pDetectCfg->Reserve(DEV_DX, &m_dx_name)){
          SetSrlState(SRLSTATE_VOICERESERVED);
          m_dx_srl_handle = dx_open(m_dx_name,0);
          LOG( RCHANDLE(m_dx_srl_handle), GetName(),
               "0x%x = dx_open(%s)",
                m_dx_srl_handle, m_dx_name);
          if (IS_VALID_HANDLE(m_dx_srl_handle )) {
              SetSrlState(SRLSTATE_VOICEOPENED);
              return true;
          } 
      } else {
          LOG(LOG_ERR2, GetName(), "dx_open:: no more devices left");
      }
  }else {
      LOG(LOG_WARNING, GetName(), "dx_open:: (%s) device is already opened",m_dx_name);
  }
  return false;
} // End of open_dx()


//*****************************************************************************
// Purpose	: 
//    setup voice device to receive cst events
// Parameters:	
//    none
// Returns:	
//    bool (true = success)
//*****************************************************************************
bool CSrlDevice::set_dx_cst(int events){
    // Enable CST events (get digits)
    int rc = dx_setevtmsk(m_dx_srl_handle,  events); // dm_digits, etc
        LOG( RC(rc), GetName(),
             "%d = dx_setevtmsk(0x%x,  events = %d)",
                  rc,m_dx_srl_handle, events);
        bool brc = ( rc != AT_FAILURE);
        if (!brc) {
             process_dx_error(m_dx_srl_handle, GetName());
        }
  return brc;
} // End of open_dx()


//*****************************************************************************
// Purpose	: 
//    Close voice device associated with this object
// Parameters:	
//    none
// Returns:	
//    bool (true = success)
//*****************************************************************************
bool CSrlDevice::close_dx(){
  int rc = 0;
  if (IS_VALID_HANDLE(m_dx_srl_handle) ) {
      rc = dx_close(m_dx_srl_handle);
             LOG( RC(rc), GetName(),
                  "%d = dx_close(0x%x) [%s]",
                  rc, m_dx_srl_handle, m_dx_name);
      m_dx_srl_handle = INV_SRL_HANDLE;
      SetSrlState(SRLSTATE_VOICEOPENED, RESET_SRL_STATE);
      glb_pDetectCfg->UnReserve(DEV_DX,m_dx_name);
      SetSrlState(SRLSTATE_VOICERESERVED, RESET_SRL_STATE);
  }
 return (rc != AT_FAILURE);
} // End of close_dx()

//*****************************************************************************
// Purpose	: 
//    Create private log for this device and arrange LOG to refer this log 
// Parameters:	
//    [in] verbosity    
// Returns:	
//    bool true = success
//*****************************************************************************
bool CSrlDevice::OpenPrivateLog(MSG_TYPE verbosity){
bool brc = true;
  if ( verbosity < LOG_NONE ) {
      char name[_MAX_PATH];
      snprintf(name, sizeof(name), "%s.log", GetName());
      m_pPrivateLog = new CFileLog();
      if (! m_pPrivateLog->InitLog(name, m_pCommonParams->m_maxlogsize)  ){
          LOG( LOG_ERR2, GetName(), 
                         "Error creating log object(%s,%d)",
                         name, m_pCommonParams->m_maxlogsize  );
          brc = false;
      } else {
           // Set new log object and chain original log  
         m_pPrivateLog->SetVerbosity(verbosity);
         m_pPrivateLog->DefineChainLog(GetLog());
         SetLog(m_pPrivateLog);
      } // if private log
  }
  return brc;
} // End of OpenPrivateLog()

//*****************************************************************************
// Purpose	: 
//    Close previously created private log and arrange LOG to refer to global log
// Parameters:	
//    none
// Returns:	
//    bool true = success
//*****************************************************************************
bool CSrlDevice::ClosePrivateLog(){
  if ( m_pPrivateLog){
       CGenLog *pLog = m_pPrivateLog->GetChainedLog();
       m_pPrivateLog->UnlinkChainedLog();
       delete m_pPrivateLog;
       m_pPrivateLog=0;
       SetLog(pLog);
  }
  return true;
} // End of ClosePrivateLog()

//*****************************************************************************
// Purpose	: 
//    Forward to StateMachine::FindInStates
// Parameters:	
//    [in] event 
//    [out] holder to return line in state table
// Returns:	
//    bool true = found
//    false = not found
//*****************************************************************************
bool CSrlDevice::FindInStates(int event, PEVENT_STRU *ppNewEventLine){
  if (m_pStateMachine){
      return m_pStateMachine->FindInStates(GetCurrentState(), event, ppNewEventLine);
  }
  return false;  
} // End of FindInStates()

//*****************************************************************************
// Purpose	: 
//    Forward to StateMachine::FindInActions
// Parameters:	
//    [in] action
//    [out] holder to return line in action table
// Returns:	
//    bool = found
//    false = not found
//*****************************************************************************
bool CSrlDevice::FindInActions(int action, PACTION_STRU *ppNewActionLine){
  if (m_pStateMachine){
      return m_pStateMachine->FindInActions(GetCurrentState(), action, ppNewActionLine);
  }
  return false;  
} // End of FindInActions()

//*****************************************************************************
// Purpose	: 
//    Advance state machine to specified stare
// Parameters:	
//    [in] state
// Returns:	
//    none
//*****************************************************************************
void CSrlDevice::SetCurrentState(int state){

    if ( m_pStateMachine){
        if( S_FINAL == GetCurrentState()){
            return;
        } 
        if (    ( state != S_SAME ) 
             && ( S_FINAL != GetCurrentState() ) ){
             m_pStateMachine->SetCurrentState(state);
             LOG( LOG_DBG, GetName(),
                  "Current state advanced to %d %s",
                  state, state_name(state));
             if ( IsInRelaxedState() ) {
                  StopTimer();
                  LOG(LOG_DBG,GetName(),"***Stop timer");

             }else {
                  StartTimer(); 
                  LOG(LOG_DBG,GetName(),"***Start timer for %d ms", GetDfltTimer());
             } 
        }else{
            LOG( LOG_DBG, GetName(),
                 "Current state not changed: %d %s",
                 m_pStateMachine->GetCurrentState(),
                 state_name( m_pStateMachine->GetCurrentState()));
        } // else - not advanced

        if( S_FINAL == GetCurrentState()){
           LOG( LOG_APP, GetName(), "***Final state reached");
        }
   } // if StateMachine exist

} // End SetCurrentState()

//-------------------------------------------------------------
//  Timer
//-------------------------------------------------------------
//*****************************************************************************
// Purpose	: 
//    Signal TMO if timer is expired
// Parameters:	
//    none
// Returns:	
//    true if timer has expired
//    false if timer is off or if it is not expired yet
//*****************************************************************************
bool CSrlDevice::CheckTmo(){
    bool brc =  CheckTimer();
    if ( brc ){
         PutEvent(USREV_TIMEOUT, GetName());
         DumpMissingEvents();
         LOG(LOG_DBG,GetName(),"***StopTimer (Timer expired)");
    }
  return brc;
}// End CheckTmo()

//*****************************************************************************
// Purpose	: 
//    indicate missing line in table actions ( log failure )
// Parameters:	
//    [in] action
// Returns:	
//    none
//*****************************************************************************
//-----------------------------------------------------------------------------
void CSrlDevice::action_table_error(int action){
   LOG( LOG_ERR2, GetName(),
        "Cannot find action_table entry for action %d %s, Current state is %d %s",
        action, action_name(action),
        GetCurrentState(), state_name(GetCurrentState()) );
  return;
} // End of action_table_error()

//*****************************************************************************
// Purpose	: 
//    Execute next step of state machine
// Parameters:	
//    [in] event
// Returns:	
//    true  = success
//    false = error
//*****************************************************************************
bool CSrlDevice::AdvanceStateMachine(int event){
bool brc = true;

// Normal flow;
  PEVENT_STRU pEventLine;

  if ( FindInStates(event, &pEventLine ) ){
       int action;
       PACTION_STRU pNewAction;
       action = pEventLine->action;

       if ( ! FindInActions(action, &pNewAction) ) {
              action_table_error(action);
              brc =  false;
       }else {
           // Action line found
           // Not waiting for event until action is complete
           SetSrlState(SRLSTATE_WAITEVENT, RESET_SRL_STATE );
           brc = Execute(action);
           if (!brc) {
               action = pNewAction->action_if_fail;
               if ( ! FindInActions(action, &pNewAction) ) {
                      action_table_error(action);
                      brc =  false;
               }else {
                      brc = Execute(action);
                      if (!brc) {
                         // Even recover failed
                          LOG( LOG_ERR2, GetName(), "Recover action failed, exiting");
                          brc = false;
                      } 
               } 
           }  
           if (brc) {
               SetCurrentState(pNewAction->state_if_pass);
           }
       }
   } 
   else {
      // Not found in event table
      // This is possible, if notification event such as GCEV_PROCEEDING is received
      LOG( LOG_WARNING, GetName(), "Can't handle this event in state  (%d %s)",
           GetCurrentState(), state_name(GetCurrentState()) );
   }

   // Indicate waiting for event
   if ( ! IsInRelaxedState() ) {
        SetSrlState(SRLSTATE_WAITEVENT, SET_SRL_STATE);
   }
 return brc;
} // End of AdvanceStateMachine()

//*****************************************************************************
// Purpose	: 
//    Handle events
// Parameters:	
//   [in] event
//   [in] event data
//   [in] data length
//   [in] METAEVENT (not used)
// Returns:	
//    none
//*****************************************************************************
void CSrlDevice::HandleEvent(int event,
                             void *evtdata, int evtlen,
                             METAEVENT *metaeventp){
 UNREFERENCED_PARAMETER(metaeventp);
 UNREFERENCED_PARAMETER(evtlen);

 LOG (LOG_DBG, GetName(), 
               "Current state is %d %s", 
                GetCurrentState(), state_name(GetCurrentState()));

    switch (event) {
        case GCEV_BLOCKED:
             SetSrlState(SRLSTATE_BLOCKED,SET_SRL_STATE);
             break;

        case GCEV_UNBLOCKED:
             SetSrlState(SRLSTATE_BLOCKED,RESET_SRL_STATE);
             break;

            
        case USREV_EXIT_REQUEST:
             SetSrlState(SRLSTATE_REQUESTED_STOP);
             break;

        case TDX_CST:
            {
              DX_CST *cstp = (DX_CST *)evtdata;
              dx_dump(cstp);
            } 
            break;

        default: 
             break;
    } // switch (event)

    if (evt_isnotification(event) ){
        // notification - dump info
        dump_notification_data(event, evtdata);
    }

    if (!AdvanceStateMachine(event) ){
        LOG ( LOG_WARNING, GetName(), 
              "Did not advance: from state %d %s",
               GetCurrentState(), state_name(GetCurrentState()));
    }

 return ;
} // End of HandleEvent()

//*****************************************************************************
// Purpose	: 
//    Indicate state machine error
// Parameters:	
//    None
// Returns:	
//    none
//*****************************************************************************
bool CSrlDevice::StateMachineError(){
  LOG( LOG_DBG, GetName(),
       "Cannot handle event in current state %d %s",
       GetCurrentState(), state_name(GetCurrentState()));

  return false;
} // End of StateMachineError()

//*****************************************************************************
// Purpose	: 
//    Request state machine to exit gracefully
// Parameters:	
//    None
// Returns:	
//    none
//*****************************************************************************
bool CSrlDevice::ExitRequest(){
     bool brc = PutEvent(USREV_EXIT_REQUEST, GetName());
 return brc;
} // End of ExitRequest()


//*****************************************************************************
// Purpose	: 
//    Request state machine to drop the call gracefully and proceed with other calls
// Parameters:	
//    None
// Returns:	
//    none
//*****************************************************************************
bool CSrlDevice::DropRequest(){
     bool brc = PutEvent(USREV_CNF_TIMEOUT, GetName());
 return brc;
} // End of DropRequest()


//*****************************************************************************
// Purpose	: 
//    Send event to this object
// Parameters:	
//    [in] event code
//    [in] sender ( string )
//    [in] optional data length
//    [in] optional event data
// Returns:	
//    true = success
//*****************************************************************************
bool CSrlDevice::PutEvent(int event,
                          const char *sender,
                          long len, void *datap){
 int rc = AT_FAILURE;
  if (IS_VALID_HANDLE(m_srl_handle) ){
      rc = sr_putevtUserContext(m_srl_handle, event, len, &datap, 0, this);
      const char *evname;
      evt_getname(event,&evname);
      LOG( LOG_DBG, GetName(), 
           "%d = [from %s] sr_putevtUserContext(0x%x, %s, %d, 0x%x, 0x%x(this %s)",
           rc, sender,
           m_srl_handle, evname, len, datap, this, GetName());
  }
  return 0 == rc;
} // End of PutEvent()

//*****************************************************************************
// Purpose	: 
//    Dump all events that can possibly advance state machine from the current state  
// Parameters:	
//    none
// Returns:	
//    none
//*****************************************************************************
void CSrlDevice::DumpMissingEvents(){

     LOG( LOG_ERR1, GetName(), "Timed out waiting for event(s):");
     int search_handle = BEGIN_SEARCH;
     int missing;   
     while (m_pStateMachine->EnumMissingEvents(&missing, &search_handle) ){
          if (     ! evt_isuser(missing) 
                &&   evt_issignificant(missing) ) {
              const char *ev_name;
              evt_getname(missing, &ev_name);
               LOG( LOG_ERR1, GetName(),
                    "  Event 0x%x %s", missing, ev_name);
          }
     }
} // End of DumpMissingEvents()

//----------------------------------------------------------------
//  Voice
//----------------------------------------------------------------
//*****************************************************************************
// Purpose	: 
//    Play a file
// Parameters:	
//    [in] file name
// Returns:	
//    true = success
//*****************************************************************************
bool CSrlDevice::PlayFile(const char *filename){
int rc;
   SetDfltTimer(VOICE_TMO);

// Termination condition: any digit
 DV_TPT      tpt[1];
    dx_clrtpt(tpt,1);
	tpt[0].tp_type   = IO_EOT;
	tpt[0].tp_termno = DX_MAXDTMF;
	tpt[0].tp_length = 1;
	tpt[0].tp_flags  = TF_MAXDTMF;

// Remove any existing digits
// otherwise, play will not start at all
	 ClearDigits();

// init IOTT block
// This is class member because we need to close file later
//   after TDX_PLAY or TDX_ERROR event is received
    memset (&m_iott,0,sizeof(m_iott));

	m_iott.io_fhandle  = dx_fileopen(filename, O_RDONLY|O_BINARY);

	m_iott.io_type = IO_DEV|IO_EOT;
	m_iott.io_bufp = 0;
	m_iott.io_offset = 0;
	m_iott.io_length = -1;

    if(m_iott.io_fhandle == INV_FILE_HANDLE) {
        LOG( LOG_ERR2, GetName(), "Cannot open file %s",filename);
        return false;
    }


    // These are valid for our sample play files
  DX_XPB      xpb;
	xpb.wFileFormat = FILE_FORMAT_VOX;
	xpb.wDataFormat = DATA_FORMAT_MULAW;
	xpb.nSamplesPerSec = DRT_8KHZ;
	xpb.wBitsPerSample = 8;

     rc = dx_playiottdata (m_dx_srl_handle, &m_iott, tpt, &xpb, EV_ASYNC);

      LOG( RC(rc), GetName(),
           "%d = dx_playiottdata(hndl=0x%x, iott(%s), tpt, xpb, EV_ASYNC)",
           rc, m_dx_srl_handle,filename );

      if ( rc == AT_FAILURE){
           process_dx_error(m_dx_srl_handle, GetName());
           return false;                             
      }

    return true;
} // End of PlayFile()

//*****************************************************************************
// Purpose	: 
//    Record a file
// Parameters:	none
// Returns:	
//    true = success
//*****************************************************************************
bool CSrlDevice::RecFile(){
int rc;
   SetDfltTimer(VOICE_TMO);

// Termination condition: any digit
 DV_TPT      tpt[1];
    dx_clrtpt(tpt,1);
	tpt[0].tp_type   = IO_EOT;
	tpt[0].tp_termno = DX_MAXDTMF;
	tpt[0].tp_length = 1;
	tpt[0].tp_flags  = TF_MAXDTMF;

// Remove any existing digits
// otherwise, play will not start at all
	 ClearDigits();

// init IOTT block
// This is class member because we need to close file later
//   after TDX_PLAY or TDX_ERROR event is received
    memset (&m_iott,0,sizeof(m_iott));

    char filename[256];
    snprintf(filename,sizeof(filename),"%s_%d.pcm",GetName(),file_index++);
	m_iott.io_fhandle  = dx_fileopen(filename, O_RDWR|O_TRUNC|O_CREAT|O_BINARY, 0666);

	m_iott.io_type = IO_DEV|IO_EOT;
	m_iott.io_bufp = 0;
	m_iott.io_offset = 0;
	m_iott.io_length = -1;

    if(m_iott.io_fhandle == INV_FILE_HANDLE) {
        LOG( LOG_ERR2, GetName(), "Cannot open file %s",filename);
        return false;
    }

  DX_XPB      xpb;
//	xpb.wFileFormat = FILE_FORMAT_WAVE;
    xpb.wFileFormat = FILE_FORMAT_VOX;
	xpb.wDataFormat = DATA_FORMAT_MULAW;
	xpb.nSamplesPerSec = DRT_8KHZ;
	xpb.wBitsPerSample = 8;

    rc = dx_reciottdata (m_dx_srl_handle, &m_iott, tpt, &xpb, EV_ASYNC);

      LOG( RC(rc), GetName(),
           "%d = dx_reciottdata(hndl=0x%x, iott(%s), tpt, xpb, EV_ASYNC)",
           rc, m_dx_srl_handle,filename );

      if ( rc == AT_FAILURE){
           process_dx_error(m_dx_srl_handle, GetName());
           return false;                             
      }

    return true;
} // End of RecFile()

//*****************************************************************************
// Purpose	: 
//    Get Digits (conference pass word)
// Parameters:	
//    none
// Returns:	
//    true = success
//*****************************************************************************
bool CSrlDevice::GetDigits(){
int rc;
unsigned short tmo = 100;    // in 1/10 sec, 10 seconds

  set_dx_cst(DM_DIGITS | DM_SILON | DM_SILOFF);
  SetDfltTimer(6*tmo*1000);  // 6 very slow digits

// Termination condition: any '#', tmo 10 sec or interdigit tmo 5 sec
 DV_TPT      tpt[2];
    dx_clrtpt(tpt,2);
	tpt[0].tp_type   = IO_CONT;
	tpt[0].tp_termno = DX_DIGMASK;
	tpt[0].tp_length = DM_P;	// #
	tpt[0].tp_flags  = TF_DIGMASK;
		
	tpt[1].tp_type   = IO_EOT;
	tpt[1].tp_termno = DX_IDDTIME;
	tpt[1].tp_length = tmo;
	tpt[1].tp_flags  = TF_IDDTIME;

    m_digits.dg_value[0] = 0; //clean old content

  // get it
  rc = dx_getdig (m_dx_srl_handle, tpt, &m_digits, EV_ASYNC);

      LOG( RC(rc), GetName(),
           "%d = dx_getdig(hndl=0x%x, tpt, m_digits, EV_ASYNC)",
           rc, m_dx_srl_handle);

      if ( rc == AT_FAILURE){
           process_dx_error(m_dx_srl_handle, GetName());
           return false;                             
      }

    return true;
} // End of GetDigits()

//*****************************************************************************
// Purpose	: 
//    Play short beep
// Parameters:	
//    [in]beep type
// Returns:	
//    true = success
//*****************************************************************************
bool CSrlDevice::DxBeep(BEEP_TYPE beep){

TN_GENCAD ToneGenCad ;
memset (&ToneGenCad, 0, sizeof(ToneGenCad));
const char *tname = "";
 DV_TPT      tpt[1];
    dx_clrtpt(tpt,1);
	tpt[0].tp_type   = IO_EOT;
	tpt[0].tp_termno = DX_DIGMASK;
	tpt[0].tp_length = DM_P;	// #
	tpt[0].tp_flags  = TF_DIGMASK;

    switch (beep){
        case BEEP_OUT:
             tname = "beep_out";
             ToneGenCad.cycles = 1;
             ToneGenCad.numsegs = 1;
             ToneGenCad.offtime[0] = 10;
             ToneGenCad.offtime[1] = 10;
             dx_bldtngen( &(ToneGenCad.tone[0]), 640, 0,-7,0,10); 
             break;

        case BEEP_IN:
             tname = "beep_in";
             ToneGenCad.cycles = 1;
             ToneGenCad.numsegs = 1;
             ToneGenCad.offtime[0] = 10;
             ToneGenCad.offtime[1] = 10;
             dx_bldtngen( &(ToneGenCad.tone[0]), 440, 0,-7,0,10); 
             dx_bldtngen( &(ToneGenCad.tone[1]), 440, 0,-7,0,10); 
             break;

        case BEEP_BYE:
             tname = "beep_bye";
             ToneGenCad.cycles = 3;
             ToneGenCad.numsegs = 1;
             ToneGenCad.offtime[0] = 10;
             ToneGenCad.offtime[1] = 10;
             dx_bldtngen( &(ToneGenCad.tone[0]), 440, 0,-7,0,10); 
             break;

        default:
             LOG(LOG_ASSERT, GetName(), "Invalid beep type %d", beep);
             break;
    } // switch (beep)

    int rc = dx_playtoneEx(m_dx_srl_handle,&ToneGenCad, tpt, EV_ASYNC);

    LOG( RC(rc),GetName(), 
         "%d = dx_playtoneEx(0x%x, tone('%s'), EV_ASYNC)",
         rc, m_dx_srl_handle, tname);

    if ( rc == AT_FAILURE){
         process_dx_error(m_dx_srl_handle, GetName());
         return false;                             
    }

 return true;
} // End of DxBeep()



//*****************************************************************************
// Purpose	: 
//    Terminate current voice operation
// Parameters:	
//    none
// Returns:	
//    true = success
//*****************************************************************************
bool CSrlDevice::StopDx(){
int rc;

  // play it
  rc = dx_stopch(m_dx_srl_handle, EV_ASYNC);

      LOG( RC(rc), GetName(),
           "%d = dx_stopch(hndl=0x%x,  EV_ASYNC)",
           rc, m_dx_srl_handle);

      if ( rc == AT_FAILURE){
           process_dx_error(m_dx_srl_handle, GetName());
           return false;                             
      }

    return true;
} // End of StopDx()

//*****************************************************************************
// Purpose	: 
//    Get reason for terminating voice operation
// Parameters:	
//    none
// Returns:	
//    true = success
//*****************************************************************************
bool CSrlDevice::GetTermReason(){

  // play it
  int termno = ATDX_TERMMSK(m_dx_srl_handle);
  const char *termname = dx_get_termination(termno);  

  LOG( LOG_API, GetName(),
       "%d = ATDX_TERMMSK(hndl=0x%x) [%s])",
       termno, m_dx_srl_handle, termname);

 return true;
} // End of GetTermReason()

//*****************************************************************************
// Purpose	: 
//    Clear FW Digit buffer
// Parameters:	
//    none
// Returns:	
//    true = success
//*****************************************************************************
bool CSrlDevice::ClearDigits(){
bool brc = true;
int rc = dx_clrdigbuf(m_dx_srl_handle);
  LOG( RC(rc), GetName(),
       "%d = dx_clrdigbuf(hndl=0x%x)",
       rc, m_dx_srl_handle);

  if (rc == AT_FAILURE){
      process_dx_error(m_dx_srl_handle, GetName());
      brc = false;
  }

  return brc;
} // End of GetTermReason()

//*****************************************************************************
// Purpose	: 
//    Close voice file handle (if any)
// Parameters:	
//    none
// Returns:	
//    true = success
//*****************************************************************************
bool CSrlDevice::CloseVoiceFile(){
     if (m_iott.io_fhandle != INV_FILE_HANDLE) {
         int rc = dx_fileclose(m_iott.io_fhandle);
         LOG(LOG_DBG, GetName(), "%d = dx_fileclose(0x%x)",rc, m_iott.io_fhandle);
         m_iott.io_fhandle = INV_FILE_HANDLE;
     }
  return true;
} // End of CloseVoiceFile()

//*****************************************************************************
// Purpose	: 
//    Voice device listen to gc/ntwk(3PCC mode) device and vice versa
// Parameters:	
//    none
// Returns:	
//    bool (true = success)
//*****************************************************************************
bool CSrlDevice::RouteVoice(){
int rc;
long ntwk_xmit;

    SC_TSINFO ts_info;
    ts_info.sc_numts = 1;
    ts_info.sc_tsarrayp = &ntwk_xmit;

    //-------------------
    // Voice listen to gc/ ntwk

    // Get gc xmit slot
    if (Is3PCCMode()){
        rc =  ipm_GetXmitSlot(m_ntwk_srl_handle, &ts_info, EV_SYNC);
        LOG(  RCFATAL(rc), GetName(),
              "%d = ipm_GetXmitSlot(ipm hndl = 0x%x, &tc_info, EV_SYNC)",
              rc, m_ntwk_srl_handle);
   
        if ( rc != GC_SUCCESS) {
             SetSrlState ( SRLSTATE_FAILED_FATAL );
             process_ipml_error(m_ntwk_srl_handle, GetName());
             return false;
        }
    }else {
        rc =  gc_GetXmitSlot(m_srl_handle, &ts_info);
        LOG(  RCFATAL(rc), GetName(),
              "%d = gc_GetXmitSlot(gc hndl = 0x%x, &tc_info)",
              rc, m_srl_handle);
   
        if ( rc != GC_SUCCESS) {
             SetSrlState ( SRLSTATE_FAILED_FATAL );
             process_gc_error();
             return false;
        }
    }
    gc_dump(&ts_info);
    // voice listen to gc
    rc = dx_listen(m_dx_srl_handle, &ts_info);
    LOG( RCFATAL(rc), GetName(),
         "%d = dx_listen(dx hndl = 0x%x, &tc_info)",
         rc, m_dx_srl_handle);

    if ( rc != EDX_NOERROR) {
         SetSrlState ( SRLSTATE_FAILED_FATAL );
         process_dx_error(m_dx_srl_handle, m_dx_name);
         return false;
    }

    //-------------------
    // gc listen to voice

    // get voice xmit slot
    long voice_xmit;
    ts_info.sc_numts = 1;
    ts_info.sc_tsarrayp = &voice_xmit;

    rc = dx_getxmitslot(m_dx_srl_handle, &ts_info);
    LOG( RCFATAL(rc) , GetName(),
         "%d = dx_getxmitslot(dx hndl=0x%x, &ts_info)",
         rc, m_dx_srl_handle);

    if ( rc == AT_FAILURE) {
         SetSrlState ( SRLSTATE_FAILED_FATAL );
         process_dx_error(m_dx_srl_handle, m_dx_name);
         return false;
    }

    gc_dump(&ts_info);

    int handle;
    const char *name;
    if ( Is3PCCMode() ) {
        handle = m_ntwk_srl_handle;
        name = "ipml";
    }else {
        handle = m_srl_handle;
        name = "gc";
    }

    if (Is3PCCMode() ){
    // ipm listen to voice
        rc = ipm_Listen(m_ntwk_srl_handle, &ts_info, EV_SYNC);
        LOG( RCFATAL(rc), GetName(),
             "%d = ipm_Listen(ntwk hndl = 0x%x, &ts_info, EV_SYNC)",
             rc, m_ntwk_srl_handle);
        PutEvent(USREV_IPML_LISTEN, GetName());

        if ( rc != GC_SUCCESS) {
             SetSrlState ( SRLSTATE_FAILED_FATAL );
             process_ipml_error(m_ntwk_srl_handle, GetName());
             return false;
        }
    }else {
    // gc listen to voice
        rc = gc_Listen(m_srl_handle, &ts_info, EV_ASYNC);
        LOG( RCFATAL(rc), GetName(),
             "%d = gc_Listen(gc hndl = 0x%x, &ts_info, EV_ASYNC)",
             rc, m_srl_handle);

        if ( rc != GC_SUCCESS) {
             SetSrlState ( SRLSTATE_FAILED_FATAL );
             process_gc_error();
             return false;
        }
    }

 return true;
} // End of RouteVoice()

// Undo RouteVoice
bool CSrlDevice::UnListen(){
int rc;

    //-------------------------
    // Voice  gc/ ntwk unlisten

    if (Is3PCCMode()){
        rc =  ipm_UnListen(m_ntwk_srl_handle, EV_ASYNC);
        LOG(  RCFATAL(rc), GetName(),
              "%d = ipm_UnListen(ipm hndl = 0x%x, EV_ASYNC)",
              rc, m_ntwk_srl_handle);
   
        if ( rc != GC_SUCCESS) {
             SetSrlState ( SRLSTATE_FAILED_FATAL );
             process_ipml_error(m_ntwk_srl_handle, GetName());
             return false;
        }
    }else {
        rc =  gc_UnListen(m_srl_handle, EV_ASYNC);
        LOG(  RCFATAL(rc), GetName(),
              "%d = gc_UnListen(gc hndl = 0x%x, EV_ASYNC)",
              rc, m_srl_handle);
   
        if ( rc != GC_SUCCESS) {
             SetSrlState ( SRLSTATE_FAILED_FATAL );
             process_gc_error();
             return false;
        }
    }
 return true;
} // End of RouteVoice()

//*****************************************************************************
// Purpose	: 
//    dev_Connect cnf party to other device
// Parameters:	
//    [in] device type (DEV_DX_PARTY or DEV_NTWK_PARTY)
// Returns:	
//    true = success
//*****************************************************************************
bool CSrlDevice::DevConnect(DEV_PARTY party){
SRL_DEVICE_HANDLE hndl;
const char *devname;
   switch ( party) {
       case DEV_DX_PARTY:
            hndl = m_dx_srl_handle;
            devname = "dx handle";
            break;
       case DEV_NTWK_PARTY:
            hndl = m_ntwk_srl_handle;
            devname = "ntwk handle";
            break;
       case DEV_CNF_PARTY:
       default:
            LOG(LOG_ASSERT, GetName(), "DevDisconnect:: Invalid parameter %d", party);
            return false;
   } // switch(party)
   int rc = dev_Connect(hndl, m_cnf_party_srl_handle, DM_FULLDUP, EV_ASYNC); 
   bool brc = true;
   LOG( RC(rc), GetName(), 
        "%d = dev_Connect(%s=0x%x, cnf_party=0x%x, DM_FULLDUP, EV_ASYNC)",
        rc, devname, hndl, m_cnf_party_srl_handle);
   
   if (rc == BRD_FAILURE){
      process_dev_error( );
      brc = false;
   }
   return brc;
}// End of DevConnect()


//*****************************************************************************
// Purpose	: 
//    dev_Disconnect party 
// Parameters:	
//    [in] device type (DEV_DX_PARTY,  DEV_NTWK_PARTY or DEV_CNF_PARTY )
// Returns:	
//    true = success
//*****************************************************************************
bool CSrlDevice::DevDisconnect(DEV_PARTY party){
SRL_DEVICE_HANDLE hndl;
const char *devname;
   switch ( party) {
       case DEV_CNF_PARTY:
            hndl = m_cnf_party_srl_handle;
            devname = "cnf party handle";
            break; 
       case DEV_DX_PARTY:
            hndl = m_dx_srl_handle;
            devname = "dx handle";
            break;
       case DEV_NTWK_PARTY:
            hndl = m_ntwk_srl_handle;
            devname = "ntwk handle";
            break;
       default:
            LOG( LOG_ASSERT, GetName(),
                 "DevDisconnect:: Invalid parameter %d", party);
            return false;
   }// switch(party)

   int rc = dev_Disconnect(hndl, EV_ASYNC); 
   bool brc = true;

   LOG( RC(rc), GetName(), 
        "%d = dev_Disconnect(%s = 0x%x, EV_ASYNC)",
        rc, devname, hndl);
   
   if ( rc == BRD_FAILURE){
        process_dev_error();
        brc = false;
   }
   return brc;
}// End of DevDisconnect()

//*****************************************************************************
// Purpose	: 
//    Find conference party 
// Parameters:	
//    none
// Returns:	
//    true = success
//*****************************************************************************
bool CSrlDevice::OpenParty(){
  CNF_ASSERT(m_pCnfBoard, "Missing m_pCnfBoard object");

  int srl_handle = m_pCnfBoard->OpenParty(this);
  if ( CNF_ERROR != srl_handle ){
       m_cnf_party_srl_handle = srl_handle;
  } 
  
  LOG(LOG_DBG, GetName(), "0x%x = OpenParty()", srl_handle);
  
 return true;
}// End of OpenParty()

//*****************************************************************************
// Purpose	: 
//    Release conference party
// Parameters:	
//    none
// Returns:	
//    true = success
//*****************************************************************************
bool CSrlDevice::CloseParty(){
  CNF_ASSERT(m_pCnfBoard, "Missing m_pCnfBoard object");

  bool brc = m_pCnfBoard->CloseParty(m_cnf_party_srl_handle);

  LOG(LOG_DBG, GetName(), "%s = CloseParty(0x%x)",brc?"true":"false", m_cnf_party_srl_handle);
  if ( brc ){
       m_cnf_party_srl_handle = INV_SRL_HANDLE;
  }
 
 return brc;
}// End of CloseParty()

//*****************************************************************************
// Purpose	: 
//    Set conference attributes according to configuration parameters
// Parameters:	
//    none
// Returns:	
//    true = success
//*****************************************************************************
bool CSrlDevice::SetPartyAttributes(ECNF_ATTR_STATE clamping){
    CNF_ATTR CnfAttr[2];
    CNF_ATTR_INFO CnfAttrInfo;
    int inx = 0;
      CnfAttrInfo.unVersion = CNF_ATTR_INFO_VERSION_0; ///< Structure version
      CnfAttrInfo.pAttrList = CnfAttr;                 ///< Pointer to attribute structure list
        
      CnfAttr[inx].unVersion   = CNF_ATTR_VERSION_0 ;           ///< Structure version
      CnfAttr[inx].unAttribute = ECNF_PARTY_ATTR_TONE_CLAMPING; ///< Attribute type
      CnfAttr[inx].unValue     = clamping;                      ///< Disable tone clamping
        
      CnfAttrInfo.unAttrCount = inx+1;                 ///< Number of attribute structures in list

      bool brc = false;
      int rc;

      rc = cnf_SetAttributes(m_cnf_party_srl_handle, &CnfAttrInfo, this);

      cnf_dump(&CnfAttrInfo);
      LOG( RC(rc), GetName(),
           "%d = cnf_SetAttributes(0x%x,attr,0x%x(this))",
           rc, m_cnf_party_srl_handle, this);

 
        if (rc == CNF_ERROR) {
            process_cnf_error();
        }else {
            brc = true;
        }

   return brc;
} // End of SetAttributes()
