/**********@@@SOFT@@@WARE@@@COPY@@@RIGHT@@@**********************************
* DIALOGIC CONFIDENTIAL
*
* Copyright (C) 2006-2007 Dialogic Corporation. All Rights Reserved.
* The source code contained or described herein and all documents related
* to the source code ("Material") are owned by Dialogic Corporation or its
* suppliers or licensors. Title to the Material remains with Dialogic Corporation
* or its suppliers and licensors. The Material contains trade secrets and
* proprietary and confidential information of Dialogic or its suppliers and
* licensors. The Material is protected by worldwide copyright and trade secret
* laws and treaty provisions. No part of the Material may be used, copied,
* reproduced, modified, published, uploaded, posted, transmitted, distributed,
* or disclosed in any way without Dialogic's prior express written permission.
*
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*
***********************************@@@SOFT@@@WARE@@@COPY@@@RIGHT@@@**********/
//***********************************************************************
//***********************************************************************
// CSrlThread.cpp: implementation of the CSrlThread class.
//
//////////////////////////////////////////////////////////////////////

#include "pdl.h"
#include <time.h>

static const char * CNF_MODULE_NAME = "CSrlThread";


//*****************************************************************************
// Purpose	: 
//    Determine if given event is a notification event
//    Notification events may cause specific action,
//        but will not Advance state machine
// Parameters:	
//    [in] event
// Returns:	
//    bool (true = yes, it is notification event )
//*****************************************************************************
bool evt_isnotification(int event){
    switch(event){
        case CNFEV_CONF_CLOSED:
        case CNFEV_CONF_OPENED:
        case CNFEV_PARTY_ADDED:
        case CNFEV_PARTY_REMOVED:
        case CNFEV_DTMF_DETECTED:
        case CNFEV_ACTIVE_TALKER:
        case GCEV_PROCEEDING:
        case GCEV_ALERTING:    
        case TDX_CST:    
             return true; 
    }
 return false;
} // End of evt_isnotification()

//*****************************************************************************
// Purpose	: 
//    determine if given event is defined in this demo
// Parameters:	
//    [in] event
// Returns:	
//    bool (true = yes  )
//*****************************************************************************
bool evt_isuser(int event){
   if (   ((unsigned)event >= EVENT_USER ) 
       && ((unsigned)event <= USREV_LAST ) ){
       return true;
   }
   return false;
}

//*****************************************************************************
// Purpose	: 
//    determine if given event is significant for changing state in normal flow
// Parameters:	
//    [in] event
// Returns:	
//    bool (true = yes  )
//*****************************************************************************
bool evt_issignificant(int event){
    switch(event){
       case GCEV_BLOCKED:
       case GCEV_UNBLOCKED:
       case GCEV_DISCONNECTED:
            return false;
    }
 return true;
}
//*****************************************************************************
// Purpose	: 
//    Determine if given event indicates a failure
// Parameters:	
//    [in] event
// Returns:	
//    bool (true = yes, it indicates a failure )
//*****************************************************************************
bool evt_isfailure(int event){
    switch(event){

        case CNFEV_ERROR:
        case CNFEV_OPEN_FAIL:
        case CNFEV_OPEN_CONF_FAIL:
        case CNFEV_OPEN_PARTY_FAIL:
        case CNFEV_ADD_PARTY_FAIL:
        case CNFEV_REMOVE_PARTY_FAIL:
        case CNFEV_GET_ATTRIBUTE_FAIL:
        case CNFEV_SET_ATTRIBUTE_FAIL:
        case CNFEV_ENABLE_EVENT_FAIL:
        case CNFEV_DISABLE_EVENT_FAIL:
        case CNFEV_GET_DTMF_CONTROL_FAIL:
        case CNFEV_SET_DTMF_CONTROL_FAIL:
        case CNFEV_GET_ACTIVE_TALKER_FAIL:
        case CNFEV_GET_PARTY_LIST_FAIL:
        case CNFEV_GET_DEVICE_COUNT_FAIL:

        case GCEV_SETCONFIGDATA_FAIL:
        case GCEV_GETCONFIGDATA_FAIL:
        case GCEV_ATTACH_FAIL:
        case GCEV_DETACH_FAIL:
        case GCEV_OPENEX_FAIL:
        case GCEV_RELEASECALL_FAIL:
        case GCEV_FATALERROR:
        case GCEV_TASKFAIL:
        case GCEV_ERROR:
        case GCEV_RESTARTFAIL:
        case GCEV_INVOKE_XFER_FAIL:
        case GCEV_ACCEPT_XFER_FAIL:
        case GCEV_XFER_FAIL:
        case GCEV_REJ_XFER_FAIL:
        case GCEV_INIT_XFER_FAIL:
        case GCEV_ACCEPT_INIT_XFER_FAIL:
        case GCEV_REJ_INIT_XFER_FAIL:
        case GCEV_MODIFY_CALL_FAIL:
        case GCEV_CANCEL_MODIFY_CALL_FAIL:
        case GCEV_ACCEPT_MODIFY_CALL_FAIL:
        case GCEV_REJECT_MODIFY_CALL_FAIL:
        case GCEV_SIP_ACK_FAIL:

        case TDX_ERROR:
        case DX_ATOMIC_ERR:
        case IPMEV_ERROR:

        case DMEV_CONNECT_FAIL:
        case DMEV_DISCONNECT_FAIL:
        case DMEV_RESERVE_RESOURCE_FAIL:
        case DMEV_RELEASE_RESOURCE_FAIL:
        case DMEV_GET_RESOURCE_RESERVATIONINFO_FAIL:             
             return true; 
    }
 return false;
} // End of evt_isfailure()


//
// Tables to search for event name by code
//
static NAME_TABLE nongc_events[] = {
  // TDX Event Names
    { "TDX_PLAY",           TDX_PLAY       },     // Play Completed 
    { "TDX_RECORD",         TDX_RECORD     },     // Record Completed 
    { "TDX_GETDIG",         TDX_GETDIG     },     // Get Digits Completed 
    { "TDX_DIAL",           TDX_DIAL       },     // Dial Completed 
    { "TDX_CALLP",          TDX_CALLP      },     // Call Progress Completed 
    { "TDX_CST",            TDX_CST        },     // CST Event Received 
    { "TDX_SETHOOK",        TDX_SETHOOK    },     // SetHook Completed 
    { "TDX_WINK",           TDX_WINK       },     // Wink Completed 
    { "TDX_ERROR",          TDX_ERROR      },     // Error Event 
    { "TDX_PLAYTONE",       TDX_PLAYTONE   },     // Play Tone Completed 
    { "TDX_GETR2MF",        TDX_GETR2MF    },     // Get R2MF completed 
    { "TDX_BARGEIN",        TDX_BARGEIN    },     // Barge in completed 
    { "TDX_NOSTOP",         TDX_NOSTOP     },     // No Stop needed to be Issued 
    { "TDX_TXDATA",         TDX_TXDATA     },     // Transmit Data Completed 
    { "TDX_RXDATA",         TDX_RXDATA     },     // Receive Data Completed 
    { "TDX_CACHEPROMPT",    TDX_CACHEPROMPT},     // Data cached 
    { "TDX_UNDERRUN",	    TDX_UNDERRUN   },     // Generated when an underrun is detected by FW 
    { "TDX_LOWWATER",	    TDX_LOWWATER   },     // Generated when a low water mark is hit in buffer. 
    { "TDX_HIGHWATER",	    TDX_HIGHWATER  },     // Generated when a high water mark is hit in buffer. 
    { "TDX_QUERYTONE",	    TDX_QUERYTONE  },     // Query Tone Completed 
    { "TDX_QUERYTONE_FAIL", TDX_QUERYTONE_FAIL }, // Query Tone Failed 
    { "TDX_DELETETONE",	    TDX_DELETETONE     }, // Delete Tone Completed 
    { "TDX_DELETETONE_FAIL",TDX_DELETETONE_FAIL}, // Delete Tone Failed 
    { "TDX_CREATETONE",	    TDX_CREATETONE     }, // Create Tone Completed 
    { "TDX_CREATETONE_FAIL",TDX_CREATETONE_FAIL}, // Create Tone Failed 
    { "TDX_VAD",	 	    TDX_VAD            },
    { "TDX_LISTEN",         TDX_LISTEN         }, // completion event for dx_listenEx 
    { "TDX_UNLISTEN",       TDX_UNLISTEN       }, // completion event for dx_unlistenEx 
    { "TDX_LISTEN_FAIL",    TDX_LISTEN_FAIL    }, // failure completion event for dx_listenEx 
    { "TDX_UNLISTEN_FAIL",  TDX_UNLISTEN_FAIL  }, // failure completion event for dx_unlistenEx 
// CNF Events
    { "CNFEV_OPEN",              CNFEV_OPEN              },///< Board opened
    { "CNFEV_OPEN_CONF",         CNFEV_OPEN_CONF         },///< Conference opened
    { "CNFEV_OPEN_PARTY",        CNFEV_OPEN_PARTY        },///< Party opened
    { "CNFEV_ADD_PARTY",         CNFEV_ADD_PARTY         },///< Party added
    { "CNFEV_REMOVE_PARTY",      CNFEV_REMOVE_PARTY      },///< Party removed
    { "CNFEV_GET_ATTRIBUTE",     CNFEV_GET_ATTRIBUTE     },///< Got attributes
    { "CNFEV_SET_ATTRIBUTE",     CNFEV_SET_ATTRIBUTE     },///< Attributes set
    { "CNFEV_ENABLE_EVENT",      CNFEV_ENABLE_EVENT      },///< Events enabled
    { "CNFEV_DISABLE_EVENT",     CNFEV_DISABLE_EVENT     },///< Events disabled
    { "CNFEV_GET_DTMF_CONTROL",  CNFEV_GET_DTMF_CONTROL  },///< DTMF control retrieved
    { "CNFEV_SET_DTMF_CONTROL",  CNFEV_SET_DTMF_CONTROL  },///< DTMF control set
    { "CNFEV_GET_ACTIVE_TALKER", CNFEV_GET_ACTIVE_TALKER },///< Active talkers retrieved
    { "CNFEV_GET_PARTY_LIST",    CNFEV_GET_PARTY_LIST    },///< Party list retrieved
    { "CNFEV_GET_DEVICE_COUNT",  CNFEV_GET_DEVICE_COUNT  },///< Device count retrieved
// notification event definitions
    { "CNFEV_CONF_OPENED",       CNFEV_CONF_OPENED       },///< Conference opened notification event
    { "CNFEV_CONF_CLOSED",       CNFEV_CONF_CLOSED       },///< Conference closed notification event
    { "CNFEV_PARTY_ADDED",       CNFEV_PARTY_ADDED       },///< Party added notification event
    { "CNFEV_PARTY_REMOVED",     CNFEV_PARTY_REMOVED     },///< Party removed notification event
    { "CNFEV_DTMF_DETECTED",     CNFEV_DTMF_DETECTED     },///< DTMF detected notification event
    { "CNFEV_ACTIVE_TALKER",     CNFEV_ACTIVE_TALKER     },///< Active talker notification event 

//Error event definitions
    { "CNFEV_ERROR",                 CNFEV_ERROR                  },///< General error
    { "CNFEV_OPEN_FAIL",             CNFEV_OPEN_FAIL              },///< Board open failed
    { "CNFEV_OPEN_CONF_FAIL",        CNFEV_OPEN_CONF_FAIL         },///< Conference open failed
    { "CNFEV_OPEN_PARTY_FAIL",       CNFEV_OPEN_PARTY_FAIL        },///< Party open failed
    { "CNFEV_ADD_PARTY_FAIL",        CNFEV_ADD_PARTY_FAIL         },///< Party added
    { "CNFEV_REMOVE_PARTY_FAIL",     CNFEV_REMOVE_PARTY_FAIL      },///< Party removed
    { "CNFEV_GET_ATTRIBUTE_FAIL",    CNFEV_GET_ATTRIBUTE_FAIL     },///< Attributes retrieved
    { "CNFEV_SET_ATTRIBUTE_FAIL",    CNFEV_SET_ATTRIBUTE_FAIL     },///< Attributes set
    { "CNFEV_ENABLE_EVENT_FAIL",     CNFEV_ENABLE_EVENT_FAIL      },///< Events enabled
    { "CNFEV_DISABLE_EVENT_FAIL",    CNFEV_DISABLE_EVENT_FAIL     },///< Events disabled
    { "CNFEV_GET_DTMF_CONTROL_FAIL", CNFEV_GET_DTMF_CONTROL_FAIL  },///< DTMF control retrieved
    { "CNFEV_SET_DTMF_CONTROL_FAIL", CNFEV_SET_DTMF_CONTROL_FAIL  },///< DTMF control set
    { "CNFEV_GET_ACTIVE_TALKER_FAIL",CNFEV_GET_ACTIVE_TALKER_FAIL },///< Active talkers retrieved
    { "CNFEV_GET_PARTY_LIST_FAIL",   CNFEV_GET_PARTY_LIST_FAIL    },///< Party list retrieved
    { "CNFEV_GET_DEVICE_COUNT_FAIL", CNFEV_GET_DEVICE_COUNT_FAIL  },///< Device count retrieved


 // DevMgmt events
    { "DMEV_CONNECT",                DMEV_CONNECT  },
    { "DMEV_CONNECT_FAIL",           DMEV_CONNECT_FAIL  },
    { "DMEV_DISCONNECT",             DMEV_DISCONNECT  },
    { "DMEV_DISCONNECT_FAIL",        DMEV_DISCONNECT_FAIL  },
    { "DMEV_RESERVE_RESOURCE",       DMEV_RESERVE_RESOURCE  },
    { "DMEV_RESERVE_RESOURCE_FAIL",  DMEV_RESERVE_RESOURCE_FAIL  },
    { "DMEV_RELEASE_RESOURCE",       DMEV_RELEASE_RESOURCE  },
    { "DMEV_RELEASE_RESOURCE_FAIL",  DMEV_RELEASE_RESOURCE_FAIL  },
    { "DMEV_GET_RESOURCE_RESERVATIONINFO",     DMEV_GET_RESOURCE_RESERVATIONINFO  },
    { "DMEV_GET_RESOURCE_RESERVATIONINFO_FAIL",DMEV_GET_RESOURCE_RESERVATIONINFO_FAIL  },
    

// Demo specific events
    { "USREV_EXIT_REQUEST", USREV_EXIT_REQUEST }, ///< Exit request

    { "USREV_TIMEOUT",      USREV_TIMEOUT      }, ///< User event timeout
    { "USREV_CNF_TIMEOUT",  USREV_CNF_TIMEOUT  }, ///< Conference timeout
    { "USREV_CHECK_PASS",   USREV_CHECK_PASS   }, ///< Conference check by passwd: success
    { "USREV_CHECK_FAIL",   USREV_CHECK_FAIL   }, ///< Conference check by passwd: not found, give up
    { "USREV_CHECK_RETRY",  USREV_CHECK_RETRY  }, //< Conference check by passwd: not found, retry

    { "USREV_CNF_OPEN",     USREV_CNF_OPEN     }, ///< A signal from Conference to it's parent board
    { "USREV_PARTY_REMOVED",USREV_PARTY_REMOVED}, ///< A signal from party to it's conference

    { "USREV_LEAVE_CNF",    USREV_LEAVE_CNF    }, ///< Leave conference ( signal from conference to party )
    { "USREV_BEEP_OUT",     USREV_BEEP_OUT     }, ///< indicate leave 
    { "USREV_BEEP_IN",      USREV_BEEP_IN      }, ///< indicate join
    { "USREV_BEEP_END",     USREV_BEEP_END     }, ///< indicate end
    { "USREV_SKIP_VOICE",   USREV_SKIP_VOICE   }, ///< skip voice party
    { "USREV_SKIP_ACCEPT",  USREV_SKIP_ACCEPT  }, ///< skip accept
    { "USREV_IPML_LISTEN",  USREV_IPML_LISTEN  }, ///< ipml listen sync mode

// IPM Events
    { "IPMEV_OPEN",                 IPMEV_OPEN  },  
    { "IPMEV_STARTMEDIA",           IPMEV_STARTMEDIA  },
    { "IPMEV_STOPPED",              IPMEV_STOPPED  },
    { "IPMEV_GET_LOCAL_MEDIA_INFO", IPMEV_GET_LOCAL_MEDIA_INFO  },
    { "IPMEV_GET_SESSION_INFO",     IPMEV_GET_SESSION_INFO  },
    { "IPMEV_LISTEN",               IPMEV_LISTEN  },
    { "IPMEV_UNLISTEN",             IPMEV_UNLISTEN  },
    { "IPMEV_GET_XMITTS_INFO",      IPMEV_GET_XMITTS_INFO  },
    { "IPMEV_SEND_DIGITS",          IPMEV_SEND_DIGITS  },
    { "IPMEV_RECEIVE_DIGITS",       IPMEV_RECEIVE_DIGITS  },
    { "IPMEV_DIGITS_RECEIVED",      IPMEV_DIGITS_RECEIVED  },
    { "IPMEV_EVENT_ENABLED",        IPMEV_EVENT_ENABLED  },
    { "IPMEV_EVENT_DISABLED",       IPMEV_EVENT_DISABLED  },

    { "IPMEV_GET_QOS_ALARM_STATUS", IPMEV_GET_QOS_ALARM_STATUS  },
    { "IPMEV_RESET_QOS_ALARM_STATUS",IPMEV_RESET_QOS_ALARM_STATUS  },
    { "IPMEV_SET_QOS_THRESHOLD_INFO",IPMEV_SET_QOS_THRESHOLD_INFO  },
    { "IPMEV_GET_QOS_THRESHOLD_INFO",IPMEV_GET_QOS_THRESHOLD_INFO  },
    { "IPMEV_QOS_ALARM",            IPMEV_QOS_ALARM  },
    { "IPMEV_SET_PARM",             IPMEV_SET_PARM  },
    { "IPMEV_GET_PARM",             IPMEV_GET_PARM  },
    { "IPMEV_PING",                 IPMEV_PING  },
    { "IPMEV_SEND_SIGNAL_TO_IP",    IPMEV_SEND_SIGNAL_TO_IP  },
    { "IPMEV_FAXTONE",              IPMEV_FAXTONE  },
    { "IPMEV_T38CALLSTATE",         IPMEV_T38CALLSTATE  },
    { "IPMEV_T38CAPFRAME_STATUS",   IPMEV_T38CAPFRAME_STATUS  },
    { "IPMEV_T38INFOFRAME_STATUS",  IPMEV_T38INFOFRAME_STATUS  },
    { "IPMEV_T38HDLCFRAME_STATUS",  IPMEV_T38HDLCFRAME_STATUS  },
    { "IPMEV_T38POSTPGCODE_STATUS", IPMEV_T38POSTPGCODE_STATUS  },
    { "IPMEV_ERROR",                IPMEV_ERROR  },
    { "IPMEV_T38POSTPGRESP_STATUS", IPMEV_T38POSTPGRESP_STATUS  },
    { "IPMEV_GET_CTINFO",           IPMEV_GET_CTINFO  },
    { "IPMEV_MODIFYMEDIA",          IPMEV_MODIFYMEDIA  },

# ifdef IPMEV_GET_CAPABILITIES 
    { "IPMEV_GET_CAPABILITIES",     IPMEV_GET_CAPABILITIES  },
# endif

# ifdef IPMEV_SEC_NOTIFY_EXPIRE_KEY_AUDIO 
    { "IPMEV_SEC_NOTIFY_EXPIRE_KEY_AUDIO",  IPMEV_SEC_NOTIFY_EXPIRE_KEY_AUDIO  },
# endif

# ifdef IPMEV_SEC_NOTIFY_EXPIRE_KEY_VIDEO 
    { "IPMEV_SEC_NOTIFY_EXPIRE_KEY_VIDEO",  IPMEV_SEC_NOTIFY_EXPIRE_KEY_VIDEO  },
# endif
    { "IPMEV_MODIFYMEDIA_FAIL",     IPMEV_MODIFYMEDIA_FAIL  },
      
    { 0,0}

}; // end of nongc_events[]

//*****************************************************************************
// Purpose	: 
//    Returns name of given event
// Parameters:	
//    [in]  event code
//    [out] holder for name
// Returns:	
//    bool (true = success, evt_name set properly in this case )
//*****************************************************************************
bool evt_getname(int event, const char **evt_name){
   if ( !str_findname(event, evt_name, nongc_events) ){
         *evt_name = GCEV_MSG(event); 
   }
   return true;
} // End of evt_getname()


//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

//*****************************************************************************
// Purpose	: 
//    Constructor
// Parameters:	
//    [in] Configuration parameters ( from configuration file )
// Returns:	
//    none
//*****************************************************************************
CSrlThread::CSrlThread(PCommonParams pCommonParams) 
          : CSrlDeviceContainer(DEV_NONE, pCommonParams, 0){
 m_srl_tmo = 0;
 m_onidle_counter = 0;
 // Exiting
 flag_exit_request = false;

 SetName("SrlThread");
} // End of Constructor()

//*****************************************************************************
// Purpose	: 
//    Destructor
// Parameters:	
//    none
// Returns:	
//    none
//*****************************************************************************
CSrlThread::~CSrlThread(){
  
  WaitSrlState(10000, SRLSTATE_OPENED, false ,0 );
  GLBLOG(LOG_DBG, CNF_MODULE_NAME, "Object deleted");
} // End of Destructor()

//*****************************************************************************
// Purpose	: 
//    Init srl thread - specify SRL model and timeout
//    Note: different threads may have different timeouts, but
//          will share same model - latest sr_setparm takes precedence
// Parameters:	
//    [in] timeout in ms 
//    [in] srl model
// Returns:	
//    true = success
//    false = failure
//*****************************************************************************
bool CSrlThread::InitMode(int srl_tmo,  // timeout in milliseconds
                          int mode){    // mode ( see user manual )
int ret_code;
// 
  m_srl_tmo=srl_tmo;
  switch (mode)  {
        default:
             GLBLOG( LOG_ERR1, CNF_MODULE_NAME,
                     "Requested mode (%d) is not supported in this demo. Use SR_STASYNC(Win32) / SR_POLLED(Linux)", mode);
             ret_code = AT_FAILURE;
             break;

        case SRL_POLLED_MODE:
             // single thread polled mode
             // (asynchronous polled)
            # ifdef WIN32
            //Windows
                int srlmodeltype = SR_STASYNC;      // single thread asyncronous
                ret_code = sr_setparm(SRL_DEVICE, SR_MODELTYPE, &srlmodeltype),
                GLBLOG(LOG_API, CNF_MODULE_NAME,
                       "%d = sr_setparm(SRL_DEVICE, SR_MODELTYPE, SR_STASYNC)", ret_code);
            # else  
            // Linux
                int srlmodeltype = SR_POLLMODE;     // Polled mode
                ret_code = sr_setparm(SRL_DEVICE, SR_MODEID, &srlmodeltype),
                GLBLOG(LOG_API, CNF_MODULE_NAME,
                       "%d = sr_setparm(SRL_DEVICE, SR_MODEID, SR_POLLMODE)", ret_code);
            # endif
                break;
  } // switch 

 return ret_code != AT_FAILURE;
} // End of InitMode()

//*****************************************************************************
// Purpose	: 
//    Receive and handle events
// Parameters:	
//    none
// Returns:	
//    none. 
//*****************************************************************************
void CSrlThread::TheThreadBody(){
int rc = 1;
int evhandle = 0;
// in WIN32 MT environment, change 0 to evhanke

  while (rc > 0) {
      // consume all available events
       rc = sr_waitevt(m_srl_tmo) ;
       if(  rc >= 0 ) {
	        EventHandler(evhandle);
            m_onidle_counter = 0;    // reset counter
       }  
  }
  OnIdle();  

 return ;
} // End of TheThreadBody()

//*****************************************************************************
// Purpose	: 
//    Handle events
// Parameters:	
//    [in] evhandle - only used in WIN32 Multithreaded models 
// Return:	
//    none
//*****************************************************************************
void CSrlThread::EventHandler(int evhandle){

  // --- Find name and log
#  ifdef WIN32
     void * evtusercontext = sr_getUserContext(evhandle);
     int evttype = sr_getevttype(evhandle);
     int srl_handle = sr_getevtdev(evhandle);
     void *evtdata = sr_getevtdatap(evhandle);
     int evtlen = sr_getevtlen(evhandle);
#  else 
  // --- LINUX
     void * evtusercontext = sr_getUserContext();
     int evttype = sr_getevttype();
     int srl_handle = sr_getevtdev();
     void *evtdata = sr_getevtdatap();
     int evtlen = sr_getevtlen();
#  endif

METAEVENT metaevent;
const char *event_name;
PSrlDevice pDevice = 0;
bool is_globalcall = false;
   gc_GetMetaEvent(&metaevent);

   if (metaevent.flags & GCME_GC_EVENT) {
       // GlobalCall event
       event_name = GCEV_MSG(evttype); 
       is_globalcall = true; 
   }else {
       str_findname(evttype, &event_name, nongc_events) ;
   }
   
   //
   // Search node:
   //
   if ( 0 != evtusercontext ) {
       // User context is device object un case of CNF events
        pDevice = static_cast<PSrlDevice>(evtusercontext);
   }else {
        if (is_globalcall) {
            // user attribute is device object in case of GlobalCall
            pDevice = static_cast<PSrlDevice>(metaevent.usrattr);
        }
   }

   if (0 == pDevice ) {
       if (!AreYouHandle(srl_handle, &pDevice) ){
           LOG( LOG_WARNING, CNF_MODULE_NAME,
                "Missing device, srl_handle = 0x%x(%d), event = 0x%x %s",
                srl_handle, srl_handle, evttype, event_name);
                return;
       }
   } // 0 == pDevice


   const char *dev_name =  CNF_MODULE_NAME;
   if (pDevice){
       dev_name = pDevice->GetName();
   }

   CGenLog *pLog;
   if (pDevice){
       pLog = pDevice->GetLog();
   }else {
       pLog = GLBLOGOBJ;
   }

   const char *txt;

   if ( evt_isnotification(evttype) ) {
        txt = "[info]";
   }else if(    (evttype >= EVENT_USER)
             && (evttype <= USREV_LAST) )  {
        txt = "[app ]";
   }else {
        txt = "[term]";
   }

   const char *event_data = "";
   switch(evttype){
       case TDX_CST:
           { DX_CST *cst = (DX_CST*)evtdata;
             switch(cst->cst_event){
                case DE_SILON:   event_data = "DE_SILON"; break;
                case DE_SILOF:   event_data = "DE_SILOF"; break;
                case DE_DIGITS:  event_data = "DE_DIGITS"; break;
                case DE_DIGOFF:  event_data = "DE_DIGOFF"; break;
                case DE_TONEON:  event_data = "DE_TONEON"; break;
                case DE_TONEOFF: event_data = "DE_TONEOFF"; break;
                case DE_VAD:     event_data = "DE_VAD"; break;
             }
 
           }
            break;
       default: 
            break;
   }

   if (is_globalcall) {
       pLog->Log(RCEVENT(evttype), dev_name,
                 "%s%s %s 0x%x (%d), usrattr = 0x%x hndl = 0x%x", 
                 event_name, txt, event_data, evttype, evttype,  metaevent.usrattr, srl_handle);
   }else {
       pLog->Log(RCEVENT(evttype), dev_name,
                 "%s%s %s 0x%x (%d), context = 0x%x hndl = 0x%x", 
                 event_name, txt, event_data, evttype, evttype,  evtusercontext, srl_handle);
   }

/*
   // test print
   if (pLog->CanOutput(LOG_DBG)) {
       printf("*** Event %s %s %d 0x%x %s\n",
              pDevice->GetName(), event_data, evttype, evttype,  event_name);
      fflush(stdout);
   }
*/

   LOG(LOG_DBG, CNF_MODULE_NAME,
            "Event routed to device: %s", 
             pDevice->GetName());
    
   pDevice->HandleEvent(evttype, evtdata, evtlen, &metaevent);

 return;
} // End of EventHandler()

//*****************************************************************************
// Purpose	: 
//    Do something while the system is idle
//    1. Pass exit request down to all devices
//    2. Calculate timeouts
//    3. Printf dots and
// Parameters:	
//    none
// Returns:	
//    none
//*****************************************************************************
void CSrlThread::OnIdle(){

 CheckTmo();

 if ( flag_exit_request ) {
     flag_exit_request = false;
     ExitRequest();
 }
 return;
} // End of OnIdle()

//*****************************************************************************
// Purpose	: 
//    Wait specific time for all devices to reach specified state
// Parameters:	
//    [in] tmo (seconds) - how long to wait
//    [in] state         - state to compare with
//    [in] compare flag: - true means compare for equal, 
//                    false means compare for not equal
//    [out] holder for SrlDevice addr
// Returns:	
//    bool (true = success, compare passed for all devices)
//         (false = operation timed out, compare failed for at least one device
//         *ppDev contains first device that DOES NOT match criteria)
//*****************************************************************************
bool CSrlThread::WaitSrlState(int tmo,
                              int srl_state,
                              bool compare_flag,
                              PSrlDevice *ppDev){
    struct timeb bgn;
    struct timeb now;
    struct timeb diff;

    ftime(&bgn);
    ftime(&now);
    timeb_diff(&diff, &bgn, &now);

    while( time_isgreater(tmo, &diff) ){
        if (AreSrlStates(srl_state, compare_flag, ppDev) ){
            return true;
        }
        // Give everybody a chance to Advance state
        TheThreadBody();

        if ( AreSrlStates(srl_state, compare_flag, ppDev) ){
            return true;
        }

        if (AreSrlStates(SRLSTATE_OPENED, false, ppDev) ){
            return false;
        }

        ftime(&now);
        timeb_diff(&diff, &bgn, &now);
    }; // while
        
    return AreSrlStates(srl_state, compare_flag, ppDev);
} // End of WaitSrlState()
