//***********************************************************************
// Copyright  2005, Intel Corporation. 
// All rights reserved All names, products, and services mentioned  
// herein are the trademarks or registered trademarks of their respective 
// organizations and are the sole property of their respective owners
//***********************************************************************


// IpDev.cpp: implementation of the CIpDev class.
//
//////////////////////////////////////////////////////////////////////

#if defined(_WIN32)
#include <windows.h>
#else
#include <unistd.h>
#endif

#include "IpDev.h"
#include "pdl.h"
#include "AppLog.h"
#include "utils.h"

#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <srllib.h>
#include <dxxxlib.h>
#include <gcip_defs.h>
#include <stdio.h>


//static members
char CIpDev::m_devTxt[] = {0};

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

//*****************************************************************************
// Function : 
//    CIp::CIpDev()
// Purpose  : 
//    Constructor
// Parameters:  
//    none
// Return type: 
//    none
//*****************************************************************************
CIpDev::CIpDev()
	: m_channelNum(-1)
	, m_crn(-1)
	, m_hDev(-1)
	, m_hVox(-1)
	, m_tsn(-1)
	, m_tsv(-1)
	, m_dropFlag(0)
	, m_playAttempt(0)
	, m_bridgeNumber(-1)
{
	
	m_xpb.wFileFormat = FILE_FORMAT_VOX;
	m_xpb.wDataFormat = DATA_FORMAT_MULAW;
	m_xpb.nSamplesPerSec = DRT_8KHZ;
	m_xpb.wBitsPerSample = 8;

	m_iott.io_type = IO_DEV|IO_EOT;
	m_iott.io_bufp = 0;
	m_iott.io_offset = 0;
	m_iott.io_length = -1;
	
//	strcpy(m_digBuf.dg_value, "123#");
}

//*****************************************************************************
// Function : 
//    CIp::~CIpDev()
// Purpose  : 
//    Destructor
// Parameters:  
//    none
// Return type: 
//    none
//*****************************************************************************
CIpDev::~CIpDev()
{
}

//////////////////////////////////////////////////////////////////////
// Public methods
//////////////////////////////////////////////////////////////////////


//*****************************************************************************
// Function : 
//    CIpDev::OpenDev()
// Purpose  : 
//    Opening channel and assigning a device handle
// Parameters:  
//    Parm1 <int channel><direction>
// Return type: 
//    int
//*****************************************************************************
int CIpDev::openDev(int channel)
{
	static int brd = 0, chn =0;
	
	// Open voice device
	chn = channel % 4;
	if(chn == 0)
		chn = 4;
	if(chn == 1)
		brd++;
	sprintf(m_voxDevName,"dxxxB%dC%d", brd, chn);
	
	if( (m_hVox = dx_open(m_voxDevName, 0)) == -1) {
		sprintf(m_devTxt, "Cannot open %s", m_voxDevName);
		logOut(WARN, m_devTxt, 1, true);
		return -1;
	}
	
	m_channelNum = channel;
	SC_TSINFO tsinfo;
	tsinfo.sc_numts = 1;
	tsinfo.sc_tsarrayp = &m_tsv;
	if(dx_getxmitslot(m_hVox, &tsinfo) == -1) {
		checkError("dx_getxmitslot");
	}
	else {
		sprintf(m_devTxt,"Media TS = %ld", m_tsv);
		logOut(INFO, m_devTxt, 1, true);
	}
	
	
	char devName[32];
	
	sprintf(m_netDevName, "iptB1T%d", channel);
	sprintf(devName, ":N_%s:P_IP:M_ipmB1C%d",m_netDevName, channel);
	
	if (gc_OpenEx(&m_hDev, devName, EV_ASYNC,&m_channelNum) != GC_SUCCESS) {
		sprintf(m_devTxt,"Cannot open %s",devName);
		logOut(ERR2, m_devTxt, 1);
		return -1;
	}
	else {
		sprintf(m_devTxt,"iptB1T%d opened", channel);
		logOut(INFO, m_devTxt, 1, true);
	}
	
	return 0;
}

//*****************************************************************************
// Function : 
//    CIpDev::CloseDev()
// Purpose  : 
//    Closing channel
// Parameters:  
//    none
// Return type: 
//    int
//*****************************************************************************
int CIpDev::closeDev()
{
	if(dx_close(m_hVox) == -1)
		checkError("dx_close");
	else
		logOut(INFO, "dx_close success", 1, true);
	
	if(gc_Close(m_hDev) != GC_SUCCESS)
		checkError("gc_Close");
	else
		logOut(INFO, "gc_Close success", 1);
	return 0;
}

//*****************************************************************************
// 
// Function	: 
//    CIpDev::enableCalls()
// Purpose	: 
//    Enables/Disables incoming calls
// Parameters:	
//    Parm1 <bool enable><direction>
// Return type:	
//    int
//*****************************************************************************
int CIpDev::enableCalls(bool enable)
{
	if(enable) {
		if (gc_WaitCall(m_hDev,NULL ,NULL ,0,EV_ASYNC) != GC_SUCCESS) {
			checkError("gc_WaitCall");
			return -1;
		}
		logOut(APP, "gc_WaitCall(EV_ASYNC) = 0", 1);
		m_portState = NULL_STATE;
		return 0;
	}
	else {
		m_portState = BLOCKED_STATE; 
	}
	return 0;
}

//*****************************************************************************
//
// Function     :
//    CIpDev::enablState()
// Purpose      :
//    Enables/Disables channel state
// Parameters:
//    Parm1 <bool enable><direction>
// Return type:
//    int
//*****************************************************************************
int CIpDev::enableState(bool enable)
{
        if(enable) 
                m_portState = NULL_STATE;
        else 
                m_portState = BLOCKED_STATE;
        return 0;
}

//*****************************************************************************
// 
// Function : 
//    CIpDev::enableSingleDigit()
// Purpose  : 
//    Allows detecting a sinfle DTMF as TDX_CST event
// Parameters:  
//    Parm1 <bool digEnable><direction>
// Return type: 
//    none
//*****************************************************************************
void CIpDev::enableSingleDigit(bool digEnable)
{
	int retVal = 0;
	m_digBuf.dg_value[0] = 0;
	if(digEnable) {
		retVal = dx_setevtmsk(m_hVox, DM_DIGITS);
	}
	else {
		retVal = dx_setevtmsk(m_hVox, DM_DIGOFF);
	}
	if(retVal == -1) {
		checkError("dx_setevtmsk(DM_DIGITS/DM_DIGOFF)");
	}
	else {
		logOut(INFO, "dx_setevtmsk(DM_DIGITS/DM_DIGOFF) = OK", 1);
	}
}


//*****************************************************************************
// 
// Function : 
//    CIpDev::isIdle()
// Purpose  : 
//    
// Parameters:  
// Return type: 
//    bool
//*****************************************************************************
bool CIpDev::isIdle() const
{
	return (m_portState == NULL_STATE);
}


//*****************************************************************************
// Function : 
//    CIpDev::dti_Listen()
// Purpose  : 
//    Routing the network timeslot to listen to the timeslot passed in
// Parameters:  
//    Parm1 <long i_ts><direction>
//    Parm2 <int mode><direction>
// Return type: 
//    int
//*****************************************************************************
int CIpDev::dti_Listen(long i_ts, int mode)
{
	long ts = i_ts;
	SC_TSINFO tsinfo;
	tsinfo.sc_numts = 1;
	tsinfo.sc_tsarrayp = &ts;
	
	if(gc_Listen(m_hDev,&tsinfo, mode) != GC_SUCCESS)
	{
		checkError("gc_Listen");
		return -1;
	}
	else {
		sprintf(m_devTxt, "gc_Listen(%ld)=0", i_ts);
		logOut(APP, m_devTxt, 1);
	}
	return 0;
}
//*****************************************************************************
// Function : 
//    CIpDev::dx_Listen()
// Purpose  : 
//    Routing the voice timeslot to listen to the timeslot passed in
// Parameters:  
//    Parm1 <long i_ts><direction>
// Return type: 
//    int
//*****************************************************************************
int CIpDev::dx_Listen(long i_ts)
{
	long ts=i_ts;
	SC_TSINFO tsinfo;
	tsinfo.sc_numts = 1;
	tsinfo.sc_tsarrayp = &ts;
	
	if(dx_listen(m_hVox,&tsinfo) == 0)
	{
		sprintf(m_devTxt, "dx_listen(%ld)=0", i_ts);
		logOut(APP, m_devTxt, 1, true);
		return 0;
	}
	else {
		checkError("dx_listen");
	}
	return -1;
	
}

//*****************************************************************************
// Function : 
//    CIpDev::dti_Unlisten()
// Purpose  : 
//    Unroute the network resource not to listen to the timeslot it was route to
//    listen to previoustly
// Parameters:  
//    none
// Return type: 
//    none
//*****************************************************************************
void CIpDev::dti_Unlisten()
{
	gc_UnListen(m_hDev, EV_ASYNC);
}

//*****************************************************************************
// Function : 
//    CIpDev::dx_Unlisten()
// Purpose  : 
//    Unroute the voice resource not to listen to the timeslot it was routed to
//    listen to previously
// Parameters:  
//    none
// Return type: 
//    none
//*****************************************************************************
void CIpDev::dx_Unlisten()
{
	dx_unlisten(m_hVox);
}

//*****************************************************************************
// Function : 
//    CIpDev::drop()
// Purpose  : 
//    Dropping the call
// Parameters:  
//    Parm1 <int cause><direction>
// Return type: 
//    int
//*****************************************************************************
int CIpDev::drop(int cause)
{
	if(gc_DropCall(m_crn, cause,  EV_ASYNC) != GC_SUCCESS) {
		checkError("gc_DropCall");
	}
	else
	{
		logOut(APP, "gc_DropCall()", 1);
		m_portState = DROPPING_STATE;
	}
	
	return 0;
}
//*****************************************************************************
// Function : 
//    CIpDev::dropCallGracerully()
// Purpose  : 
//    Playing a goodbye file before dropping the caller.
// Parameters:  
//    Parm1 <char * byeFileName><direction>
// Return type: 
//    int
//*****************************************************************************
int CIpDev::dropCallGracefully(char *byeFileName)
{
	if(playFile(byeFileName) == 0)
	{
		m_dropFlag = 1;
		return 0;
	}
	else
		return (drop());
}

//*****************************************************************************
// Function : 
//    CIpDev::collectPasscodeDigits()
// Purpose  : 
//    Plays a file that ask the user to enter digits.
// Parameters:  
//    Parm1 <char * playFileName><input>
// Return type: 
//    int
//*****************************************************************************
int CIpDev::collectPasscodeDigits(char *playFileName)
{
	if(m_playAttempt <= 0)
		return -1;
	return (playFile(playFileName));
}

//*****************************************************************************
// Function : 
//    CIpDev::collectPasscodeDigits() (overloaded)
// Purpose  : 
//    Playing a file a certain number of times 
// Parameters:  
//    Parm1 <char * playFileName><input>
//    Parm2 <int numOfAttempt><input>
// Return type: 
//    int
//*****************************************************************************
int CIpDev::collectPasscodeDigits(char *playFileName, int numOfAttempts)
{
	if(playFile(playFileName) == 0)
	{
		m_playAttempt = numOfAttempts;
		return 0;
	};
	return -1;
}

//*****************************************************************************
// Function : 
//    CIpDev::playFile()
// Purpose  : 
//    Playing a file on that device
// Parameters:  
//    Parm1 <char * file><direction>
// Return type: 
//    int
//*****************************************************************************
int CIpDev::playFile(const char *file)
{
	
	m_tpt[0].tp_type   = IO_EOT;
	m_tpt[0].tp_termno = DX_MAXDTMF;
	m_tpt[0].tp_length = 1;
	m_tpt[0].tp_flags  = TF_MAXDTMF;
	dx_clrdigbuf(m_hVox);
	m_iott.io_fhandle  = dx_fileopen(file, GH_FILE_OPEN_FLAG_PLAY);
	
	if(m_iott.io_fhandle == -1)
	{
		sprintf(m_devTxt,"Cannot open %s to play", file);
		logOut(WARN, m_devTxt, 1, true);
		return -1;
	}
	if(dx_playiottdata (m_hVox,&m_iott,m_tpt,&m_xpb,EV_ASYNC) == -1) {
		checkError("dx_playiottdata");
		return -1;
	}
	else {
		sprintf(m_devTxt, "dx_playiottta(%s)=0", file);
		logOut(APP, m_devTxt, 1, true);
	}
	return 0;
}

//*****************************************************************************
// Function : 
//    CIpDev::recordVoice()
// Purpose  : 
//    Record to a file
// Parameters:  
//    Parm1 <char *filename><direction>
//    Parm2 <int timeout><direction>
// Return type: 
//    int
//*****************************************************************************
int CIpDev::recordVoice(char *filename, int timeout)
{
	/*
	m_tpt[0].tp_type   = IO_CONT;
	m_tpt[0].tp_termno = DX_MAXDTMF;
	m_tpt[0].tp_length = 1;
	m_tpt[0].tp_flags  = TF_MAXDTMF;
	*/

	m_tpt[1].tp_type   = IO_EOT;
	m_tpt[1].tp_termno = DX_MAXTIME;
	m_tpt[1].tp_length = timeout;
	m_tpt[1].tp_flags  = TF_MAXTIME;
	
	m_iott.io_type     =  IO_DEV|IO_EOT;
	m_iott.io_fhandle  = dx_fileopen(filename, GH_FILE_OPEN_FLAG_REC, GH_FILE_OPEN_PERMISSION_REC);
	if(m_iott.io_fhandle == -1)
	{
		sprintf(m_devTxt,"Cannot open %s to record", filename);
		logOut(WARN, m_devTxt, 1, true);
		return -1;
	}
	dx_clrdigbuf(m_hVox);
	logOut(INFO, "dx_reciottta() on entry", 0, true);
	
	
	if(dx_reciottdata (m_hVox, &m_iott, &m_tpt[1] ,&m_xpb,EV_ASYNC|RM_TONE) == -1) {
		checkError("Rec file");
		dx_fileclose(m_iott.io_fhandle);
		return -1;
	}
	else {
		sprintf(m_devTxt, "dx_reciottta(%s)=0",filename);
		logOut(APP, m_devTxt, 1, true);
	}
	return 0;
}

//*****************************************************************************
// 
// Function : 
//    CIpDev::stopVoice()
// Purpose  : 
//    Stops all voice activity on the channel
// Parameters:  
// Return type: 

//    none
//*****************************************************************************
void CIpDev::stopVoice()
{
	dx_stopch(m_hVox, EV_ASYNC);
}

//*****************************************************************************
// Function : 
//    CIpDev::processEvent()
// Purpose  : 
//    Processing the event
// Parameters:  
//    none
// Return type: 
//    int
//*****************************************************************************
int CIpDev::processEvent()
{
	METAEVENT metaevent;
	GC_PARM_BLKP  gcParmBlk = NULL;
	EXTENSIONEVTBLK 	*pextensionBlk;
	long event, termno;
	char txt[128];
	DX_CST * cst;
	int ret_val = 0;
	int state = -1;
	
	if(gc_GetMetaEvent(&metaevent) != GC_SUCCESS) {
		checkError("gc_GetMetaEvent");
		return -1;
	}
	event = metaevent.evttype;
	switch(event)
	{
	case USR_EVENT_EXIT:
		logOut(EVT, "USER_EXIT_EVENT", 1);
		return USR_EVENT_EXIT;
		
	case GCEV_TASKFAIL:
		checkError("GCEV_TASKFAIL");
		break;
		
	case GCEV_OPENEX_FAIL:
		logOut(EVT, "GCEV_OPENEX_FAIL", 1);
		break;
	case GCEV_OPENEX:  
		logOut(EVT, "GCEV_OPENEX", 1);
		return USR_EVENT_OPEN_CMPT;
	case GCEV_LISTEN:
		logOut(EVT, "GCEV_LISTEN", 1);
		if(m_portState == OFFERED_STATE) {
			answer();
		}
		break;
	case GCEV_UNLISTEN:
		logOut(EVT, "GCEV_UNLISTEN", 1);
		break;
	case GCEV_BLOCKED:
		logOut(EVT, "GCEV_BLOCKED", 1);
		m_portState = BLOCKED_STATE;
		break;
		
	case GCEV_UNBLOCKED:
		long timeslot;
		SC_TSINFO tsinfo;
		tsinfo.sc_numts = 1;
		tsinfo.sc_tsarrayp = &timeslot;
		if(gc_GetXmitSlot(m_hDev, &tsinfo) != GC_SUCCESS)
			checkError("gc_GetXmitSlot");
		else
			m_tsn=timeslot;
		logOut(EVT,"GCEV_UNBLOCKED", 1);
		
		setDtmfTransferMode(IP_DTMF_TYPE_INBAND_RTP);
		/* Also supported: IP_DTMF_TYPE_USERINPUT_SIGNAL,IP_DTMF_TYPE_RFC_2833		  
		IP_DTMF_TYPE_INBAND_RTP,	IP_DTMF_TYPE_ALPHANUMERIC	 */
		enableCalls(true); 
		break;
	case GCEV_SETCHANSTATE:
		logOut(EVT,"GCEV_SETCHANSTATE",1);
		break;
	case GCEV_RESETLINEDEV:
		logOut(EVT,"GCEV_RESETLINEDEV", 1);
		m_portState = BLOCKED_STATE;
		break;
	case GCEV_ATTACH_FAIL:
		logOut(EVT,"GCEV_ATTACH_FAIL", 0);
		break;
	case GCEV_ATTACH:
		logOut(EVT,"GCEV_ATTACH", 0);
		break;
	case GCEV_DETACH:
		logOut(EVT,"GCEV_DETACH", 0);
		break;
	case GCEV_ALARM:
		logOut(EVT,"GCEV_ALARM", 1);
		break;
	case GCEV_DISCONNECTED:
		logOut(EVT,"GCEV_DISCONNECTED", 1);
		drop();
		break;
	case GCEV_DROPCALL:
		//			enableSingleDigit(false);
		m_dropFlag = 0;
		dx_stopch(m_hVox, EV_ASYNC);
		logOut(EVT,"GCEV_DROPCALL", 1);
		release();
		m_portState = NULL_STATE;
		ret_val = USER_DROPPED;
		break;
	case GCEV_OFFERED:		// incoming call
		if(m_portState == NULL_STATE)
		{
			m_playAttempt  = 0;
			logOut(EVT,"GCEV_OFFERED", 0);
			m_crn=metaevent.crn;
			m_portState = OFFERED_STATE;
			setCapabilities();
			dx_Listen(m_tsn);
			dti_Listen(m_tsv,EV_ASYNC);
		}
		else
			drop();
		break;
	case GCEV_PROCEEDING:
		logOut(EVT, "GCEV_PROCEEDING", 0);
		break;
	case GCEV_ALERTING:
		m_portState = ALERTING_STATE;
		logOut(EVT,"GCEV_ALERTING", 0);
		break;
	case GCEV_ACCEPT:
		logOut(EVT,"GCEV_ACCEPT", 0);
		m_portState = ACCEPTED_STATE;
		//			setCapabilities();
		answer();
		break;
	case GCEV_EXTENSIONCMPLT:
		logOut(EVT, "GCEV_EXTENSIONCMPLT", 0);
		pextensionBlk = (EXTENSIONEVTBLK *)(metaevent.extevtdatap);
		gcParmBlk = (&(pextensionBlk->parmblk));
		onExtension(gcParmBlk);
		break;
		
	case GCEV_EXTENSION:
		logOut(EVT,"GCEV_EXTENSION", 0);
		pextensionBlk = (EXTENSIONEVTBLK *)(metaevent.extevtdatap);
		gcParmBlk = (&(pextensionBlk->parmblk));
		onExtension(gcParmBlk);
		break;
		
	case GCEV_CONNECTED:	// Remote side answered (outgoing )
		logOut(EVT,"GCEV_CONNECTED", 1);
		m_portState = CONNECTED_STATE;
		ret_val = CALL_CONNECTED;
		break;
	case GCEV_ANSWERED:    // Application picked up ( incoming )
		logOut(EVT,"GCEV_ANSWERED", 1);
		m_portState = CONNECTED_STATE;
		ret_val = CALL_CONNECTED;
		break;
	case TDX_ERROR:
		checkError("TDX_ERROR");
		break;
		
	case TDX_PLAY:
		dx_fileclose(m_iott.io_fhandle);
		termno = ATDX_TERMMSK(m_hVox);
		sprintf(m_devTxt,"TDX_PLAY, termNo=%#lx, willDrop = %d", termno, m_dropFlag);
		logOut(EVT,m_devTxt, 0, true);
		if(m_portState != CONNECTED_STATE) {
			break;
		}
		if(m_dropFlag) {
			m_dropFlag = 0;
			drop();
		}
		else {
			if(m_playAttempt > 0)
				m_playAttempt--;
			getDigits();
		}
		break;
		
	case TDX_RECORD:
		sprintf(m_devTxt,"TDX_RECORD, termNo=%#lx", ATDX_TERMMSK(m_hVox));
		logOut(EVT,m_devTxt, 0, true);
		dx_fileclose(m_iott.io_fhandle);
		break;
		
	case TDX_PLAYTONE:
		sprintf(m_devTxt,"TDX_PLAYTONE, termNo=%#lx", ATDX_TERMMSK(m_hVox));
		logOut(EVT,m_devTxt, 0, true);
		break;
	case TDX_CST:
		cst = (DX_CST *) metaevent.evtdatap;
		sprintf(m_devTxt,"TDX_CST, cst.event = %#x, data = %#x", cst->cst_event,
			cst->cst_data);
		logOut(EVT,m_devTxt, 0, true);
		if(cst->cst_event == DE_DIGITS)
		{
			m_digBuf.dg_value[0] = (char) cst->cst_data;
			ret_val = SINGLE_DIGIT;
		}
		break;
	case TDX_GETDIG:	// get digits completed
		termno = ATDX_TERMMSK(m_hVox);
		sprintf(m_devTxt,"TDX_GETDIG, termNo=%#x, digits = %s", termno,
			m_digBuf.dg_value);
		logOut(EVT,m_devTxt, 1, true);
		if(m_portState == CONNECTED_STATE)
			ret_val = DIGITS_RECEIVED;
		break;
	case TDX_DIAL:
		sprintf(m_devTxt,"TDX_DIAL, termNo=%#x", ATDX_TERMMSK(m_hVox));
		
		break;
	default:
		sprintf(txt,"Unprocessed evt %#lx" ,event);
		logOut(EVT,m_devTxt, 0, true);
		break;
	}
	// Release this event
	return(ret_val);
}

//////////////////////////////////////////////////////////////////////
// Private methods
//////////////////////////////////////////////////////////////////////


//*****************************************************************************
// Function : 
//    CIpDev::restart()
// Purpose  : 
//    Resetting the channel
// Parameters:  
//    none
// Return type: 
//    int
//*****************************************************************************
int CIpDev::restart()
{
	if(gc_ResetLineDev(m_hDev, EV_ASYNC) != GC_SUCCESS)
		checkError("gc_ResetLineDev");
	else
		logOut(APP, "gc_ResetLineDev()", 1);
	return 0;
}

//*****************************************************************************
// Function : 
//    CIpDev::accept()
// Purpose  : 
//    Accepting the call.
// Parameters:  
//    none
// Return type: 
//    int
//*****************************************************************************
int CIpDev::accept()
{
	if(gc_AcceptCall(m_crn, 1, EV_ASYNC) != GC_SUCCESS)
		checkError("gc_AcceptCall");
	else
		logOut(APP, "gc_AcceptCall()", 1);
	return 0;
}

//*****************************************************************************
// Function : 
//    CIpDev::answer()
// Purpose  : 
//    Answering the call
// Parameters:  
//    none
// Return type: 
//    int
//*****************************************************************************
int CIpDev::answer()
{
	
	if(gc_AnswerCall(m_crn, 0, EV_ASYNC) != GC_SUCCESS)
		checkError("gc_AnswerCall");
	else
		logOut(APP, "gc_AnswerCall()", 1);
	return 0;
}

//*****************************************************************************
// Function : 
//    CIpDev::release()
// Purpose  : 
//    Sending a Release to CO
// Parameters:  
//    none
// Return type: 
//    int
//*****************************************************************************
int CIpDev::release()
{
	if(gc_ReleaseCall(m_crn) != GC_SUCCESS)
		checkError("gc_ReleaseCall()");
	else
		logOut(APP,"gc_ReleaseCall()", 1);
	return 0;
}

//*****************************************************************************
// Function : 
//    CIpDev::checkError()
// Purpose  : 
//    Checking the error for falure for the function being passed in
// Parameters:  
//    Parm1 <char * funcName><direction>
// Return type: 
//    none
//*****************************************************************************
void CIpDev::checkError(const char *funcName)
{
	GC_INFO t_Info;
	
	if(ATDV_LASTERR(m_hVox) == 0) // error occured on a network device
	{
		if(gc_ErrorInfo(&t_Info) != GC_SUCCESS)
		{
			sprintf(m_devTxt,"%s failed, Error Info unavailable", funcName);
			logOut(ERR1, m_devTxt, 1);
			return;
		}
		else
		{
			sprintf(m_devTxt, "%s failed, gcError = %s, ccError = %s",funcName, t_Info.gcMsg, t_Info.ccMsg);
			logOut(ERR1,m_devTxt, 1);
		}
	}
	else
	{
		sprintf(m_devTxt,"%s failed due to %s", funcName, ATDV_ERRMSGP(m_hVox));
		logOut(ERR1,m_devTxt, 1, 1);
	}
}




//*****************************************************************************
// Function : 
//    CIpDev::setCapabilities()
// Purpose  : 
//    Setting the IP Capabilities for the make call function later on.
// Parameters:  
//    none
// Return type: 
//    int
//*****************************************************************************
int CIpDev::setCapabilities()
{
	GC_PARM_BLKP pGcParmBlk = NULL;
	int duration = GC_SINGLECALL;
	IP_CAPABILITY capability = {0};
	
	//	printf("Setting capabilities for device\n");
	//Initialize capability information
	capability.type = GCCAPTYPE_AUDIO;
	capability.direction = IP_CAP_DIR_LCLTRANSMIT;
	
	
	capability.capability = GCCAP_AUDIO_g711Ulaw64k;
	capability.extra.audio.frames_per_pkt = 30;
	//	capability.capability = GCCAP_AUDIO_g7231;
	//	capability.extra.audio.frames_per_pkt = 1;
	capability.payload_type = IP_USE_STANDARD_PAYLOADTYPE;
	capability.extra.audio.VAD = GCPV_DISABLE;
	
	if(gc_util_insert_parm_ref(&pGcParmBlk, GCSET_CHAN_CAPABILITY,
		IPPARM_LOCAL_CAPABILITY, sizeof(IP_CAPABILITY),
		&capability) != GC_SUCCESS) 
	{
		printf("Error setting Transmit capabilities\n");
		return -1;
		
	}
	capability.direction = IP_CAP_DIR_LCLRECEIVE;
	if(gc_util_insert_parm_ref(&pGcParmBlk, GCSET_CHAN_CAPABILITY,
		IPPARM_LOCAL_CAPABILITY, sizeof(IP_CAPABILITY),
		&capability) != GC_SUCCESS) 
	{
		printf("Error setting Receive capabilities\n");
		return -1;
	}
	
	if(gc_SetUserInfo(GCTGT_GCLIB_CHAN,
		m_hDev,
		pGcParmBlk,
		duration) != GC_SUCCESS)
	{
		printf("Error gc_SetUserInfo()\n");
		return -1;
	}
	gc_util_delete_parm_blk(pGcParmBlk);
	return 0;
}



//*****************************************************************************
// Function	: 
//    CIpDev::getDigits()
// Purpose	: 
//    Get the digits that the caller entered
// Parameters:	
//    Parm1 <int timeout><direction>
// Return type:	
//    int
//*****************************************************************************
int CIpDev::getDigits(int timeout)
{
	m_tpt[0].tp_type   = IO_CONT;
	m_tpt[0].tp_termno = DX_DIGMASK;
	m_tpt[0].tp_length = DM_P;	// #
	m_tpt[0].tp_flags  = TF_DIGMASK;
	
	m_tpt[1].tp_type   = IO_CONT;
	m_tpt[1].tp_termno = DX_MAXTIME;
	m_tpt[1].tp_length = timeout;
	m_tpt[1].tp_flags  = TF_MAXTIME;
	
	m_tpt[2].tp_type   = IO_EOT;
	m_tpt[2].tp_termno = DX_IDDTIME;
	//m_tpt[2].tp_length = 60;// Maximum delay between digits
	m_tpt[2].tp_length = 100;// Maximum delay between digits
	m_tpt[2].tp_flags  = TF_IDDTIME;
	dx_Listen(m_tsn);
	m_digBuf.dg_value[0] = 0;

	if(dx_getdig(m_hVox,  m_tpt , &m_digBuf, EV_ASYNC) == 0) {
		logOut(APP, "dx_getdig() = 0", 1, true);
	}
	else {
		checkError("dx_getdig");
		return -1;
	}
	return 0;
}


//*****************************************************************************
// Function : 
//    CIpDev::onExtention()
// Purpose  : 
//    Find the specific data in the block structure
// Parameters:  
//    Parm1 <GC_PARM_BLKP parm_blk><direction>
// Return type: 
//    none
//*****************************************************************************
void CIpDev::onExtension(GC_PARM_BLKP parm_blk)
{
	GC_PARM_DATA* parmp = NULL;
	while ((parmp = gc_util_next_parm(parm_blk,parmp)) != NULL)
	{
		switch (parmp->set_ID)
		{
		case IPSET_CALLINFO:
			switch (parmp->parm_ID)
			{
			case IPPARM_CONNECTIONMETHOD:
				if(parmp->value_size != 0)
				{
					sprintf(m_devTxt, "IP data - IPPARM_CONNECTIONMETHOD: 0x%x", (*(int *)(parmp->value_buf)));
					logOut(INFO, m_devTxt, 0);
				}
				break;
			case IPPARM_H245TUNNELING:
				if(parmp->value_size != 0)
				{
					sprintf(m_devTxt, "IP data - IPPARM_H245TUNNELING: 0x%x",(*(int *)(parmp->value_buf)));
					logOut(INFO, m_devTxt, 0);
				}
				break;
			case IPPARM_DISPLAY:
				if(parmp->value_size != 0)
				{
					sprintf(m_devTxt, "IP data - DISPLAY: %s",parmp->value_buf);
					logOut(INFO, m_devTxt, 0);
					
				}
				break;
				
			case IPPARM_USERUSER_INFO:
				if(parmp->value_size != 0){
					sprintf(m_devTxt, "IP data - UUI: %s",parmp->value_buf);
					logOut(INFO, m_devTxt, 0);
				}
				break;
				
			case IPPARM_PHONELIST:
				if(parmp->value_size != 0)
				{
					sprintf(m_devTxt, "IP data - PHONELIST: %s",parmp->value_buf);
					logOut(INFO, m_devTxt, 0);
				}
				break;
				
			case IPPARM_CALLDURATION:
				if(parmp->value_size != 0)
				{
					sprintf(m_devTxt, "IP data - CALLDURATION: %d",(unsigned int)(*(parmp->value_buf)));
					logOut(INFO, m_devTxt, 0);
				}
				break;
			case IPPARM_RTCPINFO:
				if(parmp->value_size == sizeof(IP_RTCPINFO))
				{
					sprintf(m_devTxt, "IP data - RTCPINFO:timestamp %d,tx_packets %d,tx_octets %d send_indication %d",
						((IP_RTCPINFO *)(parmp->value_buf))->remote.sender.timestamp,
						((IP_RTCPINFO *)(parmp->value_buf))->remote.sender.tx_packets,
						((IP_RTCPINFO *)(parmp->value_buf))->remote.sender.tx_octets,
						((IP_RTCPINFO *)(parmp->value_buf))->remote.sender.send_indication);
					logOut(INFO, m_devTxt, 0);
				}
				break;
				
			default:
				break;
			}/* end switch (parmp->parm_ID) for IPSET_CALLINFO */
			break;
			
			
		case GCSET_CHAN_CAPABILITY:
			{
				IP_CAPABILITY *IPCapability;
				if (parmp->parm_ID == IPPARM_LOCAL_CAPABILITY)
				{
					if(parmp->value_size == sizeof(IP_CAPABILITY))
					{
						IPCapability = (IP_CAPABILITY *)(&(parmp->value_buf));
						sprintf(m_devTxt, "IP data - CAPABILITY: capability %d,type %d, direction %d",
							IPCapability->capability,
							IPCapability->type,
							IPCapability->direction);
						logOut(INFO, m_devTxt, 0);
					}
				}
			}
			break;
			
		case IPSET_VENDORINFO:
			switch (parmp->parm_ID)
			{
			case IPPARM_H221NONSTD:
				{
					if(parmp->value_size == sizeof(IP_H221NONSTANDARD))
					{
						IP_H221NONSTANDARD *pH221NonStandard;
						pH221NonStandard = (IP_H221NONSTANDARD *)(&(parmp->value_buf));
						sprintf(m_devTxt, "IP data - H221NONSTANDARD: country_code %d,\n\textension %d, manufacturer_code %d",
							pH221NonStandard->country_code,
							pH221NonStandard->extension,
							pH221NonStandard->manufacturer_code);
						logOut(INFO, m_devTxt, 0);
					}
				}
				break;
				
			case IPPARM_VENDOR_PRODUCT_ID:
				if(parmp->value_size != 0)
				{
					sprintf(m_devTxt, "IP data -IPPARM_VENDOR_PRODUCT_ID:\n \t%s",parmp->value_buf);
					logOut(INFO, m_devTxt, 0);
				}
				break;
				
			case IPPARM_VENDOR_VERSION_ID:
				if(parmp->value_size != 0)
				{
					sprintf(m_devTxt, "IP data -IPPARM_VENDOR_VERSION_ID %s",parmp->value_buf);
					logOut(INFO, m_devTxt, 0);
				}
				break;
				
			default:
				break;
				
			}/* end switch (parmp->parm_ID) for IPSET_VENDORINFO */
			break;
			
			
			
		case IPSET_MSG_Q931:
			
			switch (parmp->parm_ID)
			{
			case IPPARM_MSGTYPE:
				
				switch ((*(int *)(parmp->value_buf)))
				{
				case IP_MSGTYPE_Q931_FACILITY:
					sprintf(m_devTxt, "IP data - IP_MSGTYPE_Q931_FACILITY");
					
					break;
					
				case IP_MSGTYPE_Q931_INFORMATION:
					sprintf(m_devTxt, "IP data - IP_MSGTYPE_Q931_INFORMATION");
					break;
					
				case IP_MSGTYPE_Q931_PROGRESS:
					sprintf(m_devTxt, "IP data - IP_MSGTYPE_Q931_PROGRESS");
					break;
					
				default:
					sprintf(m_devTxt, "Unknown IPPARM_MSGTYPE");
					break;
				} /* end  switch ((int)(parmp->value_buf)) */
				logOut(INFO, m_devTxt, 0);
				
			}/* end switch (parmp->parm_ID) for IPSET_MSG_Q931 */
			break;
			
			
		case IPSET_MSG_H245:
			
			switch (parmp->parm_ID)
			{
			case IPPARM_MSGTYPE:
				
				switch ((*(int *)(parmp->value_buf)))
				{
				case IP_MSGTYPE_H245_INDICATION:
					
					sprintf(m_devTxt, "IP data - IP_MSGTYPE_H245_INDICATION");
					break;
					
				case IP_MSGTYPE_H245_COMMAND:
					sprintf(m_devTxt, "IP data - IP_MSGTYPE_H245_COMMAND");
					break;
					
				case IP_MSGTYPE_H245_REQUEST:
					sprintf(m_devTxt, "IP data - IP_MSGTYPE_H245_REQUEST");
					break;
					
				case IP_MSGTYPE_H245_RESPONSE:
					sprintf(m_devTxt, "IP data - IP_MSGTYPE_H245_RESPONSE");
					break;
					
				case IP_MSGTYPE_H245_UNDEFINED:
					sprintf(m_devTxt, "IP data - IP_MSGTYPE_H245_UNDEFINED");
					break;
					
				default:
					sprintf(m_devTxt, "Unknown IPPARM_MSGTYPE");
					break;
				} /* end  switch ((int)(parmp->value_buf)) */
				
				logOut(INFO, m_devTxt, 0);
				
			}/* end switch (parmp->parm_ID) for IPSET_MSG_H245 */
			break;
			
		case IPSET_USERINPUTINDICATION:
			
			switch (parmp->parm_ID)
			{
				
			case IPPARM_UII_ALPHANUMERIC:
				sprintf(m_devTxt, "IP data -  IPPARM_UII_ALPHANUMERIC: %s",parmp->value_buf);
				break;
				
			default:
				sprintf(m_devTxt, "Unknown IPSET_USERINPUTINDICATION"); 
				break;
				
			}/* end switch (parmp->parm_ID) for IPSET_USERINPUTINDICATION */
			logOut(INFO, m_devTxt, 0);
			break;
			
		case IPSET_NONSTANDARDDATA:
			switch (parmp->parm_ID)
			{
				
			case IPPARM_NONSTANDARDDATA_DATA:
				sprintf(m_devTxt, "IP data -  IPPARM_NONSTANDARDDATA_DATA: %s",parmp->value_buf);
				break;
				
			case IPPARM_NONSTANDARDDATA_OBJID:
				
				sprintf(m_devTxt, "IP data -  IPPARM_NONSTANDARDDATA_OBJID: %s",parmp->value_buf);
				break;
				
			default:
				break;
				
			}/* end switch (parmp->parm_ID) for IPSET_NONSTANDARDDATA */
			logOut(INFO, m_devTxt, 0);
			
			break;
		}/* end switch (parmp->set_ID) */
	}
}


//*****************************************************************************
// Function : 
//    CIpDev::GetDigBuffer()
// Purpose  : 
//    Return the buffer that contains the digits, trimming the trailing '#'
// Parameters:  
//    none
// Return type: 
//    const char *
//*****************************************************************************
const char * CIpDev::getDigBuffer()
{
	char * poundSign = strchr(m_digBuf.dg_value, '#');
	if(poundSign)
		*poundSign = 0; // strip the # sign, if any
	return m_digBuf.dg_value;
}


//*****************************************************************************
// Function : 
//    CIpDev::logOut()
// Purpose  : 
//    Logging information to the Log file.
// Parameters:  
//    Parm1 <int type><input>
//    Parm2 <char * info><input>
//    Parm3 < int printConsoleNum> <input> - number of console window to pu the info
//    Parm4 < bool voxDevInfo ><input>
// Return type: 
//    none
//*****************************************************************************
void CIpDev::logOut(int type, const char * info, int printConsoleNum, bool voxDevInfo)
{
	char * tmpName = (voxDevInfo)? m_voxDevName : m_netDevName;
	appLog->addToLog(type, tmpName, info);
#if 0
	appConsole->printInfo(type, tmpName, info, printConsoleNum);    
#endif
	printInfo(type, tmpName, info, brdStatusWindow); 
	//	scrn_wrefresh(brdStatusWindow);
}


//*****************************************************************************
// 
// Function : 
//    CIpDev::setDtmfTransferMode()
// Purpose  : 
//    Allows to change a DTMF mode for H323/SIP 
// Parameters:  
//    Parm1 <char xferMode><direction>
// Return type: 
//    int
//*****************************************************************************
int CIpDev::setDtmfTransferMode(char xferMode)
{
	// IP_DTMF_TYPE_INBAND_RTP
	GC_PARM_BLKP parmblkp = NULL;
	gc_util_insert_parm_val(&parmblkp,IPSET_DTMF,IPPARM_SUPPORT_DTMF_BITMASK,
		sizeof(char),	xferMode);
	int retval = gc_SetUserInfo(GCTGT_GCLIB_CHAN, m_hDev, parmblkp, GC_ALLCALLS);
	gc_util_delete_parm_blk(parmblkp); 
	
	if (retval != GC_SUCCESS) 
		checkError("gc_SetUserInfo(IP_DTMF_TYPE)");
	else
		logOut(APP, "gc_SetUserInfo(IP_DTMF_TYPE) = OK", 1);
	return 0;
}

//////////////////////////////////////////////////////////////////////
// End of IpDev.cpp
//////////////////////////////////////////////////////////////////////
