/**
* @file ipchannel.cpp
* @brief Multimedia demo call control
* @date Feb 16, 2007
*
* DIALOGIC CONFIDENTIAL 
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to 
* the source code ("Material") are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its 
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced, 
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual 
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

// Application Level Include Files
#include "ipchannel.h"
#include "connection.h"
#include "config.h"

// Extern Variables
extern bool g_bDone;
extern bool g_bAnswerThenDrop;
extern int  g_nCallType;
const  char g_cDtmfMode = IP_DTMF_TYPE_INBAND_RTP;
extern char g_cPhoneNumber[31];
extern char g_cProxyIp[20];
extern int  g_nSipUdpPort;
extern bool g_bUseSipInfoDtmf;
extern bool g_bWaitOnSIPAck;
extern int  g_nDtmfDetectMode;

// Global Variables
device_type CIPChannel::s_eType(IP_DEVICE);
char        s_brdDevName[DEV_NAME] = ":N_iptB1:P_IP";

bool    CIPChannel::s_bRegistered = false;           
bool    CIPChannel::s_bBrdOpened  = false;
long    CIPChannel::s_gcBrdDev    = 0;
int CIPChannel::s_nRegister_TTL = 60; // seconds to resend REGISTER

// State Table Structure for IP Call Control Events
CIPChannel::stateTransition *CIPChannel::s_ipCallTable[] = 
{
    CIPChannel::s_initialTable,
    CIPChannel::s_openedTable,
    CIPChannel::s_nullTable,
    CIPChannel::s_proceedingTable,
    CIPChannel::s_offeredTable,
    CIPChannel::s_alertingTable,
    CIPChannel::s_acceptedTable,
    CIPChannel::s_connectedTable,
    CIPChannel::s_disconnectedTable,
    CIPChannel::s_usrDisconnectedTable,
    CIPChannel::s_idleTable,
    CIPChannel::s_appStoppingTable,
};


// state table definition for the connections
CIPChannel::stateTransition CIPChannel::s_initialTable[] = // 0
{
    { GCEV_OPENEX,          &CIPChannel::OnOpenEx,      CIPChannel::state_opened,       "state_initial"},
    { GCEV_OPENEX_FAIL,         NULL,                   CIPChannel::state_initial,  "state_initial"},
    { GCEV_ERROR,               NULL,                   CIPChannel::state_initial,  "state_initial"},
    { GCEV_UNBLOCKED,       NULL,                   CIPChannel::state_initial,  "state_initial"},
    { GCEV_BLOCKED,             NULL,                   CIPChannel::state_initial,  "state_initial"},
    { GCEV_TASKFAIL,            &CIPChannel::OnTaskFail,    CIPChannel::state_initial,  "state_initial"},
    { GCEV_SETCONFIGDATA,       &CIPChannel::OnSetConfigData, CIPChannel::state_initial,       "state_initial"},
    { UNDECLARED,              &CIPChannel::OnUndeclared, CIPChannel::state_initial,   "state_initial"}
};

CIPChannel::stateTransition CIPChannel::s_openedTable[] = // 1
{
    { GCEV_UNBLOCKED,       &CIPChannel::mmWaitCall,        CIPChannel::state_null,         "state_opened"},
    { GCEV_OPENEX_FAIL,         NULL,                       CIPChannel::state_opened,       "state_opened"},
    { GCEV_ERROR,               NULL,                       CIPChannel::state_opened,       "state_opened"},
    { GCEV_BLOCKED,             NULL,                       CIPChannel::state_opened,       "state_opened"},
    { GCEV_TASKFAIL,           &CIPChannel::OnTaskFail,     CIPChannel::state_opened,       "state_opened"},
    { GCEV_SETCONFIGDATA,      &CIPChannel::OnSetConfigData,  CIPChannel::state_opened,       "state_opened"},
    { UNDECLARED,              &CIPChannel::OnUndeclared,   CIPChannel::state_opened,       "state_opened"}
};

CIPChannel::stateTransition CIPChannel::s_nullTable[] = // 2
{
    { GCEV_OFFERED,             &CIPChannel::OnOffered,         CIPChannel::state_offered,  "state_null"},
    { GCEV_LISTEN,          &CIPChannel::SetListenFlag,     CIPChannel::state_null,         "state_null"},
    { GCEV_ALERTING,            NULL,                       CIPChannel::state_alerting,     "state_null"},
    { GCEV_CONNECTED,       &CIPChannel::OnConnected,       CIPChannel::state_connected,    "state_null"},
    { GCEV_PROCEEDING,          NULL,                       CIPChannel::state_proceeding, "state_null"},
    { GCEV_DISCONNECTED,    &CIPChannel::mmDropCall,        CIPChannel::state_disconnected, "state_null"},
    { GCEV_ERROR,               &CIPChannel::mmDropCall,        CIPChannel::state_disconnected, "state_null"},
//    { GCEV_ERROR,               NULL,                       CIPChannel::state_null,         "state_null"},
    { GCEV_TASKFAIL,            &CIPChannel::OnTaskFail,        CIPChannel::state_null,         "state_null"},
    { GCEV_BLOCKED,             NULL,                       CIPChannel::state_opened,       "state_null"},
    { GCEV_UNLISTEN,            &CIPChannel::OnUnlisten,        CIPChannel::state_null,         "state_null"},
    { GCEV_DROPCALL,            &CIPChannel::mmReleaseCall,     CIPChannel::state_idle,         "state_null"},
    { GCEV_RESETLINEDEV,    &CIPChannel::mmWaitCall,        CIPChannel::state_null,         "state_null"},
    { GCEV_SETCONFIGDATA,       &CIPChannel::OnSetConfigData,   CIPChannel::state_null,       "state_null"},
    { UNDECLARED,               &CIPChannel::OnUndeclared,  CIPChannel::state_null,         "state_null"}
};

CIPChannel::stateTransition CIPChannel::s_offeredTable[] = // 3
{
    { GCEV_LISTEN,          &CIPChannel::OnListen,          CIPChannel::state_offered,  "state_offered"},
    { GCEV_CALLINFO,            &CIPChannel::OnGetCallInfo,     CIPChannel::state_offered,  "state_offered"},
    { GCEV_EXTENSIONCMPLT,      &CIPChannel::OnExtensionEvent,  CIPChannel::state_offered,  "state_offered"},
    { GCEV_EXTENSION,       &CIPChannel::OnExtensionEvent,  CIPChannel::state_offered,  "state_offered"},
    { GCEV_ACCEPT,          &CIPChannel::OnAccept,          CIPChannel::state_accepted,     "state_offered"},
    //{ GCEV_ANSWERED,            &CIPChannel::OnAnswered,        CIPChannel::state_connected,    "state_offered"},
    { GCEV_ANSWERED,            &CIPChannel::OnAnswered,        CIPChannel::state_offered,    "state_offered"},
    { GCEV_SIP_ACK,            &CIPChannel::OnSIPAck,        CIPChannel::state_connected,    "state_offered"},
    { GCEV_DISCONNECTED,    &CIPChannel::mmDropCall,        CIPChannel::state_disconnected, "state_offered"},
    { GCEV_DROPCALL,            &CIPChannel::mmReleaseCall,     CIPChannel::state_idle,         "state_offered"},
    { GCEV_ERROR,               &CIPChannel::mmDropCall,        CIPChannel::state_disconnected, "state_offered"},
    { GCEV_TASKFAIL,            &CIPChannel::OnTaskFail,        CIPChannel::state_disconnected, "state_offered"},
    { GCEV_CALLPROC,            NULL,                       CIPChannel::state_offered,  "state_offered"},
    { GCEV_BLOCKED,             NULL,                       CIPChannel::state_opened,       "state_offered"},
    { GCEV_EXTENSION,       NULL,                       CIPChannel::state_offered,  "state_offered"},
    { GCEV_RESETLINEDEV,    &CIPChannel::mmWaitCall,        CIPChannel::state_null,         "state_offered"},
    { UNDECLARED,               &CIPChannel::OnUndeclared,  CIPChannel::state_offered,  "state_offered"}
};

CIPChannel::stateTransition CIPChannel::s_acceptedTable[] = // 5
{
    { GCEV_LISTEN,          &CIPChannel::OnListen,          CIPChannel::state_offered,  "state_accepted"},
    //{ GCEV_ANSWERED,            &CIPChannel::OnAnswered,        CIPChannel::state_connected,    "state_accepted"},
    { GCEV_ANSWERED,            &CIPChannel::OnAnswered,        CIPChannel::state_offered,    "state_accepted"},
    { GCEV_SIP_ACK,            &CIPChannel::OnSIPAck,        CIPChannel::state_connected,    "state_accepted"},
    { GCEV_CALLINFO,            &CIPChannel::OnGetCallInfo,     CIPChannel::state_accepted,     "state_accepted"}, 
    { GCEV_EXTENSIONCMPLT,      &CIPChannel::OnExtensionEvent,  CIPChannel::state_accepted,     "state_accepted"},
    { GCEV_EXTENSION,       &CIPChannel::OnExtensionEvent,  CIPChannel::state_accepted,     "state_accepted"},
    { GCEV_DISCONNECTED,    &CIPChannel::mmDropCall,        CIPChannel::state_disconnected, "state_accepted"},
    { GCEV_DROPCALL,            &CIPChannel::mmReleaseCall,     CIPChannel::state_idle,         "state_accepted"},
    { GCEV_ERROR,               &CIPChannel::mmDropCall,        CIPChannel::state_disconnected, "state_accepted"},
    { GCEV_TASKFAIL,            &CIPChannel::OnTaskFail,        CIPChannel::state_disconnected, "state_accepted"},
    { GCEV_BLOCKED,             NULL,                       CIPChannel::state_opened,       "state_accepted"},
    { GCEV_RESETLINEDEV,    &CIPChannel::mmWaitCall,        CIPChannel::state_null,         "state_accepted"},
    { UNDECLARED,               &CIPChannel::OnUndeclared,  CIPChannel::state_accepted,     "state_accepted"}
};

CIPChannel::stateTransition CIPChannel::s_connectedTable[] = // 6
{
    { GCEV_LISTEN,          &CIPChannel::SetListenFlag,     CIPChannel::state_connected,    "state_connected"},
    { GCEV_UNLISTEN,            &CIPChannel::OnUnlisten,        CIPChannel::state_connected,    "state_connected"},
    { GCEV_DISCONNECTED,    &CIPChannel::mmDropCall,        CIPChannel::state_disconnected, "state_connected"},
    { GCEV_EXTENSION,       &CIPChannel::OnExtensionEvent,  CIPChannel::state_connected,    "state_connected"},
    { GCEV_EXTENSIONCMPLT,      &CIPChannel::OnExtensionEvent,  CIPChannel::state_connected,    "state_connected"},
    { GCEV_DROPCALL,            &CIPChannel::mmReleaseCall,     CIPChannel::state_idle,         "state_connected"},
    { GCEV_ERROR,               &CIPChannel::mmDropCall,        CIPChannel::state_disconnected, "state_connected"},
    { GCEV_TASKFAIL,            &CIPChannel::OnTaskFail,        CIPChannel::state_disconnected, "state_connected"},
    { GCEV_BLOCKED,             NULL,                       CIPChannel::state_opened,       "state_connected"},
    { GCEV_RESETLINEDEV,    &CIPChannel::mmWaitCall,        CIPChannel::state_null,         "state_connected"},
    { GCEV_CALLINFO,            &CIPChannel::OnGetCallInfo,     CIPChannel::state_connected,    "state_connected"},
    { UNDECLARED,               &CIPChannel::OnUndeclared,  CIPChannel::state_connected,    "state_connected"}
};

CIPChannel::stateTransition CIPChannel::s_proceedingTable[] = // 7
{
    { GCEV_ALERTING,            NULL,                       CIPChannel::state_alerting,     "state_proceeding"},
    { GCEV_EXTENSION,       &CIPChannel::OnExtensionEvent,  CIPChannel::state_proceeding, "state_proceeding"},
    { GCEV_LISTEN,          &CIPChannel::SetListenFlag,     CIPChannel::state_proceeding, "state_proceeding"},
    { GCEV_EXTENSIONCMPLT,      &CIPChannel::OnExtensionEvent,  CIPChannel::state_proceeding, "state_proceeding"},
    { GCEV_DISCONNECTED,    &CIPChannel::mmDropCall,        CIPChannel::state_disconnected, "state_proceeding"},
    { GCEV_CONNECTED,       &CIPChannel::OnConnected,       CIPChannel::state_connected,    "state_proceeding"},
    { GCEV_ERROR,               &CIPChannel::mmDropCall,        CIPChannel::state_disconnected, "state_proceeding"},
    { GCEV_TASKFAIL,            &CIPChannel::OnTaskFail,        CIPChannel::state_disconnected, "state_proceeding"},
    { GCEV_BLOCKED,             NULL,                       CIPChannel::state_opened,       "state_proceeding"},
    { GCEV_RESETLINEDEV,    &CIPChannel::mmWaitCall,        CIPChannel::state_null,         "state_proceeding"},
    { GCEV_CALLINFO,            &CIPChannel::OnGetCallInfo,     CIPChannel::state_proceeding, "state_proceeding"},
    { UNDECLARED,               &CIPChannel::OnUndeclared,  CIPChannel::state_proceeding, "state_proceeding"}
};

CIPChannel::stateTransition CIPChannel::s_alertingTable[] = // 8
{
    { GCEV_LISTEN,          &CIPChannel::SetListenFlag,     CIPChannel::state_alerting,     "state_alerting"},
    { GCEV_UNLISTEN,            &CIPChannel::OnUnlisten,        CIPChannel::state_disconnected, "state_alerting"},
    { GCEV_CONNECTED,       &CIPChannel::OnConnected,       CIPChannel::state_connected,    "state_alerting"},
    { GCEV_DISCONNECTED,    &CIPChannel::mmDropCall,        CIPChannel::state_disconnected, "state_alerting"},
    { GCEV_EXTENSIONCMPLT,      &CIPChannel::OnExtensionEvent,  CIPChannel::state_alerting,     "state_alerting"},
    { GCEV_EXTENSION,       &CIPChannel::OnExtensionEvent,  CIPChannel::state_alerting,     "state_alerting"},
    { GCEV_DROPCALL,            &CIPChannel::mmReleaseCall,     CIPChannel::state_idle,         "state_alerting"},
    { GCEV_ERROR,               &CIPChannel::mmDropCall,        CIPChannel::state_disconnected, "state_alerting"},
    { GCEV_TASKFAIL,            &CIPChannel::OnTaskFail,        CIPChannel::state_disconnected, "state_alerting"},
    { GCEV_BLOCKED,             NULL,                       CIPChannel::state_opened,       "state_alerting"},
    { GCEV_RESETLINEDEV,    &CIPChannel::mmWaitCall,        CIPChannel::state_null,         "state_alerting"},
    { UNDECLARED,               &CIPChannel::OnUndeclared,  CIPChannel::state_alerting,     "state_alerting"}
};

CIPChannel::stateTransition CIPChannel::s_disconnectedTable[] = // 9
{
    { GCEV_UNLISTEN,            &CIPChannel::OnUnlisten,        CIPChannel::state_disconnected, "state_disconnected"},
    { GCEV_DROPCALL,            &CIPChannel::mmReleaseCall,     CIPChannel::state_idle,         "state_disconnected"},
    { GCEV_ERROR,               &CIPChannel::mmDropCall,        CIPChannel::state_disconnected, "state_disconnected"},
    { GCEV_TASKFAIL,            &CIPChannel::OnTaskFail,        CIPChannel::state_disconnected, "state_disconnected"},
    { GCEV_BLOCKED,             NULL,                       CIPChannel::state_opened,       "state_disconnected"},
    { GCEV_EXTENSION,       &CIPChannel::OnExtensionEvent,  CIPChannel::state_disconnected, "state_disconnected"},
    { GCEV_EXTENSIONCMPLT,      &CIPChannel::OnExtensionEvent,  CIPChannel::state_disconnected, "state_disconnected"},
    { GCEV_RESETLINEDEV,    &CIPChannel::mmWaitCall,        CIPChannel::state_null,         "state_disconnected"},
    { UNDECLARED,               &CIPChannel::OnUndeclared,  CIPChannel::state_disconnected, "state_disconnected"}
};

CIPChannel::stateTransition CIPChannel::s_usrDisconnectedTable[] = // 10
{
    { GCEV_DROPCALL,            &CIPChannel::mmReleaseCall,     CIPChannel::state_idle,             "state_usrDisconnected"},
    { GCEV_ERROR,               &CIPChannel::mmReleaseCall,     CIPChannel::state_idle,             "state_usrDisconnected"},
    { GCEV_TASKFAIL,            &CIPChannel::OnTaskFail,        CIPChannel::state_idle,             "state_usrDisconnected"},
    { GCEV_BLOCKED,             NULL,                       CIPChannel::state_opened,           "state_usrDisconnected"},
    { GCEV_EXTENSION,       NULL,                       CIPChannel::state_usrDisconnected,  "state_usrDisconnected"},
    { GCEV_RESETLINEDEV,    &CIPChannel::mmWaitCall,        CIPChannel::state_null,             "state_usrDisconnected"},
    { UNDECLARED,               &CIPChannel::OnUndeclared,  CIPChannel::state_usrDisconnected,  "state_usrDisconnected"}
};

CIPChannel::stateTransition CIPChannel::s_idleTable[] = // 11
{
    { GCEV_RELEASECALL,         &CIPChannel::OnRelease,         CIPChannel::state_null,         "state_idle"},
    { GCEV_UNLISTEN,            &CIPChannel::OnUnlisten,        CIPChannel::state_idle,         "state_idle"},
    { GCEV_ERROR,               &CIPChannel::OnRelease,         CIPChannel::state_null,         "state_idle"},
    { GCEV_TASKFAIL,            &CIPChannel::OnTaskFail,        CIPChannel::state_null,         "state_idle"},
    { GCEV_DROPCALL,            &CIPChannel::mmReleaseCall,     CIPChannel::state_idle,         "state_idle"},
    { GCEV_BLOCKED,             NULL,                       CIPChannel::state_opened,       "state_idle"},
    { GCEV_EXTENSION,       NULL,                       CIPChannel::state_idle,         "state_idle"},
    { GCEV_RESETLINEDEV,    &CIPChannel::mmWaitCall,        CIPChannel::state_null,         "state_idle"},
    { UNDECLARED,               &CIPChannel::OnUndeclared,  CIPChannel::state_idle,         "state_idle"}
};

CIPChannel::stateTransition CIPChannel::s_appStoppingTable[] = // 12
{
    { GCEV_RESETLINEDEV,    NULL,                       CIPChannel::state_appStopping,  "state_appStopping"},
    { GCEV_ERROR,               &CIPChannel::mmDropCall,        CIPChannel::state_disconnected, "state_appStopping"},
    { GCEV_TASKFAIL,            &CIPChannel::OnTaskFail,        CIPChannel::state_disconnected, "state_appStopping"},
    { GCEV_BLOCKED,             NULL,                       CIPChannel::state_opened,       "state_appStopping"},
    { GCEV_EXTENSION,       NULL,                       CIPChannel::state_appStopping,  "state_appStopping"},
    { UNDECLARED,              &CIPChannel::OnUndeclared,  CIPChannel::state_appStopping,  "state_appStopping"}
};


// Function Descriptions


//*****************************************************************************
// 		  NAME : CIPChannel::CIPChannel(const char *pDevName, 
// 							char *pDestIp, unsigned long ulConnectionId,
// 							CMMStream *pMMChan)
// DESCRIPTION : CIPChannel Constructor
// 		 INPUT : Device Name, Dest IP Addr, Connection Id and MMStream object
// 		 OUTPUT : None
// 		RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
CIPChannel::CIPChannel(const    char *pDevName, 
                       char *pDestIp, 
                       unsigned long ulConnectionId,
                       CMMStream *pMMChan)
:   m_gcDev(-1),
m_talkTs(-1),
m_nCount(0),
m_nConnectionId(ulConnectionId),
m_coderCounter(0),
m_bCallInProgress(false),
m_bCallDropped(false),
m_bListening(false),
m_bIsFree(true),
m_pMediaChan(pMMChan)
{
    strcpy(m_devName, pDevName);
    strcpy(m_cDial, pDestIp);
    memset(m_cOrigIP, 0, GC_ADDRSIZE);

    m_state_name = "state_initial";
    m_eState = state_initial;
    m_callDuration = 0;
    m_nConnects = 0;
    m_nFailedCalls = 0;
    m_callStart = 0;
    m_crn = -1;
    m_bCallAttempt = false;
    m_crnForCurrentCall = -1;
    m_cOriginationNum[0] = '\0';
    m_SDP.clear();
		m_bSdpReceived=false;
}

//*****************************************************************************
// 		  NAME : CIPChannel::~CIPChannel()
// DESCRIPTION : CIPChannel Destructor
// 		 INPUT : None
// 		 OUTPUT : None
// 		RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
CIPChannel::~CIPChannel()
{
}

//*****************************************************************************
// 		  NAME : bool CIPChannel::Init()
// DESCRIPTION : Opens the board and devices and registers them
// 		 INPUT : None
// 		 OUTPUT : None
// 		RETURNS : bool - True if the gc_OpenEx succeeded
// 	  CAUTIONS : None
//*****************************************************************************
bool CIPChannel::Init()
{
    // open a board level device first for registration
    if (s_bBrdOpened == false) {
        if ((gc_OpenEx(&(s_gcBrdDev), s_brdDevName, EV_SYNC, (void*)IP_BOARD_DEVICE)) < 0) {
            mmReport(ERROR_GCALL, IP_DEVICE, "gc_OpenEx() failed on %s", s_brdDevName);
            return false;
        }
        mmReport(INFO_DEBUG, IP_DEVICE, "Opened IP %s, dev=%d", s_brdDevName, s_gcBrdDev);
        s_bBrdOpened = true;
#ifdef TDM_AUDIO
        // Initialze GC for a Call Info SIP header
        GC_PARM_BLKP parmblkp = NULL;
        gc_util_insert_parm_ref(   &parmblkp,
                                   IPSET_CONFIG,
                                   IPPARM_REGISTER_SIP_HEADER,
                                   (unsigned char)strlen("TextLine1") + 1,
                                   (void*)"CallInfo");

        long request_id = 0;
        // This should be called async, but as we're only opening a board at this point...
        if (gc_SetConfigData(   GCTGT_CCLIB_NETIF, 
                                s_gcBrdDev,
                                parmblkp,
                                0,
                                GCUPDATE_IMMEDIATE,
                                &request_id,
                                EV_ASYNC) != GC_SUCCESS) {
            mmReport(INFO_ERROR, s_eType, "Error in gc_SetConfigData() call");
        }
        gc_util_delete_parm_blk( parmblkp);
#endif	
    }

    //Register();
    // Open devices
    if ((gc_OpenEx(&(m_gcDev), m_devName, EV_ASYNC, (void*)IP_DEVICE)) < 0) {
        mmReport(ERROR_GCALL, IP_DEVICE, "gc_OpenEx() failed on %s", m_devName);
        return false;
    }
    mmReport(INFO_DEBUG, IP_DEVICE, "Opened IP %s, dev=%d", m_devName, m_gcDev);

    return true;
}

//*****************************************************************************
// 		  NAME : void CIPChannel::ProcessEvent(METAEVENT metaEvt, 
// 														CIPChannel *pChan)
// DESCRIPTION : Event Handler Function
// 		 INPUT : MetaEvt data, Pointer to the IPChannel where event occured
// 		 OUTPUT : None
// 		RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CIPChannel::ProcessEvent(METAEVENT metaEvt, CIPChannel *pChan)
{
    pChan->SetMetaEvt(metaEvt);

    CIPChannel::stateTransition* pTable = 0;
    CIPChannel::APPHANDLER ipCallAppHdlr = 0;

    // Get the table that contains this state.
    pTable = (s_ipCallTable)[pChan->GetState()];

    // Find the entry that matches the current state event.
    for ( ; ; ) {

        if (pTable->st_event == metaEvt.evttype)
            break;

        else if (pTable->st_event == UNDECLARED)
            break;

        else
            pTable++;
    }

    pChan->m_state_name = pTable->st_name;
    mmReport(INFO_DEBUG, s_eType, "--- [%s] Got event %s for IP (crn=0x%x) in state = %s ----", pChan->m_devName, mmEvt2Str(metaEvt.evttype), pChan->m_crn, pChan->m_state_name);

    // Call the appropriate handler function
    ipCallAppHdlr = pTable->st_action;

    if (ipCallAppHdlr)
        (pChan->*ipCallAppHdlr)(metaEvt.extevtdatap);
    else
        mmReport(INFO_ERROR, s_eType, "Invalid APP_HANDLER.......");


    pChan->SetState(pTable->st_next);
    return;
}

//*****************************************************************************
// 		  NAME : bool CIPChannel::Shutdown()
// DESCRIPTION : Cleanup Routine
// 		 INPUT : None
// 		 OUTPUT : None
// 		RETURNS : Bool - True
// 	  CAUTIONS : None
//*****************************************************************************
bool CIPChannel::Shutdown()
{
    //DeRegister();
    // Reset line device
    mmRestart(NULL);
    if (m_gcDev > 0) {
        gc_Close(m_gcDev);
        m_gcDev = 0;
    }
    return true;
}

//*****************************************************************************
// 		  NAME : bool CIPChannel::Register()
// DESCRIPTION : Register 
// 		 INPUT : None
// 		 OUTPUT : None
// 		RETURNS : Bool - True if the function succeeded or False otherwise
// 	  CAUTIONS : None
//*****************************************************************************
bool CIPChannel::Register()
{
    // Local Variable Declaration 	
    int                 frc         = GC_SUCCESS;
    bool                bOk         = true;
    char                genbuf[256] = "";
    GC_PARM_BLKP        pParmBlock  = NULL;
    IP_REGISTER_ADDRESS registerAddress;
    unsigned long       serviceID;

    mmReport(INFO_DEBUG, s_eType, "Register()");

    // Register with a SIP Proxy only if the cfg file has a valid IP address - non-zero IP
    if (strcmp(g_cProxyIp, "0.0.0.0") == 0) {
        return false;
    }

    // making sure we do have a brd handle
    if (s_bBrdOpened == false)
        return false;

    // we want to register just once
    if (s_bRegistered == true)
        return true;

    if (gc_util_insert_parm_val(&pParmBlock,
                                GCSET_SERVREQ,
                                PARM_REQTYPE,
                                sizeof(char),
                                IP_REQTYPE_REGISTRATION) < 0) {
        mmReport(ERROR_GCALL, s_eType, "Register() -> gc_util_insert_parm_val() PARM_REQTYPE failed");
        bOk = false;
    }

    if (gc_util_insert_parm_val(&pParmBlock,
                                GCSET_SERVREQ,
                                PARM_ACK,
                                sizeof(char),
                                1) < 0) {
        mmReport(ERROR_GCALL, s_eType, "Register() -> gc_util_insert_parm_val() PARM_ACK failed");
        bOk = false;
    }

    // Setting the protocol target
    if (gc_util_insert_parm_val(&pParmBlock,
                                IPSET_PROTOCOL,
                                IPPARM_PROTOCOL_BITMASK,
                                sizeof(char),
                                IP_PROTOCOL_SIP) < 0) {
        mmReport(ERROR_GCALL, s_eType, "Register() -> gc_util_insert_parm_val() IPPARM_PROTOCOL_BITMASK failed");
        bOk = false;
    }

    // Setting the operation to perform 
    if (gc_util_insert_parm_val(&pParmBlock,
                                IPSET_REG_INFO,
                                IPPARM_OPERATION_REGISTER, /* can be Register or Deregister */
                                sizeof(char),
                                IP_REG_SET_INFO) < 0) {       /* can be other relevant "sub" operations */
        mmReport(ERROR_GCALL, s_eType, "Register() -> gc_util_insert_parm_val() IPPARM_OPERATION_REGISTER failed");
        bOk = false;
    }

    // Setting address information 
    memset(&registerAddress, 0, sizeof(IP_REGISTER_ADDRESS));
    strcpy(registerAddress.reg_server, g_cProxyIp); /* set server address*/
    sprintf(registerAddress.reg_client,"%s@%s", g_cPhoneNumber, g_cProxyIp); /* set alias for SIP*/
    registerAddress.time_to_live = s_nRegister_TTL;

    if (gc_util_insert_parm_ref(&pParmBlock,
                                IPSET_REG_INFO,
                                IPPARM_REG_ADDRESS,
                                (unsigned char)sizeof(IP_REGISTER_ADDRESS),
                                &registerAddress) < 0) {
        mmReport(ERROR_GCALL, s_eType, "Register() -> gc_util_insert_parm_ref() IPPARM_REG_ADDRESS failed");
        bOk = false;
    }



    // Set the contact
    sprintf(genbuf, "Contact: <sip:%s@%s:%d>;methods=\"INVITE, INFO, SUBSCRIBE, BYE, CANCEL, NOTIFY, ACK, REFER\"", g_cPhoneNumber, m_cDial, g_nSipUdpPort);
    if (gc_util_insert_parm_ref(&pParmBlock,
                                IPSET_SIP_MSGINFO,
                                IPPARM_CONTACT_URI,
                                (strlen(genbuf)+1),
                                genbuf) < 0) {
        mmReport(ERROR_GCALL, s_eType, "Register() -> gc_util_insert_parm_ref() IPPARM_CONTACT_URI failed");
        bOk = false;
    }

    if (bOk == false)
        return false;

    // Now send the REGISTER
    mmReport(INFO_MSG, s_eType, "Register() Sending REGISTER to %s, with %s", registerAddress.reg_server, registerAddress.reg_client);

    // Send the Request
    if (gc_ReqService(GCTGT_CCLIB_NETIF,
                      CIPChannel::s_gcBrdDev,
                      &serviceID,
                      pParmBlock,
                      NULL,
                      EV_ASYNC) < 0) {
        mmReport(ERROR_GCALL, IP_DEVICE, "gc_ReqService() failed on iptB1");
        bOk = false;
    }

    // Set a flag to indicate we are done registering
    s_bRegistered = true;
    gc_util_delete_parm_blk(pParmBlock);

    return bOk;
}

//*****************************************************************************
// 		  NAME : bool CIPChannel::DeRegister()
// DESCRIPTION : UnRegister
// 		 INPUT : None
// 		 OUTPUT : None
// 		RETURNS : Bool - True if the function succeeded or False otherwise
// 	  CAUTIONS : None
//*****************************************************************************
bool CIPChannel::DeRegister()
{
    // Local Variable Declaration
    bool bOk = true;

    if (s_bRegistered == false)
        return false;

    mmReport(INFO_DEBUG, s_eType, "DeRegister()");

    GC_PARM_BLKP pParmBlock = NULL;
    unsigned long serviceID = 0;
    if (gc_util_insert_parm_val(&pParmBlock,
                                IPSET_REG_INFO,
                                IPPARM_OPERATION_DEREGISTER,
                                sizeof(unsigned char),
                                IP_REG_DELETE_ALL) < 0) {
        mmReport(ERROR_GCALL, s_eType, "DeRegister() -> gc_util_insert_parm_ref() IPPARM_OPERATION_DEREGISTER failed");
        bOk = false;
    }

    if (gc_util_insert_parm_val(&pParmBlock,
                                GCSET_SERVREQ,
                                PARM_REQTYPE,
                                sizeof(char),
                                IP_REQTYPE_REGISTRATION) < 0) {
        mmReport(ERROR_GCALL, s_eType, "Register() -> gc_util_insert_parm_val() PARM_REQTYPE failed");
        bOk = false;
    }

    if (gc_util_insert_parm_val(&pParmBlock,
                                GCSET_SERVREQ,
                                PARM_ACK,
                                sizeof(char),
                                1) < 0) {
        mmReport(ERROR_GCALL, s_eType, "Register() -> gc_util_insert_parm_val() PARM_ACK failed");
        bOk = false;
    }


    // Setting address information
    IP_REGISTER_ADDRESS registerAddress;
    memset(&registerAddress, 0, sizeof(IP_REGISTER_ADDRESS));
    strcpy(registerAddress.reg_server,g_cProxyIp); /* set server address*/
    sprintf(registerAddress.reg_client,"%s@%s", g_cPhoneNumber, g_cProxyIp); /* set alias for SIP*/
    registerAddress.time_to_live = s_nRegister_TTL; 

    if (gc_util_insert_parm_ref(&pParmBlock,
                                IPSET_REG_INFO,
                                IPPARM_REG_ADDRESS,
                                (unsigned char)sizeof(IP_REGISTER_ADDRESS),
                                &registerAddress) < 0) {
        mmReport(ERROR_GCALL, s_eType, "Register() -> gc_util_insert_parm_ref() IPPARM_REG_ADDRESS failed");
        bOk = false;
    }

    if (gc_util_insert_parm_val(&pParmBlock,
                                IPSET_PROTOCOL,
                                IPPARM_PROTOCOL_BITMASK,
                                sizeof(char),
                                IP_PROTOCOL_SIP) < 0) {
        mmReport(ERROR_GCALL, s_eType, "DeRegister() -> gc_util_insert_parm_ref() IPPARM_PROTOCOL_BITMASK failed");
        bOk = false;
    }

    if (gc_ReqService(GCTGT_CCLIB_NETIF,
                      CIPChannel::s_gcBrdDev,
                      &serviceID,
                      pParmBlock,
                      NULL,
                      EV_ASYNC) < 0) {
        mmReport(ERROR_GCALL, s_eType, "DeRegister() -> gc_ReqService() failed");
        bOk = false;
    }

    s_bRegistered = false;
    gc_util_delete_parm_blk(pParmBlock);
    return bOk;
}

//*****************************************************************************
// 		  NAME : void CIPChannel::SetState(state_t nextState)
// DESCRIPTION : This function sets the state requested
// 		 INPUT : nextState - Variable of the type state_t with the state
// 			 Information
// 		 OUTPUT : None
// 		RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CIPChannel::SetState(state_t nextState)
{
    m_lock.Lock();
    m_eState = nextState;
    m_lock.Unlock();
}

//*****************************************************************************
// 		  NAME : int CIPChannel::GetState()
// DESCRIPTION : Returns the current state
// 		 INPUT : None
// 		 OUTPUT : None
// 		RETURNS : integer - corresponding to the current state
// 	  CAUTIONS : None
//*****************************************************************************
int CIPChannel::GetState()
{
    return m_eState;
}

//*****************************************************************************
// 		  NAME : void CIPChannel::SetListenFlag(void* pData)
// DESCRIPTION : Sets the listen flag to true when receiving GCEV_LISTEN event
// 		 INPUT : None
// 		 OUTPUT : None
// 		RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CIPChannel::SetListenFlag(void* pData)
{
    // Setting this flag to true indicates that the app. received GCEV_LISTEN
    m_bListening = true;
}

//*****************************************************************************
// 		  NAME : void CIPChannel::SetMetaEvt(METAEVENT& mEvt)
// DESCRIPTION : With the information provided through the METAEVENT structure
// 			 m_crn value is updated
// 		 INPUT : object of type METAEVENT
// 		 OUTPUT : None
// 		RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CIPChannel::SetMetaEvt(METAEVENT& mEvt)
{
    m_metaEvt = mEvt;

    // ignore the CRN for the LISTEN and UNLISTEN events ????
    switch (mEvt.evttype) {
    case GCEV_LISTEN:
    case GCEV_UNLISTEN:
        m_crn = m_crnForCurrentCall;
        break;
    default:
        m_crn = mEvt.crn;
        break;
    }
}

//*****************************************************************************
// 		  NAME : void CIPChannel::OnOpenEx(void* pData)
// DESCRIPTION : Populates the Outbound Capability Structure
// 		 INPUT : None
// 		 OUTPUT : None
// 		RETURNS : Bool - True if the function succeeds or False otherwsie
// 	  CAUTIONS : None
//*****************************************************************************
void CIPChannel::OnOpenEx(void* pData)
{

    mmReport(INFO_DEBUG, s_eType, "OnGCOpenEx()");

    // set the handle in CMMStream object
    m_pMediaChan->m_gcDev = m_gcDev;
    strcpy(m_pMediaChan->m_devName, m_devName);

    //JM - Enable Unsolicited SIP 3PCC Events (GCEV_SIP_ACK) 
    GCSetConfigData_EnableEvents();

    return;
}

//*****************************************************************************
// 		  NAME : void CIPChannel::OnUnlisten(void *pData)
// DESCRIPTION : Drop the call when moving to Unlisten state
// 		 INPUT : pData - void *
// 		 OUTPUT : None
// 		RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CIPChannel::OnUnlisten(void *pData)
{
    mmReport(INFO_DEBUG, s_eType, "OnUnlisten()");
    mmDropCall(pData);
    return;
}

//*****************************************************************************
// 		  NAME : void CIPChannel::OnListen(void* pData)
// DESCRIPTION : Do Nothing
// 		 INPUT : None
// 		 OUTPUT : None
// 		RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CIPChannel::OnListen(void* pData)
{
    mmReport(INFO_DEBUG, s_eType, "OnListen()");
    return;
}

//*****************************************************************************
// 		  NAME : void CIPChannel::OnOffered(void *pData)
// DESCRIPTION : Populates the Outbound Capability Structure
// 		 INPUT : pData - void *
// 		 OUTPUT : None
// 		RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CIPChannel::OnOffered(void *pData)
{
    if (m_bCallAttempt) {
        m_nFailedCalls++;
        mmReport(INFO_DEBUG, s_eType, "OnOffered(): m_bCallAttempt == true, returning");
        return;
    } else {
        m_bCallAttempt = true;
    }

    // Mark this object as being used.
    m_bIsFree = false;

    // Store this new CRN to our CurrentCall CRN
    m_crnForCurrentCall = m_crn;

    // Set the handle in CMMStream object
    m_pMediaChan->m_gcCurrentCrn = m_crn;

		// SKS reset call based variable
	 	m_pMediaChan->ResetVariables();

    // set the flag to indicate our SDP status
    m_bIsSDPOffered = false;

    // Retrieve the ANI and DNIS strings
    gc_GetCallInfo(m_crn, ORIGINATION_ADDRESS, m_cAni);
    gc_GetCallInfo(m_crn, DESTINATION_ADDRESS, m_cDnis);
    mmReport(INFO_DEBUG, s_eType, "ANI: %s  DNIS: %s	CRN: 0x%X", m_cAni, m_cDnis, m_crn);
    if (g_nCallType == CALL_TYPE_H323) {
        mmReport(INFO_DEBUG, s_eType, "Original m_cDnis = %s", m_cDnis);
        std::string dnis(m_cDnis);
        if (dnis.size() > 0) {
            // look for a : now in the string
            string part1, part2;
            std::basic_string <int>::size_type ret = dnis.find(':');
            if (ret != -1) { // if there is a :
                part1 = dnis.substr(dnis.find(':'));
                if (part1.size() > 0) {
                    ret = part1.find(',');
                    if (ret != -1) { // if there is a ','
                        part2 = part1.substr(part1.find(',')).substr(1);
                        strcpy(m_cDnis, part2.c_str());
                    } else {
                        part2 = part1.substr(1);
                        strcpy(m_cDnis, part2.c_str());
                    }
                }
            }
        }
    }		// H323

    if (g_nCallType == CALL_TYPE_SIP) {
        char *cLocation = 0;
        char cTemp[31];
        int len = 0;

        // Store the origination's ANI information
        memset(cTemp, 0, 31);
        cLocation = strstr(m_cAni, "@");
        if (cLocation != NULL) {
            strcpy(m_cOrigIP, ++cLocation);
        }

        memset(cTemp, 0, 31);
        cLocation = 0, len = 0;
        cLocation = strchr(m_cAni, '@');
        if (cLocation != NULL) {
            len = strlen(m_cAni) - strlen(cLocation);
            strncpy(cTemp, m_cAni, len);
            strcpy(m_cAni, cTemp);
        } else {
            mmReport(INFO_ERROR, s_eType, "No Phone Number Specified in Origination Number");
        }

        mmReport(INFO_MSG, s_eType, "Incoming Call From = %s@%s", m_cAni, m_cOrigIP);

        // In SIP calls, the DNIS is of the form 8005551000@192.168.1.10 type.
        // The string needs to be parsed
        memset(cTemp, 0, 31);
        cLocation = strchr(m_cDnis, '@');
        if (cLocation != NULL) {
            len = strlen(m_cDnis) - strlen(cLocation);
            strncpy(cTemp, m_cDnis, len);
            strcpy(m_cDnis, cTemp);
        } else {
            mmReport(INFO_ERROR, s_eType, "No Phone Number Specified in Destination Number, will use default ANI");
            strcpy(m_cDnis, m_cAni);
        }
    }	//SIP

    // Retrieve the SDP in the incoming INVITE
    bool bParseError;
    bParseError=process_inbound_sdp(IPPARM_SDP_OFFER);

    if(bParseError==false && m_bIsSDPOffered==true) {
       mmReport(INFO_DEBUG, s_eType, "Incoming call from %s has OFFER SDP", m_cAni);
       // we found the OFFER SDP in the INVITE
       // Accept the incoming call
       // JM [03/2010] Start media for Eyebeam1.5 issue
       if(!g_bWaitOnSIPAck){
          m_pMediaChan->StartStream();
       } 
       mmAccept(IPPARM_SDP_ANSWER);
    } else if(bParseError==true && m_bIsSDPOffered==true) {
        mmReport(INFO_DEBUG, s_eType, "Incoming call from %s doesn't have any support CODEC in SDP", m_cAni);
       	METAEVENT metaevent;
        metaevent.evttype =GCEV_ERROR;
        metaevent.evtdatap = pData;
        metaevent.crn=m_crn;
        metaevent.evtlen = 1;
        ProcessEvent( metaevent, this);
    } else if(bParseError==false && m_bIsSDPOffered==false) {
        mmReport(INFO_DEBUG, s_eType, "Incoming call from %s has NO SDP, we will wait for SDP in ACK msg", m_cAni);
        // there was no OFFER SDP in the INVITE
        // Accept the incoming call
        mmAccept(IPPARM_SDP_OFFER);
    }
}

//*****************************************************************************
// 		  NAME : void CIPChannel::OnAccept(void* pData)
// DESCRIPTION : Calls the appropriate function to answer the call
// 		 INPUT : None
// 		 OUTPUT : None
// 		RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CIPChannel::OnAccept(void* pData)
{
    m_bCallInProgress = true;
    mmAnswer(pData);
}

//*****************************************************************************
// 		  NAME : void CIPChannel::OnGetCallInfo(void* pData)
// DESCRIPTION : Incoming Call Information
// 		 INPUT : None
// 		 OUTPUT : None
// 		RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CIPChannel::OnGetCallInfo(void* pData)
{
    // assuming that we get this event only for SIP INFO messages
    GC_PARM_BLKP    gcParmBlk_info = 0;
    // insert the message type
    if (gc_util_insert_parm_val(&gcParmBlk_info,
                                IPSET_MSG_SIP,
                                IPPARM_MSGTYPE,
                                sizeof(int),
                                IP_MSGTYPE_SIP_INFO_OK) < 0) {
        mmReport(ERROR_GCALL, s_eType, "OnGetCallInfo() -> gc_util_insert_parm_val() failed on %s for SIP INFO OK", m_devName);
    }

    if (gc_util_insert_parm_val(&gcParmBlk_info,
                                IPSET_MSG_SIP,
                                IPPARM_MSG_SIP_RESPONSE_CODE,
                                sizeof(int),
                                200) < 0) {
        mmReport(ERROR_GCALL, s_eType, "OnGetCallInfo() -> gc_util_insert_parm_val() failed on %s for SIP INFO RESPONSE CODE", m_devName);
    }

    if (gc_Extension(GCTGT_GCLIB_CRN, m_crn, IPEXTID_SENDMSG, gcParmBlk_info, NULL, EV_ASYNC) < 0) {
        mmReport(ERROR_GCALL, s_eType, "OnGetCallInfo() -> gc_Extension failed");
    }
    mmReport(INFO_DEBUG, s_eType, "Successfully sent 200 OK for incoming Call Info");
    gc_util_delete_parm_blk(gcParmBlk_info);

    if (g_bUseSipInfoDtmf || g_nDtmfDetectMode == dtmfMode_sipinfo) {
        GC_PARM_BLK* parmBlock = (GC_PARM_BLK*)pData;
        GC_PARM_DATA  *parm = 0;
        char* info = NULL;
        char* infoType = NULL;

        // going thru each parameter block data to find MIME
        while (( parm = gc_util_next_parm( parmBlock, parm)) != 0) {
            switch ( parm->set_ID) {
            case IPSET_MIME:
                switch (parm->parm_ID) {
                case IPPARM_MIME_PART:
                    {
                        GC_PARM_BLK* parmBlockBody = (GC_PARM_BLK*)(*(unsigned int*)(parm->value_buf));
                        GC_PARM_DATA* parmBody = 0;
                        unsigned int infoSize = 0;
                        unsigned int infoTypeSize = 0;
                        while ( ( parmBody = gc_util_next_parm( parmBlockBody, parmBody)) != 0) {
                            switch ( parmBody->parm_ID) {
                            case IPPARM_MIME_PART_TYPE:
                                infoTypeSize = parmBody->value_size;
                                infoType = new char[infoTypeSize + 1];
                                memcpy( infoType, (char*)( parmBody->value_buf), infoTypeSize);
                                infoType[ infoTypeSize] = '\0';
                                break;

                            case IPPARM_MIME_PART_BODY_SIZE:
                                infoSize = *(unsigned int*)( parmBody->value_buf);
                                break;

                            case IPPARM_MIME_PART_BODY:
                                info = new char[infoSize+1];
                                if ( info != 0) {
                                    memcpy( info, (char*)(*(unsigned int*)( parmBody->value_buf)), infoSize);
                                    info[ infoSize] = '\0';
                                }
                                break;
                            }
                        }
                    }
                    break;
                }
                break;
            }
        }
        if (info) {
            mmReport(INFO_DEBUG, s_eType, "SIP INFO received: %s", info);
            char *result = strtok( info, "Signal=");
            if (result && (result != info)) {
                mmReport(INFO_DEBUG, s_eType, "Faking TDX_GETDIG Event for SIPINFO: %c", result[0]);
                METAEVENT metaevent;
                metaevent.evttype = TDX_GETDIG;
                metaevent.evtdatap = result;
                metaevent.evtlen = 1;
                CMMStream::ProcessEvent( metaevent, m_pMediaChan);
            }
        } else {
            mmReport(INFO_DEBUG, s_eType, "SIP INFO received: empty");
        }
        if (info) {
            delete[] info;
        }
        if (infoType) {
            delete[] infoType;
        }
    }
}

//*****************************************************************************
// 		  NAME : void CIPChannel::OnSIPAck(void* pData)
// DESCRIPTION : Updates the connect count
// 		 INPUT : None
// 		 OUTPUT : None
// 		RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CIPChannel::OnSIPAck(void* pData)
{
    mmReport(INFO_DEBUG, s_eType, "OnSIPAck()");

  if(g_bWaitOnSIPAck){
    m_nConnects++;
    m_bCallAttempt = false;

    if (m_bIsSDPOffered == false) {
        mmReport(INFO_DEBUG, s_eType, "OnSIPAck() m_bIsSDPOffered == false");
    		bool bParseError;
    		bParseError=process_inbound_sdp(IPPARM_SDP_ANSWER);

                if(bParseError==false && m_bIsSDPOffered==true) {
        		mmReport(INFO_DEBUG, s_eType, "Received Answer SDP in SIP ACK message");
        		m_pMediaChan->StartStream();
    		} else if(bParseError==true && m_bIsSDPOffered==true) {
        		mmReport(INFO_DEBUG, s_eType, "Incoming call from %s doesn't have any supported Codec in SDP", m_cAni);
        		METAEVENT metaevent;
        		metaevent.evttype =GCEV_ERROR;
        		metaevent.evtdatap = pData;
        		metaevent.crn=m_crn;
        		metaevent.evtlen = 1;
        		ProcessEvent( metaevent, this);
    		} else if(bParseError==false && m_bIsSDPOffered==false) {
        		mmReport(INFO_DEBUG, s_eType, "No SDP offered yet from %s, Dropping call", m_cAni);
        		METAEVENT metaevent;
        		metaevent.evttype =GCEV_ERROR;
        		metaevent.evtdatap = pData;
        		metaevent.crn=m_crn;
        		metaevent.evtlen = 1;
        		ProcessEvent( metaevent, this);
    		}
		}		// else SDP already offered with INVITE	
		else {
                   m_pMediaChan->StartStream();
		}
  }
}

//*****************************************************************************
// 		  NAME : void CIPChannel::OnAnswered(void* pData)
// DESCRIPTION : Updates the connect count
// 		 INPUT : None
// 		 OUTPUT : None
// 		RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CIPChannel::OnAnswered(void* pData)
{
    mmReport(INFO_DEBUG, s_eType, "OnAnswered()");
    m_nConnects++;
    m_bCallAttempt = false;

    if (m_bIsSDPOffered == false) {
        mmReport(INFO_DEBUG, s_eType, "OnAnswered() m_bIsSDPOffered == false");
    		bool bParseError;
    		bParseError=process_inbound_sdp(IPPARM_SDP_ANSWER);

    		if(bParseError==false && m_bIsSDPOffered==true) {
        		mmReport(INFO_DEBUG, s_eType, "Received Answer SDP in ACK message");
        		//m_pMediaChan->StartStream();
    		} else if(bParseError==true && m_bIsSDPOffered==true) {
        		mmReport(INFO_DEBUG, s_eType, "Incoming call from %s doesn't have any supported Codec in SDP", m_cAni);
        		METAEVENT metaevent;
        		metaevent.evttype =GCEV_ERROR;
        		metaevent.evtdatap = pData;
        		metaevent.crn=m_crn;
        		metaevent.evtlen = 1;
        		ProcessEvent( metaevent, this);
    		} else if(bParseError==false && m_bIsSDPOffered==false) {
        		mmReport(INFO_DEBUG, s_eType, "No SDP offered yet from %s, Dropping call", m_cAni);
        		METAEVENT metaevent;
        		metaevent.evttype =GCEV_ERROR;
        		metaevent.evtdatap = pData;
        		metaevent.crn=m_crn;
        		metaevent.evtlen = 1;
        		ProcessEvent( metaevent, this);
    		}
     }	// else SDP already offered with INVITE	
     else {
          //m_pMediaChan->StartStream();
     }
}

//*****************************************************************************
// 		  NAME : void CIPChannel::OnConnected(void* pData)
// DESCRIPTION : Calls the funtion to do a Listen
// 		 INPUT : None
// 		 OUTPUT : None
// 		RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CIPChannel::OnConnected(void* pData)
{
    m_nConnects++;
    m_bCallAttempt = false;
    m_bCallInProgress = true;

    mmReport(INFO_DEBUG, s_eType, "SIP Call Now Connected");
    // if we are not already listening we need to now

    if (m_bListening == false) {
        mmListen();
    }
}

//*****************************************************************************
// 		  NAME : void CIPChannel::DropCall(void *pData)
// DESCRIPTION : Calls the appropriate function to drop the call
// 		 INPUT : pData - void *
// 		 OUTPUT : None
// 		RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CIPChannel::DropCall(void *pData)
{
    // If the app. received a GCEV_LISTEN event, then the m_bListening flag
    // would be true, else false. Based on this, we will either initiate
    // the Unlisten() or just drop call the call.

    mmReport(INFO_DEBUG, s_eType, "DropCall() Dropping IP Call");
    if (m_bListening == true) {
        // we need to unlisten from the TDM Timeslot we are listening to
        mmReport(INFO_DEBUG, s_eType, "DropCall() Calling Unlisten() and then drop");
        mmUnlisten(pData);
    } else {
        // just drop the call
        mmReport(INFO_DEBUG, s_eType, "DropCall() Dropping IP Call without Unlisten");
        mmDropCall(pData);
    }
}

//*****************************************************************************
// 		  NAME : void CIPChannel::OnRelease(void* pData)
// DESCRIPTION : Resets variables pertaining to the call
// 		 INPUT : None
// 		 OUTPUT : None
// 		RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CIPChannel::OnRelease(void* pData)
{
    m_crn = -1;
    m_crnForCurrentCall = -1;
    m_bIsSDPOffered = false;
    // reset member variables
    m_bCallAttempt = false;
    // reset the call dropped indicator
    m_bCallDropped = false;
}

//*****************************************************************************
// 		  NAME : void CIPChannel::OnExtensionEvent(void *pData)
// DESCRIPTION : Handles the Extension Event received
// 		 INPUT : pData - void *
// 		 OUTPUT : None
// 		RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CIPChannel::OnExtensionEvent(void *pData)
{
    // Local Variable Declaration
    GC_PARM_BLKP        gcParmBlk;
    GC_PARM_DATA        *pgcParmData    = 0;
    EXTENSIONEVTBLK     *pExtensionBlk = 0; 

    // Retrieve the GC parm block information.
    // NOTE: We cannot use pData here as it is the evtdatp member of METAEVENT, we need
    // the extevtdatap member.
    pExtensionBlk = (EXTENSIONEVTBLK *)(m_metaEvt.extevtdatap); 
    gcParmBlk = (&(pExtensionBlk->parmblk));

    char cPhoneList[GCNAME_MAXSIZE+1];
    int i=0;
    while ((pgcParmData = gc_util_next_parm(gcParmBlk, pgcParmData)) != 0) {
        switch (pExtensionBlk->ext_id) {
        case IPEXTID_RECEIVEMSG:
            {
                mmReport(INFO_DEBUG, s_eType, "OnExtensionEvent() Received IPEXTID_RECEIVEMSG--->");
                switch (pgcParmData->set_ID) {
                case IPSET_MSG_Q931:
                    {
                        mmReport(INFO_DEBUG, s_eType, "OnExtensionEvent() Received IPSET_MSG_Q931--->");
                        switch (pgcParmData->value_buf[0]) {
                        case IP_MSGTYPE_Q931_PROGRESS:
                            mmReport(INFO_DEBUG, s_eType, "OnExtensionEvent() Received IP_MSGTYPE_Q931_PROGRESS--->");
                            break;

                        default:
                            mmReport(INFO_DEBUG, s_eType, "OnExtensionEvent() Received PARM=0x%X--->", pgcParmData->parm_ID);
                            break;
                        }
                    }
                    break;
                case IPSET_CALLINFO:
                    mmReport(INFO_DEBUG, s_eType, "OnExtensionEvent() Received IPSET_CALLINFO--->");
                    break;

                default:
                    mmReport(INFO_DEBUG, s_eType, "OnExtensionEvent() Received 0x%X--->\n", pgcParmData->set_ID);
                    break;
                }
            }
            break;
        case IPEXTID_GETINFO:
            {
                mmReport(INFO_DEBUG, s_eType, "OnExtensionEvent() Received IPEXTID_GETINFO--->");
                switch (pgcParmData->set_ID) {
                case IPSET_CALLINFO:
                    switch (pgcParmData->parm_ID) {
                    case IPPARM_CALLID:
                        mmReport(INFO_DEBUG, s_eType, "OnExtensionEvent() for CallId");
                        if (pgcParmData->value_size != 0) {
                            for (i=0; i<16; i++) {
                                printf("%02X", pgcParmData->value_buf[i]);
                            }
                        }
                        printf("\n");
                        break;
                    case IPPARM_CONNECTIONMETHOD:
                        break;
                    case IPPARM_H245TUNNELING:
                        break;
                    case IPPARM_DISPLAY:
                        break;
                    case IPPARM_USERUSER_INFO:
                        break;
                    case IPPARM_PHONELIST:
                        {
                            if (pgcParmData->value_size != 0) {
                                strncpy(cPhoneList, (const char *)pgcParmData->value_buf, GCNAME_MAXSIZE-1);
                                cPhoneList[GCNAME_MAXSIZE+1] = '\0';
                            }
                            // now store this phonelist into the PSTN phoneNum, so that we MakeCall to
                            // this number.
                            CConnection *pConnection = MyConnection(m_nConnectionId);
                            // If we are running in SIP mode, then we get the PHONELIST in a format e.g 8005550001@191.168.1.10
                            // and we need to truncate the string and only use the user-name part of it.
                            if (g_nCallType == CALL_TYPE_SIP) {
                                // In SIP calls, the DNIS is of the form 8005551000@192.168.1.10 type. we need to split the string
                                mmReport(INFO_DEBUG, s_eType, "OnExtensionEvent() state = %s - PhoneList/DNIS = %s", m_state_name, m_cDnis);
                            } else {
                                mmReport(INFO_DEBUG, s_eType, "OnExtensionEvent() state = %s - PhoneList = %s", m_state_name, cPhoneList);
                            }
                        }
                        break;
                    case IPPARM_CALLDURATION:
                        break;
                    case IPPARM_RTCPINFO:
                        break;
                        break;
                    }
                    break;
                }
            }
        }
    } // end of switch for ext_id
}

//*****************************************************************************
// 		  NAME : void CIPChannel::OnTaskFail(void *pData)
// DESCRIPTION : Handles the TaskFail Event received
// 		 INPUT : pData - void *
// 		 OUTPUT : None
// 		RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CIPChannel::OnTaskFail(void *pData)
{
    switch (m_eState) {
    case CIPChannel::state_opened:
        mmReport(INFO_ERROR, s_eType, "OnTaskFail() in state_opened");
        break;
    case CIPChannel::state_initial:
        mmReport(INFO_ERROR, s_eType, "OnTaskFail() in state_initial");
        break;
    case CIPChannel::state_null:
        mmReport(INFO_ERROR, s_eType, "OnTaskFail() in state_null");
        break;
    case CIPChannel::state_proceeding:
        mmReport(INFO_ERROR, s_eType, "OnTaskFail() in state_proceeding");
        break;
    case CIPChannel::state_offered:
        mmReport(INFO_ERROR, s_eType, "OnTaskFail() in state_offered");
        break;
    case CIPChannel::state_alerting:
        mmReport(INFO_ERROR, s_eType, "OnTaskFail() in state_alerting");
        break;
    case CIPChannel::state_connected:
        mmReport(INFO_ERROR, s_eType, "OnTaskFail() in state_connected");
        break;
    case CIPChannel::state_disconnected:
        mmReport(INFO_ERROR, s_eType, "OnTaskFail() in state_disconnected");
        break;
    case CIPChannel::state_usrDisconnected:
        mmReport(INFO_ERROR, s_eType, "OnTaskFail() in state_usrDisconnected");
        break;
    case CIPChannel::state_idle:
        mmReport(INFO_ERROR, s_eType, "OnTaskFail() in state_idle");
        break;
    case CIPChannel::state_appStopping:
        mmReport(INFO_ERROR, s_eType, "OnTaskFail() in state_stopping");
        break;
    }

    // Reset the line device
    mmRestart(pData);
}

//*****************************************************************************
// 		  NAME : void CIPChannel::OnError(void *pData)
// DESCRIPTION : Handles the Error Event 
// 		 INPUT : pData - void *
// 		 OUTPUT : None
// 		RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CIPChannel::OnError(void *pData)
{
    mmReport(INFO_ERROR, s_eType, "OnError()");
    switch (m_eState) {
    case state_appStopping:
        break;
    }
    // Reset the line device
    mmRestart(pData);
}

//*****************************************************************************
// 		  NAME : void CIPChannel::OnSetConfigData(void *pData)
// DESCRIPTION : Handles the SetConfigData return event
// 		 INPUT : pData - void *
// 		 OUTPUT : None
// 		RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CIPChannel::OnSetConfigData(void *pData)
{
    mmReport(INFO_DEBUG, s_eType, "OnSetConfigData()");
   //nothing to do

}
//*****************************************************************************
// 		  NAME : void CIPChannel::OnError(void *pData)
// DESCRIPTION : Handles the Undeclared Event 
// 		 INPUT : pData - void *
// 		 OUTPUT : None
// 		RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CIPChannel::OnUndeclared(void *pData)
{
    METAEVENT *pEvt = (METAEVENT *)pData;
    mmReport(INFO_ERROR, s_eType, "OnUndeclared() state = %d, event = %s (%s)", m_eState, mmEvt2Str(pEvt->evttype), m_state_name);

    // Reset the line device
    //JM - Removed, may just be unexpected and unhandled...
    //mmRestart(pData);
}

//*****************************************************************************
// 		  NAME : void CIPChannel::mmWaitCall(void* pData)
// DESCRIPTION : Issues waitcall
// 		 INPUT : None
// 		 OUTPUT : None
// 		RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CIPChannel::mmWaitCall(void* pData)
{
    // Initiate gc_WaitCall()
    if ((gc_WaitCall(m_gcDev,
                     NULL,
                     NULL,
                     0,
                     EV_ASYNC)) < 0) {
        mmReport(ERROR_GCALL, s_eType, "gc_WaitCall failed");
    } else {
        mmReport(INFO_DEBUG, s_eType, "gc_WaitCall complete");
    }
}

//*****************************************************************************
// 		  NAME : void CIPChannel::mmRestart(void* pData)
// DESCRIPTION : Results in Resetting of the line device
// 		 INPUT : None
// 		 OUTPUT : None
// 		RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CIPChannel::mmRestart(void* pData)
{
    if ((gc_ResetLineDev(m_gcDev, EV_ASYNC)) < 0) {
        mmReport(ERROR_GCALL, s_eType, "gc_ResetLineDev failed");
    } else {
        mmReport(INFO_DEBUG, s_eType, "gc_ResetLineDev complete");
    }
}

//*****************************************************************************
// 		  NAME : void CIPChannel::mmAccept()
// DESCRIPTION : Accepts the offered call
// 		 INPUT : sdpType - OFFER or ANSWER
// 		 OUTPUT : None
// 		RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CIPChannel::mmAccept(unsigned long sdpType)
{
    // Local Variable Declaration
    int           frc;
    char          str[MAX_STRING_SIZE];
    GC_PARM_BLKP  gcParmBlk = NULL;
    char          sdpOfferAnswer[4096];
    int           len;

    if (sdpType == IPPARM_SDP_OFFER) {
        mmReport(INFO_DEBUG, s_eType, "No SDP Offered, we will Send OFFER SDP ");
    } else if (sdpType == IPPARM_SDP_ANSWER) {
        mmReport(INFO_DEBUG, s_eType, "SDP was offered, we will send ANSWER SDP");
    }

    if(m_bIsSDPOffered==false)
    {
        m_pMediaChan->PopulateOfferSdp(sdpOfferAnswer);
    	len = strlen(sdpOfferAnswer);
    	mmReport(INFO_DEBUG, s_eType,"Sending Offer SDP length = %d and SdpText=\n%s",sdpOfferAnswer, sdpOfferAnswer);
    } else {
    	// Fill in local SDP info
    	sdpSessionDescription* a_pSDP;
    	a_pSDP = m_pMediaChan->PopulateAnswerSdp();
    	a_pSDP->exportSDP(sdpOfferAnswer,4095,false);
    	len = strlen(sdpOfferAnswer);
    	mmReport(INFO_DEBUG, s_eType,"Sending AnswerSDP length = %d and SdpText=\n%s",len+1, sdpOfferAnswer);
    }

    // Pass SDP to GC
    frc = gc_util_insert_parm_ref_ex(&gcParmBlk, IPSET_SDP, sdpType, len+1, sdpOfferAnswer);

    if (frc != GC_SUCCESS) {
        sprintf(str, "mmAccept : gc_util_insert_parm_ref_ex failed");
        mmReport(INFO_ERROR, s_eType, str);
        return;
    }

    frc = gc_SetUserInfo(GCTGT_GCLIB_CRN, m_crn, gcParmBlk, GC_SINGLECALL);

    if (frc != GC_SUCCESS) {
        sprintf(str, "mmAccept:gc_SetUserInfo failed");
        mmReport(INFO_ERROR, s_eType, str);
        return;
    }

    // Set the codec information based on the incoming SDP
    // Create your own SDP and attach it to this AcceptCall
    if (gc_AcceptCall(m_crn, 0, EV_ASYNC) < 0) {
        mmReport(ERROR_GCALL, s_eType, "gc_AcceptCall failed CRN: 0x%X", m_crn);
    }

    gc_util_delete_parm_blk(gcParmBlk);
    return;
}

//*****************************************************************************
// 		  NAME : void CIPChannel::mmListen()
// DESCRIPTION : Routes two IP devices
// 		 INPUT : None
// 		 OUTPUT : None
// 		RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CIPChannel::mmListen()
{
    // Local Variable Declaration
    SC_TSINFO   tsInfo;
    long        lTemp = 0;

    tsInfo.sc_numts = 1;
    tsInfo.sc_tsarrayp = &lTemp;

    CConnection *pConnection = MyConnection(m_nConnectionId);
    lTemp = (pConnection->m_pMediaChan)->m_ipmTalkTs;

    // Route this IP call to its partner IP call
    if (gc_Listen(m_gcDev, &tsInfo, EV_ASYNC) < 0) {
        mmReport(INFO_ERROR, s_eType, "gc_Listen() failed for %s", m_devName);
    } else {
        m_bListening = true;
        mmReport(INFO_DEBUG, s_eType, "%d listening to ts=%d", m_talkTs, (pConnection->m_pMediaChan)->m_ipmTalkTs);
    }
}

//*****************************************************************************
// 		  NAME : void CIPChannel::mmUnListen(void* pData)
// DESCRIPTION : UnRoutes two IP Devices
// 		 INPUT : None
// 		 OUTPUT : None
// 		RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CIPChannel::mmUnlisten(void* pData)
{
    CConnection *pConnection = MyConnection(m_nConnectionId);
    if (gc_UnListen(m_gcDev, EV_ASYNC) < 0) {
        mmReport(INFO_ERROR, s_eType, "gc_UnListen() failed for %s", m_devName);
    } else {
        // unset the listening flag
        m_bListening = false;
        mmReport(INFO_DEBUG, s_eType, "Unlisten Success from %d to ts=%d", m_talkTs, (pConnection->m_pMediaChan)->m_ipmTalkTs);
    }
}

//*****************************************************************************
// 		  NAME : void CIPChannel::mmAnswer(void *pData)
// DESCRIPTION : Issues gc_AnswerCall
// 		 INPUT : None
// 		 OUTPUT : None
// 		RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CIPChannel::mmAnswer(void *pData)
{
    if (gc_AnswerCall(m_crn, 0, EV_ASYNC) < 0) {
        mmReport(ERROR_GCALL, s_eType, "gc_AnswerCall failed CRN: 0x%X", m_crn);
    } else {
        mmReport(INFO_DEBUG, s_eType, "mmAnswer() Success for CRN: 0x%x", m_crn);
        mmReport(INFO_MSG, s_eType, "[%s] Answered Incoming Call From = %s@%s", m_devName, m_cAni, m_cOrigIP);
    }
}


//*****************************************************************************
// 		  NAME : void CIPChannel::mmDropCall(void *pData)
// DESCRIPTION : Issues gc_DropCall and stops streaming
// 		 INPUT : pData - void *
// 		 OUTPUT : None
// 		RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CIPChannel::mmDropCall(void *pData)
{
    mmReport(INFO_DEBUG, s_eType, "+++++ m_crn = %d, m_bCallInProgress = %d, m_bListening = %d", m_crn, m_bCallInProgress, m_bListening);
    if (m_crn != m_crnForCurrentCall) {
        mmReport(INFO_DEBUG, s_eType, "******** mmDropCall() for Incorrect CRN, m_crn = 0x%x, ccCRN = 0x%x", m_crn, m_crnForCurrentCall);
    }
// SKS reset call based variable
		m_pMediaChan->ResetVariables();

//    if ((m_crn > 0) && (m_bCallInProgress == true)) {
    	if (m_crn > 0) {
        mmReport(INFO_MSG, s_eType, "[%s]  Dropping call From = %s@%s", m_devName, m_cAni, m_cOrigIP);
        if (gc_DropCall(m_crn, GC_NORMAL_CLEARING, EV_ASYNC) < 0) {
            mmReport(ERROR_GCALL, s_eType, "gc_DropCall failed CRN: 0x%X", m_crn);
        }
        m_bCallDropped = true;
    }

    // stop streaming
    m_pMediaChan->StopStream(pData);
}

//*****************************************************************************
// 		  NAME : void CIPChannel::mmReleaseCall(void* pData)
// DESCRIPTION : Issues gc_ReleaseCall
// 		 INPUT : None
// 		 OUTPUT : None
// 		RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CIPChannel::mmReleaseCall(void* pData)
{
    // First indicate that the call is completed. This boolean is used to 
    // figure out if the call was Dropped or not.
    m_bCallInProgress = false;

    if (gc_ReleaseCallEx(m_crn, EV_ASYNC) < 0) {
        mmReport(ERROR_GCALL, s_eType, "gc_ReleaseCallEx failed CRN: 0x%X", m_crn);
    }
}

//*****************************************************************************
// 		  NAME : void CIPChannel::process_inbound_sdp()
// DESCRIPTION : Parses incoming SDP information
// 		 INPUT : None
// 		 OUTPUT : None
// 		RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
bool CIPChannel::process_inbound_sdp(unsigned long sdpType)
{
    mmReport(INFO_DEBUG, s_eType, "process_inbound_sdp()");
// SKS
    bool bParseError=false;

    GC_PARM_BLK *pParmBlock = (GC_PARM_BLK*)(m_metaEvt.extevtdatap);

    mmReport(INFO_DEBUG, s_eType, "\t\tRetrieving inbound SDP");
    GC_PARM_DATA_EXT parm_data_ext;
    INIT_GC_PARM_DATA_EXT(&parm_data_ext);

    /* going thru each parameter block data*/
    while (gc_util_next_parm_ex(pParmBlock, &parm_data_ext) == 0) {
        switch (parm_data_ext.set_ID) {

#ifdef TDM_AUDIO
        // Although this isn't exactly SDP information, it can be collected
        // here on an offered event
        // We are looking for a user-defined SIP header called "CallInfo"
        // It contains information to be used in case a 2G audio call is requested.
        case IPSET_SIP_MSGINFO:
            {
                char* header = new char[ parm_data_ext.data_size + 1];
                strncpy( header, (char*)parm_data_ext.pData, parm_data_ext.data_size);
                header[parm_data_ext.data_size] = '\0';
                switch ( parm_data_ext.parm_ID) {
                case IPPARM_SIP_HDR:
                    mmReport(INFO_DEBUG, s_eType, "SIP HDR: %s",  header); 
                    // Have the full header; parse and save info it contains
                    m_pMediaChan->SaveISDNGwCmd(header);
                    break;
                }
                delete[] header;
                break;
            }
            break;
#endif  

            /* Handle SDP information */
        case IPSET_SDP:
            switch (parm_data_ext.parm_ID) {
            case IPPARM_SDP_OFFER:
                mmReport(INFO_DEBUG, s_eType, "\t\tSDP Offer Retrieved");
                char *sdpOffer = new char[parm_data_ext.data_size + 1];
                strncpy(sdpOffer,(char*)parm_data_ext.pData,parm_data_ext.data_size);
                sdpOffer[parm_data_ext.data_size]='\0';

                mmReport(INFO_DEBUG, s_eType, "\t\tSDP Offer(Length=%d) = \n%s",strlen(sdpOffer),sdpOffer);

                m_SDP.clear();
                m_SDP.importSDP(sdpOffer, false);
                mmReport(INFO_DEBUG, s_eType, "SDP Import successful");
                // set our flag to indicate that we did get SDP
                m_bIsSDPOffered = true;
                // Pass this incoming PORT info to local IPML
								// SKS additional check
                if(m_pMediaChan->ParseOfferSdp(&m_SDP)<0) {
                   mmReport(INFO_ERROR, s_eType,"Unsupported codec dropping call check offered SDP content"); 
                   bParseError=true;
                }
                break;
            }
            break;
        }
    }
    return bParseError;
}

#if 1
//*****************************************************************************
// 		  NAME : void CIPChannel::GCSetConfigData_EnableEvents()
// DESCRIPTION : Enables extra SIP 3PCC events
// 		 INPUT : None 
// 		 OUTPUT : None
// 		RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CIPChannel::GCSetConfigData_EnableEvents()
{     
   int          target_type;
   long         target_id;
   long         request_idp = 0;
   long         long_value;
   GC_PARM_BLK *parmblkp = NULL;
   
   mmReport(INFO_DEBUG, s_eType, "GCSetConfigData(GCMSK_SIP_ACK)");

   // JM - Added code for enabling GCEV_SIP_ACK event via gc_SetConfigData
   // First call GC PARM utility function to insert the parameter used to be updated
   long_value = GCMSK_SIP_ACK;  //GCMSK_SIP_ACK=0x8000
   gc_util_insert_parm_val(&parmblkp, GCSET_CALLEVENT_MSK, GCACT_ADDMSK, sizeof(long), long_value);

   target_type = GCTGT_GCLIB_CHAN;
   target_id = m_gcDev;
   /* Call gc_SetConfigData() function */
   if (gc_SetConfigData(target_type, target_id, parmblkp, 0, GCUPDATE_IMMEDIATE, &request_idp, EV_ASYNC) != GC_SUCCESS) {
       mmReport(INFO_DEBUG, s_eType, "gc_SetConfigData(EnableEvt=GCEV_SIP_ACK, devicename=%s, mode=EV_ASYNC) FAIL!", m_devName);
   }
   mmReport(INFO_DEBUG, s_eType, "gc_SetConfigData(EnableEvt=GCEV_SIP_ACK, devicename=%s, mode=EV_ASYNC) Success!", m_devName);

   // delete the parm data block after using it 
   gc_util_delete_parm_blk(parmblkp);

}

#endif
