/**
* @file mmdemo.cpp
* @brief Multimedia demo - main program
* @date Feb 16, 2007
*
* DIALOGIC CONFIDENTIAL 
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to 
* the source code ("Material") are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its 
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced, 
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual 
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

// Application Level Include Files
#include "mmdemo.h"
#include "exitsync.h"
#include "config.h"
#include "appevents.h"

#ifdef WIN32
    #include <conio.h>
    #include "process.h"
    #define strncasecmp _strnicmp
    #define strcasecmp strcmpi
#endif
#ifdef HRC_TARGET // Required for HMP 5.0
#include "hrclib.h"
#endif

extern const char* MULTIMEDIADEMO_VERSION_STRING;

// Global Variables
bool        g_bTraceFlag = false;
int         g_nTraceLevel = 0;
char        g_cAVFileDir[256];
int         g_nAudioFileFormat = MMD_DMF;
int         g_nVideoFileFormat = MMD_DMF;
bool        g_bAnswerThenDrop = false;
int         g_nNumberOfCalls = 0;
const int   MAX_IP_CHANS = 4;
static const char *s_DefaultConfigFile = "mmdemo.cfg";
CExitSync   *g_pExit = CExitSync::Instance();       // Create a Exit Synchronization object.
CLocker     g_lock;

bool        g_bWaitOnSIPAck = false;
int         g_IFrameInterval = 0;
bool        g_bUseSipInfoDtmf = false;
bool        g_bRepeatMenus = false;

int         g_nDtmfDetectMode = dtmfMode_inband;

int         g_nAudioTranscodeDir=RX;
int         g_nVideoTranscodeDir=RX;

// SKS
unsigned char g_cRxDciQcifValue[128] = {0};
unsigned char g_cRxDciQcifSize = 0;
char g_cRxDciQcifStrValue[128] = {'\0'};

unsigned char g_cRxDciCifValue[128] = {0};
unsigned char g_cRxDciCifSize = 0;
char g_cRxDciCifStrValue[128] = {'\0'};

unsigned char g_cTxDciQcifValue[128] = {0};
unsigned char g_cTxDciQcifSize = 0;
char g_cTxDciQcifStrValue[128] = {'\0'};

unsigned char g_cTxDciCifValue[128] = {0};
unsigned char g_cTxDciCifSize = 0;
char g_cTxDciCifStrValue[128] = {'\0'};

unsigned int g_maxBandwidth = 0;
unsigned int g_maxFrameRate = 0;
int g_nPlatformNameRelease=HMP_4_1;

int g_nVideoCodec;
int g_nAudioCodec;
int g_nVideoRes;

unsigned char g_h264SdpPacketizationMode = 1;
unsigned char g_recordControl = RECORDING_DEFAULT;
int g_h264SdpProfileLevel = -1;
bool g_bUseRecordMsg = false;
bool g_bRecordAudioOnly = false;
int  g_nRecordTimeout = 0;
int g_nMsgRepeatInterval = 0;

// JS: added RTSP support (8/15/08)
RTSP_URLS         g_rtspUrls;

// Class Function Definition
//*****************************************************************************
// 		  NAME : CConnection::CConnection(int nId)
// DESCRIPTION : Constructor of CConnection Class
// 		 INPUT : nId - Connection ID
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
CConnection::CConnection(int nId)
: m_nConnectionId(nId)
{
   m_pIpChan   = 0;
}

//*****************************************************************************
// 		  NAME : CConnection::~CConnection(int nId)
// DESCRIPTION : Destructor of CConnection Class
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
CConnection::~CConnection()
{
   delete m_pIpChan;
   m_nConnectionId = 0;
}

//*****************************************************************************
// 		  NAME : CConnection *MyConnection(int nConnectionId)
// DESCRIPTION : Retrieves the connection object based on the call object
// 		 INPUT : nConnectionId - Connection ID
// 	    OUTPUT : None
// 	   RETURNS : CConnection - Pointer to object of type CConnection
// 	  CAUTIONS : None
//*****************************************************************************
CConnection *MyConnection(int nConnectionId)
{
   // Local Variable Declaration
   CONNECTION_MAP::iterator mapIterator;

   g_lock.Lock();
   mapIterator = g_connectionMap.find(nConnectionId);
   if ( mapIterator != g_connectionMap.end() )
   {
      g_lock.Unlock();
      return((*mapIterator).second);
   }
   else
   {
      g_lock.Unlock();
      return 0;
   }
}


// Other Function Descriptions
//*****************************************************************************
// 		  NAME : void InitLogfiles()
// DESCRIPTION : Opens up the log file for future use
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void InitLogfiles()
{
   char mode[3] = {"w+"};
   g_logfile = fopen("./mmdemo.log", mode);
}

//*****************************************************************************
// 		  NAME : bool InitGC()
// DESCRIPTION : Initializes the GlobalCall Subsystem
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : Bool, True - If the process ended successfully else False
// 	  CAUTIONS : None
//*****************************************************************************
bool InitGC()
{
   // Local Variables
   bool                bOk      = true;
   GC_START_STRUCT     gcstart;
   IPCCLIB_START_DATA  ipData;
   IP_VIRTBOARD        ipVirtBoard;

   // Start the GC Subsystem
   mmReport(INFO_MSG, NONE, "Starting GC ....");

   INIT_IP_VIRTBOARD(&ipVirtBoard);
   ipData.version              = 0x100;
   ipData.num_boards           = 1;
   ipVirtBoard.total_max_calls = g_nNumberOfCalls;
   ipVirtBoard.version         = IP_VIRTBOARD_VERSION;
   ipVirtBoard.localIP.ip_ver = IPVER4;
   ipVirtBoard.localIP.u_ipaddr.ipv4 = (unsigned int)g_localIPinByteOrder;

   if ( g_nCallType == CALL_TYPE_H323 )
   {
      ipVirtBoard.h323_max_calls      = g_nNumberOfCalls;
      ipVirtBoard.sip_max_calls       = IP_CFG_NO_CALLS;
      ipVirtBoard.h323_signaling_port = 1720;
   }
   else
   {
      ipVirtBoard.sip_max_calls       = g_nNumberOfCalls;
      ipVirtBoard.h323_max_calls      = IP_CFG_NO_CALLS;
      ipVirtBoard.sip_signaling_port  = g_nSipUdpPort;
      ipVirtBoard.sip_msginfo_mask    = IP_SIP_MSGINFO_ENABLE | IP_SIP_MIME_ENABLE;
   }
   ipData.board_list = &ipVirtBoard;
   INIT_IPCCLIB_START_DATA(&ipData, 1,&ipVirtBoard);

   // Enable 3PCC mode
   ipData.max_parm_data_size       = 4096; //IP_CFG_PARM_DATA_MAXLEN;
   ipData.media_operational_mode   = MEDIA_OPERATIONAL_MODE_3PCC;

   memset(&gcstart, 0, sizeof(GC_START_STRUCT));

   // pass the IP Data that configures the RV stack
   CCLIB_START_STRUCT ccstart[] = 
   {
      {"GC_IPM_LIB",NULL},
      {"GC_H3R_LIB",&ipData},
#ifndef HRC_TARGET
      {"GC_DM3CC_LIB",NULL},
#endif
   };
   gcstart.cclib_list = ccstart;
   gcstart.num_cclibs = sizeof(ccstart)/sizeof(ccstart[0]); 

   // load specified libraries.
   if ( gc_Start( &gcstart ) != GC_SUCCESS )
   {
      mmReport(ERROR_GCALL, NONE, "Failed on GC_START");
      bOk = false;
   }
   return bOk;
}


//*****************************************************************************
// 		  NAME : bool Initialize(int nChansPerBoard)
// DESCRIPTION : Initializes the application for Inbound and Outbound modes in
//               different processes
// 		 INPUT :
//               nChansPerBoard - Channels to use per board
// 	    OUTPUT : None
// 	   RETURNS : Bool, True - If the process ended successfully else False
// 	  CAUTIONS : None
//*****************************************************************************
bool Initialize(int nChansPerBoard)
{
   // Local Variables
   int         gc_error                = 0;
   int         cclibid                 = 0;
   long        cc_error                = 0;
   bool        bError                  = false;
   int         numBoards               = 0;
   int         numIpmChansPerBoard     = 0;
   int         ipmBrdH                 = 0;
   int         iptChan                 = 0;
   int         nBrd                    = 0;
   int         nChan                   = 0;
   int         nVoxBrd             = 0;
   int         nVoxChan                = 0;
   char        cDevName[DEV_NAME]      = "";
   char        cMMDevName[DEV_NAME]    = "";
   char        cIpmCCDevName[DEV_NAME]   = ""; // JS
   char        cIpmRTSPDevName[DEV_NAME]   = ""; // JS
   char        cVoxDevName[DEV_NAME]   = "";
#ifdef TDM_AUDIO
   char            cDtiDevName[DEV_NAME]   = "";
#endif
   CIPChannel  *pIpChan                = 0;
   CMMStream   *pMMChan                = 0;

   pair<map<long, CIPChannel *>::iterator, bool> bIpIterator;
   pair<map<unsigned long, CConnection *>::iterator, bool> bConnectionIterator;

   CConnection *pConnection = 0;

   mmReport(INFO_OK, NONE, "Collecting Device Information....");
   for ( nBrd = 0, iptChan = 1; nBrd < 1; nBrd++ )
   {
      for ( nChan = 1; nChan <= nChansPerBoard; nChan++, iptChan++ )
      {
         // Now create the IPCall objects
         if ( g_nCallType == CALL_TYPE_H323 )
         {
            sprintf(cDevName, ":N_iptB1T%d:P_H323:M_ipmB%dC%d", iptChan, nBrd+1, nChan);
         }
         else
         { // SIP
            sprintf(cDevName, ":N_iptB1T%d:P_SIP", nChan);
            sprintf(cMMDevName, "mmB1C%d", nChan); // only 1 board supported
            sprintf(cIpmCCDevName, "ipmB1C%d", nChan); // only 1 board supported
#ifdef USE_RTSP
            sprintf(cIpmRTSPDevName, "ipmB1C%d", nChan+1); // only 1 board supported. JS
#endif

#ifdef TDM_AUDIO
            sprintf(cDtiDevName, "dtiB5T%d", nChan); // only 1 board supported
#endif

            // create the vox device name
            nVoxChan = nChan % 4; //there are only 4 sub-devices per voice channel
            if ( nVoxChan == 0 )
               nVoxChan = 4;
            if ( nVoxChan == 1 )
               nVoxBrd++;
            sprintf(cVoxDevName, "dxxxB%dC%d", nVoxBrd, nVoxChan);
#ifdef USE_RTSP
            nChan++; // JS. Two IPM devices used above: CC and RTSP
#endif
         }

         // Create MMStream Objects
#ifdef TDM_AUDIO
         pMMChan = new CMMStream(cMMDevName, cIpmCCDevName, cVoxDevName, cDtiDevName);
#else
         pMMChan = new CMMStream(cMMDevName, cIpmCCDevName, cIpmRTSPDevName, cVoxDevName);
#endif
         if ( pMMChan == 0 )
         {
            mmReport(INFO_ERROR, NONE, "Failed to Create MM Call object.");
            return false;
         }

         // Insert all IP channels into our CIPChannel map.
         pIpChan = new CIPChannel(cDevName, g_cDestIp, iptChan, pMMChan);
         
         if ( pIpChan == 0 )
         {
            mmReport(INFO_ERROR, NONE, "Failed to Create IP Call object.");
            delete pMMChan;
            return false;
         }

         // Create a connection object
         pConnection = new CConnection(iptChan);
         pConnection->m_pIpChan = pIpChan;

         // Insert this CConnection object into our Connection Map.
         bConnectionIterator = g_connectionMap.insert(make_pair<unsigned long, CConnection *>
                                                      (iptChan, pConnection));
         if ( bConnectionIterator.second == false )
         {
            mmReport(INFO_ERROR, NONE, "Failed to insert Connection object into Connection Map.");
            bError = true;
         }

         // Initialize MM and IPML devices
         if ( pMMChan->Init() == false )
         {
            mmReport(INFO_ERROR, NONE, "Failed to Initialize MMStream Call object.");
            bError = true;
         }

         // Initialize GC devices
         if ( pIpChan->Init() == false )
         {
            mmReport(INFO_ERROR, NONE, "Failed to Initialize IP Call object.");
            bError = true;
         }
         // Insert the IP channel object into our IP Call Map for use later.
         bIpIterator = g_ipCallMap.insert(make_pair<long, CIPChannel *>
                                          (pIpChan->m_gcDev, pIpChan));

         if ( bIpIterator.second == false )
         {
            mmReport(INFO_ERROR, NONE, "Failed to insert IP Call object into IP Call Map.");
            bError = true;
         }

         // now register the ip channel with a sip proxy
         pIpChan->Register();
      }
   }

   mmReport(INFO_OK, NONE, "......DONE!");

   // check if there were any errors during opening.
   if ( bError == true )
      Shutdown();

   return true;
}

//*****************************************************************************
// 		  NAME : bool Shutdown()
// DESCRIPTION : Cleans up all the resources
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : Bool, True - If the process ended successfully else False
// 	  CAUTIONS : None
//*****************************************************************************
bool Shutdown()
{
   // Local Variable Declaration
   IP_CALL_MAP::iterator ipIterator;

   ipIterator = g_ipCallMap.begin();

   mmReport(INFO_DEBUG, NONE, "Shutting down ......");
   while ( ipIterator != g_ipCallMap.end() )
   {
      // now deregister the ip channel with a sip proxy
      (ipIterator->second)->DeRegister();

      // shutdown GC device
      (ipIterator->second)->Shutdown();

      // shutdown MM+IPM device
      (ipIterator->second)->m_pMediaChan->Shutdown();
      delete (ipIterator->second)->m_pMediaChan;
      delete (ipIterator->second);
      ipIterator++;
   }

   // Stop the GlobalCall Subsystem
   gc_Stop();
   mmReport(INFO_DEBUG, NONE, "........ gc_Stop() done!");

   // clear all the maps
   g_ipCallMap.clear();
   g_connectionMap.clear();
   return true;
}

//*****************************************************************************
// 		  NAME : void PrintHelp(void)
// DESCRIPTION : Prints the help menu for the demo
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void PrintHelp(void)
{
   printf("Usage:\n");
   printf("\tmmdemo = uses the default config file mmdemo.cfg\n");
   printf("\tConfigFileName = use config_file as the config file, e.g. mmdemo.cfg\n");
   printf("\t\t(Note: does NOT append .cfg to ConfigFileName\n");
   printf("\tmmdemo -anything prints the help message\n");
   exit(0);
}

//*****************************************************************************
// 		  NAME : void PrintDebugHelp(void)
// DESCRIPTION : Prints the Debug help menu for the demo
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void PrintDebugHelp(void)
{
   // Local Varaible Declaration
   int Index = 0;

   printf("%s <Debug Options>\n", PROGRAM_NAME);
   for ( Index=0; DebugMenu[Index][0] != '\0'; Index++ )
   {
      printf("%s\n", DebugMenu[Index]);
   }
   printf("\n");
}

//*****************************************************************************
// 		  NAME : GetHostInfo()
// DESCRIPTION : Retrieve the systems host information - Ip address etc
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : -1 if the app failed or 0 if the app succeeded
// 	  CAUTIONS : None
//*****************************************************************************
#ifdef WIN32
int GetHostInfo(char *a_cIpAddrInDotNotation, unsigned long *a_IpAddrInByteOrder)
#else
int GetHostInfo(char *a_cIpAddrInDotNotation, in_addr_t *a_IpAddrInByteOrder)
#endif
{
#ifdef WIN32
   WORD wVersionRequested;
   WSADATA wsaData;
   int err;

   /* Use the MAKEWORD(lowbyte, highbyte) macro declared in Windef.h */
   wVersionRequested = MAKEWORD(2, 2);

   err = WSAStartup(wVersionRequested, &wsaData);
   if (err != 0) {
       /* Tell the user that we could not find a usable */
       /* Winsock DLL.                                  */
       printf("WSAStartup failed with error: %d\n", err);
       return 1;
   }

   /* Confirm that the WinSock DLL supports 2.2.*/
   /* Note that if the DLL supports versions greater    */
   /* than 2.2 in addition to 2.2, it will still return */
   /* 2.2 in wVersion since that is the version we      */
   /* requested.                                        */

   if (LOBYTE(wsaData.wVersion) != 2 || HIBYTE(wsaData.wVersion) != 2) {
       /* Tell the user that we could not find a usable */
       /* WinSock DLL.                                  */
       printf("Could not find a usable version of Winsock.dll\n");
       WSACleanup();
       return 1;
   }
   else
       printf("The Winsock 2.2 dll was found okay\n");
# endif // WIN32
   // retrieve the server IP address to use for Call Control
   // we will use the first enabled NIC
   char hostname[GCNAME_MAXSIZE];
   if ( (gethostname(hostname, GCNAME_MAXSIZE)) < 0 )
   {
      printf("ERROR: Could not read host name information. Exiting...%d\n", errno);
      return -1;
   }

   //printf("Host Name = %s, IP Address: ", hostname);
   struct hostent *hp;
   hp = gethostbyname(hostname);
   struct in_addr in;
   char **p;
   for ( p = hp->h_addr_list; *p != 0; p++ )
   {
      (void) memcpy(&in.s_addr, *p, sizeof (in.s_addr));
//           (void) printf("%s\t%s\n", inet_ntoa(in), hp->h_name);
      strcpy(a_cIpAddrInDotNotation, inet_ntoa(in));
      (void) memcpy(a_IpAddrInByteOrder, *p, sizeof (in.s_addr));
      //for (q = hp->h_aliases; *q != 0; q++)
      //    (void) printf(" %s", *q);
      //(void) putchar('\n');
   }
   return 0;
}
void GetPlatformName()
{
// SKS
   char *name="DIALOGIC_PLATFORM";
   char *release="DIALOGIC_RELNUM";
   char *platformName;
   char *releaseNumber;
   platformName=getenv(name);
   releaseNumber=getenv(release);
#ifndef HRC_TARGET
   if( (strcasecmp(platformName,"OVL")==0) && (strcasecmp(releaseNumber,"1.0")==0) ) {
      g_nPlatformNameRelease=OVL_1_0;
   }else if( (strcasecmp(platformName,"OVL")==0) && (strcasecmp(releaseNumber,"2.0")==0) ) {
      g_nPlatformNameRelease=OVL_2_0;
   }else if( (strcasecmp(platformName,"ZVL")==0) && (strcasecmp(releaseNumber,"1.0")==0) ) {
      g_nPlatformNameRelease=ZVL_1_0;
   }else if( (strcasecmp(platformName,"HMP")==0) && (strcasecmp(releaseNumber,"1.5")==0) ) {
      g_nPlatformNameRelease=HMP_1_5;
   }else if( (strcasecmp(platformName,"HMP")==0) && (strcasecmp(releaseNumber,"3.1")==0) ) {
      g_nPlatformNameRelease=HMP_3_1;
   }else if( (strcasecmp(platformName,"HMP")==0) && (strcasecmp(releaseNumber,"4.1")==0) ) {
      g_nPlatformNameRelease=HMP_4_1;
   }
   mmReport(INFO_MSG, NONE, "MultiMediaDemo running on [%s_%s]",platformName,releaseNumber);
#else
  printf("[HRC MM demo]");
#endif
}

//*****************************************************************************
// 		  NAME : int main(int argc, char *argv[])
// DESCRIPTION : Main entry point in the Application
// 		 INPUT : argc - No of Arguments, argv - Array of arguments
// 	    OUTPUT : None
// 	   RETURNS : -1 if the app failed or 0 if the app succeeded
// 	  CAUTIONS : None
//*****************************************************************************
int main(int argc, char *argv[])
{
   // Local Variable Declaration
   const char          *configFile     = s_DefaultConfigFile;
   bool                bOutboundFlag   = false;//FALSE;
   int                 nStartSpan      = 0;
   int                 nStartIpmChan   = 0;
   PTHREAD_START       pStartRoutine   = 0;
   CONFIGFILEOPTIONS   ConfigFileOpts;

   // Create a log file. We will print messages to this log file only if we are running 1 or 2 channels
   InitLogfiles();

   // Retrieve IP address of this host server
   if ( GetHostInfo(g_cDestIp, &g_localIPinByteOrder) < 0 )
      return -1;

   mmReport(INFO_MSG, NONE, "--------------------------------------------------------------------------------------");
   mmReport(INFO_MSG, NONE, "MultiMediaDemo version = [%s]",MULTIMEDIADEMO_VERSION_STRING);
   GetPlatformName();
   mmReport(INFO_MSG, NONE, "and will use IP:%s for Call Control", g_cDestIp);
   mmReport(INFO_MSG, NONE, "--------------------------------------------------------------------------------------");
   
   if ( argc >= 3 ) {
      PrintHelp();                            
   }
   if ( argc == 2 ) {
      configFile = argv[1];
      if ( *configFile == '-' ) {
         PrintHelp();                            
      }
   }
   if ( ProcessConfigFile(&ConfigFileOpts, configFile) == false ) {
      printf("ProcessCommandLine failed\n");
      return -1;
   } 

   // get the parameters the user wants to use.
   g_bTraceFlag                = ConfigFileOpts.bTraceFlag;
   g_nTraceLevel               = ConfigFileOpts.nTraceLevel;
   g_nNumberOfCalls            = ConfigFileOpts.nNumberOfCalls;
   g_nCallType                 = ConfigFileOpts.nCallType;
   g_nTimeToRecord             = 5; 
   g_nSipUdpPort               = ConfigFileOpts.nSipUdpPort;
   strcpy(g_cProxyIp, ConfigFileOpts.cProxyIp);
   strcpy(g_cPhoneNumber, ConfigFileOpts.cPhoneNumber);
   g_bLowResVideo              = ConfigFileOpts.bLowResVideo;
   strcpy(g_cAVFileDir, ConfigFileOpts.cAVFileDir);
   
   //audiofileformat cfg file setting
   if(strcasecmp(ConfigFileOpts.cAudioFileFormat,"3gp")==0) {	 
      g_nAudioFileFormat=MMD_3GP;
   } else if(strcasecmp(ConfigFileOpts.cAudioFileFormat,"wav")==0) {
      g_nAudioFileFormat=MMD_WAV;
   } else {
      g_nAudioFileFormat=MMD_DMF;
   }

   //videofileformat cfg file setting
   if(strcasecmp(ConfigFileOpts.cVideoFileFormat,"3gp")==0) {	 
      g_nVideoFileFormat=MMD_3GP;
   } else {
      g_nVideoFileFormat=MMD_DMF;
   }
   g_IFrameInterval = ConfigFileOpts.IFrameInterval;
#ifdef USE_RTSP
   g_rtspUrls = ConfigFileOpts.m_rtspUrls; // JS
#endif
   g_bUseSipInfoDtmf = ConfigFileOpts.bUseSipInfoDtmf;
   g_nDtmfDetectMode = ConfigFileOpts.nDtmfDetectMode;
   g_bRepeatMenus = ConfigFileOpts.bRepeatMenus;
#ifdef TDM_AUDIO
   strcpy(g_cISDNGatewayIP, ConfigFileOpts.cISDNGatewayIP);
   g_nISDNGatewayPort = ConfigFileOpts.nISDNGatewayPort;
#endif

   //videoxcodedir cfg file setting
   if(strcasecmp(ConfigFileOpts.cVideoTranscodeDir,"native")==0) {
      g_nVideoTranscodeDir=NATIVE;
   } else if(strcasecmp(ConfigFileOpts.cVideoTranscodeDir,"rx")==0) {
      g_nVideoTranscodeDir=RX;
   } else if(strcasecmp(ConfigFileOpts.cVideoTranscodeDir,"tx")==0) {
      g_nVideoTranscodeDir=TX;
   } else if(strcasecmp(ConfigFileOpts.cVideoTranscodeDir,"rxtx")==0 || strcasecmp(ConfigFileOpts.cVideoTranscodeDir,"txrx")==0) {
      g_nVideoTranscodeDir=RXTX;
   } else {
      g_nVideoTranscodeDir=NATIVE;
   }

   //audioxcodedir cfg file setting
   if(strcasecmp(ConfigFileOpts.cAudioTranscodeDir,"native")==0) {
      g_nAudioTranscodeDir=NATIVE;
   } else if(strcasecmp(ConfigFileOpts.cAudioTranscodeDir,"rx")==0) {
      g_nAudioTranscodeDir=RX;
   } else if(strcasecmp(ConfigFileOpts.cAudioTranscodeDir,"tx")==0) {
      g_nAudioTranscodeDir=TX;
   } else if(strcasecmp(ConfigFileOpts.cAudioTranscodeDir,"rxtx")==0 || strcasecmp(ConfigFileOpts.cAudioTranscodeDir,"txrx")==0) {
      g_nAudioTranscodeDir=RXTX;
   } else {
      g_nAudioTranscodeDir=RXTX;
   }

   //maxBandwidth cfg file setting - Set the maximum bandwidth
   g_maxBandwidth=ConfigFileOpts.maxBandwidth;
   if(g_maxBandwidth<=0 || g_maxBandwidth>384) {
      g_maxBandwidth=384; //QCIF-CIF maxbandwidth
   } 

   //maxFrameRate cfg file setting
   g_maxFrameRate=ConfigFileOpts.maxFrameRate;
   if(g_maxFrameRate<=0 || g_maxFrameRate>30) {
      g_maxFrameRate=30;
   }
 
   g_h264SdpPacketizationMode = ConfigFileOpts.h264SdpPacketizationMode;
   g_recordControl = ConfigFileOpts.recordControl;
   g_h264SdpProfileLevel = ConfigFileOpts.h264SdpProfileLevel;

   //videoResolution cfg file setting
   if(strcasecmp(ConfigFileOpts.videoRes,"SQCIF")==0) {
      g_nVideoRes=SQCIF;
   } else if(strcasecmp(ConfigFileOpts.videoRes,"CIF")==0) {
       mmReport(INFO_MSG, NONE, "Config Options setting global to CIF");
      g_nVideoRes=CIF;
   } else {
      g_nVideoRes=QCIF;
   }
  
   //videoCodec cfg file setting 
   if(strcasecmp(ConfigFileOpts.videoCodec,"MPEG4")==0) {
      g_nVideoCodec=MPEG4;
   } else if(strcasecmp(ConfigFileOpts.videoCodec,"H264")==0) {
      g_nVideoCodec=H264;
   } else if((strcasecmp(ConfigFileOpts.videoCodec,"H263-1998")==0) || (strcasecmp(ConfigFileOpts.videoCodec,"H263+")==0)) {
      g_nVideoCodec=H263_1998;
   } else if((strcasecmp(ConfigFileOpts.videoCodec,"H263-2000")==0) || (strcasecmp(ConfigFileOpts.videoCodec,"H263++")==0)) {
      g_nVideoCodec=H263_2000;
   } else {
      g_nVideoCodec=H263;
   }

   //audioCodec mmdemo.cfg file setting
   if((strcasecmp(ConfigFileOpts.audioCodec,"G711U")==0) || (strcasecmp(ConfigFileOpts.audioCodec,"G711")==0) ) {
      g_nAudioCodec=G711U;
   } else if(strcasecmp(ConfigFileOpts.audioCodec,"G711A")==0) {
      g_nAudioCodec=G711A;
   } else if(strcasecmp(ConfigFileOpts.audioCodec,"G723")==0) {
      g_nAudioCodec=G723;
   } else if(strcasecmp(ConfigFileOpts.audioCodec,"G726")==0) {
      g_nAudioCodec=G726;
   } else if(strcasecmp(ConfigFileOpts.audioCodec,"G729")==0) {
      g_nAudioCodec=G729;
   } else if((strcasecmp(ConfigFileOpts.audioCodec,"PCM")==0) ||(strcasecmp(ConfigFileOpts.audioCodec,"PCM8K")==0) ) {
      g_nAudioCodec=PCM;
   } else if(strcasecmp(ConfigFileOpts.audioCodec,"PCM16K")==0) {
      g_nAudioCodec=PCM16K;
   } else if((strcasecmp(ConfigFileOpts.audioCodec,"G722")==0)) {
      g_nAudioCodec=G722;
   } else if((strcasecmp(ConfigFileOpts.audioCodec,"AMRWB")==0) || (strcasecmp(ConfigFileOpts.audioCodec,"AMR-WB")==0))  {
      g_nAudioCodec=AMRWB;
   } else if((strcasecmp(ConfigFileOpts.audioCodec,"AMR")==0) || (strcasecmp(ConfigFileOpts.audioCodec,"AMRNB")==0))  {
      g_nAudioCodec=AMR;
   } else{
      g_nAudioCodec=G711U;
   }


   // Set Rx DCI from config file
   g_cRxDciQcifSize=ConfigFileOpts.rxdciQcifSize;
   memcpy(g_cRxDciQcifValue,ConfigFileOpts.rxdciQcifValue,ConfigFileOpts.rxdciQcifSize);
   if(ConfigFileOpts.rxdciQcifSize>0) {
      strcpy(g_cRxDciQcifStrValue,ConfigFileOpts.rxdciQcifStrValue);
      mmReport(INFO_MSG, NONE, "Config file RxDCI QCIF [size=%d]:%s",g_cRxDciQcifSize,g_cRxDciQcifStrValue);
   }
   g_cRxDciCifSize=ConfigFileOpts.rxdciCifSize;
   memcpy(g_cRxDciCifValue,ConfigFileOpts.rxdciCifValue,ConfigFileOpts.rxdciCifSize);
   if(ConfigFileOpts.rxdciCifSize>0) {
      strcpy(g_cRxDciCifStrValue,ConfigFileOpts.rxdciCifStrValue);
      mmReport(INFO_MSG, NONE, "Config file RxDCI CIF [size=%d]:%s",g_cRxDciCifSize,g_cRxDciCifStrValue);
   }

   g_cTxDciQcifSize=ConfigFileOpts.txdciQcifSize;
   memcpy(g_cTxDciQcifValue,ConfigFileOpts.txdciQcifValue,ConfigFileOpts.txdciQcifSize);
   if(ConfigFileOpts.txdciQcifSize>0) {
      strcpy(g_cTxDciQcifStrValue,ConfigFileOpts.txdciQcifStrValue);
      mmReport(INFO_MSG, NONE,"Config file TxDCI QCIF [size=%d]:%s",g_cTxDciQcifSize,g_cTxDciQcifStrValue);
   }
   g_cTxDciCifSize=ConfigFileOpts.txdciCifSize;
   memcpy(g_cTxDciCifValue,ConfigFileOpts.txdciCifValue,ConfigFileOpts.txdciCifSize);
   if(ConfigFileOpts.txdciCifSize>0) {
      strcpy(g_cTxDciCifStrValue,ConfigFileOpts.txdciCifStrValue);
      mmReport(INFO_MSG, NONE,"Config file TxDCI CIF [size=%d]:%s",g_cTxDciCifSize,g_cTxDciCifStrValue);
   }

   g_bWaitOnSIPAck = ConfigFileOpts.bWaitOnSIPAck;
   g_bUseRecordMsg = ConfigFileOpts.bUseRecordMsg;
   g_bRecordAudioOnly = ConfigFileOpts.bRecordAudioOnly;
   g_nRecordTimeout = ConfigFileOpts.nRecordTimeout;
   g_nMsgRepeatInterval = ConfigFileOpts.nMsgRepeatInterval;

   // Set the start time
#ifdef WIN32
   _ftime(&g_StartTime);
#else
   ftime(&g_StartTime);
#endif
   // set internal counter of ready channels
   g_nReadyChannels = 1;

   // Start the GC system.
   if ( InitGC() == false )
      return -1;

   pStartRoutine = (PTHREAD_START)RunDemo;

   // Initialize the demo
   if ( !Initialize(g_nNumberOfCalls) )
   {
      mmReport(INFO_ERROR, NONE, "Failed to Initialize Application Resources.");
      return -1;
   }

#ifdef WIN32
   // Create and spawn a worker thread for SRL.
   HANDLE hThread = INVALID_HANDLE_VALUE;
   hThread = ((HANDLE) _beginthreadex( (void *)        (NULL), 
                                       (unsigned)      (0), 
                                       pStartRoutine, 
                                       (void *)        (NULL), 
                                       (unsigned)      (0), 
                                       (unsigned *)    (NULL)));

   if ( hThread == INVALID_HANDLE_VALUE )
   {
      mmReport(ERROR_SYSTEM, NONE, "Failed to Create WorkerThread, Exiting ...");
      Shutdown();
      return -1;
   }

#else

   // Creating worker thread.
   pthread_t hThread, hCallDurationThread, hInterCallDelayThread;
   pthread_create(&hThread, NULL, pStartRoutine, NULL);
#endif

   // Runtime options.
   char ch;
   IP_CALL_MAP::iterator ipIterator;
   while ( !g_bDone )
   {
      ch = getchar();
      switch ( ch )
      {
         case '?':
         case 'h':
            PrintDebugHelp();
            break;
         case 'r': // toggle for tracing
            if ( g_bTraceFlag == true )
            {
               printf("\n----------- TRACE is Now OFF --------------\n");
               g_bTraceFlag = false;
            }
            else if ( g_bTraceFlag == false )
            {
               printf("\n----------- TRACE is Now ON --------------\n");
               g_bTraceFlag = true;
            }
            break;

         case 'q':
            g_bDone = true;
            break;

         case 'm':
            if (g_recordControl == RECORDING_MANUAL)
            {
                printf("Starting Manual Recording\n");
                ipIterator = g_ipCallMap.begin();
                (ipIterator->second)->m_pMediaChan->Record();
            }
            break;
      }
   }

   // Wait for all threads to exit.
   g_pExit->WaitExitSignal();

   Shutdown();
   mmReport(INFO_OK, NONE, "Exiting Application...");

#ifdef WIN32
   Sleep(1000);
#endif

   return 0;
}

//*****************************************************************************
// 		  NAME : DWORD WINAPI RunDemo(PVOID pContext) /
//				 void *RunDemo(void *pContext)
// DESCRIPTION : Thread function that handles SRL Events
// 		 INPUT : pContext - void *
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
#ifdef WIN32
DWORD WINAPI RunDemo(PVOID pContext)
#else
void *RunDemo(void *pContext)
#endif
{
   long                    lDev        = 0;
   int                     nType       = 0;
   CIPChannel              *pIpChan    = 0;
   METAEVENT               metaevt;
   IP_CALL_MAP::iterator   ipIterator;


#if 1 //PERIODIC_IFRAMES
   // Timer to generate I-Frame requests
   time_t currentTime;
   time_t lastTime;
   time(&lastTime);
   double duration;
   static const double iFrameInterval = g_IFrameInterval;
   static const double msgRepeatInterval = g_nMsgRepeatInterval;

   while ( !g_bDone )
   {
      if ( sr_waitevt(1000) < 0 )
      {
         //  Add I-Frame timer
         time (&currentTime);
         duration = difftime(currentTime, lastTime);
         if ( (iFrameInterval && (duration >= iFrameInterval)) || 
              (msgRepeatInterval && (duration >= msgRepeatInterval)) )
         {

            IP_CALL_MAP::iterator ipIterator;

            ipIterator = g_ipCallMap.begin();
            while ( ipIterator != g_ipCallMap.end() )
            {
               
               if(iFrameInterval){
                  // Send event requesting IFrame to any media channels 
                  // in a recording state.
                  METAEVENT appMetaevent;
                  appMetaevent.evttype = APPEV_IFRAME_REQ;
                  if ( (ipIterator->second)->m_pMediaChan->RecordingNow() )
                  {
                     // send a SIP INFO to request an I-Frame
                     mmReport(INFO_MSG, NONE, "Timer generated APPEV_IFRAME_REQ");
                     (ipIterator->second)->m_pMediaChan->ProcessEvent(appMetaevent, 
                                                                   (ipIterator->second)->m_pMediaChan);
                  }
               }
               if(msgRepeatInterval){
                  METAEVENT appMetaevent;
                  appMetaevent.evttype = APPEV_WAIT_TIMEOUT;
                  if ( (ipIterator->second)->m_pMediaChan->WaitingNow() )
                  {
                     mmReport(INFO_MSG, NONE, "Timer generated APPEV_WAIT_TIMEOUT");
                     (ipIterator->second)->m_pMediaChan->ProcessEvent(appMetaevent,(ipIterator->second)->m_pMediaChan);
                  }
               }
               ipIterator++;
            }

            time(&lastTime);
         }
         continue;
      }
#else
   while ( !g_bDone )
   {
      if ( sr_waitevt(5000) < 0 )
      { // look for events every 5 seconds
         continue;
      }
#endif
      // put a check to see if this is a GC event.
      memset(&metaevt, 0, sizeof(METAEVENT));
      if ( gc_GetMetaEvent(&metaevt) < 0 )
      {
         mmReport(ERROR_GCALL, NONE, "gc_GetMetaEvent() Failed");
         continue;
      }

      // check to see if we have a GC event
      if ( (metaevt.flags & GCME_GC_EVENT) == 0 )
      {
#ifdef HRC_TARGET
         if (metaevt.evttype == HRCEV_LINK_UP)
         {
            mmReport(INFO_DEBUG, NONE, "--- Got event HRCEV_LINK_UP - Connection to Media Server established");
         }
         else if (metaevt.evttype == HRCEV_LINK_DOWN)
         {
            mmReport(INFO_DEBUG, NONE, "--- Got event HRCEV_LINK_DOWN - Lost Connection to Media Server");
            g_pExit->ForceExit();
         }   
         else
#endif  
         ProcessNonGCEvent(metaevt);
      }
      else
      {
         lDev = metaevt.evtdev;
         // retrieve the userAttribute that was associated with this device handle at time of Open
         if ( gc_GetUsrAttr(lDev, (void**)&(nType)) < 0 )
         {
            mmReport(ERROR_GCALL, nType, "gc_GetUsrAttr() Failed, Ignoring Event");
            continue;
         }
         // find the CConnection object for this device handle based on the device type
         switch ( nType )
         {
            case IP_DEVICE:
               {
                  // look for this device handle in the IP Call map
                  ipIterator = g_ipCallMap.find(lDev);
                  if ( ipIterator != g_ipCallMap.end() )
                  {
                     // get the connection object from the map.
                     pIpChan = (*ipIterator).second;
                     pIpChan->ProcessEvent(metaevt, pIpChan);
                  }
                  else
                  {
                     mmReport(INFO_ERROR, nType, "Could not find Connection Object for Device = %d in the IP Call Map (evt=0x%x)", lDev, metaevt.evttype);
                     continue;
                  }
               }
               break;
            case IP_BOARD_DEVICE:
               {
                  OnRegister((void *)&metaevt);
               }
         }
      }       
   } // end of while

   g_pExit->BeginExit();
   mmReport(INFO_DEBUG, NONE, "RunDemo() Exiting....");
   return 0;
}

//*****************************************************************************
// 		  NAME : void ProcessNonGCEvent(METAEVENT &metaevt)
// DESCRIPTION : If a non gc event occurred, pass it the stream event handler
// 		 INPUT : metaevt - Pointer to METAEVENT object
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void ProcessNonGCEvent(METAEVENT &metaevt)
{
   // Local Variable Declaration
   long        lDev        = metaevt.evtdev;
   int         nType       = metaevt.evttype;
   CMMStream   *pMMStream  = 0;

   sr_getparm(lDev, SR_USERCONTEXT, &pMMStream);
   pMMStream->ProcessEvent(metaevt, (CMMStream *)pMMStream);
   return;
}

//*****************************************************************************
// 		  NAME : void OnRegister(void *pData)
// DESCRIPTION : Handle the event from Registration
// 		 INPUT : pData - void *
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void OnRegister(void *pData)
{
   METAEVENT   *metaevt    = (METAEVENT *)pData;
   GC_INFO     gcError;

   // process board device events here
   mmReport(INFO_DEBUG, IP_DEVICE, "--- Got event %s for IPBOARD", mmEvt2Str(metaevt->evttype));

   gc_ResultInfo(metaevt, &gcError);
   switch ( metaevt->evttype )
   {
      case GCEV_UNBLOCKED:
         break;
      case GCEV_EXTENSION:
         break;
      case GCEV_SERVICEREQ:
         break;
      case GCEV_SERVICERESP:
         if ( gcError.gcValue != GC_SUCCESS )
         {
            mmReport(INFO_ERROR, NONE, "Registration with SIP Proxy Failed.");
            mmReport(INFO_ERROR, NONE, "SIP Registration: SIP Specific Msg: %s", gcError.ccMsg);
         }
         else
         {
            mmReport(INFO_MSG, NONE, "Registration with SIP Proxy was Successful.");
         }
         break;
      case GCEV_SERVICERESPCMPLT:
         break;
      default:
         // Anu:
         break;
   }
}

//*****************************************************************************
// 		  NAME : void mmReport(int iType, int eType, char *fmt, ...)
// DESCRIPTION : Writes messages to the Report file
// 		 INPUT : int iType, int eType, char *fmt, ...
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void mmReport(int iType, int eType, const char *fmt, ...)
{
   // Local Variable Declaration
   char    sTemp[1024];
   char    *sCcMsg;
   char    *sLibName;
   int     iCcLibId;
   int     iGcError;
   long    iCcError;
   va_list args;
   int     iPos        = 0;


#ifdef WIN32
   struct  _timeb timebuffer;
   char    *timeline;

   _ftime(&timebuffer);

   timeline = ctime(&(timebuffer.time));

   iPos += sprintf(sTemp + iPos, "[%.15s.%03hu] ", &timeline[4], timebuffer.millitm);
#else
   struct  timeb timebuffer;
   char    *timeline;

   ftime(&timebuffer);

   timeline = ctime(&(timebuffer.time));

   iPos += sprintf(sTemp + iPos, "[%.15s.%03hu] ", &timeline[4], timebuffer.millitm);
#endif

   switch ( eType )
   {
      case IP_DEVICE:
         if ( g_nCallType == CALL_TYPE_H323 )
            iPos += sprintf(sTemp + iPos, "[H323] ");
         else
            iPos += sprintf(sTemp + iPos, "[SIP]  ");
         break;
      case MMS_DEVICE:
         iPos += sprintf(sTemp + iPos, "[MMS]  ");
         break;
      case NONE:
         iPos += sprintf(sTemp + iPos, "[APP]  ");
         break;
   }

   switch ( iType )
   {
      case INFO_OK:
      case INFO_MSG:
         iPos += sprintf(sTemp + iPos, "[INFO] ");
         break;

      case INFO_DEBUG:
         iPos += sprintf(sTemp + iPos, "[DEBUG]");
         break;

      case INFO_DEBUG1:
         iPos += sprintf(sTemp + iPos, "[DEBUG_L1]");
         break;

      case INFO_DEBUG2:
         iPos += sprintf(sTemp + iPos, "[DEBUG_L2]");
         break;
      
      case INFO_ERROR:
         iPos += sprintf(sTemp + iPos, "[ERROR]");
         break;

      case ERROR_SYSTEM:
         iPos += sprintf(sTemp + iPos, "[SYSTEM ERROR]: 0x%X", errno);
         break;

      case ERROR_GCALL:
         mmReport(INFO_OK, eType, "Getting GC Error Info.");
         if ( gc_ErrorValue(&iGcError, &iCcLibId, &iCcError) < 0 )
         {
            // error getting error!
            mmReport(INFO_ERROR, eType, "Error - gc_ErrorValue");
         }
         if ( gc_ResultMsg(iCcLibId, iCcError, &sCcMsg) < 0 )
         {
            mmReport(INFO_ERROR, eType, "Error - gc_ResultMsg (CcLibId)");
         }
         if ( gc_CCLibIDToName(iCcLibId, &sLibName) < 0 )
         {
            mmReport(INFO_ERROR, eType, "Error - gc_CCLibIDToName");
         }
         iPos += sprintf(sTemp + iPos, "\t%s ERROR: 0x%X: %s",
                         sLibName,
                         iCcError,
                         sCcMsg);
         break;
      case ERROR_SRL:
         iPos += sprintf(sTemp + iPos, "[SRL ERROR]: 0x%X: %s", ATDV_LASTERR(SRL_DEVICE),
                         ATDV_ERRMSGP(SRL_DEVICE));
         break;

      default:
         iPos += sprintf(sTemp + iPos, "[INFO TYPE NOT DEFINED]");           
         break;
   }

   iPos += sprintf(sTemp + iPos, " -> ");
   va_start(args, fmt);
   iPos += vsprintf(sTemp + iPos, fmt, args);
   va_end(args);

   int nLen = strlen(sTemp);
   strcpy(sTemp + nLen, "\n");         
   if ( g_nNumberOfCalls <= 5 )
   {
      if ( iType >  INFO_DEBUG )
      {
         if ( g_bTraceFlag == true && g_nTraceLevel >= (iType-INFO_DEBUG))
         {
            fprintf(g_logfile, sTemp); 
         }
      }
      else
         fprintf(g_logfile, sTemp); 
       
   }
   // only display to stdout if traceflag is enabled
   if ( iType >= INFO_DEBUG )
   {
      if ( g_bTraceFlag == true && g_nTraceLevel >= (iType-INFO_DEBUG))
      {
         fprintf(stdout, sTemp); 
      }
   }
   else
   {
      fprintf(stdout, sTemp); 
   }
}

//*****************************************************************************
// 		  NAME : char* mmEvt2Str(int nEvent)
// DESCRIPTION : Converts the event identification into a string for convenience
// 		 INPUT : nEvent - Event ID
// 	    OUTPUT : Pointer to a string containing the event description
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
const char* mmEvt2Str(int nEvent)
{
   // Local Variable Declaration
   static char cTemp[25];

   switch ( nEvent )
   {
      case GCEV_LISTEN:
         return("GCEV_LISTEN");
         break;
      case GCEV_UNLISTEN:
         return("GCEV_UNLISTEN");
         break;
      case GCEV_OPENEX:
         return("GCEV_OPENEX");
         break;
      case GCEV_OPENEX_FAIL:
         return("GCEV_OPENEX_FAIL");
         break;
      case GCEV_RELEASECALL:
         return("GCEV_RELEASECALL");
         break;
      case GCEV_TASKFAIL:
         return("GCEV_TASKFAIL");
         break;
      case GCEV_ANSWERED:
         return("GCEV_ANSWERED");
         break;
      case GCEV_ACCEPT:
         return("GCEV_ACCEPT");
         break;
      case GCEV_DROPCALL:
         return("GCEV_DROPCALL");
         break;
      case GCEV_RESETLINEDEV:
         return("GCEV_RESETLINEDEV");
         break;
      case GCEV_CALLINFO:
         return("GCEV_CALLINFO");
         break;
      case GCEV_REQANI:
         return("GCEV_REQANI");
         break;
      case GCEV_SETCHANSTATE:
         return("GCEV_SETCHANSTATE");
         break;
      case GCEV_FACILITY_ACK:
         return("GCEV_FACILITY_ACK");
         break;
      case GCEV_FACILITY_REJ:
         return("GCEV_FACILITY_REJ");
         break;
      case GCEV_MOREDIGITS:
         return("GCEV_MOREDIGITS");
         break;
      case GCEV_SETBILLING:
         return("GCEV_SETBILLING");
         break;
      case GCEV_ALERTING:
         return("GCEV_ALERTING");
         break;
      case GCEV_CONNECTED:
         return("GCEV_CONNECTED");
         break;
      case GCEV_ERROR:
         return("GCEV_ERROR");
         break;
      case GCEV_OFFERED:
         return("GCEV_OFFERED");
         break;
      case GCEV_DISCONNECTED:
         return("GCEV_DISCONNECTED");
         break;
      case GCEV_PROCEEDING:
         return("GCEV_PROCEEDING");
         break;
      case GCEV_PROGRESSING:
         return("GCEV_PROGRESSING");
         break;
      case GCEV_USRINFO:
         return("GCEV_USRINFO");
         break;
      case GCEV_FACILITY:
         return("GCEV_FACILITY");
         break;
      case GCEV_CONGESTION:
         return("GCEV_CONGESTION");
         break;
      case GCEV_D_CHAN_STATUS:
         return("GCEV_D_CHAN_STATUS");
         break;
      case GCEV_NOUSRINFOBUF:
         return("GCEV_NOUSRINFOBUF");
         break;
      case GCEV_NOFACILITYBUF:
         return("GCEV_NOFACILITYBUF");
         break;
      case GCEV_BLOCKED:
         return("GCEV_BLOCKED"); 
         break;
      case GCEV_UNBLOCKED:
         return("GCEV_UNBLOCKED"); 
         break;
      case GCEV_EXTENSIONCMPLT:
         return("GCEV_EXTENSIONCMPLT"); 
         break;
      case GCEV_EXTENSION:
         return("GCEV_EXTENSION"); 
         break;
      case GCEV_ISDNMSG:
         return("GCEV_ISDNMSG");
         break;
      case GCEV_NOTIFY:
         return("GCEV_NOTIFY");
         break;
      case GCEV_L2FRAME:
         return("GCEV_L2FRAME");
         break;
      case GCEV_L2BFFRFULL:
         return("GCEV_L2BFFRFULL");
         break;
      case GCEV_L2NOBFFR:
         return("GCEV_L2NOBFFR");
         break;
      case GCEV_SETUP_ACK:
         return("GCEV_SETUP_ACK");
         break;
      case GCEV_SETCONFIGDATA:
         return("GCEV_SETCONFIGDATA");
         break;
      case GCEV_SETCONFIGDATA_FAIL:
         return("GCEV_SETCONFIGDATA_FAIL");
         break;
      case GCEV_DIVERTED:
         return("GCEV_DIVERTED");
         break;
      case GCEV_HOLDACK:
         return("GCEV_HOLDACK");
         break;
      case GCEV_HOLDCALL:
         return("GCEV_HOLDCALL");
         break;
      case GCEV_HOLDREJ:
         return("GCEV_HOLDREJ");
         break;
      case GCEV_RETRIEVEACK:
         return("GCEV_RETRIEVEACK");
         break;
      case GCEV_RETRIEVECALL:
         return("GCEV_RETRIEVECALL");
         break;
      case GCEV_RETRIEVEREJ:
         return("GCEV_RETRIEVEREJ");
         break;
      case GCEV_NSI:
         return("GCEV_NSI");
         break;
      case GCEV_TRANSFERACK:
         return("GCEV_TRANSFERACK");
         break;
      case GCEV_TRANSFERREJ:
         return("GCEV_TRANSFERREJ");
         break;
      case GCEV_TRANSIT:
         return("GCEV_TRANSIT");
         break;
      case GCEV_RESTARTFAIL:
         return("GCEV_RESTARTFAIL");
         break;
      case GCEV_ALARM:
         return("GCEV_ALARM");
         break;
      case GCEV_CALLPROC:
         return("GCEV_CALLPROC");
         break;
      case GCEV_SERVICEREQ:
         return("GCEV_SERVICEREQ");
         break;
      case GCEV_SERVICERESP:
         return("GCEV_SERVICERESP");
         break;
      case GCEV_SERVICERESPCMPLT:
         return("GCEV_SERVICERESPCMPLT");
         break;
      case GCEV_SIP_ACK:
         return("GCEV_SIP_ACK");
         break;
			case SMEV_ADD_OVERLAY:
         return("SMEV_ADD_OVERLAY");
         break;
			case SMEV_ADD_OVERLAY_FAIL:
         return("SMEV_ADD_OVERLAY_FAIL");
         break;
			case SMEV_REMOVE_OVERLAY:
         return("SMEV_REMOVE_OVERLAY");
         break;
			case SMEV_REMOVE_OVERLAY_FAIL:
         return("SMEV_REMOVE_OVERLAY_FAIL");
         break;
			case SMEV_REMOVE_ALL_OVERLAYS:
				 return("SMEV_REMOVE_ALL_OVERLAYS");
				 break;
			case SMEV_REMOVE_ALL_OVERLAYS_FAIL:
				 return("SMEV_REMOVE_ALL_OVERLAYS_FAIL");
				 break;
      case IPMEV_OPEN:
         return("IPMEV_OPEN");
         break;
      case IPMEV_ERROR:
         return("IPMEV_ERROR");
         break;
      case IPMEV_STARTMEDIA:
         return("IPMEV_STARTMEDIA");
         break;
      case IPMEV_STOPPED:
         return("IPMEV_STOPPED");
         break;
      case IPMEV_GET_LOCAL_MEDIA_INFO:
         return("IPMEV_GET_LOCAL_MEDIA_INFO");
         break;
      case IPMEV_LISTEN:
         return("IPMEV_LISTEN");
         break;
      case IPMEV_UNLISTEN:
         return("IPMEV_UNLISTEN");
         break;
      case IPMEV_GET_XMITTS_INFO:
         return("IPMEV_GET_XMITTS_INFO");
         break;
      case IPMEV_SET_PARM:
         return("IPMEV_SET_PARM");
         break;
      case IPMEV_GET_PARM:
         return("IPMEV_GET_PARM");
         break;
      case IPMEV_TELEPHONY_EVENT:
         return("IPMEV_TELEPHONY_EVENT");
         break;
      case IPMEV_EVENT_ENABLED:
         return("IPMEV_EVENT_ENABLED");
         break;
      case MMEV_OPEN:
         return("MMEV_OPEN");
         break;
      case MMEV_OPEN_FAIL:
         return("MMEV_OPEN_FAIL");
         break;
      case MMEV_ERROR:
         return("MMEV_ERROR");
         break;
      case MMEV_PLAY:
         return("MMEV_PLAY");
         break;
      case MMEV_PLAY_FAIL:
         return("MMEV_PLAY_FAIL");
         break;
      case MMEV_PLAY_ACK:
         return("MMEV_PLAY_ACK");
         break;
      case MMEV_PLAY_ACK_FAIL:
         return("MMEV_PLAY_ACK_FAIL");
         break;
      case MMEV_RECORD:
         return("MMEV_RECORD");
         break;
      case MMEV_RECORD_FAIL:
         return("MMEV_RECORD_FAIL");
         break;
      case MMEV_PAUSE:
         return("MMEV_PAUSE");
         break;
      case MMEV_PAUSE_FAIL:
         return("MMEV_PAUSE_FAIL");
         break;
      case MMEV_CAPTURE:
         return("MMEV_CAPTURE");
         break;
      case MMEV_CAPTURE_FAIL:
         return("MMEV_CAPTURE_FAIL");
         break;
      case MMEV_RESUME:
         return("MMEV_RESUME");
         break;
      case MMEV_RESUME_FAIL:
         return("MMEV_RESUME_FAIL");
         break;
      case MMEV_SEEK:
         return("MMEV_SEEK");
         break;
      case MMEV_SEEK_FAIL:
         return("MMEV_SEEK_FAIL");
         break;
      case MMEV_RESET:
         return("MMEV_RESET");
         break;
      case MMEV_RESET_FAIL:
         return("MMEV_RESET_FAIL");
         break;
      case MMEV_RESET_ACK:
         return("MMEV_RESET_ACK");
         break;
      case MMEV_RESET_ACK_FAIL:
         return("MMEV_RESET_ACK_FAIL");
         break;
      case MMEV_RECORD_ACK:
         return("MMEV_RECORD_ACK");
         break;
      case MMEV_RECORD_ACK_FAIL:
         return("MMEV_RECORD_ACK_FAIL");
         break;
      case MMEV_VIDEO_RECORD_STARTED:
         return("MMEV_VIDEO_RECORD_STARTED");
         break;
      case MMEV_STOP_ACK:
         return("MMEV_STOP_ACK");
         break;
      case MMEV_STOP_ACK_FAIL:
         return("MMEV_STOP_ACK_FAIL");
         break;
      case DMEV_CONNECT:
         return("DMEV_CONNECT");
         break;
      case DMEV_DISCONNECT:
         return("DMEV_DISCONNECT");
         break;
      case DMEV_RESERVE_RESOURCE :
         return("DMEV_RESERVE_RESOURCE");
         break;
      case DMEV_CONNECT_FAIL:
         return("DMEV_CONNECT_FAIL");
         break;
      case DMEV_GET_TX_PORT_INFO:
         return("DMEV_GET_TX_PORT_INFO");
         break;
      case DMEV_GET_RX_PORT_INFO:
         return("DMEV_GET_RX_PORT_INFO");
         break;
      case DMEV_PORT_CONNECT:
         return ("DMEV_PORT_CONNECT");
         break;
      case DMEV_PORT_CONNECT_FAIL:
         return ("DMEV_PORT_CONNECT_FAIL");
         break;
      case TDX_GETDIG:
         return("TDX_GETDIG");
         break;
      case TDX_CST:
         return("TDX_CST");
         break;
#ifdef TDM_AUDIO
      case TDX_RECORD:
         return("TDX_RECORD");
         break;
#endif

      case APPEV_IFRAME_REQ:
         return("APPEVT_IFRAME_REQ");
         break;
      case APPEV_WAIT_TIMEOUT:
         return("APPEVT_WAIT_TIMEOUT");
         break;
      default:
         sprintf(cTemp, "UNKNOWN EVENT 0x%X", nEvent);
         return(cTemp);
         break;
   }
}

// end of file //
