/**
* @file sprop.cpp
* @brief Multimedia demo - sprop convertion utilities between base64 and bytestream
* @date Dec 16, 2009
*
* DIALOGIC CONFIDENTIAL 
* Copyright  2009 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to 
* the source code ("Material") are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its 
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced, 
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual 
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/



#include <stdio.h>
#include <stdlib.h>

#include "sprop.h"

/*
** Translation Table as described in RFC1113
*/
static const char cb64[]="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

/*
** Translation Table to decode (created by author)
*/
static const char cd64[]="|$$$}rstuvwxyz{$$$$$$$>?@ABCDEFGHIJKLMNOPQRSTUVW$$$$$$XYZ[\\]^_`abcdefghijklmnopq";



/*
** encodeblock
**
** encode 3 8-bit binary bytes as 4 '6-bit' characters
*/
void encodeblock( unsigned char in[3], unsigned char out[4], int len )
{
    out[0] = cb64[ in[0] >> 2 ];
    out[1] = cb64[ ((in[0] & 0x03) << 4) | ((in[1] & 0xf0) >> 4) ];
    out[2] = (unsigned char) (len > 1 ? cb64[ ((in[1] & 0x0f) << 2) | ((in[2] & 0xc0) >> 6) ] : '=');
    out[3] = (unsigned char) (len > 2 ? cb64[ in[2] & 0x3f ] : '=');
}

/*
** encodeb64
**
** base64 encode a stream adding padding and line breaks as per spec.
*/

void encodeb64( unsigned char *inMessage, int inSizeRead, char *outMessage, int *outSizeRead )
{
    unsigned char in[3], out[4];
    int i, len, blocksout = 0;
    int indexIN=0, indexOUT = 0;

    while (indexIN < inSizeRead)  {
       
        len = 0;
        for( i = 0; i < 3; i++ ) { 
            if( indexIN < inSizeRead ) {
                in[i] = inMessage[indexIN];
                len++;
            }
            else {
                in[i] = 0;
            }
            indexIN++;
        }

        if( len ) {
            encodeblock( in, out, len );
            for( i = 0; i < 4; i++ ) {
                outMessage[indexOUT] = out[i];
                indexOUT++;
            }
        }

    }

    *outSizeRead  = indexOUT;
}

/*
** decodeblock
**
** decode 4 '6-bit' characters into 3 8-bit binary bytes
*/
void decodeblock( unsigned char in[4], unsigned char out[3] )
{   
    out[ 0 ] = (unsigned char ) (in[0] << 2 | in[1] >> 4);
    out[ 1 ] = (unsigned char ) (in[1] << 4 | in[2] >> 2);
    out[ 2 ] = (unsigned char ) (((in[2] << 6) & 0xc0) | in[3]);
}

/*
** decodeb64
**
** decode a base64 encoded stream discarding padding, line breaks and noise
*/

void decodeb64( char *inMessage, int inSizeRead, unsigned char *outMessage, int *outSizeRead )
{
    unsigned char in[4], out[3], v;
    int i, len;
    int indexIN=0, indexOUT = 0;

    
    while( indexIN <= inSizeRead) {
        for( len = 0, i = 0; i < 4 && (indexIN <= inSizeRead); i++ ) {
            v = 0;
            while( (indexIN <= inSizeRead) && v == 0 ) {
                v = inMessage[indexIN];
                v = (unsigned char) ((v < 43 || v > 122) ? 0 : cd64[ v - 43 ]);
                if( v ) {
                    v = (unsigned char) ((v == '$') ? 0 : v - 61);
                }
                indexIN++;
            }
            if( indexIN <= inSizeRead ) {
                len++;
                if( v ) {
                    in[ i ] = (unsigned char) (v - 1);
                }
            }
            else {
                in[i] = 0;
            }
            
        }

        if( len ) {
            decodeblock( in, out );
            for( i = 0; i < len - 1; i++ ) {
                outMessage[indexOUT] = out[i];
                indexOUT++;
            }
        }
    }

    *outSizeRead = indexOUT;
   
}


/*
** insertStartCode
**
** insert an H264 start code, update the message pointer and size accordingly
*/

void insertStartCode (unsigned char **message, int *size)
{

   **message=0x0;
   *message +=1;

   **message=0x0;
   *message +=1;

   **message=0x0;
   *message +=1;

   **message=0x1;
   *message +=1;
   
   *size +=4;

}


/*
** insertSpropDelimiter
**
** insert an sprop delimiter, update the message pointer and size accordingly
*/

void insertSpropDelimiter (char **message, int *size)
{
   
   **message=',';
   *message +=1;
   
   *size +=1;

}


/*
** lookForStartCode
**
** look for an H264 start code at the current position
*/

unsigned char lookForStartCode (unsigned char *message)
{

   unsigned char nalUnitType, ret;

   if ( (message[0]==0x0) && (message[1]==0x0) && (message[2]==0x0) && (message[3]==0x1) )
   {
      nalUnitType=message[4] & 0x1F;
      //printf ("\nNAL UNIT TYPE <%d>\n", nalUnitType);
      ret=1;
   }
   else
      ret=0;
         
    return ret;
}


/*
** encodeSprop
**
** encode Sprop in base 64 format from DCI in byte stream format
*/

int encodeSprop( unsigned char *inDci, int inDciSize, char *outB64, int *outB64Size )

{

   unsigned char *pInMessage = inDci;
   unsigned char foundStartCode;

   int index;
   int psSize=0;
   char *pOutMessage = outB64;
   int decodedSize=0;


   if (inDciSize > DCI_MAX_SIZE)
      return (SPROP_SIZE_ERROR);
   if (inDci==NULL || outB64==NULL || outB64Size==NULL)
      return (SPROP_PARAMETER_ERROR);

   *outB64Size =0;

   // must start with a Start Code
   foundStartCode = lookForStartCode(pInMessage);
   if (!foundStartCode) 
   {
      return SPROP_START_CODE_NOT_FOUND;
   }
   else
   {

      inDciSize-=4;
      pInMessage+=4;
      inDci+=4;
      
      for (index=0; index < inDciSize; index++)
      {
         foundStartCode = lookForStartCode(pInMessage);
         
         if (foundStartCode)
         {
            encodeb64( inDci, psSize, pOutMessage, &decodedSize );
            
            inDciSize-=3;
            pInMessage+=4;
            inDci=inDci + psSize +4;
            psSize=0;
            *outB64Size += decodedSize;
            pOutMessage += decodedSize;

            insertSpropDelimiter (&pOutMessage, outB64Size);
         }
         else 
         {
            psSize++;
            pInMessage++;
         }
      }
      encodeb64( inDci, psSize, pOutMessage, &decodedSize );
      *outB64Size += decodedSize;
   }

   return SPROP_SUCCESS;
}



/*
** decodeSprop
**
** decode Sprop from base 64 format into DCI in byte stream format
*/

int decodeSprop( char *inB64, int inB64Size, unsigned char *outDci, int *outDciSize )
{
   int index;
   int psSize=0;
   unsigned char *pOutMessage = outDci;
   char *pInMessage = inB64;
   int decodedSize=0;

   if (inB64Size > B64_MAX_SIZE)
      return (SPROP_SIZE_ERROR);
   if (inB64==NULL || outDci==NULL || outDciSize==NULL)
      return (SPROP_PARAMETER_ERROR);

   *outDciSize =0;


   for (index=0; index < inB64Size; index++)
   {
      if (inB64[index] == ',')
      {
         insertStartCode (&pOutMessage, outDciSize);

         decodeb64( pInMessage, psSize, pOutMessage, &decodedSize );
         
         pInMessage=pInMessage + psSize +1;
         psSize=0;
         *outDciSize += decodedSize;
         pOutMessage += decodedSize;
      }
      else 
         psSize++;
   }

   insertStartCode (&pOutMessage, outDciSize);

   decodeb64( pInMessage, psSize, pOutMessage, &decodedSize );
   *outDciSize += decodedSize;

   return SPROP_SUCCESS;
}
