/**
 * @file  Main.cpp
 * @brief Media Toolkit Media Parsing Sample Code Implementation
 * @date  April 4. 22, 2010
 *
 * DIALOGIC CONFIDENTIAL      
 * Copyright (C) 2011 Dialogic Corporation. All Rights Reserved.
 * 
 * The source code contained or described herein and all documents related to
 * the source code ("Material") are owned by Dialogic Corporation or its 
 * suppliers or licensors.  Title to the Material remains with Dialogic 
 * Corporation or its suppliers and licensors.  The Material contains trade
 * secrets and proprietary and confidential information of Dialogic or its 
 * suppliers and licensors.  The Material is protected by worldwide copyright
 * and trade secret laws and treaty provisions.  No part of the Material may be
 * used, copied, reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way without Dialogic's prior express written
 * permission.
 * 
 * No license under any patent, copyright, trade secret or other intellectual 
 * property right is granted to or conferred upon you by disclosure or delivery
 * of the Materials, either expressly, by implication, inducement, estoppel or
 * otherwise.  Any license under such intellectual property rights must be
 * express and approved by Dialogic in writing.
 */

/*---------------------------- Standard Includes ----------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <time.h>




/*---------------------------- MTK Includes ---------------------------*/
#include "mp_mtklib.h"





/*----------------------------- Definitions --------------------------------*/
#define VERSION_X   1       //Version of program (vX.Y.Z)
#define VERSION_Y   0       //X.Y is version
#define VERSION_Z   70811   //Z is Month & Day & Year combined

#define SECONDS_FROM_1904_TO_1970   2082758400   //Adjusts POSIX date format to MPEG-4 FF date format

#define	FILE_INFO_SIZE      20000   //Max length of file information recepient
#define MAX_FILENAME_SIZE   120     //Max length of the file's name






/*---------------------------- Application globals -------------------------*/
unsigned int    bSdp       = 1;  //Print SDP info, default yes
unsigned int    bTrunc     = 0;  //Indicates possible truncation of file info (insufficient space)
unsigned int    fileType   = 1;  //Indicates type of file

/*! Prompt string printed with date and version */
char *PromptStr = "mptest - Dialogic Corporation 2011";


/*! File type strings */
char *FileTypeStr [] =
{
    "Unknown",
    "ISO",      //MP_FILE_TYPE_ISO = 1
    "3GP",
    "MP4",
    "F4V",
    "3G2",
    "AVC",
    "Not Supported",
    "Not Supported",
    "Not Supported",
    "DMF"      //MP_FILE_TYPE_DMF = 10
};

/*! Strings for block types */
char *BlockTypeStr [] =
{
    "Undefined",
    "Presentation",     //MP_BLK_TYPE_PRESENTATION=1
    "Unknown",
    "AUDIO",
    "VIDEO",
    "HINT",
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",
    "AMR",
    "AAC",
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",
    "H263",
    "MPEG4",
    "H264",
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",
    "RTP"
};

/*! Strings for codec types */
char *CodecStrAud [] =
{
    "Unknown",      //MP_S_CODEC_UNKNOWN = 0
    "OKI ADPCM",    //Values 1-99 are reserved to match MM_DATA_FORMAT_* definitions in mmlib.h
    "Undefined",
    "G.711 ALAW",
    "G.726",
    "Undefined",
    "Undefined",
    "G.711 MULAW",
    "PCM",
    "Undefined",
    "Undefined",  //10
    "Undefined",
    "G.729A",
    "Microsoft GSM",
    "ETSI GSM",
    "ETSI TIPHON Bit Order GSM",
    "TRUESPEECH",
    "G.711 ALAW 8-Bit Rev",
    "G.711 ALAW 16-Bit Rev",
    "G.711 MULAW 8-Bit Rev",
    "G.711 MULAW 16-Bit Rev",  //20
    "G.721",
    "G.721 8-Bit Rev",
    "G.721 16-Bit Rev",
    "G.721 16-Bit Rev Nibble Swap",
    "IMA ADPCM",
    "AMR-NB 4.75",
    "AMR-NB 5.15",
    "AMR-NB 5.9",
    "AMR-NB 6.7",
    "AMR-NB 7.4",  //30
    "AMR-NB 7.95",
    "AMR-NB 10.2",
    "AMR-NB 12.2",
    "G.723.1 5.3",
    "G.723.1 6.3",
    "EVRC",
    "QCELP 8",
    "QCELP 13",
    "GSM EFR",
    "G.722 48",  //40
    "G.722 56",
    "G.722 64",
    "AMR-WB 6.6",
    "AMR-WB 8.85",
    "AMR-WB 12.65",
    "AMR-WB 14.25",
    "AMR-WB 15.85",
    "AMR-WB 18.25",
    "AMR-WB 19.85",
    "AMR-WB 23.05",  //50
    "AMR-WB 23.85",
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",  //60
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",  //70
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",  //80
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",  //90
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",
    "Undefined",
    "AMR-NB",  //100   //MP_ISO_AUD_CODEC_AMR_NB
    "AMR-WB",
    "AAC"
};

/*! Strings for video codec types */
char *CodecStrVid [] =
{
    "Unknown",      //MP_S_CODEC_UNKNOWN = 0
    "H263",
    "MPEG4",
    "H264"
};

/*! Strings for hint protocol types */
char *CodecStrHint [] =
{
    "Unknown",
    "RTP"       //MP_HINT_PROTOCOL_RTP
};

/*! Strings for packetization modes */
char *PackModeStr [] =
{
    "None",     //MP_S_PACK_NONE = 0
    "Unknown",
    "MPEG4",
    "H263_2190",
    "H263_2429",
    "H264_MODE0 (single-NAL)",
    "H264_MODE1 (non-interleaved)",
    "H264_MODE2 (interleaved)",
    "AMR_OCTET_ALIGN",
    "AMR_BW_EFF"
};

/*! Strings for AMR modes */
char *AmrModeStr [] =
{
    "4.75",     //mode 0
    "5.15",
    "5.90",
    "6.70",
    "7.40",
    "7.95",
    "10.2",
    "12.2",
    "AMR_SID",
    "NO_DATA"
};

char *AmrWBModeStr [] =
{
    "6.60",     //mode 0
    "8.85",
    "12.65",
    "14.25",
    "15.85",
    "18.25",
    "19.85",
    "23.05",
    "23.85",
    "AMR_SID",
    "NO_DATA"
};

char *TextHJustStr [] =
{
    "Right justified",      // -1
    "Left justified",
    "Centered"
};

char *TextVJustStr [] =
{
    "Bottom justified",     // -1
    "Top justified",
    "Centered"
};

char *HintStreamProtStr [] =
{
    "No streaming protocol",
    "RTP streaming protocol"        //MP_HINT_PROTOCOL_RTP = 1
};

char *HintRTPFlagStr [] =
{
    "No flag",
    "Use timestamp offset field",       //MP_HINTRTP_FLAG_TIMESTAMP = 1
    "Use sequence offset field"         //MP_HINTRTP_FLAG_SEQUENCE = 2
    "Use timestamp and sequence offset field"
};

char *MpErrorStr [] =
{
    "",                             //0x00
    "MPERR_INVALID_FILE_TYPE",
    "",
    "MPERR_INVALID_FILE",
    "",
    "MPERR_INVALID_POINTER",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "MPERR_FILE_OPEN_FAILED",       //0x10
    "MPERR_FILE_ACCESS_DENIED",
    "MPERR_INCOMP_ACCESS_MODE",
    "",
    "",
    "MPERR_INTERNAL_RESOURCE",
    "MPERR_FILE_FORMAT",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",                             //0x20
    "MPERR_INVALID_STRUCT_VERSION"
};


/*---------------------------- Functions ---------------------------*/


/*!
***************************************************************************
\brief      Prints program/version information

\param[in]  none

\return     none																				
****************************************************************************/


void PrintProgramAndVersion(void)
{
    printf("\n%s - v%d.%d.%d\n", PromptStr, VERSION_X, VERSION_Y, VERSION_Z);
}


/*!
***************************************************************************
\brief      Dump a buffer in hexadecimal

                DumpHex dumps the 'len' bytes of the buffer pointed by 'pbuf',
                with up to 16 bytes per line.

\param[in]  pbuf    points on the buffer to dump
\param[in]  len     number of bytes to dump

\return     none																				
****************************************************************************/
void DumpHex (unsigned char *pbuf, int len)
{
    int	n;

    while (len)
    {
        n = (len > 16) ? 16 : len;
        len -= n;

        printf("\t      ");

        for ( ; n>0 ; n--)
        {
            printf("%.2X ", *pbuf++);
        }

        printf("\n");
    }
}

/*!
***************************************************************************
\brief      Append/print the profile/level type for MPEG-4

\param[in]  profile_and_level_ind   value of profile/level indication

\return     none
****************************************************************************/
void AppendMPEG4ProfileLevelTypeString(unsigned char profile_and_level_ind)
{
    switch(profile_and_level_ind)
    {
        case 1:
            printf(" (Simple Profile/Level 1)");
        break;
        case 2:
            printf(" (Simple Profile/Level 2)");
        break;
        case 3:
            printf(" (Simple Profile/Level 3)");
        break;
        case 8:
            printf(" (Simple Profile/Level 0)");
        break;
        case 17:
            printf(" (Simple Scalable Profile/Level 1)");
        break;
        case 18:
            printf(" (Simple Scalable Profile/Level 2)");
        break;
        case 33:
            printf(" (Core Profile/Level 1)");
        break;
        case 34:
            printf(" (Core Profile/Level 2)");
        break;
        case 50:
            printf(" (Main Profile/Level 2)");
        break;
        case 51:
            printf(" (Main Profile/Level 3)");
        break;
        case 52:
            printf(" (Main Profile/Level 4)");
        break;

        //Will ignore the higher profiles for now

        default:
            break;
    }

    printf("\n");

    return;
}



/*!
***************************************************************************
\brief      Append/print the profile/level type for MPEG-4

\param[in]  profile_and_level_ind   value of profile/level indication

\return     none
****************************************************************************/
void PrintH264ProfileLevelInfo(unsigned char profile, unsigned char profile_iop, unsigned char level)
{

    //Print Profile Info
    printf("\t      profile                      = %u", profile);

    switch(profile)
    {
        case 66:
            printf("  (Baseline)");
        break;
        case 77:
            printf("  (Main)");
        break;
        case 88:
            printf("  (Extended)");
        break;
        case 100:
            printf("  (High)");
        break;
        case 110:
            printf("  (High 10)");
        break;
        case 122:
            printf("  (High 4:2:2)");
        break;
        case 144:
            printf("  (High 4:4:4)");
        break;

        default:
            break;
    }
    printf("\n");

    //Print Profile_iop Info
    printf("\t      profile_iop                  = %u (0x%x)\n", profile_iop, profile_iop);

    //Print Level Info
    if (level == 11 && (profile_iop & 0x10))
    {
        printf("\t      level                        = %u  (Level 1b)\n", level);
    }
    else if (level % 10 == 0)
    {
        printf("\t      level                        = %u  (Level %u)\n", level, level/10);
    }
    else
    {
        printf("\t      level                        = %u  (Level %.1f)\n", level, (double)level/10);
    }


    return;

}




/*!
***************************************************************************
\brief      Print the file-level information retrieved.

\param[in]  pFileInfoDesc   pointer to a File Information Descriptor

\return     none
****************************************************************************/
void PrintFileLevelInfo (MP_FILE_INFO_DESC *pFileInfoDesc)
{
    unsigned int i;

    printf("\t------------------------------------------------------\n");
    printf("\tFile Info Descriptor:\n");
    printf("\t------------------------------------------------------\n");
    printf("\t   total size                      = %u\n", pFileInfoDesc->fileInfoSize);
    printf("\t   file type                       = %u (%s)\n", pFileInfoDesc->fileType, FileTypeStr [pFileInfoDesc->fileType]);
    printf("\t   format (major brand)            = %s\n", pFileInfoDesc->format);
    if (pFileInfoDesc->fileType == MP_FILE_TYPE_ISO ||
        pFileInfoDesc->fileType == MP_FILE_TYPE_3GP ||
        pFileInfoDesc->fileType == MP_FILE_TYPE_MP4 ||
        pFileInfoDesc->fileType == MP_FILE_TYPE_F4V)
    {
        int	v, x, y;
        x = pFileInfoDesc->version / 256;
        y = pFileInfoDesc->version & 0xFF;
        v = pFileInfoDesc->format [3] - '0';
        printf("\t   version                         = %lu (v%d.%d.%d)\n", pFileInfoDesc->version, v, x, y);
    }
    else
    {
        printf("\t   version                         = %lu\n", pFileInfoDesc->version);
    }
    printf("\t   compatible brands               = %s\n", pFileInfoDesc->compatible_brands);
    printf("\t   flags                           = 0x%.8lX", pFileInfoDesc->flags);
    if (pFileInfoDesc->flags & MP_DESC_FLAG_INSUFF_SPACE)
    {
        printf(" (insuff space!!!)");
        bTrunc = 1;
    }
    printf("\n");
    printf("\t   num video tracks                = %lu\n", pFileInfoDesc->NbVideoTracks);
    for (i=0; i< pFileInfoDesc->NbVideoTracks; i++)
    {
        printf("\t   Video Track Descriptor [%d]:\n", i);
        printf("\t       stream ID                   = %lu\n", pFileInfoDesc->video_track_descriptor[i].track_id);
        printf("\t       num sync points             = %lu\n", pFileInfoDesc->video_track_descriptor[i].NbSyncPoints);
    }
    printf("\n");
    printf("\t   num audio tracks                = %lu\n", pFileInfoDesc->NbAudioTracks);
    for (i=0; i< pFileInfoDesc->NbAudioTracks; i++)
    {
        printf("\t   Audio Track Descriptor [%d]:\n", i);
        printf("\t       stream ID                   = %lu\n", pFileInfoDesc->audio_track_descriptor[i].track_id);
    }
    printf("\n");
    printf("\t   num video hint tracks           = %lu\n", pFileInfoDesc->NbVideoHintTracks);
    printf("\t   num audio hint tracks           = %lu\n", pFileInfoDesc->NbAudioHintTracks);


    return;
}



/*!
***************************************************************************
\brief      Print the stream-level information retrieved (and presentation info).

\param[in]  pFileInfoDesc   points to the FILE_INFO_DESC descriptor to print

\return none
****************************************************************************/
void PrintStreamLevelInfo (MP_FILE_INFO_DESC *pFileInfoDesc)
{

    unsigned char   *pFileInfo;         //pointer on the file info
    unsigned short  infoSizeLeft;       //size left in file info buffer
    unsigned short  mode;               //AMR stream mode
    unsigned short  mask;               //AMR mode mask
    unsigned int    blockSize;          //size of current block
    unsigned int    blockType;          //size of current block
    unsigned long   creationTime;       //Creation time for given block
    time_t          c_time;             //Time structure
    char            ct[100];            //Array to store creation time strings
    unsigned short  dciSize;            //Capped DCI size
    unsigned short  sdpSize;            //Capped SDP size

    //Block structure pointers
    MP_BLK_HEADER               *pHdr;              //pointer on Header block
    MP_BLK_PRESENTATION         *pPresentation;     //pointer on Presentation block
    MP_BLK_STREAM_AUDIO         *pStreamAudio = NULL;      //pointer on Audio Stream block
    MP_BLK_STREAM_VIDEO         *pStreamVideo = NULL;      //pointer on Video Stream block
    MP_BLK_STREAM_HINT          *pStreamHint  = NULL;      //pointer on Text Stream block
    MP_BLK_STREAM_UNKNOWN       *pStreamUnknown;    //pointer on Unknown Stream block
    MP_BLK_CODEC_AUDIO_AMR      *pAmr;              //pointer on AMR codec block
    MP_BLK_CODEC_AUDIO_AAC      *pAac;              //pointer on AAC block
    MP_BLK_CODEC_VIDEO_H263     *pH263;             //pointer on H.263 block
    MP_BLK_CODEC_VIDEO_MPEG4    *pMpeg4;            //pointer on MPEG-4 block
    MP_BLK_CODEC_VIDEO_H264     *pH264;             //pointer on H.264 block
    MP_BLK_PROTOCOL_HINT_RTP	*pRtp;              //pointer on RTP protocol block


    //Initialize pointer to file info and info size
    pFileInfo = pFileInfoDesc -> pFileInfo;
    infoSizeLeft = pFileInfoDesc -> fileInfoSize;
	
    while (infoSizeLeft > 0)
    {
        //Determine blockSize and blockType
        pHdr = (MP_BLK_HEADER *)pFileInfo;
        blockSize = pHdr->blkSize;
        blockType = pHdr->blkType;

	    switch (blockType)
	    {
            //*****************************************************************************
            //     PRESENTATION BLOCK
            //*****************************************************************************

            case MP_BLK_TYPE_PRESENTATION:
                printf("\t------------------------------------------------------\n");
                printf("\tPresentation Block Info:\n");
                printf("\t------------------------------------------------------\n");
                pPresentation = (MP_BLK_PRESENTATION *)pFileInfo;
                printf("\t   blk size                        = %u\n", blockSize);
                printf("\t   blk type                        = %u (%s)\n", blockType, BlockTypeStr [blockType]);
                creationTime = pPresentation -> creationTime;
                c_time = (time_t)(creationTime - SECONDS_FROM_1904_TO_1970);
                if (ctime(&c_time) != NULL)
                {
                    strcpy(ct, (const char *)ctime(&c_time));
                }
                else
                {
                    strcpy(ct, "\n");
                }
                printf("\t   creation time                   = %s", ct);
                printf("\t   duration                        = %lu ms\n", pPresentation -> duration);
                printf("\t   stream count                    = %u\n", pPresentation -> streamCount);
                printf("\t   max interleave                  = %lu ms\n", pPresentation -> maxInterleave);
                sdpSize = pPresentation -> sdpSize;
                printf("\t   session-level SDP size          = %u bytes\n", sdpSize);
                if (bSdp)
                {
                    if (sdpSize > MP_MAX_SDP_SIZE)
                    {
                        sdpSize = MP_MAX_SDP_SIZE;
                    }
                    if (sdpSize && pPresentation -> pSdp)
                    {
                        printf("=========================\n");
                        printf("%s", pPresentation -> pSdp);
                        printf("=========================\n");
                    }
                }

                break;  // end of Presentation block
               
            //*****************************************************************************
            //     STREAM BLOCKS
            //*****************************************************************************

            case MP_BLK_TYPE_STREAM_AUDIO:
                printf("\t------------------------------------------------------\n");
                printf("\tAudio Stream Info:\n");
                printf("\t------------------------------------------------------\n");
                pStreamAudio = (MP_BLK_STREAM_AUDIO *)pFileInfo;
                printf("\t   blk size                        = %u\n", blockSize);
                printf("\t   blk type                        = %u (%s)\n", blockType, BlockTypeStr [blockType]);
                printf("\t   stream ID                       = %lu\n", pStreamAudio -> streamID);
                printf("\t   codec                           = %u (%s)\n", pStreamAudio -> codec, CodecStrAud [pStreamAudio -> codec]);
                creationTime = pStreamAudio -> creationTime;
                c_time = (time_t)(creationTime - SECONDS_FROM_1904_TO_1970);
                if (ctime(&c_time) != NULL)
                {
                    strcpy(ct, (const char *)ctime(&c_time));
                }
                else
                {
                    strcpy(ct, "\n");
                }
                printf("\t   creation time                   = %s", ct);
                printf("\t   duration                        = %lu ms\n", pStreamAudio -> duration);
                printf("\t   bitrate                         = %.2f kbps\n", pStreamAudio -> bitRate);
                printf("\t   frame rate                      = %.2f fps\n", pStreamAudio -> frameRate);
                printf("\t   sample rate (timescale )        = %lu Hz\n", pStreamAudio -> sampleRate);
                printf("\t   sample size                     = %lu\n", pStreamAudio -> sampleSize);				
                printf("\t   max sample size                 = %lu\n", pStreamAudio -> maxSampleSize);
                printf("\t   sample count                    = %lu\n", pStreamAudio -> sampleCount);				
                printf("\t   byte count                      = %lu\n", pStreamAudio -> byteCount);
                printf("\t   media handler Name              = %s\n", pStreamAudio -> handlerName);
                printf("\t   channel count                   = %u\n", pStreamAudio -> channelCount);
                printf("\t   bits per sample                 = %u bits\n", pStreamAudio -> bitsPerSample);

                break;

            case MP_BLK_TYPE_STREAM_VIDEO:
                printf("\t------------------------------------------------------\n");
                printf("\tVideo Stream Info:\n");
                printf("\t------------------------------------------------------\n");
                pStreamVideo = (MP_BLK_STREAM_VIDEO *)pFileInfo;
                printf("\t   blk size                        = %u\n", blockSize);
                printf("\t   blk type                        = %u (%s)\n", blockType, BlockTypeStr [blockType]);
                printf("\t   stream ID                       = %lu\n", pStreamVideo -> streamID);
                printf("\t   codec                           = %u (%s)\n", pStreamVideo -> codec, CodecStrVid [pStreamVideo -> codec]);
                creationTime = pStreamVideo -> creationTime;
                c_time = (time_t)(creationTime - SECONDS_FROM_1904_TO_1970);
                if (ctime(&c_time) != NULL)
                {
                    strcpy(ct, (const char *)ctime(&c_time));
                }
                else
                {
                    strcpy(ct, "\n");
                }
                printf("\t   creation time                   = %s", ct);
                printf("\t   duration                        = %lu ms\n", pStreamVideo -> duration);
                printf("\t   bitrate                         = %.2f kbps\n", pStreamVideo -> bitRate);
                printf("\t   frame rate                      = %.2f fps\n", pStreamVideo -> frameRate);
                printf("\t   sample rate (timescale )        = %lu Hz\n", pStreamVideo -> sampleRate);
                printf("\t   sample size                     = %lu\n", pStreamVideo -> sampleSize);				
                printf("\t   max sample size                 = %lu\n", pStreamVideo -> maxSampleSize);
                printf("\t   sample count                    = %lu\n", pStreamVideo -> sampleCount);				
                printf("\t   byte count                      = %lu\n", pStreamVideo -> byteCount);
                printf("\t   media handler Name              = %s\n", pStreamVideo -> handlerName);
                printf("\t   width                           = %u pixels\n", pStreamVideo -> width);
                printf("\t   height                          = %u pixels\n", pStreamVideo -> height);
                printf("\t   layer                           = %d\n", pStreamVideo -> layer);
                printf("\t   horizontal resolution           = %u dpi\n", pStreamVideo -> hResolution);
                printf("\t   vertical resolution             = %u dpi\n", pStreamVideo -> vResolution);

                break;

            case MP_BLK_TYPE_STREAM_HINT:
                printf("\t------------------------------------------------------\n");
                printf("\tHint Stream Info:\n");
                printf("\t------------------------------------------------------\n");
                pStreamHint = (MP_BLK_STREAM_HINT *)pFileInfo;
                printf("\t   blk size                        = %u\n", blockSize);
                printf("\t   blk type                        = %u (%s)\n", blockType, BlockTypeStr [blockType]);
                printf("\t   stream ID                       = %lu\n", pStreamHint -> streamID);
                printf("\t   codec                           = %u (%s)\n", pStreamHint -> codec, CodecStrHint [pStreamHint -> codec]);
                creationTime = pStreamHint -> creationTime;
                c_time = (time_t)(creationTime - SECONDS_FROM_1904_TO_1970);
                if (ctime(&c_time) != NULL)
                {
                    strcpy(ct, (const char *)ctime(&c_time));
                }
                else
                {
                    strcpy(ct, "\n");
                }
                printf("\t   creation time                   = %s", ct);
                printf("\t   duration                        = %lu ms\n", pStreamHint -> duration);
                printf("\t   bitrate                         = %.2f kbps\n", pStreamHint -> bitRate);
                printf("\t   frame rate                      = %.2f fps\n", pStreamHint -> frameRate);
                printf("\t   sample rate (timescale )        = %lu Hz\n", pStreamHint -> sampleRate);
                printf("\t   sample size                     = %lu\n", pStreamHint -> sampleSize);				
                printf("\t   max sample size                 = %lu\n", pStreamHint -> maxSampleSize);
                printf("\t   sample count                    = %lu\n", pStreamHint -> sampleCount);				
                printf("\t   byte count                      = %lu\n", pStreamHint -> byteCount);
                printf("\t   media handler Name              = %s\n", pStreamHint -> handlerName);
                printf("\t   reference stream IDs            = %lu %lu %lu %lu\n", 
                            pStreamHint -> refStreamID[0], pStreamHint -> refStreamID[1],
                            pStreamHint -> refStreamID[2], pStreamHint -> refStreamID[3]);
                sdpSize = pStreamHint -> sdpSize;
                printf("\t   track-level SDP size            = %u bytes\n", sdpSize);
                if (bSdp)
                {
                    if (sdpSize > MP_MAX_SDP_SIZE)
                    {
                        sdpSize = MP_MAX_SDP_SIZE;
                    }
                    if (sdpSize && pStreamHint -> pSdp)
                    {
                        printf("=========================\n");
                        printf("%s", pStreamHint -> pSdp);
                        printf("=========================\n");
                    }
                }

                break;

            case MP_BLK_TYPE_STREAM_UNKNOWN:
                printf("\t------------------------------------------------------\n");
                printf("\tUnknown Stream Info:\n");
                printf("\t------------------------------------------------------\n");
                pStreamUnknown = (MP_BLK_STREAM_UNKNOWN *)pFileInfo;
                printf("\t   blk size                        = %u\n", blockSize);
                printf("\t   blk type                        = %u (%s)\n", blockType, BlockTypeStr [blockType]);
                printf("\t   stream ID                       = %lu\n", pStreamUnknown -> streamID);
                printf("\t   codec                           = %u (%s)\n", pStreamUnknown -> codec, CodecStrVid [pStreamUnknown -> codec]);
                creationTime = pStreamUnknown -> creationTime;
                c_time = (time_t)(creationTime - SECONDS_FROM_1904_TO_1970);
                if (ctime(&c_time) != NULL)
                {
                    strcpy(ct, (const char *)ctime(&c_time));
                }
                else
                {
                    strcpy(ct, "\n");
                }
                printf("\t   creation time                   = %s", ct);

                break;

            //*****************************************************************************
            //     CODEC/PROTOCOL BLOCKS
            //*****************************************************************************

            case MP_BLK_TYPE_AUDIO_CODEC_AMR:
                if (pStreamAudio != NULL)
                {
                    if (pStreamAudio->codec == MP_ISO_AUD_CODEC_AMR_NB || pStreamAudio->codec == MP_ISO_AUD_CODEC_AMR_WB)
                    {
                        printf("\t   ------------------------------------------------------\n");
                        printf("\t   AMR Codec Info:\n");
                        printf("\t   ------------------------------------------------------\n");
                        pAmr = (MP_BLK_CODEC_AUDIO_AMR *)pFileInfo;
                        printf("\t      blk size                     = %u\n", blockSize);
                        printf("\t      blk type                     = %u (%s)\n", blockType, BlockTypeStr [blockType]);
                        printf("\t      packetization mode           = %u (%s)\n", pAmr -> packetizationMode, PackModeStr [pAmr -> packetizationMode] );
                        if (pAmr -> ptime == 0)
                        {
                            printf("\t      ptime                        = %u (Unknown)\n", pAmr -> ptime);
                        }
                        else
                        {
                            printf("\t      ptime                        = %u\n", pAmr -> ptime);
                        }
                        printf("\t      AMR mode set                 = 0x%.4X - ", pAmr -> modeSet);
                        if (pStreamAudio->codec == MP_ISO_AUD_CODEC_AMR_NB)
                        {
                            for (mode=0, mask=MP_S_AMR_MODE_475; mode<8; mode++, mask<<=1)
                            {
                                if (pAmr -> modeSet & mask)
                                {
                                    printf("%s ", AmrModeStr [mode]);
                                }
                            }
                            if (pAmr -> modeSet & MP_S_AMR_MODE_SID)
                            {
                                printf("%s ", AmrModeStr [8]);
                            }
                            if (pAmr -> modeSet & MP_S_AMR_MODE_NODATA)
                            {
                                printf("%s ", AmrModeStr [9]);
                            }
                        }
                        else if (pStreamAudio->codec == MP_ISO_AUD_CODEC_AMR_WB)
                        {
                            for (mode=0, mask=MP_S_AMR_WB_MODE_660; mode<9; mode++, mask<<=1)
                            {
                                if (pAmr -> modeSet & mask)
                                {
                                    printf("%s ", AmrWBModeStr [mode]);
                                }
                            }
                            if (pAmr -> modeSet & MP_S_AMR_WB_MODE_SID)
                            {
                                printf("%s ", AmrWBModeStr [9]);
                            }
                            if (pAmr -> modeSet & MP_S_AMR_WB_MODE_NODATA)
                            {
                                printf("%s ", AmrWBModeStr [10]);
                            }
                        }
                        printf("kbits/s\n");
                        printf("\t      mode chg period              = %u\n", pAmr -> modeChangePeriod);
                        printf("\t      frames / sample              = %u\n", pAmr -> framesPerSample);
                        printf("\t      vendor                       = %s\n", pAmr -> vendor);
                    }
                    else
                    {
                        printf("ERROR - INCORRECT CODEC (%d) DURING AMR CODEC BLOCK - skipping ahead\n\n", pStreamAudio->codec);
                    }
                }

                //Now set pStreamAudio back to NULL at end of audio codec block.  Will need to be reset
                //for next audio stream (if there is another)
                pStreamAudio = NULL;


                break;

            case MP_BLK_TYPE_AUDIO_CODEC_AAC:
                if (pStreamAudio != NULL)
                {
                    if (pStreamAudio->codec == MP_ISO_AUD_CODEC_AAC)
                    {
                        printf("\t   ------------------------------------------------------\n");
                        printf("\t   AAC Codec Info:\n");
                        printf("\t   ------------------------------------------------------\n");
                        pAac = (MP_BLK_CODEC_AUDIO_AAC *)pFileInfo;
                        printf("\t      blk size                     = %u\n", blockSize);
                        printf("\t      blk type                     = %u (%s)\n", blockType, BlockTypeStr [blockType]);
                        printf("\t      stream type                  = %u\n", pAac -> dcStreamType);
                        printf("\t      buffer size                  = %u\n", pAac -> dcBufferSize);
                        printf("\t      avg bitrate                  = %lu\n", pAac -> dcAvgBitrate);
                        printf("\t      max bitrate                  = %lu\n", pAac -> dcMaxBitrate);
                        dciSize = pAac -> dcSpecInfoSize;
                        printf("\t      DCI size                     = %u bytes\n", dciSize);
                        if (dciSize > MP_MAX_DCI_SIZE)
                        {
                            dciSize = MP_MAX_DCI_SIZE;
                        }
                        if (dciSize && pAac -> pDcSpecInfo)
                        {
                            printf("\t      ===============================================\n");
                            DumpHex (pAac -> pDcSpecInfo, dciSize);
                            printf("\t      ===============================================\n");
                        }
                    }
                    else
                    {
                        printf("ERROR - INCORRECT CODEC (%d) DURING AAC CODEC BLOCK - skipping ahead\n\n", pStreamAudio->codec);
                    }
                }

                //Now set pStreamAudio back to NULL at end of audio codec block.  Will need to be reset
                //for next audio stream (if there is another)
                pStreamAudio = NULL;

                break;

            case MP_BLK_TYPE_VIDEO_CODEC_H263:
                if (pStreamVideo != NULL)
                {
                    if (pStreamVideo->codec == MP_VID_CODEC_H263)
                    {
                        printf("\t   ------------------------------------------------------\n");
                        printf("\t   H.263 Codec Info:\n");
                        printf("\t   ------------------------------------------------------\n");
                        pH263 = (MP_BLK_CODEC_VIDEO_H263 *)pFileInfo;
                        printf("\t      blk size                     = %u\n", blockSize);
                        printf("\t      blk type                     = %u (%s)\n", blockType, BlockTypeStr [blockType]);
                        printf("\t      packetization mode           = %u (%s)\n", pH263 -> packetizationMode, PackModeStr [pH263 -> packetizationMode] );
                        printf("\t      profile                      = %u", pH263 -> profile);
                        if (pH263 -> profile == 0)
                        {
                            printf(" (Baseline)");
                        }
                        printf("\n");

                        printf("\t      level                        = %u\n", pH263 -> level);
                        printf("\t      vendor                       = %s\n", pH263 -> vendor);
                    }
                    else
                    {
                        printf("ERROR - INCORRECT CODEC (%d) DURING H263 CODEC BLOCK - skipping ahead\n\n", pStreamVideo->codec);
                    }
                }

                //Now set pStreamVideo back to NULL at end of video codec block.  Will need to be reset
                //for next video stream (if there is another)
                pStreamVideo = NULL;

                break;

            case MP_BLK_TYPE_VIDEO_CODEC_MPEG4:
                if (pStreamVideo != NULL)
                {
                    if (pStreamVideo->codec == MP_VID_CODEC_MPEG4)
                    {
                        printf("\t   ------------------------------------------------------\n");
                        printf("\t   MPEG-4 Codec Info:\n");
                        printf("\t   ------------------------------------------------------\n");
                        pMpeg4 = (MP_BLK_CODEC_VIDEO_MPEG4 *)pFileInfo;
                        printf("\t      blk size                     = %u\n", blockSize);
                        printf("\t      blk type                     = %u (%s)\n", blockType, BlockTypeStr [blockType]);
                        printf("\t      packetization mode           = %u (%s)\n", pMpeg4 -> packetizationMode, PackModeStr [pMpeg4 -> packetizationMode] );
                        printf("\t      profile/level indication     = %u", pMpeg4 -> profile_and_level_ind);
                        AppendMPEG4ProfileLevelTypeString(pMpeg4 -> profile_and_level_ind);
                        printf("\t      stream type                  = %u\n", pMpeg4 -> dcStreamType);
                        printf("\t      buffer size                  = %u\n", pMpeg4 -> dcBufferSize);
                        printf("\t      avg bitrate                  = %lu\n", pMpeg4 -> dcAvgBitrate);
                        printf("\t      max bitrate                  = %lu\n", pMpeg4 -> dcMaxBitrate);
                        dciSize = pMpeg4 -> dcSpecInfoSize;
                        printf("\t      DCI size                     = %u bytes\n", dciSize);
                        if (dciSize > MP_MAX_DCI_SIZE)
                        {
                        dciSize = MP_MAX_DCI_SIZE;
                        }
                        if (dciSize && pMpeg4 -> pDcSpecInfo)
                        {
                            printf("\t      ===============================================\n");
                            DumpHex (pMpeg4 -> pDcSpecInfo, dciSize);
                            printf("\t      ===============================================\n");
                        }
                    }
                    else
                    {
                        printf("ERROR - INCORRECT CODEC (%d) DURING MPEG4 CODEC BLOCK - skipping ahead\n\n", pStreamVideo->codec);
                    }
                }

                //Now set pStreamVideo back to NULL at end of video codec block.  Will need to be reset
                //for next video stream (if there is another)
                pStreamVideo = NULL;

                break;

            case MP_BLK_TYPE_VIDEO_CODEC_H264:
                if (pStreamVideo != NULL)
                {
                    if (pStreamVideo->codec == MP_VID_CODEC_H264)
                    {
                        printf("\t   ------------------------------------------------------\n");
                        printf("\t   H264 Codec Info:\n");
                        printf("\t   ------------------------------------------------------\n");
                        pH264 = (MP_BLK_CODEC_VIDEO_H264 *)pFileInfo;
                        printf("\t      blk size                     = %u\n", blockSize);
                        printf("\t      blk type                     = %u  (%s)\n", blockType, BlockTypeStr [blockType]);
                        printf("\t      packetization mode           = %u   (%s)\n", pH264 -> packetizationMode, PackModeStr [pH264 -> packetizationMode] );
                        PrintH264ProfileLevelInfo(pH264 -> profile, pH264 -> profile_iop, pH264 -> level);

                        dciSize = pH264 -> dcSpecInfoSize;
                        printf("\t      DCI size                     = %u bytes\n", dciSize);
                        if (dciSize > MP_MAX_DCI_SIZE)
                        {
                        dciSize = MP_MAX_DCI_SIZE;
                        }
                        if (dciSize && pH264 -> pDcSpecInfo)
                        {
                            printf("\t      ===============================================\n");
                            DumpHex (pH264 -> pDcSpecInfo, dciSize);
                            printf("\t      ===============================================\n");
                        }
                        dciSize = pH264 -> dcSpecInfoSizeIsoFormat;
                        printf("\t      DCI size (ISO format)        = %u bytes\n", dciSize);
                        if (dciSize > MP_MAX_DCI_SIZE)
                        {
                            dciSize = MP_MAX_DCI_SIZE;
                        }
                        if (dciSize && pH264 -> pDcSpecInfoIsoFormat)
                        {
                            printf("\t      ===============================================\n");
                            DumpHex (pH264 -> pDcSpecInfoIsoFormat, dciSize);
                            printf("\t      ===============================================\n");
                        }
                    }
                    else
                    {
                        printf("ERROR - INCORRECT CODEC (%d) DURING H264 CODEC BLOCK - skipping ahead\n\n", pStreamVideo->codec);
                    }
                }

                //Now set pStreamVideo back to NULL at end of video codec block.  Will need to be reset
                //for next video stream (if there is another)
                pStreamVideo = NULL;

                break;

            case MP_BLK_TYPE_HINT_PROTOCOL_RTP:
                if (pStreamHint != NULL)
                {
                    if (pStreamHint->codec == MP_HINT_PROTOCOL_RTP)
                    {
                        printf("\t   ------------------------------------------------------\n");
                        printf("\t   RTP Protocol Info:\n");
                        printf("\t   ------------------------------------------------------\n");
                        pRtp = (MP_BLK_PROTOCOL_HINT_RTP *)pFileInfo;
                        printf("\t      blk size                     = %u\n", blockSize);
                        printf("\t      blk type                     = %u (%s)\n", blockType, BlockTypeStr [blockType]);
                        printf("\t      hint track version           = %u\n", pRtp -> hintTrackVersion);
                        printf("\t      hint version compatibility   = %u\n", pRtp -> hintVersionComp);
                        printf("\t      max packet size              = %lu bytes\n", pRtp -> maxPacketSize);
                        printf("\t      RTP time base (timescale)    = %lu\n", pRtp -> timeScale);
                        printf("\t      flags                        = %lu (%s)\n", pRtp -> flags, HintRTPFlagStr [pRtp -> flags]);
                        printf("\t      RTP timestamp offset         = %lu\n", pRtp -> timestampOffset);
                        printf("\t      RTP sequence offset          = %lu\n", pRtp -> sequenceOffset);
                    }
                    else
                    {
                        printf("ERROR - INCORRECT CODEC (%d) DURING RTP PROTOCOL BLOCK - skipping ahead\n\n", pStreamHint->codec);
                    }
                }

                //Now set pStreamHint back to NULL at end of RTP protocol block.  Will need to be reset
                //for next hint stream (if there is another)
                pStreamHint = NULL;

                break;

            default:
                printf("\tUnknown block type - Skipping\n\n");
                printf("\t  blk size                         = %u\n", blockSize);
                printf("\t  blk type                         = %u\n", blockType);
                break;
        }   // end of switch (blkType)

        // go to next block
        pFileInfo += blockSize;
        infoSizeLeft -= blockSize;

    }   // end of while (infoSizeLeft >0)

    if (bTrunc)
    {
        printf("\nFILE INFORMATION MAY BE TRUNCATED - INSUFFICIENT SPACE!!!\n");
    }


    return;

}   //  end of PrintStreamLevelInfo


/*!
***************************************************************************
\brief      Get file information in a multimedia file

\param[in]  fname    Filename

\return     0 on success, -1 on failure
****************************************************************************/
int CmdGetFileInfo (char *fname)
{

    unsigned long       ret;            //Variable to store the return value of a call
    MP_MMFILE           mmFile;         //MP_MMFILE structure of the 3gp file opened
    MP_FILE_INFO_DESC   fileInfoDesc;   //File Information Descriptor structure
    unsigned long       minFileInfoSize;//Minimum File Information size

    unsigned char       FileInfo [FILE_INFO_SIZE];  //Other option is to dynamically malloc memory

    //Initialize version field in MP_MMFILE structure
    mmFile.unVersion    = MP_MMFILE_VERSION;

    //Initialize necessary fields in File Info Desc structure
    fileInfoDesc.unVersion    = MP_FILE_INFO_DESC_VERSION;
    fileInfoDesc.fileInfoSize = FILE_INFO_SIZE;
    fileInfoDesc.pFileInfo    = &FileInfo [0];

    //Print argument/file information
    printf("\nGETTING FILE INFO FROM %s\n\n", fname);
    printf("File type is set as %d (%s)\n",fileType, FileTypeStr[fileType]);

    //Open the file
    if ((ret = mp_OpenFile(fname, fileType, &mmFile)) != 0)
    {
        printf("\nERROR in mp_OpenFile (%s)\n", MpErrorStr[ret]);
        return -1;
    }

    //Get file info size (this is optional)
    if ((ret = mp_GetFileInfoSize(&mmFile, &minFileInfoSize)) != 0)
    {
        printf("\nERROR in mp_GetFileInfoSize (%s)\n", MpErrorStr[ret]);
        return -1;
    }

    if (minFileInfoSize > FILE_INFO_SIZE)
    {
        printf("\nWARNING: Min file info size (%ld bytes) exceeds memory allocated (%d bytes) - file info may be truncated\n",
                 minFileInfoSize, FILE_INFO_SIZE);
    }

    //Get file info
    if ((ret = mp_GetFileInfo(&mmFile, &fileInfoDesc)) != 0)
    {
        printf("\nERROR in mp_GetFileInfo (%s)\n", MpErrorStr[ret]);
        return -1;
    }

    //Print file-level info
    PrintFileLevelInfo(&fileInfoDesc);

    //Print stream-level info
    PrintStreamLevelInfo(&fileInfoDesc);

    if ((ret = mp_CloseFile(&mmFile)) != 0)
    {
        printf("\nERROR: cannot close 3gp file: %s (%s)\n", fname, MpErrorStr[ret]);
        return -1;
    }

    return 0;

} // end of CmdGetFileInfo


/*!
***************************************************************************
\brief      Command line usage


\param      none

\return     none
****************************************************************************/
void PrintUsage (void)
{
    printf("\nUsage: mptest [command options] filename\n");
}

/*!
***************************************************************************
\brief      Help list of commands (-h)

\param      none

\return     none
****************************************************************************/
void PrintHelp(void)
{

    PrintUsage ();
    printf("where commands options are:\n\n" );

    printf("Command options:\n");
    printf("-h          Display this help screen and exit\n");
    printf("-f<value>   File type:\n");
    printf("               1  = ISO (any ISO-based file) - default\n");
    printf("               2  = 3GP\n");
    printf("               3  = MP4\n");
    printf("               4  = F4V\n");
    printf("               5  = 3G2\n");
    printf("               6  = AVC\n");
    printf("               10 = DMF (Dialogic proprietary)\n");
    printf("-s          Ignore SDP information\n");

	return;
}

/*!
***************************************************************************
\brief      Parsing of command line arguments


\param[in]  argc    as provided to main()
\param[in]  argv    as provided to main()
\param[in]  fname   Filename

\return     none
****************************************************************************/
void ParseArguments (int argc, char **argv, char *fname)
{
    int             i, c, err=0;
    unsigned int    bFnameFound = 0;
    size_t          fnameLen;

    //Scan parameters
    for (i=1; i<argc; i++)
    {
	    if (argv[i][0] != '-')
	    {
            fnameLen = strlen(&argv[i][0]);
            if (fnameLen > MAX_FILENAME_SIZE)
            {
                PrintProgramAndVersion();
                printf("\n??? Filename length %d exceeds max allowed %d\n", fnameLen, MAX_FILENAME_SIZE);
                exit(1);
            }
            bFnameFound = 1;
            //Strings for file names
            err = (sscanf(&argv[i][0], "%s", fname) != 1);
            if (err)
            {
                PrintProgramAndVersion();
                printf("Error (%d) reading filename %s\n", err, fname);
                exit(1);
            }
        }
        else
        {
            c = argv[i][1];
            switch ( c )
            {
                case 's':
                case 'S':
                    bSdp = 0;
                    break;

                case 'f':
                case 'F':
                    err = (sscanf(&argv[i][2], "%d", &fileType) != 1);
                    if (err)
                    {
                        PrintProgramAndVersion();
                        printf("Error (%d) reading file type (option -%c)\n", err, c);
                        exit(1);
                    }
                    break;

                case 'h':
                case 'H':
                case '?':
                    PrintProgramAndVersion();
                    PrintHelp();
                    exit( 1 );
                    break;
                default:
                    PrintProgramAndVersion();
                    printf("\n??? Unrecognizable option -%c\n", c);
                    PrintHelp();
                    exit( 1 );
                    break;
            }
        }
    }

    //Must have a valid filename
    if ( bFnameFound == 0)
    {
        PrintProgramAndVersion();
        printf("\n??? No file name provided\n");
        PrintHelp();
        exit( 1 );
    }

    return;
}



/*!
***************************************************************************
\brief      Main block of the application


\param[in]  argc    number of parameters in command line
\param[in]  argv    string array of command line parameters

\return 0
****************************************************************************/
int main (int argc, char *argv[])
{
    int             ret;
    char            fname[MAX_FILENAME_SIZE];

    //Scan/parse arguments
    ParseArguments (argc,argv, fname);

    PrintProgramAndVersion();

    ret = CmdGetFileInfo(fname);
    if (ret != 0)
    {
        printf("\n\n AN ERROR HAS OCCURRED - EXITING!\n\n");
    }

    return 0;

} /* end of main */


