/**
* @file closinglcs_state.cpp
* @brief Definition of ClosingLCsState class
* @date Feb. 16, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#include "closinglcs_state.h"
#include "m3glib.h"
#include "logger.h"
#include "endpoint.h"
#include "endpointmgr.h"
#include "appevents.h"



// static constant members:
const int ClosingLCsState::AUDIO_TX_LCN_CLOSED = 0x01;
const int ClosingLCsState::AUDIO_RX_LCN_CLOSED = 0x02;
const int ClosingLCsState::VIDEO_TX_LCN_CLOSED = 0x04;
const int ClosingLCsState::VIDEO_RX_LCN_CLOSED = 0x08;
const int ClosingLCsState::H245_STOPPING       = 0x10;
const int ClosingLCsState::TX_LCNS_CLOSED      = 0x05;



//*****************************************************************************
// Function: ClosingLCsState::ClosingLCsState(Endpoint *pEndpoint)
// Description: Initializing constructor
// Return:  ClosingLCsState*
// Parameters: Endpoint *pEndpoint 
//*****************************************************************************
ClosingLCsState::ClosingLCsState(Endpoint* pEndpoint) :
   EPState(pEndpoint)
{
}

//*****************************************************************************
// Function: ClosingLCsState::~ClosingLCsState()
// Description: Destructor
// Return:  none
// Parameters: none 
//*****************************************************************************
ClosingLCsState::~ClosingLCsState()
{
}

//*****************************************************************************
// Function: void ClosingLCsState::ProcessEvent(long evtType, void *pEvtData, long evtLen, long evtDev)
// Description: Process an event
// Return: none
// Parameters: long evtType 
//             void *pEvtData 
//             long evtLen 
//             long evtDev 
//*****************************************************************************
void ClosingLCsState::ProcessEvent(long  evtType, void *pEvtData, long  evtLen, long  evtDev)
{
   switch ( evtType )
   {
      case M3GEV_CLOSE_LC_CMPLT:
         {
            M3G_LOGICAL_CHANNEL_NUMBER* pLCN = reinterpret_cast<M3G_LOGICAL_CHANNEL_NUMBER*>(pEvtData);
            LOG_ENTRY(evtDev,"\t  LCN:%d\n\n", *pLCN);
            if (*pLCN == m_pEndpoint->GetAudioTxLCN())
            {
               SetStatus(AUDIO_TX_LCN_CLOSED);
               m_pEndpoint->SetAudioTxLCN(0);
            }
            else // assume video LCN
            {
               SetStatus(VIDEO_TX_LCN_CLOSED);
               m_pEndpoint->SetVideoTxLCN(0);
            }
            CheckAllLCNsClosed();
         }
         break;

      case M3GEV_CLOSE_LC_FAIL:
         LOG_ERROR(evtDev, "outbound CLC failure\n");
         //Shutdown();
         break;

      case M3GEV_REMOTE_CLOSE_LC_RCVD:
         {
            M3G_REMOTE_CLOSED_LC* pCLCInd = reinterpret_cast<M3G_REMOTE_CLOSED_LC*>(pEvtData);
            M3G_LOGICAL_CHANNEL_NUMBER lcn = pCLCInd->logicalChannelNumber;
            LOG_ENTRY(evtDev,"\t  version:%d\n", pCLCInd->version);
            LOG_ENTRY(evtDev,"\t  LCN:%d\n", lcn);
            LOG_ENTRY(evtDev,"\t  reason:%d\n", pCLCInd->reason);
            if (lcn == m_pEndpoint->GetAudioRxLCN())
            {
               SetStatus(AUDIO_RX_LCN_CLOSED);
               m_pEndpoint->SetAudioRxLCN(0);
            }
            else if (lcn == m_pEndpoint->GetAudioTxLCN())
            {
               SetStatus(AUDIO_TX_LCN_CLOSED);
               m_pEndpoint->SetAudioTxLCN(0);
            }
            else if (lcn == m_pEndpoint->GetVideoRxLCN())
            {
               SetStatus(VIDEO_RX_LCN_CLOSED);
               m_pEndpoint->SetVideoRxLCN(0);
            }
            else if (lcn == m_pEndpoint->GetVideoTxLCN())
            {
               SetStatus(VIDEO_TX_LCN_CLOSED);
               m_pEndpoint->SetVideoTxLCN(0);
            }
            break;
         }

      case M3GEV_FRAMING_LOST:
         StopH245();
         break;   // pend in this state until EndSession

      case M3GEV_ENDSESSION_RCVD:
         // pend in this state until StopH245 termination event indicates session fully reset
         StopH245();
         break;

      case M3GEV_ENDSESSION_SENT:
         // pend in this state until StopH245 termination event
         break;

      case M3GEV_STOP_H245_CMPLT:
         H245Terminated(); // H.245 session fully reset, can now transition state
         break;

      case M3GEV_STOP_H245_FAIL:
         LOG_ERROR(evtDev, "H.245 session termination failed.\n");
         //Shutdown();
         break;

      case M3GEV_H245_MISC_CMD_RCVD:
         {
            M3G_H245_MISC_CMD* pMiscCmdInd = reinterpret_cast<M3G_H245_MISC_CMD *>(pEvtData);
            LOG_ENTRY(evtDev,"\tversion: %d LCN:%d\n",
                      pMiscCmdInd->version, pMiscCmdInd->logicalChannelNumber);
            switch (pMiscCmdInd->h245MiscCmdType)
            {
               case M3G_E_FAST_UPDATE_PICTURE:
                  LOG_ENTRY(evtDev,"\t\tM3G_E_FAST_UPDATE_PICTURE (no parms)\n\n");
                  m_pEndpoint->SetLastMessageString("R[FVU]");
                  break;

               case M3G_E_FAST_UPDATE_GOB:
                  LOG_ENTRY(evtDev,"\t\tM3G_E_FAST_UPDATE_GOB: 1stGOB: %d num GOBs:%d\n",
                            pMiscCmdInd->h245MiscCmdParams.fastUpdateGOB.numFirstGOB, 
                            pMiscCmdInd->h245MiscCmdParams.fastUpdateGOB.numGOBs);
                  break;

               case M3G_E_FAST_UPDATE_MB:
                  LOG_ENTRY(evtDev,"\t\tM3G_E_FAST_UPDATE_MB: 1stGOB:%d 1stMB:%d numMBs: %d\n",
                            pMiscCmdInd->h245MiscCmdParams.fastUpdateMB.numFirstMB,
                            pMiscCmdInd->h245MiscCmdParams.fastUpdateMB.numFirstGOB,
                            pMiscCmdInd->h245MiscCmdParams.fastUpdateMB.numMBs);
                  break;

               case M3G_E_TEMP_SPAT_TRDFF:
                  LOG_ENTRY(evtDev,"\t\tM3G_E_TEMP_SPAT_TRDFF: tempSpatialTrdff:%d\n",
                            pMiscCmdInd->h245MiscCmdParams.tempSpatialTrdff);
                  break;

               case M3G_E_VIDEO_FREEZE:
                  LOG_ENTRY(evtDev,"\t\tM3G_E_VIDEO_FREEZE\n");
                  break;

               case M3G_E_SYNC_EVERY_GOB:
                  LOG_ENTRY(evtDev,"\t\tM3G_E_SYNC_EVERY_GOB\n");
                  break;

               case M3G_E_NOSYNC_EVERY_GOB:
                  LOG_ENTRY(evtDev,"\t\tM3G_E_NOSYNC_EVERY_GOB\n");
                  break;

               default:
                  LOG_ERROR(evtDev,"Unknown Miscellaneous Command Indication.\n");
            }
            break;
         }

      case M3GEV_H245_UII_RCVD:
         {
            char pLastActionBuffer[MAX_LAST_MSG_STRING] = {'\0'};
            sprintf(pLastActionBuffer,"R[");

            M3G_H245_UII* pH245UII = reinterpret_cast<M3G_H245_UII *>(pEvtData);
            LOG_ENTRY(evtDev,"\tversion: %d  numDigits:%d : \n",
                      pH245UII->version, pH245UII->numDigits);
            for (int i=0;i<pH245UII->numDigits;i++)
            {
               sprintf(pLastActionBuffer+2,"%c",pH245UII->digitBuffer[i]);
            }
            sprintf(pLastActionBuffer + int(pH245UII->numDigits) + 2,"]");
            LOG_ENTRY(evtDev,"pH245UII->digitBuffer:%s\n",pLastActionBuffer);
            m_pEndpoint->SetLastMessageString(pLastActionBuffer);
            break;
         }

      case M3GEV_SEND_H245_UII_CMPLT:
         m_pEndpoint->SetLastMessageString("S[UII]");
         break;

      case M3GEV_SEND_H245_MISC_CMD_CMPLT:
         // only support sending fastVideoUpdate in this demo:
         m_pEndpoint->SetLastMessageString("S[FVU]");
         break;

#ifdef MONA
      case M3GEV_CALL_STATISTICS:
         m_pEndpoint->RecordStatistics(evtDev, reinterpret_cast<M3G_CALL_STATISTICS*>(pEvtData));
         break;
#endif

      case M3GEV_RESET_CMPLT:
         ProcessResetComplete();
        break;

      default:
         LOG_ERROR(evtDev,"ClosingLCsState: Unexpected event type: 0x%x\n",evtType);
         break;
   }
}


//*****************************************************************************
// Function: void ClosingLCsState::ProcessUserPrompt(EPState::E_USER_PROMPT_TYPE eUserPrompt)
// Description: Process a request from the user interface
// Return: void 
// Parameters: EPState::E_USER_PROMPT_TYPE eUserPrompt 
//*****************************************************************************
void  ClosingLCsState::ProcessUserPrompt(EPState::E_USER_PROMPT_TYPE eUserPrompt)
{
     LOG_ERROR(0,"Unexpected user prompt on endpoint[%d] in CLOSING_LCS state: %d\n",
               m_pEndpoint->GetIndex(), static_cast<int>(eUserPrompt));
}


//*****************************************************************************
// Function: void ClosingLCsState::Shutdown()
// Description: Process a shutdown request
// Return: void 
// Parameters: none 
//*****************************************************************************
void ClosingLCsState::Shutdown()
{
  LOG_ENTRY(m_pEndpoint->GetControlHandle(), "Shutdown called in %s state.  Resetting M3g\n", GetStateStr());
  m_pEndpoint->Reinitialize();
  m_pEndpoint->StopNbupMedia();
}


//*****************************************************************************
// Function: void ClosingLCsState::CheckAllLCNsClosed()
// Description: Stop H245 if all LCN are closed
// Return: void 
// Parameters: none 
//*****************************************************************************
void ClosingLCsState::CheckAllLCNsClosed()
{
   if (IsStatusSet(TX_LCNS_CLOSED))
   {
      StopH245();
   }
}

//*****************************************************************************
// Function: void ClosingLCsState::StopH245()
// Description: Stop the H245 session
// Return: void 
// Parameters: none 
//*****************************************************************************
void ClosingLCsState::StopH245()
{
   if (false == IsStatusSet(H245_STOPPING))
   {
      LOG_ENTRY(m_pEndpoint->GetControlHandle(), "Calling Stop_H245 from ClosingLCSState\n");
      SetStatus(H245_STOPPING);
      m_pEndpoint->Stop_H245();
   }
}

//*****************************************************************************
// Function: void ClosingLCsState::H245Terminated()
// Description: Process an H245 session terminated event
// Return: void 
// Parameters: none 
//*****************************************************************************
void ClosingLCsState::H245Terminated()
{
  m_pEndpoint->ChangeState(H245_INACTIVE_STATE);
  LOG_ENTRY(0,"Endpoint[%d] Notify(APP_M3G_ENDPOINT_DISCONNECTED)\n",
            m_pEndpoint->GetIndex());
  m_pEndpoint->Notify(APP_M3G_ENDPOINT_DISCONNECTED);
}

//*****************************************************************************
// Function: void ClosingiLCsState::ProcessResetComplete()
// Description: M3G reset done; set state to inactive to trigger shutdown
// Return: void
// Parameters: none
//*****************************************************************************
void ClosingLCsState::ProcessResetComplete()
{
      LOG_ENTRY(0,"Endpoint[%d] M3g reset complete.  Triggering shutdown.\n",
                m_pEndpoint->GetIndex());
      LOG_ENTRY(0,"Endpoint[%d] Notify(APP_M3G_ENDPOINT_DISCONNECTED)\n",
                m_pEndpoint->GetIndex());
      m_pEndpoint->Notify(APP_M3G_ENDPOINT_DISCONNECTED);
      m_pEndpoint->ChangeState(H245_INACTIVE_STATE); 
}

