/**
* @file logger.cpp
* @brief Definition of Logger class
* @date Feb. 16, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#include "logger.h"
#include <stdarg.h> 
#include <sys/timeb.h>
#include "srllib.h"
#include <stdio.h>
#include <iostream>


//*****************************************************************************
// Function: Logger::Logger(const char *logFileName, bool &isError)
// Description: Initializing constructor
// Return:  Logger*
// Parameters: const char *logFileName 
//             bool &isError 
//*****************************************************************************
Logger::Logger(const char *logFileName, bool& isError) :
   m_FileHandle(NULL),
   m_IsEnabled(false)
{
   // if logging disabled, no-op
   if ( NULL == logFileName )
   {
      isError = false;
      return;
   }

   m_FileHandle = fopen(logFileName,"w");
   if ( NULL == m_FileHandle )
   {
     printf("ERROR: Cannot open log file %s\n", logFileName);
     isError = true;
   }
   else
   {
      m_IsEnabled = true;
      isError = false;
   }
}


//*****************************************************************************
// Function: Logger::~Logger()
// Description: Destructor
// Return:  none
// Parameters: none 
//*****************************************************************************
Logger::~Logger()
{
   if ( NULL != m_FileHandle )
   {
      fclose(m_FileHandle);
   }
}

//*****************************************************************************
// Function: void Logger::LogEntry(SRL_DEVICE_HANDLE dev, const char *fmt, ...)
// Description: Log a message
// Return: void 
// Parameters: SRL_DEVICE_HANDLE dev 
//             char *fmt 
//             ... 
//*****************************************************************************
void Logger::LogEntry(SRL_DEVICE_HANDLE dev, const char *fmt, ...)
{
   char    logLineEntry[2048];
   struct  timeb timeBuffer;
   char   *timeline;

   // if logging disabled, no-op
   if ( false == m_IsEnabled )
      return;

   int     lineOffset = 0;
   va_list args;

   ftime(&timeBuffer);
   timeline = ctime(&timeBuffer.time);

   const char   *pDevName = (0 < dev) ? ATDV_NAMEP(dev) : "";

   lineOffset = sprintf(logLineEntry, "[%.15s.%03hu] %.20s ", 
                        &timeline[4], timeBuffer.millitm, pDevName);

   va_start(args, fmt);
   lineOffset += vsprintf(logLineEntry + lineOffset, fmt, args);
   va_end(args);

   fprintf(m_FileHandle, logLineEntry);
   fflush(m_FileHandle);
}

//*****************************************************************************
// Function: void Logger::LogError(SRL_DEVICE_HANDLE dev, const char *errStr, ...)
// Description: Log an error
// Return: void 
// Parameters: SRL_DEVICE_HANDLE dev 
//             char *errStr 
//             ... 
//*****************************************************************************
void Logger::LogError(SRL_DEVICE_HANDLE dev, const char *errStr, ...)
{
   char    logLineEntry[256];
   struct  timeb timeBuffer;
   char   *timeline;
   int     lineOffset = 0;
   va_list args;

   // if logging disabled, no-op
   if ( false == m_IsEnabled )
      return;

   ftime(&timeBuffer);
   timeline = ctime(&timeBuffer.time);

   const char   *pDevName = (0 < dev) ? ATDV_NAMEP(dev) : "";

   if ( 0 < dev )
   {
      // If nothing interesting returned with ATDV_LASTERR(), don't display confusing "NO ERROR" 
      int lastErr = ATDV_LASTERR(dev);
      if (lastErr == 0)  {
        lineOffset = sprintf(logLineEntry, "[%.15s.%03hu] %.20s ERROR:", &timeline[4],
                           timeBuffer.millitm, pDevName);
      }
      else  {
        lineOffset = sprintf(logLineEntry, "[%.15s.%03hu] %.20s %ld:%.20s   ERROR:", &timeline[4],
                             timeBuffer.millitm, pDevName, (long)lastErr, ATDV_ERRMSGP(dev));
      }
   }
   else
   {
      lineOffset = sprintf(logLineEntry, "[%.15s.%03hu]   ERROR:", &timeline[4],timeBuffer.millitm);
   }

   va_start(args, errStr);
   lineOffset += vsprintf(logLineEntry + lineOffset, errStr, args);
   va_end(args);

   fprintf(m_FileHandle, logLineEntry);
   fflush(m_FileHandle);

}
