/**
* @file mmidle_state.cpp
* @brief Definition of MMIdleState class
* @date Feb. 20, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#include "mmlib.h"
#include "mmidle_state.h"
#include "mmendpoint.h"
#include "endpointmgr.h"
#include "logger.h"

#include "appevents.h"

//*****************************************************************************
// Function: MMIdleState::MMIdleState(MMEndpoint *pMMEndpoint)
// Description: Initializing constructor
// Return:  MMIdleState*
// Parameters: MMEndpoint *pMMEndpoint 
//*****************************************************************************
MMIdleState::MMIdleState(MMEndpoint* pMMEndpoint) :
   MMEPState(pMMEndpoint)
{
   m_PortDiscCount = 0;
}

//*****************************************************************************
// Function: MMIdleState::~MMIdleState()
// Description: Destructor
// Return:  none
// Parameters: none 
//*****************************************************************************
MMIdleState::~MMIdleState()
{
}

//*****************************************************************************
// Function: void MMIdleState::ProcessEvent(METAEVENT metaevent)
// Description: Process an event
// Return: none 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void MMIdleState::ProcessEvent(METAEVENT metaevent)
{
   int retCode;
   MM_INFO mmInfo;
   INIT_MM_INFO(&mmInfo);
   
   switch ( metaevent.evttype )
   {
    case MMEV_PLAY_ACK:
      LOG_ENTRY(metaevent.evtdev, "MMIdle state: Event MMEV_PLAY_ACK received\n");
      ProcessPlayStarted(metaevent);
        break;
    case MMEV_RECORD_ACK:
      LOG_ENTRY(metaevent.evtdev, "MMIdle state: Event MMEV_RECORD_ACK received\n");
      ProcessRecordStarted(metaevent);
        break;
    case MMEV_PLAY_ACK_FAIL:
      LOG_ENTRY(metaevent.evtdev, "MMIdle state: Event MMEV_PLAY_ACK_FAIL received\n");
      retCode = mm_ErrorInfo(&mmInfo);
      LOG_ENTRY(0, "  mm_value=%#x,mmMsg=%s,addInfo=%s \n",mmInfo.mmValue,mmInfo.mmMsg,mmInfo.additionalInfo);
      break; 
    case MMEV_RECORD_ACK_FAIL:
      LOG_ENTRY(metaevent.evtdev, "MMIdle state: Event MMEV_PLAY_ACK_FAIL received\n");
        break; 
    case DMEV_PORT_DISCONNECT:
	LOG_ENTRY(metaevent.evtdev, "MMIdle state: DMEV_PORT_DISCONNECT received\n");
	ProcessPortDisconnect(metaevent.evtdev);  
	break;
    case M3GEV_H245_UII_RCVD:
      LOG_ENTRY(metaevent.evtdev, "MMIdle state: Event M3GEV_H245_UII_RCVD received\n");
      // This event is a little odd, as it has arrived at this MM endpoints 3G peer.  But
      // might as well use it as any other event value for UII/DTMF notification.
      ProcessUIIRcvd(metaevent);
        break;
    default:
      LOG_ERROR(metaevent.evtdev,"MMIdle state: Unexpected event type: 0x%x\n",metaevent.evttype);
       break;
   }
}
//*****************************************************************************
// Function: void MMIdleState::ProcessUIIRcvd()
// Description: Process UII Recvd from 3G peer
// Return: void 
// Parameters: none 
//*****************************************************************************
void MMIdleState::ProcessUIIRcvd(METAEVENT metaevent)
{
  char *UIIPtr = (char *) metaevent.evtdatap;

  // Simple here - any UII/DTMF triggers next action.
  m_pMMEndpoint->SetLastDigitRcvd(UIIPtr[0]);
  LOG_ENTRY(metaevent.evtdev, "UII %c received\n",*UIIPtr);
  
  if (!strcmp(EndpointMngr::Instance()->GetMMOperation(m_pMMEndpoint->GetLastDigitRcvd()), "NO OP"))  {
     // No operation exists for this dtmf
     m_pMMEndpoint->SetLastMessageString("No MM Operation for dtmf");
  }
  else {   
     // Send M3G Connected to notify this endpoint's BridgedMedia of operation
     m_pMMEndpoint->Notify(APP_M3G_ENDPOINT_CONNECTED);
  }
}

//*****************************************************************************
// Function: void MMIdleState::ProcessPlayStarted()
// Description: Process MMEV_PLAY_ACK
// Return: void 
// Parameters: none 
//*****************************************************************************
void MMIdleState::ProcessPlayStarted(METAEVENT metaevent)
{
  m_pMMEndpoint->Notify(APP_MM_ENDPOINT_PLAYING);
  m_pMMEndpoint->ChangeState(PLAYING_STATE);
}

//*****************************************************************************
// Function: void MMIdleState::ProcessRecordStarted()
// Description: Process MMEV_RECORD_ACK
// Return: void 
// Parameters: none 
//*****************************************************************************
void MMIdleState::ProcessRecordStarted(METAEVENT metaevent)
{
  m_pMMEndpoint->Notify(APP_MM_ENDPOINT_RECORDING);
  m_pMMEndpoint->ChangeState(RECORDING_STATE);
}

//*****************************************************************************
// Function: void MMIdleState::Shutdown()
// Description: Process shutdown request
// Return: void 
// Parameters: none 
//*****************************************************************************
void MMIdleState::Shutdown()
{
  LOG_ENTRY(m_pMMEndpoint->GetMMHandle(), "Shutdown called in %s state.  No Op\n", GetStateStr());
}

void MMIdleState::ProcessPortDisconnect(int evtDev)
{
      m_PortDiscCount++;
      if (m_PortDiscCount == 2) {
	 LOG_ENTRY(evtDev," Audio and Video Ports Disconnected\n");
	 m_pMMEndpoint->Notify(APP_MM_ENDPOINT_DISCONNECTED);
	 m_pMMEndpoint->CloseSubDevs();
      }
}
