/**
* @file sipconnected_state.cpp
* @brief Definition of SIPConnectedState class
* @date Feb. 16, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#include "ipmlib.h"
#include "sipconnected_state.h"
#include "sipendpoint.h"
#include "endpointmgr.h"
#include "logger.h"
#include "appevents.h"

//*****************************************************************************
// Function: SIPConnectedState::SIPConnectedState(SIPEndpoint *pSIPEndpoint)
// Description: Initializing constructor
// Return:  SIPConnectedState*
// Parameters: SIPEndpoint *pSIPEndpoint 
//*****************************************************************************
SIPConnectedState::SIPConnectedState(SIPEndpoint* pSIPEndpoint):
   SIPEPState(pSIPEndpoint)
{
   m_pSIPEndpoint = pSIPEndpoint;
}

//*****************************************************************************
// Function: SIPConnectedState::~SIPConnectedState()
// Description: Destructor
// Return:  none
// Parameters: none 
//*****************************************************************************
SIPConnectedState::~SIPConnectedState()
{
}

//*****************************************************************************
// Function: void SIPConnectedState::ProcessEvent(METAEVENT metaevent)
// Description: Process an event
// Return: none 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void SIPConnectedState::ProcessEvent(METAEVENT metaevent)
{
   switch ( metaevent.evttype )
   {
      // GC/R4 events
      case GCEV_DISCONNECTED:
         LOG_ENTRY(metaevent.evtdev, "SIPConnectedState: GCEV_DISCONNECTED received\n");
         ProcessDisconnect(metaevent);
         break;
      case IPMEV_STARTMEDIA:
         LOG_ENTRY(metaevent.evtdev, "SIPConnectedState: IPMEV_STARTMEDIA received\n");
         // Media is moving...
         break;
      case GCEV_CALLINFO:
         LOG_ENTRY(metaevent.evtdev,"SIPConnectedState: GCEV_CALLINFO received\n");
         ProcessIFrameUpdate(metaevent);
         break;
      case GCEV_EXTENSIONCMPLT:
         LOG_ENTRY(metaevent.evtdev,"SIPConnectedState: GCEV_EXTENSIONCMPLT received - SIP OK successfully sent\n");
         break;
      case GCEV_EXTENSION:
         LOG_ENTRY(metaevent.evtdev,"SIPConnectedState: GCEV_EXTENSION received \n");
         ProcessExtensionEvent(metaevent);
         break;
      default:
         LOG_ERROR(metaevent.evtdev, "SIPConnectedState: Unexpected event type: 0x%x\n", 
                   metaevent.evttype);
         break;
   }
}

//*****************************************************************************
// Function: void SIPConnectedState::ProcessDisconnect(METAEVENT metaevent)
// Description: Process disconnect event
// Return: void 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void SIPConnectedState::ProcessDisconnect(METAEVENT metaevent)
{ 
   // Advance to dropping state
   m_pSIPEndpoint->ChangeState(DROPPING_STATE);
   m_pSIPEndpoint->Notify(APP_SIP_ENDPOINT_DISCONNECT);
   m_pSIPEndpoint->DropCall(metaevent);
}

//*****************************************************************************
// Function: void SIPConnectedState::ProcessIFrameUpdate(METAEVENT metaevent)
// Description: Process IFrame request
// Return: void 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void SIPConnectedState::ProcessIFrameUpdate(METAEVENT metaevent)
{
   // Have a likely SIP info message that contains an IFrame update.
   // Verify that, and respond with an OK.
   // Then send out an H245 video fast update.

   if ( m_pSIPEndpoint->IsSIPIFrameUpdate(metaevent) )
   {
      // Got in an IFrame update request via SIP, so send one out
      // on 3G
      LOG_ENTRY(metaevent.evtdev, "SIP IFrame request received, sending out SIP OK and 3G fast frame request\n");
      m_pSIPEndpoint->SendSIPOK();

      BridgedCall* myBridgedCall = m_pSIPEndpoint->GetBridgedCall();
      Endpoint* pEndpoint = EndpointMngr::Instance()->GetEPFromIdx(myBridgedCall->M3gIndex());
      pEndpoint->SendFastVideoUpdate();
   }
   else
   {
      LOG_ENTRY(metaevent.evtdev, "SIP INFO message received, but its not an IFRame request\n");

#if 1 // Send a FVU on any SIP INFO message     
      LOG_ENTRY(metaevent.evtdev, "Sending FastVideoUpdate anyway...\n");
      m_pSIPEndpoint->SendSIPOK();

      BridgedCall* myBridgedCall = m_pSIPEndpoint->GetBridgedCall();
      Endpoint* pEndpoint = EndpointMngr::Instance()->GetEPFromIdx(myBridgedCall->M3gIndex());
      pEndpoint->SendFastVideoUpdate();
#endif 
   }
   // No state change needed here...
}

//*****************************************************************************
// Function: void SIPConnectedState::Shutdown()
// Description: Process shutdown request
// Return: void 
// Parameters: none 
//*****************************************************************************
void SIPConnectedState::Shutdown()
{
  LOG_ENTRY(m_pSIPEndpoint->GetGCSIPHandle(), "Shutdown called in %s state.  No Op\n", GetStateStr());
}


void SIPConnectedState::ProcessExtensionEvent(METAEVENT metaevent)
{
    // Local Variable Declaration
    GC_PARM_BLKP        gcParmBlk;
    GC_PARM_DATA        *pgcParmData    = 0;
    EXTENSIONEVTBLK     *pExtensionBlk = 0; 

    // Retrieve the GC parm block information.
    // NOTE: We cannot use pData here as it is the evtdatp member of METAEVENT, we need
    // the extevtdatap member.
    pExtensionBlk = (EXTENSIONEVTBLK *)(metaevent.extevtdatap); 
    gcParmBlk = (&(pExtensionBlk->parmblk));

    #define GCNAME_MAXSIZE 128
    char cPhoneList[GCNAME_MAXSIZE+1];
    int i=0;
    while ((pgcParmData = gc_util_next_parm(gcParmBlk, pgcParmData)) != 0) {
        switch (pExtensionBlk->ext_id) {
        case IPEXTID_RECEIVEMSG:
            {
                LOG_ENTRY(metaevent.evtdev, "OnExtensionEvent() Received IPEXTID_RECEIVEMSG--->\n");
                switch (pgcParmData->set_ID) {
                case IPSET_MSG_Q931:
                    {
                        LOG_ENTRY(metaevent.evtdev, "OnExtensionEvent() Received IPSET_MSG_Q931--->");
                        switch (pgcParmData->value_buf[0]) {
                        case IP_MSGTYPE_Q931_PROGRESS:
                            LOG_ENTRY(metaevent.evtdev, "OnExtensionEvent() Received IP_MSGTYPE_Q931_PROGRESS--->\n");
                            break;

                        default:
                            LOG_ENTRY(metaevent.evtdev, "OnExtensionEvent() Received PARM=0x%X--->\n", pgcParmData->parm_ID);
                            break;
                        }
                    }
                    break;
                case IPSET_CALLINFO:
                    LOG_ENTRY(metaevent.evtdev, "OnExtensionEvent() Received IPSET_CALLINFO--->\n");
                    break;

                default:
                    LOG_ENTRY(metaevent.evtdev, "OnExtensionEvent() Received 0x%X--->\n", pgcParmData->set_ID);
                    break;
                }
            }
            break;
        case IPEXTID_GETINFO:
            {
                LOG_ENTRY(metaevent.evtdev, "OnExtensionEvent() Received IPEXTID_GETINFO--->\n");
                switch (pgcParmData->set_ID) {
                case IPSET_CALLINFO:
                    switch (pgcParmData->parm_ID) {
                    case IPPARM_CALLID:
                        LOG_ENTRY(metaevent.evtdev, "OnExtensionEvent() for CallId");
                        if (pgcParmData->value_size != 0) {
                            for (i=0; i<16; i++) {
                                printf("%02X", pgcParmData->value_buf[i]);
                            }
                        }
                        printf("\n");
                        break;
                    case IPPARM_CONNECTIONMETHOD:
                        break;
                    case IPPARM_H245TUNNELING:
                        break;
                    case IPPARM_DISPLAY:
                        break;
                    case IPPARM_USERUSER_INFO:
                        break;
                    case IPPARM_PHONELIST:
                        {
                            if (pgcParmData->value_size != 0) {
                                strncpy(cPhoneList, (const char *)pgcParmData->value_buf, GCNAME_MAXSIZE-1);
                                cPhoneList[GCNAME_MAXSIZE+1] = '\0';
                            }
                            LOG_ENTRY(metaevent.evtdev, "OnExtensionEvent() PhoneList/DNIS \n");
                        }
                        break;
                    case IPPARM_CALLDURATION:
                        break;
                    case IPPARM_RTCPINFO:
                        break;
                    }
                    break;
                }
            }
        }
    } // end of switch for ext_id
}
