/**
* @file sipendpoint.cpp
* @brief Definition of SIPEndpoint class
* @date Feb. 16, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#include <time.h>
#include "ipmlib.h"
#include "gcip.h"
#include "logger.h"
#include "sipepstate.h"
#include "sipendpoint.h"
#include "endpointmgr.h"
#include "locker.h"
#include "appevents.h"
#include "endpoint.h"

//  For SIP INFO messages
const char *c_iFrameRequest = "<?xml version=\"1.0\" encoding=\"utf-8\" ?><media_control><vc_primitive><to_encoder><picture_fast_update></picture_fast_update></to_encoder></vc_primitive></media_control>";

//*****************************************************************************
// Function: SIPEndpoint::SIPEndpoint(int numEP, const char *SIPName, const char *IPMName, const char *mapID)
// Description: Initializing constructor
// Return:  SIPEndpoint*
// Parameters: int numEP 
//             const char *SIPName 
//             const char *IPMName 
//             const char *mapID 
//*****************************************************************************
SIPEndpoint::SIPEndpoint(int numEP, const char *SIPName, const char *IPMName, const char *mapID):
   m_Index(numEP),
   m_pCurrentState(NULL),
   m_GCSIPHandle(-1),
   m_IPMHandle(-1),
   // Dynamic Payload type definitions
   m_nAMR_PT(96),
   m_nH263_PT(34),
   m_nMPEG4_PT(102),
   m_nH264_PT(103),
   m_nRFC2833_PT(101)
{
   SetEndpointType(SIP_ENDPOINT);
   sprintf(m_SIPName, ":N_%s:P_SIP", SIPName);
   strcpy(m_IPMName, IPMName);
   strcpy(m_mapID, mapID);
   strncpy(m_CurrentStateStr, "OPENING        ", MAX_CURRENT_STATE_STR);
   strcpy(m_LastMessageBuff, "");
   m_pSIPOpeningState = new SIPOpeningState(this);
   m_pSIPReservingMediaState = new SIPReservingMediaState(this);
   m_pSIPPortConnectingState = new SIPPortConnectingState(this);
   m_pSIPIdleState = new SIPIdleState(this);
   m_pSIPCallingState = new SIPCallingState(this);
   m_pSIPConnectedState = new SIPConnectedState(this);
   m_pSIPDroppingState = new SIPDroppingState(this);
   m_pSIPReleasingState = new SIPReleasingState(this);
   m_pSIPOfferedState = new SIPOfferedState(this);
   m_pSIPClosedState = new SIPClosedState(this);
}

//*****************************************************************************
// Function: SIPEndpoint::~SIPEndpoint()
// Description: Destructor
// Return:  none
// Parameters: none 
//*****************************************************************************
SIPEndpoint::~SIPEndpoint()
{
   LOG_ENTRY(m_GCSIPHandle, "SIP Destructor called\n");
   delete m_pSIPOpeningState;
   delete m_pSIPReservingMediaState;
   delete m_pSIPPortConnectingState;
   delete m_pSIPIdleState;
   delete m_pSIPCallingState;
   delete m_pSIPConnectedState;
   delete m_pSIPDroppingState;
   delete m_pSIPReleasingState;
   delete m_pSIPOfferedState;
   delete m_pSIPClosedState;
}

//*****************************************************************************
// Function: void SIPEndpoint::ChangeState(SIPEPState::E_SIPEPSTATE e_NewState)
// Description: Change the object state
// Return: void 
// Parameters: SIPEPState::E_SIPEPSTATE e_NewState 
//*****************************************************************************
void SIPEndpoint::ChangeState(SIPEPState::E_SIPEPSTATE e_NewState)
{
   char oldStateStr[MAX_CURRENT_STATE_STR];
   strcpy(oldStateStr, m_pCurrentState->GetStateStr());

   // reset the current state for the next call:
   m_pCurrentState->Reset();

   switch ( e_NewState )
   {
      case SIPEPState::RESERVINGMEDIA_STATE:
         m_pCurrentState = m_pSIPReservingMediaState;
         break;

      case SIPEPState::PORTCONNECTING_STATE:
         m_pCurrentState = m_pSIPPortConnectingState;
         break;

      case SIPEPState::IDLE_STATE:
         m_pCurrentState = m_pSIPIdleState;
         break;

      case SIPEPState::CALLING_STATE:
         m_pCurrentState = m_pSIPCallingState;
         break;

      case SIPEPState::CONNECTED_STATE:
         m_pCurrentState = m_pSIPConnectedState;
         break;

      case SIPEPState::DROPPING_STATE:
         m_pCurrentState = m_pSIPDroppingState;
         break;

      case SIPEPState::RELEASING_STATE:
         m_pCurrentState = m_pSIPReleasingState;
         break;

      case SIPEPState::OFFERED_STATE:
         m_pCurrentState = m_pSIPOfferedState;
         break;

      case SIPEPState::CLOSED_STATE:
         m_pCurrentState = m_pSIPClosedState;
         break;

      default:
         LOG_ERROR(0,"Unexpected state: %d\n",static_cast<int>(e_NewState));
         break;
   }

   // cache the state string in the endpoint
   strcpy(m_CurrentStateStr,m_pCurrentState->GetStateStr());
   LOG_ENTRY(m_GCSIPHandle, "SIPEndpoint[%d] State transition: %s ---> %s\n",
		GetIndex(), oldStateStr, m_CurrentStateStr);
}

//*****************************************************************************
// Function: void SIPEndpoint::ProcessEvent(METAEVENT metaevent)
// Description: Process an event
// Return: none
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void SIPEndpoint::ProcessEvent(METAEVENT metaevent)
{
   m_StateLock.Lock();
   // dispatch event information to appropriate state:
   m_pCurrentState->ProcessEvent(metaevent);
   m_StateLock.Unlock();
}

//*****************************************************************************
// Function: SIPEPState::E_SIPEPSTATE SIPEndpoint::GetState()
// Description: Return the state
// Return: SIPEPState::E_SIPEPSTATE 
// Parameters: none 
//*****************************************************************************
SIPEPState::E_SIPEPSTATE SIPEndpoint::GetState()
{
   m_StateLock.Lock();
   SIPEPState::E_SIPEPSTATE state = m_pCurrentState->GetState();
   m_StateLock.Unlock();
   return state;
}

//*****************************************************************************
// Function: void SIPEndpoint::OpenSubDevs()
// Description: Open all associated devices
// Return: void 
// Parameters: none 
//*****************************************************************************
void SIPEndpoint::OpenSubDevs()
{
   // initialize the endpoint state to opening
   m_pCurrentState = m_pSIPOpeningState;

   // Open the GC SIP device
   LOG_ENTRY(0, "Opening %s\n", m_SIPName);

   if ( gc_OpenEx(&m_GCSIPHandle, (char*)m_SIPName, EV_ASYNC, (void *)this) != GC_SUCCESS )
   {
      LOG_ERROR(0, "gc_OpenEx(%s) failure\n",m_SIPName);
      LogGlobalCallError();
   }
   else
   {
      LOG_ENTRY(0, "%s successfully opened\n", m_SIPName);
   }

   // Now the IPM device
   LOG_ENTRY(0, "Opening %s\n", m_IPMName);
   if ( (m_IPMHandle = ipm_Open(m_IPMName, NULL, EV_ASYNC)) == -1 )
   {
      LOG_ERROR(0, "ipm_Open(%s) failure\n",m_IPMName);
   }
   else
   {
      LOG_ENTRY(m_IPMHandle, "%s successfully opened\n", m_IPMName);
   }
}

//*****************************************************************************
// Function: void SIPEndpoint::ReserveResources()
// Description: Reserve media resources
// Return: void 
// Parameters: none 
//*****************************************************************************
void SIPEndpoint::ReserveResources()
{
   // Reserve codecs needed for IPM device
   LOG_ENTRY(m_IPMHandle, "Reserving codec resources\n");

   // bug - Set Dtmf mode before Reserve Resource
   SetDtmfMode(RFC2833);

   // using NATIVE mode so don't reserve any resources
   // just send event for the state machine

   // However, keep this here in case non-native is used
   // again.
   
   METAEVENT tmpMetaevent;
   tmpMetaevent.evttype = DMEV_RESERVE_RESOURCE;
   tmpMetaevent.evtdev = GetGCSIPHandle();
   tmpMetaevent.evtdatap = NULL;
   tmpMetaevent.evtlen = 0;
   ProcessEvent(tmpMetaevent);
}

//*****************************************************************************
// Function: void SIPEndpoint::ReleaseResources()
// Description: Release media resources
// Return: void 
// Parameters: none 
//*****************************************************************************
void SIPEndpoint::ReleaseResources()
{
//   Since a fake reserve is done (above) we don't want to do this.
//   It causes a crash.
// 
//   Leave this here in case non-native mode is re-instated and we do the
//   reserve/release again.

//   if ( dev_ReleaseResourceEx(m_IPMHandle, &m_resourceList, EV_ASYNC) < 0 )
//   {
//      LOG_ERROR(m_IPMHandle, "ReleaseResourceEx() failed\n");
//      LogDevError();
//   }
//   else
//   {
//      LOG_ENTRY(m_IPMHandle, "ReleaseResourceEx() called successfully\n", m_IPMName);
//   }
}

//*****************************************************************************
// Function: void SIPEndpoint::GetIPMPortInfo()
// Description: Get RTP port information
// Return: void 
// Parameters: none 
//*****************************************************************************
void SIPEndpoint::GetIPMPortInfo ()
{
   // Info arrives with successful completion event on async call
   if ( dev_GetTransmitPortInfo(m_IPMHandle, this) != DEV_SUCCESS )
   {
      LOG_ERROR(m_IPMHandle,"dev_GetTransmitPortInfo() failure\n");
   }
   else
   {
      LOG_ENTRY(m_IPMHandle,"Successful dev_GetTransmitPortInfo() call\n");
   }

   if ( dev_GetReceivePortInfo(m_IPMHandle, this) != DEV_SUCCESS )
   {
      LOG_ERROR(m_IPMHandle,"dev_GetTransmitPortInfo() failure\n");
   }
   else
   {
      LOG_ENTRY(m_IPMHandle,"Successful dev_GetReceivePortInfo() call\n");
   }
}

//*****************************************************************************
// Function: void SIPEndpoint::RecordPortInfoList(METAEVENT metaevent, DM_PORT_INFO_LIST *pPortInfoList)
// Description: Save the RTP port information
// Return: void 
// Parameters: METAEVENT metaevent 
//             DM_PORT_INFO_LIST *pPortInfoList 
//*****************************************************************************
void SIPEndpoint::RecordPortInfoList(METAEVENT metaevent, DM_PORT_INFO_LIST* pPortInfoList)
{
   LOG_ENTRY(metaevent.evtdev,"SIPEndpoint::RecordPortInfoList: DM_PORT_INFO_LIST:\n");
   LOG_ENTRY(metaevent.evtdev,"\tunVersion:%d\n", pPortInfoList->unVersion);
   LOG_ENTRY(metaevent.evtdev,"\tunCount:%d\n", pPortInfoList->unCount);
   for ( unsigned int i=0; i< pPortInfoList->unCount; i++ )
   {
      LOG_ENTRY(metaevent.evtdev,"\tport_info[%d]:\n",i);
      LOG_ENTRY(metaevent.evtdev,"\t\tunVersion: %d\n",pPortInfoList->port_info[i].unVersion);

      char tmpBuff[128] = {0};
      char tokenBuff[10] = {0};
      for ( int j=0; j<16; j++ )
      {
         sprintf(tokenBuff, "0x%x ", int(pPortInfoList->port_info[i].device_ID[j]));
         strcat(tmpBuff, tokenBuff);
      }
      LOG_ENTRY(metaevent.evtdev,"\t\tdevice_ID: %s\n", tmpBuff);
      LOG_ENTRY(metaevent.evtdev,"\t\tport_ID: %s\n",reinterpret_cast<char *>(pPortInfoList->port_info[i].port_ID));
      LOG_ENTRY(metaevent.evtdev,"\t\tport_media_type: %d\n", pPortInfoList->port_info[i].port_media_type);

      switch ( pPortInfoList->port_info[i].port_media_type )
      {
         case DM_PORT_MEDIA_TYPE_AUDIO:
            if ( metaevent.evttype == DMEV_GET_TX_PORT_INFO )
            {
               LOG_ENTRY(metaevent.evtdev,"\t\tSaving Tx audio port info\n");
               memcpy(&m_audioPortTxInfo, &pPortInfoList->port_info[i], sizeof(DM_PORT_INFO));
            }
            else
            {
               LOG_ENTRY(metaevent.evtdev,"\t\tSaving Rx audio port info\n");
               memcpy(&m_audioPortRxInfo, &pPortInfoList->port_info[i], sizeof(DM_PORT_INFO));
            }
            break;
         case DM_PORT_MEDIA_TYPE_VIDEO:
            if ( metaevent.evttype == DMEV_GET_TX_PORT_INFO )
            {
               LOG_ENTRY(metaevent.evtdev,"\t\tSaving Tx video port info\n");
               memcpy(&m_videoPortTxInfo, &pPortInfoList->port_info[i], sizeof(DM_PORT_INFO));
            }
            else
            {
               LOG_ENTRY(metaevent.evtdev,"\t\tSaving Rx video port info\n");
               memcpy(&m_videoPortRxInfo, &pPortInfoList->port_info[i], sizeof(DM_PORT_INFO));
            }
            break;
         case DM_PORT_MEDIA_TYPE_AUDIO_LINEAR:
            LOG_ENTRY(metaevent.evtdev,"\t\tDetected (but not saving) audio linear port info\n");
            break;
         case DM_PORT_MEDIA_TYPE_NBUP:
            LOG_ENTRY(metaevent.evtdev,"\t\tDetected (but not saving) H223(NBUP) port info\n");
            break;
         case DM_PORT_MEDIA_TYPE_NONE:
            LOG_ENTRY(metaevent.evtdev,"\t\tDetected (but not saving) MEDIA_TYPE_NONE port info\n");
            break;
         default:
            LOG_ENTRY(metaevent.evtdev,"\t\tUnknown DM_PORT_MEDIA_TYPE port info\n");
            break;
      }
   }
}


//*****************************************************************************
// Function: void SIPEndpoint::ConnectToPeer(DM_PORT_INFO &a_audioPortRxInfo, DM_PORT_INFO &a_videoPortRxInfo)
// Description: Connect media streams to peer endpoint
// Return: void 
// Parameters: DM_PORT_INFO &a_audioPortRxInfo 
//             DM_PORT_INFO &a_videoPortRxInfo 
//*****************************************************************************
void SIPEndpoint::ConnectToPeer(DM_PORT_INFO& a_audioPortRxInfo, DM_PORT_INFO& a_videoPortRxInfo)
{
   // Connect audio and video RTP packet streams to 3G device's corresponding streams

   /************************************************************************

    Looking for this:

                    IPM                 3G

    Audio
          Tx --------------->  >----------------- Rx
          Rx ---------------<  <----------------- Tx

    Video
          Tx --------------->  >----------------- Rx
          Rx ---------------<  <----------------- Tx

   ********************************************************************/        
   DM_PORT_CONNECT_INFO_LIST portConnectInfoList;

   // Have TX and RX port info lists (one each) from IPM
   // Each list has an audio, video and linear audio node
   // 
   // Have TX and RX port info lists from 3G.  Audio device and video device both have
   // a TX and RX list.  Should only be a video PORT_INFO in the video device's, and
   // an audio PORT_INFO in the audio device's.
   // 

   // Important info here!!!!!
   ///////////////////////////
   // "Tx" device is the device whose handle is used in the dev_PortConnect call.
   // port_info_tx gets the transmit port of the device whose handle is used in the dev_PortConnect call.
   // port_info_rx gets the receive port of the other device.
   // BUT - port_info_rx must contain the device_ID of the device whose handle is used in the dev_PortConnect
   // call, or a "multiple device" error will be found in the RTF log

   // This call connects the Tx of IPMdevice to Rx of 3G device (Audio)
   EndpointMngr::Instance()->InitPortConnectInfoList(&portConnectInfoList, EndpointMngr::AUDIO_PORT, GetAudTranscodeEnabled());
   memcpy(&(portConnectInfoList.port_connect_info[0].port_info_tx),
          &m_audioPortTxInfo,
          sizeof(DM_PORT_INFO));
   memcpy(&(portConnectInfoList.port_connect_info[0].port_info_rx),
          &a_audioPortRxInfo,
          sizeof(DM_PORT_INFO));

   if ( DEV_SUCCESS != dev_PortConnect(m_IPMHandle, &portConnectInfoList, this) )
   {
      LOG_ERROR(m_IPMHandle,
                "dev_PortConnect() failure in SIP endpoint - %s\n",
                ATDV_ERRMSGP(m_IPMHandle));
   }
   else
   {
      LOG_ENTRY(m_IPMHandle,"Successful dev_PortConnect() call in SIP endpoint\n");
   }

   // This call connects the Tx of IPMdevice to Rx of 3G device (Video)
   EndpointMngr::Instance()->InitPortConnectInfoList(&portConnectInfoList, EndpointMngr::VIDEO_PORT, GetVidTranscodeEnabled());
   memcpy(&(portConnectInfoList.port_connect_info[0].port_info_tx), 
          &m_videoPortTxInfo,
          sizeof(DM_PORT_INFO));
   memcpy(&(portConnectInfoList.port_connect_info[0].port_info_rx), 
          &a_videoPortRxInfo,
          sizeof(DM_PORT_INFO));

   if ( DEV_SUCCESS != dev_PortConnect(m_IPMHandle, &portConnectInfoList, this) )
   {
      LOG_ERROR(m_IPMHandle, 
                "dev_PortConnect() failure in SIP endpoint - %s\n",
                ATDV_ERRMSGP(m_IPMHandle));
   }
   else
   {
      LOG_ENTRY(m_IPMHandle,"Successful dev_PortConnect() call in SIP endpoint\n");
   }
}

//*****************************************************************************
// Function: void SIPEndpoint::DisconnectFromPeer()
// Description: Disconnect media streams from peer endpoint
// Return: void 
// Parameters: none 
//*****************************************************************************
void SIPEndpoint::DisconnectFromPeer()
{
   Endpoint *pEndpoint = EndpointMngr::Instance()->GetEPFromIdx( GetBridgedCall()->M3gIndex());
   DM_PORT_CONNECT_INFO_LIST portConnectInfoList;

   if (m_IPMHandle == -1) {
    LOG_ENTRY(m_IPMHandle, "PortDisconnect (audio) not needed on device %s\n", m_IPMName);
   }
   else {
     // This call disconnects the Tx of IPMdevice to Rx of 3G device (Audio)
     EndpointMngr::Instance()->InitPortConnectInfoList(&portConnectInfoList, EndpointMngr::AUDIO_PORT,GetAudTranscodeEnabled());
     memcpy(&(portConnectInfoList.port_connect_info[0].port_info_tx),
	    &(GetAudioTxPortInfo()),
	    sizeof(DM_PORT_INFO));
     memcpy(&(portConnectInfoList.port_connect_info[0].port_info_rx),
	    &(pEndpoint->GetAudioRxPortInfo()),
	    sizeof(DM_PORT_INFO));
     
     if ( DEV_SUCCESS != dev_PortDisconnect(m_IPMHandle, &portConnectInfoList, this) )
       {
	 LOG_ERROR(m_IPMHandle,
		   "dev_PortDisconnect() failure in SIP endpoint - %s\n",
		   ATDV_ERRMSGP(m_IPMHandle));
       }
     else
       {
	 LOG_ENTRY(m_IPMHandle,"Successful dev_PortDisconnect() call in SIP endpoint\n");
       }
   }
   if (m_IPMHandle == -1) {
    LOG_ENTRY(m_IPMHandle, "PortDisconnect (video) not needed on device %s\n", m_IPMName);
   }
   else {
     // This call disconnects the Tx of IPMdevice to Rx of 3G device (Video)
     EndpointMngr::Instance()->InitPortConnectInfoList(&portConnectInfoList, EndpointMngr::VIDEO_PORT,GetVidTranscodeEnabled());
     memcpy(&(portConnectInfoList.port_connect_info[0].port_info_tx), 
	    &(GetVideoTxPortInfo()),
	    sizeof(DM_PORT_INFO));
     memcpy(&(portConnectInfoList.port_connect_info[0].port_info_rx), 
	    &(pEndpoint->GetVideoRxPortInfo()),
	    sizeof(DM_PORT_INFO));
     
     if ( DEV_SUCCESS != dev_PortDisconnect(m_IPMHandle, &portConnectInfoList, this) )
       {
	 LOG_ERROR(m_IPMHandle, 
		   "dev_PortDisconnect() failure in SIP endpoint - %s\n",
		   ATDV_ERRMSGP(m_IPMHandle));
       }
     else
       {
	 LOG_ENTRY(m_IPMHandle,"Successful dev_PortDisconnect() call in SIP endpoint\n");
       }
   }
}

void SIPEndpoint::WaitForCall ()
{
  LOG_ENTRY(m_GCSIPHandle, "Setting up to receive inbound call\n");

  if (gc_WaitCall(m_GCSIPHandle, NULL, NULL, -1, EV_ASYNC) != 0)
    LOG_ERROR(m_GCSIPHandle, "gc_WaitCall(%s) failure\n", m_SIPName);
}


void SIPEndpoint::BuildAnswerSDP(sdpSessionDescription *sdp)
{
    // Use the sdpapi to build an SDP from scratch
    // Local IP adress and audio/video ports have already been found
    // and stored using an IPML GetMedia call

    // Need to find current E-net interface/IP Address in use for this IPML device
    GetLocalMediaInfo();

    // Main choice when building SDP is audio only or audio/video
    sdp->version()->setVersion("0");

    sdp->origin()->setUserName("M3gGateway");
    sdp->origin()->setSessionId("1234");
    sdp->origin()->setVersion("5678");
    sdp->origin()->setNetworkType("IN");
    sdp->origin()->setAddressType("IP4");
    sdp->origin()->setAddress(m_localVideoRTPIPAddress);

    sdp->sessionName()->setName("Intel_SIP_CCLIB");
    sdp->sessionInformation()->setInfo("M3gGateway Session Info");

    sdp->connection()->setNetworkType("IN");
    sdp->connection()->setAddressType("IP4");
    sdp->connection()->setAddress(m_localVideoRTPIPAddress);

    sdpTimeDescription* pTD=sdp->timeDescriptionList()->addItem();
    unsigned long Start = 0;
    unsigned long Stop = 0;
    pTD->time()->setStart(Start);
    pTD->time()->setStop(Stop);

    // Add AMR and RFC2833 
    sdpMediaDescription* pMD=sdp->mediaDescriptionList()->addItem();
    pMD->media()->setMedia("audio");
    pMD->media()->setPort(m_localAudioRTPPort);
    pMD->media()->setTransport("RTP/AVP");
    pMD->media()->setNumPorts(1);
    pMD->media()->addFormat(m_sdpAudioFormat);
    sprintf(m_sdpRFC2833Format,"%d", m_nRFC2833_PT);
    pMD->media()->addFormat(m_sdpRFC2833Format);
    
    sdpAttributeList* pAttributeList = pMD->attributeList();
    sdpAttribute* pAttribute = pAttributeList->addItem();
    pAttribute->setProperty("rtpmap");
    pAttribute->setPropertyValue(m_sdpAudioRtpmap);
    
    pAttribute = pAttributeList->addItem(); 
    pAttribute->setProperty("fmtp"); 
    pAttribute->setPropertyValue(m_sdpAudioFmtp); 
    
    pAttribute = pAttributeList->addItem();
    pAttribute->setProperty("rtpmap");
    sprintf(m_sdpRFC2833Rtpmap, "%d telephone-event/8000", m_nRFC2833_PT); 
    pAttribute->setPropertyValue(m_sdpRFC2833Rtpmap);
    
    // Doing video, so add another media type and attributes
    pMD=sdp->mediaDescriptionList()->addItem();
    
    pMD->media()->setMedia("video");
    pMD->media()->setPort(m_localVideoRTPPort);
    pMD->media()->setTransport("RTP/AVP");
    pMD->media()->setNumPorts(1);
    pMD->media()->addFormat(m_sdpVideoFormat);

    pAttributeList = pMD->attributeList();
    pAttribute = pAttributeList->addItem();
    pAttribute->setProperty("rtpmap");
    pAttribute->setPropertyValue(m_sdpVideoRtpmap);

    pAttribute = pAttributeList->addItem();
    pAttribute->setProperty("fmtp");
    pAttribute->setPropertyValue(m_sdpVideoFmtp);
   
    if(GetVidTranscodeEnabled())
    {
      pMD->bandwidth()->setModifier("AS");
      char l_bandwidth[32];
      sprintf(l_bandwidth,"%d",(m_localBitRate/1000));
      LOG_ENTRY(0,"Bandwidth =%d,%s\n",m_localBitRate,l_bandwidth);
      pMD->bandwidth()->setBandwidthValue(l_bandwidth);
    }
    else
    {
      // For NATIVE mode, 
      // need to request to remote to limit the video  bandwidth (RFC2327)
      pMD->bandwidth()->setModifier("AS");
      pMD->bandwidth()->setBandwidthValue("40");
    }

}

void SIPEndpoint::AnswerCall(METAEVENT metaevent)
{
   GC_PARM_BLKP gcParmBlk = NULL;
   sdpSessionDescription answerSdp;
   char sdpAnswerString[1024];

   m_curCallCRN = metaevent.crn;

   answerSdp.clear();
   BuildAnswerSDP(&answerSdp);

   if (answerSdp.exportSDP(sdpAnswerString, 1024, false) == 0) {
     LOG_ENTRY(m_GCSIPHandle, "Export of Answer SDP successful\n");
   } else {
     LOG_ERROR(m_GCSIPHandle, "Export of Answer SDP failed\n");
   }
   // Add 1 to strlen for null termination 
   int len = strlen(sdpAnswerString) + 1;

   LOG_ENTRY(m_GCSIPHandle, "SDP length = %d bytes\n", strlen(sdpAnswerString));
   LOG_ENTRY(m_GCSIPHandle, sdpAnswerString);
   
   // Note that the "ex" version is used here to handle potentially long parameter (SDP)
   if (gc_util_insert_parm_ref_ex(&gcParmBlk,
				  IPSET_SDP,
				  IPPARM_SDP_ANSWER,
				  len,
				  sdpAnswerString) == -1) {
     LOG_ENTRY(m_GCSIPHandle, "gc_util_insert_parm_ref_ex failed\n");
   }
   
   if (gc_SetUserInfo(GCTGT_GCLIB_CRN, m_curCallCRN, gcParmBlk, GC_SINGLECALL) == -1) {
     LOG_ERROR(m_GCSIPHandle, "gc_SetUserInfo failed\n");
   }
   
   gc_util_delete_parm_blk(gcParmBlk);

   if (gc_AnswerCall(m_curCallCRN, 0, EV_ASYNC) != 0)
     {
       LOG_ERROR(m_GCSIPHandle, "Error calling gc_AnswerCall\n");
       LogGlobalCallError(); 
     }
}

//*****************************************************************************
// Function: void SIPEndpoint::ProcessCallAnswered(METAEVENT metaevent)
// Description: Get media streaming going on an inbound call that has been
//              answered
// Return: void 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void SIPEndpoint::ProcessCallAnswered(METAEVENT metaevent)
{
  StartMedia();
}

//*****************************************************************************
// Function: void SIPEndpoint::MakeCall(METAEVENT metaevent)
// Description: Make outbound SIP call
// Return: void 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void SIPEndpoint::MakeCall(METAEVENT metaevent)
{

   LOG_ENTRY(m_GCSIPHandle, "Setting up for outbound SIP call on %s\n", m_SIPName);

   SetupSipCoders();
   
   // Build and set sdp portion for INVITE to be sent
   int rc;
   GC_PARM_BLKP gcParmBlk = NULL;

   // Need to find current E-net interface/IP Address in use for this IPML device
   GetLocalMediaInfo();

   char tempId[32];
   sprintf( tempId, "%lu", (unsigned long)this);

   sdpSessionDescription sdp;
   sdp.clear();

   sdp.version()->setVersion("0");
   sdp.sessionName()->setName("Dialogic_SIP_CCLIB");
   sdp.sessionInformation()->setInfo("session information");
   sdp.origin()->setUserName("Dialogic_IPCCLib");
   sdp.origin()->setSessionId(tempId);
   sdp.origin()->setVersion(tempId);
   sdp.origin()->setNetworkType("IN");
   sdp.origin()->setAddressType("IP4");
   sdp.origin()->setAddress(m_localVideoRTPIPAddress);

   sdp.connection()->setNetworkType("IN");
   sdp.connection()->setAddressType("IP4");
   sdp.connection()->setAddress(m_localVideoRTPIPAddress);

   sdpTimeDescription* pTD=sdp.timeDescriptionList()->addItem();
   pTD->time()->setStart(0);
   pTD->time()->setStop(0);

   sdpMediaDescription* pMD=sdp.mediaDescriptionList()->addItem();
   pMD->media()->setMedia("audio");
   pMD->media()->setPort(m_localAudioRTPPort);
   pMD->media()->setTransport("RTP/AVP");
   pMD->media()->setNumPorts(1);
   pMD->media()->addFormat(m_sdpAudioFormat);
   sprintf(m_sdpRFC2833Format, "%d", m_nRFC2833_PT);
   pMD->media()->addFormat(m_sdpRFC2833Format);

   sdpAttributeList* pAttributeList = pMD->attributeList();
   sdpAttribute* pAttribute = pAttributeList->addItem();
   pAttribute->setProperty("rtpmap");
   pAttribute->setPropertyValue(m_sdpAudioRtpmap);

   pAttribute = pAttributeList->addItem(); 
   pAttribute->setProperty("fmtp"); 
   pAttribute->setPropertyValue(m_sdpAudioFmtp); 

   pAttribute = pAttributeList->addItem();
   pAttribute->setProperty("rtpmap");
   sprintf(m_sdpRFC2833Rtpmap, "%d telephone-event/8000", m_nRFC2833_PT); 
   pAttribute->setPropertyValue(m_sdpRFC2833Rtpmap);

   pMD=sdp.mediaDescriptionList()->addItem();
   pMD->media()->setMedia("video");
   pMD->media()->setPort(m_localVideoRTPPort);
   pMD->media()->setTransport("RTP/AVP");
   pMD->media()->setNumPorts(1);
   pMD->media()->addFormat(m_sdpVideoFormat);

   pAttributeList = pMD->attributeList();
   pAttribute = pAttributeList->addItem();
   pAttribute->setProperty("rtpmap");
   pAttribute->setPropertyValue(m_sdpVideoRtpmap);

   pAttribute = pAttributeList->addItem();
   pAttribute->setProperty("fmtp");
   pAttribute->setPropertyValue(m_sdpVideoFmtp);
   
//   if(EndpointMngr::Instance()->GetVidTranscodeModeEnabled())
   if(GetVidTranscodeEnabled())
    {
      pMD->bandwidth()->setModifier("AS");
      char l_bandwidth[32];
      sprintf(l_bandwidth,"%d",(m_localBitRate/1000));
      pMD->bandwidth()->setBandwidthValue(l_bandwidth);
    }
    else
    {
      // For NATIVE mode,
      // need to request to remote to limit the video  bandwidth (RFC2327)
      pMD->bandwidth()->setModifier("AS");
      pMD->bandwidth()->setBandwidthValue("40");
    }

   char sdpMsg[1024];
   if (sdp.exportSDP(sdpMsg, 1024, false) != 0) 
   {
      LOG_ERROR(m_GCSIPHandle, "Export of Answer SDP failed\n");
   }

   // Add 1 to strlen for null termination 
   int len = strlen(sdpMsg) + 1;

   LOG_ENTRY(m_GCSIPHandle, "SDP length = %d bytes\n", strlen(sdpMsg));
   LOG_ENTRY(m_GCSIPHandle, sdpMsg);

   // put the SDP content into the GC parameter block
   LOG_ENTRY(m_GCSIPHandle, "Setting IPSET_SDP\n");
   rc = gc_util_insert_parm_ref_ex(&gcParmBlk,
                                   IPSET_SDP, // set value 
                                   IPPARM_SDP_OFFER, // parm value 
                                   len,
                                   (void *) sdpMsg);
   if ( rc != 0 )
   {
      LOG_ERROR(m_GCSIPHandle, "Error inserting SDP OFFER\n");
      LogGlobalCallError();
      return;
   }

   char callInfoHeader[255];
   sprintf( callInfoHeader, "CallInfo:%s", (char*) metaevent.evtdatap);

   LOG_ENTRY(m_GCSIPHandle, "CallInfo: length = %d bytes\n", strlen(callInfoHeader));
   LOG_ENTRY(m_GCSIPHandle, "%s\n", callInfoHeader);

   rc = gc_util_insert_parm_ref_ex(&gcParmBlk,
                                   IPSET_SIP_MSGINFO,
                                   IPPARM_SIP_HDR,
                                   (unsigned long) (strlen( callInfoHeader) + 1),
                                   callInfoHeader);
   if ( rc != 0 )
   {
      LOG_ERROR(m_GCSIPHandle, "Error adding CallInfo SIP header\n");
      LogGlobalCallError(); 
      return; 
   }

   // Prepare for gc_MakeCall
   GC_MAKECALL_BLK gcMakeCallBlock;
   GCLIB_MAKECALL_BLK GcLibBlock;
   gcMakeCallBlock.gclib = &GcLibBlock;
   gcMakeCallBlock.cclib = 0;
   gcMakeCallBlock.gclib->ext_datap = gcParmBlk;

   strcpy(gcMakeCallBlock.gclib->destination.address, EndpointMngr::Instance()->GetDestSIPAddr());
   gcMakeCallBlock.gclib->destination.address_type = GCADDRTYPE_TRANSPARENT;

   strcpy(gcMakeCallBlock.gclib->origination.address, EndpointMngr::Instance()->GetOriginSIPAddr());
   gcMakeCallBlock.gclib->destination.address_type = GCADDRTYPE_TRANSPARENT;

   LOG_ENTRY(m_GCSIPHandle, "Making outbound SIP call (gcMakeCall)\n");
   if ( gc_MakeCall(GetGCSIPHandle(), &m_curCallCRN, (char *) 0, &gcMakeCallBlock, 
                    0, EV_ASYNC ) != GC_SUCCESS )
   {
      LogGlobalCallError();
   }
   gc_util_delete_parm_blk(gcParmBlk);
}

//*****************************************************************************
// Function: void SIPEndpoint::GetLocalMediaInfo()
// Description: Get media stream port info from local device
// Return: void 
// Parameters: none 
//*****************************************************************************
void SIPEndpoint::GetLocalMediaInfo()
{
   IPM_MEDIA_INFO LocalMediaInfo;
   IPM_MEDIA_INFO *pLocalMediaInfo = &LocalMediaInfo;

   LOG_ENTRY(m_IPMHandle, "Getting Local Media info\n");

   LocalMediaInfo.unCount = 2;
   LocalMediaInfo.MediaData[0].eMediaType = MEDIATYPE_VIDEO_LOCAL_RTP_INFO;
   LocalMediaInfo.MediaData[1].eMediaType = MEDIATYPE_AUDIO_LOCAL_RTP_INFO;
   if ( ipm_GetLocalMediaInfo(m_IPMHandle, &LocalMediaInfo, EV_SYNC) == -1 )
   {
      LOG_ENTRY(m_IPMHandle, "ipm_GetLocalMediaInfo failed - %s with error = %ld\n",
                ATDV_NAMEP(m_IPMHandle), ATDV_LASTERR(m_IPMHandle));
   }
   else
   {
      for ( unsigned int i=0; i<pLocalMediaInfo->unCount; i++ )
      {
         switch ( pLocalMediaInfo->MediaData[i].eMediaType )
         {
            case MEDIATYPE_VIDEO_LOCAL_RTP_INFO:
               LOG_ENTRY(m_IPMHandle, "MediaType=MEDIATYPE_VIDEO_LOCAL_RTP_INFO\n");
               LOG_ENTRY(m_IPMHandle, "PortId=%d\n", pLocalMediaInfo->MediaData[i].mediaInfo.PortInfo.unPortId);
               LOG_ENTRY(m_IPMHandle, "IP=%s\n", pLocalMediaInfo->MediaData[i].mediaInfo.PortInfo.cIPAddress);
               m_localVideoRTPPort = pLocalMediaInfo->MediaData[i].mediaInfo.PortInfo.unPortId;
               strcpy(m_localVideoRTPIPAddress, pLocalMediaInfo->MediaData[i].mediaInfo.PortInfo.cIPAddress);
               break;
            case MEDIATYPE_VIDEO_LOCAL_RTCP_INFO:
               LOG_ENTRY(m_IPMHandle, "MediaType=MEDIATYPE_VIDEO_LOCAL_RTCP_INFO\n");
               LOG_ENTRY(m_IPMHandle, "PortId=%d\n",pLocalMediaInfo->MediaData[i].mediaInfo.PortInfo.unPortId);
               LOG_ENTRY(m_IPMHandle, "IP=%s\n",pLocalMediaInfo->MediaData[i].mediaInfo.PortInfo.cIPAddress);
               m_localVideoRTCPPort = pLocalMediaInfo->MediaData[i].mediaInfo.PortInfo.unPortId;
               strcpy(m_localVideoRTCPIPAddress, pLocalMediaInfo->MediaData[i].mediaInfo.PortInfo.cIPAddress);
               break;
            case MEDIATYPE_AUDIO_LOCAL_RTP_INFO:
               LOG_ENTRY(m_IPMHandle, "MediaType=MEDIATYPE_AUDIO_LOCAL_RTP_INFO\n");
               LOG_ENTRY(m_IPMHandle, "PortId=%d\n",pLocalMediaInfo->MediaData[i].mediaInfo.PortInfo.unPortId);
               LOG_ENTRY(m_IPMHandle, "IP=%s\n",pLocalMediaInfo->MediaData[i].mediaInfo.PortInfo.cIPAddress);
               m_localAudioRTPPort = pLocalMediaInfo->MediaData[i].mediaInfo.PortInfo.unPortId;
               strcpy(m_localAudioRTPIPAddress, pLocalMediaInfo->MediaData[i].mediaInfo.PortInfo.cIPAddress);
               break;
            case MEDIATYPE_AUDIO_LOCAL_RTCP_INFO:
               LOG_ENTRY(m_IPMHandle, "MediaType=MEDIATYPE_AUDIO_LOCAL_RTCP_INFO\n");
               LOG_ENTRY(m_IPMHandle, "PortId=%d\n",pLocalMediaInfo->MediaData[i].mediaInfo.PortInfo.unPortId);
               LOG_ENTRY(m_IPMHandle, "IP=%s\n",pLocalMediaInfo->MediaData[i].mediaInfo.PortInfo.cIPAddress);
               m_localAudioRTCPPort = pLocalMediaInfo->MediaData[i].mediaInfo.PortInfo.unPortId;
               strcpy(m_localAudioRTCPIPAddress, pLocalMediaInfo->MediaData[i].mediaInfo.PortInfo.cIPAddress);
               break;
            default:
               LOG_ENTRY(m_IPMHandle, "Uexpected value returned from ipm_GetLocalMedia - %d\n",
                         pLocalMediaInfo->MediaData[i].eMediaType);
               break;
         }
      }
   }
}


//*****************************************************************************
// Function: void SIPEndpoint::StartMedia()
// Description: Start the media stream
// Return: void 
// Parameters: none 
//*****************************************************************************
void SIPEndpoint::StartMedia()
{

   IPM_VIDEO_CODER_INFO_EX l_ExtraVideoCoderInfoEx;

   // Set up and get audio and video RTP streams going via ipm_startmedia
   LOG_ENTRY(m_IPMHandle, "Priming StartMedia:RTP streams starting\n");
   LOG_ENTRY(m_IPMHandle, 
             "Remote RTP audio port = %d  Remote RTP video port = %d Remote Audio IP address = %s Remote Video IP address = %s\n",
             m_remoteAudioRTPPort, m_remoteVideoRTPPort, m_remoteAudioIPAddress, m_remoteVideoIPAddress);

   // Set  DTMF mode - INBAND or RFC2833.
   SetDtmfMode(RFC2833);

   // clear the MediaInfo structures
   memset(&m_mediaInfo, 0, sizeof(IPM_MEDIA_INFO));

   int miCnt = 0;

   // remote audio RTP port
   m_mediaInfo.MediaData[miCnt].eMediaType = MEDIATYPE_AUDIO_REMOTE_RTP_INFO;
   m_mediaInfo.MediaData[miCnt].mediaInfo.PortInfo.unPortId = m_remoteAudioRTPPort;
   strcpy(m_mediaInfo.MediaData[miCnt].mediaInfo.PortInfo.cIPAddress, m_remoteAudioIPAddress);
   miCnt++;

   // remote audio RTCP port
   m_mediaInfo.MediaData[miCnt].eMediaType = MEDIATYPE_AUDIO_REMOTE_RTCP_INFO;
   m_mediaInfo.MediaData[miCnt].mediaInfo.PortInfo.unPortId = m_remoteAudioRTPPort + 1;
   strcpy(m_mediaInfo.MediaData[miCnt].mediaInfo.PortInfo.cIPAddress, m_remoteAudioIPAddress);
   miCnt++;

   // remote video RTP port
   m_mediaInfo.MediaData[miCnt].eMediaType = MEDIATYPE_VIDEO_REMOTE_RTP_INFO;
   m_mediaInfo.MediaData[miCnt].mediaInfo.PortInfo.unPortId = m_remoteVideoRTPPort;
   strcpy(m_mediaInfo.MediaData[miCnt].mediaInfo.PortInfo.cIPAddress, m_remoteVideoIPAddress);
   miCnt++;

   // remote video RTCP port
   m_mediaInfo.MediaData[miCnt].eMediaType = MEDIATYPE_VIDEO_REMOTE_RTCP_INFO;
   m_mediaInfo.MediaData[miCnt].mediaInfo.PortInfo.unPortId = m_remoteVideoRTPPort + 1; 
   strcpy(m_mediaInfo.MediaData[miCnt].mediaInfo.PortInfo.cIPAddress, m_remoteVideoIPAddress);
   miCnt++;

   // remote audio codec
   LOG_ENTRY(m_IPMHandle, "AUDIO_REMOTE_CODER_INFO[%d]:\n",miCnt+1);
   LOG_ENTRY(m_IPMHandle, "\tremCoderType=%d, remCoderPayload=%d, remRedPayload=%d\n", 
                m_remoteAudioCoderType, m_remoteAudioCoderPayloadType, m_remoteAudioRedPayloadType);
   LOG_ENTRY(m_IPMHandle, "\tremFrameSize=%d, remFFP=%d, remVadEnable=%d\n", 
                m_remoteAudioCoderFramesize, m_remoteAudioFramesPerPkt, m_remoteAudioVadEnable);
   m_mediaInfo.MediaData[miCnt].eMediaType = MEDIATYPE_AUDIO_REMOTE_CODER_INFO;
   m_mediaInfo.MediaData[miCnt].mediaInfo.CoderInfo.eCoderType = m_remoteAudioCoderType;
   m_mediaInfo.MediaData[miCnt].mediaInfo.AudioCoderInfo.eFrameSize = m_remoteAudioCoderFramesize;
   m_mediaInfo.MediaData[miCnt].mediaInfo.AudioCoderInfo.unFramesPerPkt = m_remoteAudioFramesPerPkt;
   m_mediaInfo.MediaData[miCnt].mediaInfo.AudioCoderInfo.eVadEnable = m_remoteAudioVadEnable;
   m_mediaInfo.MediaData[miCnt].mediaInfo.AudioCoderInfo.unCoderPayloadType = m_remoteAudioCoderPayloadType;
   m_mediaInfo.MediaData[miCnt].mediaInfo.AudioCoderInfo.unRedPayloadType = m_remoteAudioRedPayloadType;   
   miCnt++;

   if((m_remoteAudioCoderType&0xFF) == CODER_TYPE_AMRNB_12_2k){
      LOG_ENTRY(m_IPMHandle, "AUDIO_REMOTE_CODER_OPTIONS[%d]:\n",miCnt+1);
      LOG_ENTRY(m_IPMHandle, "\tlocCoderType=CODER_TYPE_AMRNB_12_2k, locCoderOptions=CODER_OPT_AMR_OCTET\n");
      m_mediaInfo.MediaData[miCnt].eMediaType = MEDIATYPE_AUDIO_REMOTE_CODER_OPTIONS_INFO;
      m_mediaInfo.MediaData[miCnt].mediaInfo.AudioCoderOptionsInfo.unVersion =
        IPM_AUDIO_CODER_OPTIONS_INFO_VERSION;
      m_mediaInfo.MediaData[miCnt].mediaInfo.AudioCoderOptionsInfo.unCoderOptions=
        CODER_OPT_AMR_CMR_TRACK | CODER_OPT_AMR_OCTET; //CODER_OPT_AMR_EFFICIENT;
      miCnt++;
   }

   // local audio codec
   LOG_ENTRY(m_IPMHandle, "AUDIO_LOCAL_CODER_INFO[%d]:\n",miCnt+1);
   LOG_ENTRY(m_IPMHandle, "\tlocCoderType=%d, locCoderPayload=%d, locRedPayload=%d\n", 
                m_localAudioCoderType, m_localAudioCoderPayloadType, m_localAudioRedPayloadType);
   LOG_ENTRY(m_IPMHandle, "\tlocFrameSize=%d, locFFP=%d, locVadEnable=%d\n", 
                m_localAudioCoderFramesize, m_localAudioFramesPerPkt, m_localAudioVadEnable);
   m_mediaInfo.MediaData[miCnt].eMediaType = MEDIATYPE_AUDIO_LOCAL_CODER_INFO;
   m_mediaInfo.MediaData[miCnt].mediaInfo.CoderInfo.eCoderType = m_localAudioCoderType;
   m_mediaInfo.MediaData[miCnt].mediaInfo.AudioCoderInfo.eFrameSize = m_localAudioCoderFramesize;
   m_mediaInfo.MediaData[miCnt].mediaInfo.AudioCoderInfo.unFramesPerPkt = m_localAudioFramesPerPkt;
   m_mediaInfo.MediaData[miCnt].mediaInfo.AudioCoderInfo.eVadEnable = m_localAudioVadEnable;
   m_mediaInfo.MediaData[miCnt].mediaInfo.AudioCoderInfo.unCoderPayloadType = m_localAudioCoderPayloadType;
   m_mediaInfo.MediaData[miCnt].mediaInfo.AudioCoderInfo.unRedPayloadType = m_localAudioRedPayloadType;   
   miCnt++;

   if((m_localAudioCoderType&0xFF) == CODER_TYPE_AMRNB_12_2k){
      LOG_ENTRY(m_IPMHandle, "AUDIO_LOCAL_CODER_OPTIONS[%d]:\n",miCnt+1);
      LOG_ENTRY(m_IPMHandle, "\tlocCoderType=CODER_TYPE_AMRNB_12_2k, locCoderOptions=CODER_OPT_AMR_OCTET\n");
      m_mediaInfo.MediaData[miCnt].eMediaType = MEDIATYPE_AUDIO_LOCAL_CODER_OPTIONS_INFO;
      m_mediaInfo.MediaData[miCnt].mediaInfo.AudioCoderOptionsInfo.unVersion =
        IPM_AUDIO_CODER_OPTIONS_INFO_VERSION;
      m_mediaInfo.MediaData[miCnt].mediaInfo.AudioCoderOptionsInfo.unCoderOptions=
        CODER_OPT_AMR_OCTET;
      miCnt++;
   }

   // remote video codec
   LOG_ENTRY(m_IPMHandle, "VIDEO_REMOTE_CODER_INFO[%d]:\n",miCnt+1);
   LOG_ENTRY(m_IPMHandle, "\tremCoderType=%d, remCoderPayload=%d, remImageProfile=%d\n", 
                m_remoteVideoCoderType, m_remoteVideoCoderPayloadType, m_remoteImageProfile);
   m_mediaInfo.MediaData[miCnt].eMediaType = MEDIATYPE_VIDEO_REMOTE_CODER_INFO;
   INIT_IPM_VIDEO_CODER_INFO(&m_mediaInfo.MediaData[miCnt].mediaInfo.VideoCoderInfo);
   m_mediaInfo.MediaData[miCnt].mediaInfo.VideoCoderInfo.eCoderType = m_remoteVideoCoderType;
   m_mediaInfo.MediaData[miCnt].mediaInfo.VideoCoderInfo.unCoderPayloadType = m_remoteVideoCoderPayloadType;
   
   INIT_IPM_VIDEO_CODER_INFO_EX(&l_ExtraVideoCoderInfoEx);
   m_mediaInfo.MediaData[miCnt].mediaInfo.VideoCoderInfo.pExtraCoderInfo = &l_ExtraVideoCoderInfoEx;
   l_ExtraVideoCoderInfoEx.eProfile = m_remoteImageProfile;
   l_ExtraVideoCoderInfoEx.eLevel = m_remoteImageLevel;
   l_ExtraVideoCoderInfoEx.eImageWidth = m_remoteImageWidth;
   l_ExtraVideoCoderInfoEx.eImageHeight = m_remoteImageHeight;
   l_ExtraVideoCoderInfoEx.eFramesPerSec = m_remoteFramesPerSec;
   l_ExtraVideoCoderInfoEx.unBitRate = m_remoteBitRate;
   l_ExtraVideoCoderInfoEx.eSamplingRate = m_remoteSamplingRate;
   l_ExtraVideoCoderInfoEx.unVisualConfigSize = 0;
   l_ExtraVideoCoderInfoEx.szVisualConfiguration = NULL;
   miCnt++;

   // local video codec
   LOG_ENTRY(m_IPMHandle, "VIDEO_LOCAL_CODER_INFO[%d]:\n",miCnt+1);
   LOG_ENTRY(m_IPMHandle, "\tlocCoderType=%d, locCoderPayload=%d, locImageProfile=%d\n", 
                m_localVideoCoderType, m_localVideoCoderPayloadType, m_localImageProfile);
   m_mediaInfo.MediaData[miCnt].eMediaType = MEDIATYPE_VIDEO_LOCAL_CODER_INFO;
   INIT_IPM_VIDEO_CODER_INFO(&m_mediaInfo.MediaData[miCnt].mediaInfo.VideoCoderInfo);
   m_mediaInfo.MediaData[miCnt].mediaInfo.VideoCoderInfo.eCoderType = m_localVideoCoderType;
   m_mediaInfo.MediaData[miCnt].mediaInfo.VideoCoderInfo.unCoderPayloadType = m_localVideoCoderPayloadType;
   
   INIT_IPM_VIDEO_CODER_INFO_EX(&l_ExtraVideoCoderInfoEx);
   m_mediaInfo.MediaData[miCnt].mediaInfo.VideoCoderInfo.pExtraCoderInfo = &l_ExtraVideoCoderInfoEx;
   l_ExtraVideoCoderInfoEx.eProfile = m_localImageProfile;
   l_ExtraVideoCoderInfoEx.eLevel = m_localImageLevel;
   l_ExtraVideoCoderInfoEx.eImageWidth = m_localImageWidth;
   l_ExtraVideoCoderInfoEx.eImageHeight = m_localImageHeight;
   l_ExtraVideoCoderInfoEx.eFramesPerSec = m_localFramesPerSec;
   l_ExtraVideoCoderInfoEx.unBitRate = m_localBitRate;
   l_ExtraVideoCoderInfoEx.eSamplingRate = m_localSamplingRate;
   l_ExtraVideoCoderInfoEx.unVisualConfigSize = 0;
   l_ExtraVideoCoderInfoEx.szVisualConfiguration = NULL;
   //JM - DCI setting local coder config required for decoder
   if((m_localVideoCoderType&0xFF) == CODER_TYPE_MP4V_ES) {
       LOG_ENTRY(m_IPMHandle, "unVisualConfigSize=%d|szVisualConfiguration[0]=0x%x|[1]=0x%x|[2]=0x%x|[3]=0x%x|[4]=0x%x|\n", m_dciOctetStrSize,m_dciOctetStr[0],m_dciOctetStr[1],m_dciOctetStr[2],m_dciOctetStr[3],m_dciOctetStr[4]);
       if(m_dciOctetStrSize){
         l_ExtraVideoCoderInfoEx.unVisualConfigSize = m_dciOctetStrSize;
         l_ExtraVideoCoderInfoEx.szVisualConfiguration = m_dciOctetStr;
       }
   }
   miCnt++;

   // Must correspond to number of elements set
   m_mediaInfo.unCount = miCnt;

   LOG_ENTRY(m_IPMHandle, "IP_MEDIA_INFO contains %d elements remoteBitRate=%u|localBitRate=%u\n", miCnt,m_remoteBitRate,m_localBitRate);
   if ( ipm_StartMedia(m_IPMHandle, &m_mediaInfo, DATA_IP_TDM_BIDIRECTIONAL, EV_ASYNC) != 0 )
   {
      LOG_ERROR(m_IPMHandle, "ipm_StartMedia() failed - %s\n", ATDV_ERRMSGP(m_IPMHandle));
   }
   else
   {
      LOG_ENTRY(m_IPMHandle, "RTP streams started\n");
   }
}

//*****************************************************************************
// Function: void SIPEndpoint::StopMedia()
// Description: Stop the media stream
// Return: void 
// Parameters: none 
//*****************************************************************************
void SIPEndpoint::StopMedia()
{
   LOG_ENTRY(m_IPMHandle, "Stoping RTP streams for %s\n", m_IPMName);
   if ( ipm_Stop(m_IPMHandle, STOP_ALL, EV_ASYNC) != 0 )
   {
      LOG_ERROR(m_IPMHandle, "ipm_Stop() failed - %s\n", ATDV_ERRMSGP(m_IPMHandle));
   }
}

//*****************************************************************************
// Function: void SIPEndpoint::LogDevError()
// Description: Lof dev API error
// Return: void 
// Parameters: none 
//*****************************************************************************
void SIPEndpoint::LogDevError()
{
   DEV_ERRINFO errorInfo;
   if ( dev_ErrorInfo(&errorInfo) != -1 )
   {
      LOG_ERROR (m_IPMHandle, "Error during dev_* call: %d - %s\n", errorInfo.dev_ErrValue, errorInfo.dev_Msg);
   }
}

//*****************************************************************************
// Function: void SIPEndpoint::LogGlobalCallError()
// Description: Log GC API error
// Return: void 
// Parameters: none 
//*****************************************************************************
void SIPEndpoint::LogGlobalCallError()
{
   GC_INFO gcInfo;

   if ( gc_ErrorInfo(&gcInfo) != GC_SUCCESS )
   {
      LOG_ERROR(m_GCSIPHandle, "gc_ErrorInfo failed\n");
   }
   else
   {
      LOG_ERROR(m_GCSIPHandle, "gcValue  : 0x%x\n", gcInfo.gcValue);
      LOG_ERROR(m_GCSIPHandle, "gcMsg    : %s\n", gcInfo.gcMsg);
      LOG_ERROR(m_GCSIPHandle, "ccLibId  : %d\n", gcInfo.ccLibId);
      LOG_ERROR(m_GCSIPHandle, "ccLibName: %s\n", gcInfo.ccLibName);
      LOG_ERROR(m_GCSIPHandle, "ccValue  : 0x%x\n", gcInfo.ccValue);
      LOG_ERROR(m_GCSIPHandle, "ccMsg    : %s\n", gcInfo.ccMsg);
      LOG_ERROR(m_GCSIPHandle, "info     : %s\n", gcInfo.additionalInfo);
   }
}

//*****************************************************************************
// Function: void SIPEndpoint::SetDtmfMode(int dtmfMode)
// Description: Set the DTMF mode
// Return: void 
// Parameters: int dtmfMode 
//*****************************************************************************
void SIPEndpoint::SetDtmfMode(int dtmfMode)
{
   // Local Variable Declaration
   IPM_PARM_INFO      parmInfo;
   eIPM_DTMFXFERMODE  value;
   int       PLType   = 0;
   int Rfc2833PayloadType = 101;

   switch ( dtmfMode )
   {
      case RFC2833:
         {
            // set the dtmf mode to RFC2833
            value       = DTMFXFERMODE_RFC2833;
            parmInfo.eParm    = PARMCH_DTMFXFERMODE;
            parmInfo.pvParmValue = &value;
            LOG_ENTRY(m_IPMHandle, "DTMFs will be detected using RFC2833\n");
            if ( ipm_SetParm(m_IPMHandle, &parmInfo, EV_SYNC) < 0 )
            {
               LOG_ERROR(m_IPMHandle, "Error in ipm_SetParm()\n");
            }

            // set the TX Payload Type
            PLType = Rfc2833PayloadType;
            parmInfo.eParm = PARMCH_RFC2833EVT_TX_PLT;
            parmInfo.pvParmValue = &PLType;
            LOG_ENTRY(m_IPMHandle, "Setting Parameter PARMCH_RFC2833EVT_TX_PLT to %d\n",Rfc2833PayloadType);
            if ( ipm_SetParm(m_IPMHandle, &parmInfo, EV_SYNC) < 0 )
            {
               LOG_ERROR(m_IPMHandle, "Error in ipm_SetParm()\n");
            }
            // set the RX Payload Type
            PLType = Rfc2833PayloadType;
            parmInfo.eParm = PARMCH_RFC2833EVT_RX_PLT;
            parmInfo.pvParmValue = &PLType;
            LOG_ENTRY(m_IPMHandle, "Setting Parameter PARMCH_RFC2833EVT_RX_PLT to %d\n", Rfc2833PayloadType);
            if ( ipm_SetParm(m_IPMHandle, &parmInfo, EV_SYNC) < 0 )
            {
               LOG_ERROR(m_IPMHandle, "Error in ipm_SetParm()\n");
            }
         }
         break;

      case INBAND:
         {
            eIPM_DTMFXFERMODE value = DTMFXFERMODE_INBAND;
            parmInfo.eParm = PARMCH_DTMFXFERMODE;
            parmInfo.pvParmValue = &value;

            LOG_ENTRY(m_IPMHandle, "DTMFs will be detected in INBAND Rtp\n");
            if ( ipm_SetParm(m_IPMHandle, &parmInfo, EV_SYNC) < 0 )
            {
               LOG_ERROR(m_IPMHandle, "Error in ipm_SetParm()\n");
            }
         }
         break;
   }
}

//*****************************************************************************
// Function: void SIPEndpoint::ProcessFarEndConnected(METAEVENT metaevent)
// Description: Process a  connected event
// Return: void 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void SIPEndpoint::ProcessFarEndConnected(METAEVENT metaevent)
{
  // A GC connection event (SIP OK) has arrived.  Extract the SDP, start the media
  // streams and send a SIP ACK.
  
  // Want to use the "Ex" versions to handle possible data > 255 bytes
  GC_PARM_BLKP    parm_blkp = (GC_PARM_BLKP)metaevent.extevtdatap;
  GC_PARM_DATA_EXT l_parm;
  INIT_GC_PARM_DATA_EXT( &l_parm);
  
  while ( (gc_util_next_parm_ex( parm_blkp, &l_parm)) == GC_SUCCESS )
  {
    LOG_ENTRY(m_GCSIPHandle, "set id = 0x%x parm id = 0x%x\n", l_parm.set_ID, l_parm.parm_ID);
    switch ( l_parm.set_ID )
    {
      case IPSET_SIP_MSGINFO:  // 0x301b
        LOG_ENTRY(m_GCSIPHandle, "IPSET_MSG_INFO\n");
        switch ( l_parm.parm_ID )
        {
           case IPPARM_SIP_HDR:
            LOG_ENTRY(m_GCSIPHandle, "IPPARM_SIP_HDR\n");
            break;
           case IPPARM_CONTACT_URI:
            LOG_ENTRY(m_GCSIPHandle, "IPPARM_CONTACT_URI\n"); // 0x02
            break;
           case IPPARM_CALLID_HDR:
            LOG_ENTRY(m_GCSIPHandle, "IPPARM_CALLID_HDR\n"); // 0x10
            break;
           case IPPARM_FROM:
            LOG_ENTRY(m_GCSIPHandle, "IPPARM_FROM\n"); // 0x12
            break;
           case IPPARM_TO:
            LOG_ENTRY(m_GCSIPHandle, "IPPARM_TO\n"); // 0x13	
            break;
           default:
            LOG_ENTRY(m_GCSIPHandle, "Unknown parm ID 0x%x\n", l_parm.parm_ID);
            break;
        }
        break;
      case IPSET_SDP:    // 0x301b
        LOG_ENTRY(m_GCSIPHandle, "IPSET_SIP\n");
        switch ( l_parm.parm_ID )
        {
          case IPPARM_SDP_OFFER:  // 0x01
            LOG_ENTRY(m_GCSIPHandle, "IPPARM_SDP_OFFER\n");
          case IPPARM_SDP_ANSWER:  // 0x02
            LOG_ENTRY(m_GCSIPHandle, "IPPARM_SDP_ANSWER\n");
            // Log the SDP just received
            char sdpBuffer1[4096];
            strncpy( sdpBuffer1, (char*)l_parm.pData, l_parm.data_size);
            sdpBuffer1[l_parm.data_size]='\0';
            LOG_ENTRY(m_GCSIPHandle, "SDP =\n%s", sdpBuffer1);
                  sdpSessionDescription offersdp;
                  ParseSdp(offersdp,sdpBuffer1);
	    break;
        } // end switch parmID 
    } // end switch setID
  } //end while
  
  // Have what we need to know from SDP, so start RTP streams
  StartMedia();
  
  // Need to manually send an ACK
  if ( gc_SipAck(m_curCallCRN, NULL, EV_ASYNC) != 0 )
  {
    LOG_ERROR(m_GCSIPHandle, "Error calling gc_SIPAck\n");
    LogGlobalCallError();
  }
  else
  {
    LOG_ENTRY(m_GCSIPHandle, "SIP Ack (GC_SipAck) sent\n");
  }
}

//*****************************************************************************
// Function: void SIPEndpoint::SaveRemoteMediaInfo(METAEVENT metaevent)
// Description: Extract media info needed from an INVITE
// Return: void 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void SIPEndpoint::SaveRemoteMediaInfo(METAEVENT metaevent)
{
  LOG_ENTRY(m_GCSIPHandle, "SaveRemoteMediaInfo()\n");

   // Want to use the "Ex" versions to handle possible data > 255 bytes
   GC_PARM_BLKP    parm_blkp = (GC_PARM_BLKP)metaevent.extevtdatap;
   GC_PARM_DATA_EXT l_parm;
   INIT_GC_PARM_DATA_EXT( &l_parm);

   while ( (gc_util_next_parm_ex( parm_blkp, &l_parm)) == GC_SUCCESS )
   {
      LOG_ENTRY(m_GCSIPHandle, "set id = 0x%x parm id = 0x%x\n", l_parm.set_ID, l_parm.parm_ID);
      switch ( l_parm.set_ID )
      {
         case IPSET_SIP_MSGINFO:  // 0x301b
            LOG_ENTRY(m_GCSIPHandle, "IPSET_MSG_INFO\n");
            switch ( l_parm.parm_ID )
            {
               case IPPARM_SIP_HDR:
                  LOG_ENTRY(m_GCSIPHandle, "IPPARM_SIP_HDR\n");
                  break;
               case IPPARM_CONTACT_URI:
                  LOG_ENTRY(m_GCSIPHandle, "IPPARM_CONTACT_URI\n"); // 0x02
                  break;
               case IPPARM_CALLID_HDR:
                  LOG_ENTRY(m_GCSIPHandle, "IPPARM_CALLID_HDR\n"); // 0x10
                  break;
               case IPPARM_FROM:
                  LOG_ENTRY(m_GCSIPHandle, "IPPARM_FROM\n"); // 0x12
                  break;
               case IPPARM_TO:
                  LOG_ENTRY(m_GCSIPHandle, "IPPARM_TO\n"); // 0x13	
                  break;
               default:
                  LOG_ENTRY(m_GCSIPHandle, "Unknown parm ID 0x%x\n", l_parm.parm_ID);
                  break;
            }
            break;
         case IPSET_SDP:    // 0x3027
            LOG_ENTRY(m_GCSIPHandle, "IPSET_SIP\n");
            switch ( l_parm.parm_ID )
            {
               case IPPARM_SDP_OFFER:  // 0x01
                  LOG_ENTRY(m_GCSIPHandle, "IPPARM_SDP_OFFER\n");
               case IPPARM_SDP_ANSWER:  // 0x02
                  LOG_ENTRY(m_GCSIPHandle, "IPPARM_SDP_ANSWER\n");
                  // Log the SDP just received
                  char sdpBuffer1[4096];
                  strncpy( sdpBuffer1, (char*)l_parm.pData, l_parm.data_size);
                  sdpBuffer1[l_parm.data_size]='\0';
                  LOG_ENTRY(m_GCSIPHandle, "SDP=\n%s", sdpBuffer1);
                  sdpSessionDescription offersdp;
                  ParseSdp(offersdp,sdpBuffer1);
                  break;
            }
      }
   }
}

//*****************************************************************************
// Function: void SIPEndpoint::ProcessDisconnect(METAEVENT metaevent)
// Description: Process a disconnect event
// Return: void 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void SIPEndpoint::ProcessDisconnect(METAEVENT metaevent)
{
   // A GC disconnect event (SIP BYE) has arrived.  
   // The call should be dropped from this side as well.

   if ( gc_DropCall(m_curCallCRN, GC_NORMAL_CLEARING, EV_ASYNC) != GC_SUCCESS )
   {
      LOG_ERROR(m_GCSIPHandle, "Error calling gc_DropCall\n");
      LogGlobalCallError();
   }
   else
   {
      LOG_ENTRY(m_GCSIPHandle, "Call dropped (gc_DropCall)\n");
   }
}

//*****************************************************************************
// Function: void SIPEndpoint::ProcessDropCall(METAEVENT metaevent)
// Description: Process dropped call event
// Return: void 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void SIPEndpoint::ProcessDropCall(METAEVENT metaevent)
{
   // A GC dropcall event has arrive.  Release GC resources.

   if ( gc_ReleaseCallEx(m_curCallCRN, EV_ASYNC) != GC_SUCCESS )
   {
      LOG_ERROR(m_GCSIPHandle, "Error calling gc_ReleaseCallEx\n");
      LogGlobalCallError();
   }
   else
   {
      LOG_ENTRY(m_GCSIPHandle, "Call released (gc_ReleaseCallEx)\n");
   }
}

//*****************************************************************************
// Function: void SIPEndpoint::CloseSubDevs()
// Description: Close devices assoicated wioth this endpoint
// Return: void 
// Parameters: none 
//*****************************************************************************
void SIPEndpoint::CloseSubDevs()
{
//  usleep(500000);

  if (m_GCSIPHandle == -1) {
    LOG_ENTRY(m_GCSIPHandle, "GC close not needed on device %s\n", m_SIPName);
  }
  else {
    LOG_ENTRY(m_GCSIPHandle, "Closing GC device %s\n", m_SIPName);
    if ( gc_Close(m_GCSIPHandle) < 0 )
      {
	LOG_ERROR(m_GCSIPHandle, "gc_Close(%s) failure\n", m_SIPName);
	LogGlobalCallError();
      }
    else {
      m_GCSIPHandle = -1;
      LOG_ENTRY(0, "Successful gc_Close completion on SIP device %s\n", m_SIPName);
    }
  }

  if (m_IPMHandle == -1) {
    LOG_ENTRY(m_IPMHandle, "IPM close not needed on device %s\n", m_IPMName);
  }
  else {
    if ( ipm_Close(m_IPMHandle, NULL) != 0 ) {
      LOG_ERROR(m_IPMHandle, "ipm_Close(%s) failure\n", m_IPMName);
    }
    else {
      m_IPMHandle = -1;
      LOG_ENTRY(m_IPMHandle, "Successful ipm_Close comletion on device %s\n", m_IPMName);
    }
  }
}

//*****************************************************************************
// Function: bool SIPEndpoint::SendIFrameRequest()
// Description: Send an IFrame request
// Return: bool 
// Parameters: none 
//*****************************************************************************
bool SIPEndpoint::SendIFrameRequest()
{
   // Local Variable Declaration
   GC_PARM_BLKP gcParmBlk_mime = 0;
   GC_PARM_BLKP gcParmBlk_mime1 = 0;
   GC_PARM_BLKP gcParmBlk_info = 0;
   bool   bOk   = true;
   const char   *pBodyType = "Content-Type:application/media_control+xml"; // specify the body type

   LOG_ENTRY(m_GCSIPHandle, "SendIFrameRequest()\n");
   if ( gc_util_insert_parm_ref(&gcParmBlk_mime,
                                IPSET_MIME,
                                IPPARM_MIME_PART_TYPE,
                                (unsigned char)(strlen(pBodyType) + 1),
                                (void*)pBodyType) < 0 )
   {
      LOG_ERROR(m_GCSIPHandle, 
                "SendIFrameRequest() -> gc_util_insert_parm_ref() failed for IPPARM_MIME_PART_TYPE\n");
      bOk = false;
   }

   // insert the body size
   if ( gc_util_insert_parm_val(&gcParmBlk_mime,
                                IPSET_MIME,
                                IPPARM_MIME_PART_BODY_SIZE,
                                sizeof(unsigned long),
                                strlen(c_iFrameRequest)) < 0 )
   {
      LOG_ERROR(m_GCSIPHandle, 
                "SendIFrameRequest() -> gc_util_insert_parm_val() failed for IPPARM_MIME_PART_BODY_SIZE\n");
      bOk = false;
   }

   // insert the body
   if ( gc_util_insert_parm_val(&gcParmBlk_mime,
                                IPSET_MIME,
                                IPPARM_MIME_PART_BODY,
                                sizeof(unsigned long),
                                (unsigned long)(c_iFrameRequest)) < 0 )
   {
      LOG_ERROR(m_GCSIPHandle, 
                "SendIFrameRequest() -> gc_util_insert_parm_val() failed for IPPARM_MIME_PART_BODY\n");
      bOk = false;
   }

   // insert the list of parmBlks into the top level parmBlk
   if ( gc_util_insert_parm_val(&gcParmBlk_mime1,
                                IPSET_MIME,
                                IPPARM_MIME_PART,
                                sizeof(unsigned long),
                                (unsigned long)gcParmBlk_mime) < 0 )
   {
      LOG_ERROR(m_GCSIPHandle, 
                "SendIFrameRequest() -> gc_util_insert_parm_val() failed for IPPARM_MIME_PART\n");
      bOk = false;
   }

   // now set it on the device
   if ( gc_SetUserInfo(GCTGT_GCLIB_CRN,
                       m_curCallCRN,
                       gcParmBlk_mime1,
                       GC_SINGLECALL) < 0 ) // for this call only
   {
      LOG_ERROR(m_GCSIPHandle, 
                "gc_SetUserInfo() failed for MIME body in INFO\n");
      bOk = false;
   }

   // insert the message type
   if ( gc_util_insert_parm_val(&gcParmBlk_info,
                                IPSET_MSG_SIP,
                                IPPARM_MSGTYPE,
                                sizeof(int),
                                IP_MSGTYPE_SIP_INFO) < 0 )
   {
      LOG_ERROR(m_GCSIPHandle, 
                "SendIFrameRequest() -> gc_util_insert_parm_val() failed for SIP INFO\n");
      bOk = false;
   }

   if ( gc_Extension(GCTGT_GCLIB_CRN, m_curCallCRN, IPEXTID_SENDMSG, 
                     gcParmBlk_info, NULL, EV_ASYNC) < 0 )
   {
      LOG_ERROR(m_GCSIPHandle, 
                "SendIFrameRequest() -> gc_Extension failed\n");
      bOk = false;
   }
   gc_util_delete_parm_blk(gcParmBlk_info);
   gc_util_delete_parm_blk(gcParmBlk_mime);

   return bOk;
}

//*****************************************************************************
// Function: bool SIPEndpoint::IsSIPIFrameUpdate(METAEVENT metaevent)
// Description: Answer if event is an IFrame update request
// Return: bool 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
bool SIPEndpoint::IsSIPIFrameUpdate(METAEVENT metaevent)
{
   // Have an event indicating a SIP Info message has arrived.  See if its
   // MIME part is requesting an IFrame update 

   // There is also SIP header info which ought to be checked for source IP
   // Address at least...

   GC_PARM_BLK* l_parmBlock = static_cast<GC_PARM_BLK*>(metaevent.extevtdatap);
   GC_PARM_DATA_EXT l_parm;
   INIT_GC_PARM_DATA_EXT( &l_parm);

   while ( ( gc_util_next_parm_ex( l_parmBlock, &l_parm)) == GC_SUCCESS )
   {
      switch ( l_parm.set_ID )
      {
         case IPSET_SIP_MSGINFO:
            switch ( l_parm.parm_ID )
            {
               case IPPARM_SIP_HDR:
                  {
                     // Just log header info for now.  Not worried in a controlled demo
                     // environment about parsing it
                     char* l_parmData = new char[l_parm.data_size+1];
                     strncpy( l_parmData,(char*)l_parm.pData, l_parm.data_size);
                     l_parmData[l_parm.data_size]='\0';
                     if ( strstr( l_parmData, "Call-ID:") )
                     {
                        LOG_ENTRY(metaevent.evtdev, "IsSIPIFrameUpdate() - Call-ID: %s\n", 
                                  &l_parmData[strlen("Call-ID:")]);
                     }
                     else if ( strstr( l_parmData, "To:") )
                     {
                        LOG_ENTRY(metaevent.evtdev, "IsSIPIFrameUpdate() - To: %s\n", 
                                  &l_parmData[strlen("To:")]);
                     }
                     else if ( strstr( l_parmData, "From:") )
                     {
                        LOG_ENTRY(metaevent.evtdev, "IsSIPIFrameUpdate() - From: %s\n", 
                                  &l_parmData[strlen("From:")]);
                     }
                     else
                     {
                        LOG_ENTRY(metaevent.evtdev, "IsSIPIFrameUpdate() - Unknown header %s\n", 
                                  l_parmData);
                     }
                     delete[] l_parmData;
                  }
                  break;
            }
            break;

         case IPSET_MIME:
            {
               // MIME info in SIP Info message.  Extract it.
               GC_PARM_BLK* l_mimeParmBlock = (GC_PARM_BLK*)(*(unsigned int*)( l_parm.pData));
               GC_PARM_DATA_EXT l_mimeParm;
               INIT_GC_PARM_DATA_EXT( &l_mimeParm);
               unsigned long l_mimeSize = 0;
               while ( gc_util_next_parm_ex( l_mimeParmBlock, &l_mimeParm) == GC_SUCCESS )
               {
                  switch ( l_mimeParm.parm_ID )
                  {
                     case IPPARM_MIME_PART_TYPE:
                        {
                           char* l_mimeType = new char[l_mimeParm.data_size+1];
                           memcpy( l_mimeType, l_mimeParm.pData, l_mimeParm.data_size);
                           l_mimeType[ l_mimeParm.data_size] = '\0';
                           LOG_ENTRY(metaevent.evtdev, "IsSIPIFrameUpdate() - MIME type %s\n", l_mimeType);
                           delete[] l_mimeType;
                        }
                        break;

                     case IPPARM_MIME_PART_BODY_SIZE:
                        l_mimeSize = *(unsigned long*)( l_mimeParm.pData);
                        LOG_ENTRY(metaevent.evtdev, "IsSIPIFrameUpdate() - MIME size %ld\n", l_mimeSize);
                        break;

                     case IPPARM_MIME_PART_BODY:
                        if ( l_mimeSize > 0 )
                        {
                           char* l_mimeData = new char[l_mimeSize+1];
                           memcpy( l_mimeData, (char*)(*(unsigned long*)(l_mimeParm.pData)), 
                                   l_mimeSize);
                           l_mimeData[ l_mimeSize] = '\0';
                           LOG_ENTRY(metaevent.evtdev, "IsSIPIFrameUpdate() - MIME data %s\n", 
                                     l_mimeData);

                           // Now compare against expected MIME data from MM demo that indicates
                           // an IFrame update request
                           if ( strcmp(c_iFrameRequest, l_mimeData) == 0 )
                              return true;
                        }
                        break;

                     default:
                        LOG_ENTRY(metaevent.evtdev, 
                                  "IsSIPIFrameUpdate() - Unexpected IPSET_MIME parm ID %d\n", 
                                  l_mimeParm.parm_ID);
                        break;
                  }
               }
            }
            break;
      }
   }
   return false;
}

//*****************************************************************************
// Function: void SIPEndpoint::SendSIPOK()
// Description: Sendf SIP OK message
// Return: void 
// Parameters: none 
//*****************************************************************************
void SIPEndpoint::SendSIPOK ()
{
   LOG_ENTRY(m_GCSIPHandle, "Sending SIP OK\n");

   GC_PARM_BLKP l_parmBlock = 0;
   GC_PARM_BLKP l_returnParmBlock = 0;

   if ( gc_util_insert_parm_val( &l_parmBlock,
                                 IPSET_MSG_SIP,
                                 IPPARM_MSGTYPE,
                                 sizeof(int) ,
                                 IP_MSGTYPE_SIP_INFO_OK) != GC_SUCCESS )
   {
      LOG_ERROR(m_GCSIPHandle, "gc_util_insert_parm_val() failed\n");
      return;
   }
   // Handle sending this also? IP_MSGTYPE_SIP_INFO_FAILED

   if ( gc_SetUserInfo(  GCTGT_GCLIB_CRN, m_curCallCRN, l_parmBlock, GC_SINGLECALL) != GC_SUCCESS )
   {
      LOG_ERROR(m_GCSIPHandle,  "gc_SetUserInfo() failed\n");
      return;
   }

   if ( gc_Extension(   GCTGT_GCLIB_CRN,
                        m_curCallCRN,
                        IPEXTID_SENDMSG,
                        l_parmBlock,
                        &l_returnParmBlock,
                        EV_ASYNC) != GC_SUCCESS )
   {
      LOG_ERROR(m_GCSIPHandle, "gc_Extension() failed\n");
      return;
   }
   gc_util_delete_parm_blk( l_parmBlock);

}

//*****************************************************************************
// Function: void SIPEndpoint::SetLastMessageString(const char *pMessageStr)
// Description: Save the last message string
// Return: void 
// Parameters: char *pMessageStr 
//*****************************************************************************
void SIPEndpoint::SetLastMessageString(const char *pMessageStr)
{
   int length = (MAX_LAST_MSG_STRING > strlen(pMessageStr)) ? strlen(pMessageStr) : MAX_LAST_MSG_STRING-1;
   memset(m_LastMessageBuff, 0, sizeof(char)*MAX_LAST_MSG_STRING);
   strncpy(&m_LastMessageBuff[0], pMessageStr, length);
   m_LastMessageBuff[length] = '\0';
}

//*****************************************************************************
// Function: char* SIPEndpoint::GetStateString()
// Description: Format and return a status string
// Return: char* 
// Parameters: none 
//*****************************************************************************
char * SIPEndpoint::GetStateString()
{
   memset(m_StateBuffer, 0, sizeof(char)*MAX_STATE_STRING);
//   cout << "EP:    STATE:    AUDIO:         Tx    VIDEO:        Tx        MISC." << endl;
//   cout << "                 TxLCN: RxLCN:  CAP:  TxLCN: RxLCN: CAP:    CMD/IND:" << endl;
   sprintf(m_StateBuffer, "SIP%2d: %15s    %3s    %3s  %5s%1s   %3s    %3s  %5s%1s     %6s",
           m_Index,
           m_CurrentStateStr,
           "--", //GetAudioTxLCN(),
           "--", //GetAudioRxLCN(),
           GetSelAudCoderStr(),
           (GetAudTranscodeEnabled()?"*":" "),
           "--", //GetVideoTxLCN(),
           "--", //GetVideoRxLCN(),
           GetSelVidCoderStr(),
           (GetVidTranscodeEnabled()?"*":" "),
           m_LastMessageBuff);
   return m_StateBuffer;
}


//*****************************************************************************
// Function: void SIPEndpoint::DropCall(METAEVENT metaevent)
// Description: Drop the call
// Return: void 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void SIPEndpoint::DropCall(METAEVENT metaevent)
{
   // Terminate RTP streaming
   StopMedia();
   ProcessDisconnect(metaevent);
}

//*****************************************************************************
// Function: void SIPEndpoint::SendSipInfoDtmf(char dtmf)
// Description: Send DTMF digit as SIP INFO message
// Return: void 
// Parameters: char dtmf 
//*****************************************************************************
void SIPEndpoint::SendSipInfoDtmf(char dtmf)
{
   // Local Variable Declaration
   GC_PARM_BLKP gcParmBlk_mime= 0;
   GC_PARM_BLKP gcParmBlk_mime1 = 0;
   GC_PARM_BLKP gcParmBlk_info= 0;
   bool bOk= true;
   const char* pBodyType = "Content-Type:application/dtmf-relay"; // specify the body type
   char infoMsgBody[128];

   LOG_ENTRY(m_GCSIPHandle, "SendSipInfoDtmf(%d)\n", dtmf);
   if ( gc_util_insert_parm_ref(&gcParmBlk_mime,
                                IPSET_MIME,
                                IPPARM_MIME_PART_TYPE,
                                (unsigned char)(strlen(pBodyType) + 1),
                                (void*)pBodyType) < 0 )
   {
      LOG_ERROR(m_GCSIPHandle, 
                "SendSipInfoDtmf() -> gc_util_insert_parm_ref() failed for IPPARM_MIME_PART_TYPE\n");
      bOk = false;
   }

   sprintf(infoMsgBody, "Signal=%c\r\nDuration=160", dtmf);

   // insert the body size
   if ( gc_util_insert_parm_val(&gcParmBlk_mime,
                                IPSET_MIME,
                                IPPARM_MIME_PART_BODY_SIZE,
                                sizeof(unsigned long),
                                strlen(infoMsgBody)) < 0 )
   {
      LOG_ERROR(m_GCSIPHandle, 
                "SendSipInfoDtmf() -> gc_util_insert_parm_val() failed for IPPARM_MIME_PART_BODY_SIZE\n");
      bOk = false;
   }

   // insert the body
   if ( gc_util_insert_parm_val(&gcParmBlk_mime,
                                IPSET_MIME,
                                IPPARM_MIME_PART_BODY,
                                sizeof(unsigned long),
                                (unsigned long)(infoMsgBody)) < 0 )
   {
      LOG_ERROR(m_GCSIPHandle, 
                "SendSipInfoDtmf() -> gc_util_insert_parm_val() failed for IPPARM_MIME_PART_BODY\n");
      bOk = false;
   }

   // insert the list of parmBlks into the top level parmBlk
   if ( gc_util_insert_parm_val(&gcParmBlk_mime1,
                                IPSET_MIME,
                                IPPARM_MIME_PART,
                                sizeof(unsigned long),
                                (unsigned long)gcParmBlk_mime) < 0 )
   {
      LOG_ERROR(m_GCSIPHandle, 
                "SendSipInfoDtmf() -> gc_util_insert_parm_val() failed for IPPARM_MIME_PART\n");
      bOk = false;
   }

   // now set it on the device
   if ( gc_SetUserInfo(GCTGT_GCLIB_CRN,
                       m_curCallCRN,
                       gcParmBlk_mime1,
                       GC_SINGLECALL) < 0 ) // for this call only
   {
      LOG_ERROR(m_GCSIPHandle, 
                "gc_SetUserInfo() failed for MIME body in INFO\n");
      bOk = false;
   }

   // insert the message type
   if ( gc_util_insert_parm_val(&gcParmBlk_info,
                                IPSET_MSG_SIP,
                                IPPARM_MSGTYPE,
                                sizeof(int),
                                IP_MSGTYPE_SIP_INFO) < 0 )
   {
      LOG_ERROR(m_GCSIPHandle, 
                "SendSipInfoDtmf() -> gc_util_insert_parm_val() failed for SIP INFO\n");
      bOk = false;
   }

   if ( gc_Extension(GCTGT_GCLIB_CRN, m_curCallCRN, IPEXTID_SENDMSG, 
                     gcParmBlk_info, NULL, EV_ASYNC) < 0 )
   {
      LOG_ERROR(m_GCSIPHandle, 
                "SendSipInfoDtmf() -> gc_Extension failed\n");
      bOk = false;
   }
   gc_util_delete_parm_blk(gcParmBlk_info);
   gc_util_delete_parm_blk(gcParmBlk_mime);
   gc_util_delete_parm_blk(gcParmBlk_mime1);
}


//*****************************************************************************
// Function: void SIPEndpoint::isSIPInboundDevice()
// Description: Return true if ithis endpoint is
//                an even numbered (iptB1Tx) device; false otherwise 
// Return: boolean 
// Parameters: none 
//*****************************************************************************
bool SIPEndpoint::isSIPInboundDevice() 
{
  bool retVal = atoi(m_SIPName + 9) % 2;
  if (retVal)
    LOG_ENTRY (m_GCSIPHandle, "is an inbound device\n");
  else
    LOG_ENTRY (m_GCSIPHandle, "is an outbound device\n");
  return retVal;
}

//*****************************************************************************
// Function: void SIPEndpoint::SetVideoToH263()
// Description: Set the video configuration to H.263
// Return: void
// Parameters: none
//*****************************************************************************
void SIPEndpoint::SetVideoToH263(bool forceQCIF)
{
 
   LOG_ENTRY (m_GCSIPHandle, "SIPEndpoint - SetVideoToH263(pt=%d)\n",m_nH263_PT);
   SetSelectedVideoCoder(VID_CODER_H263);
   sprintf(m_sdpVideoFormat, "%d", m_nH263_PT );
   sprintf(m_sdpVideoRtpmap, "%d H263/90000", m_nH263_PT);
   
   // When transcoding is on and inbound call will always offer CIF/QCIF both and select one based on answer SDP
   if(GetVidTranscodeEnabled() && !forceQCIF) {
     sprintf(m_sdpVideoFmtp, "%d CIF=2 QCIF=2", m_nH263_PT);
   }
   // else will only offer QCIF
   else {
      sprintf(m_sdpVideoFmtp, "%d QCIF=2", m_nH263_PT);
   }

   //Local coder ipmedia values
   m_localVideoCoderType = CODER_TYPE_H263;
   m_localVideoCoderPayloadType = m_nH263_PT;
   m_localImageProfile = VIDEO_PROFILE_0_H263;
   m_localImageLevel = VIDEO_LEVEL_10_H263;

   m_localImageWidth = VIDEO_IMAGE_WIDTH_176;
   m_localImageHeight = VIDEO_IMAGE_HEIGHT_144;
   m_localFramesPerSec = VIDEO_FRAMESPERSEC_15;
   m_localBitRate = 42000;
   m_localSamplingRate = VIDEO_SAMPLING_RATE_DEFAULT;
   
   // Remote coder ipmedia values
   m_remoteVideoCoderType = CODER_TYPE_H263;
   m_remoteVideoCoderPayloadType = m_nH263_PT;
   m_remoteImageProfile = VIDEO_PROFILE_0_H263;
   m_remoteImageLevel = VIDEO_LEVEL_10_H263;
   
   m_remoteImageWidth = VIDEO_IMAGE_WIDTH_176;
   m_remoteImageHeight = VIDEO_IMAGE_HEIGHT_144;
   m_remoteFramesPerSec = VIDEO_FRAMESPERSEC_15;
   m_remoteBitRate = 42000;
   m_remoteSamplingRate = VIDEO_SAMPLING_RATE_DEFAULT;
}


//*****************************************************************************
// Function: void SIPEndpoint::UpdateH263Attributes(char videosize[5], int bitRate, int fps)
// Description: Update the H263 values based on the SDP negotiation
// Return: void
// Parameters: char videoSize[5]
//             int bitRate
//             int fp 
//*****************************************************************************
void SIPEndpoint::UpdateH263Attributes(char videoSize[5],int bitRate,int fps)
{
   LOG_ENTRY (m_GCSIPHandle, "SIPEndpoint::UpdateH263Attributes(%s,%d,%d)\n",videoSize,bitRate,fps);
   switch(fps)
   {
     case 6: 
       m_localFramesPerSec=VIDEO_FRAMESPERSEC_6;
       m_remoteFramesPerSec = VIDEO_FRAMESPERSEC_6;
       break;
     case 10:
       m_localFramesPerSec=VIDEO_FRAMESPERSEC_10;
       m_remoteFramesPerSec = VIDEO_FRAMESPERSEC_10;
       break;
     case 15:
       m_localFramesPerSec=VIDEO_FRAMESPERSEC_15;
       m_remoteFramesPerSec = VIDEO_FRAMESPERSEC_15;
       break;
     case 30:
       m_localFramesPerSec=VIDEO_FRAMESPERSEC_30;
       m_remoteFramesPerSec = VIDEO_FRAMESPERSEC_30;
       break;
     default:
       LOG_ERROR(m_GCSIPHandle,"UpdateH263Attributes(%s,%d,%d) Invalid FPS\n",videoSize,bitRate,fps);
   }
   if(bitRate==0)
   {	
     if(strcmp(videoSize,"CIF")==0) {
			 bitRate=384000;
     }
     else {
			 bitRate=64000;
     }
   }
   //QCIF: 176(w) x 144(h)
   //CIF: 352(w) x 288(h)
   //SQCIF: 128(w) x 96(h)
   if(strcmp(videoSize,"SQCIF")==0) {
     m_localImageWidth = VIDEO_IMAGE_WIDTH_128;
     m_localImageHeight = VIDEO_IMAGE_HEIGHT_96;
     m_localBitRate = bitRate;
     m_remoteImageWidth = VIDEO_IMAGE_WIDTH_128;
     m_remoteImageHeight = VIDEO_IMAGE_HEIGHT_96;
     m_remoteBitRate = bitRate;
   }
   else if(strcmp(videoSize,"QCIF")==0) {
     m_localImageWidth = VIDEO_IMAGE_WIDTH_176;
     m_localImageHeight = VIDEO_IMAGE_HEIGHT_144;
     m_localBitRate = bitRate;
   
     m_remoteImageWidth = VIDEO_IMAGE_WIDTH_176;
     m_remoteImageHeight = VIDEO_IMAGE_HEIGHT_144;
     m_remoteBitRate = bitRate;
   }
   else if(strcmp(videoSize,"CIF")==0) { 
      m_localImageWidth = VIDEO_IMAGE_WIDTH_352;
      m_localImageHeight = VIDEO_IMAGE_HEIGHT_288;
      m_localBitRate = bitRate;

      m_remoteImageWidth = VIDEO_IMAGE_WIDTH_352;
      m_remoteImageHeight = VIDEO_IMAGE_HEIGHT_288;
      m_remoteBitRate = bitRate;
   }
	if(m_remoteBitRate>128000) {
		m_localImageLevel = VIDEO_LEVEL_30_H263;
		m_remoteImageLevel = VIDEO_LEVEL_30_H263;	
	}
	else if(m_remoteBitRate>64000) {
		m_localImageLevel = VIDEO_LEVEL_20_H263;
		m_remoteImageLevel = VIDEO_LEVEL_20_H263;	
	}
	else {
		m_localImageLevel = VIDEO_LEVEL_10_H263;
		m_remoteImageLevel = VIDEO_LEVEL_10_H263;	
  }

}

//*****************************************************************************
// Function: void SIPEndpoint::SetVideoToMPEG4()
// Description: Set the video configuration to MPEG4
// Return: void
// Parameters: none
//*****************************************************************************
void SIPEndpoint::SetVideoToMPEG4()
{

   LOG_ENTRY (m_GCSIPHandle, "SIPEndpoint - SetVideoToMPEG4(pt=%d)\n", m_nMPEG4_PT);
   LOG_ENTRY (m_GCSIPHandle, "SIPEndpoint - DCI=%s\n", Get3GEPDCI());
   SetSelectedVideoCoder(VID_CODER_MPEG4);
   sprintf(m_sdpVideoFormat, "%d", m_nMPEG4_PT);
   sprintf(m_sdpVideoRtpmap, "%d mp4v-es/90000", m_nMPEG4_PT);
   sprintf(m_sdpVideoFmtp, "%d profile-level-id=0;config=%s", m_nMPEG4_PT,Get3GEPDCI());

   //Local coder ipmedia values
   m_localVideoCoderType = CODER_TYPE_MP4V_ES;
   m_localVideoCoderPayloadType = m_nMPEG4_PT;
   m_localImageProfile = VIDEO_PROFILE_LEVEL_SP0_MPEG4;
   m_localImageLevel = VIDEO_LEVEL_DEFAULT;

   m_localImageWidth = VIDEO_IMAGE_WIDTH_176;
   m_localImageHeight = VIDEO_IMAGE_HEIGHT_144;
   m_localFramesPerSec = VIDEO_FRAMESPERSEC_15;
   m_localBitRate = 42000;
   m_localSamplingRate = VIDEO_SAMPLING_RATE_DEFAULT;
   
   //Remote Coder ipmedia values
   m_remoteVideoCoderType = CODER_TYPE_MP4V_ES;
   m_remoteVideoCoderPayloadType = m_nMPEG4_PT;
   m_remoteImageProfile = VIDEO_PROFILE_LEVEL_SP0_MPEG4;
   m_remoteImageLevel = VIDEO_LEVEL_DEFAULT;
   
   m_remoteImageWidth = VIDEO_IMAGE_WIDTH_176;
   m_remoteImageHeight = VIDEO_IMAGE_HEIGHT_144;
   m_remoteFramesPerSec = VIDEO_FRAMESPERSEC_15;
   m_remoteBitRate = 42000;
   m_remoteSamplingRate = VIDEO_SAMPLING_RATE_DEFAULT;
}

//*****************************************************************************
// Function: void SIPEndpoint::UpdateMPEG4Attributes(int profileAndLevel,int bitRate)
// Description: Update the MPEG4 values based on the SDP negotiation
// Return: void
// Parameters: int profileAndLevel
//             int bitRate
//*****************************************************************************
void SIPEndpoint::UpdateMpeg4Attributes(int profileAndLevel, int bitRate, char dciValue[MAX_STATE_STRING])
{
  LOG_ENTRY (m_GCSIPHandle, "SIPEndpoint::UpdateMpeg4Attributes(%d,%d)\n",profileAndLevel,bitRate);
  LOG_ENTRY (m_GCSIPHandle, "SIPEndpoint::UpdateMpeg4Attributes(config=%s)\n",dciValue);
  if(bitRate==0)
  {
    if(profileAndLevel==3) {
			bitRate=384000;
    }
    else if(profileAndLevel==2) {
			bitRate=128000;
    }
    else {
			bitRate=64000;
    }
  }
  //QCIF: 176(w) x 144(h)
  //CIF: 352(w) x 288(h)
  //SQCIF: 128(w) x 96(h)
  if(profileAndLevel==0 || profileAndLevel==1) {
    m_localImageWidth = VIDEO_IMAGE_WIDTH_176;
    m_localImageHeight = VIDEO_IMAGE_HEIGHT_144;
    m_localFramesPerSec=VIDEO_FRAMESPERSEC_15;
    m_localBitRate = bitRate;

    m_remoteImageWidth = VIDEO_IMAGE_WIDTH_176;
    m_remoteImageHeight = VIDEO_IMAGE_HEIGHT_144;
    m_remoteFramesPerSec = VIDEO_FRAMESPERSEC_15;
    m_remoteBitRate = bitRate;
  }
  else if(profileAndLevel==2 || profileAndLevel==3) {
    m_localImageWidth = VIDEO_IMAGE_WIDTH_352;
    m_localImageHeight = VIDEO_IMAGE_HEIGHT_288;
    m_localFramesPerSec=VIDEO_FRAMESPERSEC_30;
    m_localBitRate = bitRate;

    m_remoteImageWidth = VIDEO_IMAGE_WIDTH_352;
    m_remoteImageHeight = VIDEO_IMAGE_HEIGHT_288;
    m_remoteFramesPerSec = VIDEO_FRAMESPERSEC_30;
    m_remoteBitRate = bitRate;
  }
// SKS need attention here
   m_dciOctetStrSize=(char)(strlen(dciValue)/2);
   for (unsigned int i = 0; i <m_dciOctetStrSize; i++) {
            m_dciOctetStr[i] = AsciiOctetToHex(&dciValue[2*i]);
  }
}



//*****************************************************************************
// Function: void SIPEndpoint::SetVideoToH264()
// Description: Set the video configuration to H264
// Return: void
// Parameters: none
//*****************************************************************************
void SIPEndpoint::SetVideoToH264()
{

   LOG_ENTRY (m_GCSIPHandle, "SIPEndpoint - SetVideoToH264(pt=%d)\n", m_nH264_PT);
   //LOG_ENTRY (m_GCSIPHandle, "SIPEndpoint - DCI=%s\n", Get3GEPDCI());
   SetSelectedVideoCoder(VID_CODER_H264);
   sprintf(m_sdpVideoFormat, "%d", m_nH264_PT);
   sprintf(m_sdpVideoRtpmap, "%d h264/90000", m_nH264_PT);

   //  not adviced to pass H264 DCI in SDP
   //  sprintf(m_sdpVideoFmtp, "%d profile-level-id=42000A;config=%s", m_nH264_PT,Get3GEPDCI());

   // Don't specify max-br if we transcode
   if(GetVidTranscodeEnabled())
      sprintf(m_sdpVideoFmtp, "%d profile-level-id=42000C; packetization-mode=1", m_nH264_PT);   // can receive up to Baseline, Level 1.2
   else
      sprintf(m_sdpVideoFmtp, "%d profile-level-id=42000A; packetization-mode=1; max-br=42", m_nH264_PT);  // Baseline, Level 1

   //Local coder ipmedia values
   m_localVideoCoderType = CODER_TYPE_H264;
   m_localVideoCoderPayloadType = m_nH264_PT;
   m_localImageProfile = VIDEO_PROFILE_BASELINE_H264;
   m_localImageLevel = VIDEO_LEVEL_1_H264;

   m_localImageWidth = VIDEO_IMAGE_WIDTH_176;
   m_localImageHeight = VIDEO_IMAGE_HEIGHT_144;
   m_localFramesPerSec = VIDEO_FRAMESPERSEC_15;
   if(GetVidTranscodeEnabled())
      m_localBitRate = 384000;
   else
      m_localBitRate = 42000;
   m_localSamplingRate = VIDEO_SAMPLING_RATE_DEFAULT;
   
   //Remote Coder ipmedia values
   m_remoteVideoCoderType = CODER_TYPE_H264;
   m_remoteVideoCoderPayloadType = m_nH264_PT;
   m_remoteImageProfile = VIDEO_PROFILE_BASELINE_H264;
   m_remoteImageLevel = VIDEO_LEVEL_1_H264;
   
   m_remoteImageWidth = VIDEO_IMAGE_WIDTH_176;
   m_remoteImageHeight = VIDEO_IMAGE_HEIGHT_144;
   m_remoteFramesPerSec = VIDEO_FRAMESPERSEC_15;
   m_remoteBitRate = 42000;
   m_remoteSamplingRate = VIDEO_SAMPLING_RATE_DEFAULT;
}

//*****************************************************************************
// Function: void SIPEndpoint::UpdateH264Attributes(int profileAndLevel,int bitRate)
// Description: Update the H264 values based on the SDP negotiation
// Return: void
// Parameters: int profileAndLevel
//             int bitRate
//*****************************************************************************
void SIPEndpoint::UpdateH264Attributes(int profileAndLevel, int bitRate, char dciValue[MAX_STATE_STRING])
{
  LOG_ENTRY (m_GCSIPHandle, "SIPEndpoint::UpdateH264Attributes(%d,%d)\n",profileAndLevel,bitRate);
  LOG_ENTRY (m_GCSIPHandle, "SIPEndpoint::UpdateH264Attributes(config=%s)\n",dciValue);

  int level = profileAndLevel & 0xFF;
  LOG_ENTRY (m_GCSIPHandle, "SIPEndpoint::UpdateH264Attributes(Level = %d)\n",level);

   if(GetVidTranscodeEnabled())
      m_localBitRate = 384000;
   else
      m_localBitRate = 42000;

  if(bitRate==0)  // no bitrate constraint specified by the remote
  {
    if(level >= 12) {
			m_remoteBitRate=384000;
    }
    else if(level == 11) {
			m_remoteBitRate=192000;
    }
    else {
			m_remoteBitRate=64000;
    }
  }
  else
     m_remoteBitRate = bitRate;

  //QCIF: 176(w) x 144(h)
  //CIF: 352(w) x 288(h)
  //SQCIF: 128(w) x 96(h)
  if(level >= 13) {
    LOG_ENTRY (m_GCSIPHandle, "SIPEndpoint::UpdateH264Attributes Set local and Remote to CIF resolution, Level 1.3\n");
    m_localImageWidth = VIDEO_IMAGE_WIDTH_352;
    m_localImageHeight = VIDEO_IMAGE_HEIGHT_288;
    m_localFramesPerSec=VIDEO_FRAMESPERSEC_30;
    m_localImageLevel = VIDEO_LEVEL_1_3_H264;

    m_remoteImageWidth = VIDEO_IMAGE_WIDTH_352;
    m_remoteImageHeight = VIDEO_IMAGE_HEIGHT_288;
    m_remoteFramesPerSec = VIDEO_FRAMESPERSEC_30;
    m_remoteImageLevel = VIDEO_LEVEL_1_3_H264;
  }
  else  if (level == 12) {
    LOG_ENTRY (m_GCSIPHandle, "SIPEndpoint::UpdateH264Attributes Set local and Remote to CIF resolution, Level 1.2\n");
    m_localImageWidth = VIDEO_IMAGE_WIDTH_352;
    m_localImageHeight = VIDEO_IMAGE_HEIGHT_288;
    m_localFramesPerSec=VIDEO_FRAMESPERSEC_15;
    m_localImageLevel = VIDEO_LEVEL_1_2_H264;

    m_remoteImageWidth = VIDEO_IMAGE_WIDTH_352;
    m_remoteImageHeight = VIDEO_IMAGE_HEIGHT_144;
    m_remoteFramesPerSec = VIDEO_FRAMESPERSEC_15;
    m_remoteImageLevel = VIDEO_LEVEL_1_2_H264;
  }
  else  if (level == 11){
    LOG_ENTRY (m_GCSIPHandle, "SIPEndpoint::UpdateH264Attributes Set local and Remote to QCIF resolution, Level 1.1\n");
    m_localImageWidth = VIDEO_IMAGE_WIDTH_176;
    m_localImageHeight = VIDEO_IMAGE_HEIGHT_144;
    m_localFramesPerSec=VIDEO_FRAMESPERSEC_30;
    m_localImageLevel = VIDEO_LEVEL_1_1_H264;

    m_remoteImageWidth = VIDEO_IMAGE_WIDTH_176;
    m_remoteImageHeight = VIDEO_IMAGE_HEIGHT_144;
    m_remoteFramesPerSec = VIDEO_FRAMESPERSEC_30;
    m_remoteImageLevel = VIDEO_LEVEL_1_1_H264;
  }
  else  {
    LOG_ENTRY (m_GCSIPHandle, "SIPEndpoint::UpdateH264Attributes Set local and Remote to QCIF resolution\n");
    m_localImageWidth = VIDEO_IMAGE_WIDTH_176;
    m_localImageHeight = VIDEO_IMAGE_HEIGHT_144;
    m_localFramesPerSec=VIDEO_FRAMESPERSEC_15;
    m_localImageLevel = VIDEO_LEVEL_1_H264;

    m_remoteImageWidth = VIDEO_IMAGE_WIDTH_176;
    m_remoteImageHeight = VIDEO_IMAGE_HEIGHT_144;
    m_remoteFramesPerSec = VIDEO_FRAMESPERSEC_15;
    m_remoteImageLevel = VIDEO_LEVEL_1_H264;
  }


}

//*****************************************************************************
// Function: void SIPEndpoint::SetAudioToAMR()
// Description: Set the audio configuration to AMR-NB
// Return: void
// Parameters: none
//*****************************************************************************
void SIPEndpoint::SetAudioToAMR()
{ 

   LOG_ENTRY (m_GCSIPHandle, "SIPEndpoint - SetAudioToAMR(pt=%d)\n", m_nAMR_PT);
   SetSelectedAudioCoder(AUD_CODER_AMR);
   sprintf(m_sdpAudioFormat, "%d", m_nAMR_PT);
   sprintf(m_sdpAudioRtpmap, "%d AMR/8000/1", m_nAMR_PT);
   sprintf(m_sdpAudioFmtp, "%d octet-align=1", m_nAMR_PT);

   if(GetAudTranscodeEnabled())
   {
      m_remoteAudioCoderType = CODER_TYPE_AMRNB_12_2k;
      m_localAudioCoderType = CODER_TYPE_AMRNB_12_2k;
   }
   else
   {
      m_remoteAudioCoderType = CODER_TYPE_AMRNB_12_2k_NATIVE;
      m_localAudioCoderType = CODER_TYPE_AMRNB_12_2k_NATIVE;
   }
   m_remoteAudioCoderFramesize = CODER_FRAMESIZE_20;
   m_remoteAudioFramesPerPkt = 1;
   m_remoteAudioVadEnable = CODER_VAD_ENABLE;
   m_remoteAudioCoderPayloadType = m_nAMR_PT;
   m_remoteAudioRedPayloadType = 97;

   m_localAudioCoderFramesize = CODER_FRAMESIZE_20;
   m_localAudioFramesPerPkt = 1;
   m_localAudioVadEnable = CODER_VAD_ENABLE;
   m_localAudioCoderPayloadType = m_nAMR_PT;
   m_localAudioRedPayloadType = 97;

   m_ipmAudioCoderResource = RESOURCE_IPM_AMR_NB;
}

//*****************************************************************************
// Function: void SIPEndpoint::SetAudioToG723()
// Description: Set the audio configuration to G.723
// Return: void
// Parameters: none
//*****************************************************************************
void SIPEndpoint::SetAudioToG723()
{
   LOG_ENTRY (m_GCSIPHandle, "SIPEndpoint - SetAudioToG723()\n");
   SetSelectedAudioCoder(AUD_CODER_G723);
   strcpy(m_sdpAudioFormat, "4");
   strcpy(m_sdpAudioRtpmap, "4 G723/8000");
   strcpy(m_sdpAudioFmtp, "\0");
  
   if(GetAudTranscodeEnabled())
   {
      m_remoteAudioCoderType = CODER_TYPE_G7231_5_3K;
      m_localAudioCoderType = CODER_TYPE_G7231_5_3K;
      //m_remoteAudioCoderType = CODER_TYPE_G7231_6_3K;
      //m_localAudioCoderType = CODER_TYPE_G7231_6_3K;
   }
   else
   {
      m_remoteAudioCoderType = CODER_TYPE_G7231_5_3K_NATIVE;
      m_localAudioCoderType = CODER_TYPE_G7231_5_3K_NATIVE;
      //m_remoteAudioCoderType = CODER_TYPE_G7231_6_3K_NATIVE;
      //m_localAudioCoderType = CODER_TYPE_G7231_6_3K_NATIVE;
   }

   m_remoteAudioCoderFramesize = CODER_FRAMESIZE_30;
   m_remoteAudioFramesPerPkt = 1;
   m_remoteAudioVadEnable = CODER_VAD_ENABLE;
   m_remoteAudioCoderPayloadType = 4;
   m_remoteAudioRedPayloadType = 4;

   m_localAudioCoderFramesize = CODER_FRAMESIZE_30;
   m_localAudioFramesPerPkt = 1;
   m_localAudioVadEnable = CODER_VAD_ENABLE;
   m_localAudioCoderPayloadType = 4;
   m_localAudioRedPayloadType = 4;

   m_ipmAudioCoderResource = RESOURCE_IPM_G723;
}

//*****************************************************************************
// Function: void SIPEndpoint::SetAudioToG729()
// Description: Set the audio configuration to G.723
// Return: void
// Parameters: none
//*****************************************************************************
void SIPEndpoint::SetAudioToG729()
{
   LOG_ENTRY (m_GCSIPHandle, "SIPEndpoint - SetAudioToG729()\n");
   SetSelectedAudioCoder(AUD_CODER_G729);
   strcpy(m_sdpAudioFormat, "18");
   strcpy(m_sdpAudioRtpmap, "18 g729/8000");
   strcpy(m_sdpAudioFmtp, "\0");
  
   if(GetAudTranscodeEnabled())
   {
      m_remoteAudioCoderType = CODER_TYPE_G729;
      m_localAudioCoderType = CODER_TYPE_G729;
      //m_remoteAudioCoderType = CODER_TYPE_G729;
      //m_localAudioCoderType = CODER_TYPE_G729;
   }
   else
   {
      m_remoteAudioCoderType = CODER_TYPE_G729_NATIVE;
      m_localAudioCoderType = CODER_TYPE_G729_NATIVE;
      //m_remoteAudioCoderType = CODER_TYPE_G729_NATIVE;
      //m_localAudioCoderType = CODER_TYPE_G729_NATIVE;
   }

   m_remoteAudioCoderFramesize = CODER_FRAMESIZE_10;
   m_remoteAudioFramesPerPkt = 2;
   m_remoteAudioVadEnable = CODER_VAD_ENABLE;
   m_remoteAudioCoderPayloadType = 18;
   m_remoteAudioRedPayloadType = 18;

   m_localAudioCoderFramesize = CODER_FRAMESIZE_10;
   m_localAudioFramesPerPkt = 2;
   m_localAudioVadEnable = CODER_VAD_ENABLE;
   m_localAudioCoderPayloadType = 18;
   m_localAudioRedPayloadType = 18;

   m_ipmAudioCoderResource = RESOURCE_IPM_G729;
}

//*****************************************************************************
// Function: void SIPEndpoint::SetAudioToG711U()
// Description: Set the audio configuration to G.711ULaw
// Return: void
// Parameters: none
//*****************************************************************************
void SIPEndpoint::SetAudioToG711U()
{
   LOG_ENTRY (m_GCSIPHandle, "SIPEndpoint - SetAudioToG711U()\n");
   SetSelectedAudioCoder(AUD_CODER_G711U);
   strcpy(m_sdpAudioFormat, "0");
   strcpy(m_sdpAudioRtpmap, "0 PCMU/8000");
   strcpy(m_sdpAudioFmtp, "\0");

   m_remoteAudioCoderType = CODER_TYPE_G711ULAW64K;
   m_remoteAudioCoderFramesize = CODER_FRAMESIZE_20;
   m_remoteAudioFramesPerPkt = 1;
   m_remoteAudioVadEnable = CODER_VAD_ENABLE;
   m_remoteAudioCoderPayloadType = 0;
   m_remoteAudioRedPayloadType = 0;

   m_localAudioCoderType = CODER_TYPE_G711ULAW64K;
   m_localAudioCoderFramesize = CODER_FRAMESIZE_20;
   m_localAudioFramesPerPkt = 1;
   m_localAudioVadEnable = CODER_VAD_ENABLE;
   m_localAudioCoderPayloadType = 0;
   m_localAudioRedPayloadType = 0;

   m_ipmAudioCoderResource = RESOURCE_IPM_G711_20MS;
}

//*****************************************************************************
// Function: void SIPEndpoint::SetAudioToG711A()
// Description: Set the audio configuration to G.711ALaw
// Return: void
// Parameters: none
//*****************************************************************************
void SIPEndpoint::SetAudioToG711A()
{
   LOG_ENTRY (m_GCSIPHandle, "SIPEndpoint - SetAudioToG711A()\n");
   SetSelectedAudioCoder(AUD_CODER_G711A);
   strcpy(m_sdpAudioFormat, "8");
   strcpy(m_sdpAudioRtpmap, "8 PCMA/8000");
   strcpy(m_sdpAudioFmtp, "\0");

   m_remoteAudioCoderType = CODER_TYPE_G711ALAW64K;
   m_remoteAudioCoderFramesize = CODER_FRAMESIZE_20;
   m_remoteAudioFramesPerPkt = 1;
   m_remoteAudioVadEnable = CODER_VAD_ENABLE;
   m_remoteAudioCoderPayloadType = 8;
   m_remoteAudioRedPayloadType = 8;

   m_localAudioCoderType = CODER_TYPE_G711ALAW64K;
   m_localAudioCoderFramesize = CODER_FRAMESIZE_20;
   m_localAudioFramesPerPkt = 1;
   m_localAudioVadEnable = CODER_VAD_ENABLE;
   m_localAudioCoderPayloadType = 8;
   m_localAudioRedPayloadType = 8;

   m_ipmAudioCoderResource = RESOURCE_IPM_G711_20MS;
}



//*****************************************************************************
// Function: void SIPEndpoint::SetupSipCoders()
// Description: Set the Audio and Video Codecs
// Return: void
// Parameters: none
//*****************************************************************************
void SIPEndpoint::SetupSipCoders()
{
   E_SEL_VID_CODER SelVidCoder = GetSelectedVideoCoder();
   E_SEL_AUD_CODER SelAudCoder = GetSelectedAudioCoder();
 
   LOG_ENTRY(m_GCSIPHandle, "SelVidCoder=%s[%d], SelAudCoder=%s[%d]\n", 
              GetSelVidCoderStr(), GetSelectedVideoCoder(), 
              GetSelAudCoderStr(), GetSelectedAudioCoder());

   //Set Video Coder
   if (SelVidCoder == VID_CODER_MPEG4)
      SetVideoToMPEG4();
   else if (SelVidCoder == VID_CODER_H264)
      SetVideoToH264();
   else
      SetVideoToH263(true);

   //Set Audio Coder
   switch(SelAudCoder)
   {
      case AUD_CODER_G723:
         SetAudioToG723();
         break;
      case AUD_CODER_G729:
         SetAudioToG729();
         break;
      case AUD_CODER_G711U:
         SetAudioToG711U();
         break;
      case AUD_CODER_G711A:
         SetAudioToG711A();
         break;
      case AUD_CODER_AMR:
      default:
         SetAudioToAMR();
         break;
   }

}   


//*****************************************************************************
// Function: void SIPEndpoint::ParseSdp(sdpSessionDescription& sdp, char* dataBuff)
// Description: Process Offer SDP from remote endpoint
// Return: void
// Parameters: none
//*****************************************************************************
void SIPEndpoint::ParseSdp(sdpSessionDescription& sdp, char* dataBuff)
{   
  int NumMD;
  int i,j,k,nAttributes;
  int  pt;
  char szAMR[10];
  char szTelephone[10];
  char szH263[10];
  char szH263_1998[10];
  char szH263_2000[10];
  char szMPEG4[10];
  char szH264[10];

  sprintf(szAMR,"%d",m_nAMR_PT);
  sprintf(szH263,"%d",m_nH263_PT);
  sprintf(szH263_1998,"105");
  sprintf(szH263_2000,"105");
  sprintf(szMPEG4,"%d",m_nMPEG4_PT);
  sprintf(szH264,"%d",m_nH264_PT);

  LOG_ENTRY(m_GCSIPHandle, "ParseSdp():\n");
  // Import it into the SDP library
  sdp.clear();
  // 2nd param is for debugging (true/false) 
  // The SDP is sent to stdout by the sdpapi
  sdp.importSDP(dataBuff, false);

  NumMD = sdp.mediaDescriptionList()->numItem();
  LOG_ENTRY(m_GCSIPHandle, "SDP contains %d media descriptor(s)\n",NumMD);

  for ( i=0;i<NumMD;i++ )
  {
    sdpMediaDescription* pMD = sdp.mediaDescriptionList()->getItem(i);
    sdpMedia* pMedia = pMD->media();
    //sdpOrigin* pOrigin = sdp.origin();

    // search through the coder list for dynamic codecs
    sdpAttributeList* pAttributeList = pMD->attributeList();
    int NumAttr = pAttributeList->numItem();
    for (k=0;k<NumAttr;k++)
    {
      sdpAttribute* pAttribute = pAttributeList->getItem(k);
      if ( (!strcmp(pAttribute->getProperty(), "rtpmap")) )
      {
        const char* rtpmapVal = pAttribute->getPropertyValue();
        LOG_ENTRY(m_GCSIPHandle, "Attribute rtpmap = %s\n",rtpmapVal);
        char codec_name[50];
        sscanf(rtpmapVal, "%d %s", &pt, codec_name);
        if (strncasecmp(codec_name, "amr", 3) == 0) {
          LOG_ENTRY(m_GCSIPHandle, "  Codec is AMR\n");
          sprintf(szAMR, "%d", pt);
          m_nAMR_PT = pt;
        }
        else if (strncasecmp(codec_name, "telephone", 9) == 0) {
          LOG_ENTRY(m_GCSIPHandle, "  attribute is for rfc2833\n");
          sprintf(szTelephone, "%d", pt);
          m_nRFC2833_PT = pt;
        }
        else if (strncasecmp(codec_name, "h263-1998", 9) == 0) {
          LOG_ENTRY(m_GCSIPHandle, "  Codec is h.263-1998\n");
          sprintf(szH263_1998, "%d", pt);
        }
        else if (strncasecmp(codec_name, "h263-2000", 9) == 0) {
          LOG_ENTRY(m_GCSIPHandle, "  Codec is h.263-2000\n");
          sprintf(szH263_2000, "%d", pt);
        }
        else if (strncasecmp(codec_name, "h263", 4) == 0) {
          LOG_ENTRY(m_GCSIPHandle, "  Codec is h.263\n");
          sprintf(szH263, "%d", pt);
          m_nH263_PT = pt;
        }
        else if (strncasecmp(codec_name, "mp4v", 4) == 0) {
          LOG_ENTRY(m_GCSIPHandle, "  Codec is mpeg4\n");
          sprintf(szMPEG4, "%d", pt);
          m_nMPEG4_PT = pt;
        }
        else if (strncasecmp(codec_name, "h264", 4) == 0) {
          LOG_ENTRY(m_GCSIPHandle, "  Codec is h264\n");
          sprintf(szH264, "%d", pt);
          m_nH264_PT = pt;
        }
      }
    }

    int NumMedia = pMedia->getNumFormat();
    LOG_ENTRY(m_GCSIPHandle, "Media descriptor #%d = %s\n",i+1,pMedia->getValue());
    LOG_ENTRY(m_GCSIPHandle, "Media descriptor #%d contains %d media format(s)\n",i+1,NumMedia);

    for (j=0;j<NumMedia;j++)
    {
      LOG_ENTRY(m_GCSIPHandle, "Media format #%d = %s\n",j+1,pMedia->getFormat(j));
      LOG_ENTRY(m_GCSIPHandle, "Media port for #%d = %d\n",j+1,pMedia->getPort());
      
      // Media format is H.263 (default=34)
      if ( !strcmp(pMedia->getFormat(j), szH263) )
      {
        //Set the coder to the first coder in sdp
        if(GetSelectedVideoCoder() == VID_CODER_NONE)
            SetVideoToH263(false);
        // SKS get bandwidth information if remote end sends
        char *bandwidthModifier=new char[10];
        char *bandwidthModifierValue=new char[10];
        sprintf(bandwidthModifier,"%s",pMD->bandwidth()->getModifier());
        if(bandwidthModifier!=NULL) {
          sprintf(bandwidthModifierValue,"%s",pMD->bandwidth()->getBandwidthValue());
        }
        LOG_ENTRY(m_GCSIPHandle, "Bandwidth Requested=%s\n",bandwidthModifierValue);
        m_remoteVideoRTPPort = pMedia->getPort();
        //Use the media IP address if one is provided, otherwise use the sdp ip addr
        if( strlen(pMD->connection()->getAddress()) )
        {
          strcpy(m_remoteVideoIPAddress, pMD->connection()->getAddress());
        }
        else
        {
          strcpy(m_remoteVideoIPAddress, sdp.connection()->getAddress());
        }
        LOG_ENTRY(m_GCSIPHandle, "Using remote Video media IP address for #%d.%d = %s\n", i+1,j+1,  m_remoteVideoIPAddress);

        // Parse fmtp attributes only of h263
        sdpAttributeList* pAttributeList = pMD->attributeList();
        int NumAttributes=pAttributeList->numItem();
        char *tempStr = 0;
        int fps=0;
        char videoSize[5];

        for(nAttributes=0;nAttributes<NumAttributes;nAttributes++) {
          sdpAttribute* pAttribute = pAttributeList->getItem(nAttributes);
          LOG_ENTRY(m_GCSIPHandle, "\t AttribuesName(%s)= AttributeValue(%s)\n",pAttribute->getProperty(),pAttribute->getPropertyValue());
          if(strcmp(pAttribute->getProperty(),"fmtp")==0) {
            char *str=new char[128];
            //strcpy(str,pAttribute->getPropertyValue());
            sprintf(str,"%s",pAttribute->getPropertyValue());
            tempStr=strstr(str,szH263);
            if(tempStr==NULL) {
              continue;
            }	
            tempStr=0;
            tempStr=strstr(str,"SQCIF");
            if(tempStr!=NULL) {
              sscanf(tempStr,"SQCIF=%d",&fps);
              strcpy(videoSize,"SQCIF");
            }
            else {
              tempStr=strstr(str,"QCIF");
              if(tempStr!=NULL) {
                sscanf(tempStr,"QCIF=%d",&fps);
                strcpy(videoSize,"QCIF");
              }
              else {
                tempStr=0;
                tempStr=strstr(str,"CIF");
                if(tempStr!=NULL) {
                  sscanf(tempStr,"CIF=%d",&fps);
                  strcpy(videoSize,"CIF");
                }
              }
              UpdateH263Attributes(videoSize,atoi(bandwidthModifierValue)*1000,(30/fps));
              LOG_ENTRY(m_GCSIPHandle, "\t Codec=H263 | Size=%s | FPS=%d\n",videoSize,(30/fps));
              break;  // Break attributes loop after fmtp
            }
          }
        }  // end of attributes loop
      }  // end of Media format H263 (34)

      // Media format is MPEG4 (default=102)
      if ( !strcmp(pMedia->getFormat(j), szMPEG4) )
      {
        //Set the coder to the first coder in sdp
        if(GetSelectedVideoCoder() == VID_CODER_NONE)
          SetVideoToMPEG4();
        m_remoteVideoRTPPort = pMedia->getPort();
        //Use the media IP address if one is provided, otherwise use the sdp ip addr
        if( strlen(pMD->connection()->getAddress()) )
        {
          strcpy(m_remoteVideoIPAddress, pMD->connection()->getAddress());
        }
        else
        {
          strcpy(m_remoteVideoIPAddress, sdp.connection()->getAddress());
        }
        LOG_ENTRY(m_GCSIPHandle, "Using remote Video media IP address for #%d.%d = %s\n", i+1,j+1,  m_remoteVideoIPAddress);
        // SKS get bandwidth information if remote end sends
        char *bandwidthModifier=new char[10];
        char *bandwidthModifierValue=new char[10];
        sprintf(bandwidthModifier,"%s",pMD->bandwidth()->getModifier());
        if(bandwidthModifier!=NULL) {
          sprintf(bandwidthModifierValue,"%s",pMD->bandwidth()->getBandwidthValue());
        }
        LOG_ENTRY(m_GCSIPHandle, "Bandwidth Requested=%s\n",bandwidthModifierValue);

        // Parse fmtp attributes only of mpeg4
        sdpAttributeList* pAttributeList = pMD->attributeList();
        int NumAttributes=pAttributeList->numItem();
        char *tempStr = 0;
        int profileLevelId=0;
        //char strDciValue[256]={0};
        int bitRate=atoi(bandwidthModifierValue)*1000;
        if(bitRate<0 || bitRate>384000){ bitRate=0;}

        for(nAttributes=0;nAttributes<NumAttributes;nAttributes++) {
          sdpAttribute* pAttribute = pAttributeList->getItem(nAttributes);
          LOG_ENTRY(m_GCSIPHandle, "\t AttributesName(%s)= AttributeValue(%s)\n",pAttribute->getProperty(),pAttribute->getPropertyValue());
          if(strcmp(pAttribute->getProperty(),"fmtp")==0) {
            char *str=new char[128];
            sprintf(str,"%s",pAttribute->getPropertyValue());
            tempStr=strstr(str,szMPEG4);
            if(tempStr==NULL) {
              continue;
            }
            tempStr=0;
            tempStr=strstr(str,"profile-level-id");
            if(tempStr!=NULL) {
              sscanf(tempStr,"profile-level-id=%d",&profileLevelId);
            }
            // SKS get DCI value
            tempStr=strstr(str,"config=");
            if(tempStr!=NULL) {
              sscanf(tempStr,"config=%s",m_SdpDciStr);
            }
            UpdateMpeg4Attributes(profileLevelId,bitRate,m_SdpDciStr);
            LOG_ENTRY(m_GCSIPHandle, "\t Codec=mp4v-es | ProfileAndLevel=%d | BitRate=%d\n", profileLevelId, bitRate);
            LOG_ENTRY(m_GCSIPHandle, "\t DCIValue=%s\n", m_SdpDciStr);
            break;    // Terminate attributes loop after ftmp
          }
        } // end of attributes loop
      }

      // Media format is H264 (default=??)
      if ( !strcmp(pMedia->getFormat(j), szH264) )
      {
        //Set the coder to the first coder in sdp
        if(GetSelectedVideoCoder() == VID_CODER_NONE)
          SetVideoToH264();
        m_remoteVideoRTPPort = pMedia->getPort();
        //Use the media IP address if one is provided, otherwise use the sdp ip addr
        if( strlen(pMD->connection()->getAddress()) )
        {
          strcpy(m_remoteVideoIPAddress, pMD->connection()->getAddress());
        }
        else
        {
          strcpy(m_remoteVideoIPAddress, sdp.connection()->getAddress());
        }
        LOG_ENTRY(m_GCSIPHandle, "Using remote Video media IP address for #%d.%d = %s\n", i+1,j+1,  m_remoteVideoIPAddress);
        // SKS get bandwidth information if remote end sends
        char *bandwidthModifier=new char[10];
        char *bandwidthModifierValue=new char[10];
        sprintf(bandwidthModifier,"%s",pMD->bandwidth()->getModifier());
        if(bandwidthModifier!=NULL) {
          sprintf(bandwidthModifierValue,"%s",pMD->bandwidth()->getBandwidthValue());
        }
        LOG_ENTRY(m_GCSIPHandle, "Bandwidth Requested=%s\n",bandwidthModifierValue);

        // Parse fmtp attributes only of H264
        sdpAttributeList* pAttributeList = pMD->attributeList();
        int NumAttributes=pAttributeList->numItem();
        char *tempStr = 0;
        int profileLevelId=0;
        //char strDciValue[256]={0};
        int bitRate=atoi(bandwidthModifierValue)*1000;
        if(bitRate<0 || bitRate>384000){ bitRate=0;}

        for(nAttributes=0;nAttributes<NumAttributes;nAttributes++) {
          sdpAttribute* pAttribute = pAttributeList->getItem(nAttributes);
          LOG_ENTRY(m_GCSIPHandle, "\t AttributesName(%s)= AttributeValue(%s)\n",pAttribute->getProperty(),pAttribute->getPropertyValue());
          if(strcmp(pAttribute->getProperty(),"fmtp")==0) {
            char *str=new char[128];
            sprintf(str,"%s",pAttribute->getPropertyValue());
            tempStr=strstr(str,szH264);
            if(tempStr==NULL) {
              continue;
            }
            tempStr=0;
            tempStr=strstr(str,"profile-level-id");
            if(tempStr!=NULL) {
              sscanf(tempStr,"profile-level-id=%x",&profileLevelId);
            }
            // SKS get DCI value
            tempStr=strstr(str,"config=");
            if(tempStr!=NULL) {
              sscanf(tempStr,"config=%s",m_SdpDciStr);
            }
            UpdateH264Attributes(profileLevelId,bitRate,m_SdpDciStr);
            LOG_ENTRY(m_GCSIPHandle, "\t Codec=h264 | ProfileAndLevel=%d | BitRate=%d\n", profileLevelId, bitRate);
            LOG_ENTRY(m_GCSIPHandle, "\t DCIValue=%s\n", m_SdpDciStr);
            break;    // Terminate attributes loop after ftmp
          }
        } // end of attributes loop
      }

      // Media format 96 is AMR
      if ( !strcmp(pMedia->getFormat(j), szAMR) )
      {
        //Set the coder to the first coder in sdp
        if(GetSelectedAudioCoder() == AUD_CODER_NONE)
            SetAudioToAMR();
        m_remoteAudioRTPPort = pMedia->getPort();
        //Use the media IP address if one is provided, otherwise use the sdp ip addr
        if( strlen(pMD->connection()->getAddress()) )
        {
          strcpy(m_remoteAudioIPAddress, pMD->connection()->getAddress());
        }
        else
        {
          strcpy(m_remoteAudioIPAddress, sdp.connection()->getAddress());
        }
        LOG_ENTRY(m_GCSIPHandle, "Using remote Audio IP address = %s\n", m_remoteAudioIPAddress);
      }
      // Media format 0 is G711ULAW
      if ( !strcmp(pMedia->getFormat(j), "0") )
      {
        //Set the coder to the first coder in sdp
        if(GetSelectedAudioCoder() == AUD_CODER_NONE)
            SetAudioToG711U();
        m_remoteAudioRTPPort = pMedia->getPort();
        //Use the media IP address if one is provided, otherwise use the sdp ip addr
        if( strlen(pMD->connection()->getAddress()) )
        {
          strcpy(m_remoteAudioIPAddress, pMD->connection()->getAddress());
        }
        else
        {
          strcpy(m_remoteAudioIPAddress, sdp.connection()->getAddress());
        }
        LOG_ENTRY(m_GCSIPHandle, "Using remote Audio IP address = %s\n", m_remoteAudioIPAddress);
      }
      // Media format 8 is G711ALAW
      if ( !strcmp(pMedia->getFormat(j), "8") )
      {
        //Set the coder to the first coder in sdp
        if(GetSelectedAudioCoder() == AUD_CODER_NONE)
          SetAudioToG711A();
        m_remoteAudioRTPPort = pMedia->getPort();
        //Use the media IP address if one is provided, otherwise use the sdp ip addr
        if( strlen(pMD->connection()->getAddress()) )
        {
          strcpy(m_remoteAudioIPAddress, pMD->connection()->getAddress());
        }
        else
        {
          strcpy(m_remoteAudioIPAddress, sdp.connection()->getAddress());
        }
        LOG_ENTRY(m_GCSIPHandle, "Using remote Audio IP address = %s\n", m_remoteAudioIPAddress);
      }
      // Media format 4 is G723
      if ( !strcmp(pMedia->getFormat(j), "4") )
      {
        //Set the coder to the first coder in sdp
        if(GetSelectedAudioCoder() == AUD_CODER_NONE)
          SetAudioToG723();
        m_remoteAudioRTPPort = pMedia->getPort();
        //Use the media IP address if one is provided, otherwise use the sdp ip addr
        if( strlen(pMD->connection()->getAddress()) )
        {
          strcpy(m_remoteAudioIPAddress, pMD->connection()->getAddress());
        }
        else
        {
          strcpy(m_remoteAudioIPAddress, sdp.connection()->getAddress());
        }
        LOG_ENTRY(m_GCSIPHandle, "Using remote Audio IP address = %s\n", m_remoteAudioIPAddress);
      }
      // Media format 18 is G729
      if ( !strcmp(pMedia->getFormat(j), "18") )
      {
        //Set the coder to the first coder in sdp
        if(GetSelectedAudioCoder() == AUD_CODER_NONE)
          SetAudioToG729();
        m_remoteAudioRTPPort = pMedia->getPort();
        //Use the media IP address if one is provided, otherwise use the sdp ip addr
        if( strlen(pMD->connection()->getAddress()) )
        {
          strcpy(m_remoteAudioIPAddress, pMD->connection()->getAddress());
        }
        else
        {
          strcpy(m_remoteAudioIPAddress, sdp.connection()->getAddress());
        }
        LOG_ENTRY(m_GCSIPHandle, "Using remote Audio IP address = %s\n", m_remoteAudioIPAddress);
      }
    } //end for NumMedia
  } //end for NumMD
}
