/**
* @file sipidle_state.cpp
* @brief Definition of SIPIdleState class
* @date Feb. 20, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#include "ipmlib.h"
#include "sipidle_state.h"
#include "sipoffered_state.h"
#include "sipendpoint.h"
#include "endpointmgr.h"
#include "baseendpoint.h"
#include "appevents.h"
#include "logger.h"

//*****************************************************************************
// Function: SIPIdleState::SIPIdleState(SIPEndpoint *pSIPEndpoint)
// Description: Initializing constructor
// Return:  SIPIdleState*
// Parameters: SIPEndpoint *pSIPEndpoint 
//*****************************************************************************
SIPIdleState::SIPIdleState(SIPEndpoint* pSIPEndpoint):
   SIPEPState(pSIPEndpoint)
{
   m_pSIPEndpoint = pSIPEndpoint;
   m_PortDiscCount = 0;
}

//*****************************************************************************
// Function: SIPIdleState::~SIPIdleState()
// Description: Destructor
// Return:  none
// Parameters: none 
//*****************************************************************************
SIPIdleState::~SIPIdleState()
{
}

//*****************************************************************************
// Function: void SIPIdleState::ProcessEvent(METAEVENT metaevent)
// Description: Process an event
// Return: none 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void SIPIdleState::ProcessEvent(METAEVENT metaevent)
{
   switch ( metaevent.evttype )
   {
      // Application events
      case APP_CALLREQ:
         ProcessOutboundSIPReq(metaevent);
         break;
         // GC/R4 events
      case GCEV_OFFERED:
         LOG_ENTRY(metaevent.evtdev, "SIPIdle state: GCEV_OFFERED received\n");
         ProcessCallOffered(metaevent);
         break;
      case GCEV_EXTENSION:
         LOG_ENTRY(metaevent.evtdev, "SIPIdle state: GCEV_EXTENSION received\n");
         break;
      case GCEV_DISCONNECTED:
         LOG_ENTRY(metaevent.evtdev, "SIPIdle state: GCEV_DISCONNECTED received\n");
         break;
      case GCEV_SETCONFIGDATA:
         LOG_ENTRY(metaevent.evtdev, "SIPIdle state: GCEV_SETCONFIGDATA received\n");
         break;
      case GCEV_SETCONFIGDATA_FAIL:
         LOG_ERROR(metaevent.evtdev, "SIPIdle state: GCEV_SETCONFIGDATA_FAIL received\n");
         break;
      case IPMEV_STOPPED:
         LOG_ENTRY(metaevent.evtdev, "SIPIdle state: IPMEV_STOPPED received\n");
         break;
      case DMEV_PORT_DISCONNECT:
         LOG_ENTRY(metaevent.evtdev, "SIPIdle state: DMEV_PORT_DISCONNECT received\n");
         ProcessPortDisconnect(metaevent.evtdev);  
	 break;
      default:
         LOG_ERROR(metaevent.evtdev,"SIPIdle state: Unexpected event type: 0x%x\n", metaevent.evttype);
         break;
   }
}

//*****************************************************************************
// Function: void SIPIdleState::ProcessUserPrompt(SIPEPState::E_USER_PROMPT_TYPE eUserPrompt)
// Description: Process a request from the user interface
// Return: void 
// Parameters: SIPEPState::E_USER_PROMPT_TYPE eUserPrompt 
//*****************************************************************************
void  SIPIdleState::ProcessUserPrompt(SIPEPState::E_USER_PROMPT_TYPE eUserPrompt)
{
   LOG_ERROR(0,"Unexpected user prompt on SIP endpoint in IDLE state: %d\n",
             static_cast<int>(eUserPrompt));
}

//*****************************************************************************
// Function: void SIPIdleState::Shutdown()
// Description: Process shutdown request
// Return: void 
// Parameters: none 
//*****************************************************************************
void SIPIdleState::Shutdown()
{
  LOG_ENTRY(m_pSIPEndpoint->GetGCSIPHandle(), "Shutdown called in %s state\n", GetStateStr());
  m_pSIPEndpoint->DisconnectFromPeer();
  m_pSIPEndpoint->ReleaseResources();
}

void SIPIdleState::ProcessCallOffered(METAEVENT metaevent)
{
  LOG_ENTRY(metaevent.evtdev, "ProcessCallOffered\n");

  // Reset selected coders to choose from incomming SDP
  m_pSIPEndpoint->SetSelectedVideoCoder(VID_CODER_NONE);
  m_pSIPEndpoint->SetSelectedAudioCoder(AUD_CODER_NONE);
  
  // Advance to offered state, save SDP info from INVITE,
  // build SDP for OK using local media info and send OK
  // with a gc_Answer
  m_pSIPEndpoint->ChangeState(OFFERED_STATE);
  m_pSIPEndpoint->SaveRemoteMediaInfo(metaevent);
  m_pSIPEndpoint->AnswerCall(metaevent);
}


//*****************************************************************************
// Function: void SIPIdleState::ProcessOutboundSIPReq(METAEVENT metaevent)
// Description: Process a request to make an outbound SIP call
// Return: void 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void SIPIdleState::ProcessOutboundSIPReq(METAEVENT metaevent)
{
   LOG_ENTRY(metaevent.evtdev, "ProcessOutboundSIPReq\n");
   
   // Advance to calling state
   m_pSIPEndpoint->ChangeState(CALLING_STATE);
   m_pSIPEndpoint->MakeCall(metaevent);
}

void SIPIdleState::ProcessPortDisconnect(int evtDev)
{
      m_PortDiscCount++;
      if (m_PortDiscCount == 2) {
	 LOG_ENTRY(evtDev," Audio and Video Ports Disconnected\n");
	 m_pSIPEndpoint->CloseSubDevs();
	 m_pSIPEndpoint->ChangeState(CLOSED_STATE);
	 m_pSIPEndpoint->Notify(APP_SIP_ENDPOINT_DISCONNECTED);
      }
}
