/**********@@@SOFT@@@WARE@@@COPY@@@RIGHT@@@**********************************
* DIALOGIC CONFIDENTIAL
*
* Copyright (C) 2006-2007 Dialogic Corporation. All Rights Reserved.
* The source code contained or described herein and all documents related
* to the source code ("Material") are owned by Dialogic Corporation or its
* suppliers or licensors. Title to the Material remains with Dialogic Corporation
* or its suppliers and licensors. The Material contains trade secrets and
* proprietary and confidential information of Dialogic or its suppliers and
* licensors. The Material is protected by worldwide copyright and trade secret
* laws and treaty provisions. No part of the Material may be used, copied,
* reproduced, modified, published, uploaded, posted, transmitted, distributed,
* or disclosed in any way without Dialogic's prior express written permission.
*
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*
***********************************@@@SOFT@@@WARE@@@COPY@@@RIGHT@@@**********/
//***********************************************************************
//***********************************************************************
// ConfigFile.cpp: implementation of the ConfigFile class.
//
//////////////////////////////////////////////////////////////////////
#include <ctype.h> 
#include "utils.h"
#include "CParser.h"

static const char * CFG_MODULE_NAME = "CParser";

// These are tokens as well as delimiters
// ( space, tab and new line are also delimiters, but are not tokens)
static const char *atomic_tokens="%^&()=|,;'[]{}?<>";
static const char *comment_line = "#*!";       // a line comment

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////
//*****************************************************************************
// Purpose	: 
//    Constructor
// Parameters:	
//    none
// Returns:	none
//*****************************************************************************
CParser::CParser(CGenLog *pLog):
                 CanLog(pLog) {
  m_section_name = 0;
  m_is_correct = true;
  ClearBuffer();
}  //	End of constructor()



//*****************************************************************************
// Purpose	: 
//    release all buffers and init internal variables
// Parameters:	
//    none
// Returns:	void
//*****************************************************************************
void CParser::ClearBuffer(){
    m_buffer = 0;
    m_current_pos = 0;
    m_current_line = 0;
    str_deletestoredstring(&m_section_name);
    m_section_id = -1;
    m_back = false;
    m_last_line = 0;
    m_last_pos = 0;
}  //	End of DeleteBuffer()

//*****************************************************************************
// Purpose	: 
//    Advance m_current_pos to the first token in buffer
// Parameters:	
//    none
// Returns:	bool 
//      false = no more data
//*****************************************************************************
bool CParser::Rewind(){
    m_current_line = 1;
    m_current_pos  = m_buffer;
    str_deletestoredstring(&m_section_name);
    m_section_id = -1;
    m_back = false;
    m_last_line = 1;
    m_last_pos = 0;

    return GotoNextToken();
}  //	End of Rewind()


//*****************************************************************************
// Purpose	: 
//    advance m_current_pos to the first non-space character
// Parameters:	
//    None
// Returns:	
//    bool (false = no more data)
//*****************************************************************************
bool CParser::SkipSpace() {
    if ( 0 == m_current_pos ){
        // null buffer
        return false;
    }

    while ( *m_current_pos ) {
        if (!isspace(*m_current_pos) ){
            return true;
        }

        if (CHAR_NEWLINE == *m_current_pos){
            ++ m_current_line;
        }

        ++m_current_pos;
    }  // while
	return false;
}  //	End of SkipSpace()

//*****************************************************************************
// Purpose	: 
//    advance m_current_pos at the beginning of the next line
//    ( skip comments )
// Parameters:	
//    none
// Returns:	
//    bool (false = no more data)
//*****************************************************************************
bool  CParser::GotoNextLine() {
    if (!m_current_pos){
        // null buffer
        return false;
    }

	m_current_pos = strchr(m_current_pos, CHAR_NEWLINE);

    if ( m_current_pos == 0) {
         return false;
    }

    ++ m_current_pos;
    ++ m_current_line;

    if ( *m_current_pos == 0) {
        return false;
    }

    return true;
}  //	End of GotoNextLine()

//*****************************************************************************
// Purpose	: 
//    advance m_current_pos at the beginning of the next token
// Parameters:	
//    None
// Returns:	
//    bool (false = no more data)
//*****************************************************************************
bool CParser::GotoNextToken() {

    if (!m_current_pos){
        // null buffer
        return false;
    }

    while ( SkipSpace( ) ) { 
        if (    (*m_current_pos   == '*') 
             && (m_current_pos[1] == '/')  
           ) {
            m_current_pos+=2;
            m_is_correct = false;
            LOG( LOG_ERR2, CFG_MODULE_NAME,
                 "Line %d: Extra closing '*/'",
                 GetCurrentLineNumber() );
        }else if(    strchr(comment_line, *m_current_pos)
            || (    (*m_current_pos   == '/') 
                 && (m_current_pos[1] == '/')  
               ) 
          ) {
            // skip this line
            if (! GotoNextLine()){
                return false;
            } 
        } else if (    (*m_current_pos   == '/') 
                    && (m_current_pos[1] == '*')  
                  ) {
            // Start of multi-line comment
            m_current_pos +=2;
            list<int> * bgn_lines = new list<int>;
            bgn_lines->push_back(GetCurrentLineNumber());
            bool brc = SkipMultiLineComment(bgn_lines);
            if ( bgn_lines->size() ){
                // Then we have unmatched openning comments
                m_is_correct = false;
                list <int>::iterator pos;
                for ( pos = bgn_lines->begin(); pos != bgn_lines->end(); ++pos ) {
                      LOG( LOG_ERR2, CFG_MODULE_NAME,
                           "Line %d: Unmatched  '/*'",*pos);
                } 
            } 
            delete bgn_lines;
            if (!brc) {
                // done, no more data
                return false;
            }
        } else {
            return true;
        }
	}
  return false;
}  //	End of GotoNextToken()

//*****************************************************************************
// Purpose	: 
//    read section name (enclosed in [] ) and section ID
// Parameters:	
//    None
// Returns:	
//    bool (false = no more data)
//*****************************************************************************
bool CParser::TrySection() {

    if (!m_current_pos){
        // null buffer
        return false;
    }

    if (*m_current_pos != '['){
        return true;
    }

    // skip "["
    ++m_current_pos;

    char token[MAXTOKEN];
    char section[MAXTOKEN];
    *section = 0;
    while ( GetNextToken(token,sizeof(token) ) ) {
        // skip anything enclosed in []
        // load m_cnf_section and m_section_id 
        // [Board Configuration] is section 'BoardConfiguration' id=0
        // [Board 12] is section 'Boars' id = 12
        if (COMPARE_EQUAL != str_compare(token,"]") ){ 
            if (   (*section == 0 ) 
                || !str_getnumber(token, &m_section_id) ){
                str_safecat(section, sizeof(section), token);
            }
        } else {
            break;
        }
    }
    str_storestring(&m_section_name,section);
    return GotoNextToken();
}  //	End of TrySection()

//*****************************************************************************
// Purpose	: 
//    store rest of the line in client's buffer and advance m_current_pos
// Parameters:	
//    [out] buffer - holder for token
//     [in] buffer size
// Returns:	
//    bool (false = no more data)
//    also returns token in token_buf.
//           Guaranteed not to overflow, but may trim long tokens
//*****************************************************************************
bool CParser::GetLine(char *token_buf, size_t token_len) {

    if (token_len == 0){
        return false ; // no buffer
    }

    if (m_current_pos == 0 ){
        // This is first call
        // position on first token;
        if ( !Rewind() ) {
             return false;
        }
    }

    if (!GotoNextToken()){
        return false;
    }

    // Back() implementation:
    {
        if ( m_back ){
             m_current_line = m_last_line;
             m_current_pos =  m_last_pos;
             return true;
        }
        m_back = false;
        m_last_line = m_current_line ;
        m_last_pos = m_current_pos;
    } // Back() block

	const char *tmp  = strchr(m_current_pos, CHAR_NEWLINE);

    if ( tmp == 0) {
		 str_safecopy(token_buf, token_len , m_current_pos);
		 m_current_pos = 0;
         return true;
    }

	size_t len = tmp-m_current_pos;
	if (len > token_len-1) {
		len = token_len-1;
	}
	if (len > 0){
	    memcpy(token_buf,m_current_pos,len);
	}
	token_buf[len]=0;

    return true;

}  //	End of GetLine()

//*****************************************************************************
// Purpose	: 
//    store next token in client's buffer and advance m_current_pos
// Parameters:	
//    [out] buffer - holder for token
//     [in] buffer size
// Returns:	
//    bool (false = no more data)
//    also returns token in token_buf.
//           Guaranteed not to overflow, but may trim long tokens
//*****************************************************************************
bool CParser::GetNextToken(char *token_buf, size_t token_len) {
bool is_in_string = false; // inside string (enclosed in ""),
                           // do not check for atomic-tokens and space characters 

    if (token_len == 0){
        return false ; // no buffer
    }

    if (m_current_pos == 0 ){
        // This is first call
        // position on first token;
        if ( !Rewind() ) {
             return false;
        }
    }

    if (!GotoNextToken()){
        return false;
    }

    // Back() implementation:
    {
        if ( m_back ){
             m_current_line = m_last_line;
             m_current_pos =  m_last_pos;
             return true;
        }
        m_back = false;
        m_last_line = m_current_line ;
        m_last_pos = m_current_pos;
    } // Back() block


    // Try if it is a section enclosed in "[]"
    // skip entire section name and advance to next token
    // update m_section_name and m_section_id with new values
    TrySection();

    if (*m_current_pos == '\"') {
        // this is a string beginning with '"'
        ++m_current_pos;
        is_in_string = true;
    }else {
        // check for other atomic token, such as '=' or '+'
        if (strchr(atomic_tokens, *m_current_pos) ) {
             *token_buf = *m_current_pos;
             ++m_current_pos;
             ++token_buf;
             *token_buf=0;
            return true;
        }
    }

    char *tmp = token_buf;
    while ( token_len > 1 ){
        if (  *m_current_pos != 0 ) {
            if ( is_in_string ){
                // When in string, watch for closing \"
                if ( *m_current_pos == '\"' ){
                     *token_buf=0;
                     ++m_current_pos;
                     return true;
                } else {
                    if ( *m_current_pos == CHAR_NEWLINE ){
                        // new line is also not good, terminate string
                         *token_buf=0;
                         ++m_current_pos;
                         LOG(LOG_WARNING, CFG_MODULE_NAME, "Line %d: closing '\"' is missing", m_current_line); 
                         ++m_current_line;
                         return true;
                    }
                }
            }else {
                // not in string,
                // watch for atomic tokens and spaces
                if ( isspace (*m_current_pos) ) {
                    *token_buf=0;
					if (*m_current_pos == CHAR_NEWLINE) {
						++m_current_line;
					}

                    ++m_current_pos;

                    return true;
                }
                if (strchr(atomic_tokens, *m_current_pos) ) {
                    *token_buf=0;
                    return true;
                }
            } 
            
            *token_buf = *m_current_pos;
            ++m_current_pos;
            ++token_buf;
            --token_len;
        } else {
            // end of buffer
            *token_buf=0;
            return true;
        }
    }

    *token_buf=0;
    LOG(LOG_ERR1, CFG_MODULE_NAME, "Line %d: The following token is too long %s", m_current_line, tmp); 
    // skip this long name
    while (   !isspace(*m_current_pos) 
           && ( *m_current_pos != '=' )
           && ( *m_current_pos != 0 )) {
        ++m_current_pos;
    }
 return false;
}  //	End of GetNextToken()

//*****************************************************************************
// Purpose	: 
//    Determine if specific string(buffer) matches Boolean values
//    if yes, store its Boolean representation in *result
//    "Yes" or "True" matches true (*result = true)
//    "No" or "False" matches false (*result = false)
//    Anything else is error (function returns false)
// Parameters:	
//    [in] buffer - contains token
//    [out] result - where to place Boolean value
// Returns:	
//    bool (false = Syntax error - cant match value)
//    true = success, Boolean value is in *result
//*****************************************************************************
bool CParser::GetYesNo(const char *buffer, bool *result){
     if ( COMPARE_EQUAL == str_compare("Yes",buffer) ){
          *result = true;
     } else if ( COMPARE_EQUAL == str_compare("True",buffer) ) {
          *result = true;
     } else if ( COMPARE_EQUAL == str_compare("No",buffer) ) {
          *result = false;
     } else if ( COMPARE_EQUAL == str_compare("False",buffer) ) {
          *result = false;
     }else {
           LOG(LOG_ERR2,CFG_MODULE_NAME,
               "Line %d: Unknown Boolean value: %s",
                GetCurrentLineNumber(), buffer );
           return false;
     }
 return true;
}  //	End of GetYesNo()

//*****************************************************************************
// Purpose	: 
//    Determine if specific string(buffer) represents numeric value
// Parameters:	
//    [in] buffer - contains token hex representation 0xnn is Ok
//    [out] result - where to place numeric (int) value
// Returns:	
//    bool (false = Syntax error - buffer does not represent any number)
//*****************************************************************************
bool CParser::GetNumber(const char *buffer, int *result){
  bool brc = str_getnumber(buffer, result);
  if (!brc) {
      LOG(LOG_ERR2,CFG_MODULE_NAME,
           "Line %d: Unknown numeric value: %s",
            GetCurrentLineNumber(), buffer );
           return false;
  } 
 return brc;
}  //	End of GetNumber()

//*****************************************************************************
// Purpose	: 
//    Determine if specific string(buffer) represents unsigned numeric value
// Parameters:	
//    [in] buffer - contains token hex representation 0xnn is Ok
//    [out] result - where to place numeric (int) value
// Returns:	
//    bool (false = Syntax error - buffer does not represent any number)
//*****************************************************************************
bool CParser::GetUNumber(const char *buffer, unsigned int *result){
  bool brc = str_getunsignednumber(buffer, result);
  if (!brc) {
      LOG(LOG_ERR2,CFG_MODULE_NAME,
           "Line %d: Unknown numeric value: %s",
           GetCurrentLineNumber(), buffer );
           return false;
  } 
 return brc;
}  //	End of GetUNumber()


//*****************************************************************************
// Purpose	: 
//    Skip multi line comments /* ... */
//    Pay attention to nested comments 
// Parameters:	
//    [in] [out] List of line numbers where comment was opened
// Returns:	
//    false = failure ( eof reached comment was not closed)
//*****************************************************************************
bool CParser::SkipMultiLineComment(list<int> * bgn_lines ){

    while ( bgn_lines->size() ) {
       if (    ( *m_current_pos   == '/') 
            && (m_current_pos[1] == '*')  
          ) {
          // Start of multi-line comment
          m_current_pos +=2;
          bgn_lines->push_back(GetCurrentLineNumber());
       }else if (    (*m_current_pos   == '*') 
                  && (m_current_pos[1] == '/')  
                ) {
           // closing
           bgn_lines->pop_back();
           m_current_pos +=2;
       }else {
          ++m_current_pos;
       }
       if (*m_current_pos == 0 ){ 
           return false;
       }
    }
 return true;
}
