/**********@@@SOFT@@@WARE@@@COPY@@@RIGHT@@@**********************************
* DIALOGIC CONFIDENTIAL
*
* Copyright (C) 2006-2007 Dialogic Corporation. All Rights Reserved.
* The source code contained or described herein and all documents related
* to the source code ("Material") are owned by Dialogic Corporation or its
* suppliers or licensors. Title to the Material remains with Dialogic Corporation
* or its suppliers and licensors. The Material contains trade secrets and
* proprietary and confidential information of Dialogic or its suppliers and
* licensors. The Material is protected by worldwide copyright and trade secret
* laws and treaty provisions. No part of the Material may be used, copied,
* reproduced, modified, published, uploaded, posted, transmitted, distributed,
* or disclosed in any way without Dialogic's prior express written permission.
*
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*
***********************************@@@SOFT@@@WARE@@@COPY@@@RIGHT@@@**********/
//***********************************************************************
//***********************************************************************
// CnfDemoPrm.cpp: implementation of the CnfDemoPrm class.
//
//////////////////////////////////////////////////////////////////////

#define PRINT_VERSION
#include "utils.h"
#include "dialogic.h"
#include "product_versions.h"
#include "CnfDemoPrm.h"


static const char * CNF_MODULE_NAME = "CnfDemoPrm";


static NAME_TABLE mode_table_stru[] = {
    {"MODE_1PCC",     MEDIA_OPERATIONAL_MODE_1PCC     },
    {"MODE_3PCC",     MEDIA_OPERATIONAL_MODE_3PCC     },
    { 0,0}
};
static CNameTable mode_table(mode_table_stru);
//////////////////////////////////////////////////////////////////////
// Global functions
//////////////////////////////////////////////////////////////////////
//*****************************************************************************
// Purpose	: 
//    Get message name from given type
// Parameters:	
//    [in] type
// Returns:	
//    name ("????" indicates unknown)
//*****************************************************************************
const char *get_call_control_mode_name(enumIPCCLIBMediaMode mode){
 return mode_table.find_name(mode);
}  //	End of log_get_msgtype_name()

//*****************************************************************************
// Purpose	: 
//    Get message type from given name
// Parameters:	
//    [in] char *name
// Returns:	
//    MSG_TYPE  ( returns MSG_TYPE_MAX in case of unresolved value )
//*****************************************************************************
enumIPCCLIBMediaMode get_call_control_value(const char *name){
  return (enumIPCCLIBMediaMode) mode_table.find_code(name);
}  //	End of log_get_msgtype_value()


//*****************************************************************************
// Purpose	: 
//    Get reserve device type from given name
// Parameters:	
//    [in] char *name
// Returns:	
//    MSG_TYPE  ( returns MSG_TYPE_MAX in case of unresolved value )
//*****************************************************************************
static NAME_TABLE  reserve_table_stru [] = {
    {"No",     RESERVE_RESOURCE_NONE },
    {"Yes",    RESERVE_RESOURCE      },
    {"Ex",     RESERVE_RESOURCE_EX   },
    { 0,0}
};

static CNameTable reserve_table(reserve_table_stru);

RESERVE_RESOURCE_TYPE get_reserve_resource_value(const char *name){
  return  (RESERVE_RESOURCE_TYPE) reserve_table.find_code(name);
}  //	End of get_reserve_resource_value()


//*****************************************************************************
// Purpose	: 
//    Get reserve device name from given type
// Parameters:	
//    [in] type
// Returns:	
//    name ("????" indicates unknown)
//*****************************************************************************
const char *get_reserve_resource_name(RESERVE_RESOURCE_TYPE mode){
    return reserve_table.find_name(mode); 
}  //	End of get_reserve_resource_name()


//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////
//*****************************************************************************
// Purpose	: 
//    Constructor
// Parameters:	
//    none
// Returns:	
//    none
//*****************************************************************************
CnfDemoPrm::CnfDemoPrm(CGenLog *pLog) 
           : ConfigFile(pLog){

 return;
}   //	End of constructor()

//*****************************************************************************
// Purpose	: 
//    Destructor
// Parameters:	
//    none
// Returns:	
//    none
//*****************************************************************************
CnfDemoPrm::~CnfDemoPrm(){


    { // delete conference parameters
        list<PCnfParams>::iterator pos;
        for ( pos = m_ConferenceContainer.begin(); pos != m_ConferenceContainer.end(); ++pos) {
              delete (*pos);
        } 
    }

    { // delete ipt params
        list<PIptParams>::iterator pos;
        for ( pos = m_IptContainer.begin(); pos != m_IptContainer.end(); ++pos) {
              delete (*pos);
        } 
    }

 return;
}  //	End of destructor()


//*****************************************************************************
// Purpose	: 
//    Read configuration file and load parameters
// Parameters:	
//    [in] file name
// Returns:	
//    true = success
//*****************************************************************************
bool CnfDemoPrm::ReadConfigFile(const char *filename){
char token[MAXTOKEN];
bool brc = true;

   CFG_ERROR err = LoadConfigFile(filename);

   LOG(err == CFG_OK?LOG_API:LOG_ERR2, CNF_MODULE_NAME,
              "[%d] %s = Open(%s)", err, cfg_get_errormsg(err),filename); 

       if ( err != CFG_OK ) {
            return false;    
       }

       while (GetNextToken(token,sizeof(token) ) ){
           if (IsSection("common") ){
               if (! ReadCommonValues(token) ){
                     brc = false;
               }
           } else if (IsSection("ipt")) { 
               if (GetSectionId() == -1) {
                   LOG(LOG_ERR2, CNF_MODULE_NAME,
                       "%s(%d): Section IPT requires unique id",
                       filename, GetCurrentLineNumber());
               }
               if (! ReadIptValues(token) ){
                     brc = false;
               }
           } else if (IsSection("dti")) { 
               if (! ReadDtiValues(token) ){
                     brc = false;
               }
           } else if (IsSection("BoardParameters")) { // no spaces in the name here!!!
               if (! ReadBoardValues(token) ){
                     brc = false;
               }
           } else if (IsSection("Conference") ){
               if (GetSectionId() == -1) {
                   LOG(LOG_ERR2, CNF_MODULE_NAME,
                           "%s(%d): Section IPT requires unique id",
                            filename, GetCurrentLineNumber());
               }
               // Read conference
               if (! ReadConference(token) ){
                     brc = false;
               }
           }
       } // while GetNextToken

       // Store last conference
       if ( brc && m_is_correct) {
            StoreConference(); 
            StoreIpt();
       }

// More validations

	   if (m_CommonParams.m_call_control_mode == MEDIA_OPERATIONAL_MODE_3PCC){
		   if (GetRequestedNumberOfIptDevices("H323") > 0) {
               LOG(LOG_ERR2, CNF_MODULE_NAME,
                  "H323 does not support 3PCC mode.",
                   filename, GetCurrentLineNumber());
		   }
	   }


       LOG(LOG_DBG, CNF_MODULE_NAME, "Done reading configuration file (%s)", filename); 
   return brc && m_is_correct;
}  //	End of ReadConfigFile()

//*****************************************************************************
// Purpose	: 
//    Read section [Common] from configuration file
// Parameters:	
//    [in] first token
// Returns:	
//    true = success
//    false = unexpected eof
//*****************************************************************************
bool CnfDemoPrm::ReadCommonValues(const char *token){
  typedef enum {
    KEYWORD_MAX_LOG_SIZE,
    KEYWORD_PRINT_LEVEL,
    KEYWORD_LOG_FILE_NAME,
    KEYWORD_WELCOME_FILE_NAME,
    KEYWORD_BAD_PASSCODE_FILE_NAME,
    KEYWORD_CALL_LATER_FILE_NAME,
    KEYWORD_GOOD_BYE_FILE_NAME,
	KEYWORD_CALL_CONTROL_MODE,
	KEYWORD_RESERVE_RESOURCE
  } COMMON_KEYWORD_CODES;
  static const char *common_keywords[] = {
   "MaxLogSize",
   "PrintLevel",
   "LogFileName",
   "WelcomeFileName",
   "BadPasscodeFileName",
   "CallLaterFileName",
   "GoodByeFileName",
   "CallControlMode",
   "ReserveResource",
  };  //	End of common_keywords

  const size_t num_common_keywords = sizeof(common_keywords) / sizeof(const char *);
  char value[MAXTOKEN];

  // skip any '='
  *value = '=';
  while( *value == '=') {
         if (!GetNextToken(value, sizeof(value))){
             return false;
         }
  } // while

  bool brc = true;
  size_t inx;
  for ( inx = 0; inx < num_common_keywords; inx++){
     if (COMPARE_EQUAL == str_compare(common_keywords[inx],token ) ) {
         switch(inx){
            case KEYWORD_MAX_LOG_SIZE:    //   "MaxLogSize",
                 brc = GetUNumber(value, &m_CommonParams.m_maxlogsize);
                 if (!brc){
                     LogInvalidValue(token, value);
                 }
                 break;

            case KEYWORD_PRINT_LEVEL:    //   "MaxLogLeve",
                 if ( ! str_getunsignednumber(value, (unsigned int *)&m_CommonParams.m_log_level ) ) {
                        m_CommonParams.m_log_level = log_get_msgtype_value(value);
                 }

                 if ( m_CommonParams.m_log_level >= MSG_TYPE_MAX ){
                     LogInvalidValue(token,value);
                     brc = false;
                 } 
                 break;

			case KEYWORD_CALL_CONTROL_MODE:	
                 if ( ! str_getunsignednumber(value, (unsigned int *)&m_CommonParams.m_call_control_mode ) ) {
                        m_CommonParams.m_call_control_mode = get_call_control_value(value);
                 }
				 switch(m_CommonParams.m_call_control_mode){
					case MEDIA_OPERATIONAL_MODE_1PCC:
					case MEDIA_OPERATIONAL_MODE_3PCC:
						 break;
					default:
                         LogInvalidValue(token,value);
                         brc = false;
						 break;
				 } // switch
				 break;

			case KEYWORD_RESERVE_RESOURCE:	
                 if ( ! str_getunsignednumber(value, (unsigned int *)&m_CommonParams.m_reserve_resource_type ) ) {
                        m_CommonParams.m_reserve_resource_type = get_reserve_resource_value(value);
                 }

                 if ( m_CommonParams.m_reserve_resource_type > RESERVE_RESOURCE_EX ){
                     LogInvalidValue(token,value);
                     brc = false;
                 } 

				 break;

            case KEYWORD_LOG_FILE_NAME:    //   "LogFileName",
                 str_storestring(&m_CommonParams.m_log_file, value);
                 break;

            case KEYWORD_WELCOME_FILE_NAME:    //   "WelcomeFileName",
                 str_storestring(&m_CommonParams.m_welcome_file, value);
                 break;

            case KEYWORD_BAD_PASSCODE_FILE_NAME:    //   "BadPasscodeFileName",
                 str_storestring(&m_CommonParams.m_bad_passcode_file, value);
                 break;

            case KEYWORD_CALL_LATER_FILE_NAME:    //   "CallLaterFileName",
                 str_storestring(&m_CommonParams.m_callater_file, value);
                 break;

            case KEYWORD_GOOD_BYE_FILE_NAME:    //   "GoodByeFileName",
                 str_storestring(&m_CommonParams.m_goodbye_file, value);
                 break;

            default:
                 LogUnmatchedParameter(token, __LINE__);
                 brc = false;
                 break;
         } // switch inx (keyword)
         break;
    } // if keyword match
  }   // for each keyword  

  if ( inx == num_common_keywords){
       LogUndefinedParameter(token);
  }

 return brc;
}  //	End of ReadCommonValues()

//*****************************************************************************
// For   ReadIptValues():
// Tables with possible parameter values
//*****************************************************************************

static NAME_TABLE digit_types[] = {
    {"RFC2833",   IP_DTMF_TYPE_RFC_2833     },
	{"INBAND",    IP_DTMF_TYPE_INBAND_RTP   },
	{"OUTOFBAND", IP_DTMF_TYPE_ALPHANUMERIC },
    {0,0}
};

const char * ip_digit_type_name(int code) {
  const char *dg_type;
  if (str_findname(code, &dg_type, digit_types)){
     return dg_type;
  } 
  return "unknown";   
}


static NAME_TABLE coder_names[] = {
    {"CODER_TYPE_NONSTANDARD",   CODER_TYPE_NONSTANDARD  },
    {"CODER_TYPE_G711ALAW64K",   CODER_TYPE_G711ALAW64K  },
    {"CODER_TYPE_G711ALAW56K",   CODER_TYPE_G711ALAW56K  },
    {"CODER_TYPE_G711ULAW64K",   CODER_TYPE_G711ULAW64K  },
    {"CODER_TYPE_G711ULAW56K",   CODER_TYPE_G711ULAW56K  },
    {"CODER_TYPE_G721ADPCM",     CODER_TYPE_G721ADPCM    },
    {"CODER_TYPE_G722_48K",      CODER_TYPE_G722_48K     },
    {"CODER_TYPE_G722_56K",      CODER_TYPE_G722_56K     },
    {"CODER_TYPE_G722_64K",      CODER_TYPE_G722_64K     },
	{"CODER_TYPE_G7231_5_3K",    CODER_TYPE_G7231_5_3K   },
	{"CODER_TYPE_G7231_6_3K",    CODER_TYPE_G7231_6_3K   },
    // etc....
    {0,0}
};

const char * ip_coder_name(int code) {
  const char *coder_name;
  if (str_findname(code, &coder_name, coder_names)){
     return coder_name;
  } 
  return "unknown";   
}

//-
static NAME_TABLE vad_names[] = {
    {"CODER_VAD_DISABLE",   CODER_VAD_DISABLE  },
	{"CODER_VAD_ENABLE",    CODER_VAD_ENABLE   },
    {0,0}
};

const char * ip_vad_name(int code) {
  const char *vad_name;
  if (str_findname(code, &vad_name, vad_names)){
     return vad_name;
  } 
  return "unknown";   
}


static NAME_TABLE framesize_names[] = {
    {"CODER_FRAMESIZE_5",  CODER_FRAMESIZE_5     },
	{"CODER_FRAMESIZE_10",  CODER_FRAMESIZE_10   },
	{"CODER_FRAMESIZE_20", CODER_FRAMESIZE_20 },
	{"CODER_FRAMESIZE_30", CODER_FRAMESIZE_30 },
    {0,0}
};

const char * ip_framesize_name(int code) {
  const char *frame_name;
  if ( str_findname(code, &frame_name, framesize_names) ){
       return frame_name;
  } 
  return "unknown";   
}

static NAME_TABLE payload_names[] = {
    {"IP_USE_STANDARD_PAYLOADTYPE",  IP_USE_STANDARD_PAYLOADTYPE     },
    {0,0}
};

const char * ip_payload_name(int code){
  const char *payload_name;
  if ( str_findname(code, &payload_name, payload_names) ){
       return payload_name;
  } 
  return "unknown";   
}

//*****************************************************************************
// Purpose	: 
//    Adjust parameters if (all available) is specified
// Parameters:	
//    [in] first token
// Returns:	
//    true = success
//    false = unexpected eof
//*****************************************************************************
bool CnfDemoPrm::AdjustNumberOfDevices(size_t dx_devices,
                                       size_t ipt_devices,
                                       size_t dti_devices){

  unsigned int ipt_requested = GetRequestedNumberOfIptDevices();
  unsigned int dti_requested = GetRequestedNumberOfDtiDevices();

  // ipt check
  if ( ipt_requested > ipt_devices) {
       LOG( LOG_ERR2, GetFileName(), 
            "Requested: %u ipt devices, available %u",
            ipt_requested, ipt_devices );
       return false;
  }

  // dti check
  if ( dti_requested > dti_devices) {
       LOG( LOG_ERR2, GetFileName(), 
            "Requested: %u dti devices, available %u",
            dti_requested, dti_devices );
       return false;
  }

  if ( ipt_requested + dti_requested > dx_devices) {
       LOG( LOG_ERR2, GetFileName(), 
            "Requested: %u dxx devices, available %u",
            ipt_requested + dti_requested, dx_devices );
       return false;
  }

  // consume 
  dx_devices  -= (ipt_requested + dti_requested);
  ipt_devices -= ipt_requested;
  dti_devices -= dti_requested;

  // adjust dti for (all available request)

  PDtiParams pDtiParams = GetDtiPrm();
  if ( MAXUINT == pDtiParams->m_number_of_channels ){
      size_t max_avl = tmin(dx_devices, dti_devices );
      if ( max_avl == 0 ){
           LOG( LOG_WARNING, GetFileName(), 
                "\"all avalilable ipt devices\" equals to 0");
      }
      pDtiParams->m_number_of_channels = max_avl;
     // consume dx device, we know there are enough
      dx_devices -= max_avl; 
  }

  return true;
}
//*****************************************************************************
// Purpose	: 
//    Read section [ipt] from configuration file
// Parameters:	
//    [in] first token
// Returns:	
//    true = success
//    false = unexpected eof
//*****************************************************************************
bool CnfDemoPrm::ReadIptValues(const char *token){
  typedef enum {
    NUM_IPT_CHANNELS,
    PRIVATE_IPT_LOG,
    IP_ACCEPT_CALL,
    IP_PROTOCOL,
    IP_DIGITS,
    IP_CODER,
    IP_FRAMESIZE,
    IP_VAD,
    IP_PAYLOAD,
  }IPT_KEYWORD_CODES;
  static const char *ipt_keywords[] = {
    "NumberOfChannels",
    "PrivateLog",
    "AcceptCall",
    "IpProtocol",
    "IpDigits",
    "IpCoder",
    "IpFrame",
    "IpVad",
    "PayLoadType"
  };  //End of common_keywords
  static const size_t num_ipt_keywords = sizeof(ipt_keywords) / sizeof(const char *);

  if (m_ipt_params.ipt_id != GetSectionId() ){
       // ipt changed, store current value
       StoreIpt();
  } 
  m_ipt_params.ipt_id = GetSectionId();

  char value[MAXTOKEN];

  // Skip any '='
  *value = '=';
  while( *value == '=') {
         if (!GetNextToken(value, sizeof(value))){
             return false;
         }
  } // while

  bool brc = true;
  size_t inx;
  for ( inx = 0; inx < num_ipt_keywords; inx++ ){
     if (COMPARE_EQUAL == str_compare(ipt_keywords[inx],token ) ) {
         switch(inx){
            case NUM_IPT_CHANNELS:    //   "NumberOfChannels",
                 if (! GetUNumber(value, &m_ipt_params.m_number_of_channels) ){
                     LogInvalidValue(token,value);
                     brc = false;
                 }
                 break;
            
            case PRIVATE_IPT_LOG:    //   "PrivateIptLog",
                 if ( ! str_getunsignednumber(value,(unsigned int *)&m_ipt_params.m_private_log ) ) {
                        m_ipt_params.m_private_log = log_get_msgtype_value(value);
                 }
 
                 if ( m_ipt_params.m_private_log >= MSG_TYPE_MAX ){
                      LogInvalidValue(token, value);
                      brc = false;
                 } 
                 break;
            case IP_ACCEPT_CALL:
                 if ( ! GetYesNo(value, &m_ipt_params.m_accept_call) ){
                      brc = false;
                      LogInvalidValue(token, value);
                 }
                 break;

            case IP_PROTOCOL:    //   "IP Protocol"
                 str_storestring(&m_ipt_params.ip_protocol, value);
                 break;

            case IP_DIGITS:
                 if (!str_findcode(value, &m_ipt_params.ip_digits, digit_types) ){
                     // not in table
                     if ( ! str_getnumber(value, &m_ipt_params.ip_digits ) ) {
                         // and not a number, failure
                          brc = false;
                          LogInvalidValue(token, value);
                     }
                 }
                 break;

            case IP_CODER:
                 if (!str_findcode(value, (int *)&m_ipt_params.ip_coder, coder_names) ){
                     // not in table
                     if ( ! str_getnumber(value, (int *)&m_ipt_params.ip_coder ) ) {
                         // and not a number, use default
                          LogInvalidValue(token, value);
                          brc = false;
                     }
                 }
                 break;

            case IP_FRAMESIZE:
                 if (!str_findcode(value, (int *)&m_ipt_params.ip_frame_size, framesize_names) ){
                     // not in table
                     if ( ! str_getnumber(value, (int *)&m_ipt_params.ip_frame_size ) ) {
                         // and not a number, failure
                          LogInvalidValue(token,value);
                          brc = false;
                     }
                 }
                 break;

            case IP_VAD:
                 if (!str_findcode(value, (int *)&m_ipt_params.ip_vad, vad_names) ){
                     // not in table
                     if ( ! str_getnumber(value, (int *)&m_ipt_params.ip_vad ) ) {
                          // and not a number, failure
                          LogInvalidValue(token, value);
                          brc = false;
                     }
                 }
                 break;

            case IP_PAYLOAD:
                 if (!str_findcode(value, (int *)&m_ipt_params.ip_payoad_type, payload_names) ){
                     // not in table
                     if ( ! str_getnumber(value, &m_ipt_params.ip_payoad_type ) ) {
                          // and not a number, failure
                          LogInvalidValue(token, value);
                          brc = false;
                     }
                 }
                 break;

            default:
                 LogUnmatchedParameter(token, __LINE__);
                 brc = false;
                 break;
         } // switch inx (keyword)
         break;
     } // if keyword match
 }  // for each keyword  

 if ( inx == num_ipt_keywords){
      LogUndefinedParameter(token);
 }
 return brc;
}  //	End of ReadIptValues()

//*****************************************************************************
// Purpose	: 
//    Read section [Dti] from configuration file
// Parameters:	
//    [in] first token
// Returns:	
//    true = success
//    false = unexpected eof
//*****************************************************************************
bool CnfDemoPrm::ReadDtiValues(const char *token){
 typedef enum {
    NUM_DTI_CHANNELS,
    PRIVATE_DTI_LOG,
    DTI_ACCEPT_CALL
 }DTI_KEYWORD_CODES;

 static const char *dti_keywords[] = {
    "NumberOfChannels",
    "PrivateLog",
    "AcceptCall"
 };  //	End of common_keywords
  static const size_t num_dti_keywords = sizeof(dti_keywords) / sizeof(const char *);

  char value[MAXTOKEN];

  // skip any '='
  *value = '=';
  while( *value == '=') {
         if (!GetNextToken(value, sizeof(value))){
             return false;
         }
  } // while

  bool brc = true;
  size_t inx;
  for ( inx = 0; inx < num_dti_keywords; inx++){
     if (COMPARE_EQUAL == str_compare(dti_keywords[inx],token ) ) {
         switch(inx){
            case NUM_DTI_CHANNELS:    //   "NumberOfChannels",
                 if ( !GetUNumber(value, &m_DtiParams.m_number_of_channels) ){
                      LogInvalidValue(token, value);
                      brc = false;
                 }
                 break;

            case PRIVATE_DTI_LOG:    //   "MaxLogSize",
                 if ( ! str_getunsignednumber(value, (unsigned int *)&m_DtiParams.m_private_log ) ) {
                        m_DtiParams.m_private_log = log_get_msgtype_value(value);
                 }
                 if ( m_DtiParams.m_private_log >= MSG_TYPE_MAX ){
                      LogInvalidValue(token, value);
                      brc = false;
                 } 
                 break;
            case DTI_ACCEPT_CALL:
                 if ( ! GetYesNo(value, &m_DtiParams.m_accept_call) ){
                      brc = false;
                      LogInvalidValue(token, value);
                 }
                 break;

            default:
                 LogUnmatchedParameter(token, __LINE__);
                 brc = false;
                 break;
         } // switch inx (keyword)
         break;
     } // if keyword match
  }  // for each keyword  

  if ( inx == num_dti_keywords){
       LogUndefinedParameter(token);
  }
 return brc;
}  //	End of ReadDtiValues()

//*****************************************************************************
// Purpose	: 
//    Read section [Board Parameters] from configuration file
// Parameters:	
//    [in] first token
// Returns:	
//    true = success
//    false = unexpected eof
//*****************************************************************************
bool CnfDemoPrm::ReadBoardValues(const char *token){
  typedef enum {
    ACTIVE_TALKER,
    AT_INTERVAL,
    DTMF_CLAMPING,
    PRIVATE_BRD_LOG
  }BOARD_KEYWORD_CODES;
  static const char *brd_keywords[]={
   "ActiveTalker",
   "ATInterval",
   "DTMFClamping",
   "PrivateLog" 
  };
  static const size_t num_brd_keywords = sizeof(brd_keywords) / sizeof(const char *);
  char value[MAXTOKEN];

  // skip any '='
  *value = '=';
  while( *value == '=') {
         if (!GetNextToken(value, sizeof(value))){
             return false;
         }
  } // while

  bool brc = true;
  size_t inx;
  for ( inx = 0; inx < num_brd_keywords; inx++){
     if (COMPARE_EQUAL == str_compare(brd_keywords[inx],token ) ) {
         switch(inx){
            case ACTIVE_TALKER:   //   "Active talker",
                 if ( ! GetYesNo(value, &m_BrdParams.m_brd_active_talker) ){
                      brc = false;
                      LogInvalidValue(token, value);
                 }
                 break;

            case AT_INTERVAL:     //   "Notification interval",
                 if ( ! GetNumber(value, &m_BrdParams.m_brd_at_interval) ){
                     brc = false;
                     LogInvalidValue(token, value);
                 }
                 break;

            case DTMF_CLAMPING:    //   "DTMFClamping",
                 if ( ! GetYesNo(value, &m_BrdParams.m_brd_dtmf_clamping) ){
                      brc = false;
                      LogInvalidValue(token, value);
                 }
                 break;

            case PRIVATE_BRD_LOG:    //   "PrivateLog",
                 if ( ! str_getunsignednumber(value,(unsigned int *)&m_BrdParams.m_log_level ) ) {
                        m_BrdParams.m_log_level = log_get_msgtype_value(value);
                 }
 
                 if ( m_BrdParams.m_log_level >= MSG_TYPE_MAX ){
                      LogInvalidValue(token, value);
                      brc = false;
                 } 
                 break;

            default:
                 LogUnmatchedParameter(token, __LINE__);
                 brc = false;
                 break;
         } // switch inx (keyword)
         break;
     } // if keyword match
  }  // for each keyword  

  if ( inx == num_brd_keywords){
       LogUndefinedParameter(token);
  }
  return brc;
}  //	End of ReadBoardValues()

//*****************************************************************************
// Purpose	: 
//    Store current conference using m_cnf_params and initialize m_cnf_params
// Parameters:	
//    None
// Returns:	
//    true = success
//    false = was not stored (incomplete)
//*****************************************************************************
bool CnfDemoPrm::StoreConference(){
bool brc = false;
   if (m_cnf_params.cnf_id != -1) {
       // Conference is defined
       if (   ( 0 == m_cnf_params.pass_code)
           || ( 0 == *m_cnf_params.pass_code)
           || ( ! str_isnumber(m_cnf_params.pass_code ) )  ){
           LOG( LOG_ERR2, GetFileName(), 
                "Line %d: Missing or invalid pass code for conference %d",
                 GetCurrentLineNumber(), m_cnf_params.cnf_id );
       }else {
           if (m_ConferenceContainer.find(m_cnf_params.pass_code) ){
               // Duplicate
               LOG( LOG_ERR2, GetFileName(),
                    "Line %d: Duplicate pass code for conference %d, pass_code = %s",
                    GetCurrentLineNumber(), m_cnf_params.cnf_id, m_cnf_params.pass_code );
           } else {
               // Store & clean
               PCnfParams pCnfParams = new CnfParams(m_cnf_params);
               m_ConferenceContainer.push_back(pCnfParams);
               brc = true;
           }
           m_cnf_params.Init();
       }
   } // cnf_id != -1
 return brc;
}  //	End of StoreConference()


//*****************************************************************************
// Purpose	: 
//    Store current ipt using m_ipt_params and initialize m_ipt_params
// Parameters:	
//    None
// Returns:	
//    true = success
//    false = was not stored (incomplete)
//*****************************************************************************
bool CnfDemoPrm::StoreIpt(){
bool brc = false;
   if ( m_ipt_params.ipt_id != -1) {
        // Store & clean
           PIptParams pIptParams = new CIptParams(m_ipt_params);
           m_IptContainer.push_back(pIptParams);
           m_ipt_params.Init();
           brc = true;
   } // ipt_id != -1
 return brc;
}  //	End of StoreIpt()


//*****************************************************************************
// Purpose	: 
//    Read all sections [Conference xxx] from configuration file
// Parameters:	
//    [in] first token
// Returns:	
//    true = success
//    false = unexpected eof
//*****************************************************************************
bool CnfDemoPrm::ReadConference(const char *token){
  typedef enum {
    PASS_CODE,
    TONE_CLAMPING,
    DETECT_DIGITS,
    TMO_KEYWORD,
    VOICE_KEYWORD,
    BEEP_NOTIFY_KEYWORD,
    CNF_PRIVATE_LOG,
  }CNF_KEYWORD_CODES;

  static const char *cnf_keywords[]={
   "PassCode",
   "DTMFClamping",
   "DetectDigits",
   "Tmo",
   "VoiceParty",
   "BeepNotify",
   "PrivateLog"
  };
  static const size_t num_cnf_keywords = sizeof(cnf_keywords) / sizeof(const char *);

  if (m_cnf_params.cnf_id != GetSectionId() ){
       // Conference changed, store current value
       StoreConference();
  } // if section is not same

  m_cnf_params.cnf_id = GetSectionId();
  char value[MAXTOKEN];

  // skip any '='
  *value = '=';
  while( *value == '=') {
         if (!GetNextToken(value, sizeof(value))){
             return false;
         }
  } // while

  bool brc = true;
  size_t inx;  
  for ( inx = 0; inx < num_cnf_keywords; inx++){
     if (COMPARE_EQUAL == str_compare(cnf_keywords[inx],token ) ) {
         switch(inx){
            case PASS_CODE:    //   "PassCode",
                 str_storestring(&m_cnf_params.pass_code,value);
                 break;

            case TONE_CLAMPING:    //   "DtmfClamping",
                 if (! GetYesNo(value, &m_cnf_params.dtmf_clamping) ){
                     LogInvalidValue(token, value);
                     brc = false;
                 }
                 break;

            case DETECT_DIGITS:    //   "DetectDigits",
                 if (! GetYesNo(value, &m_cnf_params.detect_digits) ){
                     LogInvalidValue(token, value);
                     brc = false;
                 }
                 break;

            case TMO_KEYWORD:        //   "Tmo",
                 if (! str_getnumber(value, &m_cnf_params.tmo) ) {
                     LogInvalidValue(token, value);
                     brc = false;
                 }
                 break;

            case VOICE_KEYWORD:      //   "VoiceParty",
                 if (! GetYesNo(value, &m_cnf_params.voice_party) ){
                     LogInvalidValue(token, value);
                     brc = false;
                 }
                 break;

            case BEEP_NOTIFY_KEYWORD:
# ifdef _FR2619
                 if (! GetYesNo(value, &m_cnf_params.beep_notify) ){
                     LogInvalidValue(token, value);
                     brc = false;
                 }
# else 
                 LOG( LOG_WARNING, GetFileName(), 
                      "Line %d: Not supported parameter <%s> in this build (%s)",
                      GetCurrentLineNumber(),token, RELNAME );
# endif // _FR2619
                 break;

            case CNF_PRIVATE_LOG:    //   "PrivateLog",
                 if ( ! str_getunsignednumber(value, (unsigned int *)&m_cnf_params.private_log) ) {
                        m_cnf_params.private_log = log_get_msgtype_value(value);
                 }
 
                 if ( m_cnf_params.private_log >= MSG_TYPE_MAX ){
                      LogInvalidValue(token, value);
                      brc = false;
                 } 
                 break;

            default:
                 LogUnmatchedParameter(token, __LINE__);
                 brc = false;
                 break;
         } // switch inx (keyword)
         break;
     } // if compare keyword
  } // for keywords

  if ( inx == num_cnf_keywords){
       LogUndefinedParameter(token);
  }

 return brc;
}  //	End of ReadConference()

//*****************************************************************************
// Purpose	: 
//    Log 'invalid value' message: ( this is not appropriate <value> for <token> )
// Parameters:	
//    [in] token 
//    [in] value
// Returns:	
//    none
//*****************************************************************************
void CnfDemoPrm::LogInvalidValue(const char *token, const char *value){
  LOG( LOG_ERR2, GetFileName(), 
       "Line %d: (cfg file error) Invalid value <%s> for parameter %s",
       GetCurrentLineNumber(), value, token );
  return;
} // End of LogInvalidValue()

//*****************************************************************************
// Purpose	: 
//    Log "This token exist, but there is no case to handle (app error)"
// Parameters:	
//    [in] token 
//    [in] line (source line number in this file)
// Returns:	
//    none
//*****************************************************************************
void CnfDemoPrm::LogUnmatchedParameter(const char *token, int line){
     LOG( LOG_ERR2, CNF_MODULE_NAME, 
          "line %d(cfg file), (app error) token <%s> is defined, but there is no case to handle it - function %s line %d, file %s",
          GetCurrentLineNumber(), token,  line,  __FILE__);
 return ;
} // End of LogUnmatchedParameter()
//*****************************************************************************
// Purpose	: 
//    Log "This token is not recognized as parameter name"
// Parameters:	
//    [in] token 
// Returns:	
//    none
//*****************************************************************************
void CnfDemoPrm::LogUndefinedParameter(const char *token){
   LOG( LOG_ERR2, GetFileName(),
        "Line %d: Token '%s' is not recognized as parameter name",
        GetCurrentLineNumber(), token);
 return;
}  // End of LogUndefinedParameter()

//*****************************************************************************
// Purpose	: 
//    Dump all parameters
// Parameters:	
//    None
// Returns:	
//    none
//*****************************************************************************
void CnfDemoPrm::Dump(){
    LOG(LOG_DBG, CNF_MODULE_NAME, "=== Global Parameters ===");
    LOG(LOG_DBG, CNF_MODULE_NAME, "  section [Common]");
    m_CommonParams.Dump(GetLog());

    LOG(LOG_DBG, CNF_MODULE_NAME, "  section [Board parameters]");
    m_BrdParams.Dump(GetLog());

    LOG(LOG_DBG, CNF_MODULE_NAME, "  section [Dti]");
    m_DtiParams.Dump(GetLog());

    {
        LOG(LOG_DBG, CNF_MODULE_NAME, "  sections [Ipt]");
        list<PIptParams>::iterator pos;
        for ( pos = m_IptContainer.begin(); pos != m_IptContainer.end(); ++pos) { 
             (*pos)->Dump(GetLog());
        } 
    }

    {
        LOG(LOG_DBG, CNF_MODULE_NAME, "  sections [Conference nnn]");
        list<PCnfParams>::iterator pos;
        for ( pos = m_ConferenceContainer.begin(); pos != m_ConferenceContainer.end(); ++pos) {
             (*pos)->Dump(GetLog());
        } 
    }


 return;
}  //	End of Dump()


