/**********@@@SOFT@@@WARE@@@COPY@@@RIGHT@@@**********************************
* DIALOGIC CONFIDENTIAL
*
* Copyright (C) 2006-2007 Dialogic Corporation. All Rights Reserved.
* The source code contained or described herein and all documents related
* to the source code ("Material") are owned by Dialogic Corporation or its
* suppliers or licensors. Title to the Material remains with Dialogic Corporation
* or its suppliers and licensors. The Material contains trade secrets and
* proprietary and confidential information of Dialogic or its suppliers and
* licensors. The Material is protected by worldwide copyright and trade secret
* laws and treaty provisions. No part of the Material may be used, copied,
* reproduced, modified, published, uploaded, posted, transmitted, distributed,
* or disclosed in any way without Dialogic's prior express written permission.
*
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*
***********************************@@@SOFT@@@WARE@@@COPY@@@RIGHT@@@**********/
//***********************************************************************
//***********************************************************************

//////////////////////////////////////////////////////////////////////
// main.cpp: implementation of the application business logics.
//
//////////////////////////////////////////////////////////////////////

#include <signal.h>

#include "pdl.h"

// Global objects
//------------------
CGenLog     * glb_pAppLog = 0;       // Application log

CDetectCfg  * glb_pDetectCfg = 0;    // Detect configuration

ConferencePool *glb_pConferencePool; // Container for all conference boards
                                     // Used to search conference by pass code

// This module static data
//--------------------------
static CnfDemoPrm  * m_pAppParams;   // Cnf Demo parameters
static CSrlThread  * m_pSrlThread;   // Represent thread and
                                     // is container for all devices & containers served by this thread 

static bool prompt_done;             // the prompt is already shown

static const char *CNF_MODULE_NAME = "MAIN";        // for logging purposes
static const int appMaxLogLines = 7200;             // defaults for logging
static const char *cfgFile = "conf_demo.cfg";

static NAME_TABLE exit_tbl_stru[] = {
    { "Normal exit",    EXIT_OK    },
    { "Config file error",   EXIT_CFG   },
    { "Resources ( service started? licenses? Other APP running(HMP)? )",  EXIT_INIT  },
    { "Log file error",   EXIT_LOG   },
    { "Open/init devices", EXIT_START },
    { "Validate ( state machine error? )",  EXIT_VALIDATE  },
    { "Failure",  EXIT_FAIL  },
    { 0, 0 }
};
static CNameTable exit_tbl(exit_tbl_stru);



// Forward Declaration
//----------------------
static void intr_handler(int exit_code);


// Global functions
//----------------------
//*****************************************************************************
// Purpose	: 
//    Request to exit
// Parameters:	
//    flag (true if it is internal request)
//          false if request is coming from OS
// Returns:	
//    none
//*****************************************************************************
static volatile bool exit_request = false; // indicates request to exit was sent
static volatile bool exit_now = false;     // indicates request for immediate exit
static volatile bool exiting = false;      // indicates exit in progress
static bool fatal_error = false;
void glb_exit_request(bool is_internal, bool is_fatal){
  if (is_fatal){
      fatal_error = true;
  }
  if (!exiting){
      if(exit_request){
         // Second attempt, exit now!
         if (!is_internal){
             GLBLOG(LOG_APP, CNF_MODULE_NAME, "Immediate exit requested");
             exit_now = true;
         }
      }else {
         // Gracefull exit
         GLBLOG(LOG_APP, CNF_MODULE_NAME, "Exit requested please wait...");
         if (m_pSrlThread) {
             m_pSrlThread->SignalExitRequest();
         }
      }
      exit_request = true;
  }else {
      GLBLOG(LOG_APP, CNF_MODULE_NAME, "Exiting ...");
  }
 return;
} //	End of glb_exit_request()


// Local functions
//----------------------
//*****************************************************************************
// Purpose	: 
//    set intr_handlers function to serve OS exit request ( kill / ctrl-C / etc.)
// Parameters:	
//    none
// Returns:	
//    none
//*****************************************************************************
static void set_intr_handler(){
#  ifdef WIN32
	signal(SIGBREAK, intr_handler);
#   else // LINUX
    signal(SIGQUIT,  intr_handler);
# endif // LINUX

	signal(SIGINT,   intr_handler);
	signal(SIGTERM,  intr_handler);
	signal(SIGILL,   intr_handler);
	signal(SIGFPE,   intr_handler);
	signal(SIGABRT,  intr_handler);
 return;
} //	End of set_intr_handler()

//*****************************************************************************
// Purpose	: 
//    handle exit request from OS - set flag for APP to exit
// Parameters:	
//    exit_code(we don't care about it's value ) - it comes from OS
// Returns:	
//    none
//*****************************************************************************
static void intr_handler(int exit_code){

  GLBLOG(LOG_APP, CNF_MODULE_NAME, "Signal (code=%d)",exit_code);

  // 
  glb_exit_request(false,false);
  
  // re-set exit handlers for customers that have lost their patience
  set_intr_handler();
    
  return;
}  //	End of intr_handler()



//*****************************************************************************
// Purpose	: 
//    display how many licenses needs current configuration
// Parameters:	
//    none
// Returns:	
//    none
//*****************************************************************************
static void display_license_info(){
   if (m_pAppParams){
       GLBLOG(LOG_ALL, CNF_MODULE_NAME, "To run current confuguration, you need at least following licenses:");
       GLBLOG(LOG_ALL, CNF_MODULE_NAME, "  Voice           = %d", m_pAppParams->GetRequestedNumberOfDxDevices());
       GLBLOG(LOG_ALL, CNF_MODULE_NAME, "  RTP             = %d", m_pAppParams->GetRequestedNumberOfIptDevices());
       GLBLOG(LOG_ALL, CNF_MODULE_NAME, "  IP Call Control = %d", m_pAppParams->GetRequestedNumberOfIptDevices());
       GLBLOG(LOG_ALL, CNF_MODULE_NAME, "  Conferencing    = %d", m_pAppParams->GetRequestedNumberOfConferences());
   }
 return;
}

//*****************************************************************************
// Purpose	: 
//    release resources and exit (assumes all state machines in NULL state)
// Parameters:	
//    return code (to OS).
//      0 indicates OK
//      1 indicates failure
// Returns:	
//    never returns
//*****************************************************************************
static void do_exit(EXIT_CODE ret_code) {
 exiting = true;

   if (EXIT_INIT == ret_code ){
       display_license_info();
   }

   delete glb_pConferencePool;

   // Delete global objects
   delete m_pSrlThread;

   // gc_Stop
   gc_stop_wrapper();
   
   // Delete global objects
   delete glb_pDetectCfg; 

   delete m_pAppParams;   

   // Provide exit reasons
   glb_pAppLog->Dump();

   const char *ret_name = exit_tbl.find_name(ret_code);

   GLBLOG(LOG_ALL, CNF_MODULE_NAME, "Exiting with code %d '%s'", ret_code, ret_name);

   // Keep console alive: print message and wait a stroke 
   con_wait_a_key(ret_code, "Cleanup complete.");

   delete glb_pAppLog;

   // Done
   printf ("\n*** End cnf Demo***\n");
   exit(ret_code); 
}  //	End of do_exit()


//*****************************************************************************
// Purpose	: 
//    Display message shown bellow
// Parameters:	
//    none
// Returns:	
//    int
//*****************************************************************************
static void LogPrompt(){
 if (!prompt_done) {
   if (!exit_request) {
       prompt_done = true;
       GLBLOG(LOG_ALL, CNF_MODULE_NAME, "***********************************");
       GLBLOG(LOG_ALL, CNF_MODULE_NAME, "*                                 *");
       GLBLOG(LOG_ALL, CNF_MODULE_NAME, "*       Ready to answer calls     *");
       GLBLOG(LOG_ALL, CNF_MODULE_NAME, "*                                 *");
       GLBLOG(LOG_ALL, CNF_MODULE_NAME, "* Name=%-27s*",glb_pDetectCfg->GetHostName() );
       GLBLOG(LOG_ALL, CNF_MODULE_NAME, "* Addr=%-27s*",glb_pDetectCfg->GetHostAddr() );
       GLBLOG(LOG_ALL, CNF_MODULE_NAME, "*                                 *");
       GLBLOG(LOG_ALL, CNF_MODULE_NAME, "* operational mode: %-14s*",get_call_control_mode_name(m_pAppParams->GetPCCMode()));
       GLBLOG(LOG_ALL, CNF_MODULE_NAME, "* using %3d ipt SIP  devices      *",m_pAppParams->GetRequestedNumberOfIptDevices("SIP"));
       GLBLOG(LOG_ALL, CNF_MODULE_NAME, "* using %3d ipt H323 devices      *",m_pAppParams->GetRequestedNumberOfIptDevices("H323"));
       GLBLOG(LOG_ALL, CNF_MODULE_NAME, "* using %3d dti devices           *",m_pAppParams->GetRequestedNumberOfDtiDevices());
       GLBLOG(LOG_ALL, CNF_MODULE_NAME, "*                                 *");
       GLBLOG(LOG_ALL, CNF_MODULE_NAME, "* using %3d conferences           *",glb_pConferencePool->GetNumberOfConferences());
       GLBLOG(LOG_ALL, CNF_MODULE_NAME, "*                                 *");
       GLBLOG(LOG_ALL, CNF_MODULE_NAME, "*                                 *");
       GLBLOG(LOG_ALL, CNF_MODULE_NAME, "*       Press <Ctrl-C> to exit    *");
       GLBLOG(LOG_ALL, CNF_MODULE_NAME, "***********************************");
   } // !exit requste
 } // ! prompt_done
}  //	End of LogPrompt()

//*****************************************************************************
// Purpose	: 
//    wait all available SRL devices to reach init state
//         exit in case pf error
// Parameters:	number of seconds to wait
//    none
// Returns:	
//    none
//*****************************************************************************
static void wait_srl(int tmo){
           
 PSrlDevice pDev;
   // Wait engine to open all conferences and conference boards
   if ( !m_pSrlThread->WaitSrlState(tmo*1000, SRLSTATE_INIT_COMPLETE, true, &pDev ) ) {
          GLBLOG( LOG_ERR2, CNF_MODULE_NAME,
                 "(%s) Failed to reach SRLSTATE_INIT_COMPLETE in %d sec",
                  pDev->GetName(), tmo);
                 do_exit(EXIT_START);
   }  
 return;
}

//*****************************************************************************
// Purpose	: 
//    Creata all conferences and wait initialisation
// Parameters:	
//    none
// Returns:	
//    int
//*****************************************************************************
static bool CreateConferences(PCommonParams pCommonParams) {

  // * --- Create conference objects
  CnfBoard    * pCnfBoard;
  PBrdParams pBrdParams = m_pAppParams->GetBrdPrm();
  PCnfPrmContainer pCnfPrmContainer = m_pAppParams->GetConferencePrmContainer();
  pCnfBoard = new CnfBoard(pCommonParams, pBrdParams, pCnfPrmContainer);
  if (!pCnfBoard->IsSrlState(SRLSTATE_RESERVED, true) ) {
      // Missing conference virtual board
      delete pCnfBoard;
      do_exit(EXIT_INIT);
  } 

  // Attach board
  m_pSrlThread->push_back(pCnfBoard);

  // Wait initialization, 5 seconds per conference
  // int sec = APP_TMO_SHORT * m_pAppParams->GetNumberOfConferences();
  int sec = 10; // 10 sec total
  wait_srl(sec);

 return true;
}
//*****************************************************************************
// Purpose	: 
//    Create all dti boards and wait initialization
// Parameters:	
//    none
// Returns:	
//    int
//*****************************************************************************
static bool CreateDtiDevices(PCommonParams pCommonParams) {

   //----- Create Ip objects, number of devices is Ok -----
   PDtiParams pDtiParams = m_pAppParams->GetDtiPrm() ;
   if(pDtiParams) {
        unsigned int inx_dev;
        for( inx_dev = 0; inx_dev < pDtiParams->m_number_of_channels; inx_dev++){
             PPstnDevice pDevice;
             pDevice = new CPstnDevice(pCommonParams, pDtiParams);
             EXIT_CODE e_code = pDevice->GetExitCode();
             if (EXIT_OK == e_code) {
                  m_pSrlThread->push_back(pDevice);
             }else {
                  delete pDevice; 
                  do_exit(e_code);
                  break;
             } 
        } // for inx_dev 

     // Wait initialization, 5 sec per IPT device
//    int sec = APP_TMO_SHORT * m_pAppParams->cp_NumIptDevices;
      int sec = 10; // 10 sec total
      wait_srl(sec);
   }// if PDtiParams
 return true;
}

//*****************************************************************************
// Purpose	: 
//    Create all dti boards and wait initialization
// Parameters:	
//    none
// Returns:	
//    int
//*****************************************************************************
static bool CreateIpDevices(PCommonParams pCommonParams) {

   //----- Create Ip objects, number of devices is Ok -----
   unsigned int inx_ipt = 0;
   PIptParams pIptParams;
   while(m_pAppParams->GetIptPrm(inx_ipt, &pIptParams)){
        unsigned int inx_dev;
        for( inx_dev = 0; inx_dev < pIptParams->m_number_of_channels; inx_dev++){
             PIptDevice pDevice;
             pDevice = new CIptDevice(pCommonParams, pIptParams);
             EXIT_CODE e_code = pDevice->GetExitCode();
             if (EXIT_OK == e_code) {
                  m_pSrlThread->push_back(pDevice);
             }else {
                  delete pDevice; 
                  do_exit(e_code);
                  break;
             } 
        } // for inx_dev: 0 to m_nunber_of_channels
        inx_ipt++;
   }// while = next group of devices

   // Wait initialization, 5 sec per IPT device
   // int sec = APP_TMO_SHORT * m_pAppParams->cp_NumIptDevices;
   int sec = 10; // 10 sec total
   wait_srl(sec);
 return true;
}

//*****************************************************************************
// Purpose	: 
//    start the app
// Parameters:	
//    none
// Returns:	
//    int
//*****************************************************************************
int main() {
EXIT_CODE ret_code = EXIT_OK;

   // setup exit handlers
   set_intr_handler();

   // Temporarily, Log to console
   glb_pAppLog = new ConsoleLog;
   
   GLBLOG(LOG_ALL, CNF_MODULE_NAME, "%s", BINARY_VERSION_STRING);

   // Read configuration file
   m_pAppParams = new CnfDemoPrm(glb_pAppLog);
   PCommonParams pCommonParams = m_pAppParams->GetCommonPrm();

   if (! m_pAppParams->ReadConfigFile(cfgFile)  || exit_request) {
         do_exit(EXIT_CFG);
   }

   // Set verbosity level

   glb_pAppLog->SetVerbosity(pCommonParams->m_log_level);

   { 
     // --- Create log file ---
       CFileLog *pFileLog = new CFileLog();
       if (! pFileLog->InitLog(pCommonParams->m_log_file, pCommonParams->m_maxlogsize) ){
             GLBLOG( LOG_ERR2, CNF_MODULE_NAME,
                     "Error creating log object(%s,%d)",
                     pCommonParams->m_log_file, pCommonParams->m_maxlogsize );
             do_exit(EXIT_LOG);
       }
       pFileLog->SetVerbosity(pCommonParams->m_log_level);

       //
       // Swap logs:
       //  pAppLog => ConsoleLog
       pFileLog->DefineChainLog(glb_pAppLog);
       glb_pAppLog = pFileLog;

       // -- Dump actual parameters 
       m_pAppParams->Dump();
   } // Block


   // -- Start GC
   if ( !gc_start_wrapper(m_pAppParams) ){
       // if gc is not started, IPT devices will not be detected
       do_exit(EXIT_INIT);
   }

   { //----- Detect & Dump configuration -----
       glb_pDetectCfg = new CDetectCfg(m_pAppParams);
       glb_pDetectCfg->DetectConfiguration(); 
       glb_pDetectCfg->DumpSize();       
   } // block (detect)

   { // Create SRL object and Devices
     //=================================
       m_pSrlThread = new CSrlThread(pCommonParams);
       if (! m_pSrlThread->InitMode(SRL_TMO, SRL_POLLED_MODE)) {
             GLBLOG( LOG_ERR2, CNF_MODULE_NAME,
                    "Error initializing SrlThread object(%d,%d)",
                    SRL_TMO, SRL_POLLED_MODE  );
             do_exit(EXIT_INIT);
       }


       if (! m_pAppParams->AdjustNumberOfDevices(glb_pDetectCfg->GetNumberOf(DEV_DX),
                                                 glb_pDetectCfg->GetNumberOf(DEV_IPT),
                                                 glb_pDetectCfg->GetNumberOf(DEV_DTI))){
            do_exit(EXIT_CFG);
       }

       // Container for all conference boards
       // Ised for searching for a conference by given pass code
       glb_pConferencePool = new ConferencePool(); 
              
       if (!exit_request) {
            CreateConferences(pCommonParams); 
       }

       if (!exit_request) {
            CreateDtiDevices(pCommonParams);
       }

       // exit request may change 
       if ( !exit_request) {
            CreateIpDevices(pCommonParams);
       }

   } // Block Create SRL Object & devices

   LogPrompt();

   //---- The main loop until:
   // 1. At least one device is still opened
   // or 2. exit-now flag is set ( but does will exit gracefully in this case)
   while ( m_pSrlThread->AnySrlState(SRLSTATE_OPENED, true, 0) ){
           if (exit_now) {
               break;
           } 
           m_pSrlThread->TheThreadBody();
           if (!m_pSrlThread->AnySrlState(SRLSTATE_CALL_ACTIVE, true, 0)) {
               LogPrompt();
           }else {
               prompt_done = false;
           }
            
   }
   if (fatal_error){
       ret_code = EXIT_FAIL;
   }else {
       if ( m_pSrlThread->AnySrlState(SRLSTATE_FAILED_FATAL, true, 0) ){
            ret_code = EXIT_FAIL;
       }
   }
   
   do_exit(ret_code);

 // fake, 
 // some compilers will complain about missing return here
 // other compilers require main() to return int
 return(0);
}   //	End of main()
