/**
* @file exitsync.cpp
* @brief Multimedia demo exit synchronization
* @date Feb 16., 2007*
* DIALOGIC CONFIDENTIAL 
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to 
* the source code ("Material") are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its 
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced, 
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual 
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

// Application Level Header Files
#include "exitsync.h"

// Extern Variables
extern bool g_bDone;

CExitSync* CExitSync::pInst = 0;
unsigned int CExitSync::exitTry = 0;
#ifdef WIN32
HANDLE  CExitSync::hExitAppEvent = 0;
#else
sem_t   CExitSync::hExitAppEvent;
#endif

// Function Descriptions
//*****************************************************************************
// 		  NAME : CExitSync* CExitSync::Instance(void)
// DESCRIPTION : Creates a new instance of the ExitSync Object
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : A pointer to the object of the type CExitSync
// 	  CAUTIONS : None
//*****************************************************************************
CExitSync* CExitSync::Instance(void)
{
    // If the Static pointer is zero, create a new object of this type.
    if (pInst == 0) {
        pInst = new CExitSync();
    }
    return pInst;
}

//*****************************************************************************
// 		  NAME : CExitSync::CExitSync()
// DESCRIPTION : Constructor for the ExitSync Class. Registers the signal 
//			     handlers
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
CExitSync::CExitSync()
{
#ifdef WIN32
    hExitAppEvent = CreateEvent(NULL, FALSE, FALSE, NULL);
    SetConsoleCtrlHandler( CtrlHandler,  true);
#else 
    sem_init(&hExitAppEvent, 0, 0);
    if (signal(SIGTERM, &CtrlHandler) == SIG_ERR) {
        printf("Couldn't register SIGTERM handler \n");
    }

    if (signal(SIGINT, &CtrlHandler) == SIG_ERR) {
        printf("Couldn't register SIGINT handler\n");
    }
#endif
    exitTry = 0;
    pInst   = 0;
}

//*****************************************************************************
// 		  NAME : CExitSync::~CExitSync()
// DESCRIPTION : Destructor for the ExitSync Class
//			     handlers
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
CExitSync::~CExitSync()
{

}


//*****************************************************************************
// 		  NAME : BOOL WINAPI CExitSync::CtrlHandler( DWORD dwCtrlType) or
//				 void CExitSync::CtrlHandler(int signum)
// DESCRIPTION : Signal handling functions for the ExitSync class
//			     handlers. Handles the CTRL-C Application Exit notification
// 		 INPUT : Control Type or Signal number based on the OS
// 	    OUTPUT : None
// 	   RETURNS : BOOL or void
// 	  CAUTIONS : None
//*****************************************************************************
#if defined(WIN32)
BOOL WINAPI CExitSync::CtrlHandler( DWORD dwCtrlType   )
{
    printf("Ctrl-C \n");
    g_bDone = true;
    if ( exitTry == 0)
        exitTry++;
    else {
        printf("Exiting \n");
        exit(-1);
    }
    BeginExit();
    return(true);
}

#else
void CExitSync::CtrlHandler(int signum)
{
    if (signum == SIGTERM)
        printf("Caught signal SIGTERM \n");
    else if (signum == SIGINT)
        printf("Caught signal SIGINT \n");

    else
        printf("Caught Signal \n");

    if (exitTry) {
        printf("Forced exit \n");
        raise(9);
    } else {
        exitTry++;
        BeginExit();
    }

}
#endif

//*****************************************************************************
// 		  NAME : void CExitSync::WaitExitSignal(void)
// DESCRIPTION : Wait for objects to be signalled
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CExitSync::WaitExitSignal(void)
{
#ifdef WIN32
    WaitForSingleObject(hExitAppEvent, INFINITE);
    CloseHandle(hExitAppEvent);
#else
    sem_wait(&hExitAppEvent);
#endif
}

//*****************************************************************************
// 		  NAME : void CExitSync::ForceExit(void)
// DESCRIPTION : Forces the exit of the Application
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CExitSync::ForceExit(void)
{
#ifdef WIN32
    exit(-1);
#else
    raise(9);
#endif
}

//*****************************************************************************
// 		  NAME : void CExitSync::BeginExit(void)
// DESCRIPTION : Set the object to the signalled state
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CExitSync::BeginExit(void)
{
#if defined(WIN32)
    SetEvent(hExitAppEvent);
#else
    sem_post(&(hExitAppEvent));
#endif
}



