/***************************************************************************
* C Source:	     gc_basic_call_model.c
* Description:   This is a demonstration program for a basic call model which performs,
*					  Basic make call and DropCall from Outbound end.It also shows the case 
*					  of simultaneous disconnect.
*					  which executes the program continuously until it encounters signal ctrl+c.
* General Notes: 1. The tab setting used for this file is 3. 
*					  2. Pagination is done with Courier New, 10 point, 64 lines per page
*					  3. For purposes of the demo, the outbound side waits 
*						  1-2 seconds in the connected state, then calls gc_DropCall()
*					  4. PDK as shipped has an issue where occasionally the first makecall may
*						  fail with an invalid state error. The demo program has a recovery
*						  mechanism built in. When connected to a switch, this should not be 
*						  a problem.
*					  5. gc_demo_open_isdn_channel may need to be modified for your environment,
*						  especially if you are using T1.

*
*	%created_by:	fiedorom %
*	%date_created: Tue May 25 11:21:03 2004 %
*
***************************************************************************/
/**********************************************************************
*  Copyright (C) 2000-2002 Intel Corporation.
*  All Rights Reserved
*
*  All names, products, and services mentioned herein are the 
*  trademarks or registered trademarks of their respective 
*  organizations and are the sole property of their respective owners.
**********************************************************************/
#ifndef lint
static char *_csrc = "@(#) %filespec: gc_basic_call_model.c-28 %  (%full_filespec: gc_basic_call_model.c-28:csrc:gc#1 %)";
#endif

/* OS Header Files */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <time.h>
#include <signal.h>
#include <sys/timeb.h>

/* OS Socket API Headers */
#ifdef _WIN32
#include <winsock2.h>
#else
#include <unistd.h>
#include <netdb.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#endif

/* Dialogic Header Files */
#include <gcip.h>
#include <gclib.h>
#include <gcisdn.h>
#include <srllib.h>

/* Macro Definitions */


#define MAXCHAN			120		/* Maximum number of channels that can be opened - 2 E1 spans	*/
#define MAX_DEVNAME		64		/* Maximum length of device name											*/
#define MAX_CODECNAME	20		/* Maximum length of codec name											*/
#define MAX_STRING_SIZE 1000	/* Maximum length of string that to write on log file			*/
									/* Keep large																	*/
#define MAX_CALLS			2		/* Maximum number of calls per channel */
#define MIN_CONNECTED_TIME	1 /* Minimum connected time, artifact of test program				*/
									/* and most likely will not be needed for a production program */

#define	MIN_MAKECALL_RETRY_TIME	5	/* min time to wait before retrying makecall under		*/
									/* error conditions															*/
#define MAXCODECS			10	/* Maximum number of codecs per VoIP channel */
#define MAX_VADSTR		10	/* Maximum length of VAD string, e.g. "ON", "OFF", "N/A" */


/* Logging defines */

#define NON_GC_APIERR	0	/* Had error on non-GC API function										*/
#define EVENT				1	/* An event																		*/
#define GC_APICALL		2	/* GC API funtion call														*/
#define GC_APIERR			3	/* Had error on GC API function											*/
#define STATE				4	/* STATE message																*/
#define GC_RESULT_INFO	5	/* want GC result information												*/
#define MISC				6	/* None of the above															*/
#define MISC_WITH_NL		7	/* None of the above with new line codes								*/

/* Inbound or Outbound Direction of device */

#define DIR_IN				1
#define DIR_OUT			2

/* Miscellaneous declarations */

#define YES					1
#define NO					0

#define ALL_DEVICES	  -1		/* This value is used to write the */
										/* miscellaneous statements into all log files */ 
#define OUT_OF_RANGE	  -1		/* callindex is out of range */

/*
 * Technology Types
 */
#define E1ORT1_CAS		1
#define ISDN				2							/* includes both E1 and T1 ISDN */
#define ANALOG				3
#define SS7					4
#define H323				5
#define SIP					6

/* Global variables */
static char		logfile[] = "gc_basic_call_model";		/* log filename without the .log	 */
static char		cfgfile[] = "gc_basic_call_model.cfg"; /* config filename */

static int		num_devices;		/* Number of devices loaded from cfg file */
static int		interrupted = NO;	/* Flag for user interrupted signals		 */
static int		reset_linedevs_left_to_complete = 0; 

/* Data structure which stores call information for each line device */
typedef struct {
	CRN					crn;							/* GlobalCall call handle */
	int					call_state;					/* state of first layer state machine */
	int					dropcall_active;			/* Flag for simultaneous disconnect check	*/
} CALL;

/* TX and RX codecs for each line device although only used for H.323 or SIP */
typedef struct {
	char					txcodec[MAX_CODECNAME];	/* TX codec type (H.323 or SIP only)					*/
	int					txrate;						/* TX codec rate (H.323 or SIP only)					*/
	char					txVAD[MAX_VADSTR];		/* TX VAD enabled/disabled (H.323 or SIP only)		*/
	char					rxcodec[MAX_CODECNAME];	/* RX codec type (H.323 or SIP only)					*/
	int					rxrate;						/* RX codec rate (H.323 or SIP only)					*/
	char					rxVAD[MAX_VADSTR];		/* RX VAD enabled/disabled (H.323 or SIP only)		*/
} CODEC;

/* Data structure which stores all information for each line device */
static struct channel {
	LINEDEV				ldev;							/* GlobalCall line device handle							*/
	LINEDEV				mediah;						/* Media handle - only for H323                    */
	CALL					call[MAX_CALLS];
	int					index;						/* Device index as loaded from cfg file				*/
	int					direction;					/* Inbound(WaitCall will be issued)
																or outbound(MakeCall will be issued)				*/
	int					blocked;						/* YES or NO													*/
	char					netdevice[MAX_DEVNAME]; /* Network device name										*/
	char					protname[MAX_DEVNAME];	/* Protocol name												*/
	char					voicedevice[MAX_DEVNAME]; /* Voice device name										*/ 
	char					devname[MAX_DEVNAME];	/* Argument to gc_OpenEx() function						*/
	char					destination_num[MAXPHONENUM]; /* Phone													*/
	int					num_codecs;					/* The No. of codecs specified (VoIP device only) 	*/
	CODEC					codec[MAXCODECS];			/* Codecs (VoIP device only)								*/
	int					numb_calls;					/* The No of calls on the device							*/
	int					waitcall_active;			/* Flag for waitcall function for issuing only once*/
	int					makecall_active;
	int					resetlinedev_active;		/* Necessary since must ignore call related GC		*/
															/* events with gc_ResetLineDev() active				*/
	int					resetlinedev_required_after_unblocked; /* for ISDN									*/
															/* and some error conditions								*/
													
	int					makecall_timeout;			/* necessary since not all technologies support		*/
															/* async makecall timeout and they might be			*/
															/* different for different technologies				*/
	GC_MAKECALL_BLK	makecallblk;				/* Variable for MAKECALL block							*/
	FILE					*log_fp;						/* logfile Pointer											*/
	int					techtype;					/* technology type: E1ORT1_CAS, ISDN, ANALOG,		*/

	time_t				makecall_time;				/* time before which makecall is not allowed       */
															/* this is to add robustness to the app				*/
															/* if makecall fails which it might under some		*/
															/* circumstances, but will work if try later			*/
															/* if this value is 0, then OK to do now           */

	/* this variable is in so know when to drop a call */
	/* it is an artifact of the demo program and 	   */
	/* and should not be used in production level 		*/
	/* programs														*/
	time_t				dropcall_time;				/* time to drop the call on the outbound side      */
															/* format is result of time() function call        */
															/* 0 <=> no dropcall required                      */
	time_t				intercall_time;			/* Time when the call was released. The next call  */
                                             /* would be made after intercall_time +            */
                                             /* intercall_delay seconds.                        */
   int               intercall_delay;        /* Inter-call delay in milliseconds                */
} port[MAXCHAN];

/* codec translation table */
typedef struct {
	char*			codecname;
	int			codecvalue;
	int			isVADsupported;
} CODECXLAT;

static const CODECXLAT codecxlat[] = {
	{"G.711Alaw",	GCCAP_AUDIO_g711Alaw64k,			NO},
	{"G.711ulaw",	GCCAP_AUDIO_g711Ulaw64k,			NO},
	{"G.723_5.3k",	GCCAP_AUDIO_g7231_5_3k,				YES},
	{"G.723_6.3k",	GCCAP_AUDIO_g7231_6_3k,				YES},
	{"G.726_16k",	GCCAP_AUDIO_g726_16k,				YES},
	{"G.726_24k",	GCCAP_AUDIO_g726_24k,				YES},
	{"G.726_32k",	GCCAP_AUDIO_g726_32k,				YES},
	{"G.726_40k",	GCCAP_AUDIO_g726_40k,				YES},
	{"G.729",		GCCAP_AUDIO_g729		,				NO},
	{"G.729A",		GCCAP_AUDIO_g729AnnexA,				NO},
	{"G.729B",		GCCAP_AUDIO_g729wAnnexB,			YES},
	{"G.729AB",		GCCAP_AUDIO_g729AnnexAwAnnexB,	YES},
	{"GSM_FULL",	GCCAP_AUDIO_gsmFullRate,			YES},
	{"T.38UDP",		GCCAP_DATA_t38UDPFax,				NO},
	{"DONT_CARE",	GCCAP_dontCare,						NO}
};

static int		in_calls, out_calls;					/* Counters for Inbound calls and Outbound calls	*/
static int		gc_started = NO;						/* Flag is set if gc_Start() is Success				*/
time_t			start_time;								/* To note down the start time for the test			*/

/* Function prototype(s) */

static void		init_srl_mode(void);							/* inits SRL mode */
static void		load_config_file(void);						/* Loads Configuration File									*/
static void		drop_outbound_calls_if_required(void); /* used by demo to make repeated calls */
static void		make_calls_if_required(void);				/* used by demo to handle makecall error conditions 	*/
static void		printandlog(int index, int log_type,	/* Prints and logs												*/ 
								METAEVENT *metavent, char *msg_str, int callindex);
static void		gc_demo_makecall(int index);				/* Function for gc_MakeCall()									*/ 
static void		exitdemo(int exitlevel);					/* Exit routine													*/
static void		initiate_exit(void);
static void		intr_hdlr(void);								/* For handling user interrupted signals					*/
static void		set_calling_num(struct channel *pline, char *phone_num);
static void		enable_alarm_notification(struct channel *pline);/* enable alarm notification                */
static void		gc_demo_open_isdn_channel(int index);	/* Technology specific initialization for ISDN			*/
static void		gc_demo_open_E1orT1_channel(int index);/* Technology specific initialization for PDK			*/ 
static void		gc_demo_open_ss7_channel(int index);	/* Technology specific initialization for SS7			*/
static void		gc_demo_open_analog_channel(int index);/* Technology specific initialization for Analog		*/
static void		gc_demo_open_H323_channel(int index);	/* Technology specific initialization for H323  		*/
static void		gc_demo_open_SIP_channel(int index);	/* Technology specific initialization for SIP   		*/
static void		gc_set_channel_codecs(int index);		/* VoIP specific initialization of codecs					*/
static void		open_all_devices(void);						/* Function invokes Technology specific initilization 
																			functions to open all devices								*/
static void		open_device(int index);						/* Function invokes Technology specific initilization 
																			functions to open specific device						*/
static void		inbound_application(struct channel *pline); /* PLACE HOLDER FOR YOUR APPLICATION CODE			*/
static void		outbound_application(struct channel *pline); /* PLACE HOLDER FOR YOUR APPLICATION CODE			*/
static void		print_alarm_info(METAEVENTP metaeventp, struct channel *pline);
static void		process_event(void);							/* Function handles the GlobalCall Events					*/
static void		process_connected_event(struct channel *pline, METAEVENT *metaeventp);
static void		process_disconnected_event(struct channel *pline, METAEVENT *metaeventp);
static void		process_offered_event(METAEVENT *metaeventp, struct channel *pline);
static void		process_call_related_inbound_event(struct channel *pline, METAEVENT *metaeventp); /* Function handles the GlobalCall Events for inbound */
static void		process_call_related_outbound_event(struct channel *pline, METAEVENT *metaeventp); /* Function handles the GlobalCall Events for outbound */
static int		open_all_logfiles(void);					/* Function to open the logfiles								*/
static void		append_call_state_name(char *buffer, int index, int callindex);	/* append call state name to buffer	*/
static void		format_error_info(char *dest_buffer);		/* formats error information									*/
static void		format_result_info(int index, METAEVENT *metaeventp, /* formats event result information									*/
											char *dest_buffer);
static int		existCRN(CRN crn, struct channel *pline);	/* Retrieve CRN if available */

/********************************************************************************
 *			NAME: main ()
 *DESCRIPTION: This is the entry point. 
 *					It loads the configuration file.
 *					Take cares the	 Event Handling process.
 *					Starts GC Library.
 *					Opens Configured Devices.
 *					Waits for user interrupted signal (ctrl+c).
 *					Whenever an event is received on any of the opened 
 *					devices, Depending upon the event received the proper 
 *					action will be taken.
 *		  INPUT: None
 *		RETURNS: 0 if success else -1
 ********************************************************************************/
int main(void)
{
	int						i, ret;
	char						str[MAX_STRING_SIZE], str1[MAX_STRING_SIZE];
	GC_CCLIB_STATUSALL	cclib_status_all;

	/* Note down the start time for the test */
	time(&start_time);
	
	/* Give control to Event Handler */ 
#ifdef _WIN32
	signal(SIGINT, (void (__cdecl*)(int)) intr_hdlr);
	signal(SIGTERM, (void (__cdecl*)(int)) intr_hdlr);
#else
	signal(SIGHUP, (void (*)()) intr_hdlr);
	signal(SIGQUIT, (void (*)()) intr_hdlr);
	signal(SIGINT, (void (*)()) intr_hdlr);
	signal(SIGTERM, (void (*)()) intr_hdlr);
#endif

	init_srl_mode();									/* set SRL mode to ASYNC, polled */

	/* Load Configuration file */
	load_config_file();

	/* Function to open all the logfiles */
	open_all_logfiles();
	if (num_devices == 0) {
		printf("No devices enabled - please modify gc_basic_call_model.cfg to enable desired technology\n");
		exitdemo(1);
	}
	printandlog(ALL_DEVICES, MISC, NULL, "********** GC DEMO - BASIC CALL MODEL ***********\n", 0);
	
	/* Start GlobalCall */
	if (gc_Start(NULL) != GC_SUCCESS) { 
		sprintf(str, "gc_Start(startp = NULL) Failed");
		printandlog(ALL_DEVICES, GC_APIERR, NULL, str, 0);
		exitdemo(1);
	}
	/* Set the flag gc_started to YES */
	gc_started = YES;
	sprintf(str, "gc_Start(startp = NULL) Success");
	printandlog(ALL_DEVICES, GC_APICALL, NULL, str, 0);


	/* this code is usefule for installation problem debugging */
	/* you may not wish to use this code in production systems */
	if (gc_CCLibStatusEx("GC_ALL_LIB", &cclib_status_all) != GC_SUCCESS) {
		sprintf(str, "gc_CCLibStatusEx(GC_ALL_LIB, &cclib_status_all) Failed");
		printandlog(ALL_DEVICES, GC_APIERR, NULL, str, 0);
		exitdemo(1);
	}	
	strcpy(str, " Call Control Library Status:\n");
	for (i = 0; i < GC_TOTAL_CCLIBS; i++) {
		switch (cclib_status_all.cclib_state[i].state) {
			case GC_CCLIB_CONFIGURED:
				sprintf(str1, "		  %s - configured\n", cclib_status_all.cclib_state[i].name);
				break;

			case GC_CCLIB_AVAILABLE:
				sprintf(str1, "		  %s - available\n", cclib_status_all.cclib_state[i].name);
				break;

			case GC_CCLIB_FAILED:
				sprintf(str1, "		  %s - is not available for use\n", cclib_status_all.cclib_state[i].name);
				break;

			default:
				sprintf(str1, "		  %s - unknown CCLIB status\n", cclib_status_all.cclib_state[i].name);
				break;


		}
		strcat(str, str1);
	}
	printandlog(ALL_DEVICES, MISC, NULL, str, 0);

	/* open all the GC devices */
	open_all_devices(); 

	/* 
	-- Forever loop where the main work is done - wait for an event or user requested exit
	*/
	for (;;) {
		ret = sr_waitevt(500);					/* 1/2 second */
		if (ret != -1) {							/* i.e. not timeout */
			process_event();
		}

		if (interrupted == NO) {				/* flag set in intr_hdlr() */
			drop_outbound_calls_if_required();/* an artifact of the demo program */
														/* where the outbound side disconnects 1-2 */
														/* seconds after connected */

			make_calls_if_required();			/* handles error conditions on makecall */
		} else {
			initiate_exit();
			if (reset_linedevs_left_to_complete == 0) {
				exitdemo(1);
			}
		}

	}
} /* End of Main */

/****************************************************************
*			NAME: void init_srl_mode(void)
* DESCRIPTION: Inits SR mode
*		RETURNS: None
*	  CAUTIONS: NA
****************************************************************/
static void init_srl_mode(void)
{
/* 
SRL MODEL TYPE - Single Threaded Async, event handler will be called in the Main thread 
*/
#ifdef _WIN32
	int	mode = SR_STASYNC | SR_POLLMODE; 
#else
	int	mode = SR_POLLMODE;
#endif
	

	/* Set SRL mode */
#ifdef _WIN32
	if (sr_setparm(SRL_DEVICE, SR_MODELTYPE, &mode) == -1) {
		printandlog(ALL_DEVICES, NON_GC_APIERR, NULL, "Unable to set to SR_STASYNC | SR_POLLMODE", 0);
		exitdemo(1);
	}
	printandlog(ALL_DEVICES, MISC, NULL, "SRL Model Set to SR_STASYNC | SR_POLLMODE", 0);
#else
	if (sr_setparm(SRL_DEVICE, SR_MODEID, &mode) == -1) {
		printandlog(ALL_DEVICES, NON_GC_APIERR, NULL, "Unable to set mode ID to SR_POLLMODE ", 0);
		exitdemo(1);
	}
	
	printandlog(ALL_DEVICES, MISC, NULL, "SRL mode ID set to SR_POLLMODE", 0);
#endif
}

/****************************************************************
*			NAME: open_all_logfiles(void)
* DESCRIPTION: Function to Open the Logfiles and put file pointers in file 
*					pointer field of port array. Later depending upon the index the
*					particular file can be accessed to dump the data.
*					If the network device name is present, then the log file
*					uses the network device name
*					else it uses the voice device name
*		RETURNS: returns 1 if successful
*	  CAUTIONS: Assumes device name includes network device name or a voice device name
****************************************************************/
static int open_all_logfiles(void)
{	
	char		temp_filename[MAX_DEVNAME], temp_str[MAX_STRING_SIZE];
	char		*i;
	int		index;

	for (index = 0; index < num_devices; index++) {
		/* build up temp_filename */
		strcpy(temp_filename, logfile);

		/* case 1: if dti name exists, use it */
		if (strcmp(port[index].netdevice, "NONE") != 0) {
			/* IP has different "dti" name than NON-IP */
			if (port[index].netdevice[0] == 'i') {
				sscanf(port[index].devname, ":N_ipt%[^:]", temp_str); 
			} else {
				sscanf(port[index].devname, ":N_dti%[^:]", temp_str); 
			}
			strcat(temp_filename,  "_");
			strcat(temp_filename, temp_str);
		}
		/* case 2: if voice name exists, use it */
		else if (strcmp(port[index].voicedevice, "NONE") != 0) {
			i = (char *)strstr(port[index].voicedevice, "B");
			strcpy(temp_str, i);
			strcat(temp_filename,  "_");
			strcat(temp_filename, temp_str);
		} else {
			printf("Missing both DTI and Voice resource in config file device name (%s)\n",
					port[index].devname);
			exitdemo(1);
		}		
		strcat(temp_filename, ".log");

		if ((port[index].log_fp = fopen(temp_filename, "w")) == NULL) {
			printf("\nCannot open log file %s for write\n", temp_filename);
			perror("Reason is: ");
			exitdemo(1);
		}
		printf("%s successfully opened\n", temp_filename); 
		
	} /* End of for loop */
	return 1;
}

/****************************************************************
*			NAME: inbound_application(struct channel *pline)
* DESCRIPTION: Place holder for your inbound application code in the connected state
*		  INPUT: pline - pointer to port data structure
*		RETURNS: NA
*	  CAUTIONS: none
****************************************************************/
static void inbound_application(struct channel *pline)
{
	/* Place holder - your app code goes here */
}
/****************************************************************
*			NAME: outbound_application(struct channel *pline)
* DESCRIPTION: Place holder for your outbound application code in the connected state
*		  INPUT: pline - pointer to port data structure
*		RETURNS: NA
*	  CAUTIONS: none
****************************************************************/
static void outbound_application(struct channel *pline)
{
	char			str[MAX_STRING_SIZE];

	/*************************************************************************/
	/* YOUR APP WOULD PUT ITS WORK HERE													 */
	/* however for purposes of the demo, stay in the connected state			 */
	/* at least one second. This is because not all technologies and all		 */
	/* protocols support dropcall immediately after connected in the default */
	/* shipped configuration																 */
	/*************************************************************************/
	time(&pline->dropcall_time);
	pline->dropcall_time += MIN_CONNECTED_TIME + 1; /* add 1 because the granularity is 1 sec */
	sprintf(str, "gc_DropCall() will be issued in %d-%d seconds", MIN_CONNECTED_TIME, MIN_CONNECTED_TIME + 1);
	printandlog(pline->index, MISC, NULL, str, 0);
}

/****************************************************************
*			NAME: drop_outbound_calls_if_required(void)
* DESCRIPTION: This drops outbound side calls if required
*					For purposes of the demo, a call stays in the connected
*					state state for a second or two, then is dropped by the outbound side
*					Your application does not need this code
*					and hence it is not optimized
*		RETURNS: NA
*	  CAUTIONS: none
****************************************************************/
static void drop_outbound_calls_if_required(void)
{
	int					index;
	struct channel		*pline;
	time_t				current_time;								/* in seconds */
	char					str[MAX_STRING_SIZE];
	int					callindex;

	/* get the current time in seconds */
	time(&current_time);

	/* check all devices for an "expired" drop call time */
	/* assumes caller of gc_ResetLinedev() set dropcall_time = 0 */
	for (index = 0; index < num_devices; index++) {
		pline = &port[index];				
		/* check if time tripped */
		if (pline->dropcall_time && (pline->dropcall_time <= current_time)) {
			/* drop call is no longer required */
			pline->dropcall_time = 0;

			/* Retrieve CRN  - assumes only 1 call is active */
			callindex = existCRN(-1, pline);
			if (callindex == OUT_OF_RANGE) {
				/* CRN not found, so log error and return from function. */
				sprintf(str, "CRN not found");
				printandlog(pline->index, MISC_WITH_NL, NULL, str, 0);
				return;
			}

			if (gc_DropCall(pline->call[callindex].crn, GC_NORMAL_CLEARING, EV_ASYNC) != GC_SUCCESS) {
				sprintf(str, "gc_DropCall(crn=0x%lx, cause=GC_NORMAL_CLEARING, mode=EV_ASYNC) Failed", pline->call[callindex].crn);
				printandlog(pline->index, GC_APIERR, NULL, str, 0);
				printandlog(pline->index, STATE, NULL, " ", callindex);
				exitdemo(1);
			}
			printandlog(index, MISC_WITH_NL, NULL, "********* Dropping outbound call from drop_outbound_calls_if_required() *********", 0);
			sprintf(str, "gc_DropCall(crn=0x%lx, cause=GC_NORMAL_CLEARING, mode=EV_ASYNC) Success", pline->call[callindex].crn);
			printandlog(pline->index, GC_APICALL, NULL, str, 0);

			/* Set the dropcall flag to YES on the related channel */
			pline->call[callindex].dropcall_active = YES;
		}
	}
}


/****************************************************************
*			NAME: make_calls_if_required(void)
* DESCRIPTION: This procedure sees if there are any calls that need to be made
*					handles some makecall error conditions
*		RETURNS: NA
*	  CAUTIONS: none
****************************************************************/
static void make_calls_if_required(void)

{
	int					index;
	struct channel		*pline;
	time_t				current_time;								/* in seconds */

	time(&current_time);												/* get the current time in seconds */

	/* check all devices for an "expired" make call time */
	for (index = 0; index < num_devices; index++) {
		pline = &port[index];				
		if (pline->makecall_time && (pline->makecall_time <= current_time)) {			/* check if time tripped */
			printandlog(index, MISC, NULL, "makecall timer tripped - will retry the makecall", 0);
			pline->makecall_time = 0;								/* makecall request will now be met */
			gc_demo_makecall(index);								/* attempt to make the call again */
		}
		if (pline->intercall_time && (pline->intercall_time <= current_time)) {			/* check if time tripped */
			printandlog(index, MISC, NULL, "Inter-Call timer tripped - New call will now be attempted", 0);
			pline->intercall_time = 0;								/* makecall request will now be met */
			gc_demo_makecall(index);								/* attempt to make the call again */
		}

	}
}

/****************************************************************
*			NAME: process_event(void)
* DESCRIPTION: Function to handle the GlobalCall Events 
*		RETURNS: NA
*	  CAUTIONS: none
****************************************************************/
static void process_event(void)
{
	METAEVENT			metaevent;
	struct channel		*pline;
	char					str[MAX_STRING_SIZE];
	int					index;				/* Device index */
	int					evttype;
	GC_INFO				t_info;	  			/* Structure which stores information about GC related errors */ 
	int					callindex;			/* CAUTION: once initialized, do not change */
													/* the value in this subroutine! */
	int					temp_callindex;	/* use the temporary callindex instead */
	
	/* Populate the metaEvent structure */
	if(gc_GetMetaEvent(&metaevent) != GC_SUCCESS)
	{
		/* serious problem - should never fail */
		printandlog(ALL_DEVICES, GC_APIERR, NULL, "gc_GetMetaEvent() failed", 0);
		exitdemo(1);
	}

	/* process GlobalCall events */
	if ((metaevent.flags & GCME_GC_EVENT) == 0)	{
		/*********************************************************************/
		/* Your application might have non-GC events , e.g. voice or fax		*/
		/* events that need to be processed. That would be done here.			*/
		/* However for the purposes of the demo, non_GC events are only		*/
		/* logged. Most likely you will need to do some sort of a				*/
		/* search of the port array to find the associated data structure		*/
		/* for this event. Since the event is not a GC event,						*/
		/* gc_GetUsrAttr() cannot help find the associated GC data structure */
		/*********************************************************************/
		sprintf(str, "Received a non-GC Event 0x%lx\n", metaevent.evttype);
		printandlog(ALL_DEVICES, MISC, NULL, str, 0);
		return;												/* RETURN POINT */
	}
	/**************************************/
	/* when here => processing a GC event */
	/**************************************/
	/* initialize pline to port[index],
	The pointer to this structure was stored in GC
	during gc_OpenEx() */
	pline = (struct channel *) metaevent.usrattr;
	
	/* Retrieve device index */
	index = pline->index;

	evttype = metaevent.evttype;

	/* skip checks if no crn for this event */

	if ((metaevent.crn != 0) && (pline->resetlinedev_active == NO)) {
		/* Assumption: Only GCEV_DETECTED/GCEV_OFFERED will require the CRN
	   	to be inserted into the array. */
		callindex = existCRN(metaevent.crn, pline);
		if (callindex == OUT_OF_RANGE) {
			/* CRN was not found, so insert CRN into the array.
			 * Should only occurs when GCEV_DETECTED or GCEV_OFFERED event is received.
			 * Note: GCEV_DETECTED is not enabled, but this code will support it if
			 * if you enable this event */
			if ((evttype == GCEV_DETECTED) || (evttype == GCEV_OFFERED)) {
				for (callindex = 0; callindex < MAX_CALLS; callindex++) {
					if (pline->call[callindex].crn == 0) {
						/* crn is not found, insert into crn array. */
						pline->call[callindex].crn = metaevent.crn;
						break;
					}
				}
			} else {
				/* Error case: should not happen */
				sprintf(str, " Received an event(0x%x) with an invalid CRN(0x%lx)", evttype, metaevent.crn);
				printandlog(index, MISC_WITH_NL, NULL, str, 0);
				return;
			}
		}
	} else {
		callindex = 0;						/* some printing routines use callindex */
	}
	/**********************************************************************/
	/* CAUTION: at this point callindex is initialized, do not change it! */
	/**********************************************************************/

	printandlog(index, MISC_WITH_NL, NULL, "********* Received a GC event *********", 0);
	printandlog(index, EVENT, NULL, GCEV_MSG(evttype), 0);
	printandlog(index, STATE, NULL, " is the current GC call state ", callindex);

	/* 1st handle the generic events */
	switch (evttype) 
	{
		case GCEV_OPENEX:
			printandlog(index, MISC, NULL, "GCEV_OPENEX received", 0);
			/* Although this demo currently only opens H.323 and SIP devices asynchronously */
			if ((pline->techtype == H323) || (pline->techtype == SIP)) {

				/* Enable alarm notification for H.323 or SIP devices which were opened ASYNC */
				enable_alarm_notification(pline);

				/* Set DTMF mode to inband for SIP as default of Alphanumeric mode
				 * valid only for H.323 devices
				 */
				if (pline->techtype == SIP) {
					GC_PARM_BLK			  *parmblkp = NULL;

					gc_util_insert_parm_val(&parmblkp, IPSET_DTMF, IPPARM_SUPPORT_DTMF_BITMASK,
													sizeof(char), IP_DTMF_TYPE_INBAND_RTP);

					if (gc_SetUserInfo(GCTGT_GCLIB_CHAN, port[index].ldev, parmblkp, GC_ALLCALLS) != GC_SUCCESS) {
							sprintf(str, "gc_SetUserInfo(linedev=%ld) Failed configuring DTMF mode", port[index].ldev);
							printandlog(index, GC_APIERR, NULL, str, 0);
							exitdemo(1);
					}

					sprintf(str, "gc_SetUserInfo(linedev=%ld) Success - DTMF mode is inband", port[index].ldev);
					printandlog(index, GC_APICALL, NULL, str, 0);
					gc_util_delete_parm_blk(parmblkp);
				}
			}
			break;

		/* This demo only opens H.323 and SIP devices asynchronously */
		case GCEV_OPENEX_FAIL:
			printandlog(index, MISC, NULL, "GCEV_OPENEX_FAIL received", 0);
			printandlog(index, GC_RESULT_INFO, &metaevent, "Reason for GCEV_OPENEX_FAIL ", 0);
			exitdemo(1);
			break;


		case GCEV_ALARM:
			print_alarm_info(&metaevent, pline);
			break;

		case GCEV_FATALERROR:
			printandlog(index, GC_RESULT_INFO, &metaevent, "Reason for fatalerror ", 0);
			printandlog(index, STATE, NULL, " ", callindex);

			if (gc_ResultInfo(&metaevent, &t_info) < 0) {
				sprintf(str, "gc_ResultInfo() call failed");
				printandlog(index, GC_APIERR, NULL, str, 0);
				exitdemo(1);
			}

			if (t_info.gcValue == GCRV_RESETABLE_FATALERROR) {
				pline->dropcall_time = 0;								/* in case on */
				pline->resetlinedev_active = YES;					/* reset linedevice is on as a background process */
				/* do nothing else= wait till the GCEV_RESETLINEDEV completion event arrives */
			} else if (t_info.gcValue == GCRV_RECOVERABLE_FATALERROR) {
				/* do a close and open on recoverable, fatal errors */
				printandlog(index, MISC, NULL, " Recoverable fatal error occurred - closing device and re-opening it", 0);
				if (gc_Close(port[index].ldev) < 0) {
					printandlog(index, GC_APIERR, NULL, "gc_Close() failed", 0);
					exitdemo(1);
				}
				open_device(index);

			} else {
				printandlog(index, MISC, NULL, " Non-recoverable fatal error occurred", 0);
				exitdemo(1);
			}
			break;				

		case GCEV_BLOCKED:
			pline->blocked = YES;						/* Nothing to do when a blocked event occurs */
			break;
			
		case GCEV_UNBLOCKED:
			/* blocked/unblocked do not cause state transitions */
			pline->blocked = NO;
			if (pline->resetlinedev_required_after_unblocked == YES) {
				pline->resetlinedev_required_after_unblocked = NO;

				if (gc_ResetLineDev(port[index].ldev, EV_ASYNC) != GC_SUCCESS) {
					sprintf(str, "gc_ResetLineDev(linedev=%ld, mode=EV_ASYNC) Failed", port[index].ldev);
					printandlog(index, GC_APIERR, NULL, str, 0);
					exitdemo(1);
				}
				pline->resetlinedev_active = YES;
				sprintf(str, "gc_ResetLineDev(linedev=%ld, mode=EV_ASYNC) Success", port[index].ldev);
				printandlog(index, GC_APICALL, NULL, str, 0);
				/* nothing more can do until the completion event arrives */
				break;
			}

			if (pline->waitcall_active == YES) {
				/* waitcall already active - do not call it again */
				break;
			}
			if (pline->resetlinedev_active == YES) {
				/* The line device is being reset */
				break;
			}
			/* FALL THROUGH ON PURPOSE */

		case GCEV_RESETLINEDEV:
			if (evttype == GCEV_RESETLINEDEV) {
				/* Reset call related flags and variables */
				pline->waitcall_active = NO;
				pline->makecall_active = NO;
				pline->resetlinedev_active = NO;
				for (temp_callindex = 0; temp_callindex < MAX_CALLS; temp_callindex++) {
					/* reset internal variables to that of beginning */
					pline->call[temp_callindex].call_state = GCST_NULL;
					pline->call[temp_callindex].crn = 0;
					pline->call[temp_callindex].dropcall_active = NO;
				}
			}

			if ((evttype == GCEV_RESETLINEDEV) 
			&& (reset_linedevs_left_to_complete != 0)) {		/* tests if processing shutdown request */
				reset_linedevs_left_to_complete--;
				sprintf(str, "One less GCEV_RESETLINEDEV to wait for, %d left",
					reset_linedevs_left_to_complete);
				printandlog(ALL_DEVICES, MISC, NULL, str, 0);

			}

			/* must further process the event if unblocked */
			if ((interrupted == NO) && (pline->blocked == NO)) {
				if (pline->direction == DIR_IN) {
					/* Note: by definition, waitcall is not active at this point */
					if (pline->blocked == NO) {
						if (gc_WaitCall(pline->ldev, NULL, NULL, -1, EV_ASYNC) != GC_SUCCESS) {
							sprintf(str, "gc_WaitCall(linedev=%ld, crnp=NULL, waittime=0, mode=EV_ASYNC) Failed", pline->ldev);
							printandlog(index, GC_APIERR, NULL, str, 0);
							exitdemo(1);
						}
						pline->waitcall_active = YES;
						sprintf(str, "gc_WaitCall(linedev=%ld, crnp=NULL, waittime=0, mode=EV_ASYNC) Success", pline->ldev);
						printandlog(index, GC_APICALL, NULL, str, 0);
					}
				} else {
					/* Make another call if possible */
					gc_demo_makecall(index); 
				}
			}
			break;

		case GCEV_TASKFAIL:
			printandlog(index, GC_RESULT_INFO, &metaevent, "Reason for taskfail ", 0);
			printandlog(index, STATE, NULL, " ", callindex);

			/* best can do is a reset linedevice */
			/* if reset lindevice is already active, then retry it */
			if (gc_ResetLineDev(port[index].ldev, EV_ASYNC) == GC_SUCCESS) {
				pline->dropcall_time = 0;								/* in case on */
				pline->call[callindex].dropcall_active = NO;		/* in case on */
				pline->makecall_active = NO;							/* in case on */
				pline->resetlinedev_active = YES;
				sprintf(str, "gc_ResetLineDev(linedev=%ld, mode=EV_ASYNC) Success", port[index].ldev);
				printandlog(index, GC_APICALL, NULL, str, 0);
			} else {
				sprintf(str, "gc_ResetLineDev(linedev=%ld, mode=EV_ASYNC) Failed", port[index].ldev);
				printandlog(index, GC_APIERR, NULL, str, 0);
				if (pline->blocked == NO) {							/* If no alarms are active */
					exitdemo(1);											/* Then should never get here */
				}
				/* when here, resetlinedev failed and in the blocked state */
				/* most likely due to an alarm being active */
				if (pline->call[callindex].crn) {					/* is there an active call? */
					if (pline->call[callindex].dropcall_active == NO) {					/* drop and release the call if can */
						if (gc_DropCall(pline->call[callindex].crn, GC_NORMAL_CLEARING, EV_ASYNC) == GC_SUCCESS)
						{
							sprintf(str, "gc_DropCall(crn=0x%lx, cause=GC_NORMAL_CLEARING, mode=EV_ASYNC) Success", pline->call[callindex].crn);
							printandlog(pline->index, GC_APICALL, NULL, str, 0);
							pline->call[callindex].dropcall_active = YES;
						} else {
							sprintf(str, "gc_DropCall(crn=0x%lx, cause=GC_NORMAL_CLEARING, mode=EV_ASYNC) Failed", pline->call[callindex].crn);
							printandlog(pline->index, GC_APIERR, NULL, str, 0);
							printandlog(pline->index, STATE, NULL, " ", callindex);
						}
					}
				}
				pline->resetlinedev_required_after_unblocked = YES;		/* try again later */
			}
			break;

		case GCEV_DISCONNECTED:
			if (pline->resetlinedev_active == NO) {
				process_disconnected_event(pline, &metaevent);	/* process the disconnected event */
			}
			break;

		default:
			if (pline->resetlinedev_active == NO) {	/* ignore all call related events with reset linedevice active */
				if (pline->direction == DIR_IN) {
					process_call_related_inbound_event(pline, &metaevent);
				} else {
					process_call_related_outbound_event(pline, &metaevent);
				}
			}
			break;
	}

	printandlog(index, STATE, NULL, " is the new GC call state after processing the event", callindex);
}

/****************************************************************
*			NAME: void	print_alarm_info(METAEVENTP metaeventp, struct channel *pline)
* DESCRIPTION: Prings alarm information
*		 INPUTS: metaeventp - pointer to the alarm event
*              pline - pointer to the channel data structure
*		RETURNS: NA
*	  CAUTIONS: Assumes already known to be an alarm event
****************************************************************/
static void	print_alarm_info(METAEVENTP metaeventp, struct channel *pline)
{
	long					alarm_number;
	char					*alarm_name;
	unsigned long		alarm_source_objectID;
	char					*alarm_source_object_name;
	char					str[MAX_STRING_SIZE];

	if (gc_AlarmNumber(metaeventp, &alarm_number) != GC_SUCCESS)
	{
		sprintf(str, "gc_AlarmNumber(...) FAILED");
		printandlog(pline->index, GC_APIERR, NULL, str, 0);
		printandlog(pline->index, STATE, NULL, " ", 0);
		exitdemo(1);
	}
	if (gc_AlarmName(metaeventp, &alarm_name) != GC_SUCCESS)
	{
		sprintf(str, "gc_AlarmName(...) FAILED");
		printandlog(pline->index, GC_APIERR, NULL, str, 0);
		printandlog(pline->index, STATE, NULL, " ", 0);
		exitdemo(1);
	}

	if (gc_AlarmSourceObjectID(metaeventp, &alarm_source_objectID) != GC_SUCCESS)
	{
		sprintf(str, "gc_AlarmSourceObjectID(...) FAILED");
		printandlog(pline->index, GC_APIERR, NULL, str, 0);
		printandlog(pline->index, STATE, NULL, " ", 0);
		exitdemo(1);
	}


	if (gc_AlarmSourceObjectName(metaeventp, &alarm_source_object_name) != GC_SUCCESS)
	{
		sprintf(str, "gc_AlarmSourceObjectName(...) FAILED");
		printandlog(pline->index, GC_APIERR, NULL, str, 0);
		printandlog(pline->index, STATE, NULL, " ", 0);
		exitdemo(1);
	}

	sprintf(str, "Alarm %s (0x%lx) occurred on ASO %s (%d)",
				alarm_name, alarm_number, alarm_source_object_name, (int) alarm_source_objectID);

	printandlog(pline->index, MISC, NULL, str, 0);
}



/****************************************************************
*			NAME: process_disconnected_event(struct channel *pline, METAEVENT *metaeventp)
* DESCRIPTION: Function to process a disconnected event
*		 INPUTS: pline - pointer to entry in port table
*					metaeventp - pointer to the current metaevent
*		RETURNS: NA
*	  CAUTIONS: Assumes event logging already done
****************************************************************/
static void process_disconnected_event(struct channel *pline, METAEVENT *metaeventp)
{
	char		str[MAX_STRING_SIZE];
	int		callindex;

	/* Retrieve CRN */
	callindex = existCRN(metaeventp->crn, pline);
	if (callindex == OUT_OF_RANGE) {
		/* CRN not found, so log error and return from function. */
		sprintf(str, "CRN not found");
		printandlog(pline->index, MISC_WITH_NL, NULL, str, 0);
		return;
	}

	switch (pline->call[callindex].call_state)
	{
		case GCST_IDLE:
			/*************************************************************/
			/* this represents a simultaneous disconnect						 */
			/* do nothing as gc_DropCall() must have already been issued */
			/* to reach this point!													 */
			/*************************************************************/
			break;

		case GCST_NULL:
			/* In case have a disconnect before dialing is reached */
			if (pline->makecall_active == NO) {
				/* this represents an error */
				printandlog(pline->index, MISC, NULL, "Received GCEV_DISCONNECTED in NULL call state", 0); 

				/* however, there is a PT - 24402 - which indicates that the inbound side */
				/* may under alarm conditions receive a disconnected event */
				/* even though the call has not been offered */
				/* the other work-around is to enable the detected state, but */
				/* for purposes of showing a second work-around, we will "create" the call */
				/* and then drop and release it */
				pline->call[callindex].crn = metaeventp->crn;					/* "create the call */
																		/* the call state will be init in default */

				/* break; - when PT24402 is fixed, put this back in and pull the call "creation" */
			}
			/* fall through on purpose - need to do a dropcall and release call since there is a call active */
			/* may have reached this condition if there is a problem before dialing */
			/* or the inbound side has not enabled detection of events and the disconnected */
			/* event arrives (usually due to alarm conditions) without the app previously */
			/* being informed the call exists */

		default:
			/* GCEV_DISCONNECTED is allowed in all other call states */
			/* If dropcall is already active, don't reissue it */
			/* this represents a simultaneous disconnect */
			if(pline->call[callindex].dropcall_active == NO)
			{
				if (gc_DropCall(pline->call[callindex].crn, GC_NORMAL_CLEARING, EV_ASYNC) != GC_SUCCESS)
				{
					sprintf(str, "gc_DropCall(crn=0x%lx, cause=GC_NORMAL_CLEARING, mode=EV_ASYNC) Failed", pline->call[callindex].crn);
					printandlog(pline->index, GC_APIERR, NULL, str, 0);
					printandlog(pline->index, STATE, NULL, " ", callindex);
					exitdemo(1);
				}
				sprintf(str, "gc_DropCall(crn=0x%lx, cause=GC_NORMAL_CLEARING, mode=EV_ASYNC) Success", pline->call[callindex].crn);
				printandlog(pline->index, GC_APICALL, NULL, str, 0);
				pline->call[callindex].dropcall_active = YES;
			}
			pline->call[callindex].call_state = GCST_DISCONNECTED;
			pline->dropcall_time = 0;								/* dropcall is no longer required */
			break;
	}
}

/****************************************************************
*			NAME: process_call_related_inbound_event(struct channel *pline, METAEVENT *metaeventp)
* DESCRIPTION: Function to process inbound GlobalCall event
*		 INPUTS: pline - pointer to entry in port table
*					metaeventp - pointer to the metaevent structure
*		RETURNS: NA
*	  CAUTIONS: Tightly coupled with process_event() for pre-processing
*					Assumes event logging already done
*					Disconnected event already handled
****************************************************************/
static void process_call_related_inbound_event(struct channel *pline, METAEVENT *metaeventp)
{
	int		evttype, index;
	char		str[MAX_STRING_SIZE];
	int		unexpected_event = 0;					/*assume not till proven otherwise */
																/* May be overridden for GCEV_UNBLOCKED */
	int		callindex;

	evttype = metaeventp->evttype;
	index = pline->index;

	/* Retrieve CRN */
	callindex = existCRN(metaeventp->crn, pline);
	if (callindex == OUT_OF_RANGE) {
		/* CRN not found, so log error and return from function. */
		sprintf(str, "CRN not found");
		printandlog(pline->index, MISC_WITH_NL, NULL, str, 0);
		return;
	}

	/* Implement the state machine */
	switch (pline->call[callindex].call_state)
	{
		/************************************/
		/* first come the call setup states */
		/************************************/
		case GCST_NULL:
		{
			switch (evttype)
			{
				case GCEV_OFFERED:
					process_offered_event(metaeventp, pline);
					break;
	
				default:
					unexpected_event = 1;
					break;
			}
			break;
			/* end of case GCST_NULL */
		}

		case GCST_OFFERED:
		{
			if (evttype == GCEV_ACCEPT) {
				pline->call[callindex].call_state = GCST_ACCEPTED;
				

				/* Microsoft Phoenix Version 5.0 Soft SIP phone requires SDP field in OK event so we */
				/* specify coders prior to calling gc_AnswerCall( ) to ensure OK event has SDP field.*/
				/* Though done specifically for Phoenix interopability, this will not affect         */
				/* interoperability with other SIP clients.                                          */
				if(pline->techtype == SIP) {
					gc_set_channel_codecs(pline->index);
				}
				
				/* Note: if both answercall and acceptcall use 0 rings  */
				/* then the outbound side may not get an alerting event */
				/* this behavior is technology/protocol dependent		  */
				if (gc_AnswerCall(pline->call[callindex].crn, 0, EV_ASYNC) != GC_SUCCESS) {
					sprintf(str, "gc_AnswerCall(crn=0x%lx, # of rings=0, mode=EV_ASYNC) Failed", pline->call[callindex].crn);
					printandlog(index, GC_APIERR, NULL, str, 0);
					exitdemo(1);
				}
				sprintf(str, "gc_AnswerCall(crn=0x%lx, mode=EV_ASYNC) Success", pline->call[callindex].crn);
				printandlog(index, GC_APICALL, NULL, str, 0);
			} else {
				unexpected_event = 1;
			}
			break;
		}

		case GCST_ACCEPTED:
		{
			if (evttype == GCEV_ANSWERED) {
				/* Do nothing but change the state to connected. The call will be cleared from
				outbound side. Just wait for Disconnect */
				/* Your application specific code will go here */
				pline->call[callindex].call_state = GCST_CONNECTED;

				/* Increment the total calls on this device */
				pline->numb_calls++;
				/* Increment the total inbound calls on all devices */
				in_calls++;
				inbound_application(pline);			/* YOUR APPLICATION CODE IS CALLED HERE */
			} else {
				unexpected_event = 1;
			}
			break;
		}

		case GCST_CONNECTED:
		{
			/*************************************************/
			/* Depending upon your application, you may or	 */
			/* may not have application specific event		 */
			/* handling here for when in the connected state */
			/* You could also have your app specific code	 */
			/* in process_event()									 */
			/*************************************************/
			unexpected_event = 1;						/* recall that GCEV_DISCONNECTED is handled elsewhere */
			break;
		}

		/*************************************/
		/* now come the call teardown states */
		/*************************************/
		case GCST_DISCONNECTED:
		{
			/* If the GCEV_OFFERED event is observed in this state,
			   then add the code that in located in GCST_IDLE state
				to resolved the issue. */
			if (evttype == GCEV_DROPCALL) {
				pline->call[callindex].dropcall_active = NO;

				/* Call ReleaseCall */
				pline->call[callindex].call_state = GCST_IDLE;

				if (gc_ReleaseCallEx(pline->call[callindex].crn, EV_ASYNC) != GC_SUCCESS) {
					sprintf(str, "gc_ReleaseCallEx(crn=0x%lx, EV_ASYNC) Failed", pline->call[callindex].crn);
					printandlog(index, GC_APIERR, NULL, str, 0);
					exitdemo(1);
				}
				sprintf(str, "gc_ReleaseCallEx(crn=0x%lx, EV_ASYNC) Success", pline->call[callindex].crn);
				printandlog(index, GC_APICALL, NULL, str, 0);
				fflush(stdout);
			} else {
				unexpected_event = 1;
			}
			break;
		}

		case GCST_IDLE:
		{
			/* In most cases, only GCEV_RELEASECALL will be seen in this state,
			   but GCEV_OFFERED can be received after gc_ReleaseCallEx() and
			   before GCEV_RELEASECALL. */
			if (evttype == GCEV_RELEASECALL) {
				pline->call[callindex].call_state = GCST_NULL;
				pline->call[callindex].crn = 0;
			} else if (evttype == GCEV_OFFERED) {
				process_offered_event(metaeventp, pline);
			} else {
				unexpected_event = 1;
			}
			break;
		}

		default:
		{
			/* Should never get here */
			printandlog(index, MISC, NULL, "Invalid state in process_call_related_inbound_event()", 0); 
			printandlog(index, STATE, NULL, " ", callindex);
			break;
		}
	}

	if (unexpected_event && (evttype != GCEV_UNBLOCKED)) {
		printandlog(index, GC_RESULT_INFO, metaeventp, "Event was unexpected in the current state", 0);
		printandlog(index, STATE, NULL, " ", callindex);
	}
}

/****************************************************************
*			NAME: process_offered_event(struct channel *pline)
* DESCRIPTION: Function to process an offered event while in the NULL state
*		 INPUTS: pline - pointer to entry in port table
*		RETURNS: NA
*	  CAUTIONS: Assumes event logging already done
****************************************************************/
static void process_offered_event(METAEVENT *metaeventp, struct channel *pline)
{
	char		dnis[GC_ADDRSIZE];
	char		ani[GC_ADDRSIZE];
	char		caller_name[GC_ADDRSIZE];
	char		str[MAX_STRING_SIZE];
	int		callindex;

	/* Retrieve CRN */
	callindex = existCRN(metaeventp->crn, pline);
	if (callindex == OUT_OF_RANGE) {
		printandlog(pline->index, MISC, NULL, "metaeventp->crn unexpectedly not found for a GCEV_OFFERED event", 0);
		exitdemo(1);
	}

	/* Save the CRN in port[index] and call Acceptcall */
	pline->call[callindex].call_state = GCST_OFFERED;
	pline->call[callindex].crn = metaeventp->crn;

	/* show some technology specific functionality */
	/* get destination and origination  if not ANALOG */
	if (pline->techtype != ANALOG) {
		/* first, get called party number */
		if (gc_GetCallInfo(pline->call[callindex].crn, DESTINATION_ADDRESS, dnis) == GC_SUCCESS) {
			sprintf(str, "gc_GetCallInfo(crn=0x%lx) Success - called party = %s", pline->call[callindex].crn, dnis);
		} else {
			sprintf(str, "gc_GetCallInfo(crn=0x%lx) Failure - called party not available ", pline->call[callindex].crn);
		}
		printandlog(pline->index, GC_APICALL, NULL, str, 0);
		/* do application specific dnis event processing here */
		/* For simplicity's sake, max DDI logic is not shown here */

		/* next get the calling party here */
		if (gc_GetCallInfo(pline->call[callindex].crn, ORIGINATION_ADDRESS, ani) == GC_SUCCESS) {
			sprintf(str, "gc_GetCallInfo(crn=0x%lx) Success - calling party = %s", pline->call[callindex].crn, ani);
		} else {
			sprintf(str, "gc_GetCallInfo(crn=0x%lx) Failure - calling party not available", pline->call[callindex].crn);
		}
		printandlog(pline->index, GC_APICALL, NULL, str, 0);
	} else {
		/* get calling party, if Analog */
		if (gc_GetCallInfo(pline->call[callindex].crn, ORIGINATION_ADDRESS, ani) == GC_SUCCESS) {
			/* get ANI */		
			sprintf(str, "gc_GetCallInfo(crn=0x%lx) Success - calling party  = %s", pline->call[callindex].crn, ani);
		} else {
			sprintf(str, "gc_GetCallInfo(crn=0x%lx) Failure - calling party not available", pline->call[callindex].crn);
		}
		printandlog(pline->index, GC_APICALL, NULL, str, 0);

		/* get the call info */
		if (gc_GetCallInfo(pline->call[callindex].crn, CALLNAME, caller_name) == GC_SUCCESS) {
			sprintf(str, "gc_GetCallInfo(crn=0x%lx, CALLNAME) Success, name = %s",
			pline->call[callindex].crn, caller_name);
		} else {
			sprintf(str, "gc_GetCallInfo(crn=0x%lx, CALLNAME) Failure, name not available",
					pline->call[callindex].crn);
		}
		printandlog(pline->index, GC_APICALL, NULL, str, 0);
	}

	/* Note: if # of rings is == 0 then the outbound side */
	/* may not get the alerting event, depending upon the protocol */ 
	/* if the number of rings is 1, there is an occasional problem in */
	/* in the outbound side's ability to recognize the tone */
	if (gc_AcceptCall(pline->call[callindex].crn, 3, EV_ASYNC) != GC_SUCCESS) {
		sprintf(str, "gc_AcceptCall(crn=0x%lx, # of rings=3, mode=EV_ASYNC) Failed", pline->call[callindex].crn);
		printandlog(pline->index, GC_APIERR, NULL, str, 0);
		exitdemo(1);
	}
	sprintf(str, "gc_AcceptCall(crn=0x%lx, mode=EV_ASYNC) Success", pline->call[callindex].crn);
	printandlog(pline->index, GC_APICALL, NULL, str, 0);
}

/****************************************************************
*			NAME: process_call_related_outbound_event(struct channel *pline, METAEVENT *metaeventp)
* DESCRIPTION: Function to process outbound GlobalCall event
*		 INPUTS: pline - pointer to entry in port table
*					metaeventp - pointer to the metaevent structure
*		RETURNS: NA
*	  CAUTIONS: Tightly coupled with process_event() for pre-processing
*					Assumes event logging already done
*					Disconnected event already handled
****************************************************************/
static void process_call_related_outbound_event(struct channel *pline, METAEVENT *metaeventp)
{
	int		evttype, index;
	int		unexpected_event = 0;					/* till proven otherwise */
	char		str[MAX_STRING_SIZE];
	int		callindex;

	evttype = metaeventp->evttype;
	index = pline->index;

	/* Retrieve CRN */
	callindex = existCRN(metaeventp->crn, pline);
	if (callindex == OUT_OF_RANGE) {
		/* CRN not found, so log error and return from function. */
		sprintf(str, "CRN not found");
		printandlog(pline->index, MISC_WITH_NL, NULL, str, 0);
		return;
	}

	switch (pline->call[callindex].call_state)
	{
		/************************************/
		/* first come the call setup states */
		/************************************/
		case GCST_NULL:
		{
			switch (evttype)
			{
				case GCEV_ALERTING:						/* just change the state to alerting - no other action is necessary */
					pline->call[callindex].call_state = GCST_ALERTING;
					break;

				case GCEV_PROCEEDING:					/* just change the state to proceeding - no other action is necessary */
					pline->call[callindex].call_state = GCST_PROCEEDING;
					break;

				/* This case can happen in some technologies/protocols if the inbound side */
				/* answered the call without ringing (i.e. if acceptcall was issued, */
				/* then the rings parameter in both acceptcall and anwercall was 0 */
				/* if only answercall was issued, then the # of rings was 0 */
				/* The internal GC call state should be Dialing, but the dialing event */
				/* is disabled by default and we did not enable it */
				case GCEV_CONNECTED:
					process_connected_event(pline, metaeventp);
					break;

				default:
					unexpected_event = 1;
					break;
			}
			break;
		}

		case GCST_ALERTING:
		{
			if (evttype == GCEV_CONNECTED) {
				process_connected_event(pline, metaeventp);
			} else {
				unexpected_event = 1;
			}
			break;
		}

      case GCST_PROCEEDING:
      {
			switch (evttype)
			{
				case GCEV_ALERTING:						/* just change the state to alerting - no other action is necessary */
					pline->call[callindex].call_state = GCST_ALERTING;
					break;

				case GCEV_CONNECTED:
					process_connected_event(pline, metaeventp);
					break;

				default:
					unexpected_event = 1;
					break;
			}
			break;
		}

		/*************************************/
		/* now come the call teardown states */
		/*************************************/
		case GCST_CONNECTED:
		{
			/*************************************************/
			/* Depending upon your application, you may or	 */
			/* may not have application specific event		 */
			/* handling here for when in the connected state */
			/* You could also have your app specific code	 */
			/* in process_event()									 */
			/*************************************************/
			if (evttype != GCEV_DROPCALL) {			/* recall the outbound side issues a gc_DropCall() */
				unexpected_event = 1;					/* for the purposes of this demo */
				break;
			}
			/* FALL THROUGH ON PURPOSE since this is one of the 2 ways of reaching Idle */
			/* the other being a disconnected event */
		}

		case GCST_DISCONNECTED:
		{
			if (evttype == GCEV_DROPCALL) {
				pline->call[callindex].dropcall_active = NO;

				/* Call ReleaseCall */
				pline->call[callindex].call_state = GCST_IDLE;

				if (gc_ReleaseCallEx(pline->call[callindex].crn, EV_ASYNC) != GC_SUCCESS) {
					sprintf(str, "gc_ReleaseCallEx(crn=0x%lx, EV_ASYNC) Failed", pline->call[callindex].crn);
					printandlog(index, GC_APIERR, NULL, str, 0);
					exitdemo(1);
				}
				sprintf(str, "gc_ReleaseCallEx(crn=0x%lx, EV_ASYNC) Success", pline->call[callindex].crn);
				printandlog(index, GC_APICALL, NULL, str, 0);
				fflush(stdout);
				break;
			} else {
				unexpected_event = 1;
			}
			break;
		}

		case GCST_IDLE:
		{
			if (evttype == GCEV_RELEASECALL) {
				pline->call[callindex].call_state = GCST_NULL;
				pline->call[callindex].crn = 0;
				pline->makecall_active = NO;					/* makecall is no longer active */
            if(pline->intercall_delay)  /* Set Inter-call Delay if specified */
            {
           		time(&pline->intercall_time);
 		         pline->intercall_time += pline->intercall_delay;
            }
				gc_demo_makecall(index); 
			} else {
				unexpected_event = 1;
			}
			break;
		}

		default:
		{
			/* Should never get here */
			printandlog(index, MISC, NULL, "Invalid state in process_call_related_outbound_event()", 0); 
			break;
		}
	}

	if (unexpected_event && (evttype != GCEV_UNBLOCKED)) {
		printandlog(index, GC_RESULT_INFO, metaeventp, "Event was unexpected in the current state", 0);
		printandlog(index, EVENT, NULL, GCEV_MSG(evttype), 0);
		printandlog(index, STATE, NULL, " ", callindex);
	}
}

/****************************************************************
*			NAME: process_connected_event(struct channel *pline, METAEVENT *metaeventp)
* DESCRIPTION: Function to process a connected event
*		 INPUTS: pline - pointer to entry in port table
*		RETURNS: NA
*	  CAUTIONS: Assumes event logging already done
*					Recall that this is only done for the outbound side
*					as the GCEV_CONNECTED event is an outbound side event only
****************************************************************/
static void process_connected_event(struct channel *pline, METAEVENT *metaeventp)
{
	char		str[MAX_STRING_SIZE];
	char		value;										/* for connect type */
	int		rc;
	int		callindex;

	/* Retrieve CRN */
	callindex = existCRN(metaeventp->crn, pline);
	if (callindex == OUT_OF_RANGE) {
		/* CRN not found, so log error and return from function. */
		sprintf(str, "CRN not found");
		printandlog(pline->index, MISC_WITH_NL, NULL, str, 0);
		return;
	}

	/* Obtain connect type reason if not H323 nor SIP */
	if ((pline->techtype != H323) && 
		(pline->techtype != SIP) && 
		(pline->techtype != SS7)){
		/* print connect type */
		rc = gc_GetCallInfo(pline->call[callindex].crn, CONNECT_TYPE, &value);
		if (rc != GC_SUCCESS) {
			if (rc == EGC_UNSUPPORTED) {
				strcpy(str, "call connected - gc_GetCallInfo(CONNECT_TYPE) not supported");
			} else {
				strcpy(str, "call connected - gc_GetCallInfo(CONNECT_TYPE) error\n");
				sprintf(str, "gc_GetCallInfo(crn=0x%lx, CONNECT_TYPE, &value) failed", pline->call[callindex].crn);
				printandlog(pline->index, GC_APIERR, NULL, str, 0);
			}
		} else {
			switch (value)
			{
				case GCCT_CAD :
					strcpy(str, "call connected - cadence break");
					break;

				case GCCT_LPC :
					strcpy(str, "call connected - loop current drop");
					break;

				case GCCT_PVD :
					strcpy(str, "call connected - voice detection");
					break;

				case GCCT_PAMD :
					strcpy(str, "call connected - answering machine");
					break;

				case GCCT_FAX :
					strcpy(str, "call connected - FAX machine");
					break;

				case GCCT_NA :
					strcpy(str, "call connected - call progress not applicable");
					break;
	
				default :
					strcpy(str, "call connected - unknown connect type");
					break;
			}
		}
		printandlog(pline->index, MISC, NULL, str, 0); 
	}

	pline->call[callindex].call_state = GCST_CONNECTED;
	pline->numb_calls++;							/* Increment the total calls on this device */
	out_calls++;									/* Increment the total outbound calls on all devices */
	outbound_application(pline);				/* call your application specific code here */
}

/****************************************************************
*			NAME: enable_alarm_notification(struct channel *pline)
* DESCRIPTION: Enables all alarms notification for pline
*					Also fills in pline->mediah
*		  INPUT: pline - pointer to channel data structure
*		RETURNS: None - exits if error
*	  CAUTIONS: Does not sanity checking as to whether or not the technology supports
*				   alarms - assumes caller has done that already
****************************************************************/
static void	enable_alarm_notification(struct channel *pline)
{
	char		str[MAX_STRING_SIZE];
	int		alarm_ldev;						/* linedevice that alarms come on */


	alarm_ldev = pline->ldev;				/* until proven otherwise */
	if (pline->techtype == H323) {
		/* Recall that the alarms for IP come on the media device, not the network device */
		if (gc_GetResourceH(pline->ldev, &alarm_ldev, GC_MEDIADEVICE) != GC_SUCCESS) {
			sprintf(str, "gc_GetResourceH(linedev=%ld, &alarm_ldev, GC_MEDIADEVICE) Failed", pline->ldev);
			printandlog(pline->index, GC_APIERR, NULL, str, 0);
			exitdemo(1);
		}
		pline->mediah = alarm_ldev;			/* save for later use */
	}

	if (gc_SetAlarmNotifyAll(alarm_ldev, ALARM_SOURCE_ID_NETWORK_ID, ALARM_NOTIFY) != GC_SUCCESS) {
		sprintf(str, "gc_SetAlarmNotifyAll(linedev=%ld, ALARM_SOURCE_ID_NETWORK_ID, \
ALARM_NOTIFY) Failed", pline->ldev);
		printandlog(pline->index, GC_APIERR, NULL, str, 0);
		printandlog(pline->index, MISC, NULL, "You may ignore this error - \
most likely the underlying technology does not support alarms yet", 0);
	} else {
		sprintf(str, "gc_SetAlarmNotifyAll(linedev=%ld, ALARM_SOURCE_ID_NETWORK_ID, ALARM_NOTIFY) PASSED", pline->ldev);
		printandlog(pline->index, MISC, NULL, str, 0);
	}
}


/****************************************************************
*			NAME: open_all_devices(void)
* DESCRIPTION: Routine calls the specific technology function, which
*					opens all the line devices. 
*		  INPUT: None
*		RETURNS: None - exits if error
*	  CAUTIONS: none
****************************************************************/
static void open_all_devices(void)
{
	int	index;

	for (index = 0; index < num_devices; index++) { 
		open_device(index);
	}
}
/****************************************************************
*			NAME: open_devices(int index)
* DESCRIPTION: Routine calls the specific technology function, which
*					opens all specific line device. 
*		  INPUT: index - index into port data structure
*		RETURNS: None - exits if error
*	  CAUTIONS: Since multiple paths here, will init run-time fields of port data structure
****************************************************************/
static void open_device(int index)
{
	int		callindex;

	/* reset run-time variables that might now be wrong */
	port[index].ldev = 0;

	/* reset the call related information */
	for (callindex = 0; callindex < MAX_CALLS; callindex++) {
		port[index].call[callindex].crn = 0;
		port[index].call[callindex].call_state = GCST_NULL;
		port[index].call[callindex].dropcall_active = NO;	
	}

	port[index].index = index;
	port[index].blocked = YES;
	port[index].waitcall_active = NO;	
	port[index].makecall_active = NO;	
	port[index].resetlinedev_active = NO;	
	port[index].resetlinedev_required_after_unblocked = NO;	
	port[index].dropcall_time = 0;
	port[index].makecall_time = 0;
	port[index].intercall_time = 0;

	if ((strcmp(port[index].protname, "isdn") == 0) ||
		(strcmp(port[index].protname, "ISDN") == 0)) {
		gc_demo_open_isdn_channel(index);
	} else if ((strstr(port[index].protname, "ss7") != NULL) ||
		(strstr(port[index].protname, "SS7") != NULL)) {
		gc_demo_open_ss7_channel(index);
	} else if ((strstr(port[index].protname, "_AN_") != NULL) ||
		(strstr(port[index].protname, "_an_") != NULL)) {
		gc_demo_open_analog_channel(index);
	} else if (strstr(port[index].protname, "H323") != NULL) {
		gc_demo_open_H323_channel(index);
	} else if (strstr(port[index].protname, "SIP") != NULL) {
		gc_demo_open_SIP_channel(index);
	} else {								/* assume everything else is E1 or T1 */
											/* if wrong, then gc_OpenEx() will fail */
		gc_demo_open_E1orT1_channel(index);
	}
}

/****************************************************************
*			NAME: gc_demo_makecall(index)
* DESCRIPTION: Routine calls the gc_Makecall() function if not blocked and there is not a call active
*		  INPUT: index into port[] for this channel
*		RETURNS: exits with exitdemo() if not successful
*	  CAUTIONS: none
****************************************************************/
static void gc_demo_makecall(int index)
{
	char					str[MAX_STRING_SIZE];
	struct channel		*pline;
	GC_MAKECALL_BLK	*makecallblkp;				/* Variable for MAKECALL block */		
	int		callindex;
	char					*numberstr;

	pline = &port[index];
	if ( (pline->blocked == YES) || (pline->resetlinedev_active == YES)
	|| (interrupted == YES) ) {
		return;										/* cannot make a call until unblocked and completion of reset line device */
	}

	/* Find empty place in array to store CRN. */
	for (callindex = 0; callindex < MAX_CALLS; callindex++) {
		if (pline->call[callindex].crn == 0) {
			break;
		}
	}

	if (pline->makecall_active == YES) {
		printandlog(index, MISC, NULL, "Trying to make a call while makecall is already active\n", 0);
		return;
	}

	/* verify OK to do a makecall now */
	if ((pline->makecall_time) ||  (pline->intercall_time)){
		/* makecall or intercall timer active, cannot do just now */
		return;
	}


	if(pline->techtype != SIP) {
		numberstr = pline->destination_num;
	} else {  /* else if SIP: */
		/* we are setting destination address in makecallblk */
		numberstr = NULL;
		/* and we are specifying coders on a per call basis */
		gc_set_channel_codecs(index);
	}

	/* if the makecall block has been initialized, use it */
	/* else use NULL for the makecall block pointer */
	makecallblkp = (pline->makecallblk.gclib != NULL) ? &pline->makecallblk : NULL;

	if (gc_MakeCall(pline->ldev, &pline->call[callindex].crn, numberstr, makecallblkp, pline->makecall_timeout, EV_ASYNC) == GC_SUCCESS) {
		pline->makecall_active = YES;					/* indicate there is a call active */
		sprintf(str, "gc_MakeCall(linedev=%ld, numberstr=%s, mode=EV_ASYNC) Success", pline->ldev, pline->destination_num);
		printandlog(index, GC_APICALL, NULL, str, 0);
	} else {
		sprintf(str, "gc_MakeCall(linedev=%ld, numberstr=%s, mode=EV_ASYNC) Failed", pline->ldev, pline->destination_num);
		printandlog(index, GC_APIERR, NULL, str, 0);
		if (pline->techtype == ISDN) {
			sprintf(str, "Please verify setup in gc_demo_open_isdn_channel() is correct");
			printandlog(index, MISC, NULL, str, 0);
		}

		/* There are circumstances under which makecall will fail now, but will succeed later */
		/* so rather than exit the app, let's schedule a retry */
		time(&pline->makecall_time);
		pline->makecall_time += MIN_MAKECALL_RETRY_TIME + 1;
		sprintf(str, "Will retry the makecall in %d-%d seconds", MIN_MAKECALL_RETRY_TIME, MIN_MAKECALL_RETRY_TIME + 1);
		printandlog(index, MISC, NULL, str, 0);
	}
} /* End of Function */

/****************************************************************
*			NAME: gc_demo_open_isdn_channel(index)
* DESCRIPTION: initialize a port used by an ISDN protocol
*		  INPUT: index into port[] for this channel
*		RETURNS: exits with exitdemo() if not successful
*	  CAUTIONS: 1. This will need to be modified for some configurations/environments
*					2. Many parameters are "commented" out. The purpose of this is to 
*					   give you a template of how to change default parameters if you need
*						to do so
*					3. Don't be intimidated by the size of this procedure.
*						As previously indicated, much of this procedure has
*						commented out code which is intended to give examples
*						for those who need it. Many developers will not.
****************************************************************/
static void gc_demo_open_isdn_channel(int index)
{
	char						str[MAX_STRING_SIZE];
	GCLIB_MAKECALL_BLK	*gclib_makecallp;
	GC_PARM_BLKP			parmblkp = NULL;


	printandlog(index, MISC, NULL, "ISDN device being opened", 0);

	/* Initial channel state is blocked */
	port[index].blocked = YES; 
	port[index].techtype = ISDN;
	port[index].makecall_timeout = 0;							/* timeout parameter is ignored for ISDN */

	/* initializes the GC_PARM_BLK with the respective set_id, parm_id and the values */

	/*******
	-- To set the bearer channel transfer mode information
	-- Commented out as this is the default. Change if you wish to use
	-- other than the default value
	--	gc_util_insert_parm_val(&parmblkp, GCIS_SET_BEARERCHNL, GCIS_PARM_TRANSFERMODE, 
	--		sizeof(int), ISDN_ITM_CIRCUIT);
	*******/

	/*******
	-- To set the bearer channel transfer rate information
	-- Commented out as this is the default. Change if you wish to use 
	-- other than the default value
	--	gc_util_insert_parm_val(&parmblkp, GCIS_SET_BEARERCHNL, GCIS_PARM_TRANSFERRATE, 
	--		sizeof(int), BEAR_RATE_64KBPS);
	*******/

	/*******
	-- To set Layer 1 protocol to use on bearer channel
	-- Commented out as this is the default. Change if you wish to use
	-- other than the default value
	--	gc_util_insert_parm_val(&parmblkp, GCSET_CHAN_CAPABILITY, GCPARM_TYPE, 
	--		sizeof(int), GCCAPTYPE_AUDIO);
	*******/

	/*******
	-- To set Layer 1 protocol to use on bearer channel
	-- The default is not good for all our E1 boards, for example Springware
	-- WARNING: this will need to be changed for T1. Commenting out this
	-- function call will work in many T1 cases
	*******/
	gc_util_insert_parm_val(&parmblkp, GCSET_CHAN_CAPABILITY, GCPARM_CAPABILITY, 
		sizeof(int), GCCAP_AUDIO_g711Alaw64k);

	/*******
	-- To set User rate to use on bearer channel (layer 1 rate)
	--  Commented out as this is the default. Change if you wish to use
	-- other than the default value
	--	gc_util_insert_parm_val(&parmblkp, GCSET_CHAN_CAPABILITY, GCPARM_RATE, 
	--		sizeof(int), GCCAPRATE_DEFAULT);
	*******/

	/*******
	-- To support the specific feature or service requested from the network
	-- Commented out as this is not supported by E1 and DM3.
	--	gc_util_insert_parm_val(&parmblkp, GCIS_SET_FACILITY, GCIS_PARM_FACILITY_CODINGVALUE, 
	--		sizeof(int), 0x03);
	*******/

	/*******
	-- To support the specific feature or service requested from the network
	-- Commented out as this is not supported by E1 and DM3.
	--	gc_util_insert_parm_val(&parmblkp, GCIS_SET_FACILITY, GCIS_PARM_FACILITY_FEATURESERVICE, 
	--		sizeof(int), ISDN_SERVICE);
	*******/

	/*******
	-- Note: gclib_makecallp is not freed since a "permanent" data
	-- structure. Assume program exit will free the resource
	*******/
	gclib_makecallp
		= (GCLIB_MAKECALL_BLKP) malloc(sizeof(GCLIB_MAKECALL_BLK) + parmblkp->parm_data_size);
	if (gclib_makecallp == NULL) {
		printandlog(ALL_DEVICES, MISC, NULL, "Could not allocate memory\n", 0);
		exitdemo(1);
	}
	port[index].makecallblk.cclib = NULL;
	port[index].makecallblk.gclib = gclib_makecallp;

	/* Set the extra data in the makecall block */
	gclib_makecallp->ext_datap = parmblkp;

	/*******
	-- Set the called number plan to unknown 
	-- Modified, as this causes issues on some switches
	-- - UNKNOWN works on 99% of E1 switches
	--	gclib_makecallp->destination.address_plan = GCADDRPLAN_ISDN;
	*******/
	gclib_makecallp->destination.address_plan = GCADDRPLAN_UNKNOWN;

	/*******
	-- Set the address type to transparent
	-- Modified, as this causes issues on some switches 
	-- - TRANSPARENT works on 99% of E1 switches
	--	gclib_makecallp->destination.address_type = GCADDRTYPE_NAT;
	*******/
	gclib_makecallp->destination.address_type = GCADDRTYPE_TRANSPARENT;

	/*******
	-- Removed sub-addresses, as this causes issues on some switches
	--	strcpy(gclib_makecallp->destination.sub_address, "456");
	--	gclib_makecallp->destination.sub_address_plan = 0;
	--	gclib_makecallp->destination.sub_address_type = GCSUBADDR_OSI;
	*******/
	gclib_makecallp->destination.sub_address[0] = '\x0';
	gclib_makecallp->destination.sub_address_plan = 0;
	gclib_makecallp->destination.sub_address_type = 0;

	/*******
	-- Note to customers: make sure that the Calling Address is set correctly, as some switches
	-- could reject the call if the calling address is not correct.
	*******/
	strcpy(gclib_makecallp->origination.address, "7654321");

	/*******
	-- Set the calling number plan to ISDN type
	-- Modified, as this causes issues on some switches
	-- gclib_makecallp->origination.address_plan = GCADDRPLAN_ISDN;
	*******/
	gclib_makecallp->origination.address_plan = GCADDRPLAN_UNKNOWN;

	/*******
   -- Set the address type
	-- Modified, as this causes issues on some switches
	-- gclib_makecallp->origination.address_type = GCADDRTYPE_NAT;
	*******/
	gclib_makecallp->origination.address_type = GCADDRTYPE_TRANSPARENT;

	/*******
	-- Removed sub-addresses, as this causes issues on some switches
	--	strcpy(gclib_makecallp->origination.sub_address, "456");
	-- gclib_makecallp->origination.sub_address_plan = 0;
	-- gclib_makecallp->origination.sub_address_type = GCSUBADDR_OSI;
	*******/
	gclib_makecallp->origination.sub_address[0] = '\x0';
	gclib_makecallp->origination.sub_address_plan = 0;
	gclib_makecallp->origination.sub_address_type = 0;

	gclib_makecallp->call_info.address_info = GCADDRINFO_ENBLOC;
	
	/* Open the line device */
	if (gc_OpenEx(&port[index].ldev, port[index].devname, EV_SYNC, (void *)&port[index]) != GC_SUCCESS) {
		sprintf(str, "gc_OpenEx(devicename=%s, mode=EV_SYNC) Failed", port[index].devname);
		printandlog(index, GC_APIERR, NULL, str, 0);
		exitdemo(1);
	}
	sprintf(str, "gc_OpenEx(devicename=%s, mode=EV_SYNC) Success", port[index].devname);
	printandlog(index, GC_APICALL, NULL, str, 0);

	/* Due to implementation, must wait until get GCEV_UNBLOCKED before issueing the reset linedevice */
	port[index].resetlinedev_required_after_unblocked = YES;

	/* Enable alarm notification */
	enable_alarm_notification(&port[index]);
} /* End of Function */

/****************************************************************
*			NAME: gc_demo_open_analog_channel(index)
* DESCRIPTION: initialize a port used by an Analog protocol
*		  INPUT: index into port[] for this channel
*		RETURNS: exits with exitdemo() if not successful
*	  CAUTIONS: none
****************************************************************/
static void gc_demo_open_analog_channel(int index)
{
	char		str[MAX_STRING_SIZE], msgbuffer[MAX_STRING_SIZE];


	printandlog(index, MISC, NULL, "Analog device being opened", 0);

	/* Initial channel state is blocked */
	port[index].blocked = YES; 
	port[index].techtype = ANALOG;
	port[index].makecall_timeout = 60;
	
	if (gc_OpenEx(&port[index].ldev, port[index].devname, EV_SYNC, (void *)&port[index]) != GC_SUCCESS) {
		sprintf(str, "gc_OpenEx(devicename=%s, mode=EV_SYNC) Failed", port[index].devname);
		printandlog(index, GC_APIERR, NULL, str, 0);
		exitdemo(1);
	}
	
	sprintf(str, "gc_OpenEx(devicename=%s, mode=EV_SYNC) Success", port[index].devname);
	printandlog(index, GC_APICALL, NULL, str, 0);

	if ((gc_LoadDxParm(port[index].ldev,"dxchan.vcp", msgbuffer, MAX_STRING_SIZE)) !=0) {
		sprintf(str, "gc_LoadDxParm(linedev=%ld) Failed)%s)", port[index].ldev, msgbuffer);
		printandlog(index, GC_APIERR, NULL, str, 0);
		exitdemo(1);
	}
	sprintf(str, "gc_LoadDxParm(linedev=%ld) Success)", port[index].ldev);
	printandlog(index, MISC, NULL, str, 0);

	/* Some implementations do not reset (i.e. go on-hook) if line is off-hook at open time */
	if (gc_ResetLineDev(port[index].ldev, EV_ASYNC) != GC_SUCCESS) {
		sprintf(str, "gc_ResetLineDev(linedev=%ld, EV_ASYNC) Failed", port[index].ldev);
		printandlog(index, GC_APIERR, NULL, str, 0);
		exitdemo(1);
	}

	port[index].resetlinedev_active = YES;
	sprintf(str, "gc_ResetLineDev(linedev=%ld, EV_ASYNC) Success", port[index].ldev);
	printandlog(index, GC_APICALL, NULL, str, 0);
} /* End of Function */

/****************************************************************
*			NAME: gc_demo_open_E1orT1_channel(index)
* DESCRIPTION: initialize a port used by an E1 or T1 protocol
*		  INPUT: index into port[] for this channel
*		RETURNS: exits with exitdemo if not successful
*	  CAUTIONS: none
****************************************************************/
static void gc_demo_open_E1orT1_channel(int index) 
{
	char		str[MAX_STRING_SIZE]; 

	printandlog(index, MISC, NULL, "E1 or T1 device being opened", 0);

	/* Initial channel state is blocked */
	port[index].blocked = YES; 
	port[index].techtype = E1ORT1_CAS;
	port[index].makecall_timeout = 60;							/* Use 1 minute */
	
	if (gc_OpenEx(&port[index].ldev, port[index].devname, EV_SYNC, (void *)&port[index]) != GC_SUCCESS) {
		sprintf(str, "gc_OpenEx(devicename=%s, mode=EV_SYNC) Failed", port[index].devname);
		printandlog(index, GC_APIERR, NULL, str, 0);
		exitdemo(1);
	}
	
	sprintf(str, "gc_OpenEx(devicename=%s, mode=EV_SYNC) Success", port[index].devname);
	printandlog(index, GC_APICALL, NULL, str, 0);

	set_calling_num(&port[index], "7654321");

	/* Enable alarm notification */
	enable_alarm_notification(&port[index]);
} /* End of function */

/****************************************************************
*			NAME: gc_demo_open_H323_channel(index)
* DESCRIPTION: initialize a port used by H323 IP protocol
*		  INPUT: index into port[] for this channel
*		RETURNS: exits with exitdemo if not successful
*	  CAUTIONS: none
****************************************************************/
static void gc_demo_open_H323_channel(int index)
{
	char				str[MAX_STRING_SIZE]; 

	printandlog(index, MISC, NULL, "H.323 device being opened", 0);

	/* Initial channel state is blocked */
	port[index].blocked = YES; 
	port[index].techtype = H323;
	port[index].makecall_timeout = 60;						/* Use 1 minute */

	if (gc_OpenEx(&port[index].ldev, port[index].devname, EV_ASYNC, (void *)&port[index]) != GC_SUCCESS) {
		sprintf(str, "gc_OpenEx(devicename=%s, mode=EV_ASYNC) Failed", port[index].devname);
		printandlog(index, GC_APIERR, NULL, str, 0);
		exitdemo(1);
	}

	sprintf(str, "gc_OpenEx(devicename=%s, mode=EV_ASYNC) Success", port[index].devname);
	printandlog(index, GC_APICALL, NULL, str, 0);
	
	/* Specify TX and RX coders for device */
	gc_set_channel_codecs(index);
	
} /* End of function */

/****************************************************************
*			NAME: gc_demo_open_SIP_channel(index)
* DESCRIPTION: initialize a port used by SIP protocol
*		  INPUT: index into port[] for this channel
*		RETURNS: exits with exitdemo if not successful
*	  CAUTIONS: none
****************************************************************/
static void gc_demo_open_SIP_channel(int index)
{
	char						 str[MAX_STRING_SIZE]; 
	GCLIB_MAKECALL_BLK	*gclib_makecallp;
	struct hostent			*phostent = NULL;
	struct in_addr			 addr;
	
	printandlog(index, MISC, NULL, "SIP device being opened", 0);

	/* Initial channel state is blocked */
	port[index].blocked = YES; 
	port[index].techtype = SIP;
	port[index].makecall_timeout = 60;						/* Use 1 minute */

	/* Note: gclib_makecallp is not freed since a "permanent" data */
	/* structure. Assume program exit will free the resource			*/
	gclib_makecallp = (GCLIB_MAKECALL_BLKP)
	malloc(sizeof(GCLIB_MAKECALL_BLK) );
	if (gclib_makecallp == NULL) {
		printandlog(ALL_DEVICES, MISC, NULL, "Could not allocate memory\n", 0);
		exitdemo(1);
	}

	port[index].makecallblk.cclib = NULL;
	port[index].makecallblk.gclib = gclib_makecallp;
	gclib_makecallp->ext_datap = NULL;

	/* set GCLIB_ADDRESS_BLK with destination string & type*/
	strcpy(gclib_makecallp->destination.address,port[index].destination_num);
	/* set the called number plan to Transparent type  */
	gclib_makecallp->destination.address_type = GCADDRTYPE_TRANSPARENT;

	/* set GCLIB_ADDRESS_BLK with local string & type*/
	/* retrieve local host name and address */
	if (gethostname(str, MAX_STRING_SIZE) != 0) {
		printandlog(index, GC_APIERR, NULL, "Failed to retrieve Host Name", 0);
		exitdemo(1);
	}

	if ((phostent = (struct hostent *)gethostbyname(str)) == 0) { 
		printandlog(index, GC_APIERR, NULL, "Failed to retrieve Host IP address", 0);
		exitdemo(1);
	}

	memcpy(&addr, phostent->h_addr_list[0], sizeof(struct in_addr));

	sprintf(gclib_makecallp->origination.address, "%s_%s@%s", port[index].netdevice,
				str, inet_ntoa(addr));
	/* set the called number plan to Transparent type  */
	gclib_makecallp->origination.address_type = GCADDRTYPE_TRANSPARENT;

	if (gc_OpenEx(&port[index].ldev, port[index].devname, EV_ASYNC, (void *)&port[index]) != GC_SUCCESS) {
		sprintf(str, "gc_OpenEx(devicename=%s, mode=EV_ASYNC) Failed", port[index].devname);
		printandlog(index, GC_APIERR, NULL, str, 0);
		exitdemo(1);
	}
	
	sprintf(str, "gc_OpenEx(devicename=%s, mode=EV_ASYNC) Success", port[index].devname);
	printandlog(index, GC_APICALL, NULL, str, 0);
} /* End of function */

/****************************************************************
*			NAME: gc_set_channel_codecs(index)
* DESCRIPTION: configure codecs for a VoIP (H.323 or SIP) channel 
*		  INPUT: index into port[] for this channel
*		RETURNS: exits with exitdemo if not successful
*	  CAUTIONS: none
****************************************************************/
static void gc_set_channel_codecs(index)
{
	int						i, j, scope;
	IP_CAPABILITY			ipcap;
	GC_PARM_BLK			  *parmblkp = NULL;
	char						str[MAX_STRING_SIZE];


	printandlog(index, MISC, NULL, "Specifying codecs", 0);

	for (i= 0; i< port[index].num_codecs; i++) {

		memset(&ipcap, 0, sizeof(IP_CAPABILITY));

		/* First obtain value of transmit codec type from translation table */
		for (j = 0; j< (sizeof(codecxlat)/sizeof(CODECXLAT)); j++) {
			if (!strcmp(port[index].codec[i].txcodec, codecxlat[j].codecname)) {
				ipcap.capability = codecxlat[j].codecvalue;
				break;
			}
		}

		/* if not found, log error and exit */
		if (ipcap.capability == 0) {
			printf("Invalid codec type entry in %s\n", cfgfile);
			exitdemo(1);
		}

		/* Specify TX codec, unless type is T.38 UDP which is bi-directional */
		/* and handled separately as data codec.                             */
		if (ipcap.capability != GCCAP_DATA_t38UDPFax) {
			ipcap.type = GCCAPTYPE_AUDIO;
			ipcap.direction = IP_CAP_DIR_LCLTRANSMIT;
			ipcap.extra.audio.frames_per_pkt = port[index].codec[i].txrate;
			/* Set VAD if and only if supported by codec type */
			if (codecxlat[j].isVADsupported == YES) {
				/* If codec supports VAD, use setting from config file. */
				/* Only check for "ON" for simplicity sake. */
				ipcap.extra.audio.VAD = (strcmp(port[index].codec[i].txVAD,"ON") == 0); 
			} else { /* VAD not supported */
				ipcap.extra.audio.VAD = NO;
			}

			/* append the GC_PARM_BLK with the respective TX codec */
			gc_util_insert_parm_ref(&parmblkp, GCSET_CHAN_CAPABILITY, IPPARM_LOCAL_CAPABILITY, 
											sizeof(IP_CAPABILITY), &ipcap);

			/* Re-initialize for determining receive codec */
			ipcap.capability = 0;
			/* Next obtain value of receive codec type from translation table */
			for (j = 0; j< (sizeof(codecxlat)/sizeof(CODECXLAT)); j++) {
				if (!strcmp(port[index].codec[i].rxcodec, codecxlat[j].codecname)) {
					ipcap.capability = codecxlat[j].codecvalue;
					break;
				}
			}

			/* If not found, log error and exit */
			if (ipcap.capability == 0) {
				printf("Invalid codec type entry in %s\n", cfgfile);
				exitdemo(1);
			}

			ipcap.direction = IP_CAP_DIR_LCLRECEIVE;
			ipcap.extra.audio.frames_per_pkt = port[index].codec[i].rxrate;
			/* Set VAD if and only if supported by codec type */
			if (codecxlat[j].isVADsupported) {
				/* string validation minimized for simplicity sake */
				ipcap.extra.audio.VAD = (strcmp(port[index].codec[i].rxVAD,"ON") == 0); 
			} else { /* VAD not supported */
				ipcap.extra.audio.VAD = NO;
			}

			/* append the GC_PARM_BLK with the respective RX codec */
			gc_util_insert_parm_ref(&parmblkp, GCSET_CHAN_CAPABILITY, IPPARM_LOCAL_CAPABILITY, 
									sizeof(IP_CAPABILITY), &ipcap);
					

		} else {		/* else codec is T.38 UDP which is bi-directional data codec */
			ipcap.type = GCCAPTYPE_RDATA;
			ipcap.direction = IP_CAP_DIR_LCLTXRX;
			ipcap.extra.data.max_bit_rate = 144;

			/* append the GC_PARM_BLK with the T.38 codec */
			gc_util_insert_parm_ref(&parmblkp, GCSET_CHAN_CAPABILITY, IPPARM_LOCAL_CAPABILITY, 
											sizeof(IP_CAPABILITY), &ipcap);
		}	
	}	/* end for loop for all codecs specified for device */

	/* To interoperate with Microsoft Phoenix Version 5.0 Soft SIP phone,*/
	/* we will set coders on SIP calls on a per call basis.  Again, this */
	/* is not required for other SIP clients.                            */
	if(port[index].techtype == H323) {
		scope = GC_ALLCALLS;
	} else {  /* else SIP */
		scope = GC_SINGLECALL;
	}

	if (gc_SetUserInfo(GCTGT_GCLIB_CHAN, port[index].ldev, parmblkp, scope) != GC_SUCCESS) {
							sprintf(str, "gc_SetUserInfo(linedev=%ld) Failed when specifying coders", port[index].ldev);
							printandlog(index, GC_APIERR, NULL, str, 0);
							exitdemo(1);
	}

	gc_util_delete_parm_blk(parmblkp);
} /* End of function */

/****************************************************************
*			NAME: gc_demo_open_ss7_channel(index)
* DESCRIPTION: initialize a port used by an SS7 protocol
*		  INPUT: index into port[] for this channel
*		RETURNS: exits with exitdemo if not successful
*	  CAUTIONS: none
****************************************************************/
static void gc_demo_open_ss7_channel(int index)
{
	char		str[MAX_STRING_SIZE];
	char		temp_devname[MAX_DEVNAME];
	printandlog(index, MISC, NULL, "SS7 device being opened", 0);

	/* Initial channel state is blocked */
	port[index].blocked = YES; 
	port[index].techtype = SS7;
	port[index].makecall_timeout = 60;							/* timeout is supported, try 1 minute */
	
	/* 
	NOTE:
	The SS7 MakeCallBlock is not initialized currently, as Gc_MakeCallBlk->cclib is set to 
	Null to use default values. For Live Testing, initialize the SS7MakeCallBlock and set 
	its address to Gc_MakeCallBlk->cclib field.
	*/

   sprintf(temp_devname,":L_SS7%s", port[index].devname);
	if (gc_OpenEx(&port[index].ldev, temp_devname, EV_SYNC, (void *)&port[index]) != GC_SUCCESS) {
		sprintf(str, "gc_OpenEx(devicename=%s, mode=EV_SYNC) Failed", temp_devname);
		printandlog(index, GC_APIERR, NULL, str, 0);
		exitdemo(1);
	}
	
	sprintf(str, "gc_OpenEx(devicename=%s, mode=EV_SYNC) Success", temp_devname);
	printandlog(index, GC_APICALL, NULL, str, 0);

	set_calling_num(&port[index], "7654321");

} /* End of function */ 

/***************************************************************************
 *			NAME: void set_calling_num(struct channel *pline, char *phone_num)
 *DESCRIPTION: Sets the calling number for the outbound direction
 *		  INPUT: pline - pointer to entry in port table
 *					phone_num - pointer to the phone number
 *		RETURNS: None.
  ***************************************************************************/
static void set_calling_num(struct channel *pline, char *phone_num)
{
	char		str[MAX_STRING_SIZE];

	if (pline->direction == DIR_OUT) {
		if (gc_SetCallingNum(pline->ldev, phone_num) < 0) {
			sprintf(str, "gc_SetCallingNum(linedev=%ld, phone_num = %s) Failed", pline->ldev, phone_num);
			printandlog(pline->index, GC_APIERR, NULL, str, 0);
			exitdemo(1);
		}
		sprintf(str, "gc_SetCallingNum(linedev=%ld, phone_num = %s) Success\n", pline->ldev, phone_num);
		printandlog(pline->index, MISC, NULL, str, 0);
	}
}


/***************************************************************************
 *			NAME: void intr_hdlr()
 *DESCRIPTION: Handler called when one of the following signals is
 *					received: SIGHUP, SIGINT, SIGQUIT, SIGTERM.
 *					This function stops I/O activity on all channels and
 *					closes all the channels.
 *		  INPUT: None
 *		RETURNS: None.
  ***************************************************************************/
static void intr_hdlr(void)
{
	if (interrupted == NO) {
		printandlog(ALL_DEVICES, MISC, NULL, " *******Received User Interrupted Signal *******\n", 0);

		interrupted = YES;
	}
	/* will handle graceful shutdown outside this handler in main's forever loop */
}

/********************************************************************************
 *			NAME: load_config_file()
 *DESCRIPTION: This function reads the configuration file line
 *					by line. Ignores commented lines. For valid lines, 
 *					it fills the devname field of corresponding element 
 *					of port (global array of struct channel) as it is
 *					to be passed as an argument to gc_OpenEx(). It also
 *					fills the direction field in port[index]. 
 * 
 *		  INPUT: None
 *		RETURNS: None - exits if error
 *	  CAUTIONS: Assumes config file is in a "reasonable" format
 *					Not all fields are case insensitive
 *					The parsing in this procedure is simplified for demo purposes,
 *					i.e. error handling is weak to non-existent.
 *					Your application will need to implement parsing that is appropriate 
 *					for your application
 ********************************************************************************/
static void load_config_file(void)
{
	static	char	line[MAX_STRING_SIZE], direction[MAX_STRING_SIZE];
	static	char	netdevice[MAX_DEVNAME], protname[MAX_DEVNAME], voicedevice[MAX_DEVNAME];
	static	char	devname[MAX_DEVNAME], destination_num[MAXPHONENUM];
	static	char	txcodec[MAX_CODECNAME], rxcodec[MAX_CODECNAME];
	static	char	txVAD[MAX_VADSTR], rxVAD[MAX_VADSTR];
   int      intercall_delay=0;
	int		txrate = 0;
	int		rxrate = 0;
	int		index, num_params, num_codec;
	FILE		*cfg_fp;
	fpos_t	fpos;
	
	/* Open Configuration file in Read mode */
	cfg_fp = fopen(cfgfile, "r");
	
	if (cfg_fp == NULL) {
		printf("Unable to open %s for read\n", cfgfile);
		perror("Reason is: ");
		exit(1);
	}

	while (fgets(line, MAX_STRING_SIZE, cfg_fp)) {
		/* strip trailing white space */
		index = strlen(line) - 1;
		while (index && (line[index] <= ' ')) {
			line[index] = '\0';
			index--;
		}

		/* ignore blank lines */
		if (index == 0) {
			continue;
		}

		/* if 1st character is not alpha numeric, then treat the line as a comment & ignore it */
		if ((line[0] < 'A') || (line[0] > 'z')) {	/* simplifed non-alpha check */
			continue; /* Ignore the line and read next */
		}
		num_devices++; /* Increment Global Variable */

		if(num_devices > MAXCHAN)
		{
			printf(" [ERROR]:		 Number of devices in cfg file exceeded maximum. Increase MAXCHAN and recompile\n");
			num_devices--;
			fclose (cfg_fp);
			exitdemo(1);
		}

		/* Parse DTI Device, Protocol Name, Direction and Voice Device Name */ 
		num_params = sscanf(line, "%s%s%s%s%s%d%s%d%s%s%d%s",
			netdevice, 
			protname, 
			direction, 
			voicedevice, 
			destination_num,
         &intercall_delay,
			/* following coder parameters apply to VoIP devices only: */
			txcodec,
			&txrate,
			txVAD,
			rxcodec,
			&rxrate,
			rxVAD);

			if ((netdevice[0] == '\0') || (protname[0] == '\0') || (direction == '\0')) {
			/* Invalid entry in the configuration file */
			fclose (cfg_fp);
			printf("Invalid entry in %s\n", cfgfile);
			printf("Line is: %s\n", line);
			exitdemo(1);
		}
		
		/* Fill devname to be passed as arg in gc_OpenEx() 
		in port array (global) of struct channel */
		
		if (strcmp(netdevice, "NONE") != 0) {
			sprintf(devname, ":N_%s:P_%s", netdevice, protname);
		} else {
			sprintf(devname, ":P_%s", protname);
		}
		/*Fill :V_ only if voice device name is not specified as NONE in cfg file */
		if (tolower(voicedevice[0]) != 'n') {
			/* IP uses :M instead of :V */
			if (voicedevice[0] != 'i') {
				strcat(devname, ":V_");
			} else {
				strcat(devname, ":M_");
			}
			strcat(devname, voicedevice);
		}
		
		/* Fill Direction in port array (global) of struct channel */
		if (tolower(direction[0]) == 'i') {
			port[num_devices - 1].direction = DIR_IN;
		} else {
			port[num_devices - 1].direction = DIR_OUT;
		}

		/* Fill device index in port array (global) of struct channel */
		port[num_devices - 1].index = num_devices - 1;

		/* Copying all the local variable values to Global Variables */
		strcpy(port[num_devices - 1].devname, devname);
		strcpy(port[num_devices - 1].netdevice, netdevice);
		strcpy(port[num_devices - 1].voicedevice, voicedevice);
		strcpy(port[num_devices - 1].protname, protname);
		strcpy(port[num_devices - 1].destination_num, destination_num);
      port[num_devices - 1].intercall_delay = intercall_delay;

		/* For VoIP devices (H.323 or SIP) copy the codec parameters */ 
		if ((!strcmp(protname, "H323")) || (!strcmp(protname, "SIP"))) {
			if (num_params == 12) {
				strcpy(port[num_devices - 1].codec[0].txcodec, txcodec);
				port[num_devices - 1].codec[0].txrate = txrate;
				strcpy(port[num_devices - 1].codec[0].txVAD, txVAD);
				strcpy(port[num_devices - 1].codec[0].rxcodec, rxcodec);
				port[num_devices - 1].codec[0].rxrate = rxrate;
				strcpy(port[num_devices - 1].codec[0].rxVAD, rxVAD);

				/* Get current position in config file for possible rewinding later */
				fgetpos(cfg_fp, &fpos);
				num_codec = 1;

				while (fgets(line, MAX_STRING_SIZE, cfg_fp)) {
					/* Strip trailing white space */
					index = strlen(line) - 1;
					while (index && (line[index] <= ' ')) {
						line[index] = '\0';
						index--;
					}

					/* ignore comments or blank lines */
					if ((line[0] == '#') || (index == 0)) {
						continue; /* Ignore the line and read next */
					}

					num_params = sscanf(line, "%s%d%s%s%d%s",
						txcodec,
						&txrate,
						txVAD,
						rxcodec,
						&rxrate,
						rxVAD);

					if (num_params !=6) {
						break;
					}

					num_codec++;
					if  (num_codec > MAXCODECS) {
							printf("More than 10 codecs specified for device - please modify gc_basic_call_model.cfg to correct\n");
							exitdemo(1);
					}

					strcpy(port[num_devices - 1].codec[num_codec - 1].txcodec, txcodec);
					port[num_devices - 1].codec[num_codec - 1].txrate = txrate;
					strcpy(port[num_devices - 1].codec[num_codec - 1].txVAD, txVAD);
					strcpy(port[num_devices - 1].codec[num_codec - 1].rxcodec, rxcodec);
					port[num_devices - 1].codec[num_codec - 1].rxrate = rxrate;
					strcpy(port[num_devices - 1].codec[num_codec - 1].rxVAD, rxVAD);

					/* Get current position in config file for possible rewinding later */
					fgetpos(cfg_fp, &fpos);
				}  /* end while there are multiple codecs for device to parse */

				/* Reset the previous file position for other line device entries to process */
				fsetpos(cfg_fp, &fpos);

			} else {  /* Not all codec parameters were in config file */
				fclose (cfg_fp);
				printf("Invalid or missing codec parameter entry in %s\n", cfgfile);
				printf("Line is: %s\n", line);
				exitdemo(1);
			}

			/* Store number of codecs specified for device */
			port[num_devices - 1].num_codecs = num_codec;

		}  /* end if VoIP device */
	
	} /* End of read all lines loop */
	
	fclose (cfg_fp);
} /* End of Function load_config_file() */


/********************************************************************************
 *		 NAME: printandlog ()
 * DESCRIPTION: prints on the screen and to the log file. 
 *				 Put time stamp to the log file. If the
 *				 message to be logged is specific to a device,
 *				 it is logged to the device specific log file.
 * 
 *		INPUT: index is the device index
 *				 log_type valid values are ERR, MISC, EVENT, GC_APICALL, GC_APIERR and GC_RESULT_INFO
 *				 metaeventp = pointer to metaevent data structure - only used for GC_RESULT_INFO
 *				 msg_str message string to be printed
 *				 callindex is the CRN index - used for retrieving callstate.
 *	 RETURNS: NA
 ********************************************************************************/
static void printandlog(int index, int log_type, METAEVENT *metaeventp, char *msg_str, int callindex)
{	
	struct tm			*Time;
	time_t				Timet;
	int					i;

#ifdef _WIN32	/* Defined in Windows */  
	struct _timeb	  Timeb;										/* used to get milliseconds */
#else			/* Defined in Unix	*/
	struct timeb		Timeb;	  
#endif
	 
	char					TimeStr[MAX_STRING_SIZE];			/* will hold the time string when built */
	char					buffer[MAX_STRING_SIZE];			/* will have everything but the time */
	char					timestamp[MAX_STRING_SIZE];		/* holds the formatted time */
	char					str[MAX_STRING_SIZE];
	char					gc_apierr[MAX_STRING_SIZE];		/* iff have GC API error - most of time will be NULL */
	char					gc_result_info[MAX_STRING_SIZE]; /* iff have GC result info request - most of time will be null */
		 
	/* initialize temporary buffers */
	buffer[0] = '\0';
	gc_apierr[0] = gc_result_info[0] = timestamp[0] = '\0';
	 
	/* Put time stamp	 - everything but the milliseconds */
	TimeStr[0] = '\0';
	time(&Timet);
	Time = localtime(&Timet);

#ifdef _WIN32	/* Defined in Windows */  
	_ftime(&Timeb);												/* needed to get the milliseconds */
#else			/* Defined in Unix	 */  
	ftime(&Timeb); 
#endif

	if (log_type == MISC_WITH_NL) {
		strcpy(timestamp, "\n");
	} 

	sprintf(TimeStr, "%02d/%02d %02d:%02d:%02d.%03d ", Time->tm_mon+1, Time->tm_mday, Time->tm_hour, Time->tm_min, Time->tm_sec, Timeb.millitm);
	strcat(timestamp, TimeStr);

	switch(log_type)
	{
		case GC_APICALL:
				strcat(buffer, " [GC_APICALL]: ");
				break;

		case GC_APIERR:
				strcat(buffer, " [GC_APIERR]:	 ");
				format_error_info(gc_apierr);
				break;

		case GC_RESULT_INFO:
				strcat(buffer, " [GC_RESULT_INFO]: ");
				format_result_info(index, metaeventp, gc_result_info);
				break;

		case NON_GC_APIERR:
				strcat(buffer, " [NON_GC_APIERR]: ");
				break;

		case EVENT:
				strcat(buffer, " [EVENT]:		 ");
				break;

		case MISC:
				strcat(buffer, " [MISC]:		 ");
				break;

		case MISC_WITH_NL:
				strcat(buffer, " [MISC]:		 ");
				break;

		case STATE:
				strcat(buffer, " [STATE]:		 ");
				append_call_state_name(buffer, index, callindex);
				break;

		default:
			sprintf(str, " [UNEXPECTED LOG TYPE %d]:	", log_type);
			strcat(buffer, str);
			break;
	} /* End of Switch */
	 
	strcat(buffer, msg_str);
	 

	/* now do the actual logging */
	/* 1st to screen */
	printf("%s\n", buffer);

	/* print GC error or GC result information if required */
	if (gc_apierr[0]) {
		printf("%s\n", gc_apierr);
	}
	if (gc_result_info[0]) {
		printf("%s\n", gc_result_info);
	}

	/* then to log file(s) */
	if (index == ALL_DEVICES) {
		for (i = 0; i < num_devices; i++) {
			if (port[i].log_fp) {
				fprintf(port[i].log_fp, "%s ", timestamp);
				fprintf(port[i].log_fp, "%s\n", buffer);
			}
		}
	} else if (port[index].log_fp) {
		fprintf(port[index].log_fp, "%s", timestamp);
		fprintf(port[index].log_fp, "%s\n", buffer);

		/* print GC error or GC result information if required */
		if (gc_apierr[0]) {
			fprintf(port[index].log_fp, "%s\n", gc_apierr);
		}
		if (gc_result_info[0]) {
			fprintf(port[index].log_fp, "%s\n", gc_result_info);
		}
	}
}

/********************************************************************************
 *			NAME: format_error_info()
 *DESCRIPTION: formats error information using gc_ErrorInfo()
 *		  INPUT: buffer - buffer to format into
 *		RETURNS: None
 *	  CAUTIONS: No integrity checking done on dest_buffer overflow
 ********************************************************************************/
static void format_error_info(char *dest_buffer)
{
	GC_INFO			  t_info;	  /* Structure which stores information about GC related errors */ 

	gc_ErrorInfo(&t_info);
			 
	strcpy(dest_buffer, "					 GC ERROR: ");
	strcat(dest_buffer, t_info.gcMsg);
	strcat(dest_buffer, "\n");


	strcat(dest_buffer, "					 CC Name: ");
	strcat(dest_buffer, t_info.ccLibName);
	strcat(dest_buffer, "\n");


	strcat(dest_buffer, "					 CC ERROR: ");
	strcat(dest_buffer, t_info.ccMsg);
	strcat(dest_buffer, "\n");

}

/********************************************************************************
 *			NAME: format_result_info()
 *DESCRIPTION: formats result information using gc_ResultInfo()
 *		  INPUT: index - index into port data structure
 *             metaeventp - pointer to metaevent structure
 *					buffer - buffer to format into
 *		RETURNS: None
 *	  CAUTIONS: No integrity checking done on dest_buffer overflow
 ********************************************************************************/
static void format_result_info(int index, METAEVENT *metaeventp, char *dest_buffer)
{
	char					str[MAX_STRING_SIZE];
	int					rc;
	GC_INFO				t_info;	  /* Structure which stores information about GC related errors */ 

	rc = gc_ResultInfo(metaeventp, &t_info);
	if (rc < 0) {
		sprintf(str, "gc_ResultInfo() call failed");
		printandlog(index, GC_APIERR, NULL, str, 0);
		exitdemo(1);
	}
			 
	strcpy(dest_buffer, "					 GC Result: ");
	strcat(dest_buffer, t_info.gcMsg);
	strcat(dest_buffer, "\n");


	strcat(dest_buffer, "					 CC Name: ");
	strcat(dest_buffer, t_info.ccLibName);
	strcat(dest_buffer, "\n");


	strcat(dest_buffer, "					 CC Result: ");
	strcat(dest_buffer, t_info.ccMsg);
	strcat(dest_buffer, "\n");
}

/********************************************************************************
 *			NAME: append_call_state_name ()
 *DESCRIPTION: Appends the call state to a given buffer for a specified channel index
 *		  INPUT: buffer - buffer to append to
 *					index - index into port structure (from 0)
 *		RETURNS: None
 *	  CAUTIONS: No integrity checking done on buffer overflow
 ********************************************************************************/
static void append_call_state_name(char *buffer, int index, int callindex)
{
	struct channel		*pline;
	char					str[100];

	if ((callindex < 0) || (callindex > MAX_CALLS)) {
		sprintf(str, "Invalid callindex(%d)", callindex);
		strcat(buffer, str);
		return;
	}

	pline = &port[index];
	switch(pline->call[callindex].call_state)
	{ 
		case GCST_NULL:
			strcat(buffer, "GCST_NULL");
			break;

		case GCST_IDLE:
			strcat(buffer, "GCST_IDLE");
			break;

		case GCST_OFFERED:
			strcat(buffer, "GCST_OFFERED");
			break;

		case GCST_ALERTING:
			strcat(buffer, "GCST_ALERTING");
			break;

        case GCST_PROCEEDING:
			strcat(buffer, "GCST_PROCEEDING");
			break;

		case GCST_ACCEPTED:
			strcat(buffer, "GCST_ACCEPTED");
			break;
				
		case GCST_CONNECTED:
			strcat(buffer, "GCST_CONNECTED");
			break;

		case GCST_DISCONNECTED:
			strcat(buffer, "GCST_DISCONNECTED");
			break;

		default:
			sprintf(str, "UNKNOWN GC CALL STATE (%d)", pline->call[callindex].call_state);
			strcat(buffer, str);
			break;
	}
}

/*******************************************************************************
 *			NAME: existCRN()
 *DESCRIPTION: Search the CRN array for a valid CRN.
 *		  INPUT: CRN crn - CRN associated with the call.
 *						Note: if crn == -1 then this means return the index of the 1st valid
 *						crn in the call array.
 *		  			struct channel *pline - pointer to data structure for the channel
 *		RETURNS: if CRN exists, then return callindex; otherwise return OUT_OF_RANGE.		
 ******************************************************************************/
static int existCRN(CRN crn, struct channel *pline)
{
	int callindex;

	if (crn == -1) {
		/* For drop_outbound_calls_if_required() case */
		/* The function does not have access to metaevent structure. */
		for (callindex = 0; callindex < MAX_CALLS; callindex++) {
			/* if crn is found, then break */
			if (pline->call[callindex].crn > 0) {
				break;
			}
		}
	} else {
		/* All other cases. */
		for (callindex = 0; callindex < MAX_CALLS; callindex++) {
			if (pline->call[callindex].crn == crn) {
				/* crn is found, break out of for loop and continue. */
				break;
			}
		}
	}

	/* If the callindex < MAX_CALLS, then return CRN. */
	/* Otherwise return (-1) indicating CRN was not found. */
	if (callindex < MAX_CALLS) {
		/* CRN found */
		return (callindex);
	}
	else {
		/* CRN not found */
		return (-1);
	}
}

/********************************************************************************
 *			NAME: initiate_exit()
 * DESCRIPTION: Initiate termination of active calls.
 * 			This will be done via gc_ResetLineDev()
 * 			which will clean up active calls
 *		  INPUT: none
 *		RETURNS: none
 *
 *******************************************************************************/
static void initiate_exit()
{
	int				index;
	char				str[MAX_STRING_SIZE]; 
	struct channel	*pline;
	static int		exit_active = NO;

	if (exit_active == YES) {
		return;
	}
	printandlog(ALL_DEVICES, MISC, NULL, "********** Exit initiated ***********\n", 0);

	exit_active = YES;
	for(index = 0; index < num_devices; index++) {
		pline = &port[index];
		if (pline->ldev) {
			if (pline->resetlinedev_active == NO) {
				if (gc_ResetLineDev(port[index].ldev, EV_ASYNC) == GC_SUCCESS) {
					reset_linedevs_left_to_complete++;
					pline->resetlinedev_active = YES;
					sprintf(str, "gc_ResetLineDev(linedev=%ld, mode=EV_ASYNC) Success",
						 port[index].ldev);
					printandlog(index, GC_APICALL, NULL, str, 0);
				} else {
					sprintf(str, "gc_ResetLineDev(linedev=%ld, mode=EV_ASYNC) failure ignored while exiting (reload boards after exit)",
						port[index].ldev);
					printandlog(index, GC_APIERR, NULL, str, 0);
				}
			} else {
				reset_linedevs_left_to_complete++;			/* reset linedev already active */
			}															/* just wait for it to complete */
		}
	}
	if (reset_linedevs_left_to_complete) {
		sprintf(str, "Please wait while %d gc_ResetLineDev()s are being done...",
			reset_linedevs_left_to_complete);
		printandlog(pline->index, MISC, NULL, str, 0);
	}
}

/********************************************************************************
 *			NAME: exitdemo()
 *DESCRIPTION: Closes all the logfiles,
 *					Closes all the devices
 *					Disables event handler
 *					Stops GlobalCall
 *					And Exits
 *		  INPUT: exitlevel 0 success, 1 some error 
 *					Exit from program with the exitlevel as specified by exitlevel
 *		RETURNS: exits		
 ********************************************************************************/
static void exitdemo(int exitlevel)
{
	int					index;
	char					str[MAX_STRING_SIZE]; 
	struct channel		*pline;
	time_t				finish_time;
	double				duration, min_time = 0;
	static int			recursive = 0;					/* called recursively. e.g. Multiple ^c. If yes, skip clean-up logic and exit */

	printandlog(ALL_DEVICES, MISC_WITH_NL, NULL, "********* Program Exiting *********", 0);
	if (recursive == 0) {
		recursive++;
		for (index = 0; index < num_devices; index++) {
			pline = &port[index];
		 
			/* The total calls run on Inbound & Outbound side */
			if (pline->direction == DIR_IN) {
				sprintf(str, "The total no of Inbound calls on this device is: %d", pline->numb_calls);
				printandlog(pline->index, MISC, NULL, str, 0);
			} else { 
				sprintf(str, "The total no of Outbound calls on this device is: %d", pline->numb_calls);
				printandlog(pline->index, MISC, NULL, str, 0);
			}
			sprintf(str, "The total no of Inbound calls over all devices is: %d", in_calls);
			printandlog(pline->index, MISC, NULL, str, 0);
			sprintf(str, "The total no of Outbound calls over all devices is: %d", out_calls);
			printandlog(pline->index, MISC, NULL, str, 0);

			/* Note down the finish time for the test */
			time(&finish_time);
			/* Calculate the duration taken by the test */
			duration = difftime(finish_time,	 start_time);
			min_time = duration / 60;
			sprintf(str, "The total duration taken by the test is: %5.2f minutes", min_time);
			printandlog(pline->index, MISC, NULL, str, 0);
		}
	}

	/* code that can never be called recursively */

	/* close all log files */
	for (index = 0; index < num_devices; index++) {
		pline = &port[index];
		if (pline->log_fp != NULL) {
			fclose(pline->log_fp);
			pline->log_fp = NULL;			
		}
	}
	if (gc_started == YES) {
		for(index = 0; index < num_devices; index++) {
			pline = &port[index];

			if (pline->ldev) {
				/* nothing can do about errors, so ignore them */			
				gc_Close(pline->ldev);
				pline->ldev = 0;
			}
			/* if GC makecall block being used, then free it */
			if (port[index].makecallblk.gclib) {
				/* free the parm block if used using the utility functions */
				/* CAUTION: this must be done before freeing the GC makecall block */
				if (port[index].makecallblk.gclib->ext_datap) {
					gc_util_delete_parm_blk(port[index].makecallblk.gclib->ext_datap);
				}

				/* now free the GC makecall block */
				free(port[index].makecallblk.gclib);
				port[index].makecallblk.gclib = NULL;
			}
		}
		gc_Stop();
	}
	exit(exitlevel);
} /* End of function exitdemo() */
/*****<< End of File >>*****/
