/**
 * @file  KMBC_low_mmap.c
 * @brief mmap() support for the KMBC_LOW device driver.
 * @date  June 28, 2006
 *
 * Dialogic CONFIDENTIAL	
 * Copyright 2006 Dialogic Corporation All Rights Reserved.
 * 
 * The source code contained or described herein and all documents related to 
 * the source code ("Material") are owned by Dialogic Corporation or its suppliers
 * or licensors.  Title to the Material remains with Dialogic Corporation or its 
 * suppliers and licensors.  The Material contains trade secrets and proprietary
 * and confidential information of Dialogic or its suppliers and licensors.  The
 * Material is protected by worldwide copyright and trade secret laws and treaty
 * provisions.  No part of the Material may be used, copied, reproduced, 
 * modified, published, uploaded, posted, transmitted, distributed, or disclosed
 * in any way without Dialogic's prior express written permission.
 * 
 * No license under any patent, copyright, trade secret or other intellectual 
 * property right is granted to or conferred upon you by disclosure or delivery
 * of the Materials, either expressly, by implication, inducement, estoppel or
 * otherwise.  Any license under such intellectual property rights must be
 * express and approved by Dialogic in writing.
 * 
 * Unless otherwise agreed by Dialogic in writing, you may not remove or alter this
 * notice or any other notice embedded in Materials by Dialogic or Dialogic's 
 * suppliers or licensors in any way.
 */




/* Header Files */
#include <linux/kernel.h>
#include <linux/version.h>
#include "KMBC_low_mmap.h"
#include "KMBC_low_mm.h"

#if (LINUX_VERSION_CODE < 0x020500)
#define LINUX24
#endif



/* Global Variables */
static struct vm_operations_struct kmbc_mmap_fops =
{
	.open   = kmbc_mmap_open,
	.close  = kmbc_mmap_close,
};




/* External Variables */
extern int g_BufferSize;




#ifdef KMBC_DEBUG
/***************************************************************************************/
/* The KMBC_TRACE macro provides a way of compiling in a lot more trace data for       */
/* initial development and subsequent debugging.  Most errors will always be logged,   */
/* regardless of the KMBC_DEBUG build option.                                          */
/***************************************************************************************/
#define KMBC_TRACE(x)     printk x
#else
#define KMBC_TRACE(x)
#endif	/* #ifdef KMBC_DEBUG */




int
kmbc_mmap( struct file *filp, struct vm_area_struct *vma )
{
	void          *buffer_addr;
	unsigned long  phys_addr;
	int            rc;


	if( vma != NULL )
	{
		/* When the user calls mmap(), we need to associate that user-mode address with */
		/* one of our HSI buffers.  We allocate the buffer from our pool, and then map  */
		/* it with a call to remap_pfn_range.                                           */
		buffer_addr = kmbc_hsi_pool_alloc();

		if( buffer_addr != NULL )
		{
			kmbc_hsi_pool_set_user_addr( buffer_addr, (void *)vma->vm_start );
			phys_addr = kmbc_hsi_pool_get_bus_addr( (void *)vma->vm_start );

			if( phys_addr )
			{
				/* Mark memory as noncached */
				vma->vm_ops        = &kmbc_mmap_fops;

#if (LINUX_VERSION_CODE == KERNEL_VERSION(2,4,21) || !defined(LINUX24))
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,11))
				rc = remap_pfn_range( vma, vma->vm_start, phys_addr >> PAGE_SHIFT, vma->vm_end - vma->vm_start,
											 vma->vm_page_prot );
#else
				rc = remap_page_range( vma, vma->vm_start, phys_addr, vma->vm_end - vma->vm_start,
											  vma->vm_page_prot );
#endif
#else
				rc = remap_page_range( vma->vm_start, phys_addr, vma->vm_end - vma->vm_start,
											  vma->vm_page_prot );
#endif

				if( rc != 0 )
				{
					KMBC_TRACE(( KERN_ALERT "kmbc_low:  remap_page_range() failed (rc=%d).\n", rc ));
					return( -EINVAL );
				}
				
				KMBC_TRACE(( KERN_ALERT "kmbc_low:  Associating Memory Map (virt_addr=%p, phys_addr=0x%lx, size=%d).\n",
							 (void*)vma->vm_start, phys_addr, g_BufferSize ));
			}
			else
			{
				KMBC_TRACE(( KERN_ALERT "kmbc_low:  Could Not Get Physical Address (kern_addr=%p).\n",
							 buffer_addr ));
			}
		}
		else
		{
			KMBC_TRACE(( KERN_ALERT "kmbc_low:  No More HSI Buffers.\n" ));
		}
	}

	return( 0 );
}




void
kmbc_mmap_open( struct vm_area_struct *vma )
{
	KMBC_TRACE(( KERN_ALERT "kmbc_low:  Entering kmbc_mmap_open().\n" ));
	return;
}




void
kmbc_mmap_close( struct vm_area_struct *vma )
{
	void *kern_addr;

	KMBC_TRACE(( KERN_ALERT "kmbc_low:  Entering kmbc_mmap_close() - (vma=%p).\n", vma ));

	if( vma != NULL )
	{
		kern_addr = kmbc_hsi_pool_convert_user_addr( (void *)vma->vm_start );

		KMBC_TRACE(( KERN_ALERT "kmbc_low:  Closing MMAP (virt_addr=%p, kern_addr=%p).\n",
					(void*)vma->vm_start, kern_addr ));

		kmbc_hsi_pool_free( kern_addr );
	}
}
