/**
 * @file  lb_mtklib.h
 * @brief Layout Builder Media Toolkit Library
 * @date  Apr. 20, 2006
 *
 * DIALOGIC CONFIDENTIAL      
 * Copyright (C) 2006-2011 Dialogic Corporation. All Rights Reserved.
 * 
 * The source code contained or described herein and all documents related to
 * the source code ("Material") are owned by Dialogic Corporation or its 
 * suppliers or licensors.  Title to the Material remains with Dialogic 
 * Corporation or its suppliers and licensors.  The Material contains trade
 * secrets and proprietary and confidential information of Dialogic or its 
 * suppliers and licensors.  The Material is protected by worldwide copyright
 * and trade secret laws and treaty provisions.  No part of the Material may be
 * used, copied, reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way without Dialogic's prior express written
 * permission.
 * 
 * No license under any patent, copyright, trade secret or other intellectual 
 * property right is granted to or conferred upon you by disclosure or delivery
 * of the Materials, either expressly, by implication, inducement, estoppel or
 * otherwise.  Any license under such intellectual property rights must be
 * express and approved by Dialogic in writing.
 */
 
#ifndef _LB_MTKLIB_H_
#define _LB_MTKLIB_H_

#include "mtklib.h"
#include "stddef.h"

/*******************************************************************************
 * Layout Builder Meida Toolkit Data Type's
 ******************************************************************************/

typedef MTK_HANDLE LB_FRAME_HANDLE;
 
/**
 * @enum eLB_LAYOUT_SIZE
 * Enumerated list of supported video layout sizes.
 */
typedef enum tagELB_LAYOUT_SIZE
{
   eLB_LAYOUT_SIZE_SUBQCIF,      ///< Layout size [128 x 96]
   eLB_LAYOUT_SIZE_QCIF,         ///< Layout size [176 x 144]
   eLB_LAYOUT_SIZE_CIF,           ///< Layout size [352 x 288] 
   eLB_LAYOUT_SIZE_VGA,           ///< Layout size [640 x 480] 
   
   eLB_LAYOUT_SIZE_SQVGA,
   eLB_LAYOUT_SIZE_QVGA, 
   eLB_LAYOUT_SIZE_WQVGA,
   eLB_LAYOUT_SIZE_W288P,
   eLB_LAYOUT_SIZE_NHD,  
   eLB_LAYOUT_SIZE_448P, 
   eLB_LAYOUT_SIZE_SD,   
   eLB_LAYOUT_SIZE_W448P, 
   eLB_LAYOUT_SIZE_FOURCIF,           ///< Layout size [640 x 480] 
   eLB_LAYOUT_SIZE_P720_4_3_HD,    ///< Layout size [960 x 720] 
   eLB_LAYOUT_SIZE_P720    ///< Layout size [960 x 720] 
} eLB_LAYOUT_SIZE;

#define LB_1_RGN_BASE 100
#define LB_4_RGN_BASE 400
#define LB_6_RGN_BASE 600
#define LB_9_RGN_BASE 900

/**
 * @enum eLB_LAYOUT_TYPE
 * Enumerated list of supported video layout types.
 */
typedef enum tagELB_LAYOUT_TYPE
{
   eLB_LAYOUT_TYPE_CUSTOM  = 0,                 ///< Custom layout type to be configured by user.
   eLB_LAYOUT_TYPE_1_1     = LB_1_RGN_BASE+1,   ///< One region layout variation 1.
   eLB_LAYOUT_TYPE_4_1     = LB_4_RGN_BASE+1,   ///< Four region layout variation 1.
   eLB_LAYOUT_TYPE_6_1     = LB_6_RGN_BASE+1,
   eLB_LAYOUT_TYPE_9_1     = LB_9_RGN_BASE+1
} eLB_LAYOUT_TYPE;

/**
 * @enum eLB_DISPLAY_MODE
 * Enumerated list of region display modes.
 */
typedef enum tagELB_DISPLAY_MODE
{
   eLB_DISPLAY_MODE_BLANK,       ///< This mode will display a blank region.
   eLB_DISPLAY_MODE_FROZEN,      ///< This mode will display a frozen image of
                                 ///< the participant within region.
   eLB_DISPLAY_MODE_LIVE         ///< This mode will display live streaming of 
                                 ///< the participant within region.
} eLB_DISPLAY_MODE;

/**
 * @enum eLB_SELECTION_MODE
 * Enumerated list of region selection modes.
 */
typedef enum tagELB_SELECTION_MODE
{
   eLB_SELECTION_MODE_VOICE_ACTIVATED,  ///< This mode will switch the region
                                       ///< frame video based on active talker.
   eLB_SELECTION_MODE_USER_SELECT       ///< This mode will switch the region
                                       ///< frame video based on user selection.
} eLB_SELECTION_MODE;


/*******************************************************************************
 * Layout Builder Meida Toolkit API's
 ******************************************************************************/
 
#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */


/**
 * @brief Add a region to a video layout.
 *
 * This API will add a new region to the specified video layout. A user can add
 * regions only on custom layout types, regions cannot be added to predefined
 * layout types. Upon successful completion, a frame handle will be returned to
 * uniquely identify the newly added region within the specified layout.
 *
 * @param [in] a_LayoutHandle - Layout frame handle.
 * @param [in] a_pRect        - Pointer to Rect structure defining region.
 *
 * @return Region frame handle if successful, else MTK_ERROR.
 * @sa RemoveRegion
 */
MTK_API
LB_FRAME_HANDLE MTK_CONV lb_AddRegion (
   LB_FRAME_HANDLE a_LayoutHandle,
   PMTK_RECT       a_pRect
   );
   

/**
 * @brief Create a new video layout template.
 *
 * This API will create a new video layout that can be applied to any device
 * which supports video layouts. If successful this API will return a unique
 * frame handle which will identify this video layout. Once a valid frame handle
 * is returned to the user, it may be customized and eventually applied to 1 or 
 * more devices. Applying a layout to a device will realize that layout on a 
 * given device. To apply a layout to a device the user will utilize the frame 
 * handle, however, once a layout has been applied to a device, modifications to
 * the layout via the frame handle will not change nor modify the layout on the 
 * devices in which the same layout handle had previously been applied to, 
 * unless the user reapplies the layout handle to the device.
 *
 * @param [in] a_eType - Layout frame type.
 *
 * @return New video layout handle if successful, else MTK_ERROR.
 * @sa lb_DestroyLayout
 */
MTK_API
LB_FRAME_HANDLE MTK_CONV lb_CreateLayoutTemplate(eLB_LAYOUT_TYPE a_eType);

   
/**
 * @brief Destroy a existing video layout template.
 *
 * This API will destroy a previously created video layout. Destroying a layout
 * will not impact the devices in which the layout was applied.
 *
 * @param [in] a_LayoutHandle - Layout frame handle.
 *
 * @return MTK_SUCCESS if successful, else MTK_ERROR.
 * @sa lb_CreateLayout
 */
MTK_API 
MTK_RETURN MTK_CONV lb_DestroyLayoutTemplate(LB_FRAME_HANDLE a_LayoutHandle);


MTK_API
MTK_RETURN MTK_CONV lb_GetDisplayMode(LB_FRAME_HANDLE a_RegionHandle, eLB_DISPLAY_MODE * a_pMode);

MTK_API
MTK_RETURN MTK_CONV lb_GetPriority(LB_FRAME_HANDLE a_RegionHandle, unsigned int * a_punPriority);

MTK_API
MTK_RETURN MTK_CONV lb_GetRect(LB_FRAME_HANDLE a_RegionHandle, PMTK_RECT a_pRect);

MTK_API
MTK_RETURN MTK_CONV lb_GetSelectionMode(LB_FRAME_HANDLE a_RegionHandle, eLB_SELECTION_MODE * a_pMode);

MTK_API
MTK_RETURN MTK_CONV lb_GetType(LB_FRAME_HANDLE a_LayoutHandle, eLB_LAYOUT_TYPE * a_pType);


/**
 * @brief Get list of layout regions.
 *
 * This API will get the list of regions contained by the specified video layout
 * handle. The a_pRegionCount variable is used to specify the size of the 
 * provided frame list. If this function completes successfully then the 
 * a_pRegionList will contain the list of regions and the a_pRegionCount will be
 * updated to reflect the actual region count provided in the list. If this
 * function fails because the actual region count is greater than the provided
 * region count, the a_pRegionCount varaiable will be updated to reflect tha 
 * actual region count.
 *
 * @param [in]  a_LayoutHandle - Layout frame handle.
 * @param [out] a_pRegionList  - Pointer to region frame handle list.
 * @param [out] a_pRegionCount - Pointer to region frame handle count.
 *
 * @return MTK_SUCCESS if successful, else MTK_ERROR.
 */
MTK_API
MTK_RETURN MTK_CONV lb_GetRegionList (
   LB_FRAME_HANDLE   a_Layouthandle,
   LB_FRAME_HANDLE * a_pRegionList,
   unsigned int *    a_pRegionCount
   );
   
   
/**
 * @brief Remove a region from a video layout.
 *
 * This API will remove a previously added region from a video layout. A user 
 * can remove regions only on custom layout types, regions cannot be removed on
 * predefined layout types.
 *
 * @param [in] a_LayoutHandle - Layout frame handle.
 * @param [in] a_RegionHandle - Region frame handle.
 *
 * @return MTK_SUCCESS if successful, else MTK_ERROR.
 * @sa lb_AddRegion
 */
MTK_API
MTK_RETURN MTK_CONV lb_RemoveRegion (
   LB_FRAME_HANDLE a_LayoutHandle,
   LB_FRAME_HANDLE a_RegionHandle
   );
   

MTK_API
MTK_RETURN MTK_CONV lb_SetDisplayMode(LB_FRAME_HANDLE a_RegionHandle, eLB_DISPLAY_MODE a_eMode);

MTK_API
MTK_RETURN MTK_CONV lb_SetPriority(LB_FRAME_HANDLE a_RegionHandle, unsigned int a_unPriority);

MTK_API
MTK_RETURN MTK_CONV lb_SetRect(LB_FRAME_HANDLE a_RegionHandle, PMTK_RECT a_pRect);

MTK_API
MTK_RETURN MTK_CONV lb_SetSelectionMode(LB_FRAME_HANDLE a_RegionHandle, eLB_SELECTION_MODE a_eMode);
   
                                     
#ifdef __cplusplus
}
#endif  /* __cplusplus */


#endif /* _LB_MTKLIB_H_ */

