/**
 * @file  sm_mtklib.h
 * @date  May 23, 2007
 * @brief Stream Manipulation Media Toolkit Library Functions
 * 
 * DIALOGIC CONFIDENTIAL      
 * Copyright (C) 2007 Dialogic Corporation. All Rights Reserved.
 * 
 * The source code contained or described herein and all documents related to
 * the source code ("Material") are owned by Dialogic Corporation or its 
 * suppliers or licensors.  Title to the Material remains with Dialogic 
 * Corporation or its suppliers and licensors.  The Material contains trade
 * secrets and proprietary and confidential information of Dialogic or its 
 * suppliers and licensors.  The Material is protected by worldwide copyright
 * and trade secret laws and treaty provisions.  No part of the Material may be
 * used, copied, reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way without Dialogic's prior express written
 * permission.
 * 
 * No license under any patent, copyright, trade secret or other intellectual 
 * property right is granted to or conferred upon you by disclosure or delivery
 * of the Materials, either expressly, by implication, inducement, estoppel or
 * otherwise.  Any license under such intellectual property rights must be
 * express and approved by Dialogic in writing.
 */

#ifndef _SMMTKLIB_H_
#define _SMMTKLIB_H_

#include "mtklib.h"
#include "ob_mtklib.h"
#include "mtkerrs.h"
#include "sm_mtkevts.h"

#define SM_VER(v,s) (sizeof(s) << 16 | (0xFFFF & (v)))

/*! the maximum number of overlays on a device */
#define SM_MAX_OVERLAYS                16
#define SM_MAX_REMOVE_OVERLAYS         SM_MAX_OVERLAYS
#define SM_MAX_EVENTS                  64

/**
 * @struct SM_EVENT_INFO
 * @struct tagSM_EVENT_INFO
 * Event information structure. Structure used to provide event information
 * when enabling and disabling events.
 */
typedef struct tagSM_EVENT_INFO
{
    unsigned int    unVersion;                  ///< Structure version
    unsigned int    unCount;                    ///< Event Count
    unsigned int    unEventList[SM_MAX_EVENTS]; ///< List of events
} SM_EVENT_INFO, *PSM_EVENT_INFO;
typedef const SM_EVENT_INFO * CPSM_EVENT_INFO;

/*! @brief SM Event info structure version*/
#define SM_EVENT_INFO_VERSION_0 SM_VER(0, SM_EVENT_INFO)
#define SM_EVENT_INFO_VERSION SM_EVENT_INFO_VERSION_0
/*! @brief SM Event info structure initialization routine */
static __inline void INIT_SM_EVENT_INFO(PSM_EVENT_INFO pStruct)
{
	int ii;
    pStruct->unVersion = SM_EVENT_INFO_VERSION;
    for (ii=0; ii<SM_MAX_EVENTS; ii++)
    {
        pStruct->unEventList[ii] = 0;
    }
}


/*!
 * @enum tagESM_ADD_OVERLAY_RESULT
 * @brief Enumerated list of add overlay result codes.
 */
typedef enum tagESM_ADD_OVERLAY_RESULT
{
   eSM_ADD_OVERLAY_RESULT_ADD,     /*!< the add created a new overlay for the device*/
   eSM_ADD_OVERLAY_RESULT_MODIFY,  /*!< the add caused a modification of an existing overlay */
   eSM_ADD_OVERLAY_RESULT_FAIL     /*!< the add failed */
} eSM_ADD_OVERLAY_RESULT;


/*!
 * @enum tagESM_OVERLAY_DIRECTION
 * @brief Enumerated list of directions for overlay direction. This specifies
 *        the direction from the device in which the overlay will be added.
 */
typedef enum tagESM_OVERLAY_DIRECTION
{
   eSM_OVERLAY_DIRECTION_DEVICE,   /*!< overlay will be presented in device connection direction*/
   eSM_OVERLAY_DIRECTION_NETWORK   /*!< overlay will be presented in network direction */
} eSM_OVERLAY_DIRECTION;


/*!
 * @struct SM_ADD_OVERLAY
 * @struct tagSM_ADD_OVERLAY
 * @brief Add Overlay info structure
 */
typedef struct tagSM_ADD_OVERLAY
{
     /*! overlay info structure version */
   unsigned int   unVersion;
     /*!  specifies the direction in which overlay is presented */
   eSM_OVERLAY_DIRECTION eDirection;
     /*! overlay builder overlay handle */
   OB_OVERLAY_HANDLE    hOverlay;
} SM_ADD_OVERLAY, *PSM_ADD_OVERLAY;
typedef const PSM_ADD_OVERLAY CPSM_ADD_OVERLAY;

/*! @brief Add overlay structure version*/
#define SM_ADD_OVERLAY_VERSION_0 SM_VER(0, SM_ADD_OVERLAY)
#define SM_ADD_OVERLAY_VERSION SM_ADD_OVERLAY_VERSION_0
/*! @brief Add overlay info structure initialization routine */
static __inline void INIT_SM_ADD_OVERLAY(PSM_ADD_OVERLAY pStruct)
{
   pStruct->unVersion = SM_ADD_OVERLAY_VERSION;
   pStruct->eDirection = eSM_OVERLAY_DIRECTION_DEVICE;
   pStruct->hOverlay = MTK_ERROR;
}


/*!
 * @struct SM_ADD_OVERLAY_LIST
 * @struct tagSM_ADD_OVERLAY_LIST
 * @brief Add overlay list structure
 */
typedef struct tagSM_ADD_OVERLAY_LIST
{
     /*! overlay list structure version */
   unsigned int   unVersion;
     /*! number of valid overlays contained in overlayInfos */
   unsigned int   unCount;
     /*! add overlay list of overlays */
   SM_ADD_OVERLAY  addOverlays[SM_MAX_OVERLAYS];
} SM_ADD_OVERLAY_LIST, *PSM_ADD_OVERLAY_LIST;
typedef const PSM_ADD_OVERLAY_LIST CPSM_ADD_OVERLAY_LIST;

/*! @brief Add overlay list structure version*/
#define SM_ADD_OVERLAY_LIST_VERSION_0 SM_VER(0, SM_ADD_OVERLAY_LIST)
#define SM_ADD_OVERLAY_LIST_VERSION SM_ADD_OVERLAY_LIST_VERSION_0
/*! @brief Add overlay list structure initialization routine */
static __inline void INIT_SM_ADD_OVERLAY_LIST(PSM_ADD_OVERLAY_LIST pStruct)
{
   int ii;
   pStruct->unVersion   = SM_ADD_OVERLAY_LIST_VERSION;
   pStruct->unCount     = 0;     
   for (ii=0; ii<SM_MAX_OVERLAYS; ii++)
   {
      INIT_SM_ADD_OVERLAY(&(pStruct->addOverlays[ii]));
   }
}

/*!
 * @struct SM_ADD_OVERLAY_RESULT
 * @struct tagSM_ADD_OVERLAY_RESULT
 * @brief Add overlay result structure
 */
typedef struct tagSM_ADD_OVERLAY_RESULT
{
     /*! overlay id structure version  */
   unsigned int           unVersion;
      /*! overlay builder overlay template handle */
   OB_OVERLAY_HANDLE    hOverlayTemplate;
     /*! overlay snapshot handle */
   OB_OVERLAY_HANDLE    hOverlaySnapshot;
     /*! add result code */
   eSM_ADD_OVERLAY_RESULT result;
} SM_ADD_OVERLAY_RESULT, *PSM_ADD_OVERLAY_RESULT;
typedef const PSM_ADD_OVERLAY_RESULT CPSM_ADD_OVERLAY_RESULT;

/*! @brief Add overlay result structure version */
#define SM_ADD_OVERLAY_RESULT_VERSION_0 SM_VER(0, SM_ADD_OVERLAY_RESULT)
#define SM_ADD_OVERLAY_RESULT_VERSION SM_ADD_OVERLAY_RESULT_VERSION_0
/*! @brief Add overlay result structure initialization routine */
static __inline void INIT_SM_ADD_OVERLAY_RESULT(PSM_ADD_OVERLAY_RESULT pStruct)
{
   pStruct->unVersion  = SM_ADD_OVERLAY_RESULT_VERSION;
   pStruct->hOverlayTemplate = MTK_ERROR;
   pStruct->hOverlaySnapshot = MTK_ERROR;
   pStruct->result     = eSM_ADD_OVERLAY_RESULT_ADD;
}

/*!
 * @struct SM_ADD_OVERLAY_RESULT_LIST
 * @struct tagSM_ADD_OVERLAY_RESULT_LIST
 * @brief Add overlay result list structure
 */
typedef struct tagSM_ADD_OVERLAY_RESULT_LIST
{
   unsigned int           unVersion;    /*! add overlay result list structure version  */
   unsigned int           unCount;      /*! number of add overlay results contained in overlayResults */
   SM_ADD_OVERLAY_RESULT  overlayResults[SM_MAX_OVERLAYS];
} SM_ADD_OVERLAY_RESULT_LIST, *PSM_ADD_OVERLAY_RESULT_LIST;
typedef const PSM_ADD_OVERLAY_RESULT_LIST CPSM_ADD_OVERLAY_RESULT_LIST;

/*! @brief Add overlay result structure version */
#define SM_ADD_OVERLAY_RESULT_LIST_VERSION_0 SM_VER(0, SM_ADD_OVERLAY_RESULT_LIST)
#define SM_ADD_OVERLAY_RESULT_LIST_VERSION SM_ADD_OVERLAY_RESULT_LIST_VERSION_0
/*! @brief Add overlay result list structure initialization routine */
static __inline void INIT_SM_ADD_OVERLAY_RESULT_LIST(PSM_ADD_OVERLAY_RESULT_LIST pStruct)
{
   int ii;
   pStruct->unVersion = SM_ADD_OVERLAY_RESULT_LIST_VERSION;
   pStruct->unCount   = 0;
   for (ii=0; ii<SM_MAX_OVERLAYS; ii++)
   {
      INIT_SM_ADD_OVERLAY_RESULT(&(pStruct->overlayResults[ii]));
   }
}


/*!
 * @struct SM_REMOVE_OVERLAY_LIST
 * @struct tagSM_REMOVE_OVERLAY_LIST
 * @brief Remove overlay list structure
 */
typedef struct tagSM_REMOVE_OVERLAY_LIST
{
	unsigned int       unVersion;                        /*! remove overlay list structure version  */
	unsigned int       unCount;                          /*! number of overlay handles contained in overlayHandles. */
	OB_OVERLAY_HANDLE  overlayHandles[SM_MAX_OVERLAYS];  /*! array of overlay snapshot handles */
} SM_REMOVE_OVERLAY_LIST, *PSM_REMOVE_OVERLAY_LIST;
typedef const PSM_REMOVE_OVERLAY_LIST CPSM_REMOVE_OVERLAY_LIST;

/*! @brief Remove overlay list structure version */
#define SM_REMOVE_OVERLAY_LIST_VERSION_0 SM_VER(0, SM_REMOVE_OVERLAY_LIST)
#define SM_REMOVE_OVERLAY_LIST_VERSION SM_REMOVE_OVERLAY_LIST_VERSION_0
/*! @brief Remove overlay list structure initialization routine */
static __inline void INIT_SM_REMOVE_OVERLAY_LIST(PSM_REMOVE_OVERLAY_LIST pStruct)
{
   int ii;
   pStruct->unVersion       = SM_REMOVE_OVERLAY_LIST_VERSION;
   pStruct->unCount         = 0;
   for (ii=0; ii<SM_MAX_OVERLAYS; ii++)
   {
      pStruct->overlayHandles[ii] = MTK_ERROR;
   }
}


/*******************************************************************************
 * Stream Manipulation Media Toolkit API functions
 ******************************************************************************/
#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

   
/**
 * @brief Enable one or more device events.
 * 
 * This function currently is not supported.
 *
 * This function attempts to enable one or more stream manipulation events for
 * a given device. 
 *
 * @param [in] a_hDev - device handle on which to enable the events
 * @param [in] a_pEventInfo - list of events to ensable 
 * @param [in] a_pUserInfo  - pointer to user defined data
 *
 * @return MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Include
 *  sm_mtklib.h
 * @par Synchronous Mode
 *  Not supported
 * @par Asynchronous Mode
 *  If MTK_SUCCESS is returned, the events as described below will be generated.
 * @par Event Data
 *  If the function returns MTK_SUCCESS, it can generate any of the following events:
 * @par
 *  \b SMEV_ENABLE_EVENT: Termination event reported on successful completion
 *  of enabling the event(s).  Event data will contain a list of the events
 *  that were successfully enabled.
 *  <b>Event Data:</b> 
 *  The event data is an SM_EVENT_INFO structure.
 * @par
 *  \b SMEV_ENABLE_EVENT_FAIL: Termination event reported on failure to enable
 *  the event(s).  In this case no events will have been enabled. Event data 
 *  will contain a list of all events that were requested to be enabled.
 *  <b>Event Data:</b> 
 *  The event data is an SM_EVENT_INFO structure.
 * @par
 *  \b SMEV_ERROR: Event reported upon encountering an unexpected error.
 *  <b>Event Data:</b> none
 * @par Notes
 *  This function can be called while the device is streaming data.
 * @par Cautions
 *  <b>Important: ATDV_LASTERR( ) and ATDV_ERRMSGP( ) are not supported for this
 *  function. </b>
 * @par Errors
 *  This function will fail if incorrect parameters specified in the call. Call
 *  mtk_GetErrorInfo() for error information upon failure.
 * @sa sm_DisableEvents()
 */
MTK_API
MTK_RETURN MTK_CONV sm_EnableEvents(SRL_DEVICE_HANDLE a_hDev,
                                    CPSM_EVENT_INFO a_pEventInfo,
                                    void * a_pUserInfo);


/**
 * @brief Disable one or more device events.
 * 
 * This function is currently not supported.
 *
 * This function attempts to disable one or more stream manipulation events for
 * a given device. 
 *
 * @param [in] a_hDev - device handle on which to disable the events
 * @param [in] a_pEventInfo - list of events to disable 
 * @param [in] a_pUserInfo  - pointer to user defined data
 *
 * @return MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Include
 *  sm_mtklib.h
 * @par Synchronous Mode
 *  Not supported
 * @par Asynchronous Mode
 *  If MTK_SUCCESS is returned, the events as described below will be generated.
 * @par Event Data
 *  If the function returns MTK_SUCCESS, it can generate any of the following events:
 * @par
 *  \b SMEV_DISABLE_EVENT: Termination event reported on successful completion
 *  of disabling the event(s).  Event data will contain a list of the events
 *  that were successfully disabled.
 *  <b>Event Data:</b> 
 *  The event data is an SM_EVENT_INFO structure.
 * @par
 *  \b SMEV_DISABLE_EVENT_FAIL: Termination event reported on failure to disable
 *  the event(s).  In this case no events will have been enabled. Event data 
 *  will contain a list of all events that were requested to be disabled.
 *  <b>Event Data:</b> 
 *  The event data is an SM_EVENT_INFO structure.
 * @par
 *  \b SMEV_ERROR: Event reported upon encountering an unexpected error.
 *  <b>Event Data:</b> none
 * @par Notes
 *  This function can be called while the device is streaming data.
 * @par Cautions
 *  <b>Important: ATDV_LASTERR( ) and ATDV_ERRMSGP( ) are not supported for this
 *  function. </b>
 * @par Errors
 *  This function will fail if incorrect parameters specified in the call. Call
 *  mtk_GetErrorInfo() for error information upon failure.
 * @sa sm_EnableEvents()
 */
MTK_API
MTK_RETURN MTK_CONV sm_DisableEvents(SRL_DEVICE_HANDLE a_hDev,
                                     CPSM_EVENT_INFO a_pEventInfo,
                                     void * a_pUserInfo);


/**
 * @brief Adds one or more overlays to a device. 
 * 
 * This function attempts to add one or more overlays to a device.  These overlays
 * will be placed over any video stream generated by that device.  The overlays are 
 * defined using templates created with ob_CreateImageOverlayTemplate() or
 * ob_CreateTextOverlayTemplate(). If any of the overlays associated with the given
 * overlay handles in list a_pOverlayList fails to be added, the entire function
 * will fail and no overlays will be added.
 *
 * The overlay template handle in each SM_ADD_OVERLAY will be mapped to a
 * snapshot of the current state of the overlay template. This mapping will be
 * returned in the event data of this function.
 * 
 * This function can be called while a device is streaming data.
 *
 * <b>Important: Only image overlays are currently supported. Text overlays are not
 * currently supported.</b>
 *
 * <b>Important: The bounding frame must be set for an overlay template 
 * before using it to add an overlay to a device or the add will fail.</b>
 *
 * <b>Important: Do not call ob_DestroyOverlayTemplate() on the handles returned
 * in the add overlay completion event data.</b>
 *
 * @param [in] a_hDev - device handle on to which to add the overlay
 * @param [in] a_pOverlayList - pointer to an add overlay list
 * @param [in] a_pUserInfo  - pointer to user defined data
 *
 * @return MTK_SUCCESS if successful, MTK_ERROR for failure. A failure will be 
 *  retured if this is called on a device that was connected using dev_Connect 
 *  or dev_PortConnect(hDev1, hDev2, DMFL_TRANSCODE_NATIVE).
 * @par Include
 *  sm_mtklib.h
 * @par Synchronous Mode
 *  Not supported
 * @par Asynchronous Mode
 *  If MTK_SUCCESS is returned, the events as described below will be generated.
 *  After a successful addition event is received, the associated overlays can
 *  then be removed from the device by using sm_RemoveOverlays(). 
 * @par Event Data
 *  If the function returns MTK_SUCCESS, it can generate any of the following
 *  events:
 * @par
 *  \b SMEV_ADD_OVERLAY: Termination event reported on successful completion
 *  of adding overlay(s).
 *  <b>Event Data:</b> The event data is a SM_ADD_OVERLAY_RESULT_LIST. The
 *  SM_ADD_OVERLAY_RESULT_LIST contains a list of SM_ADD_OVERLAY_RESULT 
 *  instances. Each of these contains a mapping from one of the template overlay
 *  handles in the original SM_ADD_OVERLAY_LIST to the handle of the snapshot 
 *  overlay. This snapshot handle is used in the call to sm_RemoveOverlays() to
 *  remove this instance of the overlay for the given device.
 * @par
 *  \b SMEV_ADD_OVERLAY_FAIL: Operation failure event reported upon encountering
 *  an error while adding overlay(s).  
 *  <b>Event Data:</b> An SM_ADD_OVERLAY_RESULT_LIST structure. When there is an
 *  add failure, all overlays referenced in a_pOverlayList passed in to
 *  sm_AddOverlays() by the user are considered invalidated. If any of the
 *  overlays referenced had been added previously, that overlay is removed from
 *  the device and from the system. Overlay templates created with 
 *  ob_Create*OverlayTemplate() are not affected. The event data is an
 *  SM_ADD_OVERLAY_RESULT_LIST structure containing an SM_ADD_OVERLAY_RESULT for
 *  each overlay referenced. The hOverlayTemplate in each will be set to the 
 *  handle of the affected overlay. The result data member of each
 *  SM_ADD_OVERLAY_RESULT will be set to eSM_ADD_OVERLAY_RESULT_FAIL.
 * @par
 *  \b SMEV_OVERLAY_CMPLT: <b>This event is temporarily not supported.</b>
 *  Unsolicited event reported when an overlay has completed displaying. 
 *  <b>Event Data:</b> TBD.
 * @par
 *  \b SMEV_ERROR: Unsolicited event reported upon encountering an unexpected
 *  error.
 * @par Notes
 *  For each overlay being added to a device, a snapshot of the parameters for
 *  that overlay is used to create the overlay on the device.  The snapshot 
 *  overlay handle is returned in the completion data for the successful add,
 *  allowing the user to map from the template to the snapshot for that device.
 *
 *  If a snapshot handle is used for an add a second time on the same device by
 *  setting the SM_ADD_OVERLAY hOverlay to the snapshot handle, the current 
 *  parameters for the snapshot overlay will be sent to the device and the 
 *  overlay will be reset on that device. This is considered a modification to
 *  the overlay on that device. In this case, both the snapshot and template
 *  overlay handles in the event data will be the snapshot overlay handle.
 *
 *  If a snapshot handle is used for an add to a different device, that snapshot
 *  overlay is considered a template for that add on the second device.
 *  The current parameters for the snapshot overlay will be sent to the device
 *  and the overlay will be added on the second device.
 *
 *  In all cases, the snapshot overlay handle returned in the 
 *  SM_ADD_OVERLAY_RESULT is to be used to remove the overlay from the device 
 *  by making a call to sm_RemoveOverlays() for the given device.
 *
 *  The SM_ADD_OVERLAY_RESULT result code will be either eSM_ADD_OVERLAY_RESULT_ADD
 *  if the overlay handle is new to the device or eSM_ADD_OVERLAY_RESULT_MODIFY
 *  if the overlay handle already was associated with the device.
 *
 *  The direction, as specified by eDirection in the SM_ADD_OVERLAY, refers to the
 *  direction in which the overlay will be displayed. A value of
 *  eSM_OVERLAY_DIRECTION_DEVICE means that the overlay will only be placed on
 *  the data stream between the device in question and any other device with which
 *  it is connected. A value of eSM_OVERLAY_DIRECTION_NETWORK means that the 
 *  overlays will only be placed on the data stream between the device in question
 *  and the external network. For a media device that does not normally have a 
 *  network, such as a multimedia device, the external network is considered the
 *  sink associated with that device such as a recorded file.
 *
 *  This function can be called while the device is or is not streaming data.
 *
 *  Changes to a given overlay template through the ob_Set*() functions do not
 *  affect those overlays already added to a device until sm_AddOverlays() is 
 *  called again for the overlay handle. In other words the snapshot of the 
 *  overlay is not updated in the firmware until an additional call to 
 *  sm_AddOverlays(), presenting the snapshot overlay handle for that overlay
 *  on that device.
 *
 * @par Cautions
 *  <b>Important: ATDV_LASTERR( ) and ATDV_ERRMSGP( ) are not supported for this
 *  function. </b>
 *  <b>Important: Only image overlays are currently supported. Text overlays are not
 *  currently supported.</b>
 *  <b>Important: All overlays must be removed for a given device before it is
 *  closed. </b>
 *  
 *  This function will fail if called on a device that was connected using
 *  dev_Connect or dev_PortConnect(hDev1, hDev2, DMFL_TRANSCODE_NATIVE).
 *  
 *  The bounding frame must be set for the Overlay template before adding
 *  the overlay to a device or the add will fail.
 *
 *  Do not call sm_DestroyOverlayTemplate() on the handles returned in the add
 *  completion event data.
 * @par Errors
 *  This function will fail if incorrect parameters specified in the call. The
 *  bounding frame must be set for the Overlay template before adding the
 *  overlay to a device or the add will fail. Call mtk_GetErrorInfo for error
 *  information upon failure.
 * @sa ob_CreateTextOverlayTemplate() 
 * @sa ob_CreateImageOverlayTemplate()
 * @sa ob_SetOverlayBoundingFrame()
 * @sa sm_GetOverlays()
 * @sa sm_RemoveOverlays()
 * @sa sm_RemoveAllOverlays()
 */
MTK_API
MTK_RETURN MTK_CONV sm_AddOverlays(SRL_DEVICE_HANDLE a_hDev,
                                   PSM_ADD_OVERLAY_LIST a_pOverlayList,
                                   void * a_pUserInfo);


/**
 * @brief Remove one or more overlays from a device. 
 * 
 * This function attempts to stop and remove one or more overlays previously
 * added to a device. 
 *
 * The overlay handles in a_pOverlayList should be those returned as a snapshot
 * handle in the add completion event data associated with sm_AddOverlays().
 *
 * This function can be called while the device is streaming data.
 *
 * @param [in] a_hDev - device handle from which to remove the overlay(s)
 * @param [in] a_pOverlayList - list of overlays to remove. 
 * @param [in] a_pUserInfo  - pointer to user defined data
 *
 * @return MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Include
 *  sm_mtklib.h
 * @par Synchronous Mode
 *  Not supported
 * @par Asynchronous Mode
 *  If MTK_SUCCESS is returned, the events as described below will be generated.
 * @par Event Data
 *  If the function returns MTK_SUCCESS, it can generate 
 *  any of the following events:
 * @par
 *  \b SMEV_REMOVE_OVERLAY: Termination event reported on successful completion
 *  of removing overlay(s).  Event data will contain a list of the overlay handles
 *  that were successfully removed.
 *  <b>Event Data:</b> 
 *  The event data is an SM_REMOVE_OVERLAY_LIST structure.
 * @par
 *  \b SMEV_REMOVE_OVERLAY_FAIL: Operation failure event reported upon encountering
 *  an error while removing overlay(s).
 *  <b>Event Data:</b> None
 * @par
 *  \b SMEV_ERROR: Event reported upon encountering an unexpected error.
 *  <b>Event Data:</b> none
 * @par Notes
 *  This function can be called while the device is streaming data.
 * @par Cautions
 *  <b>Important: ATDV_LASTERR( ) and ATDV_ERRMSGP( ) are not supported for this
 *  function. </b>
 *  This function should not be called on overlay handles returned from 
 *  ob_CreateTextOverlayTemplate() or ob_CreateImageOverlayTemplate().
 * @par Errors
 *  This function will fail if incorrect parameters specified in the call. Call
 *  mtk_GetErrorInfo() for error information upon failure.
 * @sa sm_AddOverlays()
 */
MTK_API
MTK_RETURN MTK_CONV sm_RemoveOverlays(SRL_DEVICE_HANDLE a_hDev,
                                      PSM_REMOVE_OVERLAY_LIST a_pOverlayList,
                                      void * a_pUserInfo);


/**
 * @brief Remove all overlays from a device. 
 * 
 * This function attempts to stop and remove all overlays previously added to a
 * device.
 *
 * This function can be called while the device is streaming data.
 *
 * @param [in] a_hDev - device handle from which to remove the overlay(s)
 * @param [in] a_pUserInfo  - Pointer to user defined data.
 *
 * @return MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Include
 *  sm_mtklib.h
 * @par Synchronous Mode
 *  Not supported
 * @par Asynchronous Mode
 *  If MTK_SUCCESS is returned, the events as described below will be generated.
 * @par Event Data
 *  If the function returns MTK_SUCCESS, it can generate
 *  any of the following events:
 * @par
 *  \b SMEV_REMOVE_ALL_OVERLAYS: Termination event reported on successful completion
 *  of removing all overlay(s). 
 *  <b>Event Data:</b> None.
 * @par
 *  \b SMEV_REMOVE_OVERLAYS_FAIL: Operation failure event reported upon encountering
 *  an error while removing overlay(s).
 *  <b>Event Data:</b> None.
 * @par
 *  \b SMEV_ERROR: Unsolicited event reported upon encountering an unexpected
 *  error.
 *  <b>Event Data:</b> None.
 * @par Notes
 *  This function can be called while the device is streaming data.
 * @par Cautions
 *  <b>Important: ATDV_LASTERR( ) and ATDV_ERRMSGP( ) are not supported for this
 *  function. </b>
 * @par Errors
 *  This function will fail if incorrect parameters specified in the call. Call
 *  mtk_GetErrorInfo() for error information upon failure.
 * @sa sm_AddOverlays()
 */
MTK_API
MTK_RETURN MTK_CONV sm_RemoveAllOverlays(SRL_DEVICE_HANDLE a_hDev, void * a_pUserInfo);

#ifdef __cplusplus
}
#endif  /* __cplusplus */

#endif /* _SMMTKLIB_H_ */
