#
# dti16_cfg.tcl: DTI 16 Configuration file generator.
# omalleyj
#
# version Alpha 0.07
#
# Copyright Intel Corporation 2002
#
# Script to combine multiple PCD and config files together
# on a per trunk basis for DTI16. Input is the name of a
# configuration and the trunks that get it in the form of a
# bitmask, 00-FF. Configuration name matches a set of PCD and config
# files in the data directory.
set version "Alpha 0.07"

# HELP
proc dti16_cfg_help {} {
puts "Usage: tclsh dti16_cfg.tcl -p1<dti16_p1> -m1<mask1> -pn<dti16_pn> -mn<maskn> -o<output_file> \[-fcd|-nofcd\] \[-h\[elp\]\] \[-v\[ersion\]\]"
puts "Example: tclsh dti16_cfg.tcl <dti16_p1> 00FF <dti16_p2> FF00"
puts "This will create a file with 8 trunks of p1 and 8 trunks of p2."
puts "-p1 though -pn are the numbers of the protocol, the index serves"
puts "only to match it to a mask. Each n must be a decimal digit >= 0."
puts "-m1 through -mn are the 16 place bit maps that indicate the trunk." 
puts "Usually they should total FFFF (all 16 trunks used) and any pair"
puts "must XOR true (no overlap)."
puts "-fcd creates an fcd with fcdgen, which is the default and may be"
puts "omitted. Note that the fcdgen executable must be in your path."
puts "-nofcd suppresses fcd file generation, usually so the config file"
puts "may be further edited manually"
puts "-h shows this message and exits with code -1"
puts "-v shows the version and continues processing other args"
exit -1
}

# MASK2NUMBER
# Counts number of bits set in the mask
proc Mask2Number { Mask } {
set Number 0
while { $Mask } {
	set Mask [expr $Mask & ($Mask - 1)]
	incr Number
}	
return $Number
}

# MAPHIT
# Returns true if Position is 1 in Mask
proc MapHit { Mask Position } {
return [expr $Mask & int(pow(2,$Position - 1))]
}

# GET ARGS
proc DTI16_cfg_Init {} {
global argv version NumberOfTrunks Trunk GenFCD CommentName PCDFileDescription
set state FLAG
set GenFCD true
set MASK_INDEX ""
set PROTOCOL_INDEX ""
foreach arg $argv {
	switch -- $state {
	FLAG {
		switch -glob -- $arg {
		-fcd { set GenFCD true }
		-nofcd { set GenFCD false }
		-v* { puts $version }
		-h* { dti16_cfg_help }
		-m? {	set state MASK
			regexp -- {-m([0-9a-fA-F])} $arg whole MASK_INDEX
		}
		-p? {	set state PROTOCOL
			regexp -- {-p([0-9a-fA-F])} $arg whole PROTOCOL_INDEX
		}
		default { puts "WARNING: Unknown flag $arg"
			 dti16_cfg_help }
		}
	}
	MASK {	set MASK_ARRAY($MASK_INDEX) "$arg"
		set state FLAG
	}
	PROTOCOL {
		set PROTOCOL_ARRAY($PROTOCOL_INDEX) $arg
		set state FLAG
	}
	}
}

set MIs [lsort -dictionary [array names MASK_ARRAY]]
set PIs [lsort -dictionary [array names PROTOCOL_ARRAY]]

foreach MI $MIs {
	if { [info exists PROTOCOL_ARRAY($MI)] } {
		if { [info exists NumberOfTrunks($PROTOCOL_ARRAY($MI))] } {
			set NumberOfTrunks($PROTOCOL_ARRAY($MI)) [expr [Mask2Number $MASK_ARRAY($MI)] + $NumberOfTrunks($PROTOCOL_ARRAY($MI)) ]
		} else {
			set NumberOfTrunks($PROTOCOL_ARRAY($MI)) [Mask2Number $MASK_ARRAY($MI)] 
		}
		if { [info exists MASK_ARRAY($PROTOCOL_ARRAY($MI))] } { 
			set MASK_ARRAY($PROTOCOL_ARRAY($MI)) [expr $MASK_ARRAY($PROTOCOL_ARRAY($MI)) | $MASK_ARRAY($MI)] 
		} else {
			set MASK_ARRAY($PROTOCOL_ARRAY($MI)) $MASK_ARRAY($MI) 
		}
		for {set i 1} {$i <= 16} {incr i} {
			if [MapHit $MASK_ARRAY($MI) $i] {
				set Trunk($i) $PROTOCOL_ARRAY($MI)
			}
		}
		unset MASK_ARRAY($MI)
	} else {
		puts "FATAL:mismatched mask and protocol numbers"
	}
}

set CommentName ""
foreach MI [lsort -dictionary [array names MASK_ARRAY]] {
	set NEXTMASK [format "%#04x %s" $MASK_ARRAY($MI) $MI]
	set CommentName "$CommentName $NEXTMASK"
}

set PCDFileDescription "DTI16, User Generated,"
foreach PROT [array names NumberOfTrunks] {
	set PCDFileDescription "$PCDFileDescription $NumberOfTrunks($PROT) $PROT,"
}
regsub {,$} $PCDFileDescription {} PCDFileDescription 

}

#========================================================
# PCD_WriteGlobalHeaderInfo
# This procedure writes the global PCD information
# The majority of the data is hard-coded as it does not
# vary across DTI16 configurations
# Exceptions are weighted averages of the numbers across
# the configurations.
# Stream size should be verified to be identical across
# all configurations by the calling procedure.
#========================================================
proc PCD_WriteGlobalHeaderInfo {} {
global DEST waPCD CommentName PCDFileDescription
puts $DEST "\[DMN160TEC] 

  GlobalMemRequired	: 16

  ModelType		: 4608
  HostConfigSize	: 64
  MaxHostMsgs		: 32
  MaxFwMsgs		: 48
  DataAlignment		: 0
  MaxFwHostDataXfer	: 48
  MaxHostFwDataXfer	: 48
  HostConfigMask	: 0
  HwIntInterval		: 30
  DataBlockLimit	: 80
  FileDescription	: $PCDFileDescription 
"
puts $DEST "	NumberMFAInboundQ	: $waPCD(NumberMFAInboundQ)"
puts $DEST "	NumberMFAOutboundQ	: $waPCD(NumberMFAOutboundQ)"
puts $DEST "	flowControlToBoard	: $waPCD(flowControlToBoard)"
puts $DEST "	flowControlFromBoard	: $waPCD(flowControlFromBoard)"

for { set i 0 } { $i <= 2 } { incr i 1 } {
if { $waPCD(StreamGroup$i\_NumStreams) == "0" || \
	$waPCD(StreamGroup$i\_StreamSize) == "0" || \
	$waPCD(StreamGroup$i\_NumStreams) == "" || \
	$waPCD(StreamGroup$i\_StreamSize) == "" } {
continue
}
puts $DEST [format "
	\[StreamGroup (%s)\]
	\{
		NumStreams	: %s
		StreamSize	: %s
	\}
" \
$i \
$waPCD(StreamGroup$i\_NumStreams) \
$waPCD(StreamGroup$i\_StreamSize)]
}

puts $DEST "
  ; parameters for SPGroup 1-3

  \[SPGroup (0-2)\]
  \{
    GMCSBaseAddress	: 0x4000
    PMCSBaseAddress	: 0x6000
  \}
"

}
# end PCD_WriteGlobalHeaderInfo

#========================================================
# PCD_WriteCPInfo
# This procedure writes the global CP information
# Weighted averages across configurations.
# exception:
# CP_PLM the CP load module is checked to be identical.
# Pool and Component numbers are is weighted a weighted
# average.
# Pool size and component options should be verified to
# be identical across configurations by the calling
# procedure.
#========================================================
proc PCD_WriteCPInfo {} {
global waPCD DEST
puts $DEST [format "
  \[CP\]
  \{

    PLMFile		: %s

    NumCStreams		: %s
    PMCSCircularBufSize	: %s
    NumFrameTimers	: %s
    MaxComponents	: %s
    MaxInstances	: %s
    DefaultTaskMsgQSize	: %s
" \
$waPCD(CP_PLMFile) \
$waPCD(CP_NumCStreams) \
$waPCD(CP_PMCSCircularBufSize) \
$waPCD(CP_NumFrameTimers) \
$waPCD(CP_MaxComponents) \
$waPCD(CP_MaxInstances) \
$waPCD(CP_DefaultTaskMsgQSize)]

for { set i 0 } { $i <= 11 } { incr i 1 } { 
puts $DEST [format "	\[LocalPool (%s)\]
	\{
		BlockSize	: %s
		NumBlocks	: %s
		MemRegion	: %s
	\}
" \
$i \
$waPCD(CP_LocalPool$i\_BlockSize) \
$waPCD(CP_LocalPool$i\_NumBlocks) \
$waPCD(CP_LocalPool$i\_MemRegion)]
}

foreach { comp } [lsort -dictionary [array names waPCD CP_COMP_*_Attribute]] {
regexp {CP_COMP_([a-zA-Z]*)_Attribute} $comp whole comp_name 

if { [string compare $waPCD(CP_COMP_$comp_name\_Attribute) "" ] } {
	puts $DEST "	\[COMP $comp_name\]"
	puts $DEST "	\{"
	puts $DEST "		Attribute	:  $waPCD(CP_COMP_$comp_name\_Attribute)"
 	if { [info exists waPCD(CP_COMP_$comp_name\_NumInstances)] } {
		puts $DEST "		NumInstances	: $waPCD(CP_COMP_$comp_name\_NumInstances)"
	}
 	if [string compare $waPCD(CP_COMP_$comp_name\_ConfigOption) "" ] {
		puts $DEST "		ConfigOption	: $waPCD(CP_COMP_$comp_name\_ConfigOption)"
 	}
	if [string compare $waPCD(CP_COMP_$comp_name\_InitOption) "" ] {
		puts $DEST "		InitOption	: $waPCD(CP_COMP_$comp_name\_InitOption)"
	}
	if { [info exists waPCD(CP_COMP_$comp_name\_InitInstance)] && [string compare $waPCD(CP_COMP_$comp_name\_InitInstance) "" ] } {
		puts $DEST "		InitInstance	: $waPCD(CP_COMP_$comp_name\_InitInstance)"
 	}
 	if [string compare $waPCD(CP_COMP_$comp_name\_DependentComp) "" ] {
		puts $DEST "		DependentComp	:  $waPCD(CP_COMP_$comp_name\_DependentComp)"
 	}
	puts $DEST "	\}
"
}
}
puts $DEST "\}"
}

# end PCD_WriteCPInfo

#========================================================
# PCD_WriteClusterInfo
# Currently a weighted average of the only valid cluster
# a TOC cluster.
#========================================================
proc PCD_WriteClusterInfo {} {
global waPCD DEST
puts $DEST "\/\* \{\{CLUSTERCONFIGURATION\}\}
PORTS
	\/\/ Definition of the Resource Ports.
	PORT	RS	IN	1	>	Ri1
	PORT	RS	OUT	1	>	Ro1
	PORT	RS	IN	2	>	Ri2
	PORT	RS	OUT	2	>	Ro2

	\/\/ SC Ports.
	PORT	SC	IN	1	>	Si1
	PORT	SC	OUT	1	>	So1
	PORT	SC	IN	2	>	Si2
	PORT	SC	OUT	2	>	So2

       	\/\/ Newtwork Ports.
	PORT	NW	IN	1	>	Ni
	PORT	NW	OUT	1	>	No
ENDPORTS
ATTRIBUTES
	\/\/ Definition of all the Component Attributes.
	ATTRIBUTE	0	0x1			>  PLY_CompAttr
	ATTRIBUTE	0	0x2			>  REC_CompAttr
	ATTRIBUTE	0	0x5			>  TG_CompAttr
	ATTRIBUTE	0	0x7			>  SD_CompAttr
	ATTRIBUTE	0	0x8	0x8	1	>  SB1_CompAttr
	ATTRIBUTE	0	0x8	0x8	2	>  SB2_CompAttr
	ATTRIBUTE	0	0x8	0x8	3	>  SB3_CompAttr
	ATTRIBUTE	0	0xa			>  CA_CompAttr
	ATTRIBUTE	0	0xe			>  Fax_CompAttr
	ATTRIBUTE	0	0x12			>  TSC_CompAttr
	ATTRIBUTE	0	0x15			>  CCS_CompAttr
	ATTRIBUTE	0	0x16			>  LineAdmin_CompAttr
	ATTRIBUTE	0	0x1a			>  VSRP_CompAttr
	ATTRIBUTE	0	0x1d			>  H323_CompAttr
	ATTRIBUTE	0	0x1e			>  NETTSC_CompAttr
	ATTRIBUTE	0	0x29			>  CID_CompAttr
	ATTRIBUTE	0	0x2b			>  FEP_CompAttr
	ATTRIBUTE	0	0x3f			>  FlcnTransP_CompAttr
	ATTRIBUTE	0	0x42			>  R2MF_CompAttr
	ATTRIBUTE	0	0x100			>  SC_CompAttr

	\/\/ Definition of all the Cluster Attributes
	ATTRIBUTE	0	0x55			>  ExitN_ClusAttr
	ATTRIBUTE	0x1202	20			>  Trc_ClusAttr_FALCON
	ATTRIBUTE	0x1203	0x1			>  Trc_ClusAttr_CAS
	ATTRIBUTE	0x1203	0x2			>  Trc_ClusAttr_ISDN
	ATTRIBUTE	0x1203	0x3			>  Trc_ClusAttr_H323
	ATTRIBUTE       0x1203	0x6			>  Trc_ClusAttr_PDK
	ATTRIBUTE	0x1203	0xFFFFFFFF		>  Trc_ClusAttr_Any

ENDATTRIBUTES
"
puts $DEST [format "CLUSTERTYPE TOC

	STARTINSTANCE > %s
	NUMBEROFCLUSTERS > %s
	 

 ALLOCATECLUSTER TRC_ClusAttr_Any > TrcCluster
 FINDINSTANCEINCLUSTER TrcCluster TSC_CompAttr > TSCInst
 ALLOCATEINSTANCEINCLUSTER TrcCluster SC_CompAttr > SCInstance
 LOCKCLUSTER TrcCluster
 FREECLUSTER TrcCluster
 ASSIGNTS SCInstance Si1

ENDCLUSTERTYPE
" \
$waPCD(STARTINSTANCE) \
$waPCD(NUMBEROFCLUSTERS)]

puts $DEST "\{\{ENDCLUSTERCONFIGURATION\}\} \*\/"

}

#========================================================
# PCD_GetValue
# Uses Token to identify key-value pair
# MergeType indicates how to composite values
# can be one of
# Average - a weighted average
# Max - the largest value
# UnionOfSets - Union of all value
# WarnNotUnique - warns if all values are not identical
# Uses SectionKey to locate, can be omitted if
# token is unique
# returns value
#
#========================================================
proc PCD_GetValue { Token MergeType SectionKey } {
global hfilePCD NumberOfTrunks Trunk
set InSection 0
set bracecount 0
set TmpList ""
set Pattern \*$Token\*
#puts $Pattern
set Section \*$SectionKey\*
#puts $Section
set Value ""
foreach { F } [array names NumberOfTrunks] {
	seek $hfilePCD($F) 0 start
	set InSection 0
	set tmpValue ""
	while { [gets $hfilePCD($F) oneline] >= 0 } {
		if { [string length $SectionKey] == 0 || [string match $Section $oneline] == 1 } {
			set InSection 1
			set bracecount 0
		}
		if { [string match *\{* $oneline] == 1 && $InSection == 1 } { incr bracecount 1 }
		if { [string match *\}* $oneline] == 1 && $InSection == 1 } {
			incr bracecount -1
			if { $bracecount == 0 } {
				set InSection 0
			}
		}
		if { [string match $Pattern $oneline] == 1 && $InSection == 1 } {
			if { [string first ";" $oneline] == -1 ||
				 ([string first ";" $oneline] > [string first $Token $oneline]) } {
				regexp "$Token\[ 	\]*\[:>\]\[ 	\]*(.*)$" $oneline whole tmpValue
				switch $MergeType {
				"Average" { set Value [expr $Value + ($tmpValue * $NumberOfTrunks($F))] }
				"Max" { if { $Value < $tmpValue} { set Value $tmpValue } }
				"UnionOfSets" { 
						if { $tmpValue != "" } {
							if { $Value != "" && [string match *$Value* $tmpValue] == 0 } {
							 	set Value "$Value, $tmpValue"
							} else {
								set Value $tmpValue
							}
						}
					      }
				"WarnNotUnique" {
						 if { $Value != "" && [string match *$Value* $tmpValue] == 0 } {
							return "Illegal non Unique Value"
						 } else {
							set Value $tmpValue
						 }
						}
				}
				break
			}
		}

	}

}

if { $MergeType == "Average" } {
	if { $Value == "" } {
		set Value 0
	}
	set Value [expr $Value / 16]
}


#puts $Value
return $Value
}

#========================================================
# GetAllPCDInfo
# Needs to cycle through multiple PCD Files
# Save significant data in arrays
#========================================================
proc GetAllPCDInfo {} {
global waPCD

set waPCD(NumberMFAInboundQ) [PCD_GetValue NumberMFAInboundQ Average ""]
set waPCD(NumberMFAOutboundQ) [PCD_GetValue NumberMFAOutboundQ Average ""]
set waPCD(flowControlToBoard) [PCD_GetValue flowControlToBoard Average ""]
set waPCD(flowControlFromBoard) [PCD_GetValue flowControlFromBoard Average ""]

set waPCD(StreamGroup0_NumStreams) [PCD_GetValue NumStreams Average {\[StreamGroup (0)\]}]
set waPCD(StreamGroup0_StreamSize) [PCD_GetValue StreamSize Max {\[StreamGroup (0)\]}]
set waPCD(StreamGroup1_NumStreams) [PCD_GetValue NumStreams Average {\[StreamGroup (1)\]}]
set waPCD(StreamGroup1_StreamSize) [PCD_GetValue StreamSize Max {\[StreamGroup (1)\]}]
set waPCD(StreamGroup2_NumStreams) [PCD_GetValue NumStreams Average {\[StreamGroup (2)\]}]
set waPCD(StreamGroup2_StreamSize) [PCD_GetValue StreamSize Max {\[StreamGroup (2)\]}]

set waPCD(CP_PLMFile) [PCD_GetValue PLMFile WarnNotUnique {\[CP\]}]

set waPCD(CP_NumCStreams) [PCD_GetValue NumCStreams Average {\[CP\]}]
set waPCD(CP_PMCSCircularBufSize) [PCD_GetValue PMCSCircularBufSize Max {\[CP\]}]
set waPCD(CP_NumFrameTimers) [PCD_GetValue NumFrameTimers Average {\[CP\]}]
set waPCD(CP_MaxComponents) [PCD_GetValue MaxComponents Max {\[CP\]}]
set waPCD(CP_MaxInstances) [PCD_GetValue MaxInstances Max {\[CP\]}]
set waPCD(CP_DefaultTaskMsgQSize) [PCD_GetValue DefaultTaskMsgQSize Average {\[CP\]}]

set waPCD(CP_LocalPool0_BlockSize) [PCD_GetValue BlockSize WarnNotUnique {\[LocalPool (0)\]}]
set waPCD(CP_LocalPool0_NumBlocks) [PCD_GetValue NumBlocks Average {\[LocalPool (0)\]}]
set waPCD(CP_LocalPool0_MemRegion) [PCD_GetValue MemRegion WarnNotUnique {\[LocalPool (0)\]}]
set waPCD(CP_LocalPool1_BlockSize) [PCD_GetValue BlockSize WarnNotUnique {\[LocalPool (1)\]}]
set waPCD(CP_LocalPool1_NumBlocks) [PCD_GetValue NumBlocks Average {\[LocalPool (1)\]}]
set waPCD(CP_LocalPool1_MemRegion) [PCD_GetValue MemRegion WarnNotUnique {\[LocalPool (1)\]}]
set waPCD(CP_LocalPool2_BlockSize) [PCD_GetValue BlockSize WarnNotUnique {\[LocalPool (2)\]}]
set waPCD(CP_LocalPool2_NumBlocks) [PCD_GetValue NumBlocks Average {\[LocalPool (2)\]}]
set waPCD(CP_LocalPool2_MemRegion) [PCD_GetValue MemRegion WarnNotUnique {\[LocalPool (2)\]}]
set waPCD(CP_LocalPool3_BlockSize) [PCD_GetValue BlockSize WarnNotUnique {\[LocalPool (3)\]}]
set waPCD(CP_LocalPool3_NumBlocks) [PCD_GetValue NumBlocks Average {\[LocalPool (3)\]}]
set waPCD(CP_LocalPool3_MemRegion) [PCD_GetValue MemRegion WarnNotUnique {\[LocalPool (3)\]}]
set waPCD(CP_LocalPool4_BlockSize) [PCD_GetValue BlockSize WarnNotUnique {\[LocalPool (4)\]}]
set waPCD(CP_LocalPool4_NumBlocks) [PCD_GetValue NumBlocks Average {\[LocalPool (4)\]}]
set waPCD(CP_LocalPool4_MemRegion) [PCD_GetValue MemRegion WarnNotUnique {\[LocalPool (4)\]}]
set waPCD(CP_LocalPool5_BlockSize) [PCD_GetValue BlockSize WarnNotUnique {\[LocalPool (5)\]}]
set waPCD(CP_LocalPool5_NumBlocks) [PCD_GetValue NumBlocks Average {\[LocalPool (5)\]}]
set waPCD(CP_LocalPool5_MemRegion) [PCD_GetValue MemRegion WarnNotUnique {\[LocalPool (5)\]}]
set waPCD(CP_LocalPool6_BlockSize) [PCD_GetValue BlockSize WarnNotUnique {\[LocalPool (6)\]}]
set waPCD(CP_LocalPool6_NumBlocks) [PCD_GetValue NumBlocks Average {\[LocalPool (6)\]}]
set waPCD(CP_LocalPool6_MemRegion) [PCD_GetValue MemRegion WarnNotUnique {\[LocalPool (6)\]}]
set waPCD(CP_LocalPool7_BlockSize) [PCD_GetValue BlockSize WarnNotUnique {\[LocalPool (7)\]}]
set waPCD(CP_LocalPool7_NumBlocks) [PCD_GetValue NumBlocks Average {\[LocalPool (7)\]}]
set waPCD(CP_LocalPool7_MemRegion) [PCD_GetValue MemRegion WarnNotUnique {\[LocalPool (7)\]}]
set waPCD(CP_LocalPool8_BlockSize) [PCD_GetValue BlockSize WarnNotUnique {\[LocalPool (8)\]}]
set waPCD(CP_LocalPool8_NumBlocks) [PCD_GetValue NumBlocks Average {\[LocalPool (8)\]}]
set waPCD(CP_LocalPool8_MemRegion) [PCD_GetValue MemRegion WarnNotUnique {\[LocalPool (8)\]}]
set waPCD(CP_LocalPool9_BlockSize) [PCD_GetValue BlockSize WarnNotUnique {\[LocalPool (9)\]}]
set waPCD(CP_LocalPool9_NumBlocks) [PCD_GetValue NumBlocks Average {\[LocalPool (9)\]}]
set waPCD(CP_LocalPool9_MemRegion) [PCD_GetValue MemRegion WarnNotUnique {\[LocalPool (9)\]}]
set waPCD(CP_LocalPool10_BlockSize) [PCD_GetValue BlockSize WarnNotUnique {\[LocalPool (10)\]}]
set waPCD(CP_LocalPool10_NumBlocks) [PCD_GetValue NumBlocks Average {\[LocalPool (10)\]}]
set waPCD(CP_LocalPool10_MemRegion) [PCD_GetValue MemRegion WarnNotUnique {\[LocalPool (10)\]}]
set waPCD(CP_LocalPool11_BlockSize) [PCD_GetValue BlockSize WarnNotUnique {\[LocalPool (11)\]}]
set waPCD(CP_LocalPool11_NumBlocks) [PCD_GetValue NumBlocks Average {\[LocalPool (11)\]}]
set waPCD(CP_LocalPool11_MemRegion) [PCD_GetValue MemRegion WarnNotUnique {\[LocalPool (11)\]}]

# [COMP LineAdmin]
set waPCD(CP_COMP_LineAdmin_Attribute) [PCD_GetValue Attribute WarnNotUnique {\[COMP LineAdmin\]}]
set waPCD(CP_COMP_LineAdmin_NumInstances) [PCD_GetValue NumInstances Average {\[COMP LineAdmin\]}]
set waPCD(CP_COMP_LineAdmin_ConfigOption) [PCD_GetValue ConfigOption WarnNotUnique {\[COMP LineAdmin\]}]
set waPCD(CP_COMP_LineAdmin_InitOption) [PCD_GetValue InitOption WarnNotUnique {\[COMP LineAdmin\]}]
set waPCD(CP_COMP_LineAdmin_DependentComp) [PCD_GetValue DependentComp UnionOfSets {\[COMP LineAdmin\]}]
# [COMP TSC]
set waPCD(CP_COMP_TSC_Attribute) [PCD_GetValue Attribute WarnNotUnique {\[COMP TSC\]}]
set waPCD(CP_COMP_TSC_NumInstances) [PCD_GetValue NumInstances Average {\[COMP TSC\]}]
set waPCD(CP_COMP_TSC_ConfigOption) [PCD_GetValue ConfigOption WarnNotUnique {\[COMP TSC\]}]
set waPCD(CP_COMP_TSC_InitOption) [PCD_GetValue InitOption WarnNotUnique {\[COMP TSC\]}]
set waPCD(CP_COMP_TSC_DependentComp) [PCD_GetValue DependentComp UnionOfSets {\[COMP TSC\]}]
# [COMP CHP]
set waPCD(CP_COMP_CHP_Attribute) [PCD_GetValue Attribute WarnNotUnique {\[COMP CHP\]}]
set waPCD(CP_COMP_CHP_NumInstances) [PCD_GetValue NumInstances Average {\[COMP CHP\]}]
set waPCD(CP_COMP_CHP_ConfigOption) [PCD_GetValue ConfigOption WarnNotUnique {\[COMP CHP\]}]
set waPCD(CP_COMP_CHP_InitOption) [PCD_GetValue InitOption WarnNotUnique {\[COMP CHP\]}]
set waPCD(CP_COMP_CHP_DependentComp) [PCD_GetValue DependentComp UnionOfSets {\[COMP CHP\]}]
# [COMP CAS]
set waPCD(CP_COMP_CAS_Attribute) [PCD_GetValue Attribute WarnNotUnique {\[COMP CAS\]}]
set waPCD(CP_COMP_CAS_NumInstances) [PCD_GetValue NumInstances Average {\[COMP CAS\]}]
set waPCD(CP_COMP_CAS_ConfigOption) [PCD_GetValue ConfigOption WarnNotUnique {\[COMP CAS\]}]
set waPCD(CP_COMP_CAS_InitOption) [PCD_GetValue InitOption WarnNotUnique {\[COMP CAS\]}]
set waPCD(CP_COMP_CAS_DependentComp) [PCD_GetValue DependentComp UnionOfSets {\[COMP CAS\]}]
# [COMP CCS]
set waPCD(CP_COMP_CCS_Attribute) [PCD_GetValue Attribute WarnNotUnique {\[COMP CCS\]}]
set waPCD(CP_COMP_CCS_NumInstances) [PCD_GetValue NumInstances Average {\[COMP CCS\]}]
set waPCD(CP_COMP_CCS_ConfigOption) [PCD_GetValue ConfigOption WarnNotUnique {\[COMP CCS\]}]
set waPCD(CP_COMP_CCS_InitOption) [PCD_GetValue InitOption WarnNotUnique {\[COMP CCS\]}]
set waPCD(CP_COMP_CCS_InitInstance) [PCD_GetValue InitInstance WarnNotUnique {\[COMP CCS\]}]
set waPCD(CP_COMP_CCS_DependentComp) [PCD_GetValue DependentComp UnionOfSets {\[COMP CCS\]}]
# [COMP NFAS]
set waPCD(CP_COMP_NFAS_Attribute) [PCD_GetValue Attribute WarnNotUnique {\[COMP NFAS\]}]
set waPCD(CP_COMP_NFAS_ConfigOption) [PCD_GetValue ConfigOption WarnNotUnique {\[COMP NFAS\]}]
set waPCD(CP_COMP_NFAS_InitOption) [PCD_GetValue InitOption WarnNotUnique {\[COMP NFAS\]}]
set waPCD(CP_COMP_NFAS_DependentComp) [PCD_GetValue DependentComp UnionOfSets {\[COMP NFAS\]}]

#set waPCD(CLUSTERTYPE) [PCD_GetValue CLUSTERTYPE UnionOfSets {\/\* \{\{CLUSTERCONFIGURATION\}\}}]
#set waPCD(STARTINSTANCE) [PCD_GetValue STARTINSTANCE WarnNotUnique {\/\* \{\{CLUSTERCONFIGURATION\}\}}]
#set waPCD(NUMBEROFCLUSTERS) [PCD_GetValue NUMBEROFCLUSTERS Average  {\/\* \{\{CLUSTERCONFIGURATION\}\}}]
set waPCD(STARTINSTANCE) [PCD_GetValue STARTINSTANCE WarnNotUnique ""]
set waPCD(NUMBEROFCLUSTERS) [PCD_GetValue NUMBEROFCLUSTERS Average  ""]

}

#========================================================
# Config_SpliceLineAdminInfo
# Needs to cycle through multiple config Files
# Save significant data in an array
#========================================================
proc Config_SpliceLineAdminInfo { } {
global Trunk hfileCFG Cfg_DEST CommentName
puts $Cfg_DEST "! $CommentName"
puts $Cfg_DEST "! The above line is used to track trunk configurations in the interface."
puts $Cfg_DEST "! It is generated automatically and should never be changed or deleted."
foreach { F } [lsort -integer [array names Trunk]] {
	seek $hfileCFG($Trunk($F)) 0 start
	set InSection 0
	while { [gets $hfileCFG($Trunk($F)) oneline] >= 0 } {
		if { $InSection == 1 && [ string match *\]*  $oneline ] == 1 } {
			set InSection 0
			puts $Cfg_DEST ""
			break
		}
		if { $InSection == 1 } {
			puts $Cfg_DEST $oneline
		}	
		if { [string match *lineAdmin.$F\]*  $oneline] == 1 } { 
			set InSection 1
puts $Cfg_DEST "! $Trunk($F)"
			puts $Cfg_DEST $oneline
		}
	}
}
}
# END Config_SpliceLineAdminInfo

#========================================================
# Config_InsertNFASInfo
# Needs to cycle through multiple config Files
# Save significant data in an array
#========================================================
proc Config_InsertNFASInfo { } {
global Cfg_DEST waPCD
if { [info exists waPCD(CP_COMP_NFAS_Attribute)] && [string compare $waPCD(CP_COMP_NFAS_Attribute) "" ] } {
puts $Cfg_DEST "\[NFAS\] 
!  NFAS_INSTANCE_MAP allows the flexibility to decide how many 
!  NFAS instances we want instantiated. So if a board contains 
!  trunks separate in 2 different groups, there will be 2 NFAS 
!  instances instantiated. 

SetParm=0x3e02, 0x1     ! INSTANCE MAP - default = 1, i.e. 1 group/board 
! SetParm=0x6,0xFF       ! NFAS DEBUG LEVEL - turns on additional tracing 

!\[NFAS.1\]                ! NFAS Instance for Group I 
!  Notes on AddNFASInterface: 
! 
! 1.    Allows an NFAS Component Instance to configure the list of NFAS 
!       trunks that it wants to place into its group. Multiple NFAS are 
!       added in recurring sets of triplets as outlined below. 
! 
! 2.    The format of the AddNFASInterface command is as follows: 
!       AddNFASInterface(GroupID)= InterfaceID,BoardNumber,InstanceNumber, 
!       .............and so on............... for multiple trunks: 
! 
!       Where InterfaceID=Unique Num for this interface assigned at subscription 
!             BoardNumber=Board Num on which the the above interface ID is present 
!             InstanceNumber=Instance Num on which the above interface ID is present 
!             GroupID=NFAS Group into which the Interface needs to be added 
! 
! AddNFASInterface(1)=0,0,1, 1,0,2, 2,0,3, 3,0,4 ! Example of adding 4 trunks in group 1 
! SetParm=0x3e04,0                   ! Set Primary Dchannel as Interface '0' 
                                     ! Note: ID MUST BE present in above group 
                        
!\[NFAS.2\]                            ! NFAS Instance for Group 2 
!AddNFASInterface(2)=0,0,2, 1,0,4    ! Example of adding 2 trunks in group 2 
!SetParm=0x3e04,0                    ! Set Primary Dchannel as Interface '0' 
                                     ! Note: ID MUST BE present in above group 

!\[NFAS.3\]                            ! NFAS Instance for Group 3 
!AddNFASInterface(3)=0,1,1, 1,1,2    ! Example of adding 2 trunks in group 3 
!SetParm=0x3e04,0                    ! Set Primary Dchannel as Interface '0' 
                                     ! Note: ID MUST BE present in above group 

!\[NFAS.4\]                            ! NFAS Instance for Group 4 
!AddNFASInterface(4)=0,1,1, 1,1,2    ! Example of adding 2 trunks in group 4 
!SetParm=0x3e04,0                    ! Set Primary Dchannel as Interface '0' 
"
}

}
# END Config_InsertNFASInfo



#========================================================
# Config_SpliceCCSInfo
# Needs to cycle through multiple config Files
# Save significant data in an array
#========================================================
proc Config_SpliceCCSInfo { } {
global Trunk hfileCFG Cfg_DEST

#Common CCS section
puts $Cfg_DEST "
\[CCS\]
SetParm=5,0xffff	! Bit mapped instance count
SetParm=2,0x10		! MaxInstNum
!SetParm=6,0xff		! Debug level
"

foreach { F } [lsort -integer [array names Trunk]] {
	seek $hfileCFG($Trunk($F)) 0 start
	set InSection 0
	while { [gets $hfileCFG($Trunk($F)) oneline] >= 0 } {
		if { $InSection == 1 && [ string match *\]*  $oneline ] == 1 } {
			set InSection 0
			puts $Cfg_DEST ""
			break
		}
		if { $InSection == 1 } {
			puts $Cfg_DEST $oneline
		}	
		if { [string match *CCS.$F\]*  $oneline] == 1 } { 
			set InSection 1
puts $Cfg_DEST "! $Trunk($F)"
			puts $Cfg_DEST $oneline
		}
	}
}
}
# END Config_SpliceCCSInfo

#========================================================
# Config_SpliceCHPInfo
# Needs to cycle through multiple config Files
# Save significant data in an array
#========================================================
proc Config_SpliceCHPInfo { } {
global Trunk hfileCFG Cfg_DEST

#Common CHP section
puts $Cfg_DEST "
\[CHP\]
!SetParm=0xc,212	     ! CHP Debug level
SetParm=0x1310,1     ! R4 Compatibility Flag
                     ! 0-default, 1-enable, 2-disable
Setparm=0x1311,1	! InitialChanState. 1=InService; 2=OutOfService
			! Default is 1
SetParm=0x1312,1    ! Parm DisableBlock set to (1), the Maintenance Message is NOT sent out on a B-channel status change.
		    ! Parm DisableBlock set to (0), the Maintenance Message IS sent out on a B-channel status change.

"
#! 
#! ISDN Protocol variant definitions
#! 

foreach { F } [lsort -integer [array names Trunk]] {
	seek $hfileCFG($Trunk($F)) 0 start
	set InSection 0
	puts $Cfg_DEST "!"
	while { [gets $hfileCFG($Trunk($F)) oneline] >= 0 } {
		if { $InSection == 1 && [ string match *\]*  $oneline ] == 1 } {
			set InSection 0
			puts $Cfg_DEST ""
			break
		}
		if { $InSection == 1 } {
			if { [string match {Variant Define *}  $oneline] == 1 } { 
				puts $Cfg_DEST "Variant Define $F"
				break				
			} else {
				puts $Cfg_DEST $oneline
			}
		}	
		if { [string match {! ISDN Protocol variant definitions}  $oneline] == 1 } { 
			set InSection 1
			puts $Cfg_DEST $oneline
			puts $Cfg_DEST "! Begin Variant $F Definition from $Trunk($F)"
		}
	}
}
}
# END Config_SpliceCHPInfo

#========================================================
# Config_SpliceTSCInfo
# Needs to cycle through multiple config Files
# Save significant data in an array
#========================================================
proc Config_SpliceTSCInfo { } {
global Trunk hfileCFG Cfg_DEST

puts $Cfg_DEST "\[TSC\]"
foreach { F } [lsort -integer [array names Trunk]] {
	set PATTERN 0,$F,
	set PATTERN defineBSet=$F$PATTERN
	seek $hfileCFG($Trunk($F)) 0 start
	set InSection 0
	while { [gets $hfileCFG($Trunk($F)) oneline] >= 0 } {
		if { $InSection == 1 && [ string match *\]*  $oneline ] == 1 } {
			set InSection 0
			puts $Cfg_DEST ""
			break
		}
		if { $InSection == 1 } {
			if { [string match $PATTERN*  $oneline] == 1 } { 
#puts "PATTERN:$PATTERN  ONELINE:$oneline MATCH:[string match $PATTERN* $oneline]"
puts $Cfg_DEST "! $Trunk($F)"
				set NewVARIANT \\1$F,$F\\2
				regsub -- {(defineBSet=[0-9]+,[0-9]+,[0-9]+,[0-9]+,[0-9]+,)1,1(,[0-9]+,[0-9]+,[0-9]+,[0-9]+,[0-9]+,[0-9]+,[0-9]+,[0-9]+)} $oneline $NewVARIANT newBSET
				puts $Cfg_DEST $newBSET
			}
		}	
		if { [string match *TSC\]*  $oneline] == 1 } { 
			set InSection 1
		}
	}
}
}
# END Config_SpliceTSCInfo

# BEGIN MAIN

# Get all applicable files
cd ../data
DTI16_cfg_Init

#foreach ONE [lsort -integer [array names Trunk]] {
#	puts "$ONE $Trunk($ONE)"
#}
set CompositeName "gdti16"
foreach ONE [array names NumberOfTrunks] {
	set whole ""
	set ProtocolValue ""
	regexp -- {dti16_(isdn_)*([_A-Za-z0-9]+)} $ONE whole junk ProtocolValue
	set CompositeName "$CompositeName\_"
	set CompositeName "$CompositeName$NumberOfTrunks($ONE)\_$ProtocolValue"
}
set CompositeCFG "$CompositeName\.config"
set CompositeFCD "$CompositeName\.fcd"
set CompositePCD "$CompositeName\.pcd"

set DEST [open $CompositePCD w]
set Cfg_DEST [open $CompositeCFG w]

# assumes starting in dialogic/bin with existing files in dialogic/data

foreach { F } [array names NumberOfTrunks] {
set hfilePCD($F) [open $F.pcd r]
set hfileCFG($F) [open $F.config r]
}

GetAllPCDInfo

Config_SpliceLineAdminInfo
Config_InsertNFASInfo
Config_SpliceCCSInfo
Config_SpliceCHPInfo
Config_SpliceTSCInfo

#foreach { D } [array names waPCD] {
#puts "$D : $waPCD($D)"
#}
PCD_WriteGlobalHeaderInfo
PCD_WriteCPInfo
PCD_WriteClusterInfo

foreach { F } [array names NumberOfTrunks] {
close $hfilePCD($F)
close $hfileCFG($F)
}

close $DEST
close $Cfg_DEST
if { [string match $GenFCD "true"] == 1 } {
	catch { exec fcdgen -f $CompositeCFG -o $CompositeFCD } result
	#puts $result
}
cd ../cfg
set DEST [open "tcutil.log" w]
puts $DEST $CompositeName
close $DEST
puts $CompositePCD

