/**********@@@SOFT@@@WARE@@@COPY@@@RIGHT@@@**********************************
* Copyright (C) 2007 Dialogic Corporation. All Rights Reserved.
* Redistribution and use in source and binary forms, with or without
* modification, are permitted provided that the following conditions
* are met:
* 1.      Redistributions of source code must retain the above copyright
* notice, this list of conditions and the following disclaimer.
*
* 2.      Redistributions in binary form must reproduce the above copyright
* notice, this list of conditions and the following disclaimer in
* the documentation and/or other materials provided with the
* distribution.
*
* 3.      Neither the name Dialogic Corporation nor the names of its contributors
* may be used to endorse or promote products derived from this software
* without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
* ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
* LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
* SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
* INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
* CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
* ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
* POSSIBILITY OF SUCH DAMAGE.
***********************************@@@SOFT@@@WARE@@@COPY@@@RIGHT@@@**********/

/***************************************************************
* FILE        : tvl2pcidrv.c
* AUTHOR      : J. Rocha
* DESCRIPTION : Kernel module for interrupt management of devices of 
*               the TVL2 media server.
* HISTORY 
* Date       Who   Description  
* 08-Aug-05  JRF   Created
* 16-Sep-05  JRF   Implemented protection against 
*                  simulatenous opens using atomic variable.
* 13-01-05   JRF   Added control of 9056 local interrupt 
*                  bit.
* 02-27-06   NS    Implement a workaround for RTM interrupts. 
*                  Connect the RTM driver instead of IRQ 193 to IRQ 185.
* 03-15-06   MLB   Export a module command to enable/disable debug logging.
* 06-23-06   MLB   In tvl2IrqCtlProcRead when down_interruptible returns -1,
*                  return the error code instead of ERESTARTSYS.  That code
*                  should not be returned to userland, and if rc is EINTR,
*                  the user process should start the read again.
* 11-09-06   NS    Added numOfLicensedtrunks field to the proc pstn.cfg file.
* 10-26-06   MLB   Revamp debug output to reduce volume, added new defines for
*                  logging, added separate AMC, RTM interrupt counts to /proc
* 04-27-07   NS    Add Zionsville Offload board detection logic.
* 05-30-07   NS    Implemented the interrupt enable macros 
*                  for Zionsville.
* 07-27-07   LMK   Added checks for Dialogic Sub-Vendor ID in addition to
*                  Intel Sub-Vendor ID everywhere we check for this in 
*                  tvl2PciDrvProbe()
* 10-02-07   BMN   Added HSI interface support to the driver.
****************************************************************/ 
#define HARDWARE 1
#define ENABLE_ZIONSVILLE_CHANGES
#define CONFIG_PROBE_AND_CLAIM 1 

#ifdef	MODULE
#include <linux/module.h>	/* Kernel module definitions */
#include <linux/moduleparam.h>
#endif
#include <linux/init.h>
#include <linux/kernel.h>	/* We will be "in the kernel" mode of execution */

#include <linux/fs.h>
#include <linux/slab.h>
#include <linux/proc_fs.h>
#include <linux/timer.h>
#include <linux/interrupt.h>
#include <linux/pci.h>
#include <linux/version.h>
#include <asm/uaccess.h>
#include <asm/semaphore.h>

#include "tvl2pcidrv.h" 
#include "mma_sync.h"


/* declare a parameter to use with insmod to enable/disable logging */
short verbose;
module_param(verbose, short, 0);
MODULE_PARM_DESC(verbose, "enable (1) or disable (0, default) debug logging.");

/* Utilities */
#define OPTION_RTM 1  // enable RTM debug printk
#define OPTION_AMC 2  // enable AMC debug printk
#define OPTION_OTH 7  // enable all debug printk

#define TVL2DRV_IRQ_DBG(fmt,args...) \
        { \
            if (verbose) \
            { \
              if ((verbose & OPTION_AMC) && \
                  pIrqCtlData == &g_IrqCtlData[AMC_IRQ_CTL_DATA]) \
              { \
                printk(KERN_INFO "tvl2PciDrv: AMC " fmt, ## args); \
              } \
              else if (verbose & OPTION_RTM && \
                  pIrqCtlData == &g_IrqCtlData[RTM_IRQ_CTL_DATA]) \
              { \
                printk(KERN_INFO "tvl2PciDrv: RTM " fmt, ## args); \
              } \
              else if (pIrqCtlData != &g_IrqCtlData[AMC_IRQ_CTL_DATA] && \
                       pIrqCtlData != &g_IrqCtlData[RTM_IRQ_CTL_DATA]) \
              { \
                printk(KERN_ERR "tvl2PciDrv: UNKNOWN DEVICE " fmt, ## args); \
              } \
           } \
        }

#define TVL2DRV_ERR(args...) \
        { {printk(KERN_ERR "tvl2PciDrv: " args);} }

#define TVL2DRV_DBG(args...) \
        { if (verbose==OPTION_OTH) {printk(KERN_ERR "tvl2PciDrv: " args);} }

/*
 * The The PLX9056 device on alpha TVL2 RTM cards contained incorrect 
 * device number. The device contained device number 3 instead of number 0. 
 * This caused the BIOS to incorrectly map the device interrupt. 
 * The workaround bellow connects the RTM driver to IRQ 185 instead 
 * of IRQ 193. The beta RTMs had this problem corrected and the workaround
 * is not needed.
 *
 * The workaround detect the presence of the TVl2 alpha RTM by
 * expecting the RTM to be on bus 0xA and have a device id of 3.
 */
#if 0
#define IRQ_WORKAROUND 
#else
#define IRQ_WORKAROUND \
  if ( ((pPCIDevice->vendor)            == TVL2_PCI_VENDOR_ID_PLX)        && \
       ((pPCIDevice->device)            == TVL2_PCI_DEVICE_ID_PLX_9056)   && \
       (((pPCIDevice->subsystem_vendor) == TVL2_PCI_VENDOR_ID_DIALOGIC)   || \
        ((pPCIDevice->subsystem_vendor) == TVL2_PCI_VENDOR_ID_INTEL))  && \
	   ((pPCIDevice->subsystem_device)  == TVL2_PCI_SUBDEVICE_ID_RTM)     && \
       ((pPCIDevice->bus->number)       == 0xA)                           && \
       ((PCI_SLOT(pPCIDevice->devfn)) == 0x3)                           && \
       ((PCI_FUNC(pPCIDevice->devfn)) == 0x0) ) \
  { \
      TVL2DRV_ERR("%s: RTM is an ALPHA board?\n", __FUNCTION__); \
      if ( pPCIDevice->irq == 193 ) \
      { \
           pPCIDevice->irq = 185; \
           TVL2DRV_ERR("%s: WORKAROUND: Changed RTM IRQ 193 to 185\n", \
                       __FUNCTION__); \
      } \
      else if ( pPCIDevice->irq == 185 ) \
      { \
          TVL2DRV_ERR("tvl2PciDrvProbe - WORKAROUND: RTM already has IRQ 185 assigned\n"); \
      } \
      else \
      { \
          TVL2DRV_ERR("%s - unable to implement workaround from IRQ %d\n", \
                      __FUNCTION__, pPCIDevice->irq); \
          return -ENODEV;  \
      } \
  }
#endif

#define TVL2_AMC_FPGA_BAR   (TVL2_PCI_BAR2)
#define TVL2_RTM_FPGA_BAR   (TVL2_PCI_BAR2)

typedef enum {
    AMC_IRQ_CTL_DATA,
    RTM_IRQ_CTL_DATA
} Tvl2IrqCtlId_t;


/*
 *      AMC timer registers used by this
 *      application.
 */
#define AMC_REG_INT_STATUS         (0x2000)
#define AMC_REG_INT_MASK           (0x2800)


#ifdef ENABLE_ZIONSVILLE_CHANGES
/* CR22070 - BMN: Adding the rest of the registers that need to be accessed */
/* in order to use the HSI functionality of the ZVL board.                  */
#define AMC_REG_TIMER_PRESET       (0x3000)
#define AMC_REG_TIMER_LOAD         (0x3800)
#define AMC_REG_TIMER_CONTROL      (0x4000)
#define AMC_REG_TIMER_LATCH        (0x4800)
#define AMC_REG_TIMER_COUNTER      (0xE000)
#define AMC_REG_INDEX_BUFFER_SIZE  (0xE800)
#define AMC_REG_SLAVE_ERR_CORR     (0xD000)
#define AMC_REG_SLAVE_RATE         (0xD800)
#define AMC_REG_SLAVE_COUNTER      (0xC000)
#endif

#define RTM_REG_INT_STATUS         (0x2000)
#define RTM_REG_INT_MASK           (0x2800)

#define TVL2_RTM_INT_MASK          (0x1fff)
#define TVL2_AMC_INT_MASK          (0x07ff)
#define ZVL_INT_MASK               (0x1fff)
#define DEFAULT_INT_MASK           (0xffff)

/*
 *      PLX9056 registers used by this
 *      application.
 */
#define PLX9056_REG_INTCSR           	(0x0068)
#define PLX9056_REG_DMA0_MODE        	(0x0080)
#define PLX9056_REG_DMA1_MODE        	(0x0094)
#define PLX9056_REG_OUTPOST_INT_STATUS  (0x0030)

#define PLX9056_INTCSR_PCI_INT_ENABLE   (1 <<  8)
#define PLX9056_INTCSR_LINT_ENABLE      (1 << 11)
#define PLX9056_INTCSR_LINT_ACTIVE      (1 << 15)

/*
 * The Command Register definitions inside PCI Config Space.
 * This driver manipulates the Interrupt Disable bit (bi 10)
 * of Command register. Setting this bit to 0 enables PCI 
 * intterupts. Setting the bit to 1 disables the interrupts.
 */

#define  TVL2_PCI_COMMAND_INT_DISABLE   (0x0400)

/***********************  Typedefs  ***************************************/
struct tvl2PciDev_s;

typedef struct tvl2IrqCtlData_s
{
	int PLX9056BarNum;
	int fpgaBarNum;
	unsigned int intMask;
	void* p9056BaseAddr;
	void* pFPGABaseAddr;
	/*
	 * The pciDeviceType filed specifies which PCI device acts as
	 * a interface to the local space memory. The device could be PLX9056 or
	 * Dialogic's Seaville.
	 * The device type will be used to perform different type of algorithm
	 * for enabling/disabling of the global PCI intterupt.
	 */
	int   pciDeviceType;
	unsigned int intStatusReg;
	unsigned int intMaskReg;
   	spinlock_t lock;
	struct tvl2PciDev_s* tvl2PciDevp;
	int isInitialized;
} tvl2IrqCtlData_t;




typedef struct tvl2PciDev_s
{
  int index;                           /* position in the device array */
  struct pci_dev* pPCIDevice;
  struct irq_ctl_s
  {
    int index;
    struct semaphore irqSem;
    struct proc_dir_entry* pProcIrqEntry;
    tvl2IrqCtlData_t* pIrqCtlData;
#if (HARDWARE == 0)
    /* this is used when simulation is enabled */
    struct timer_list simTimer;
    int timerActive;
    spinlock_t simTimerLock;
#endif
  } irqCtl;
  tvl2PciUsrDev_t tvl2PciUsrDev;
} tvl2PciDev_t;  


typedef struct tvl2PciDrv_s
{
  int major;
  int nDevicesFound;
  struct proc_dir_entry* pProcRootEntry;
  struct proc_dir_entry* pProcPstnConfigEntry;
  tvl2PciDev_t tvl2PciDev [N_TVL2_PCI_DEVS];
} tvl2PciDrv_t;



#ifdef ENABLE_ZIONSVILLE_CHANGES
/* CR22070 - BMN:  Adding the necessary type definitions, prototypes, and */
/* global variables that are used to support the HSI interface.           */
/* Type Definitions */
typedef struct tvl2HsiInfo_s
{
  unsigned int     bridgeId;
  unsigned int     numHsiBuffers;
  HSI_TICK_RATE    tickRate;
  HSI_START_TYPE   startType;
  HSI_ISR_CALLBACK isrFunc;
  unsigned int     SlaveMode;
}
tvl2HsiInfo_t;

typedef struct tvl2HsiCallback_s
{
  HSI_RESPONSE_TYPE type;
  union
  {
    HSI_BRIDGE_INIT_OUTPUT         bridge_init;
    HSI_BRIDGE_STOP_OUTPUT         bridge_stop;
    HSI_BRIDGE_INTR_ENABLE_OUTPUT  intr_enable;
    HSI_BRIDGE_INTR_DISABLE_OUTPUT intr_disable;
    HSI_BRIDGE_SYNC_CLOCK_OUTPUT   sync_clock;
  } u;
}
tvl2HsiCallback_t;

/* Function Prototypes */
static int     tvl2HsiOpen (struct inode *inodep, struct file *filep);
static int     tvl2HsiRelease (struct inode *inode, struct file *file);
static int     tvl2HsiIoctl (struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg);
static ssize_t tvl2HsiRead(struct file *file, char *buf, size_t count, loff_t *ppos);
static int     tvl2HsiIoctlBridgeInit( HSI_BRIDGE_INIT_INPUT *init_data );
static int     tvl2HsiIoctlBridgeStop( unsigned long logical_id );
static int     tvl2HsiIoctlIntrEnable( unsigned long logical_id );
static int     tvl2HsiIoctlIntrDisable( unsigned long logical_id );
static int     tvl2HsiIoctlBridgeSync( HSI_BRIDGE_CLOCK_SYNC_INPUT *sync_data );

/* Constants */
#define  MAX_HSI_RESPONSES		( 10 )

/* Global Variables */
static struct file_operations tvl2Hsi_fops = {
	owner:    THIS_MODULE, 
	ioctl:    tvl2HsiIoctl, 
	open:     tvl2HsiOpen,
	read:     tvl2HsiRead,
	release:  tvl2HsiRelease 
};

static tvl2HsiInfo_t     g_HsiInfo = {
	bridgeId:      0,
	numHsiBuffers: 0,
	tickRate:      0,
	startType:	   0,
	isrFunc:       NULL,
	SlaveMode:     0
};

static int               g_HsiMajorNum;
static tvl2HsiCallback_t g_HsiCallbackData[MAX_HSI_RESPONSES];
static unsigned int      g_HsiCallbackReadIndex  = 0;
static unsigned int      g_HsiCallbackWriteIndex = 0;
static atomic_t          g_HsiSyncData;
static spinlock_t        g_HsiLock;
static struct semaphore  g_HsiSem;
static atomic_t          g_HsiCounter = ATOMIC_INIT(1);
#endif	/* #ifdef ENABLE_ZIONSVILLE_CHANGES */



/***********************     Prototypes ***************************************/
/* module init/cleanup */
static int __init tvl2PciDrvInit (void);
static void __exit tvl2PciDrvStop(void);

/* file operations for the driver */
static int tvl2PciDrvOpen (struct inode *inodep, struct file *filep);
static int tvl2PciDrvRelease (struct inode *inode, struct file *file);
static int tvl2PciDrvIoctl (struct inode *inode, struct file *file, unsigned int cmd, 
				unsigned long arg);

/* file operations for the irq proc entries */
static int tvl2IrqCtlProcOpen (struct inode *inodep, struct file *filep);
static int tvl2IrqCtlProcRelease (struct inode *inode, struct file *file);
static ssize_t tvl2IrqCtlProcRead(struct file *file, char *buf, 
	size_t count, loff_t *ppos);
static int tvl2IrqCtlProcIoctl (struct inode *inode, struct file *file, unsigned int cmd, 
				unsigned long arg);
/* use this function before using the device */
static int tvl2PciDrvClaimDevice (struct pci_dev* pPCIDevice);
static void* tvl2PciDrvMapRegion (struct pci_dev* pPCIDevice, int bar);
/* PCI probe function */
static int tvl2PciDrvProbe (struct pci_dev* pDev, const struct pci_device_id* pDevId);
static void tvl2PciDrvRemove (struct pci_dev *pPCIDev);
/* interrupt handler */
static int tvl2PciDrvReleaseDevice (struct pci_dev* pPCIDevice);
static irqreturn_t tvl2PciDrvIsr (int irq, void* ack, struct pt_regs* regs);

static void tvl2PciDrvRemoveIrqProcEntry (tvl2PciDev_t *pDev);
/*
 *	Interrupt control interface. All titusville 2 devices
 *      which are compatible with tvl2pcidrv need to implement
 *      this set of operations. 
 */
static int  tvl2IrqCtlInit          (tvl2IrqCtlData_t* , struct pci_dev*);
static void tvl2IrqCtlDeInit        (tvl2IrqCtlData_t*);
static int  tvl2IrqCtlGetIsrReg     (tvl2IrqCtlData_t*);
static int  tvl2IrqCtlSetMaskIsrReg (tvl2IrqCtlData_t*, unsigned long);
static void tvl2IrqCtlEnableIntr    (tvl2IrqCtlData_t*);
static void tvl2IrqCtlDisableIntr   (tvl2IrqCtlData_t*);
static unsigned int  tvl2IrqCtlIsIntrActive  (tvl2IrqCtlData_t*);

#if (HARDWARE == 0) 
static void tvl2PciDrvInitSimulation (tvl2PciDev_t *tvl2PciDevp);
static void tvl2PciDrvSimTimerCallback (unsigned long data);
static void tvl2PciDrvStartStopSimTimer (tvl2PciDev_t *tvl2PciDevp,int startStopCmd);
static void tvl2PciDrvStartSimTimer (tvl2PciDev_t *tvl2PciDevp);
static void tvl2PciDrvStopSimTimer (tvl2PciDev_t *tvl2PciDevp);
#endif

static int my_atoi (const char* str);

atomic_t openCounter = ATOMIC_INIT(1);
static tvl2PciDrv_t tvl2PciDrv;
static Tvl2PstnHwInfo_t hwInfo; 
static volatile unsigned long g_amcIsrCount;
static volatile unsigned long g_rtmIsrCount;
static volatile unsigned long g_totalIsrCount;
/********************* OS-specific Device Driver structures *********************/

/*
 *   Add Titusville and Zionsville pci devices here
 */
static struct pci_device_id tvl2PciDevIdTbl[] = 
{
#if (HARDWARE == 0) /* these are devices in my development machine, simulation only. */
  { PCI_DEVICE(PCI_VENDOR_ID_INTEL, PCI_ANY_ID) },
  { PCI_DEVICE(PCI_VENDOR_ID_ATI, PCI_ANY_ID) },
#else
  /* TVL2 AMC w/ Dialogic Sub-Vendor ID */
  {
     .vendor      = TVL2_PCI_VENDOR_ID_PLX,
     .device      = TVL2_PCI_DEVICE_ID_PLX_9056,
     .subvendor   = TVL2_PCI_VENDOR_ID_DIALOGIC,
	 .subdevice   = TVL2_PCI_SUBDEVICE_ID_AMC,
     .class       = 0,
     .class_mask  = 0,
     .driver_data = 0
  },
  /* TVL2 AMC w/ Intel Sub-Vendor ID */
  {
     .vendor      = TVL2_PCI_VENDOR_ID_PLX,
     .device      = TVL2_PCI_DEVICE_ID_PLX_9056,
     .subvendor   = TVL2_PCI_VENDOR_ID_INTEL,
     .subdevice   = TVL2_PCI_SUBDEVICE_ID_AMC,
     .class       = 0,
     .class_mask  = 0,
     .driver_data = 0
  },
  /* TVL2 RTM w/ Dialogic Sub-Vendor ID */
  {
     .vendor      = TVL2_PCI_VENDOR_ID_PLX,
     .device      = TVL2_PCI_DEVICE_ID_PLX_9056,
     .subvendor   = TVL2_PCI_VENDOR_ID_DIALOGIC,
	 .subdevice   = TVL2_PCI_SUBDEVICE_ID_RTM,
     .class       = 0,
     .class_mask  = 0,
     .driver_data = 0
  },
  /* TVL2 RTM w/ Intel Sub-Vendor ID */
  {
     .vendor      = TVL2_PCI_VENDOR_ID_PLX,
     .device      = TVL2_PCI_DEVICE_ID_PLX_9056,
     .subvendor   = TVL2_PCI_VENDOR_ID_INTEL,
     .subdevice   = TVL2_PCI_SUBDEVICE_ID_RTM,
     .class       = 0,
     .class_mask  = 0,
     .driver_data = 0
  },
  /* Zionsville Offload card */
  {
     .vendor      = TVL2_PCI_VENDOR_ID_DIALOGIC,
     .device      = TVL2_PCI_DEVICE_ID_DIALOGIC_SEAVILLE,
     .subvendor   = TVL2_PCI_VENDOR_ID_DIALOGIC,
     .subdevice   = TVL2_PCI_SUBDEVICE_ID_ZVL,
     .class       = 0,
     .class_mask  = 0,
     .driver_data = 0
  },
#endif
  { 0, 0, 0, 0, 0, 0, 0},
}; 

/* pci driver structure */
static struct pci_driver tvl2PciDrv_pci =
{
  .name = DEVNAME,
  .id_table = tvl2PciDevIdTbl,
  .probe = tvl2PciDrvProbe,
  .remove = tvl2PciDrvRemove 
};

/* export the pci_device struct to user space */
MODULE_DEVICE_TABLE(pci, tvl2PciDevIdTbl);

/*   File operation structure: this defines 
 *   the operations which can be performed
 *   by driver.
 */
static struct file_operations tvl2PciDrv_fops = {
  owner:          THIS_MODULE, 
  ioctl:          tvl2PciDrvIoctl, 
  open:           tvl2PciDrvOpen,
  release:        tvl2PciDrvRelease 
};

/*   File operation structure: this defines 
 *   the operations which can be performed
 *   by an interrupt proc entry.
 */
static struct file_operations tvl2IrqCtlProc_fops = {
  owner:          THIS_MODULE, 
  ioctl:          tvl2IrqCtlProcIoctl, 
  open:           tvl2IrqCtlProcOpen,
  read:           tvl2IrqCtlProcRead,
  release:        tvl2IrqCtlProcRelease 
};


tvl2IrqCtlData_t g_IrqCtlData [] =
{
	[AMC_IRQ_CTL_DATA] = 
	{
	    .PLX9056BarNum = TVL2_PCI_BAR0,
	    .fpgaBarNum    = TVL2_PCI_BAR2,
	    .intMask       = DEFAULT_INT_MASK,
	    .p9056BaseAddr = 0,
	    .pFPGABaseAddr = 0,
	    .pciDeviceType = 0,
	    .intStatusReg  = AMC_REG_INT_STATUS,
	    .intMaskReg	   = AMC_REG_INT_MASK,
   	    .lock          = SPIN_LOCK_UNLOCKED,
	    .tvl2PciDevp   = NULL,
	    .isInitialized = 0
	},
	[RTM_IRQ_CTL_DATA] = 
	{
	    .PLX9056BarNum = TVL2_PCI_BAR0,
	    .fpgaBarNum    = TVL2_PCI_BAR2,
	    .intMask       = TVL2_RTM_INT_MASK,
	    .p9056BaseAddr = 0,
	    .pFPGABaseAddr = 0,
	    .pciDeviceType = 0,
	    .intStatusReg  = RTM_REG_INT_STATUS,
	    .intMaskReg	   = RTM_REG_INT_MASK,
   	    .lock          = SPIN_LOCK_UNLOCKED,
	    .tvl2PciDevp   = NULL,
	    .isInitialized = 0
	}
};

/*
 * The section bellow contains inline functions to access the registers of
 * PLX9056, Seaville and FPGA registers. PLX9056 and Seaville devices have 
 * compatible register definitions but Seaville device contains erratas
 * which effect the implementation of some of the functions. See the description
 * of the implemenation bellow.
 *
 * The functions in this section include the following:
 *
 *    PLX9056_READ_REG32
 *    PLX9056_WRITE_REG32
 *    PLX9056_ENABLE_LINT
 *    PLX9056_DISABLE_LINT
 *    PLX9056_ENABLE_PCI_INT
 *    PLX9056_DISABLE_PCI_INT
 *    FPGA_READ_REG32
 *    FPGA_WRITE_REG32
 *    
 * The PLX9056 device acts as a bridge between PCI and the Local memory while
 * the Seaville device implements similar functionality between PCIe and the Local memory.
 * Each of these devices processes the accesses to the device internal register 
 * differently than the accesses to the Local memory. The different processing
 * could cause the sequential commands to the device be executed out of sequence.
 * This topic is described further bellow. We will Seaville device as an example.
 * 
 * The write and read accesses to the Seaville's internal registers are executed
 * rigth away in the sequence that they came in. The writes to the Local memory are
 * posted into the internal command queue inside Seaville and are executed
 * when Seaville has time to do it. The reads to the Local memory are not posted.
 * The read from the Local memory causes Seaville to execute all the queued write 
 * commands before completing the read command. The buffering of the writes
 * top the Local memory can cause the problem described bellow.
 *
 * Lets take the following sequence of calls by the application:
 *
 * 1) Disable the PCIe interrupts by setting bit 8 in INTCSR register to 0.
 * 2) Clear the timer interrupt inside FPGA.
 * 3) Enable the PCIe interrupts by setting bit 8 in INTCSR register to 1.
 *
 * Since the write to the FPGA in step (2) is posted, it is possible for step (3)
 * to be executed before step (2) is completed. This would cuase the timer interrupt
 * be active when PCIe interrupt is enabled.
 *
 * To quarantee the order of execution an additional read of the FPGA regsiter
 * must be issued as follows:
 *
 * 1) Disable the PCIe interrupts by setting bit 8 in INTCSR register to 0.
 * 2) Clear the timer interrupt inside FPGA.
 * 3) Read FPGA register.
 * 4) Enable the PCIe interrupts by setting bit 8 in INTCSR register to 1.
 *
 * In some instances this driver code will issue additional FPGA reads in order
 * to guarantee the execution sequence.
 */

/*
 *      Inlines for PLX9056 register access. 
 */
inline unsigned int  PLX9056_READ_REG32(tvl2IrqCtlData_t* pIrqCtlData, int reg) 
{
	if (pIrqCtlData && pIrqCtlData->p9056BaseAddr != 0) 
		return *((volatile unsigned int *) (pIrqCtlData->p9056BaseAddr + reg));

	return 0;

}

inline void PLX9056_WRITE_REG32(tvl2IrqCtlData_t* pIrqCtlData, int reg,unsigned int val) 
{
	if (pIrqCtlData && pIrqCtlData->p9056BaseAddr != 0) 
		*((volatile unsigned int *) (pIrqCtlData->p9056BaseAddr + reg)) = val;
}

/*
 * Function to enable/disable the Local Interrupt Input Enable (LINT)
 * bit inside PLX 9056 INTCSR register. The Seaville device is
 * compatible with PLX 9056 and contains LINT bit in the same location.
 */

inline void PLX9056_ENABLE_LINT(tvl2IrqCtlData_t* pIrqCtlData) 
{
    unsigned int intCsrVal;

	intCsrVal = PLX9056_READ_REG32(pIrqCtlData, PLX9056_REG_INTCSR);
	intCsrVal |= PLX9056_INTCSR_LINT_ENABLE;

	TVL2DRV_IRQ_DBG("%s: INTCSR = 0x%x\n", __FUNCTION__, intCsrVal);
	PLX9056_WRITE_REG32(pIrqCtlData, PLX9056_REG_INTCSR, intCsrVal);
}

inline void PLX9056_DISABLE_LINT(tvl2IrqCtlData_t* pIrqCtlData) 
{
    unsigned int intCsrVal;
	intCsrVal = PLX9056_READ_REG32(pIrqCtlData, PLX9056_REG_INTCSR);
	intCsrVal &= ~PLX9056_INTCSR_LINT_ENABLE;
	TVL2DRV_IRQ_DBG("%s: INTCSR = 0x%x\n", __FUNCTION__, intCsrVal);
	PLX9056_WRITE_REG32(pIrqCtlData, PLX9056_REG_INTCSR, intCsrVal);
}

/*
 * Function to enable/disable the generation of PCI Interrupts to the host.
 *
 * Two different approaches are used to implement this function on
 * TVL2 and Zionsville platforms. The reason for this is listed bellow.
 *
 * TVL2 platform uses PLX 9056 as a PCI interface device. The PCI interrupts
 * are enabled/disabled using PCI Interrupt Enable bit inside INTCSR register.
 *
 * Zionsville platform uses Seaville ASIC as a PCIe interface device.
 * The Seaville ASIC implements a register compatibility with PLX 9056.
 * But the approach os manipulating PCI Interrupt Enable bit inisde INTCSR
 * can not be used on Seaville because of the hardware bug.
 *
 * To workaround the hardware problem with Seaville, the PCI 
 * interrupts will be enabled/disabled using the Interrupt Disable
 * bit inside Command Register of the Cofiguration space. This approach
 * can be used only for Seaville since Seaville is PCI Express device
 * and that bit is supported for PCIe devices only. The PLX 9056
 * does not support Interrupt Disable bit in the command register.
 *
 */

inline void PLX9056_ENABLE_PCI_INT(tvl2IrqCtlData_t* pIrqCtlData) 
{
    if ( pIrqCtlData->pciDeviceType == TVL2_PCI_DEVICE_ID_PLX_9056 )
    {
        unsigned int intCsrVal;

	intCsrVal = PLX9056_READ_REG32(pIrqCtlData, PLX9056_REG_INTCSR);
	intCsrVal |= PLX9056_INTCSR_PCI_INT_ENABLE;

	TVL2DRV_IRQ_DBG("%s: INTCSR = 0x%x\n", __FUNCTION__, intCsrVal);
	PLX9056_WRITE_REG32(pIrqCtlData, PLX9056_REG_INTCSR, intCsrVal);
    }
    else if ( pIrqCtlData->pciDeviceType == TVL2_PCI_DEVICE_ID_DIALOGIC_SEAVILLE )
    {
        u16 wordCmd;

        TVL2DRV_IRQ_DBG("%s: Enable PCI Int in config space\n", __FUNCTION__);

        /* Set Interrupt Disable bit to 0 */
        pci_read_config_word(pIrqCtlData->tvl2PciDevp->pPCIDevice, PCI_COMMAND, &wordCmd);

        wordCmd = wordCmd & (~((u16)TVL2_PCI_COMMAND_INT_DISABLE));

        pci_write_config_word(pIrqCtlData->tvl2PciDevp->pPCIDevice, PCI_COMMAND, wordCmd);
    }
    else
    {
        TVL2DRV_IRQ_DBG("%s: Invalid PCI device type = 0x%x\n", 
            __FUNCTION__, pIrqCtlData->pciDeviceType);
    }
}

inline void PLX9056_DISABLE_PCI_INT(tvl2IrqCtlData_t* pIrqCtlData) 
{
    if ( pIrqCtlData->pciDeviceType == TVL2_PCI_DEVICE_ID_PLX_9056 )
    {
        unsigned int intCsrVal;
	intCsrVal = PLX9056_READ_REG32(pIrqCtlData, PLX9056_REG_INTCSR);
	intCsrVal &= ~PLX9056_INTCSR_PCI_INT_ENABLE;

        TVL2DRV_IRQ_DBG("%s: INTCSR = 0x%x\n", __FUNCTION__, intCsrVal);
	PLX9056_WRITE_REG32(pIrqCtlData, PLX9056_REG_INTCSR, intCsrVal);
    }
    else if ( pIrqCtlData->pciDeviceType == TVL2_PCI_DEVICE_ID_DIALOGIC_SEAVILLE )
    {
        u16 wordCmd;

        TVL2DRV_IRQ_DBG("%s: Disable PCI Int in config space\n", __FUNCTION__);

        /* Set Interrupt Disable bit to 1 */
        pci_read_config_word(pIrqCtlData->tvl2PciDevp->pPCIDevice, PCI_COMMAND, &wordCmd);

        wordCmd = wordCmd | TVL2_PCI_COMMAND_INT_DISABLE;

        pci_write_config_word(pIrqCtlData->tvl2PciDevp->pPCIDevice, PCI_COMMAND, wordCmd);
    }
    else
    {
        TVL2DRV_IRQ_DBG("%s: Invalid PCI device type = 0x%x\n", 
            __FUNCTION__, pIrqCtlData->pciDeviceType);
    }
}


/*
 *      Inlines for fpga register access. 
 */
inline unsigned int FPGA_READ_REG32(tvl2IrqCtlData_t* pIrqCtlData, int reg) 
{
	if (pIrqCtlData && pIrqCtlData->pFPGABaseAddr != 0) 
		return *((volatile unsigned int *) (pIrqCtlData->pFPGABaseAddr + reg));
	return 0;
}

inline void  FPGA_WRITE_REG32(tvl2IrqCtlData_t* pIrqCtlData, int reg, unsigned int val) 
{
	if (pIrqCtlData && pIrqCtlData->pFPGABaseAddr != 0) 
	{
		*((volatile unsigned int *) (pIrqCtlData->pFPGABaseAddr + reg)) = val;
	}

}
/********************* Implementation of tvl2PciDrv file operations  *********************/

/******************************************************************************
* Function Name: <tvl2PciDrvOpen(struct inode* inodep, struct file* filep)>
* Input :
*    <inodep> - not used. 
*    <filep>  - not used. 
* Output       : None
*
* Global Data  : None 
* Description  : Invoked when a user space application opens the device.
*                dirver.  
* Return Value : Return -EBUSY if device driver is opened more than once. 
*                otherwise returns 0 (success).
*****************************************************************************/
static ssize_t tvl2PciDrvOpen(struct inode *inodep, struct file *filep)
{
  TVL2DRV_DBG("%s: open count increment/check\n", __FUNCTION__);

  /* Make sure user space apps open this device only once */
  if (!atomic_dec_and_test (&openCounter))
  {
      TVL2DRV_ERR("%s: multiple opens not supported \n", __FUNCTION__);
      return -EBUSY;  	
  }
  return 0;
  
}
/******************************************************************************
* Function Name: <tvl2PciDrvRelease(struct inode* inodep, struct file* filep)>
* Input :
*    <inodep> - not used. 
*    <filep>  - not used. 
* Output       : None
*
* Global Data  : None 
* Description  : Invoked when a user space application closes the device.
*                dirver (invokes close). Re-initializes atomic open counter.
* Return Value : Always returns 0 (success).
*****************************************************************************/
static int tvl2PciDrvRelease(struct inode *inode, struct file *file)
{
  TVL2DRV_DBG("%s: open count decrement\n", __FUNCTION__);

  atomic_set (&openCounter, 1);
  return 0;
}

/******************************************************************************
* Function Name: <tvl2PciDrvIoctlGetNumDevFound(int __user* pUser)>
* Input        : None
* Output       : <pUser> - user space variable that will store  
*                the number of devices found.   
* Global Data  : None 
* Description  : Retrieves the number of titusville2 devices detected  
* Return Value : Returns -EFAULT if copy fails otherwise returns 0 (success).
*****************************************************************************/
static int tvl2PciDrvIoctlGetNumDevFound (int __user* pUser) 
{
	int* pKernel;

        pKernel  =  &tvl2PciDrv.nDevicesFound;
	return (copy_to_user(pUser, pKernel, sizeof (int)) ? -EFAULT : 0);
}

static int tvl2PciDrvIoctlGetDevice (tvl2PciUsrDev_t __user* pUser)
{
      int index;
      tvl2PciUsrDev_t* pKernel;  

      index = pUser->index;

      if ((index >= tvl2PciDrv.nDevicesFound) || (index < 0))
      {
          TVL2DRV_ERR("Argument invalid (index)\n");
	  return -EINVAL;
      }

      pKernel  = &tvl2PciDrv.tvl2PciDev[index].tvl2PciUsrDev;
      return (copy_to_user(pUser, pKernel, sizeof (tvl2PciUsrDev_t)) ? -EFAULT : 0);
}

static int tvl2PciDrvIoctl (struct inode *inode, struct file *file, unsigned int cmd, 
				unsigned long arg)
{
  TVL2DRV_DBG("%s entered\n", __FUNCTION__);

  /*
   * Check commands before decoding them 
   */
  if (_IOC_TYPE(cmd) !=  TVL2_RTM_DRV_IOC_MAGIC) 
  {
    TVL2DRV_ERR ("%s: invalid command %d\n", __FUNCTION__, cmd);
    return -ENOTTY;
  }

  if (_IOC_NR(cmd) >  TVL2_RTM_DRV_IOC_MAXNR) 
  {
    TVL2DRV_ERR ("%s: invalid command %d\n", __FUNCTION__, cmd);
    return -ENOTTY;
  }

  switch (cmd)
  {
  case IOC_TVL2_RTM_DRV_GET_NUM_DEV:
    {
      int __user* pUser = (int __user *)arg;
      TVL2DRV_DBG("%s: GET_NUM_DEV\n", __FUNCTION__);
      return tvl2PciDrvIoctlGetNumDevFound (pUser);
    }
  case IOC_TVL2_RTM_DRV_GET_DEV:
    {
      tvl2PciUsrDev_t __user *pUser;
      TVL2DRV_DBG("%s: GET_DEV\n", __FUNCTION__);
      pUser = (tvl2PciUsrDev_t __user *)arg;
      return tvl2PciDrvIoctlGetDevice (pUser);
    }
  case IOC_TVL2_RTM_DRV_GET_NW_INFO:
    {
      Tvl2PstnHwInfo_t __user* pUser =  (Tvl2PstnHwInfo_t __user*)arg;  
      Tvl2PstnHwInfo_t * pKernel =  &hwInfo;  
      TVL2DRV_DBG("%s: GET_NW_INFO\n", __FUNCTION__);
      return (copy_to_user(pUser, pKernel, sizeof (Tvl2PstnHwInfo_t)) ? -EFAULT : 0);
    }
    break; 
 case IOC_TVL2_RTM_DRV_GET_TICKS:
    {
      unsigned long __user* pUser;
      unsigned long* pKernel;

      pUser =  (unsigned long __user*)arg;
      pKernel  =  (unsigned long *)&jiffies;
      TVL2DRV_DBG("%s: GET_TICKS\nn", __FUNCTION__);
      return (copy_to_user(pUser, pKernel, sizeof (unsigned long)) ? -EFAULT : 0);
    }

  default:
    TVL2DRV_ERR ("%s: invalid command %d\n", __FUNCTION__, cmd);
    return -EINVAL;
  }
}

static int tvl2PciDrvCreateIrqProcEntry (tvl2PciDev_t *pDev)
{
  TVL2DRV_DBG("%s create %s\n", __FUNCTION__, pDev->tvl2PciUsrDev.name);

  pDev->irqCtl.pProcIrqEntry = 
	create_proc_entry (pDev->tvl2PciUsrDev.name, S_IFREG | S_IRUGO, 
		tvl2PciDrv.pProcRootEntry); 
  if (pDev->irqCtl.pProcIrqEntry)
  {
#ifdef MODULE
      pDev->irqCtl.pProcIrqEntry->owner = THIS_MODULE;
#endif
      pDev->irqCtl.pProcIrqEntry->data       = (void *)pDev; 
      pDev->irqCtl.pProcIrqEntry->read_proc  = 0; 
      pDev->irqCtl.pProcIrqEntry->write_proc = 0;
      pDev->irqCtl.pProcIrqEntry->proc_fops  = &tvl2IrqCtlProc_fops;
#if HARDWARE == 0
      tvl2PciDrvInitSimulation (pDev); 
#endif
  }
  else
  {
      TVL2DRV_ERR("%s: can't install proc dir %s\n", 
                  __FUNCTION__,  pDev->tvl2PciUsrDev.name);
      return -ENOMEM;
  } 
  return 0;
}

static void tvl2PciDrvRemoveIrqProcEntry (tvl2PciDev_t *pDev)
{
  TVL2DRV_DBG("%s remove %s\n", __FUNCTION__, pDev->tvl2PciUsrDev.name);
  if (pDev->irqCtl.pProcIrqEntry)
  {
#if (HARDWARE == 0)
     tvl2PciDrvStartStopSimTimer (pDev, 0);
#endif
     remove_proc_entry (pDev->tvl2PciUsrDev.name, tvl2PciDrv.pProcRootEntry);
  }
}


/************** Implementation of tvl2IrqCtlProc file operations  *************/
/******************************************************************************
* Function Name: <tvl2PciDrvIrqCtlProcOpen(struct inode* inodep, struct file* filep)>
* Input :
*    <inodep> - pointer to entry in /proc which represents the interrupt
*               control device. 
*    <filep>  - not used. 
* Output       : None
*
* Global Data  : None 
* Description  : Invoked when a user space application opens the interrupt.
*                device.  
* Return Value : Return -EBUSY if interrupts cannot be proprerly installed. 
*                , returnrs -EINVAL if device is unsupported, otherwise 
*		 returns 0 (success).
*****************************************************************************/
static ssize_t tvl2IrqCtlProcOpen(struct inode *inodep, struct file *filep)
{
int rc = 0;
struct proc_dir_entry* entry = PDE(inodep);
tvl2PciDev_t* tvl2PciDevp = (tvl2PciDev_t*)entry->data;
struct pci_dev* pPCIDevice = tvl2PciDevp->pPCIDevice;


//    IRQ_WORKAROUND


#if (CONFIG_PROBE_AND_CLAIM == 0) 
    tvl2PciDrvClaimDevice (pPCIDevice);
#endif

    switch (tvl2PciDevp->pPCIDevice->subsystem_device)
    {
       case TVL2_PCI_SUBDEVICE_ID_AMC:
       case TVL2_PCI_SUBDEVICE_ID_ZVL:
          TVL2DRV_DBG("%s opening AMC device -- IRQ %d\n", 
                      __FUNCTION__, pPCIDevice->irq);
	       tvl2PciDevp->irqCtl.pIrqCtlData = &g_IrqCtlData[AMC_IRQ_CTL_DATA];
	       tvl2PciDevp->irqCtl.pIrqCtlData->tvl2PciDevp = tvl2PciDevp;
           if ( tvl2PciDevp->pPCIDevice->subsystem_device == TVL2_PCI_SUBDEVICE_ID_AMC )
           {
               tvl2PciDevp->irqCtl.pIrqCtlData->pciDeviceType = TVL2_PCI_DEVICE_ID_PLX_9056;
           }
           else
           {
               tvl2PciDevp->irqCtl.pIrqCtlData->pciDeviceType = TVL2_PCI_DEVICE_ID_DIALOGIC_SEAVILLE;
           }
	       break;
       case TVL2_PCI_SUBDEVICE_ID_RTM:
          TVL2DRV_DBG("%s opening RTM device -- IRQ %d\n", 
                      __FUNCTION__, pPCIDevice->irq);
          tvl2PciDevp->irqCtl.pIrqCtlData = &g_IrqCtlData[RTM_IRQ_CTL_DATA];
          tvl2PciDevp->irqCtl.pIrqCtlData->tvl2PciDevp = tvl2PciDevp;  
          tvl2PciDevp->irqCtl.pIrqCtlData->pciDeviceType = TVL2_PCI_DEVICE_ID_PLX_9056;
          break;
       default:
          TVL2DRV_ERR("%s: Invalid device tvl2PciDevp = 0x%p, "
                        "pPCIDevice = 0x%p, subsystem_device = %d\n",
                      __FUNCTION__, tvl2PciDevp, tvl2PciDevp->pPCIDevice, 
                      tvl2PciDevp->pPCIDevice->subsystem_device);
          rc = -EINVAL;   
          goto unsupported_device_error;
    } /* switch */

    rc = tvl2IrqCtlInit (tvl2PciDevp->irqCtl.pIrqCtlData, pPCIDevice);
    if (rc != 0) {
        TVL2DRV_ERR("%s: Failed during tvl2IrqCtlInit, rc = %d\n", 
                    __FUNCTION__, rc);
    	goto irq_ctl_iface_error;
    }

    filep->private_data = tvl2PciDevp;

    tvl2IrqCtlEnableIntr(tvl2PciDevp->irqCtl.pIrqCtlData);

    return 0;


irq_ctl_iface_error:
unsupported_device_error:
    TVL2DRV_ERR("%s: exit with error, rc = %d\n", __FUNCTION__, rc);
	return rc;
}


static int tvl2IrqCtlProcRelease(struct inode *inodep, struct file *file)
{
    struct proc_dir_entry* entry = PDE(inodep);
    tvl2PciDev_t* tvl2PciDevp = (tvl2PciDev_t*)entry->data;
    struct pci_dev* pPCIDevice = tvl2PciDevp->pPCIDevice;
    tvl2IrqCtlData_t* pIrqCtlData = tvl2PciDevp->irqCtl.pIrqCtlData;

    TVL2DRV_IRQ_DBG("%s: close\n", __FUNCTION__);
    if (pPCIDevice->irq) 
    {
#if HARDWARE
        /* Release resources allocated for interrupt handling */
        tvl2IrqCtlDeInit(pIrqCtlData);
#else
	tvl2PciDrvStartStopSimTimer (pIrqCtlData->tvl2PciDevp, 0);
#endif
    }
#if (CONFIG_PROBE_AND_CLAIM == 0) 
    tvl2PciDrvReleaseDevice (pPCIDevice);
#endif
    return 0;
}
                                                                                                 

static int tvl2IrqCtlProcIoctl (struct inode *inode, struct file *file, unsigned int cmd, 
				unsigned long arg)
{
int rc = 0;

  TVL2DRV_DBG("%s entered\n", __FUNCTION__);

  /*
   * Check commands before decoding them 
   */
  if (_IOC_TYPE(cmd) !=  TVL2_IRQ_CTL_IOC_MAGIC) 
  {
    TVL2DRV_ERR ("%s: invalid command %d\n", __FUNCTION__, cmd);
    return -ENOTTY;
  }

  if (_IOC_NR(cmd) >  TVL2_IRQ_CTL_IOC_MAXNR) 
  {
    TVL2DRV_ERR ("%s: invalid command %d\n", __FUNCTION__, cmd);
    return -ENOTTY;
  }

  switch (cmd)
  {
      case IOC_TVL2_IRQ_CTL_ENABLE_LINT:
          TVL2DRV_DBG("%s ENABLE_LINT not implemented\n", __FUNCTION__);
	  break;
      case IOC_TVL2_IRQ_CTL_DISABLE_LINT:
          TVL2DRV_DBG("%s ENABLE_LINT not implemented\n", __FUNCTION__);
	  break;
  default:
        TVL2DRV_ERR ("%s: invalid command %d\n", __FUNCTION__, cmd);
	rc = -ENOTTY;
	break;
  }

  return rc;
}

static ssize_t tvl2IrqCtlProcRead(struct file *file, char *buf, 
	size_t count, loff_t *ppos)
{
int rc;
tvl2PciDev_t* pTvl2PciDev;
unsigned long isrReg = 0;
tvl2IrqCtlData_t* pIrqCtlData;

  pTvl2PciDev  = (tvl2PciDev_t*)file->private_data;
  pIrqCtlData  = pTvl2PciDev->irqCtl.pIrqCtlData;

  rc = down_interruptible (&pTvl2PciDev->irqCtl.irqSem);

  if (rc) 
  {
      TVL2DRV_IRQ_DBG("%s: signal received (rc = %d)\n", __FUNCTION__, rc);
      PLX9056_ENABLE_PCI_INT(pIrqCtlData);
      return rc;
  }
  /* 
   * Make sure user's buffer size is big enough to
   * store the 32-bit interrupt status register of
   * the titusville device
   */
  if (count < sizeof (isrReg)) {
        TVL2DRV_ERR("%s: buffer size %d bytes < required %d\n", 
                    __FUNCTION__, count, sizeof(isrReg));
        PLX9056_ENABLE_PCI_INT(pIrqCtlData);
	return -EINVAL;	
  }

  /* reset count to the size of the ISR reg */
  count = sizeof (isrReg);

  /* Read the interrupt status bits */
  isrReg = tvl2IrqCtlGetIsrReg(pTvl2PciDev->irqCtl.pIrqCtlData);

  /* 
   * Mask the sources that generated the interrupts.
   * The interrupts will be unmasked ince they are processed
   * by higher layers.
   */
  tvl2IrqCtlSetMaskIsrReg(pTvl2PciDev->irqCtl.pIrqCtlData, isrReg);

  if (copy_to_user(buf, (char *)&isrReg, count)) {
        TVL2DRV_ERR("tvl2IrqCtlProcRead copy_to_user failed\n");
        PLX9056_ENABLE_PCI_INT(pIrqCtlData);
	return -EFAULT;
  }

  PLX9056_ENABLE_PCI_INT(pIrqCtlData);

  return count; 

}
/****************************************************************
*                    ISR 
****************************************************************/
static irqreturn_t 
tvl2PciDrvIsr (int irq, void* dev, struct pt_regs* regs)
{
#ifdef ENABLE_ZIONSVILLE_CHANGES
	/* CR22070 - BMN:  In order to support the HSI interface through the KMBC, */
	/* the ISR needs to be able to detect the Timer and Slave interrupts so    */
	/* they can be routed to the KMBC.                                         */
	unsigned int      fpga_IntStatusVal = 0;
	unsigned int      fpga_MaskVal      = 0;
	unsigned int      fpga_CounterVal   = 0;
	unsigned long     flags;
	HSI_ISR_CALLBACK  rate_func;
#endif

tvl2IrqCtlData_t* pIrqCtlData;
int handled = 0;
unsigned int source;

	pIrqCtlData  = (tvl2IrqCtlData_t *)dev;

        TVL2DRV_IRQ_DBG("%s: enter, jiffies = %lu, cpu = %d\n", 
            __FUNCTION__, jiffies, smp_processor_id());

	if (pIrqCtlData == NULL)
        {
	    TVL2DRV_ERR("%s: ISR with NULL board data, unhandled interrupt.\n",
                        __FUNCTION__);
	    return IRQ_NONE;
        }

	g_totalIsrCount++;


#ifdef ENABLE_ZIONSVILLE_CHANGES
	/* CR22070 - BMN:  In order to support the HSI interface through the KMBC, */
	/* the ISR needs to be able to detect the Timer and Slave interrupts so    */
	/* they can be routed to the KMBC.                                         */
	if(( source = tvl2IrqCtlIsIntrActive( pIrqCtlData )) != PLX9056_INTR_SOURCE_NONE )
	{
		PLX9056_DISABLE_PCI_INT( pIrqCtlData );
		
		if( pIrqCtlData == &g_IrqCtlData[AMC_IRQ_CTL_DATA] )
		{
			g_amcIsrCount++;

			/* Only execute this code path if the the Bridge Device has been initialized. */
			if( g_HsiInfo.bridgeId != 0 )
			{
				if( source & ( 1 << PLX9056_INTR_SOURCE_LOCAL1 ))
				{
					/* We are looking to handle the Timer and Slave Interrupts differently,   */
					/* so look into the FPGA and see if they are asserted.  If they are, then */
					/* we will send them off to the KMBC.  If there are other interrupts that */
					/* are still pending, they will be handled in the normal way.             */
					fpga_IntStatusVal = FPGA_READ_REG32( pIrqCtlData, AMC_REG_INT_STATUS );
					fpga_MaskVal      = FPGA_READ_REG32( pIrqCtlData, AMC_REG_INT_MASK );

					if(( fpga_IntStatusVal & 0x1 ) && !( fpga_MaskVal & 0x1 ))
					{
						/* This is a Timer Interrupt - if the interrupt isn't being masked by */
						/* the FPGA, then deliver it to the KMBC.                             */
						fpga_CounterVal = FPGA_READ_REG32( pIrqCtlData, AMC_REG_TIMER_COUNTER );

						spin_lock_irqsave( &g_HsiLock, flags );

						rate_func = g_HsiInfo.isrFunc;

						if( rate_func )
						{
							rate_func( g_HsiInfo.bridgeId, fpga_CounterVal );
						}

						spin_unlock_irqrestore( &g_HsiLock, flags );

						/* Clear the Timer Interrupt */
						FPGA_WRITE_REG32( pIrqCtlData, AMC_REG_INT_STATUS, 0x1 );
						FPGA_READ_REG32( pIrqCtlData, AMC_REG_INT_STATUS );
					}

					if(( fpga_IntStatusVal & 0x2000 ) && !( fpga_MaskVal & 0x2000 ))
					{
						/* This is a slave interrupt - deliver to KMBC if we are in slave mode, */
						/* otherwise ignore it.                                                 */
						fpga_CounterVal = FPGA_READ_REG32( pIrqCtlData, AMC_REG_SLAVE_COUNTER );

						spin_lock_irqsave( &g_HsiLock, flags );

						if( g_HsiInfo.SlaveMode )
						{
							rate_func = g_HsiInfo.isrFunc;

							if( rate_func )
							{
								rate_func( g_HsiInfo.bridgeId, fpga_CounterVal );
							}
						}

						spin_unlock_irqrestore( &g_HsiLock, flags );

						/* Clear the slave interrupt */
						FPGA_WRITE_REG32( pIrqCtlData, AMC_REG_INT_STATUS, 0x2000 );
						FPGA_READ_REG32( pIrqCtlData, AMC_REG_INT_STATUS );
					}

					/* Clear both bits so that we can tell if there were other interrupts */
					/* to process.                                                        */
					fpga_IntStatusVal &= 0xFFFE;
					fpga_IntStatusVal &= 0xDFFF;

					if( fpga_IntStatusVal == 0x0 )
					{
						/* There are no more interrupts from the FPGA, so clear that local */
						/* interrupt as a source.                                          */
						source &= ~(1 << PLX9056_INTR_SOURCE_LOCAL1 );
					}
					else
					{
						/* There are other interrupts being reported by the FPGA, check to    */
						/* see if they are being masked out.  If not, than let the existing   */
						/* interrupt mechanism take care of it (through the /proc interface). */
						if(( fpga_MaskVal & fpga_IntStatusVal ) == fpga_IntStatusVal )
						{
							/* All bits that are active in the Interrupt Status register are   */
							/* also being masked out in the Interrupt Mask register, so we are */
							/* done here.                                                      */
							source &= ~(1 << PLX9056_INTR_SOURCE_LOCAL1 );
						}
					}
				}

				if( source == 0 )
				{
					/* All interrupts have already been handled */
					PLX9056_ENABLE_PCI_INT( pIrqCtlData );
					handled = 1;
				}
			}
		}
		else if( pIrqCtlData == &g_IrqCtlData[RTM_IRQ_CTL_DATA] )
		{
			g_rtmIsrCount++;
		}

		if( handled == 0 )
		{
			up( &(pIrqCtlData->tvl2PciDevp->irqCtl.irqSem ));
			handled = 1;
		}
	}
#else
	if ((source = tvl2IrqCtlIsIntrActive (pIrqCtlData)) != PLX9056_INTR_SOURCE_NONE)
        {
		PLX9056_DISABLE_PCI_INT(pIrqCtlData);
                if (pIrqCtlData == &g_IrqCtlData[AMC_IRQ_CTL_DATA])
		    g_amcIsrCount++;
                else if (pIrqCtlData == &g_IrqCtlData[RTM_IRQ_CTL_DATA])
		    g_rtmIsrCount++;
		up(&(pIrqCtlData->tvl2PciDevp->irqCtl.irqSem));
		handled = 1;
	}
#endif

#if 0
        if ( handled == 0 )
        {
            TVL2DRV_ERR("%s: Error, Unhandled interrupt, irq = %d, disable timer\n",
                        __FUNCTION__, irq);
        }
        else
        {
            TVL2DRV_ERR("%s: Handled interrupt, irq = %d\n",
                        __FUNCTION__, irq);
        }
#endif
	return IRQ_RETVAL(handled);
}

/****************************************************************
*                    PCI FUNCTIONS 
****************************************************************/
static void* tvl2PciDrvMapRegion (struct pci_dev* pPCIDevice, int bar)
{
    unsigned int mmioStart;
    unsigned int mmioLen;
    unsigned int base;
    unsigned int offset;
    unsigned int size;

    mmioStart    = pci_resource_start (pPCIDevice, bar);
    mmioLen      = pci_resource_len   (pPCIDevice, bar);

    if (mmioStart % PAGE_SIZE)
    {
  	base     =  mmioStart & ~PAGE_SIZE;
    }
    else
    {
  	base     =  mmioStart;
    }
    offset   = (mmioStart  - base);
    size     = (offset + mmioLen);
  
    return ioremap (mmioStart, mmioLen); 
}


static void tvl2PciDrvUnMapRegion (void* virtualAddr)
{
	iounmap (virtualAddr); 
}


static int tvl2PciDrvClaimDevice (struct pci_dev* pPCIDevice)
{
  int err;

  TVL2DRV_DBG("%s pPCIDevice = 0x%p, irq = %d\n",
              __FUNCTION__, pPCIDevice, pPCIDevice->irq);
#if HARDWARE
    
  /*first, perform device initialization */
  if ((err = pci_enable_device (pPCIDevice))) 
     return err;

  IRQ_WORKAROUND

  /* and do not forget to claim all the device's regions */
  if (pci_request_regions (pPCIDevice, DEVNAME)) 
  {
    TVL2DRV_ERR("tvl2PciDrvClaimDevice :  failed to request pci regions");
    return -EBUSY;
  }
#endif /* HARDWARE */

  return 0;
}


static int tvl2PciDrvReleaseDevice (struct pci_dev* pPCIDevice)
{
#if HARDWARE
  if ( pPCIDevice->subsystem_device==TVL2_PCI_SUBDEVICE_ID_AMC ||
       pPCIDevice->subsystem_device==TVL2_PCI_SUBDEVICE_ID_ZVL)
  {
     TVL2DRV_DBG("%s %s: irq = %d\n", "AMC", __FUNCTION__, pPCIDevice->irq);
  }
  else if (pPCIDevice->subsystem_device==TVL2_PCI_SUBDEVICE_ID_RTM)
  {
     TVL2DRV_DBG("%s %s: irq = %d\n", "RTM", __FUNCTION__, pPCIDevice->irq);
  }
  else
  {
     TVL2DRV_DBG("%s %s: irq = %d\n", "Invalid Device", __FUNCTION__, pPCIDevice->irq);
  }
    
  /* first release the device's resources */
  pci_release_regions (pPCIDevice); 
  /*then disable device  */
  pci_disable_device (pPCIDevice); 

#endif

  return 0;
}


static int tvl2PciDrvProbe (struct pci_dev* pPCIDevice, const struct pci_device_id* pDevId)
{
  int err;
  int bar = 0;
  char name[32];
  tvl2PciDev_t* tvl2PciDevp; 
  tvl2PciUsrDev_t* tvl2PciUsrDevp; 

  TVL2DRV_DBG("%s - Device 0x%x has been found at bus %d dev %d func %d\n",
              __FUNCTION__, pPCIDevice->device, pPCIDevice->bus->number, 
              PCI_SLOT(pPCIDevice->devfn), PCI_FUNC(pPCIDevice->devfn));

  tvl2PciDevp = 0;
  tvl2PciUsrDevp = 0;

  if (tvl2PciDrv.nDevicesFound > N_TVL2_PCI_DEVS)
  {
    TVL2DRV_ERR("%s: devices found %d > max devices %d\n",
                __FUNCTION__, tvl2PciDrv.nDevicesFound, N_TVL2_PCI_DEVS);
    return -ENODEV;   
  }
  tvl2PciDevp = &tvl2PciDrv.tvl2PciDev[tvl2PciDrv.nDevicesFound]; 

  tvl2PciDevp->index = tvl2PciDrv.nDevicesFound;
  tvl2PciDevp->pPCIDevice = pPCIDevice;


  //IRQ_WORKAROUND

#if CONFIG_PROBE_AND_CLAIM
  if ((err = tvl2PciDrvClaimDevice (pPCIDevice)))
     return err;
#endif

  tvl2PciUsrDevp = &tvl2PciDevp->tvl2PciUsrDev;
  tvl2PciUsrDevp->index = tvl2PciDevp->index;
  /* then, find the memory region */
  for (bar = 0; bar < 6; bar++)
  {
    /* this assumes device has only one pci memory region */
    if (pci_resource_flags(pPCIDevice, bar) & IORESOURCE_MEM)
    {
       tvl2PciUsrDevp->regions[bar].type = TVL2_MEMORY_REGION;
    }
    else
    {
       tvl2PciUsrDevp->regions[bar].type = TVL2_IO_REGION;
    }
    tvl2PciUsrDevp->vendor_id = pPCIDevice->vendor;
    tvl2PciUsrDevp->subvendor_id = pPCIDevice->subsystem_vendor;
    tvl2PciUsrDevp->device_id = pPCIDevice->device;
    tvl2PciUsrDevp->subdevice_id = pPCIDevice->subsystem_device;
    tvl2PciUsrDevp->bus = pPCIDevice->bus->number;
    tvl2PciUsrDevp->slot = PCI_SLOT(pPCIDevice->devfn);
    tvl2PciUsrDevp->function = PCI_FUNC(pPCIDevice->devfn);
    tvl2PciUsrDevp->regions[bar].bar = bar;
    tvl2PciUsrDevp->regions[bar].base = pci_resource_start (pPCIDevice, bar);
    tvl2PciUsrDevp->regions[bar].size = pci_resource_len (pPCIDevice, bar);
    TVL2DRV_DBG("tvl2PciDrvProbe - bar = %d, base = %lx, size = %x\n", bar,
		tvl2PciUsrDevp->regions[bar].base, 
		tvl2PciUsrDevp->regions[bar].size);
  }

  TVL2DRV_DBG("tvl2PciDrvProbe - irq = %d\n", pPCIDevice->irq);
  if (pPCIDevice->irq)
  {
    tvl2PciUsrDevp->irq = pPCIDevice->irq; 
    /* set device name */
    sprintf (name, "%d.%d.%d.%d", tvl2PciUsrDevp->bus, 
	tvl2PciUsrDevp->slot, tvl2PciUsrDevp->function, tvl2PciUsrDevp->irq); 
    strcpy (tvl2PciUsrDevp->name, name);
    tvl2PciDrvCreateIrqProcEntry (tvl2PciDevp);
  } 

  init_MUTEX_LOCKED (&tvl2PciDrv.tvl2PciDev[tvl2PciDrv.nDevicesFound].irqCtl.irqSem);

  ++tvl2PciDrv.nDevicesFound;

  /*
   * The number of interrupt bits differ between TVL2 AMC and
   * Zionsville Oflload board. Set the product specific interrupt mask.
   * 
   */
  if ( (pPCIDevice->vendor            == TVL2_PCI_VENDOR_ID_PLX)           &&
       (pPCIDevice->device            == TVL2_PCI_DEVICE_ID_PLX_9056)      &&
       ((pPCIDevice->subsystem_vendor == TVL2_PCI_VENDOR_ID_DIALOGIC) ||
        (pPCIDevice->subsystem_vendor == TVL2_PCI_VENDOR_ID_INTEL))        &&
	   (pPCIDevice->subsystem_device  == TVL2_PCI_SUBDEVICE_ID_AMC) )
  {
    g_IrqCtlData[AMC_IRQ_CTL_DATA].intMask = TVL2_AMC_INT_MASK;
  }
  else if ( (pPCIDevice->vendor           == TVL2_PCI_VENDOR_ID_DIALOGIC)           &&
            (pPCIDevice->device           == TVL2_PCI_DEVICE_ID_DIALOGIC_SEAVILLE)  &&
            (pPCIDevice->subsystem_vendor == TVL2_PCI_VENDOR_ID_DIALOGIC)           &&
            (pPCIDevice->subsystem_device == TVL2_PCI_SUBDEVICE_ID_ZVL) )
  {
    g_IrqCtlData[AMC_IRQ_CTL_DATA].intMask = ZVL_INT_MASK;
  }
  return 0; 
}


inline static int tvl2PciDrvSearchEntry (struct pci_dev *pPCIDevice)
{
int i=0;

  tvl2PciDev_t* tvl2PciDevp;
  for (i=0; i < tvl2PciDrv.nDevicesFound; i++)
  {
    tvl2PciDevp = &tvl2PciDrv.tvl2PciDev[i];
    if ((pPCIDevice->bus->number == tvl2PciDevp->pPCIDevice->bus->number)
		&& (pPCIDevice->devfn == tvl2PciDevp->pPCIDevice->devfn))
    {
	break;
    }
  }
  return i;
}


static void tvl2PciDrvRemove (struct pci_dev *pPCIDevice)
{
    int index;
    tvl2PciDev_t* tvl2PciDevp;

    TVL2DRV_DBG("%s - Device 0x%x at bus %d dev %d func %d\n",
              __FUNCTION__, pPCIDevice->device, pPCIDevice->bus->number, 
              PCI_SLOT(pPCIDevice->devfn), PCI_FUNC(pPCIDevice->devfn));

    index = tvl2PciDrvSearchEntry(pPCIDevice);
    if (index < tvl2PciDrv.nDevicesFound)
    {
       tvl2PciDevp  = &tvl2PciDrv.tvl2PciDev[index]; 
    }
    else
       return;

    if (pPCIDevice->irq)
        tvl2PciDrvRemoveIrqProcEntry (tvl2PciDevp);

#if CONFIG_PROBE_AND_CLAIM 
	tvl2PciDrvReleaseDevice( pPCIDevice );
#endif

}

/*********************** Tvl2 device-specific ***************************/
static int  
tvl2IrqCtlInit(
    tvl2IrqCtlData_t* pIrqCtlData, 
    struct pci_dev* pPCIDevice)
{
    int rc = 0;

//    spin_lock (&pIrqCtlData->lock);

    if ( pIrqCtlData->isInitialized != 0 )
    {
        rc = -EBUSY;
        goto out;	
    }

    /*
     *     Map BAR0 of the PLX9056, so we can access PLX9056 INTCSR
     */
    TVL2DRV_IRQ_DBG("tvl2IrqCtlInit : mapping BAR%d\n", \
		pIrqCtlData->PLX9056BarNum);
    pIrqCtlData->p9056BaseAddr = \
		tvl2PciDrvMapRegion (pPCIDevice, pIrqCtlData->PLX9056BarNum);

    if (pIrqCtlData->p9056BaseAddr)
    {
        TVL2DRV_IRQ_DBG("tvl2IrqCtlInit mapped 9056 to %p\n", \
                    pIrqCtlData->p9056BaseAddr);
    }
    else
    {
        TVL2DRV_ERR("tvl2IrqCtlInit unable to map 9056\n");
        rc = -EIO;
        goto out;
    }
    /*
    *     Map BAR2 of the PLX9056, so we can access FPGA registers. 
    */
    pIrqCtlData->pFPGABaseAddr = \
		tvl2PciDrvMapRegion (pPCIDevice, pIrqCtlData->fpgaBarNum);

    if (pIrqCtlData->pFPGABaseAddr != 0) 
    {
        TVL2DRV_IRQ_DBG("tvl2IrqCtlInit mapped FPGA to %p\n", \
                    pIrqCtlData->pFPGABaseAddr);
    }
    else
    {
        TVL2DRV_ERR("tvl2IrqCtlInit unable to map FPGA\n");
        rc = -EIO;
        tvl2PciDrvUnMapRegion (pIrqCtlData->p9056BaseAddr);
        goto out;
    }

#if HARDWARE
   TVL2DRV_IRQ_DBG("%s: request irq = %d\n", __FUNCTION__, pPCIDevice->irq);

    /* install ISR as shared handler */
    if (request_irq (pPCIDevice->irq, tvl2PciDrvIsr, SA_SHIRQ, \
                     DEVNAME, (void*) pIrqCtlData)) 
    {
        TVL2DRV_ERR("tvl2PciDrvProcOpen - unable to install ISR");
        tvl2PciDrvUnMapRegion (pIrqCtlData->p9056BaseAddr);
        tvl2PciDrvUnMapRegion (pIrqCtlData->pFPGABaseAddr);
        rc = -EBUSY; 
        goto out;  
    }
#else
    tvl2PciDrvStartStopSimTimer (tvl2PciDevp, 1);
#endif  
    pIrqCtlData->isInitialized = 1;

out:
//    spin_unlock(&pIrqCtlData->lock);
    return rc;
}

static void tvl2IrqCtlDeInit (tvl2IrqCtlData_t* pIrqCtlData)
{
	if (pIrqCtlData == NULL)
	{
		TVL2DRV_ERR("tvl2IrqCtlDeInit: invalid argument\n");
		return;
	}

  	spin_lock (&pIrqCtlData->lock);

        if ( pIrqCtlData->isInitialized == 0 )
	{
  		spin_unlock (&pIrqCtlData->lock);
		return;	
	}
	tvl2IrqCtlDisableIntr(pIrqCtlData);

	/* Wait for all handlers to complete */
	synchronize_irq(pIrqCtlData->tvl2PciDevp->pPCIDevice->irq);
#if HARDWARE
	TVL2DRV_IRQ_DBG("%s: freeing  irq = %d\n",
	    __FUNCTION__, pIrqCtlData->tvl2PciDevp->pPCIDevice->irq);
	free_irq(pIrqCtlData->tvl2PciDevp->pPCIDevice->irq, (void*) pIrqCtlData);
#endif
	/* Wait for all handlers to complete */
	synchronize_irq(pIrqCtlData->tvl2PciDevp->pPCIDevice->irq);

	if (pIrqCtlData->pFPGABaseAddr != 0) 
	   tvl2PciDrvUnMapRegion (pIrqCtlData->pFPGABaseAddr);
        pIrqCtlData->pFPGABaseAddr = 0;
	if (pIrqCtlData->p9056BaseAddr != 0) 
	   tvl2PciDrvUnMapRegion (pIrqCtlData->p9056BaseAddr);
        pIrqCtlData->p9056BaseAddr = 0;
	pIrqCtlData->isInitialized = 0;
  	spin_unlock (&pIrqCtlData->lock);
}

static int tvl2IrqCtlGetIsrReg(tvl2IrqCtlData_t* pIrqCtlData)
{
	/* TODO: check local interrupt bit is set */

        if (pIrqCtlData != 0)
		return (FPGA_READ_REG32 (pIrqCtlData, pIrqCtlData->intStatusReg));

	return (0);
}

static int tvl2IrqCtlSetMaskIsrReg(tvl2IrqCtlData_t* pIrqCtlData, unsigned long maskValue)
{
    if (pIrqCtlData == NULL)
    {
        TVL2DRV_ERR("tvl2IrqCtlSetMaskIsrReg, Error pIrqCtlData = NULL\n");
	return(0);
    }

    if (pIrqCtlData->pFPGABaseAddr)
    {
        unsigned long intMaskVal = 0;
        intMaskVal = FPGA_READ_REG32(pIrqCtlData, pIrqCtlData->intMaskReg);
        intMaskVal |= maskValue;
        FPGA_WRITE_REG32(pIrqCtlData, pIrqCtlData->intMaskReg, intMaskVal);
        /* perform dummy read to make sure that write was completed */
        intMaskVal = FPGA_READ_REG32(pIrqCtlData, pIrqCtlData->intMaskReg);
    }

    return (0);
}

static void tvl2IrqCtlEnableIntr(tvl2IrqCtlData_t* pIrqCtlData)
{
    unsigned int intStatusRegVal;

    if (pIrqCtlData == NULL)
    {
        TVL2DRV_ERR("tvl2IrqCtlEnableIntr, Error pIrqCtlData = NULL\n");
        return;
    }

    if (pIrqCtlData->pFPGABaseAddr)
    {
        FPGA_WRITE_REG32(pIrqCtlData, pIrqCtlData->intMaskReg, pIrqCtlData->intMask);

        intStatusRegVal = FPGA_READ_REG32(pIrqCtlData, pIrqCtlData->intStatusReg);
        FPGA_WRITE_REG32(pIrqCtlData, pIrqCtlData->intStatusReg, intStatusRegVal);
        /* perform dummy read to make sure that write was completed */
        intStatusRegVal = FPGA_READ_REG32(pIrqCtlData, pIrqCtlData->intStatusReg);
    }

    if (pIrqCtlData->p9056BaseAddr)
    {
        PLX9056_ENABLE_LINT(pIrqCtlData);
        PLX9056_ENABLE_PCI_INT(pIrqCtlData);
    }
}

static void tvl2IrqCtlDisableIntr(tvl2IrqCtlData_t* pIrqCtlData)
{
    if (pIrqCtlData == NULL)
    {
        TVL2DRV_ERR("tvl2IrqCtlDisableIntr, Error pIrqCtlData = NULL\n");
        return;
    }
    PLX9056_DISABLE_LINT(pIrqCtlData);
    PLX9056_DISABLE_PCI_INT(pIrqCtlData);
}

static unsigned int tvl2IrqCtlIsIntrActive(tvl2IrqCtlData_t* pIrqCtlData)
{
    unsigned int intCsrVal;
    unsigned int plxIntrSource = PLX9056_INTR_SOURCE_NONE;
    unsigned int dma0Mode, dma1Mode;
    unsigned int outpostIntrMask, outpostIntrStatus;
//    unsigned int tempRegVal;

    /* test PLX9056 local interrupt is active */
    intCsrVal 	= PLX9056_READ_REG32(pIrqCtlData, PLX9056_REG_INTCSR);

//    TVL2DRV_DBG("%s: INTCSR = 0x%x\n", __FUNCTION__, intCsrVal);
//    tempRegVal = FPGA_READ_REG32(pIrqCtlData, pIrqCtlData->intStatusReg);
//    TVL2DRV_DBG("%s: FPGA ISR = 0x%x\n", __FUNCTION__, tempRegVal);
//    tempRegVal = FPGA_READ_REG32(pIrqCtlData, pIrqCtlData->intMaskReg);
//    TVL2DRV_DBG("%s: FPGA IMR = 0x%x\n", __FUNCTION__, tempRegVal);

    /****************************************************
    * If the chip is in a low power state, then local
    * register reads are disabled and will always return
    * 0xFFFFFFFF.  If the PLX chip's interrupt is shared
    * with another PCI device, the PXL ISR may continue
    * to be called.  This case is handled to avoid
    * erroneous reporting of an active interrupt.
    ***************************************************/
    if (intCsrVal == 0xFFFFFFFF)
        return ((unsigned int) PLX9056_INTR_SOURCE_NONE);

    // Check for master PCI interrupt enable
    if ((intCsrVal & (1 << 8)) == 0)
        return ((unsigned int) PLX9056_INTR_SOURCE_NONE);

    // Verify that an interrupt is truly active

    // Check if PCI Doorbell Interrupt is active and not masked
    if ((intCsrVal & (1 << 13)) && (intCsrVal & (1 << 9)))
    {
        plxIntrSource |= (1 << PLX9056_INTR_SOURCE_DOORBELL);
        TVL2DRV_IRQ_DBG("tvl2IrqCtlIsIntrActive DOORBELL INT active\n");
    }

    // Check if PCI Abort Interrupt is active and not masked
    if ((intCsrVal & (1 << 14)) && (intCsrVal & (1 << 10)))
    {
        plxIntrSource |= (1 << PLX9056_INTR_SOURCE_PCI_ABORT);
        TVL2DRV_IRQ_DBG("tvl2IrqCtlIsIntrActive ABORT INT active\n");
    }

    // Check if Local->PCI Interrupt is active and not masked
    if ((intCsrVal & (1 << 15)) && (intCsrVal & (1 << 11)))
    {
        plxIntrSource |= (1 << PLX9056_INTR_SOURCE_LOCAL1);
        TVL2DRV_IRQ_DBG("tvl2IrqCtlIsIntrActive LOCAL INT active\n");
    }

    // Check if DMA Channel 0 Interrupt is active and not masked
    if ((intCsrVal & (1 << 21)) && (intCsrVal & (1 << 18)))
    {
        // Verify the DMA interrupt is routed to PCI
        dma0Mode = PLX9056_READ_REG32(pIrqCtlData, PLX9056_REG_DMA0_MODE);

        if (dma0Mode & (1 << 17))
        {	
            plxIntrSource |= (1 << PLX9056_INTR_SOURCE_DMA0);
            TVL2DRV_IRQ_DBG("tvl2IrqCtlIsIntrActive DMA0 INT active\n");
        }
    }

    // Check if DMA Channel 1 Interrupt is active and not masked
    if ((intCsrVal & (1 << 22)) && (intCsrVal & (1 << 19)))
    {
        // Verify the DMA interrupt is routed to PCI
        dma1Mode = PLX9056_READ_REG32(pIrqCtlData, PLX9056_REG_DMA1_MODE);

        if (dma1Mode & (1 << 17))
        {
            plxIntrSource |= (1 << PLX9056_INTR_SOURCE_DMA1);
            TVL2DRV_IRQ_DBG("tvl2IrqCtlIsIntrActive DMA1 INT active\n");
        }
    }

    // Check if MU Outbound Post interrupt is active
    outpostIntrStatus = \
    PLX9056_READ_REG32(pIrqCtlData, PLX9056_REG_OUTPOST_INT_STATUS);

    if (outpostIntrStatus & (1 << 3))
    {
        // Check if MU Outbound Post interrupt is not masked
        outpostIntrMask =
            PLX9056_READ_REG32(pIrqCtlData, PLX9056_REG_OUTPOST_INT_MASK);

        if ((outpostIntrMask & (1 << 3)) == 0)
        {
            plxIntrSource |= (1 << PLX9056_INTR_SOURCE_OUTBOUND_POST);
            TVL2DRV_IRQ_DBG("tvl2IrqCtlIsIntrActive OUTBOUND POST INT active\n");
        }
    }

    return (plxIntrSource); 
}


static int tvl2PciDrvReadProc(char *buf, char **start, off_t pos, int count, int *eof, void *data)
{
  char    *p = buf;

  p += sprintf(p, "\t jiffies: %lu\n",              jiffies);
  p += sprintf(p, "\t AMC interrupt count : %lu\n", g_amcIsrCount);
  p += sprintf(p, "\t RTM interrupt count : %lu\n", g_rtmIsrCount);
  p += sprintf(p, "\t Total interrupt count : %lu\n", g_totalIsrCount);
  p += sprintf(p, "\t DataValid: %d\n",             hwInfo.isDataValid);
  p += sprintf(p, "\t RTM present : %d\n",          hwInfo.isRtmPresent);
  p += sprintf(p, "\t Framer Device Type : %d\n",   hwInfo.framerType);
  p += sprintf(p, "\t Number of Framer Devices : %d\n", hwInfo.numberOfFramers);
  p += sprintf(p, "\t Trunks per Framer : %d\n",    hwInfo.trunksPerFramer);
  p += sprintf(p, "\t Default Line Type : %d\n",    hwInfo.defaultLineType);
  p += sprintf(p, "\t Number of licensed trunks : %d\n",    hwInfo.numberOfLicensedTrunks);

  return (p-buf);
}

static int tvl2PciDrvWriteProc(struct file* file, const char* buffer, 
	unsigned long count, void* data)
{
  unsigned char cmd[32], param[32], val[32];
  unsigned char *pos;
  int len;
  int index;

  memset (param, 0, sizeof(param));
  memset (val, 0, sizeof(val));

  if (count > sizeof(cmd)-1)
  {
     TVL2DRV_ERR("tvl2PciDrvWriteProc : string length %ld exceeds maximum %d\n",
                 count, sizeof(cmd)-1);
     return -EINVAL;
  }
  if (copy_from_user(cmd, buffer, count))
  {
    return -EFAULT;
  }
                                                                                                 
  cmd[count] = '\0';
  if (cmd[count-1] == '\n')
        cmd [count - 1] = 0;
  len = strlen(cmd);

  TVL2DRV_DBG("%s: processing string '%s'\n", __FUNCTION__, cmd);
  pos = strchr (cmd, '=');
  if (pos == NULL)
  {
    TVL2DRV_ERR("tvl2PciDrvWriteProc - Invalid input format, did not find '='\n");
    return -EINVAL;
  }
  index = cmd - pos; 
  if (index < 0) 
    index = -1 * index;

  strncpy (param, cmd, index);
  strcpy (val, &cmd[index+1]);
  TVL2DRV_DBG("tvl2PciDrvWriteProc processing %s = %s\n", param, val);

  if (!strcmp (RTMDETECTED, param))
  {
     hwInfo.isRtmPresent = my_atoi (val);
  }
  else if (!strcmp (DATAVALID, param))
  {
     hwInfo.isDataValid = my_atoi (val);
  }
  else if (!strcmp (FRAMERTYPE, param))
  {
     hwInfo.framerType = my_atoi (val);
  }
  else if (!strcmp (NUMFRAMERS, param))
  {
     hwInfo.numberOfFramers = my_atoi(val);
  }
  else if (!strcmp (TRUNKSPERFRAMER, param))
  {
     hwInfo.trunksPerFramer = my_atoi(val);
  }
  else if (!strcmp (NUMBER_LICENSED_TRUNKS, param))
  {
     hwInfo.numberOfLicensedTrunks = my_atoi(val);
  }
  else if (!strcmp (DEFAULTLINETYPE, param))
  {
     hwInfo.defaultLineType = my_atoi(val);
  }
  else
  {
    TVL2DRV_ERR("tvl2PciDrvWriteProc - parameter %s is unknown\n", param);
    return -EINVAL;
  }
  return (count);
}

static int my_atoi (const char* str)
{
  int sign, number = 0, placeValue = 1;
  int len = strlen (str);
  int i=0;
  int j=0;

  if ((str[0] == '+') || (str[0] == '-'))
  {
    sign = (str[0] == '-') ? -1 : 1;
    j++;
  }

  for (i=len-1; i >=  j; i--)
  {
    if ((str[i] < '0') || (str[i] > '9')) 
      break;
    number += ((str[i]  - '0') * placeValue);
    placeValue *= 10;
  }
  return number;
}


#if (HARDWARE == 0)
static void tvl2PciDrvSimTimerCallback (unsigned long data)
{
     tvl2PciDev_t* tvl2PciDevp = (tvl2PciDev_t*) data;
     TVL2DRV_DBG("tvl2PciDrvTimerCallback entered\n");
     tvl2PciDrvIsr (tvl2PciDevp->pPCIDevice->irq, (void*)data, 0);
     mod_timer(&tvl2PciDevp->irqCtl.simTimer,jiffies +HZ*5);
     return;
}
/*  
 * Start/Stop timer which simulates 
 * device interrupts.
 */
static void tvl2PciDrvStartStopSimTimer (tvl2PciDev_t* tvl2PciDevp, int  startStopCmd)
{
  spin_lock (&tvl2PciDevp->irqCtl.simTimerLock);
  if (startStopCmd  == tvl2PciDevp->irqCtl.timerActive)
  {
      TVL2DRV_DBG("tvl2PciDrvStartStopTimer - no change in timer status\n");
  }
  else
  {  
    tvl2PciDevp->irqCtl.timerActive = startStopCmd;
    if (startStopCmd) 
    {
      TVL2DRV_DBG("tvl2PciDrvStartStopTimer - Starting timer\n");
      tvl2PciDrvStartSimTimer (tvl2PciDevp);
    }
    else
    {
      TVL2DRV_DBG("tvl2PciDrvStartStopTimer - Stopping timer\n");
      tvl2PciDrvStopSimTimer (tvl2PciDevp);
    }
  }
  spin_unlock(&tvl2PciDevp->irqCtl.simTimerLock);
       
  return;
}

static void tvl2PciDrvStartSimTimer (tvl2PciDev_t* tvl2PciDevp)
{
  tvl2PciDevp->irqCtl.simTimer.function = tvl2PciDrvSimTimerCallback;
  tvl2PciDevp->irqCtl.simTimer.expires = jiffies + HZ*5;
  tvl2PciDevp->irqCtl.simTimer.data = (long)tvl2PciDevp;
  add_timer (&tvl2PciDevp->irqCtl.simTimer);
}
static void tvl2PciDrvStopSimTimer (tvl2PciDev_t* tvl2PciDevp)
{
  del_timer_sync(&tvl2PciDevp->irqCtl.simTimer);
}
static void tvl2PciDrvInitSimulation (tvl2PciDev_t* tvl2PciDevp) 
{
  tvl2PciDevp->irqCtl.simTimerLock = SPIN_LOCK_UNLOCKED;
  init_timer(&tvl2PciDevp->irqCtl.simTimer);
}
#endif



#ifdef ENABLE_ZIONSVILLE_CHANGES
/******************************************************************************
 * Function Name: <tvl2HsiOpen( struct inode *inodep, struct file *filep)>
 *
 * Description  : This function is called by the OS when a user-space app opens
 *                the tvl2hsi device.  It is responsible for initializing the
 *                tvl2PciDrv structure (if it isn't already initialized) so we
 *                can use the HSI interface.
 *
 * Input        :
 *    <inodep>  - Not used.
 *    <filep>   - Not used.
 *
 * Output       : None.
 *
 * Return Value : Returns -EBUSY if the device is already in use, or if the
 *                initialization of the global struct fails.
 *****************************************************************************/
static ssize_t
tvl2HsiOpen( struct inode *inodep, struct file *filep )
{
	int              rc;
	tvl2PciDev_t*    tvl2PciDevp; 
	struct pci_dev*  pPCIDevice;

	TVL2DRV_DBG( "%s: open count increment/check\n", __FUNCTION__ );

	/* Make sure user space apps open this device only once */
	if( !atomic_dec_and_test( &g_HsiCounter ))
	{
		TVL2DRV_ERR( "%s: multiple opens not supported \n", __FUNCTION__ );
		return( -EBUSY );
	}

	/* For now, ZVL only supports systems with one board, so there should only be one in the system */
	tvl2PciDevp = &tvl2PciDrv.tvl2PciDev[0];
	pPCIDevice  = tvl2PciDevp->pPCIDevice;

	TVL2DRV_DBG( "%s opening HSI device -- IRQ %d\n", __FUNCTION__, pPCIDevice->irq );


	/* Only perform initialization if it hasn't already been done. */
	if( tvl2PciDevp->irqCtl.pIrqCtlData == NULL )
	{
#if (CONFIG_PROBE_AND_CLAIM == 0) 
		tvl2PciDrvClaimDevice( pPCIDevice );
#endif

		tvl2PciDevp->irqCtl.pIrqCtlData                = &g_IrqCtlData[AMC_IRQ_CTL_DATA];
		tvl2PciDevp->irqCtl.pIrqCtlData->tvl2PciDevp   = tvl2PciDevp;
		tvl2PciDevp->irqCtl.pIrqCtlData->pciDeviceType = TVL2_PCI_DEVICE_ID_DIALOGIC_SEAVILLE;

		rc = tvl2IrqCtlInit( tvl2PciDevp->irqCtl.pIrqCtlData, pPCIDevice );

		if( rc != 0 )
		{
			TVL2DRV_ERR( "%s: Failed during tvl2IrqCtlInit, rc = %d\n", __FUNCTION__, rc );
			return( rc );
		}

		filep->private_data = tvl2PciDevp;
		tvl2IrqCtlEnableIntr( tvl2PciDevp->irqCtl.pIrqCtlData );
	}

	return( 0 );
}




/******************************************************************************
 * Function Name: <tvl2HsiRelease( struct inode *inodep, struct file *filep )>
 *
 * Description  : This function is called when a user-space app releases the
 *						tvl2hsi device.  The function uninitializes the interrupt.
 *
 * Input        :
 *    <inodep>  - Not Used
 *    <filep>   - Not Used
 *
 * Output       : None
 *
 * Return Value : The function always returns 0.
 *****************************************************************************/
static int
tvl2HsiRelease( struct inode *inodep, struct file *filep )
{
	tvl2PciDev_t*    tvl2PciDevp; 
	struct pci_dev*  pPCIDevice;

	TVL2DRV_DBG( "%s: close\n", __FUNCTION__ );

	tvl2PciDevp = &tvl2PciDrv.tvl2PciDev[0];
	pPCIDevice  = tvl2PciDevp->pPCIDevice;

	if( pPCIDevice->irq )
	{
		/* Release resources allocated for interrupt handling */
		tvl2IrqCtlDeInit( tvl2PciDevp->irqCtl.pIrqCtlData );
	}

#if (CONFIG_PROBE_AND_CLAIM == 0) 
	tvl2PciDrvReleaseDevice( pPCIDevice );
#endif

	TVL2DRV_DBG( "%s: open count decrement\n", __FUNCTION__ );
	atomic_set( &g_HsiCounter, 1 );
	return( 0 );
}




/******************************************************************************
 * Function Name: <tvl2HsiIoctl( struct inode *inodep, struct file *filep,
 *                               unsigned int cmd, unsigned long arg )>
 *
 * Description  : This function just acts as a dispatcher to route the ioctl
 *                request to the appropriate handler function.
 *
 * Input        :
 *    <inodep>  - Not Used.
 *    <filep>   - Not Used.
 *    <cmd>     - Specifies the control code sent via the ioctl() system call.
 *    <arg>     - Pointer to the user-supplied data for the request.
 *
 * Output       : None.
 *
 * Return Value : The function returns -ENOTTY on any error.
 *****************************************************************************/
static int
tvl2HsiIoctl( struct inode *inodep, struct file *filep, unsigned int cmd, unsigned long arg )
{
	int cmd_type;
	int cmd_code;

	TVL2DRV_DBG( "%s entered\n", __FUNCTION__ );

	/* Check commands before decoding them */
	cmd_type = _IOC_TYPE( cmd );
	cmd_code = _IOC_NR( cmd );

	if( cmd_type != HSI_FILE_DEVICE )
	{
		TVL2DRV_ERR( "%s: invalid command %d\n", __FUNCTION__, cmd );
		return( -ENOTTY );
	}
	else
	{
		if(( cmd < HSI_BRIDGE_INIT_IOCTL ) || ( cmd > HSI_BRIDGE_SYNC_CLOCK_IOCTL ))
		{
			TVL2DRV_ERR( "%s: invalid command %d\n", __FUNCTION__, cmd );
			return( -ENOTTY );
		}
	}


	switch( cmd )
	{
	case HSI_BRIDGE_INIT_IOCTL:
		{
			int                   status;
			HSI_BRIDGE_INIT_INPUT payload;

			TVL2DRV_DBG( "%s: BRIDGE_INIT\nn", __FUNCTION__ );

			status = copy_from_user( &payload, (HSI_BRIDGE_INIT_INPUT __user *)arg,
											 sizeof( HSI_BRIDGE_INIT_INPUT ));

			if( status == 0 )
			{
				status = tvl2HsiIoctlBridgeInit( &payload );
			}
			else
			{
				TVL2DRV_ERR( "%s: copy failure\n", __FUNCTION__ );
				status = -ENOTTY;
			}
			return( status );
		}

	case HSI_BRIDGE_STOP_IOCTL:
		{
			int           status;
			unsigned long logical_id;

			TVL2DRV_DBG( "%s: BRIDGE_STOP\nn", __FUNCTION__ );

			status = copy_from_user( &logical_id, (unsigned long __user *)arg, sizeof( unsigned long ));

			if( status == 0 )
			{
				status = tvl2HsiIoctlBridgeStop( logical_id );
			}
			else
			{
				TVL2DRV_ERR( "%s: copy failure\n", __FUNCTION__ );
				status = -ENOTTY;
			}
			return( status );
		}

	case HSI_BRIDGE_ROUTE_IOCTL:
		{
			/* This board does not support HSI streams */
			TVL2DRV_ERR( "%s: command not supported %d\n", __FUNCTION__, cmd );
			return( -EINVAL );
		}

	case HSI_INTR_RATE_ENABLE_IOCTL:
		{
			int           status;
			unsigned long logical_id;

			TVL2DRV_DBG( "%s: INTR_RATE_ENABLE\nn", __FUNCTION__ );

			status = copy_from_user( &logical_id, (unsigned long __user *)arg, sizeof( unsigned long ));

			if( status == 0 )
			{
				status = tvl2HsiIoctlIntrEnable( logical_id );
			}
			else
			{
				TVL2DRV_ERR( "%s: copy failure\n", __FUNCTION__ );
				status = -ENOTTY;
			}
			return( status );
		}

	case HSI_INTR_RATE_DISABLE_IOCTL:
		{
			int           status;
			unsigned long logical_id;

			TVL2DRV_DBG( "%s: INTR_RATE_DISABLE\nn", __FUNCTION__ );

			status = copy_from_user( &logical_id, (unsigned long __user *)arg, sizeof( unsigned long ));

			if( status == 0 )
			{
				status = tvl2HsiIoctlIntrDisable( logical_id );
			}
			else
			{
				TVL2DRV_ERR( "%s: copy failure\n", __FUNCTION__ );
				status = -ENOTTY;
			}
			return( status );
		}

	case HSI_BRIDGE_SYNC_CLOCK_IOCTL:
		{
			int                         status;
			HSI_BRIDGE_CLOCK_SYNC_INPUT payload;

			TVL2DRV_DBG( "%s: BRIDGE_SYNC\nn", __FUNCTION__ );

			status = copy_from_user( &payload, (HSI_BRIDGE_CLOCK_SYNC_INPUT __user *)arg,
			sizeof( HSI_BRIDGE_CLOCK_SYNC_INPUT ));

			if( status == 0 )
			{
				status = tvl2HsiIoctlBridgeSync( &payload );
			}
			else
			{
				TVL2DRV_ERR( "%s: copy failure\n", __FUNCTION__ );
				status = -ENOTTY;
			}
			return( status );
		}

	default:
		TVL2DRV_ERR( "%s: invalid command %d\n", __FUNCTION__, cmd );
		return( -EINVAL );
	}
}




/******************************************************************************
 * Function Name: <tvl2HsiIoctlBridgeInit( HSI_BRIDGE_INIT_INPUT *init_data )>
 *
 * Input        :
 *    <init_data> - Contains data for the BRIDGE_INIT request.
 *
 * Output       : The function creates a BRIDGE_INIT_CMPLT response and sends
 *                it to the response queue (handled by the read() system call).
 *
 * Description  : This function is called to initialize the Bridge Device
 *                functionality of this driver.  There is no special processing
 *                other than to store some relevant configuration parameters
 *                and to note whether or not the bridge is to be configured in
 *                slave mode.
 *
 * Return Value : This function always returns 0.
 *****************************************************************************/
static int
tvl2HsiIoctlBridgeInit( HSI_BRIDGE_INIT_INPUT *init_data )
{
	tvl2IrqCtlData_t *pIrqCtlData = &g_IrqCtlData[AMC_IRQ_CTL_DATA];
	unsigned int      mask_value  = 0;
	unsigned long     flags;

	spin_lock_irqsave( &g_HsiLock, flags );

	/* If the start type is HSI_START_ON_INTR_ENABLE, then just store the relevant */
	/* parameters and complete the request                                         */
	g_HsiInfo.bridgeId      = init_data->bridgeId;
	g_HsiInfo.tickRate      = init_data->tickRate;
	g_HsiInfo.startType     = init_data->startType;
	g_HsiInfo.isrFunc       = init_data->hmpIsrFunc;
	g_HsiInfo.numHsiBuffers = init_data->numHsiBuffers;

	TVL2DRV_DBG( "%s:  HSI_BRIDGE_INIT (bid=%d, rate=%d, start=%d, callback=%p, numbufs=%d).\n",
					 __FUNCTION__, g_HsiInfo.bridgeId, g_HsiInfo.tickRate,
					 g_HsiInfo.startType, g_HsiInfo.isrFunc, g_HsiInfo.numHsiBuffers );

	if( g_HsiInfo.startType == HSI_START_SLAVE_SYNC )
	{
		/* Unmask the Slave Interrupt Bit */
		mask_value  = FPGA_READ_REG32( pIrqCtlData, AMC_REG_INT_MASK );
		mask_value &= ~0x2000; /* Clear Bit 13 */
		FPGA_WRITE_REG32( pIrqCtlData, AMC_REG_INT_MASK, mask_value );

		g_HsiInfo.SlaveMode = 1;
	}
	else
	{
		g_HsiInfo.SlaveMode = 0;
	}

	/* Prepare a buffer to complete the BRIDGE_INIT request asynchronously */
	if( g_HsiCallbackData[g_HsiCallbackWriteIndex].type == HSI_BRIDGE_MAX_COMPLETION )
	{
		g_HsiCallbackData[g_HsiCallbackWriteIndex].type = HSI_BRIDGE_INIT_COMPLETION;

		g_HsiCallbackData[g_HsiCallbackWriteIndex].u.bridge_init.type        = HSI_BRIDGE_INIT_COMPLETION;
		g_HsiCallbackData[g_HsiCallbackWriteIndex].u.bridge_init.bridgeId    = g_HsiInfo.bridgeId;
		g_HsiCallbackData[g_HsiCallbackWriteIndex].u.bridge_init.txDelay     = 0;
		g_HsiCallbackData[g_HsiCallbackWriteIndex].u.bridge_init.rxDelay     = 0;
		g_HsiCallbackData[g_HsiCallbackWriteIndex].u.bridge_init.txSyncDelay = 0;
		g_HsiCallbackData[g_HsiCallbackWriteIndex].u.bridge_init.rxSyncDelay = 0;

		g_HsiCallbackWriteIndex = (g_HsiCallbackWriteIndex + 1) % MAX_HSI_RESPONSES;
		spin_unlock_irqrestore( &g_HsiLock, flags );
		up( &g_HsiSem );
	}
	else
	{
		spin_unlock_irqrestore( &g_HsiLock, flags );
		TVL2DRV_DBG( "%s:  Could not store completion data.\n", __FUNCTION__ );
	}

	return 0;
}




/******************************************************************************
 * Function Name: <tvl2HsiIoctlBridgeStop( unsigned long logical_id )>
 *
 * Input        :
 *    <logical_id> - Specifies the driver-specific ID of the bridge to stop.
 *
 * Output       : The function creates a BRIDGE_STOP_CMPLT response and sends
 *                it to the response queue (handled by the read() system call).
 *
 * Description  : This function is used to stop the Bridge Device.  It will
 *                clear out the global HSI structure and if the bridge was
 *                configured to operate in slave mode, the slave interrupt will
 *                be masked out.
 *
 * Return Value : This function always returns 0.
 *****************************************************************************/
static int
tvl2HsiIoctlBridgeStop( unsigned long logical_id )
{
	tvl2IrqCtlData_t *pIrqCtlData = &g_IrqCtlData[AMC_IRQ_CTL_DATA];
	unsigned int      mask_value  = 0;
	unsigned int      bridge_id;
	unsigned long     flags;

	TVL2DRV_DBG( "%s:  HSI_BRIDGE_STOP (lid=%lu).\n", __FUNCTION__, logical_id );

	spin_lock_irqsave( &g_HsiLock, flags );

	bridge_id = g_HsiInfo.bridgeId;

	g_HsiInfo.bridgeId  = 0;
	g_HsiInfo.tickRate  = 0;
	g_HsiInfo.startType = 0;
	g_HsiInfo.isrFunc   = NULL;

	if( g_HsiInfo.SlaveMode == 1 )
	{
		/* Mask the Slave Interrupt again */
		mask_value  = FPGA_READ_REG32( pIrqCtlData, AMC_REG_INT_MASK );
		mask_value |= 0x2000; /* Set Bit 13 */
		FPGA_WRITE_REG32( pIrqCtlData, AMC_REG_INT_MASK, mask_value );

		g_HsiInfo.SlaveMode = 0;
	}

	spin_unlock_irqrestore( &g_HsiLock, flags );

	/* Prepare a buffer to complete the HSI_BRIDGE_STOP request asynchronously */
	if( g_HsiCallbackData[g_HsiCallbackWriteIndex].type == HSI_BRIDGE_MAX_COMPLETION )
	{
		g_HsiCallbackData[g_HsiCallbackWriteIndex].type = HSI_BRIDGE_STOP_COMPLETION;

		g_HsiCallbackData[g_HsiCallbackWriteIndex].u.intr_enable.type     = HSI_BRIDGE_STOP_COMPLETION;
		g_HsiCallbackData[g_HsiCallbackWriteIndex].u.intr_enable.bridgeId = bridge_id;

		g_HsiCallbackWriteIndex = (g_HsiCallbackWriteIndex + 1) % MAX_HSI_RESPONSES;

		up( &g_HsiSem );
	}
	else
	{
		TVL2DRV_DBG( "%s:  Could not store completion data.\n", __FUNCTION__ );
	}

	return 0;
}




/******************************************************************************
 * Function Name: <tvl2HsiIoctlIntrEnable( unsigned long logical_id )>
 *
 * Description  : This function will enable the Timer interrupt on a Bridge
 *                Device.  This function is called by the KMBC whenever a Bridge
 *                is designated as the HMP clock master.
 *
 * Input        :
 *    <logical_id> - Specifies the driver-specific ID of the bridge to enable
 *                   interrupts on.
 *
 * Output       : The function creates a BRIDGE_INTR_ENABLE_CMPLT response and
 *                sends it to the response queue (handled by the read() system
 *                call).
 *
 * Return Value : This function always returns 0.
 *****************************************************************************/
static int
tvl2HsiIoctlIntrEnable( unsigned long logical_id )
{
	tvl2IrqCtlData_t *pIrqCtlData = &g_IrqCtlData[AMC_IRQ_CTL_DATA];
	unsigned int      mask_value  = 0;
	unsigned long     flags;
	unsigned int      index_value = 0;

	TVL2DRV_DBG( "%s:  HSI_BRIDGE_INTR_ENABLE (lid=%lu).\n", __FUNCTION__, logical_id );

	spin_lock_irqsave( &g_HsiLock, flags );

	if( g_HsiInfo.SlaveMode == 1 )
	{
		/* Mask the Slave Interrupt again */
		mask_value  = FPGA_READ_REG32( pIrqCtlData, AMC_REG_INT_MASK );
		mask_value |= 0x2000; /* Set Bit 13 */
		FPGA_WRITE_REG32( pIrqCtlData, AMC_REG_INT_MASK, mask_value );

		g_HsiInfo.SlaveMode = 0;
	}

	/* Mask out the Timer Interrupt */
	mask_value  = FPGA_READ_REG32( pIrqCtlData, AMC_REG_INT_MASK );
	mask_value &= ~0x1; /* Clear bit 0 */
	FPGA_WRITE_REG32( pIrqCtlData, AMC_REG_INT_MASK, mask_value );

	/* Disable the Timer Interrupt */
	FPGA_WRITE_REG32( pIrqCtlData, AMC_REG_TIMER_CONTROL, 0x2 );

	/* Clear the Timer Interrupt */
	if( FPGA_READ_REG32( pIrqCtlData, AMC_REG_INT_STATUS ) & 0x0001 )
	{
		FPGA_WRITE_REG32( pIrqCtlData, AMC_REG_INT_STATUS, 0x1 );
	}

	/* Load the Preset Data Register */
	FPGA_WRITE_REG32( pIrqCtlData, AMC_REG_TIMER_PRESET, 0x8 * g_HsiInfo.tickRate );

	/* Load Timer with contents of Preset Data Register */
	FPGA_WRITE_REG32( pIrqCtlData, AMC_REG_TIMER_LOAD, 0x0 );

	/* Set the Index Buffer Size Register */
	FPGA_WRITE_REG32( pIrqCtlData, AMC_REG_INDEX_BUFFER_SIZE, g_HsiInfo.numHsiBuffers );
	index_value = FPGA_READ_REG32( pIrqCtlData, AMC_REG_INDEX_BUFFER_SIZE );
//	printk( KERN_INFO "tvl2PciDrv: BMN - Index Buffer Value Register = %d\n", index_value );

	/* Unmask the Timer Interrupt */
	mask_value &= 0xFFFE;
	FPGA_WRITE_REG32( pIrqCtlData, AMC_REG_INT_MASK, mask_value );

	/* Enable the Timer Interrupt */
	FPGA_WRITE_REG32( pIrqCtlData, AMC_REG_TIMER_CONTROL, 0x3 );

	/* Read the Interrupt Status Register to force write completion */
	FPGA_READ_REG32( pIrqCtlData, AMC_REG_INT_STATUS );

	spin_unlock_irqrestore( &g_HsiLock, flags );


	/* Prepare a buffer to complete the HSI_BRIDGE_INTR_ENABLE request asynchronously */
	if( g_HsiCallbackData[g_HsiCallbackWriteIndex].type == HSI_BRIDGE_MAX_COMPLETION )
	{
		g_HsiCallbackData[g_HsiCallbackWriteIndex].type = HSI_BRIDGE_INTR_ENABLE_COMPLETION;

		g_HsiCallbackData[g_HsiCallbackWriteIndex].u.intr_enable.type     = HSI_BRIDGE_INTR_ENABLE_COMPLETION;
		g_HsiCallbackData[g_HsiCallbackWriteIndex].u.intr_enable.bridgeId = g_HsiInfo.bridgeId;

		g_HsiCallbackWriteIndex = (g_HsiCallbackWriteIndex + 1) % MAX_HSI_RESPONSES;
		up( &g_HsiSem );
	}
	else
	{
		TVL2DRV_DBG( "%s:  Could not store completion data.\n", __FUNCTION__ );
	}

  return 0;
}




/******************************************************************************
 * Function Name: <tvl2HsiIoctlIntrDisable( unsigned long logical_id )>
 *
 * Description  : This function will disable the Timer interrupt on a Bridge
 *                Device.
 *
 * Input        :
 *    <logical_id> - Specifies the driver-specific ID of the bridge to disable
 *                   interrupts on.
 *
 * Output       : The function creates a BRIDGE_INTR_DISABLE_CMPLT response and
 *                sends it to the response queue (handled by the read() system
 *                call).
 *
 * Return Value : This function always returns 0.
 *****************************************************************************/
static int
tvl2HsiIoctlIntrDisable( unsigned long logical_id )
{
	tvl2IrqCtlData_t *pIrqCtlData = &g_IrqCtlData[AMC_IRQ_CTL_DATA];
	unsigned int      mask_value  = 0;
	unsigned long     flags;

	TVL2DRV_DBG( "%s:  HSI_BRIDGE_INTR_DISABLE (lid=%lu).\n", __FUNCTION__, logical_id );

	spin_lock_irqsave( &g_HsiLock, flags );

	/* Mask out Timer Interrupt */
	mask_value  = FPGA_READ_REG32( pIrqCtlData, AMC_REG_INT_MASK );
	mask_value |= 0x1;
	FPGA_WRITE_REG32( pIrqCtlData, AMC_REG_INT_MASK, mask_value );

	/* Disable Timer */
	FPGA_WRITE_REG32( pIrqCtlData, AMC_REG_TIMER_CONTROL, 0x2 );

	/* Clear Timer Interrupt */
	if( FPGA_READ_REG32( pIrqCtlData, AMC_REG_INT_STATUS ) & 0x0001 )
	{
		FPGA_WRITE_REG32( pIrqCtlData, AMC_REG_INT_STATUS, 0x1 );
	}

	/* Unmask the Slave Interrupt Bit */
	mask_value  = FPGA_READ_REG32( pIrqCtlData, AMC_REG_INT_MASK );
	mask_value &= ~0x2000; /* Clear Bit 13 */
	FPGA_WRITE_REG32( pIrqCtlData, AMC_REG_INT_MASK, mask_value );
	g_HsiInfo.SlaveMode = 1;

	spin_unlock_irqrestore( &g_HsiLock, flags );


	/* Prepare a buffer to complete the HSI_BRIDGE_INTR_DISABLE request asynchronously */
	if( g_HsiCallbackData[g_HsiCallbackWriteIndex].type == HSI_BRIDGE_MAX_COMPLETION )
	{
		g_HsiCallbackData[g_HsiCallbackWriteIndex].type = HSI_BRIDGE_INTR_DISABLE_COMPLETION;

		g_HsiCallbackData[g_HsiCallbackWriteIndex].u.intr_enable.type     = HSI_BRIDGE_INTR_DISABLE_COMPLETION;
		g_HsiCallbackData[g_HsiCallbackWriteIndex].u.intr_enable.bridgeId = g_HsiInfo.bridgeId;

		g_HsiCallbackWriteIndex = (g_HsiCallbackWriteIndex + 1) % MAX_HSI_RESPONSES;

		up( &g_HsiSem );
	}
	else
	{
		spin_unlock_irqrestore( &g_HsiLock, flags );
		TVL2DRV_DBG( "%s:  Could not store completion data.\n", __FUNCTION__ );
	}

  return 0;
}




/******************************************************************************
 * Function Name: <tvl2HsiIoctlBridgeSync( HSI_BRIDGE_CLOCK_SYNC_INPUT *sync_data )>
 *
 * Description  : This function is used to modify the clock rate of the board
 *                in order to keep it's 2ms timer in sync with another HMP
 *                clock master in the system.
 *
 * Input        :
 *    <sync_data> - Specifies data used to adjust the period of the Timer
 *                  Interrupt.
 *
 * Output       : The function creates a BRIDGE_CLOCK_SYNC_CMPLT response and
 *                sends it to the response queue (handled by the read() system
 *                call).
 *
 * Return Value : This function always returns 0.
 *****************************************************************************/
static int
tvl2HsiIoctlBridgeSync( HSI_BRIDGE_CLOCK_SYNC_INPUT *sync_data )
{
	tvl2IrqCtlData_t *pIrqCtlData     = &g_IrqCtlData[AMC_IRQ_CTL_DATA];
	int               CorrectionIndex = 0;
	unsigned int      CorrectionVal   = 0;


	/* The sync parameter comes as a 32-bit integer.  The correction factor    */
	/* can be determined by subtracting 1,000,000 from the value.  With that   */
	/* the function will consult a lookup table to determine the proper value  */
	/* to write to the Slave Error Correction Register.                        */
	if( sync_data->scaleFactor >= 1000000 )
	{
		/* Slave clock is running too fast, we need to slow it down.  Calculate */
		/* the ppm correction index (supporting 5 ppm resolution).              */
		CorrectionIndex = (sync_data->scaleFactor - 1000000) / 5;

		/* Correction is limited to +/- 150 ppm, which at 5 ppm resolution      */
		/* results in a maximum index of 30.                                    */
		if( CorrectionIndex > 30 )  CorrectionIndex = 30;

		CorrectionVal = MMA_SLOWDOWN_LUT[CorrectionIndex];

		TVL2DRV_DBG( KERN_ALERT "%s:  Clock Correction (sf=%d, ppm=%d, cv=%d).\n",
			          __FUNCTION__, sync_data->scaleFactor, CorrectionIndex * 5,
						 CorrectionVal );
	}
	else
	{
		/* Slave clock is running too slow, we need to speed it up.  Calculate  */
		/* the ppm correction index (supporting 5 ppm resolution).              */
		CorrectionIndex = (1000000 - sync_data->scaleFactor) / 5;

		/* Correction is limited to +/- 150 ppm, which at 5 ppm resolution      */
		/* results in a maximum index of 30.                                    */
		if( CorrectionIndex > 30 )  CorrectionIndex = 30;

		CorrectionVal = MMA_SPEEDUP_LUT[CorrectionIndex];

		TVL2DRV_DBG(KERN_ALERT "%s:  Clock Correction (sf=%d, ppm=-%d, cv=%d).\n",
			          __FUNCTION__, sync_data->scaleFactor, CorrectionIndex * 5,
						 CorrectionVal );
	}

	/* Write the new correction parameters to the MMA card. */
	FPGA_WRITE_REG32( pIrqCtlData, AMC_REG_SLAVE_ERR_CORR, CorrectionVal );


	/* Prepare a buffer to complete the HSI_BRIDGE_SYNC_CLOCK request asynchronously */
	if( g_HsiCallbackData[g_HsiCallbackWriteIndex].type == HSI_BRIDGE_MAX_COMPLETION )
	{
		g_HsiCallbackData[g_HsiCallbackWriteIndex].type = HSI_BRIDGE_SYNC_CLOCK_COMPLETION;

		g_HsiCallbackData[g_HsiCallbackWriteIndex].u.sync_clock.type     = HSI_BRIDGE_SYNC_CLOCK_COMPLETION;
		g_HsiCallbackData[g_HsiCallbackWriteIndex].u.sync_clock.bridgeId = g_HsiInfo.bridgeId;

		g_HsiCallbackWriteIndex = (g_HsiCallbackWriteIndex + 1) % MAX_HSI_RESPONSES;

		up( &g_HsiSem );
	}
	else
	{
		TVL2DRV_DBG( "%s:  Could not store completion data.\n", __FUNCTION__ );
	}

	return 0;
}




/******************************************************************************
 * Function Name: <tvl2HsiRead( struct file *filep, char *buf, size_t count,
 *                              loff_t *ppos )>
 *
 * Description  : This is the handled for the read() system call, which is used
 *                by the device driver to communicate asynchronous responses to
 *                the application that has opened the tvl2hsi device.
 *
 * Input        :
 *    <filep> - Not Used.
 *    <buf>   - Pointer to the user buffer to copy read data to.
 *    <count> - Length of the 'buf' data.
 *    <ppos>  - Not Used.
 *
 * Output       : The function will block if no data is available.  When data
 *                is available, the function copies the response message to the
 *                user buffer and completes the system call.
 *
 * Return Value : The function returns the number of bytes read 
 *****************************************************************************/
static ssize_t
tvl2HsiRead( struct file *filep, char *buf, size_t count, loff_t *ppos )
{
	/* This function is new, and should only be called by the KMBC in order to get command responses */
	int ret_val;
	int bytes_to_copy = 0;
	int rc            = 0;

   TVL2DRV_DBG( "%s:  User App is attempting to perform a read().\n", __FUNCTION__ );

	/* Check to see if there are any messages to complete right away */
	if( g_HsiCallbackData[g_HsiCallbackReadIndex].type == HSI_BRIDGE_MAX_COMPLETION )
	{
	   TVL2DRV_DBG( "%s:  No response pending - blocking the read call.\n", __FUNCTION__ );

		/* Block the read until a command is completed, when this function will wake up and */
		/* complete the read.                                                               */
		ret_val = down_interruptible( &g_HsiSem );

		if( ret_val ) 
		{
			TVL2DRV_DBG("%s: signal received (rc = %d)\n", __FUNCTION__, ret_val );
			return( ret_val );
		}

	}

	if( g_HsiCallbackData[g_HsiCallbackReadIndex].type != HSI_BRIDGE_MAX_COMPLETION )
	{
		switch( g_HsiCallbackData[g_HsiCallbackReadIndex].type )
		{
		case( HSI_BRIDGE_INIT_COMPLETION ):
			bytes_to_copy = sizeof( HSI_BRIDGE_INIT_OUTPUT );
			if( bytes_to_copy > count ) bytes_to_copy = count;
			rc = copy_to_user( buf, &g_HsiCallbackData[g_HsiCallbackReadIndex].u.bridge_init, bytes_to_copy );
			break;


		case( HSI_BRIDGE_STOP_COMPLETION ):
			bytes_to_copy = sizeof( HSI_BRIDGE_STOP_OUTPUT );
			if( bytes_to_copy > count ) bytes_to_copy = count;
			rc = copy_to_user( buf, &g_HsiCallbackData[g_HsiCallbackReadIndex].u.bridge_stop, bytes_to_copy );
			break;


		case( HSI_BRIDGE_INTR_ENABLE_COMPLETION ):
			bytes_to_copy = sizeof( HSI_BRIDGE_INTR_ENABLE_OUTPUT );
			if( bytes_to_copy > count ) bytes_to_copy = count;
			rc = copy_to_user( buf, &g_HsiCallbackData[g_HsiCallbackReadIndex].u.intr_enable, bytes_to_copy );
			break;


		case( HSI_BRIDGE_INTR_DISABLE_COMPLETION ):
			bytes_to_copy = sizeof( HSI_BRIDGE_INTR_DISABLE_OUTPUT );
			if( bytes_to_copy > count ) bytes_to_copy = count;
			rc = copy_to_user( buf, &g_HsiCallbackData[g_HsiCallbackReadIndex].u.intr_disable, bytes_to_copy );
			break;


		case( HSI_BRIDGE_SYNC_CLOCK_COMPLETION ):
			bytes_to_copy = sizeof( HSI_BRIDGE_SYNC_CLOCK_OUTPUT );
			if( bytes_to_copy > count ) bytes_to_copy = count;
			rc = copy_to_user( buf, &g_HsiCallbackData[g_HsiCallbackReadIndex].u.sync_clock, bytes_to_copy );
			break;

		default:
			break;
		}

		TVL2DRV_DBG( "%s:  Completing Response (read_idx=%d, type=%d, bytes=%d, rc=%d).\n",
						 __FUNCTION__, g_HsiCallbackReadIndex, 
						 g_HsiCallbackData[g_HsiCallbackReadIndex].type,
						 bytes_to_copy, rc );
		
		g_HsiCallbackData[g_HsiCallbackReadIndex].type = HSI_BRIDGE_MAX_COMPLETION;
		g_HsiCallbackReadIndex = (g_HsiCallbackReadIndex + 1) % MAX_HSI_RESPONSES;

		if( rc != 0 )
		{
			return( -EFAULT );
		}
	}

	TVL2DRV_DBG( "%s:  Completing Read (bytes=%d).\n", __FUNCTION__, bytes_to_copy );
	return( bytes_to_copy );
}
#endif	/* #ifdef ENABLE_ZIONSVILLE_CHANGES */




/****************************************************************
*                    MODULE FUNCTIONS 
****************************************************************/
static int __init tvl2PciDrvInit (void)
{
#ifdef ENABLE_ZIONSVILLE_CHANGES
  unsigned int index;
#endif

  TVL2DRV_DBG("tvl2PciDrvInit enter.\n");
  memset (&tvl2PciDrv, sizeof tvl2PciDrv, 0);

  tvl2PciDrv.major = register_chrdev (0, DEVNAME, &tvl2PciDrv_fops);

  if (tvl2PciDrv.major)
  {
    TVL2DRV_DBG("tvl2PciDrvInit registered driver : major is %d\n", tvl2PciDrv.major);
  }  

  tvl2PciDrv.pProcRootEntry = proc_mkdir (DEVNAME,  NULL);
  if (tvl2PciDrv.pProcRootEntry)
  {
#ifdef MODULE
      tvl2PciDrv.pProcRootEntry->owner = THIS_MODULE;
#endif
      tvl2PciDrv.pProcRootEntry->read_proc = 0;
      tvl2PciDrv.pProcRootEntry->write_proc = 0;
  }
  else
  {
      TVL2DRV_ERR("tvl2PciDrv: can't install proc dir '%s'\n", DEVNAME);
      return -ENOMEM;
  } 

  /* this entry will allow OAM to configure the framer parameters */ 
  tvl2PciDrv.pProcPstnConfigEntry = 
	create_proc_entry (PSTNCONFIG, S_IFREG | S_IRUGO, 
		tvl2PciDrv.pProcRootEntry); 

  if (tvl2PciDrv.pProcPstnConfigEntry)
  {
#ifdef MODULE
      tvl2PciDrv.pProcPstnConfigEntry->owner = THIS_MODULE;
#endif
      tvl2PciDrv.pProcPstnConfigEntry->read_proc  = tvl2PciDrvReadProc;
      tvl2PciDrv.pProcPstnConfigEntry->write_proc = tvl2PciDrvWriteProc;
  }
  else
  {
      TVL2DRV_ERR("tvl2PciDrv: can't install proc dir '%s'\n", PSTNCONFIG);
      return -ENOMEM;
  } 

#ifdef ENABLE_ZIONSVILLE_CHANGES
	g_HsiMajorNum = register_chrdev (0, "tvl2hsi", &tvl2Hsi_fops);

	if( g_HsiMajorNum )
	{
		TVL2DRV_DBG( "tvl2PciDrvInit registered driver : major is %d\n", g_HsiMajorNum );
	}  

	/* Initialize the data used by the HSI interface to respond to requests asynchronously */
	for( index = 0; index < MAX_HSI_RESPONSES; index++ )
	{
		g_HsiCallbackData[index].type = HSI_BRIDGE_MAX_COMPLETION;
	}

  atomic_set( &g_HsiSyncData, 0 );
  g_HsiLock = SPIN_LOCK_UNLOCKED;
  init_MUTEX_LOCKED( &g_HsiSem );
#endif

  return pci_register_driver (&tvl2PciDrv_pci);
}
/* Just before the module is removed, execute the stop routine. */
static void __exit tvl2PciDrvStop(void)
{
  TVL2DRV_DBG("tvl2PciDrvStop enter.\n");
  pci_unregister_driver (&tvl2PciDrv_pci);
  if (tvl2PciDrv.pProcPstnConfigEntry)
	remove_proc_entry (PSTNCONFIG, tvl2PciDrv.pProcRootEntry);
  if (tvl2PciDrv.pProcRootEntry)
	remove_proc_entry (DEVNAME, NULL);
   unregister_chrdev(tvl2PciDrv.major, DEVNAME);
#ifdef ENABLE_ZIONSVILLE_CHANGES
	unregister_chrdev( g_HsiMajorNum, "tvl2hsi" );
#endif
  return;
}
module_init(tvl2PciDrvInit);
module_exit(tvl2PciDrvStop);


MODULE_AUTHOR("Dialogic");
#ifdef	MODULE_LICENSE
MODULE_LICENSE("BSD");
#endif
