/****************************************************************************
*              Copyright (c) 1997 Dialogic Corp.
*              All Rights Reserved.
*
*              THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF Dialogic Corp.
*              The copyright notice above does not evidence any actual or
*              intended publication of such source code.
****************************************************************************/

/****************************************************************************
* FILE:    ccs_defs.h
* Author: B. Thorne
* $Revision: 1.16 $
*
* DESCRIPTION:  Common Channnel Signalling component Definitions file.
*               Defines the messaging interface to the CCS.
*
****************************************************************************/

/*
 * The CCS PRI Component uses messages from the Standard Kernel Message Set,
 * the Standard Component Message Set, and the Mercury Common Channel
 * Signaling PRI Message Set.
 */

#ifndef   _ccs_defs_h      /* Prevent multiple inclusion */
#define   _ccs_defs_h

#include "qtypes.h"
#include "qcomplib.h"
#include "ccs_ext_defs.h"      /* Machine built header file extensions to ccs_defs */

/*
 * CCS COMPONNT Std_MsgxxxParm parameter equates
*/
#define  CCS_INSTANCE_MAP  5  /* READ/WRITE parameter.
                                 Instance creation bitmap. CCS instances
                                 correlate directly to the network
                                 interfaces. For example, 0x06 would specify
                                 the creation of CCS instances 2 & 3 that
                                 would communicate with network interfaces
                                 2 & 3, respectively. */

#define CCS_DEBUG_LEVEL    6  /* READ/WRITE parameter. Default = 0x01 (DBG_ERROR). */
                              /* CCS Debug Level                                   */

/*
 * CCS INSTANCE Std_MsgxxxParm parameter equates
*/
#define  CCS_DSL_ID        5  /* READ ONLY parameter.
                                 Digital Subscriber Loop ID. Implicitly
                                 defined by instance# (instance# - 1).
                              */
#define  INTERFACE_MAP     6  /* READ/WRITE parameter. Default = 0.
                                 Network Interface Map Identifier.
                                 A value of 0 implies that this CCS instance
                                 will only carry signalling info for the NI
                                 that is associated with it.
                                 If one or more bits is set, then one bit has to
                                 specify the NI that is associated with the CCS
                                 instance.
                                 If more than one bit is set, then an NFAS
                                 configuration is implied.
                              */
#define  CCS_SWITCH_TYPE   7  /* READ/WRITE parameter.
                                 Network switch type.
                                 For builds with only one switch type, the
                                 default value will be that switchtype. See
                                 CCS_SW_TYPE for possible values.
                              */
#define  DCHAN_INVERSION   8  /* READ/WRITE parameter. Default = 0.
                                 D channel inversion flag.
                                 0 = no inversion.
                                 1 = inversion.
                              */
#define  L2_ACCESS         9  /* READ/WRITE parameter. Default = 0.
                                 Layer 2 access flag.
                                 0 = No L2 access.
                                 1 = Communicate directly with L2.
                              */
#define  L2_TRACE         10  /* READ/WRITE parameter. Default = 0.
                                 Layer 2 trace flag.
                                 0 = Trace off.
                                 1 = Trace on.
                              */

#define  CCS_TMR_303      11  /* READ/WRITE prameters. Layer3 Timers.     */
#define  CCS_TMR_304      12  /* See Q.931 & corresponding switch         */
#define  CCS_TMR_305      13  /* specifications for default values &      */
#define  CCS_TMR_308      14  /* purpose. Not all timers are applicable to*/
#define  CCS_TMR_310      15  /* all switches.                            */
#define  CCS_TMR_313      16  /*                                          */
#define  CCS_TMR_318      17  /* Values are in milliseconds.              */
#define  CCS_TMR_319      18  /*   0 == Default value.                    */
                              /*  -1 == Disable timer. Note: This         */
                              /*        currently has the same effect as  */
                              /*        setting the default value.        */

#define  L2_SYM_CR_BIT    19  /* READ/WRITE parameter. Default = 0.
                                 Layer 2 Synchronous DL flag for DTE-DTE
                                 datalink.
                                 0 = DCE--DTE datalink.
                                 1 = DTE--DTE datalink.
                              */

#define  CCS_TMR_302               20  /* READ/WRITE prameters. Layer3 T302 Timer. */
#define  CCS_TMR_TEI_RETRY         21  /* READ/WRITE prameters. TEI Link Establishment Retry. */
#define  CCS_TMR_TEI_S4_STABILITY  22  /* READ/WRITE prameters. TEI State 4
                                          Stability. */
#define  CCS_PROTOCOL_MODE         23  /* READ/WRITE parameter to setup
                                          USER_MODE or NETWORK MODE */    /*GKG */
#define  CCS_ENBL_FEAT_TESTS       24  /* Enable Feature Test */
                                       /*GKG */

#define  CCS_NFAS_MODE             25  /* NFAS Mode - 0 == disabled (default)
                                                      1 == Primary Dchannel
                                                      2 == Standby Dchannel
                                                      3 == Nfas Enabled */  
                                                                          
#define  CCS_INTERFACE_ID          26  /* Interface ID - Range is from 0 -31 */
                                        
#define  CCS_NFAS_GRP_ID           27  /* PRI NFAS Server Group ID */


#define  CCS_MONITOR_MODE          28  /* parameter for HI-Z feature */

#define  PERST_L2_ACT              29  /* READ/WRITE parameter. Default = 0.
                                          Persistent Layer 2 activation flag.
                                             0 = off.
                                             1 = on.
                                       */
                                       
#define  CCS_FEATURE_BITMASK       30  /* READ/WRITE parameter. Default = 0.
                                          ISDN feature Mask.
                                       */                 

#define  CCS_RESTINSERV_FLAG       31  /* READ/WRITE parameter. Default = 0.
                                          ISDN RESTART/IN_SERVICE parameter for PTR 27563
                                       */                 
#define  CCS_TMR_306              32
#define  CCS_TMR_309              33
#define  CCS_TMR_312              34
#define  CCS_TMR_314              35
#define  CCS_TMR_322              36

/* CCS Std_MsgSetParm parameter values for Feature Bit Mask types */
#define  CCSMSK_CALL_PROC_SEND   0x80
#define  CCSMSK_SETUP_ACK        0x100

/* CCS Std_MsgSetParm parameter values for Switch types */
typedef  enum
{
   CCS_UNSPECIFIED_STYPE,     /* Unspecified/unknown switch type */
   CCS_4ESS_STYPE,            /* AT&T 4ESS PRI */
   CCS_5ESS_STYPE,            /* AT&T 5ESS PRI */
   CCS_DMS100_STYPE,          /* NTI DMS100 PRI */
   CCS_NTT_STYPE,             /* JAPANESE NTT PRI (WAS KDD_NTT) */
   CCS_1TR6_STYPE,            /* GERMAN 1TR6 PRI */
   CCS_DASS2_STYPE,           /* DASS2 PRI */
   CCS_NET5_STYPE,            /* NET 5 PRI */
   CCS_TS014_STYPE,           /* AUSTRALIAN TS014 PRI (WAS TPH_1856) */
   CCS_VN_STYPE,              /* FRENCH VN2/3 PRI (WAS VN2) */
   CCS_QSIGE1_STYPE,          /* QSIG E1 PRI */
   CCS_QSIGT1_STYPE,          /* QSIG T1 PRI */
	CCS_NI2_STYPE,					/* Added for PRI NI2 switch type */
   CCS_MAX_NUM_SW_TYPE        /* Number of switch types supported */
} CCS_SW_TYPE;


/* CCS Std_MsgDetectEvt Types
   see file ccs_ext_defs.h */
#define CCS_Event_Type_LinkState  CCS_Event_Label_LinkState   /* Event type, for requesting
                                                                 RTC events, for CCS HDLC
                                                                 Link State */
/*
 * Mercury Kernel Standard Message Set defined in mercdefs.doc
 */
/*
 * Standard Component Message Set defined in stddefs.doc.
 */

/*
 * COMPONENT CCS (value = 0x0015) uses:
 *   MESSAGE Std_MsgComtest
 *   MESSAGE Std_MsgComtestCmplt
 *   MESSAGE Std_MsgExit
 *   MESSAGE Std_MsgExitCmplt
 *   MESSAGE Std_MsgGetParm
 *   MESSAGE Std_MsgGetParmCmplt
 *   MESSAGE Std_MsgInit
 *   MESSAGE Std_MsgInitCmplt
 *   MESSAGE Std_MsgSetParm
 *   MESSAGE Std_MsgSetParmCmplt
 *   MESSAGE Std_MsgDetectEvt
 *   MESSAGE Std_MsgDetectEvtCmplt
 *   MESSAGE Std_MsgEvtDetected
 */

/* CCS PRI Message Definitions.
 * The following message definition structure is the same for all Mercury
 * messages between CCS and the CHP instances. The structure is for the
 * message body of the Mercury message. Messages are generated by and
 * delivered to the host user software which relate to the GlobalCall
 * interface. However, the Mercury messages detailed in this section
 * are generated from the CHP call control to CCS, and vice versa.
 * Generally the CHP generates Requests(Req.) to CCS. These traverse the
 * network to the  peer call control and arrive as an Indicator(Ind.).
 * The Peer call control generates a Response(Resp) back to the
 * network. This arrives back at the originating user call  control as an
 * Confirmation(Conf).  So the host call control generates Requests and
 * eventually expects a Confirmation. It also will answer an Indicator with
 * some kind  of Response.
 */

#define     CCS_COMPONENT            0x15        /* CCS component number */
#define     CCS_COMP                 0x15        /* Alias */
#define     HIF_COMPONENT            0x51        /* dump HIF component number */
#define     HIF_COMP                 0x51        /* Alias */
#define     CCS_COMP_ID_STR          "ccscomp"   /* ccs component name */
#define     CCS_HIF_ID_STR           "ccshif"    /* hif component name */

#define  CCS_Std_ComponentType   0x15
typedef  QCompAttr   CCS_Std_ComponentType_t;
/*
 * Unique identifier for the Channel Protocol Component.
 */

#define  CCSMSGBASE     0x1500      /* Base number for CCS unique messages */


/******************************************************************************
* This structure defines the information beginning at the "data[]" field
* of the QMsg message for CHP<--->CCS ISDN messages destined for the
* Call Control application. So for example, if the CHP needs to
* send an ALERTING_IND to the CCS, the QMsg.type field will be CCS_ieMessage.
* Then the MessageID of the CCS_IE_HEADER will be ALERTING_IND.
******************************************************************************/
#define  CCS_ieMessage      (CCSMSGBASE+0x00)

/*  This is a rediffinition of tCCS_IE_HEADER in
 *  FILE:iefuncts.h. Since qhost has a BYTE typedefed to
 *  a signed char I cannot include this file in some of the
 *  protocol stack APIs. If you change this structre you MUST
 *  UPDATE tCCS_IE_HEADER IN FILE IEFUNCTS.H !!!!
 */

typedef     struct   {
   UInt16   CallId;                    /* Layer 3 Call Connection Identifier */
   UInt16   CallRef;                    /* Network Call Reference - LOW 11301999 */
   UInt16   IEBufLen;                  /* Length of Private elements */
   UInt8    MessageID;                 /* Message ID, i.e. ALERTING_IND. */
} CCS_IE_HEADER;

/* Field addresses have changed, need to recalculate LOW - 11301999 */
/* 
#define  CCS_ieHeader_Size      ((2*QT_UINT16_SIZE) + QT_UINT8_SIZE)
#define  CCS_ieMessage_Size      ((2*QT_UINT16_SIZE) + QT_UINT8_SIZE)

#define  CCS_IE_HEADER_CallId     ((QT_UINT16 << QFIELDDEF_DATATYPE)+0)
#define  CCS_IE_HEADER_IEBufLen   ((QT_UINT16 << QFIELDDEF_DATATYPE) + \
                                   (QT_UINT16_SIZE << QFIELDDEF_OFFSET))
#define  CCS_IE_HEADER_MessageID  ((QT_UINT8 << QFIELDDEF_DATATYPE) + \
                                   ((2*QT_UINT16_SIZE) << QFIELDDEF_OFFSET))

#define  CCS_IE_DATA_data         ((QT_UINT8 << QFIELDDEF_DATATYPE) + \
                                   ((CCS_ieHeader_Size) << QFIELDDEF_OFFSET))

#define  CCS_ieHeader_get(msgPtr, structAddr, offset)   \
            qMsgVarFieldGet((msgPtr), 3, &(offset),   \
            (CCS_IE_HEADER_CallId), &((structAddr)->CallId),   \
            (CCS_IE_HEADER_IEBufLen), &((structAddr)->IEBufLen),   \
            (CCS_IE_HEADER_MessageID), &((structAddr)->MessageID))


#define  CCS_ieHeader_put(msgPtr, structAddr, offset)   \
            qMsgVarFieldPut((msgPtr), 3, &(offset),   \
            (CCS_IE_HEADER_CallId), &((structAddr)->CallId),   \
            (CCS_IE_HEADER_IEBufLen), &((structAddr)->IEBufLen),   \
            (CCS_IE_HEADER_MessageID), &((structAddr)->MessageID))
*/
#define  CCS_ieHeader_Size      ((3*QT_UINT16_SIZE) + QT_UINT8_SIZE)
#define  CCS_ieMessage_Size      ((3*QT_UINT16_SIZE) + QT_UINT8_SIZE)

#define  CCS_IE_HEADER_CallId     ((QT_UINT16 << QFIELDDEF_DATATYPE)+0)
#define  CCS_IE_HEADER_CallRef   ((QT_UINT16 << QFIELDDEF_DATATYPE) + \
                                   (QT_UINT16_SIZE << QFIELDDEF_OFFSET))
#define  CCS_IE_HEADER_IEBufLen  ((QT_UINT16 << QFIELDDEF_DATATYPE) + \
                                   ((2*QT_UINT16_SIZE) << QFIELDDEF_OFFSET))
#define  CCS_IE_HEADER_MessageID  ((QT_UINT8 << QFIELDDEF_DATATYPE) + \
                                   ((3*QT_UINT16_SIZE) << QFIELDDEF_OFFSET))

#define  CCS_IE_DATA_data         ((QT_UINT8 << QFIELDDEF_DATATYPE) + \
                                   ((CCS_ieHeader_Size) << QFIELDDEF_OFFSET))

#define  CCS_ieHeader_get(msgPtr, structAddr, offset)   \
            qMsgVarFieldGet((msgPtr), 4, &(offset),   \
            (CCS_IE_HEADER_CallId), &((structAddr)->CallId),   \
            (CCS_IE_HEADER_CallRef), &((structAddr)->CallRef),   \
            (CCS_IE_HEADER_IEBufLen), &((structAddr)->IEBufLen),   \
            (CCS_IE_HEADER_MessageID), &((structAddr)->MessageID))

#define  CCS_ieHeader_put(msgPtr, structAddr, offset)   \
            qMsgVarFieldPut((msgPtr), 4, &(offset),   \
            (CCS_IE_HEADER_CallId), &((structAddr)->CallId),   \
            (CCS_IE_HEADER_CallRef), &((structAddr)->CallRef),   \
            (CCS_IE_HEADER_IEBufLen), &((structAddr)->IEBufLen),   \
            (CCS_IE_HEADER_MessageID), &((structAddr)->MessageID))
                  /* LOW - 11301999 */

#define  CCS_ieData_get(msgPtr, byteArray, length, offset)   \
            qMsgVarFieldGet((msgPtr), 1, &(offset),   \
            (QMsgField)(((length-1) << QFIELDDEF_COUNT) + CCS_IE_DATA_data), \
            (byteArray))

/*
 *#define  CCS_ieData_get(msgPtr, byteArray, length, offset)   \
 *           qMsgVarFieldGet((msgPtr), 1, &(offset),   \
 *           (QMsgField)(((length) << QFIELDDEF_COUNT) + CCS_IE_DATA_data), \
 *           (byteArray))
 *
 */

#if  (CPU == SPARC)

#define  CCS_ieData_put(msgPtr, byteArray, length, offset)   \
            qMsgVarFieldPut((msgPtr), 1, &(offset),   \
            (QMsgField) (((length) > ((msgPtr)->msgsize - CCS_ieHeader_Size)) ? \
            ((((msgPtr)->msgsize - CCS_ieHeader_Size) << QFIELDDEF_COUNT) + \
            CCS_IE_DATA_data) : \
            (((length-1) << QFIELDDEF_COUNT) + CCS_IE_DATA_data)), \
            (byteArray))
#else


#define  CCS_ieData_put(msgPtr, byteArray, length, offset)   \
            qMsgVarFieldPut((msgPtr), 1, &(offset),   \
            (QMsgField) (((length) > ((msgPtr)->size - CCS_ieHeader_Size)) ? \
            ((((msgPtr)->size - CCS_ieHeader_Size) << QFIELDDEF_COUNT) + \
            CCS_IE_DATA_data) : \
            (((length-1) << QFIELDDEF_COUNT) + CCS_IE_DATA_data)), \
            (byteArray))

/*
 *#define  CCS_ieData_put(msgPtr, byteArray, length, offset)   \
 *           qMsgVarFieldPut((msgPtr), 1, &(offset),   \
 *           (QMsgField) (((length) > ((msgPtr)->size - CCS_ieHeader_Size)) ? \
 *           ((((msgPtr)->size - CCS_ieHeader_Size) << QFIELDDEF_COUNT) + \
 *           CCS_IE_DATA_data) : \
 *           (((length) << QFIELDDEF_COUNT) + CCS_IE_DATA_data)), \
 *           (byteArray))
 */

#endif

/*
 * The following structure describes the standard header for
 * package INFO buffers. The INFO buffer is pointed to by the PKG buffer.
 * The overall length of this buffer is defined in the PKG.InfoLen field.
 * The msgID field is synonymous to the PKG.MsgType field.
*/
typedef  struct   {
   UInt8     msgID;                     /* Message ID */
   UInt8     dslID;                     /* DSL ID */
   UInt8     reserv1;                   /* CES for North American ISDN */
   UInt8     reserv2;                   /* Reserved byte 2 */
   UInt8     reserv3;                   /* Reserved byte 3 */
   UInt8     callID1;                   /* MSB of Call ID */
   UInt8     callID2;                   /* LSB of Call ID */
   UInt8     lenIEs;                    /* Length of privates IE's */
} INFO_HEAD;

#define     msgID_OFFSET      0
#define     dslID_OFFSET      1
#define     reserv1_OFFSET    2
#define     reserv2_OFFSET    3
#define     reserv3_OFFSET    4
#define     callID1_OFFSET    5
#define     callID2_OFFSET    6
#define     lenIEs_OFFSET     7
#define     IEs_OFFSET        8

/*
 * The following structure is actually defined in ccie.h and is here for
 * information purposes only. This structure defines the header data
 * beginning, and repeating, within the IE data.
*/
#if 0
typedef struct
    {
    INT16 ident;                                 /* Element identifier */
    UInt8 len;          /* Element size (in bytes) excluding the header */
    } IE_HEAD;
#endif

/* Note, offset is not provided in the following definition */
#define  IE_HEAD_ident           (QT_INT16 << QFIELDDEF_DATATYPE)

/******************************************************************************
* CCS_CHPRegister is an input message for component CCS.
******************************************************************************/
#define  CCS_CHPRegister      (CCSMSGBASE+0x0001)    /* Msg # for CHPRegister */

#define  CCS_CHPRegister_DSLid   (QT_UINT8 << QFIELDDEF_DATATYPE)
typedef  UInt8   CCS_CHPRegister_DSLid_t;

#define  CCS_CHPRegister_BChannelId   ((QT_UINT8 << QFIELDDEF_DATATYPE) + \
                                     ((QT_UINT8_SIZE) << QFIELDDEF_OFFSET))
typedef  UInt8   CCS_CHPRegister_BChannelId_t;    /* Line Id */

#define  CCS_CHPRegister_InterfaceId  ((QT_UINT8 << QFIELDDEF_DATATYPE) + \
                                     ((2*QT_UINT8_SIZE) << QFIELDDEF_OFFSET))
typedef  UInt8   CCS_CHPRegister_InterfaceId_t;

typedef struct {
   UInt8      DSLid;
   UInt8      BChannelId;
   UInt8      InterfaceId;
} CCS_CHPRegister_t;

#define  CCS_CHPRegister_get(msgPtr, structAddr, offset)   \
          qMsgVarFieldGet((msgPtr), 3, &(offset),   \
          (CCS_CHPRegister_DSLid), &((structAddr)->DSLid),   \
          (CCS_CHPRegister_BChannelId), &((structAddr)->BChannelId),   \
          (CCS_CHPRegister_InterfaceId), &((structAddr)->InterfaceId))


#define  CCS_CHPRegister_put(msgPtr, structAddr, offset)   \
          qMsgVarFieldPut((msgPtr), 3, &(offset),   \
          (CCS_CHPRegister_DSLid), &((structAddr)->DSLid),   \
          (CCS_CHPRegister_BChannelId), &((structAddr)->BChannelId),   \
          (CCS_CHPRegister_InterfaceId), &((structAddr)->InterfaceId))


#define  CCS_CHPRegister_Size      3

/******************************************************************************
* CCS_CHPRegisterCmplt is an output message for component CCS. It is sent in
* response to the message CCS_CHPRegister to indicate that the CHP has been
* registered with the CCS.
******************************************************************************/
#define  CCS_CHPRegisterCmplt      (CCSMSGBASE+0x0002)    /* Msg # for
                                                         CHPRegisterCmplt */

#define  CCS_CHPRegisterCmplt_Size      0


/******************************************************************************
* CCS_CHPRegisterErr is an output message for component CCS. It is sent in
* response to the message CCS_CHPRegister to indicate that an error has
* occured in registering the CHP with the CCS.
******************************************************************************/
#define  CCS_CHPRegisterErr       (CCSMSGBASE+0x0003)    /* Msg # for
                                                         CHPRegisterErr */

#define  CCS_CHPRegisterErr_Size      0

/******************************************************************************
* CCS_CHPHandoff is an input message for component CCS.
******************************************************************************/

#define  CCS_CHPHandoff      (CCSMSGBASE+0x0004)   /* Msg # for CHPHandoff */

/*
 * Sent by CHP instances to the CCS component to handoff a call from one B
 * channel(CHP) to a new B Channel(CHP) during B channel negotiation. The
 * CCS maps CHP instances task address to a specific B channel(BChannelId)
 * within a specific T1 line or NFAS(DSLid).
 * Direction               - IN
 * Information Elements    - None
 */
/* Already diffined ahead 
#define  CCS_IE_HEADER_MessageID  ((QT_UINT8 << QFIELDDEF_DATATYPE) + \
                                   ((2*QT_UINT16_SIZE) << QFIELDDEF_OFFSET))
                                   */
#define  CCS_CHPHandoff_Old_CHPtskAddr   (QT_COMPDESC << QFIELDDEF_DATATYPE)
typedef  QCompDesc   CCS_CHPHandoff_Old_CHPtskAddr_t;    /* Old CHP address */

#define  CCS_CHPHandoff_New_CHPtskAddr   ((QT_COMPDESC << QFIELDDEF_DATATYPE) + \
                                   ((QT_COMPDESC_SIZE) << QFIELDDEF_OFFSET))
typedef  QCompDesc   CCS_CHPHandoff_New_CHPtskAddr_t;    /* New CHP address */

#define  CCS_CHPHandoff_BChannelId   ((QT_UINT8 << QFIELDDEF_DATATYPE) + \
                                   ((2*QT_COMPDESC_SIZE) << QFIELDDEF_OFFSET))
typedef  UInt8   CCS_CHPHandoff_BChannelId_t;

#define  CCS_CHPHandoff_InterfaceId   ((QT_UINT8 << QFIELDDEF_DATATYPE) + \
                  (((2*QT_COMPDESC_SIZE)+QT_UINT8_SIZE) << QFIELDDEF_OFFSET))
typedef  UInt8   CCS_CHPHandoff_InterfaceId_t;          /* Line Id */

typedef struct {
   QCompDesc   Old_CHPtskAddr;
   QCompDesc   New_CHPtskAddr;
   UInt8       BChannelId;
   UInt8       InterfaceId;
} CCS_CHPHandoff_t;

#define  CCS_CHPHandoff_get(msgPtr, structAddr, offset)   \
          qMsgVarFieldGet((msgPtr), 4, &(offset),   \
          (CCS_CHPHandoff_Old_CHPtskAddr), &((structAddr)->Old_CHPtskAddr),   \
          (CCS_CHPHandoff_New_CHPtskAddr), &((structAddr)->New_CHPtskAddr),   \
          (CCS_CHPHandoff_BChannelId), &((structAddr)->BChannelId),   \
          (CCS_CHPHandoff_InterfaceId), &((structAddr)->InterfaceId))


#define  CCS_CHPHandoff_put(msgPtr, structAddr, offset)   \
          qMsgVarFieldPut((msgPtr), 4, &(offset),   \
          (CCS_CHPHandoff_Old_CHPtskAddr), &((structAddr)->Old_CHPtskAddr),   \
          (CCS_CHPHandoff_New_CHPtskAddr), &((structAddr)->New_CHPtskAddr),   \
          (CCS_CHPHandoff_BChannelId), &((structAddr)->BChannelId),   \
          (CCS_CHPHandoff_InterfaceId), &((structAddr)->InterfaceId))


#define  CCS_CHPHandoff_Size      14

/******************************************************************************
* CCS_CHPHandoffCmplt is an output message for component CCS. It is sent in
* response to the message CCS_CHPHandoff to indicate that the CHP has
* accomplished a call handoff with the CCS.
******************************************************************************/

#define  CCS_CHPHandoffCmplt      (CCSMSGBASE+0x0005)   /* Msg # for
                                                        CHPHandoffCmplt */

#define  CCS_CHPHandoffCmplt_Size      0

/******************************************************************************
* CCS_CHPHandoffErr is an output message for component CCS. It is sent in
* response to the message CCS_CHPHandoff to indicate that an error has
* occured in handing-off the CHP call with the CCS.
******************************************************************************/
#define  CCS_CHPHandoffErr       (CCSMSGBASE+0x0006)    /* Msg # for
                                                         CHPHandoffErr */

#define  CCS_CHPHandoffErr_Size      0

/******************************************************************************
* CCS_ErrorInd is an output message for component CCS.
******************************************************************************/

#define  CCS_ErrorInd       (CCSMSGBASE+0x0007)   /* Msg # for ErrorInd */

/*
 * Sent by CCS instances to indicate an error has occurred in the HDLC
 * controller, TN L3/L2 software, or the CCS instance.  The CHP uses this
 * information to inform the Line Administration instance of Line status
 * changes.
 * Direction               - OUT
 * Information Elements    - None
 */

#define  CCS_ErrorInd_Callid   0x280000
typedef  UInt16   CCS_ErrorInd_Callid_t;

#define  CCS_ErrorInd_DSLid   0x200002
typedef  UInt8   CCS_ErrorInd_DSLid_t;

#define  CCS_ErrorInd_BChannelId   0x200003
typedef  UInt8   CCS_ErrorInd_BChannelId_t;

#define  CCS_ErrorInd_ErrorCode   0x200004
typedef  UInt8   CCS_ErrorInd_ErrorCode_t;

#define  CCS_ErrorInd_ErrorDescription   0x50f805
typedef  char *  CCS_ErrorInd_ErrorDescription_t;

typedef struct {
   UInt16     Callid;
   UInt8      DSLid;
   UInt8      BChannelId;
   UInt8      ErrorCode;
   char       ErrorDescription[32];
} CCS_ErrorInd_t;

#define  CCS_ErrorInd_get(msgPtr, structAddr, offset)   \
          qMsgVarFieldGet((msgPtr), 5, &(offset),   \
          (CCS_ErrorInd_Callid), &((structAddr)->Callid),   \
          (CCS_ErrorInd_DSLid), &((structAddr)->DSLid),   \
          (CCS_ErrorInd_BChannelId), &((structAddr)->BChannelId),   \
          (CCS_ErrorInd_ErrorCode), &((structAddr)->ErrorCode),   \
          (CCS_ErrorInd_ErrorDescription), &((structAddr)->ErrorDescription))


#define  CCS_ErrorInd_put(msgPtr, structAddr, offset)   \
          qMsgVarFieldPut((msgPtr), 5, &(offset),   \
          (CCS_ErrorInd_Callid), &((structAddr)->Callid),   \
          (CCS_ErrorInd_DSLid), &((structAddr)->DSLid),   \
          (CCS_ErrorInd_BChannelId), &((structAddr)->BChannelId),   \
          (CCS_ErrorInd_ErrorCode), &((structAddr)->ErrorCode),   \
          (CCS_ErrorInd_ErrorDescription), &((structAddr)->ErrorDescription))


#define  CCS_ErrorInd_Size      37

/******************************************************************************
* CCS_StatusInd is an output message for component CCS.
******************************************************************************/

#define  CCS_StatusInd       (CCSMSGBASE+0x0008)   /* Msg # for StatusInd */

/*
 * This message is use for reporting firmware status codes.  These status
 * codes are generated by the firmware and not by the ISDN network. The CHP
 * uses this information to inform the Line Administration instance of
 * Line status changes.
 * Direction        - OUT
 * Status codes
 */

#define  CCS_StatusInd_CallId   0x280000
typedef  UInt16   CCS_StatusInd_CallId_t;

#define  CCS_StatusInd_StatusCode   0x200002
typedef  UInt8   CCS_StatusInd_StatusCode_t;

typedef struct {
   UInt16   CallId;
   UInt8      StatusCode;
} CCS_StatusInd_t;

#define  CCS_StatusInd_get(msgPtr, structAddr, offset)   \
          qMsgVarFieldGet((msgPtr), 2, &(offset),   \
          (CCS_StatusInd_CallId), &((structAddr)->CallId),   \
          (CCS_StatusInd_StatusCode), &((structAddr)->StatusCode))


#define  CCS_StatusInd_put(msgPtr, structAddr, offset)   \
          qMsgVarFieldPut((msgPtr), 2, &(offset),   \
          (CCS_StatusInd_CallId), &((structAddr)->CallId),   \
          (CCS_StatusInd_StatusCode), &((structAddr)->StatusCode))


#define  CCS_StatusInd_Size      3

/******************************************************************************
* CCS_DataLinkInd is an output message for component CCS.
******************************************************************************/

#define  CCS_DataLinkInd      (CCSMSGBASE+0x0009)   /* Msg # for DataLinkInd */

/*
 * This message is use for reporting the D-channel status change for a
 * specified PRI line. Note: the DSL_Id is in the header of the message.
 * Direction               - OUT
 * Information Elements    - None
 */

#define  CCS_DataLinkInd_StatusCode   0x200000
typedef  UInt8   CCS_DataLinkInd_StatusCode_t;

typedef struct {
   UInt8   StatusCode;
} CCS_DataLinkInd_t;

#define  CCS_DataLinkInd_get(msgPtr, structAddr, offset)   \
          qMsgVarFieldGet((msgPtr), 1, &(offset),   \
          (CCS_DataLinkInd_StatusCode), &((structAddr)->StatusCode))


#define  CCS_DataLinkInd_put(msgPtr, structAddr, offset)   \
          qMsgVarFieldPut((msgPtr), 1, &(offset),   \
          (CCS_DataLinkInd_StatusCode), &((structAddr)->StatusCode))


#define  CCS_DataLinkInd_Size      1

/******************************************************************************
* CCS_GetStatistics is an input message for component CCS.
******************************************************************************/

#define  CCS_GetStatistics      (CCSMSGBASE+0x000A)

#define  CCS_GetStatistics_Size      0

/******************************************************************************
* CCS_GetNClearStats is an input message for component CCS.
******************************************************************************/

#define  CCS_GetNClearStats      (CCSMSGBASE+0x000B)

#define  CCS_GetNClearStats_Size      0

/******************************************************************************
* CCS_StatisticsCmplt is an output message for component CCS. It is sent in
* response to the message CCS_GetStatistics or CCS_GetNClearStats.
******************************************************************************/

#define  CCS_StatisticsCmplt    (CCSMSGBASE+0x000C)

#define  MAX_MDL_ERR 15

#define  CCS_Statistics_mdlErr   ((QT_INT16 << QFIELDDEF_DATATYPE) + \
                                        ((MAX_MDL_ERR-1) << QFIELDDEF_COUNT))
typedef  Int16   CCS_Statistics_mdlErr_t[MAX_MDL_ERR];

#define  CCS_Statistics_invalFrameCnt ((QT_INT16 << QFIELDDEF_DATATYPE) \
                          + ((MAX_MDL_ERR*QT_INT16_SIZE) << QFIELDDEF_OFFSET))
typedef  Int16   CCS_Statistics_invalFrameCnt_t;

#define  CCS_Statistics_rxOverflowCnt ((QT_INT16 << QFIELDDEF_DATATYPE) \
                      + (((MAX_MDL_ERR+1)*QT_INT16_SIZE) << QFIELDDEF_OFFSET))
typedef  Int16   CCS_Statistics_rxOverflowCnt_t;

#define  CCS_Statistics_crcErrorCnt ((QT_INT16 << QFIELDDEF_DATATYPE) \
                      + (((MAX_MDL_ERR+2)*QT_INT16_SIZE) << QFIELDDEF_OFFSET))
typedef  Int16   CCS_Statistics_crcErrorCnt_t;

#define  CCS_Statistics_rxAbortedCnt ((QT_INT16 << QFIELDDEF_DATATYPE) \
                      + (((MAX_MDL_ERR+3)*QT_INT16_SIZE) << QFIELDDEF_OFFSET))
typedef  Int16   CCS_Statistics_rxAbortedCnt_t;

typedef struct {
   Int16   mdlErr[MAX_MDL_ERR];  /* MDL Error Indications, Table II.1/Q.921 */
   Int16   invalFrameCnt;                        /* Invalid RX frames, HDLC */
   Int16   rxOverflowCnt;                             /* RX overflows, HDLC */
   Int16   crcErrorCnt;                              /* RX CRC errors, HDLC */
   Int16   rxAbortedCnt;            /* RX aborted frames (via sender), HDLC */
   } CCS_Statistics_t;

#define  CCS_Statistics_get(msgPtr, structAddr, offset)   \
          qMsgVarFieldGet((msgPtr), 5, &(offset),   \
            (CCS_Statistics_mdlErr), &((structAddr)->mdlErr), \
            (CCS_Statistics_invalFrameCnt), &((structAddr)->invalFrameCnt), \
            (CCS_Statistics_rxOverflowCnt), &((structAddr)->rxOverflowCnt), \
            (CCS_Statistics_crcErrorCnt), &((structAddr)->crcErrorCnt), \
            (CCS_Statistics_rxAbortedCnt), &((structAddr)->rxAbortedCnt))

#define  CCS_Statistics_put(msgPtr, structAddr, offset)   \
          qMsgVarFieldPut((msgPtr), 5, &(offset),   \
            (CCS_Statistics_mdlErr), &((structAddr)->mdlErr), \
            (CCS_Statistics_invalFrameCnt), &((structAddr)->invalFrameCnt), \
            (CCS_Statistics_rxOverflowCnt), &((structAddr)->rxOverflowCnt), \
            (CCS_Statistics_crcErrorCnt), &((structAddr)->crcErrorCnt), \
            (CCS_Statistics_rxAbortedCnt), &((structAddr)->rxAbortedCnt))


#define  CCS_StatisticsCmplt_Size   ((MAX_MDL_ERR + 4)*QT_INT16_SIZE)

/******************************************************************************
* CCS_ActivateLink is an input message for component CCS.
******************************************************************************/

#define  CCS_ActivateLink      (CCSMSGBASE+0x000D)

#define  CCS_ActivateLink_Size      0

/******************************************************************************
* CCS_ActivateLinkCmplt is an output message for component CCS.
******************************************************************************/

#define  CCS_ActivateLinkCmplt      (CCSMSGBASE+0x000E)

#define  CCS_ActivateLinkCmplt_Size      0

/******************************************************************************
* CCS_DeActivateLink is an input message for component CCS.
******************************************************************************/

#define  CCS_DeActivateLink      (CCSMSGBASE+0x000F)

#define  CCS_DeActivateLink_Size      0

/******************************************************************************
* CCS_DeActivateLinkCmplt is an output message for component CCS.
******************************************************************************/

#define  CCS_DeActivateLinkCmplt      (CCSMSGBASE+0x0010)

#define  CCS_DeActivateLinkCmplt_Size      0


/******************************************************************************
* CCS component indexing for message count statistics. REQs and RESPs are
* generated by the CHP component and flow down the ISDN stack. INDs and CONFs
* flow up from the ISDN stack to the CHP component. The response to a REQ is
* a CONF, if there is any response. The response to an IND is a RESP, if there
* is any response.
******************************************************************************/
typedef enum ccs_msg_count {
              /************/
              /* Incoming */
              /************/
         CCS_ALERTING_IND,             /* 0x00 */
         CCS_CALL_PROCEEDING_IND,      /* 0x01 */
         CCS_CLEAR_CONF,               /* 0x02 Release_Comp */
         CCS_CLEAR_IND,                /* 0x03 Release */
         CCS_CONG_CONTROL_IND,         /* 0x04 */
         CCS_CONNECT_IND,              /* 0x05 */
         CCS_CONNECT_CONF,             /* 0x06 */
         CCS_INFO_IND,                 /* 0x07 */
         CCS_NOTIFY_IND,               /* 0x08 */
         CCS_PROGRESS_IND,             /* 0x09 */
         CCS_RESUME_CONF,              /* 0x0A */
         CCS_RESUME_IND,               /* 0x0B */
         CCS_RESUME_REJ_CONF,          /* 0x0C */
         CCS_SETUP_CONF,               /* 0x0D */
         CCS_SETUP_IND,                /* 0x0E */
         CCS_SUSPEND_CONF,             /* 0x0F */
         CCS_SUSPEND_IND,              /* 0x10 */
         CCS_SUSPEND_REJ_CONF,         /* 0x11 */
         CCS_USER_INFO_IND,            /* 0x12 */
         CCS_REGISTER_CONF,            /* 0x13 */
         CCS_MANAGEMENT_IND,           /* 0x14 No associated C.R. */
         CCS_DISC_IND,                 /* 0x15 */
         CCS_CHAN_STATUS_IND,          /* 0x16 */
         CCS_LINK_STATUS_IND,          /* 0x17 */
         CCS_FACILITY_CONF,            /* 0x18 */
         CCS_FACILITY_IND,             /* 0x19 */
         CCS_FACILITY_REJ_CONF,        /* 0x1A */
         CCS_HOST_CONTROL_IND,         /* 0x1B */
              /************/
              /* Outgoing */
              /************/
         CCS_ALERTING_REQ,             /* 0x1C */
         CCS_CALL_PROCEEDING_REQ,      /* 0x1D */
         CCS_CLEAR_REQ,                /* 0x1E Release */
         CCS_CLEAR_RESP,               /* 0x1F Release_Comp */
         CCS_CONG_CONTROL_REQ,         /* 0x20 */
         CCS_CONNECT_REQ,              /* 0x21 */
         CCS_INFO_REQ,                 /* 0x22 */
         CCS_NOTIFY_REQ,               /* 0x23 */
         CCS_PROGRESS_REQ,             /* 0x24 */
         CCS_RESUME_REJ_RESP,          /* 0x25 */
         CCS_RESUME_REQ,               /* 0x26 */
         CCS_RESUME_RESP,              /* 0x27 */
         CCS_SETUP_REQ,                /* 0x28 */
         CCS_SETUP_RESP,               /* 0x29 */
         CCS_SUSPEND_REJ_RESP,         /* 0x2A */
         CCS_SUSPEND_REQ,              /* 0x2B */
         CCS_SUSPEND_RESP,             /* 0x2C */
         CCS_USER_INFO_REQ,            /* 0x2D */
         CCS_REGISTER_REQ,             /* 0x2E */
         CCS_MANAGEMENT_REQ,           /* 0x2F No associated C.R. */
         CCS_DISC_REQ,                 /* 0x30 */
         CCS_CHAN_STATUS_REQ,          /* 0x31 */
         CCS_LINK_STATUS_REQ,          /* 0x32 */
         CCS_FACILITY_REJ_RESP,        /* 0x33 */
         CCS_FACILITY_REQ,             /* 0x34 */
         CCS_FACILITY_RESP,            /* 0x35 */
         CCS_HOST_CONTROL_REQ,         /* 0x36 */
              /*********/
              /* Other */
              /*********/
         CCS_UNDEFINED_MSG             /* 0x37 Should never occur */
} CCSMsgCount;

#define  CCS_MSG_CNT_ARRAY_SIZE   (CCS_UNDEFINED_MSG+1) /* 0x37 */

/******************************************************************************
* CCS_GetMsgCnts is an input message for component CCS.
******************************************************************************/

#define  CCS_GetMsgCnts      (CCSMSGBASE+0x0011)

#define  CCS_GetMsgCnts_Size      0

/******************************************************************************
* CCS_GetNClearMsgCnts is an input message for component CCS.
******************************************************************************/

#define  CCS_GetNClearMsgCnts      (CCSMSGBASE+0x0012)

#define  CCS_GetNClearMsgCnts_Size      0

/******************************************************************************
* CCS_MsgCntsCmplt is an output message for component CCS. It is sent in
* response to the message CCS_GetMsgCnts or CCS_GetNClearMsgCnts. The response
* is an array, of 32 bit integer values, the size of CCS_MSG_CNT_ARRAY_SIZE.
******************************************************************************/

#define  CCS_MsgCntsCmplt    (CCSMSGBASE+0x0013)

#define  CCS_MsgCnts   ((QT_INT32 << QFIELDDEF_DATATYPE) + \
                           ((CCS_MSG_CNT_ARRAY_SIZE-1) << QFIELDDEF_COUNT))
typedef  Int32    CCS_MsgCnts_t[CCS_MSG_CNT_ARRAY_SIZE];

#define  CCS_MsgCnts_get(msgPtr, arrayAddr, offset)   \
          qMsgVarFieldGet((msgPtr), 1, &(offset),   \
            (CCS_MsgCnts), (arrayAddr))

#define  CCS_MsgCnts_put(msgPtr, arrayAddr, offset)   \
          qMsgVarFieldPut((msgPtr), 1, &(offset),   \
            (CCS_MsgCnts), (arrayAddr))

#define  CCS_MsgCntsCmplt_Size   ((CCS_MSG_CNT_ARRAY_SIZE)*QT_INT32_SIZE)

/******************************************************************************
* CCS_ChanMngmtReq is an input message for component CCS.
*
* CCS_ChanMngmtInd is an output message from component CCS, to the Registered
* B-Channel.
******************************************************************************/
#define  CCS_ChanMngmtReq      (CCSMSGBASE+0x0014)
#define  CCS_ChanMngmtInd      (CCSMSGBASE+0x0015)

#define  CCS_ChanMngmt_Action   (QT_UINT8 << QFIELDDEF_DATATYPE)
typedef  UInt8   CCS_ChanMngmt_Action_t;

#define  CCS_ChanMngmt_CallState   ((QT_UINT8 << QFIELDDEF_DATATYPE) + \
                                     ((QT_UINT8_SIZE) << QFIELDDEF_OFFSET))
typedef  UInt8   CCS_ChanMngmt_CallState_t;

#define  CCS_ChanMngmt_BChannelId   ((QT_UINT8 << QFIELDDEF_DATATYPE) + \
                                     ((2*QT_UINT8_SIZE) << QFIELDDEF_OFFSET))
typedef  UInt8   CCS_ChanMngmt_BChannelId_t;

#define  CCS_ChanMngmt_InterfaceId  ((QT_UINT8 << QFIELDDEF_DATATYPE) + \
                                     ((3*QT_UINT8_SIZE) << QFIELDDEF_OFFSET))
typedef  UInt8   CCS_ChanMngmt_InterfaceId_t;

/*
 * Action Values
*/
#define  CCS_CHAN_IN_SERVICE           0  /* Req/Ind for channel in service */
#define  CCS_CHAN_MAINTENANCE          1  /* Req/Ind for channel in maint. */
#define  CCS_CHAN_OUT_OF_SERVICE       2  /* Req/Ind for channel out of serv. */
#define  CCS_CHAN_LOC_OUT_OF_SERVICE   3  /* Req/Ind for chan locally out of serv. */
#define  CCS_CHAN_RESTART              4  /* Req/Ind for channel restart */
#define  CCS_CHAN_RESTART_ERR          5  /* Ind for channel restart error */

/*
 * CallState Values (Meaningfull only for CCS_ChanMngmtInd message.)
*/
#define  CCS_CHAN_IDLE              0     /* Channel available state */
#define  CCS_CHAN_PROPOSED          1     /* Channel being negotiated */
#define  CCS_CHAN_BUSY              2     /* Channel unavailable state */
#define  CCS_CHAN_RESERVED          3     /* Channel has been reserved */
#define  CCS_CHAN_RESTART_PEND      4     /* Channel restart initiated */
#define  CCS_CHAN_MAINT_PEND        5     /* Channel maint. pending */


typedef struct {
   UInt8    Action;			/* Service action/report */
   UInt8    CallState;			/* Channel call state */
   UInt8    BChannelId;			/* B-Channel for service req./ind. */
   UInt8    InterfaceId;		/* Interface Id (NFAS only) */
} CCS_ChanMngmt_t, CCS_ChanMngmtReq_t, CCS_ChanMngmtInd_t;

#define  CCS_ChanMngmt_get(msgPtr, structAddr, offset)   \
          qMsgVarFieldGet((msgPtr), 4, &(offset),   \
          (CCS_ChanMngmt_Action), &((structAddr)->Action),   \
          (CCS_ChanMngmt_CallState), &((structAddr)->CallState),   \
          (CCS_ChanMngmt_BChannelId), &((structAddr)->BChannelId),   \
          (CCS_ChanMngmt_InterfaceId), &((structAddr)->InterfaceId))

#define  CCS_ChanMngmtReq_get  CCS_ChanMngmt_get
#define  CCS_ChanMngmtInd_get  CCS_ChanMngmt_get


#define  CCS_ChanMngmt_put(msgPtr, structAddr, offset)   \
          qMsgVarFieldPut((msgPtr), 4, &(offset),   \
          (CCS_ChanMngmt_Action), &((structAddr)->Action),   \
          (CCS_ChanMngmt_CallState), &((structAddr)->CallState),   \
          (CCS_ChanMngmt_BChannelId), &((structAddr)->BChannelId),   \
          (CCS_ChanMngmt_InterfaceId), &((structAddr)->InterfaceId))

#define  CCS_ChanMngmtReq_put       CCS_ChanMngmt_put
#define  CCS_ChanMngmtInd_put       CCS_ChanMngmt_put

#define  CCS_ChanMngmt_Size          (4*QT_UINT8_SIZE)
#define  CCS_ChanMngmtReq_Size       CCS_ChanMngmt_Size
#define  CCS_ChanMngmtInd_Size       CCS_ChanMngmt_Size


/* The following definitions were moved here from ccscomp.h 
   since this is where they really should belong */
/*****************************************************************/
#define  UNDECLARED                 (CCSMSGBASE + 0x00FF)
#define  ALL_INST_REG               (CCSMSGBASE + 0x00FE)
#define  CCS_DATA_LNK_DOWN          (CCSMSGBASE + 0x00FD)
#define  CCS_DATA_LNK_UP            (CCSMSGBASE + 0x00FC)
#define  CCS_IE_ERR                 (CCSMSGBASE + 0x00FB)
#define  CCS_CHPREG_ERR             (CCSMSGBASE + 0x00FA)
#define  CCS_CHPHAND_ERR            (CCSMSGBASE + 0x00F9)
#define  CCS_STACK_INITED           (CCSMSGBASE + 0x00F8)
#define  CCS_ALL_INST_EXITED        (CCSMSGBASE + 0x00F7)
/*****************************************************************/

#ifdef DLGC_NFAS
#define  CCS_GettingPriDchanInfo    (CCSMSGBASE + 0x00F6)
#endif
#define  CCS_GenericDM3Message      (CCSMSGBASE + 0x00F5)
#define  CCS_PERST_L2_ACT           (CCSMSGBASE + 0x00F4)
#endif   /* _ccs_defs_h */

