#ifndef _DTMFDBG_H__
#define _DTMFDBG_H__
/****************************************************************************
 *		Copyright (c) 1995 Dialogic Corp.
 *		All Rights Reserved.
 *
 *		THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF Dialogic Corp.
 *		The copyright notice above does not evidence any actual or
 *		intended publication of such source code.
 ***************************************************************************/

/****************************************************************************
 * FILE:         
 * AUTHOR:       Chris Chuba
 * ORG:          Dialogic Corporation
 * ORIG-DATE:     1-May-96 at 10:47:28
 * LAST-MOD:     10-Mar-03 at 16:12:28 by Baowen Chen
 * PURPOSE:
 *    This module contains a limited configuration ability for DTMF detection.
 *
 * HISTORY:
 * Date     Who   Description
 * 10-Mar-03 BC   Task 17490: Enhanced CPA phase 1b. 
 *                Reduced the PVD and PAMD templates to support 5 templates.
 *                Also removed some of previous commented out defines.
 * 27-Feb-03 JPO  Task 17153: Fix PT26745. Added FT_SILENCE_POSTPONE and
 *		  FT_SILENCE_INITIAL_TIME values for the FT_ bitmap.
 * 26-Feb-03 AE   Task 17470: Merged (TASK-15524:PT27713): Added the SILENCE_DEB template.   
 * 03-Feb-03 BC   Task 17185: Fixed PT28822. This PT was filed against
 *                SR.5.1.1 but this fix should also go into 6.0.
 *                Modified the default SIT tone templates to meet BellCore
 *                and ITU spec.
 * 02-Apr-02 AR   Commented out the R2MF related signal tone templates and
 *                frequency templates
 *                (TASK-13253:PT-N/A)
 * 21-Jan-02 BC   Task 12136: Removed all Luna/HMP related defines. 
 *                Actually it reversed back task 11877.
 * 09-Jan-02 BC   Task 11877: Added/Moved Luna related defs to here.
 *                "X86" and "HMP" compiler switch are defined here. Any file
 *                which need to use "HMP" compile switch must include this
 *                file. All the HMP related global variables should defined
 *                here.
 *                Added "#include <qcomplib.h>" for QCompDesc defines.
 * 14-Jun-01 BC   Changed the tone amplitude defs in FTEMP structure from
 *                Int16 to signed char because they are defined as Int8 in CIS
 *                and not correct. Task 9016 to fix PT 17545.
 * 29-Mar-01 AR   Modified SD_MAX_TONES and SD_MAX_GTD_TEMPLATES 
 * 11-Feb-01 BC   Added new defs and supports for the new sigdet code. 
 *                Also enable NEWPVD, NEWPAMD. 
 *                Disabled BitStorage.h since not used in the
 *                new sigdet code.
 * 15-Jul-99 AR   Preserving the the contents of Qual Templates till we figure
                  out how to incorporate the fcd changes to all our products.
                  For similar reason the pvd and pamd templates have been disabled 
                  temporarily till we figure out the way of updating fcd changes 
                  in all our products.
 * 11-Jul-99 AR   Added support for dynamic Qualification, Pvd, Pamd templates.
 * 3-2-99    TH   Addition of R2Mf support
 * 5-1-96    CC   Initial release.
 ***************************************************************************/
#include <dtmf.h>
#include <mf.h>
#include <r2.h>

/*
Author               : Amin Remtulla
File                 : BitStorage.h
Date Last Modified   : 
Modified By          : Amin Remtulla (x.5079)
*/

#if 0 /* it is not used in the new sigdet code. baowen chen.*/
#ifndef __BitStorage_H__
#define __BitStorage_H__

/* 
 * The typedef for storage unit.
 */
typedef UInt32           BITStorage;


/* 
 * Computes the zero based index of the storage units.
 * --idx : bit number for which index to storage unit is to be computed.
 */
#define BIT_FINDINDEX(idx)     (((BITStorage)idx) / (sizeof(BITStorage) * 8))


/* 
 * Computes the zero based offset within the storage unit.
 * --idx : bit number for which offset to storage unit is to be computed.
 */
/*
#define BIT_FINDOFFSET(idx)       ((BITStorage)idx - (BIT_FINDINDEX(idx) * sizeof(BITStorage) * 8))
*/
#define BIT_FINDOFFSET(idx)       (((BITStorage)idx) % (sizeof(BITStorage) * 8))


/* 
 * Computes the number of storage units required to store MaxElt bits.
 * -- numBits : Number of bits to be stored.
 */
#define BIT_MAXSTORAGE(numBits) (BIT_FINDINDEX((numBits) - 1) + 1)


/*
 * Sets the bit number, elt of stat (the array of storage units).
 * --Stat   : Array of the storage units.
 * --size   : Total number of bits in stat.
 * --elt    : which bit to SET.
 * --retVal : 0=Error, 1=Success
 * 
 */
#define BIT_SET(stat, size, elt, retVal) \
    if (elt >= size) retVal = 0; \
    else { \
        stat[BIT_FINDINDEX(elt)] |= (1 << BIT_FINDOFFSET(elt)); \
        retVal = 1; \
    }


/*
 * Sets all the bit of stat (the array of storage units).
 * --Stat     : Array of the storage units.
 * --size     : Total number of bits in stat.
 * --loopIdx  : The counter for the for loop.
 * --retVal   : 0=Error, 1=Success
 * 
 */
#define BIT_SETALL(stat, size, loopIdx, retVal) \
    for(loopIdx = 0; loopIdx < size; loopIdx++) { \
        BIT_SET(stat, size, loopIdx, retVal); \
        if (retVal == 0) break; \
    }


/*
 * Resets the bit number, elt of stat (the array of storage units).
 * --Stat   : Array of the storage units.
 * --size   : Total number of bits in stat.
 * --elt    : which bit to RESET.
 * --retVal : 0=Error, 1=Success.
 * 
 */
#define BIT_RESET(stat, size, elt, retVal)\
    if (elt >= size) retVal = 0; \
    else { \
        stat[BIT_FINDINDEX(elt)] = ~(stat[BIT_FINDINDEX(elt)]); \
        BIT_SET(stat,size,elt,retVal); \
        stat[BIT_FINDINDEX(elt)] = ~(stat[BIT_FINDINDEX(elt)]); \
        retVal = 1; \
    }



/*
 * Resets all the bit of stat (the array of storage units).
 * --Stat     : Array of the storage units.
 * --size     : Total number of bits in stat.
 * --loopIdx  : The counter for the for loop.
 * --retVal   : 0=Error, 1=Success
 * 
 */
#define BIT_RESETALL(stat, size, loopIdx, retVal) \
    for(loopIdx = 0; loopIdx < size; loopIdx++) { \
        BIT_RESET(stat, size, loopIdx, retVal); \
        if (retVal == 0) break; \
    }


/*
 * Determines whether or not the bit number, elt of stat (the array of storage units)
 * is set to 1. The result is stored in theRetVal. The interpretation of the result
 * is as follows:
 * - The value of theRetVal is 1 implies the bit number, elt is set to 1.
 * - The value of theRetVal is -1 implies the bit number, elt is NOT set to 1.
 * --Stat      : Array of the storage units.
 * --size      : Total number of bits in stat.
 * --elt       : which bit to see if set to 1.
 * --theRetVal : 0=Error, -1=ISNOTSET, 1=ISSET
 */
#define BIT_ISSET(stat, size, elt, theRetVal) \
    if (elt >= size) \
        theRetVal = 0; \
    else  \
        theRetVal = ((stat[BIT_FINDINDEX(elt)] >> BIT_FINDOFFSET(elt)) & 1) ? 1 : -1; 


/*
 * Finds the first bit number of stat (the array of storage units) which 
 * is set to 1. The result is stored in theRetVal. The interpretation of the result
 * is as follows:
 * - The value of theRetVal is 0 means serious error.
 * - The value of theRetVal is -1 implies none of the bits are SET
 * - else theRetVal is the bit number of the first bit that is SET.
 * --Stat      : Array of the storage units.
 * --size      : Total number of bits in stat.
 * --loopIdx   : The counter for the for loop.
 * --theRetVal : -1=NOTFOUND else the Index ...
 */
#define BIT_FINDFIRSTSET(stat, size, loopIdx, theRetVal) \
    theRetVal = -1; \
    for(loopIdx = 0; loopIdx < size; loopIdx++) { \
        BIT_ISSET(stat, size, loopIdx, theRetVal); \
        if (theRetVal == 1) { \
            theRetVal = loopIdx; \
            break; \
        } \
    }


 /*
 * Determines whether or not the bit number, elt of stat (the array of storage units)
 * is reset to 0. The result is stored in theRetVal. The interpretation of the result
 * is as follows:
 * - The value of theRetVal is 1 implies the bit number, elt is reset to 0.
 * - The value of theRetVal is -1 implies the bit number, elt is NOT reset to 0.
 * --Stat      : Array of the storage units.
 * --size      : Total number of bits in stat.
 * --elt       : which bit to see if reset to 0.
 * --theRetVal : 0=Error, -1=ISNOTRESET, 1=ISRESET
 */
#define BIT_ISRESET(stat, size, elt, theRetVal) \
    BIT_ISSET(stat, size, elt, theRetVal)\
    if (theRetVal != 0) \
        theRetVal = (theRetVal == 1)?-1:1;


/*
 * Finds the first bit number of stat (the array of storage units) which 
 * is reset to 0. The result is stored in theRetVal. The interpretation of the result
 * is as follows:
 * - The value of theRetVal is -1 implies none of the bits are reset to 0
 * - else theRetVal is the bit number of the first bit that is reset to 0.
 * --Stat      : Array of the storage units.
 * --size      : Total number of bits in stat.
 * --loopIdx   : The counter for the for loop.
 * --theRetVal : -1=NOTFOUND else the Index ...
 */
#define BIT_FINDFIRSTRESET(stat, size, loopIdx, theRetVal) \
    theRetVal = -1; \
    for(loopIdx = 0; loopIdx < size; loopIdx++) { \
        BIT_ISRESET(stat, size, loopIdx, theRetVal); \
        if (theRetVal == 1) { \
            theRetVal = loopIdx; \
            break; \
        } \
    }


/*
 * Finds the first bit number of stat (the array of storage units) which 
 * is set to 1. It also reset found bit.
 * The result is stored in pos and theRetVal. 
 * The interpretation of the result is as follows:
 * - The value of theRetVal is -1 implies none of the bits are reset to 0
 * - else theRetVal is the bit number of the first bit that is reset to 0.
 * --Stat      : Array of the storage units.
 * --size      : Total number of bits in stat.
 * --loopIdx   : The counter for the for loop.
 * --pos       : which bit number first set. -1 indicates no Set value exists.
 * --retVal    : 0 - Error and 1 - Success.
 */
#define BIT_FINDFIRSTSETex(stat, size, loopIdx, pos, retVal) \
    BIT_FINDFIRSTSET(stat, size, loopIdx, pos) \
    retVal = 1; \
    if (pos != -1)     BIT_RESET(stat, size, pos, retVal);

/*
 * Finds the first bit number of stat (the array of storage units) which 
 * is reset to 0. It also set found bit.
 * The result is stored in pos and theRetVal. 
 * The interpretation of the result is as follows:
 * - The value of theRetVal is -1 implies none of the bits are reset to 0
 * - else theRetVal is the bit number of the first bit that is reset to 0.
 * --Stat      : Array of the storage units.
 * --size      : Total number of bits in stat.
 * --loopIdx   : The counter for the for loop.
 * --pos       : which bit number first reset. -1 indicates no reset value exists.
 * --retVal    : 0 - Error and 1 - Success.
 */
#define BIT_FINDFIRSTRESETex(stat, size, loopIdx, pos, retVal) \
    BIT_FINDFIRSTRESET(stat, size, loopIdx, pos) \
    retVal = 1; \
    if (pos != -1)     BIT_SET(stat, size, pos, retVal);


#endif /*__BitStorage_H__*/
#endif

#define KILLPARM  3
/* (A.R) Memory Savings. Reduced MAX_TONES. */
#define SD_MAX_TONES 128 /*255*/
/* (A.R|0.46) defines pertaining to Qualification Templates */
#define NEWQT
#define MAX_QT    16

/* (A.R|0.46) defines pertaining to Pvd Templates */
#define NEWPVD 
#define MAX_PVD   5 /* was 10 */

/* (A.R|0.46) defines pertaining to Pamd Templates */ 
#define NEWPAMD 
#define MAX_PAMD        7  /* was 10. 7 will support 5 different templates
                             since the first 3 are the defaults and use the
                             same qualification templates.*/
#define MAX_PAMD_QT     5

/* (A.R) Memory Savings. Reduced MAX_TONES. */
#define  SD_MAX_GTD_TMPLTS    SD_MAX_TONES /*SD_MAX_TONES*3*/


/*
 * The following is the type def for the qualification template.
 */
typedef struct qtemp {
   Uint16   nfrm;          /* num frms needed to verify */
   Uint16   snr0;          /* min snr for 1st frame     */
   Uint16   snr1;          /* min snr for 2nd - nfrm    */
   Uint16   b0;            /* min broad for 1st frame   */
   Uint16   b1;            /* min broad for 2nd - nfrm  */
   Uint16   trat;          /* maximum twist ratio       */
   Uint16   qmax;          /* pass frame if qcnt>=qmax  */
   Uint16   tnfrm;         /* num frame to test talkoff */
   Uint16   tflg;          /* flg to test talkoff or not*/
   Uint16   minfrm;        /* min frms to verify        */
   Uint16   killParm;      /* This number may not exceed 7 */
   UInt16   longKill;      /* Sometime this is used instead of killparm*/
} QTEMP;

/* 
 * longKill - This is used to keep a DTMF's from being split.  If what
 *          breaks a DTMF tone is audio energy above a threshold then 
 *          this parameter is used instead of killParm.
 */

/*
 * FT_??? - Is a mask for the mask field of the Frequency template
 */
#define  FT_DUALTONE 1
#define  FT_RING     2
#define  FT_SILENCE  4 
#define  FT_DIALTONE 8
#define  FT_SILENCE_POSTPONE      0x10 /* 16, JPO Task 17153 PT26745 */
#define  FT_SILENCE_INITIAL_TIME  0x20 /* 32, JPO Task 17153 PT26745 */
#define  FT_FREE     0xff00 /* added to indicate this entry is free. Note,
                               now 0 is used as the default for sigle
                               tone. Late it will change to other value,
                               and let FT_FREE use 0. Baowen Chen*/



/*
 * The following structure defines the frequency range for all the 16 DTMF
 * digits.  Note, gtd stands for General Tone Detection.
 */
typedef struct ftemp {
   Uint16    mask;       /* dual/single tone flag                  */
   Uint16   f0l;        /* low bound of 0th or lowest freq peak   */
   Uint16   f0h;        /* hi bound of 0th or lowest freq peak    */
   Uint16   f1l;        /* low bound of 1st or lowest freq peak   */
   Uint16   f1h;        /* hi bound of 1st or lowest freq peak    */
   Uint16   qt;         /* The Qualification Template index (see QTEMP) */

   signed char   amp0l;      /* dBm0 (0 - 40)The lower amplitude range for f0 */
   signed char   amp0h;      /* The upper amplitude range for f0 */
   signed char   amp1l;      /* The lower amplitude range for f1 */
   signed char   amp1h;      /* The upper amplitude range for f1 */

   Uint16   fTwinL;     /* Lower frequency bound for the twin template */
   Uint16   fTwinH;     /* Upper frequence bount for the twin template */
} FTEMP;

/*
 * This is the typedef for the tone segment
 */

typedef  struct 
{
   Uint16   toneDefNum;
   Uint16   minOnDuration;
   Uint16   maxOnDuration;
   Uint16   minOffDuration;
   Uint16   maxOffDuration;
}  ToneSeg;

/*
 * This typedef is for the tone template definition.
 */
typedef  struct 
{
   Uint32   DeleteHndl;    
   Uint32   id;
   Uint8    label[4];
   Uint8    labelSize;     /* The number of Uint8's in the label */
   Uint16   repCount;      /* The number of times to cycle the seg list */
   Uint16   numOfSegs;     /* The size of the segment list */
   ToneSeg  seg[4];        /* For now assume a maximum of 4 segments */
} FixedToneSignalTmplt;    /* This def is added for new sigdet code. Baowen Chen.*/

typedef  struct 
{
   Uint16   DeleteHndl;    /* was Uint32.*/
   Uint16   refCount;      /* Number of instances using this tone.*/
   Uint32   id;
   Uint8    label[4];
   Uint8    labelSize;     /* The number of Uint8's in the label */
   Uint16   repCount;      /* The number of times to cycle the seg list */
   Uint16   numOfSegs;     /* The size of the segment list */
   ToneSeg  seg[4];        /* For now assume a maximum of 4 segments */
} ToneSignalTmplt;

/*
 * The following structure is for the PVD algorithm.
 */
typedef  struct {
   Uint32   id;
   Uint24   minSnr;
   Uint24   maxSnr;
   Uint16   maxPk;
   Uint16   maxRing;
   Uint24   ringThreshold;
   Uint24   windowSize; /* # of frames before doing voice tests */
   Uint24   VThreshold; /* Min. energy required for collecting frame */
   Uint16   rbLow;   /* Ringback frequency lower limit */
   Uint16   rbHigh;  /* Ringback frequence upper limit */
} FixedPvdSignalTmplt; /* this def is added for the new sigdet code. Baowen Chen.*/

typedef  struct {
   Uint32   id;
   Uint24   minSnr;
   Uint24   maxSnr;
   Uint16   maxPk;
   Uint16   maxRing;
   Uint24   ringThreshold;
   Uint24   windowSize; /* # of frames before doing voice tests */
   Uint24   VThreshold; /* Min. energy required for collecting frame */
   Uint16   rbLow;   /* Ringback frequency lower limit */
   Uint16   rbHigh;  /* Ringback frequence upper limit */
   Uint16   refCount; /* Number of instances using this PVD. Baowen Chen*/
} PvdSignalTmplt;

/*
 * The following structure is for the PAMD algorithm.
 * Note, this is a qualification template for the algorithm.  An isntance
 * of this structure may be shared by multiple PAMD templates definitions
 * PamdTmplt.
 */
typedef  struct {
   Uint16   minRing;
   Uint16   maxAnsiz1;
   Uint16   maxAnsiz2;
   Uint16   maxAnsiz3;
   Uint16   loHiss;
   Uint16   hiHiss;
   Uint16   bhParm;
   Uint16   cvThresh1;
   Uint16   cvThresh2;
   Uint16   maxCvThresh;
   Uint16   nMaxBroad;
   Uint16   nMaxErg;
   Uint16   maxSilence;
   Uint16   voiceThresh;
   Uint16   silenceThresh;
   Uint16   rjFbandLow;
   Uint16   rgFbandHigh;
}  PamdQualTmplt;

typedef  struct {
   Uint32   signalId;   /* The signal ID of course */
   Uint16   pamdQTmp;   /* The pamd qualification template */
}  FixedPamdSignalTmplt;/* This def is added for the new sigdet code. Baowen Chen.*/


typedef  struct {
   Uint32   signalId;   /* The signal ID of course */
   Uint16   pamdQTmp;   /* The pamd qualification template */
   Uint16   refCount;   /* Number of instances using this PAMD. Baowen Chen*/
}  PamdSignalTmplt;

/*
 * The SIT?_MIN/MAX defines bracket the U.S. SIT tone
 * Below is the Bellcore spec for SIT tones:
 *       1st tone                2nd tone            3rd tone
 *    Freq.(Hz) Time(ms)    Freq.(Hz) Time(ms)  Freq.(Hz)  Time(ms)
 * NC    985.2    380         1428.5   380         1776.7   380
 * IC    913.8    274         1370.6   274         1776.7   380
 * VC    985.2    380         1370.6   274         1776.7   380
 * RO    913.8    274         1428.5   380         1776,7   380   
 * The frequency is +/-1.5%. The duration shift is +/-5%.
 */

#define  OLD_SIT  0
#if OLD_SIT
#define  SIT1a_MIN   874
#define  SIT1a_MAX   940
#define  SIT1b_MIN   950
#define  SIT1b_MAX   1001
#define  SIT2a_MIN   1310
#define  SIT2a_MAX   1400
#define  SIT2b_MIN   1400
#define  SIT2b_MAX   1450
#define  SIT3_MIN    1740
#define  SIT3_MAX    1800

/* added in task 17185 to meet Bellcore spec. 
 * Added correct duration defines for each SIT tone segments to 
 * allow SIT tone detection able to distingush with some SIT tone
 * segments which may have frequency overlapping.
 */
#else 
#define  SIT1a_MIN   874
#define  SIT1a_MAX   955   /* Increased from 940 to 955. */
#define  SIT1b_MIN   950
#define  SIT1b_MAX   1001
#define  SIT2a_MIN   1310
#define  SIT2a_MAX   1430  /* Increased from 1400 to 1430.*/
#define  SIT2b_MIN   1400
#define  SIT2b_MAX   1450
#define  SIT3_MIN    1740
#define  SIT3_MAX    1850  /* increased from 1800 to 1850.*/

#define  High380     45   /* Nominal 380 ms, no more than 450 ms.*/
#define  Low380      32   /* Nominal 380 ms, no less than 320 ms.*/
#define  High274     30   /* Nominal 274 ms, no more than 300 ms.*/
#define  Low274      15   /* Nominal 274 ms, no less than 150 ms.*/
#endif


#ifdef   _SIGINIT_C__
/*
 * Setting qDtmfShow to 1 will print any DTMF digit that is detected by the 
 * control processor to the VxWorks shell window.  This does validate the 
 * SP <-> CP connection but NOT the CP <-> Host connection.
 */
int qDtmfShow=0;

/*
 * The actual qualification templates used for all the Tone Definitions.
 * The first structure is used for DTMF detection.
 * The second structure is used for Call Analysis. (Ringback,Busy, etc).
 */
   QTEMP    qt[] = {
   #ifdef SdJpFarallon
      {4, 82, 72, 6, 32, 10, 4, 0, 0, 3, 2,2 }, /* Used for DTMF */
   #else
      {4, 82, 72, 6, 32, 10, 4, 0, 0, 3, KILLPARM, 12}, /* Used for DTMF */
   #endif
      {8, 10, 10, 2,  2, 10, 4, 0, 0, 3, KILLPARM,KILLPARM}, /* Used for Call Analysis*/
      {4, 10, 10, 2,  2, 10, 4, 0, 0, 3, KILLPARM,KILLPARM}, /* Used for SIT Tone */
      {3, 76, 76, 6, 32, 10, 4, 0, 0, 1, KILLPARM,KILLPARM}, /* Used for MF Set */
      {4, 82, 72, 6, 32, 10, 4, 3, 1, 2, KILLPARM,KILLPARM}, /* Used for LKP of MS set*/
      {4, 72, 72, 6, 32, 10, 4, 0, 0, 1, KILLPARM,KILLPARM}, /* Used for R2MF Set */
      {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}, /* A NULL ENTRY */
      {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}, /* A NULL ENTRY */
      {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}, /* A NULL ENTRY */
      {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}, /* A NULL ENTRY */
      {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}, /* A NULL ENTRY */
      {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}, /* A NULL ENTRY */
      {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}, /* A NULL ENTRY */
      {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}, /* A NULL ENTRY */
      {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}, /* A NULL ENTRY */
      {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0} /* A NULL ENTRY */
   };

#ifndef NEWQT
   #define  NUM_OF_QUAL_TMPLTS ((sizeof(qt)/sizeof(qt[1]))-1)
   /* int      qNumOfQualTmplts = MAX_QUAL_TMPLTS; */

#else /* NEWQT */

   /* QTEMP qt[MAX_QT]; */
   /* int      qNumOfQualTmplts = MAX_QUAL_TMPLTS; */

#endif /* NEWQT */


/* (A.R) Memory Savings. Reduced MAX_TONES. */
FTEMP qGtdTemplates[SD_MAX_GTD_TMPLTS/*SD_MAX_TONES*3*/];
FTEMP qStaticGtdTemplates[] = { 
/* 
 * Reserved for the Silence on transition 
 */
{0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},  
      
/*  The DTMF Set
 *  1=Dual   f1    f1    f2       f2       Qual    Amp      Amp
 *  0=Single low   high  Low      High     Tmplt   Range    Range
 *  Tone     limit limit Limit    Limit            f0       f1
 */
{FT_DUALTONE, 678,  716,  1177,    1241,    0,     0, 0,    0,0, 0,0}, /* '1' */
{FT_DUALTONE, 678,  716,  1301,    1371,    0,	   0, 0,    0,0, 0,0}, /* '2' */
{FT_DUALTONE, 678,  716,  1439,    1515,    0,     0, 0,    0,0, 0,0}, /* '3' */
{FT_DUALTONE, 678,  716,  1591,    1675,    0,     0, 0,    0,0, 0,0}, /* 'a' */
{FT_DUALTONE, 749,  791,  1177,    1241,    0,     0, 0,    0,0, 0,0}, /* '4' */
{FT_DUALTONE, 749,  791,  1301,    1371,    0,     0, 0,    0,0, 0,0}, /* '5' */
{FT_DUALTONE, 749,  791,  1439,    1515,    0,     0, 0,    0,0, 0,0}, /* '6' */
{FT_DUALTONE, 749,  791,  1591,    1675,    0,     0, 0,    0,0, 0,0}, /* 'b' */
{FT_DUALTONE, 829,  875,  1177,    1241,    0,     0, 0,    0,0, 0,0}, /* '7' */
{FT_DUALTONE, 829,  875,  1301,    1371,    0,     0, 0,    0,0, 0,0}, /* '8' */
{FT_DUALTONE, 829,  875,  1439,    1515,    0,     0, 0,    0,0, 0,0}, /* '9' */
{FT_DUALTONE, 829,  875,  1591,    1675,    0,     0, 0,    0,0, 0,0}, /* 'c' */
{FT_DUALTONE, 916,  966,  1177,    1241,    0,     0, 0,    0,0, 0,0}, /* '*' */
{FT_DUALTONE, 916,  966,  1301,    1371,    0,     0, 0,    0,0, 0,0}, /* '0' */
{FT_DUALTONE, 916,  966,  1439,    1515,    0,     0, 0,    0,0, 0,0}, /* '#' */
{FT_DUALTONE, 916,  966,  1591,    1675,    0,     0, 0,    0,0, 0,0}, /* 'd' */
/*
 *  Sys4 was  678: 716 became  678: 749
 *  Sys4 was  749: 791 became  749: 813
 *  Sys4 was  829: 875 became  811: 876
 *  Sys4 was  916: 966 became  916: 966
 *  Sys4 was 1177:1241 became 1177:1251.
 *  Sys4 was 1301:1371 became 1301:1376
 *  Sys4 was 1439:1515 became 1436:1515
 *  Sys4 was 1591:1675 became 1591:1688
 */
 /* End of DTMF Tone Definitions definitions */
 /* Start of the Call Analysis definitions */
{ 0,     SIT1a_MIN,SIT1a_MAX,  0, 0,    2,     0, 0,    0,0, 0,0}, /* 913 */
{ 0,     SIT2a_MIN,SIT2a_MAX,  0, 0,    2,     0, 0,    0,0, 0,0}, /* 1370 */
{ 0,     SIT3_MIN, SIT3_MAX,   0, 0,    2,     0, 0,    0,0, 0,0}, /* 1776 */
{ 0,     SIT1b_MIN,SIT1b_MAX,  0, 0,    2,     0, 0,    0,0, 0,0}, /* 985 */
{ 0,     SIT2b_MIN,SIT2b_MAX,  0, 0,    2,     0, 0,    0,0, 0,0}, /* 1420 */


/* was 410/510*/
{ FT_RING+FT_DIALTONE+FT_DUALTONE, 350,  550, 350,550,    1,     0, 0,    0,0, 0,0}, /* RING*/
{ FT_DUALTONE, 450,  510,           590,650,  1,     0, 0,    0,0, 0,0}, /* Busy */

{ 0,     2000,2300,           0,0,  1,     0, 0,    0,0, 0,0}, /* FAX CED1 */

/* This detects the HDLC flags for called fax machines */
{ 0,     1550,1750,           0,0,  1,     0, 0,    0,0, 0,0}, /* FAX CED2 */

/*  The MF Set
 *  1=Dual   f1    f1    f2       f2       Qual    Amp      Amp
 *  0=Single low   high  Low      High     Tmplt   Range    Range
 *  Tone     limit limit Limit    Limit            f0       f1
 */
{FT_DUALTONE, 682,  718,   877,     923,    3,	  0, 0,    0,0, 0,0}, /* '1' */
{FT_DUALTONE, 682,  718,  1072,    1128,    3,     0, 0,    0,0, 0,0}, /* '2' */
{FT_DUALTONE, 877,  923,  1072,    1128,    3,     0, 0,    0,0, 0,0}, /* '3' */
{FT_DUALTONE, 682,  718,  1267,    1333,    3,     0, 0,    0,0, 0,0}, /* '4' */
{FT_DUALTONE, 877,  923,  1267,    1333,    3,     0, 0,    0,0, 0,0}, /* '5' */
{FT_DUALTONE,1072, 1128,  1267,    1333,    3,     0, 0,    0,0, 0,0}, /* '6' */
{FT_DUALTONE, 682,  718,  1462,    1538,    3,     0, 0,    0,0, 0,0}, /* '7' */
{FT_DUALTONE, 877,  923,  1462,    1538,    3,     0, 0,    0,0, 0,0}, /* '8' */
{FT_DUALTONE,1072, 1128,  1462,    1538,    3,     0, 0,    0,0, 0,0}, /* '9' */
{FT_DUALTONE,1267, 1333,  1462,    1538,    3,     0, 0,    0,0, 0,0}, /* '0' */
{FT_DUALTONE, 877,  923,  1657,    1743,    3,     0, 0,    0,0, 0,0}, /* 'a'-STP */
{FT_DUALTONE,1267, 1333,  1657,    1743,    3,     0, 0,    0,0, 0,0}, /* 'b'-KP2 */
{FT_DUALTONE, 682,  718,  1657,    1743,    3,     0, 0,    0,0, 0,0}, /* 'c'-ST3P */
{FT_DUALTONE,1462, 1538,  1657,    1743,    3,     0, 0,    0,0, 0,0}, /* '#'-ST */
{FT_DUALTONE,1072, 1128,  1657,    1743,    4,     0, 0,    0,0, 0,0}, /* '*'-(L)KP */

/* 
   Sys4 was  682: 718 became  650: 750
   Sys4 was  877: 923 became  850: 950
   Sys4 was 1072:1128 became 1050:1150
   Sys4 was 1267:1333 became 1250:1350
   Sys4 was 1462:1538 became 1450:1550
   Sys4 was 1657:1743 became 1650:1750

*/

/*
 * This is the Dial Tone Definition 350/440 (310/434 are bin centered values)
 * DIALTONE1
 */
{FT_DUALTONE+FT_DIALTONE,   300, 380,  400,    480,    2,     0, 0,    0,0, 0,0},
/*
 * The CNG tone definition 1100Hz tone 500ms
 */
{0,1050, 1150,  0, 0, 2, 0, 0, 0,0, 0,0},

#if 0
#define R2SIDE 35
/*  The R2 MF forward set
 *  1=Dual    f1           f1           f2           f2        Qual  Amp   Amp
 *  0=Single  low          high         Low          High      Tmplt Range Range
 *  Tone      limit        limit        Limit        Limit           f0    f1
 */
{FT_DUALTONE, 1380-R2SIDE, 1380+R2SIDE, 1500-R2SIDE, 1500+R2SIDE, 5, 0, 0, 0,0, 0,0}, /* '1' */  
{FT_DUALTONE, 1380-R2SIDE, 1380+R2SIDE, 1620-R2SIDE, 1620+R2SIDE, 5, 0, 0, 0,0, 0,0}, /* '2' */
{FT_DUALTONE, 1500-R2SIDE, 1500+R2SIDE, 1620-R2SIDE, 1620+R2SIDE, 5, 0, 0, 0,0, 0,0}, /* '3' */
{FT_DUALTONE, 1380-R2SIDE, 1380+R2SIDE, 1740-R2SIDE, 1740+R2SIDE, 5, 0, 0, 0,0, 0,0}, /* '4' */
{FT_DUALTONE, 1500-R2SIDE, 1500+R2SIDE, 1740-R2SIDE, 1740+R2SIDE, 5, 0, 0, 0,0, 0,0}, /* '5' */
{FT_DUALTONE, 1620-R2SIDE, 1620+R2SIDE, 1740-R2SIDE, 1740+R2SIDE, 5, 0, 0, 0,0, 0,0}, /* '6' */
{FT_DUALTONE, 1380-R2SIDE, 1380+R2SIDE, 1860-R2SIDE, 1860+R2SIDE, 5, 0, 0, 0,0, 0,0}, /* '7' */
{FT_DUALTONE, 1500-R2SIDE, 1500+R2SIDE, 1860-R2SIDE, 1860+R2SIDE, 5, 0, 0, 0,0, 0,0}, /* '8' */
{FT_DUALTONE, 1620-R2SIDE, 1620+R2SIDE, 1860-R2SIDE, 1860+R2SIDE, 5, 0, 0, 0,0, 0,0}, /* '9' */
{FT_DUALTONE, 1740-R2SIDE, 1740+R2SIDE, 1860-R2SIDE, 1860+R2SIDE, 5, 0, 0, 0,0, 0,0}, /* 'a' */
{FT_DUALTONE, 1380-R2SIDE, 1380+R2SIDE, 1980-R2SIDE, 1980+R2SIDE, 5, 0, 0, 0,0, 0,0}, /* 'b' */ 
{FT_DUALTONE, 1500-R2SIDE, 1500+R2SIDE, 1980-R2SIDE, 1980+R2SIDE, 5, 0, 0, 0,0, 0,0}, /* 'c' */
{FT_DUALTONE, 1620-R2SIDE, 1620+R2SIDE, 1980-R2SIDE, 1980+R2SIDE, 5, 0, 0, 0,0, 0,0}, /* 'd' */
{FT_DUALTONE, 1740-R2SIDE, 1740+R2SIDE, 1980-R2SIDE, 1980+R2SIDE, 5, 0, 0, 0,0, 0,0}, /* 'e' */
{FT_DUALTONE, 1860-R2SIDE, 1860+R2SIDE, 1980-R2SIDE, 1980+R2SIDE, 5, 0, 0, 0,0, 0,0}, /* 'f' */
/*  The R2 MF backward set
 *  1=Dual    f1           f1           f2           f2        Qual  Amp   Amp
 *  0=Single  low          high         Low          High      Tmplt Range Range
 *  Tone      limit        limit        Limit        Limit           f0    f1
 */
{FT_DUALTONE, 1020-R2SIDE, 1020+R2SIDE, 1140-R2SIDE, 1140+R2SIDE, 5, 0, 0, 0,0, 0,0}, /* '1' */  
{FT_DUALTONE,  900-R2SIDE,  900+R2SIDE, 1140-R2SIDE, 1140+R2SIDE, 5, 0, 0, 0,0, 0,0}, /* '2' */
{FT_DUALTONE,  900-R2SIDE,  900+R2SIDE, 1020-R2SIDE, 1020+R2SIDE, 5, 0, 0, 0,0, 0,0}, /* '3' */
{FT_DUALTONE,  780-R2SIDE,  780+R2SIDE, 1140-R2SIDE, 1140+R2SIDE, 5, 0, 0, 0,0, 0,0}, /* '4' */
{FT_DUALTONE,  780-R2SIDE,  780+R2SIDE, 1020-R2SIDE, 1020+R2SIDE, 5, 0, 0, 0,0, 0,0}, /* '5' */
{FT_DUALTONE,  780-R2SIDE,  780+R2SIDE,  900-R2SIDE,  900+R2SIDE, 5, 0, 0, 0,0, 0,0}, /* '6' */
{FT_DUALTONE,  660-R2SIDE,  660+R2SIDE, 1140-R2SIDE, 1140+R2SIDE, 5, 0, 0, 0,0, 0,0}, /* '7' */
{FT_DUALTONE,  660-R2SIDE,  660+R2SIDE, 1020-R2SIDE, 1020+R2SIDE, 5, 0, 0, 0,0, 0,0}, /* '8' */
{FT_DUALTONE,  660-R2SIDE,  660+R2SIDE,  900-R2SIDE,  900+R2SIDE, 5, 0, 0, 0,0, 0,0}, /* '9' */
{FT_DUALTONE,  660-R2SIDE,  660+R2SIDE,  780-R2SIDE,  780+R2SIDE, 5, 0, 0, 0,0, 0,0}, /* 'a' */
{FT_DUALTONE,  540-R2SIDE,  540+R2SIDE, 1140-R2SIDE, 1140+R2SIDE, 5, 0, 0, 0,0, 0,0}, /* 'b' */ 
{FT_DUALTONE,  540-R2SIDE,  540+R2SIDE, 1020-R2SIDE, 1020+R2SIDE, 5, 0, 0, 0,0, 0,0}, /* 'c' */
{FT_DUALTONE,  540-R2SIDE,  540+R2SIDE,  900-R2SIDE,  900+R2SIDE, 5, 0, 0, 0,0, 0,0}, /* 'd' */
{FT_DUALTONE,  540-R2SIDE,  540+R2SIDE,  780-R2SIDE,  780+R2SIDE, 5, 0, 0, 0,0, 0,0}, /* 'e' */
{FT_DUALTONE,  540-R2SIDE,  540+R2SIDE,  660-R2SIDE,  660+R2SIDE, 5, 0, 0, 0,0, 0,0}, /* 'f' */
#endif
#define DISCNT_FTEMP 43
{FT_DUALTONE+FT_DIALTONE,  360,  410,  430,  440, 1, 0, 0, 0,0, 0,0}, /* Dm3_DISCONNECT */
#define JP_FTEMP 44
{ 0, 380,420,  0, 0,    2,     0, 0,    0,0, 0,0}, /* Japanese tones */
{0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0} /* A NULL ENTRY */
};   


/*
 * This is the tone template definitions which is the top of the food chain.
 * Tone templates join a Signal ID to the tone definition. see sd_defs.doc.
 * Caution - The '# of Tone segments' column is a 0 based number, 0 = 1 tone
 * segment.
 */

ToneSignalTmplt   qToneSignalTmplts[SD_MAX_TONES];
FixedToneSignalTmplt   qStaticToneSignalTmplts[]= {
 /*
  * Signal Label  Label Rep      # of        Tone 
  *   ID          Size  Count    Tone        Segment Definition
  *                              Segments
  */

#define  TN_BASE  1 /* The start of the DTMF in the tone definitions */
   {0,DTMF1, {"1"},    1,  1,      0,       {0+TN_BASE,  0, 0, 0, 0}},
   {0,DTMF2, {"2"},    1,  1,      0,       {1+TN_BASE,  0, 0, 0, 0}},
   {0,DTMF3, {"3"},    1,  1,      0,       {2+TN_BASE,  0, 0, 0, 0}},
   {0,DTMFA, {"a"},    1,  1,      0,       {3+TN_BASE,  0, 0, 0, 0}},
   {0,DTMF4, {"4"},    1,  1,      0,       {4+TN_BASE,  0, 0, 0, 0}},
   {0,DTMF5, {"5"},    1,  1,      0,       {5+TN_BASE,  0, 0, 0, 0}},
   {0,DTMF6, {"6"},    1,  1,      0,       {6+TN_BASE,  0, 0, 0, 0}},
   {0,DTMFB, {"b"},    1,  1,      0,       {7+TN_BASE,  0, 0, 0, 0}},
   {0,DTMF7, {"7"},    1,  1,      0,       {8+TN_BASE,  0, 0, 0, 0}},
   {0,DTMF8, {"8"},    1,  1,      0,       {9+TN_BASE,  0, 0, 0, 0}},
   {0,DTMF9, {"9"},    1,  1,      0,       {10+TN_BASE, 0, 0, 0, 0}},
   {0,DTMFC, {"c"},    1,  1,      0,       {11+TN_BASE, 0, 0, 0, 0}},
   {0,DTMFSTAR,{"*"},  1,  1,      0,       {12+TN_BASE, 0, 0, 0, 0}},
   {0,DTMF0, {"0"},    1,  1,      0,       {13+TN_BASE, 0, 0, 0, 0}},
   {0,DTMFNUM,{"#"},   1,  1,      0,       {14+TN_BASE, 0, 0, 0, 0}},
   {0,DTMFD, {"d"},    1,  1,      0,       {15+TN_BASE, 0, 0, 0, 0}},

   /*
    * Call Analysis Definitions
    */
#if OLD_SIT
   {0,SIT_NC,{"x"},     1,  1,      2,       {{19+TN_BASE, 10, 50, 0, 5},
                                           {20+TN_BASE,  10, 50, 0, 5},
                                           {18+TN_BASE,   0, 0, 0, 0}}
   },
   {0,SIT_IC,{"x"},     1,  1,      2,       {{16+TN_BASE, 10, 50, 0, 5},
                                           {17+TN_BASE,  10, 50, 0, 5},
                                           {18+TN_BASE,   0, 0, 0, 0}}
   },
   {0,SIT_VC,{"x"},     1,  1,      2,       {{19+TN_BASE, 10, 50, 0, 5},
                                           {17+TN_BASE,  10, 50, 0, 5},
                                           {18+TN_BASE,   0,  0, 0, 0}}
   },

   {0,SIT_RO,{"x"},     1,  1,      2,       {{16+TN_BASE, 10, 50, 0, 5},
                                           {20+TN_BASE,  10, 50, 0, 5},
                                           {18+TN_BASE,   0,  0, 0, 0}}
   },

#else
/* Task 16862. Bellcore spec uses also duration information to distingush
 * different SIT tones.
 */
   {0,SIT_NC,{"x"},     1,  1,      2,       {{19+TN_BASE, Low380, High380, 0, 5},
                                           {20+TN_BASE,  Low380, High380, 0, 5},
                                           {18+TN_BASE,   0, 0, 0, 0}}
   },
   {0,SIT_IC,{"x"},     1,  1,      2,       {{16+TN_BASE, Low274, High274, 0, 5},
                                           {17+TN_BASE, Low274, High274, 0, 5},
                                           {18+TN_BASE,   0, 0, 0, 0}}
   },
   {0,SIT_VC,{"x"},     1,  1,      2,       {{19+TN_BASE, Low380, High380, 0, 5},
                                           {17+TN_BASE,  Low274, High274, 0, 5},
                                           {18+TN_BASE,   0,  0, 0, 0}}
   },

   {0,SIT_RO,{"x"},     1,  1,      2,       {{16+TN_BASE, Low274, High274, 0, 5},
                                           {20+TN_BASE,  Low380, High380, 0, 5},
                                           {18+TN_BASE,   0,  0, 0, 0}}
   },

#endif

   /* The Single Ring Template */
   {0,RINGING1,{"R"}, 1,  1,      0,       {21+TN_BASE, 75, 300, 0, 800}},

   /* The Double Ring Template */
   {0,RINGING2,{"r"}, 1,  1,      1,       {{21+TN_BASE, 20, 100,  20,  100},
                                           {21+TN_BASE, 20, 100,  100, 600}}
    
   },
   {0,BUSY_FAST, {"b"},1, 2,       0,       {22+TN_BASE, 10,  40, 10,  40}},

   {0,BUSY, {"B"},    1,  2,       0,       {22+TN_BASE, 30, 100, 30, 100}},

   {0,FAX_CED1, {""}, 0,  1,       0,       {23+TN_BASE, 10, 0, 0, 0}},
   {0,FAX_CED2, {""}, 0,  1,       0,       {24+TN_BASE, 10, 0, 0, 0}},
   {0,FAX_CNG1, {""}, 0,  1,       0,       {42,         10, 60, 0, 0}},

/* 
 * The start of the MF in the tone definitions 
 */
#define  MF_TN_BASE  25+TN_BASE 
 /* The R1 MF Set
  * Signal Label  Label Rep      # of        Tone 
  *   ID          Size  Count    Tone        Segment Definition
  *                              Segments
  */
   {0,MF1,   {"1"},    1,  1,      0,       {MF_TN_BASE + 0,  0, 0, 0, 0}},
   {0,MF2,   {"2"},    1,  1,      0,       {MF_TN_BASE + 1,  0, 0, 0, 0}},
   {0,MF3,   {"3"},    1,  1,      0,       {MF_TN_BASE + 2,  0, 0, 0, 0}},
   {0,MF4,   {"4"},    1,  1,      0,       {MF_TN_BASE + 3,  0, 0, 0, 0}},
   {0,MF5,   {"5"},    1,  1,      0,       {MF_TN_BASE + 4,  0, 0, 0, 0}},
   {0,MF6,   {"6"},    1,  1,      0,       {MF_TN_BASE + 5,  0, 0, 0, 0}},
   {0,MF7,   {"7"},    1,  1,      0,       {MF_TN_BASE + 6,  0, 0, 0, 0}},
   {0,MF8,   {"8"},    1,  1,      0,       {MF_TN_BASE + 7,  0, 0, 0, 0}},
   {0,MF9,   {"9"},    1,  1,      0,       {MF_TN_BASE + 8,  0, 0, 0, 0}},
   {0,MF0,   {"0"},    1,  1,      0,       {MF_TN_BASE + 9,  0, 0, 0, 0}},
   {0,MFA,   {"a"},    1,  1,      0,       {MF_TN_BASE + 10, 0, 0, 0, 0}},
   {0,MFB,   {"b"},    1,  1,      0,       {MF_TN_BASE + 11, 0, 0, 0, 0}},
   {0,MFC,   {"c"},    1,  1,      0,       {MF_TN_BASE + 12, 0, 0, 0, 0}},
   {0,MFNUM, {"#"},    1,  1,      0,       {MF_TN_BASE + 13, 0, 0, 0, 0}},
   {0,MFSTAR,{"*"},    1,  1,      0,       {MF_TN_BASE + 14, 0, 0, 0, 0}},

   /*
    * These are all the silence on definitions
    */
   {0,SILENCE_ON,{""}, 0,  1,     0,       {0, 1000,  0, 0, 0}},
   {0,SILENCE_1, {""}, 0,  1,     0,       {0,  100,  0, 0, 0}},
   {0,SILENCE_2, {""}, 0,  1,     0,       {0,  200,  0, 0, 0}},
   {0,SILENCE_3, {""}, 0,  1,     0,       {0,  300,  0, 0, 0}},
   {0,SILENCE_5, {""}, 0,  1,     0,       {0,  500,  0, 0, 0}},
   {0,SILENCE_8, {""}, 0,  1,     0,       {0,  800,  0, 0, 0}},
   {0,SILENCE_15,{""}, 0,  1,     0,       {0, 1500,  0, 0, 0}},
   {0,SILENCE_30,{""}, 0,  1,     0,       {0, 3000,  0, 0, 0}},

   /*
    * This is the dial tone definition      Tone on after 100ms
    *                                          V
    */
   {0,DIALTONE1,{"DL1"}, 0,  1,      0,       {41, 10, 0, 0, 0}},
   {0,Dm3_DIALTONE2,{"DL2"}, 0,  1,      0,       {41,100, 0, 0, 0}},

#if 0
#define R2_TN_BASE 17+MF_TN_BASE
   /* The R2 MF Forward Set
    * Signal      Label  Label Rep      # of        Tone 
    *   ID               Size  Count    Tone        Segment Definition
    *                                   Segments
    */
   {0,Dm3_R2MFF1,   {"1"},    1,  1,      0,       {R2_TN_BASE + 0,  0, 0, 0, 0}},
   {0,Dm3_R2MFF2,   {"2"},    1,  1,      0,       {R2_TN_BASE + 1,  0, 0, 0, 0}},
   {0,Dm3_R2MFF3,   {"3"},    1,  1,      0,       {R2_TN_BASE + 2,  0, 0, 0, 0}},
   {0,Dm3_R2MFF4,   {"4"},    1,  1,      0,       {R2_TN_BASE + 3,  0, 0, 0, 0}},
   {0,Dm3_R2MFF5,   {"5"},    1,  1,      0,       {R2_TN_BASE + 4,  0, 0, 0, 0}},
   {0,Dm3_R2MFF6,   {"6"},    1,  1,      0,       {R2_TN_BASE + 5,  0, 0, 0, 0}},
   {0,Dm3_R2MFF7,   {"7"},    1,  1,      0,       {R2_TN_BASE + 6,  0, 0, 0, 0}},
   {0,Dm3_R2MFF8,   {"8"},    1,  1,      0,       {R2_TN_BASE + 7,  0, 0, 0, 0}},
   {0,Dm3_R2MFF9,   {"9"},    1,  1,      0,       {R2_TN_BASE + 8,  0, 0, 0, 0}},
   {0,Dm3_R2MFFA,   {"a"},    1,  1,      0,       {R2_TN_BASE + 9,  0, 0, 0, 0}},
   {0,Dm3_R2MFFB,   {"b"},    1,  1,      0,       {R2_TN_BASE + 10, 0, 0, 0, 0}},
   {0,Dm3_R2MFFC,   {"c"},    1,  1,      0,       {R2_TN_BASE + 11, 0, 0, 0, 0}},
   {0,Dm3_R2MFFD,   {"d"},    1,  1,      0,       {R2_TN_BASE + 12, 0, 0, 0, 0}},
   {0,Dm3_R2MFFE,   {"e"},    1,  1,      0,       {R2_TN_BASE + 13, 0, 0, 0, 0}},
   {0,Dm3_R2MFFF,   {"f"},    1,  1,      0,       {R2_TN_BASE + 14, 0, 0, 0, 0}},
   /* The R2 MF Backward Set
    * Signal      Label  Label Rep      # of        Tone 
    *   ID               Size  Count    Tone        Segment Definition
    *                                   Segments
    */
   {0,Dm3_R2MFB1,   {"1"},    1,  1,      0,       {R2_TN_BASE + 15,  0, 0, 0, 0}},
   {0,Dm3_R2MFB2,   {"2"},    1,  1,      0,       {R2_TN_BASE + 16,  0, 0, 0, 0}},
   {0,Dm3_R2MFB3,   {"3"},    1,  1,      0,       {R2_TN_BASE + 17,  0, 0, 0, 0}},
   {0,Dm3_R2MFB4,   {"4"},    1,  1,      0,       {R2_TN_BASE + 18,  0, 0, 0, 0}},
   {0,Dm3_R2MFB5,   {"5"},    1,  1,      0,       {R2_TN_BASE + 19,  0, 0, 0, 0}},
   {0,Dm3_R2MFB6,   {"6"},    1,  1,      0,       {R2_TN_BASE + 20,  0, 0, 0, 0}},
   {0,Dm3_R2MFB7,   {"7"},    1,  1,      0,       {R2_TN_BASE + 21,  0, 0, 0, 0}},
   {0,Dm3_R2MFB8,   {"8"},    1,  1,      0,       {R2_TN_BASE + 22,  0, 0, 0, 0}},
   {0,Dm3_R2MFB9,   {"9"},    1,  1,      0,       {R2_TN_BASE + 23,  0, 0, 0, 0}},
   {0,Dm3_R2MFBA,   {"a"},    1,  1,      0,       {R2_TN_BASE + 24,  0, 0, 0, 0}},
   {0,Dm3_R2MFBB,   {"b"},    1,  1,      0,       {R2_TN_BASE + 25,  0, 0, 0, 0}},
   {0,Dm3_R2MFBC,   {"c"},    1,  1,      0,       {R2_TN_BASE + 26,  0, 0, 0, 0}},
   {0,Dm3_R2MFBD,   {"d"},    1,  1,      0,       {R2_TN_BASE + 27,  0, 0, 0, 0}},
   {0,Dm3_R2MFBE,   {"e"},    1,  1,      0,       {R2_TN_BASE + 28,  0, 0, 0, 0}},
   {0,Dm3_R2MFBF,   {"f"},    1,  1,      0,       {R2_TN_BASE + 29,  0, 0, 0, 0}},
#endif

   {0,Dm3_DISCONNECT,   {"d"},    1,  1,      0,       {DISCNT_FTEMP/*73*/,  30, 60, 30, 60}},
   /* allowing for timing deviations of 20% */
   {0,JP_RINGING1,      {"E"},    1,  1,      0,       {JP_FTEMP/*74*/,  80, 120, 160, 360}},
   {0,JP_BUSY,          {"B"},    1,  1,      0,       {JP_FTEMP/*74*/,  30, 60, 30, 60}},
   {0,JP_DIALTONE1,     {"D"},    1,  1,      0,       {JP_FTEMP/*74*/,  50,  0,  0,  0}},
   {0,JP_SIT_ANY,       {"S"},     1,  1,      2,       {{JP_FTEMP/*74*/, 0, 12, 0, 12},
                                                         {JP_FTEMP/*74*/, 0, 12, 0, 12},
                                                         {JP_FTEMP/*74*/, 0, 12, 0, 12}}
   },
   /*
    * TASK-15524:PT27713 - Added SILENCE_DEB for DE_SILON,DE_SILOFF R4 events
    */
   {0,SILENCE_DEB,{""}, 0,  1,     0,       {0, 0,  0, 0, 0}},
   {0,0, {""}, 0, 0, 0, {0,0,0,0,0} } /* A NULL ENTRY */
};


#ifndef NEWPVD
   PvdSignalTmplt qPvdSignalTmplts[]={
      /*
       * First Template Definition
       */
      {PVD1,   /* Signal ID*/
       50,     /* minSnr */
       600,    /* maxSnr */
       2,      /* maxPk */
       5,      /* maxRing */
       10000,  /* ringThreshold */
       8,      /* windowSize */
       30,     /* VThreshold */
       340,    /* rbLow */
       510},   /* rbHigh */

      /*
       * Last template definition NULL
       */
      {0,0,0,0,0,0,0,0}
   };
   #define  NUM_OF_PVD_SIGNAL_TMPLTS ((sizeof(qPvdSignalTmplts) \
                                       / sizeof(qPvdSignalTmplts[1]))-1)
   int      qNumOfPvdSignalTmplts=NUM_OF_PVD_SIGNAL_TMPLTS;
#else /* NEWPVD, Baowen Chen */
   FixedPvdSignalTmplt qStaticPvdSignalTmplts[]={
      /*
       * First Template Definition
       */
      {PVD1,   /* Signal ID*/
       50,     /* minSnr */
       600,    /* maxSnr */
       2,      /* maxPk */
       5,      /* maxRing */
       10000,  /* ringThreshold */
       8,      /* windowSize */
       30,     /* VThreshold */
       340,    /* rbLow */
       510},   /* rbHigh */

      /*
       * Last template definition NULL
       */
      {0,0,0,0,0,0,0,0}
   };
   #define  NUM_OF_PVD_SIGNAL_TMPLTS ((sizeof(qStaticPvdSignalTmplts) \
                                       / sizeof(qStaticPvdSignalTmplts[1]))-1)
   int      qNumOfPvdSignalTmplts=NUM_OF_PVD_SIGNAL_TMPLTS;/* just for init stage.*/

   PvdSignalTmplt qPvdSignalTmplts[MAX_PVD];
/*
   BITStorage     qPvdFreeStatus[BIT_MAXSTORAGE(MAX_PVD)] = {BIT_BITMASK};
   int            qNumOfPvdSignalTmplts = 0; 
*/
#endif /* NEWPVD */


#ifndef NEWPAMD
   PamdQualTmplt qPamdQualTmplts[]={
      /*
       * First Template Definition
       */
      {190,    /* minRing */
       125,    /* maxAnsiz1 */
       50,     /* maxAnsiz2 */
       220,    /* maxAnsiz3 */
       22,     /* loHiss */
       16,     /* hiHiss */
       5,      /* bhParm */
       80,     /* cvThresh1 */
       165,    /* cvThresh2 */
       390,    /* maxCvThresh */
       2,      /* nMaxBroad */
       65,     /* nMaxErg */
       30,     /* maxSilence */
       25,     /* voiceThresh */
       30,     /* silenceThresh */
       0,      /* rjFbandLow */
       0       /* rgFbandHigh */
      },
  
      {0,0,0,0,0,0,0,0,0,0,0,0,0,0}
   };

   PamdSignalTmplt  qPamdSignalTmplts[]={
      /*
       * This is the default template set for PAMD.
       */
      {Dm3_PAMD_DEFAULT, /* id -  Safest and slowest. */
       0},           /* Pamd Qualification template # */
      {Dm3_PAMD_MEDIUM,  /* id -  Complete algorithm */
       0},
      {Dm3_PAMD_QUICK,   /* id -  Quick decistion only fastest and most unreliable */
       0},

      {0,0} /* NULL Entry */
   };

   #define  NUM_OF_PAMD_SIGNAL_TMPLTS ((sizeof(qPamdSignalTmplts) \
                                     / sizeof(qPamdSignalTmplts[1]))-1)
   int      qNumOfPamdSignalTmplts=NUM_OF_PAMD_SIGNAL_TMPLTS;

   #define  NUM_OF_PAMD_QUAL_TMPLTS ((sizeof(qPamdQualTmplts) \
                                   / sizeof(qPamdQualTmplts[1]))-1)
   int      qNumOfPamdQualTmplts=NUM_OF_PAMD_QUAL_TMPLTS;

#else /* NEWPAMD */

   PamdQualTmplt qStaticPamdQualTmplts[]={
      /*
       * First Template Definition
       */
      {190,    /* minRing */
       125,    /* maxAnsiz1 */
       50,     /* maxAnsiz2 */
       220,    /* maxAnsiz3 */
       22,     /* loHiss */
       16,     /* hiHiss */
       5,      /* bhParm */
       80,     /* cvThresh1 */
       165,    /* cvThresh2 */
       390,    /* maxCvThresh */
       2,      /* nMaxBroad */
       65,     /* nMaxErg */
       30,     /* maxSilence */
       25,     /* voiceThresh */
       30,     /* silenceThresh */
       0,      /* rjFbandLow */
       0       /* rgFbandHigh */
      },
  
      {0,0,0,0,0,0,0,0,0,0,0,0,0,0}
   };

   FixedPamdSignalTmplt  qStaticPamdSignalTmplts[]={
      /*
       * This is the default template set for PAMD.
       */
      {Dm3_PAMD_DEFAULT, /* id -  Safest and slowest. */
       0},           /* Pamd Qualification template # */
      {Dm3_PAMD_MEDIUM,  /* id -  Complete algorithm */
       0},
      {Dm3_PAMD_QUICK,   /* id -  Quick decistion only fastest and most unreliable */
       0},

      {0,0} /* NULL Entry */
   };

   #define  NUM_OF_PAMD_SIGNAL_TMPLTS ((sizeof(qStaticPamdSignalTmplts) \
                                     / sizeof(qStaticPamdSignalTmplts[1]))-1)
   int      qNumOfPamdSignalTmplts=NUM_OF_PAMD_SIGNAL_TMPLTS;/* for init
                                                                phase only.*/

   #define  NUM_OF_PAMD_QUAL_TMPLTS ((sizeof(qStaticPamdQualTmplts) \
                                   / sizeof(qStaticPamdQualTmplts[1]))-1)
   int      qNumOfPamdQualTmplts=NUM_OF_PAMD_QUAL_TMPLTS;/* for init phase only.*/


   PamdSignalTmplt  qPamdSignalTmplts[MAX_PAMD];
   PamdQualTmplt    qPamdQualTmplts[MAX_PAMD_QT];

#endif /* NEWPAMD */


#define  NUM_OF_GTD_TMPLTS ((sizeof(qStaticGtdTemplates)/sizeof(qStaticGtdTemplates[1]))-1)
int      qNumOfGtdTmplts=NUM_OF_GTD_TMPLTS;
      
#define  NUM_OF_TONE_SIGNAL_TMPLTS ((sizeof(qStaticToneSignalTmplts) \
                                     / sizeof(qStaticToneSignalTmplts[1]))-1)

int      qNumOfToneSignalTmplts=NUM_OF_TONE_SIGNAL_TMPLTS;/* for fixed tones only.*/

#else /* _SIGINIT_C__ */ 

extern   FTEMP qGtdTemplates[];
extern   FTEMP qStaticGtdTemplates[];
extern   int   qNumOfGtdTmplts;
extern   int   qDtmfShow;
extern   int   qNumOfToneSignalTmplts;
extern   ToneSignalTmplt   qToneSignalTmplts[]; /* Tone signal templates */
extern   FixedToneSignalTmplt   qStaticToneSignalTmplts[];

/* Qualification Related */
extern   QTEMP qt[];

/* Pvd Related */
extern PvdSignalTmplt qPvdSignalTmplts[];
extern int qNumOfPvdSignalTmplts;
/* #ifdef NEWPVD */
extern FixedPvdSignalTmplt qStaticPvdSignalTmplts[];

/* #endif */

/* Pamd Related */
extern   PamdQualTmplt     qPamdQualTmplts[];   /* PAMD Qualification
                                                   templates */
extern   PamdSignalTmplt   qPamdSignalTmplts[]; /* PAMD signal templates*/
extern   int   qNumOfPamdSignalTmplts;
extern   int   qNumOfPamdQualTmplts;
/* #ifdef NEWPAMD */
extern   FixedPamdSignalTmplt qStaticPamdSignalTmplts[]; 
extern   PamdQualTmplt     qStaticPamdQualTmplts[];

/* #endif*/ /*NEWPAMD */

#endif /* _SIGINIT_C__ */


#endif /*  _DTMFDBG_H__ */
