/******************************************************************************
* M3GLIB.H:
*
* M3G API:  Control of 3G-324M transporting of audio and video functionality.
*******************************************************************************
* DIALOGIC CONFIDENTIAL	
* Copyright 2007 Dialogic Corporation All Rights Reserved.
* 
* The source code contained or described herein and all documents related to 
* the source code ("Material") are owned by Dialogic Corporation or its suppliers
* or licensors.  Title to the Material remains with Dialogic Corporation or its 
* suppliers and licensors.  The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors.  The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions.  No part of the Material may be used, copied, reproduced, 
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual 
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise.  Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
* 
* Unless otherwise agreed by Dialogic in writing, you may not remove or alter this
* notice or any other notice embedded in Materials by Dialogic or Dialogic's 
* suppliers or licensors in any way.
******************************************************************************/

#ifndef _M3GLIB_H_
#define _M3GLIB_H_

#include "m3gevts.h"
#include "m3gerrs.h"
#include "srllib.h"


/* Pack all structures on byte boundaries */
#pragma pack(1)


#ifdef _WIN32
#define M3GLIB_API __declspec(dllexport)
#else
#define M3GLIB_API
#endif

#if ((!defined EV_SYNC) && (!defined(EV_ASYNC)))
#define EV_SYNC   0x0000
#define EV_ASYNC  0x8000
#endif

typedef int M3G_RETVAL;

#define M3G_SUCCESS  0
#define M3G_ERROR    -1

#define M3G_LIBRARY_VERSION_2_0        0x0200
#define M3G_LIBRARY_VERSION_1_0        1
#define M3G_LIBRARY_VERSION            M3G_LIBRARY_VERSION_2_0

#define MAX_CAPABILITIES_PER_DEVTYPE   20
#define MAX_NUM_DIGITS                 30
#define OCTET_STRING_SIZE              255
#define OBJECTID_SIZE                  255

/* initialization structure used in m3g_Start(): */
typedef struct
{
   unsigned int   version;
   unsigned short numVirtBoards;  /* must be 1 this release */
   unsigned short numEndpoints;   /* 0 := allocate maximum number of 3G licenses */
} M3G_START_STRUCT;

/* in-line macro used to initialize M3G_START_STRUCT with defaults of current release: */
static __inline void INIT_M3G_START_STRUCT(M3G_START_STRUCT *pLibStart)
{
   pLibStart->version = M3G_LIBRARY_VERSION;
   pLibStart->numVirtBoards = 1;
   pLibStart->numEndpoints = 0;
}

/* structure M3G_OPEN_INFO is unused in this release */
#define M3G_OPEN_INFO  void

/* parameter IDs for m3g_Set/GetParm(): */
typedef enum
{
   M3G_E_PRM_RELAY_DIGIT_TO_MEDIA_DEV = 0,
   M3G_E_PRM_RELAY_DIGIT_TO_H245UII,
   M3G_E_PRM_RELAY_FASTUPDATE_TO_MEDIA_DEV,
   M3G_E_PRM_RELAY_FASTUPDATE_TO_H245,
   M3G_E_PRM_RELAY_TEMPORALSPATIALTRADEOFF_TO_MEDIA_DEV,
   M3G_E_PRM_RELAY_TEMPORALSPATIALTRADEOFF_TO_H245,
   M3G_E_PRM_RELAY_VIDEOFREEZE_TO_MEDIA_DEV,
   M3G_E_PRM_RELAY_VIDEOFREEZE_TO_H245,
   M3G_E_PRM_SKEWINDICATION,
   M3G_E_PRM_AUDIOVISUALSYNC,
   M3G_E_PRM_H245_TERMINAL_TYPE,
   M3G_E_PRM_MAX_CCSRL_SEGMENT,
   M3G_E_PRM_RETRANSMIT_ON_IDLE,
   M3G_E_PRM_AMR_PAYLOAD_FORMAT,
   M3G_E_PRM_BOARD_DEBUG,                                 /* for internal use only */
   M3G_E_PRM_ALL_INSTANCES_DEBUG,                         /* for internal use only */
   M3G_E_PRM_INSTANCE_DEBUG,                              /* for internal use only */
   M3G_E_PRM_STACK_DEBUG_LEVEL,                           /* for internal use only */
   M3G_E_PRM_STACK_DEBUG_MODULES,                         /* for internal use only */
   M3G_E_PRM_MPEG4_TX_DCI,
   M3G_E_PRM_MPEG4_RX_DCI,
   M3G_E_PRM_TX_SKEW_ADJUSTMENT,
   M3G_E_PRM_RX_SKEW_ADJUSTMENT,
   M3G_E_PRM_VIDEO_BIT_RATE,
   M3G_E_PRM_VIDEO_FRAME_RATE,
   M3G_E_PRM_EARLY_MES,
   M3G_E_PRM_AUTO_VFU_PERIOD,
   M3G_E_PRM_H223_SYNC_TIMER,
   M3G_E_PRM_TX_VENDORID, 
   M3G_E_PRM_H264_TX_DCI,
   M3G_E_PRM_IFRAME_NOTIFY_CONTROL_MASK
} M3G_E_PRM_TYPE;

typedef enum
{
   M3G_FALSE = 0,
   M3G_TRUE  = 1
} M3G_BOOL;

typedef unsigned char              M3G_H245_TERMINAL_TYPE;
typedef unsigned short             M3G_MAX_CCSRL_SEGMENT_SIZE;
typedef unsigned char              M3G_AMR_PAYLOAD_FORMAT;
typedef unsigned int               M3G_BITMASK;
typedef int                        M3G_SKEW_ADJUSTMENT;
typedef unsigned int               M3G_VIDEO_BIT_RATE;
typedef unsigned int               M3G_VIDEO_FRAME_RATE;
typedef unsigned short             M3G_VFU_TIMER_VAL;
typedef unsigned short             M3G_H223_SYNC_TIMER_VAL;
typedef unsigned char              M3G_IFRAME_NOTIFY_CONTROL_MASK;

typedef struct
{
   unsigned char             length;
   unsigned char             octet[OCTET_STRING_SIZE];
} M3G_OCTET_STRING;


/* Iframe bit masks (me Notify Control Masks M3G_E_PRM_IFRAME_NOTIFY_CONTROL_MASK): */
#define IFRAME_NOTIFY_RX   ((M3G_IFRAME_NOTIFY_CONTROL_MASK) 1)
#define IFRAME_NOTIFY_TX   ((M3G_IFRAME_NOTIFY_CONTROL_MASK) 2)
#define IFRAME_NOTIFY_DCI_CHANGE_RX   ((M3G_IFRAME_NOTIFY_CONTROL_MASK) 4)
#define IFRAME_NOTIFY_DCI_CHANGE_TX   ((M3G_IFRAME_NOTIFY_CONTROL_MASK) 8)


/* AMR payload formats (M3G_E_PRM_AMR_PAYLOAD_FORMAT): */
#define AMR_PAYLOAD_BW_EFFICIENT   ((M3G_AMR_PAYLOAD_FORMAT) 1)
#define AMR_PAYLOAD_OCTET_ALIGNED  ((M3G_AMR_PAYLOAD_FORMAT) 2)

/* video frame rates (M3G_E_PRM_VIDEO_FRAME_RATE): */
#define VIDEO_FRAME_RATE_6_FPS     ((M3G_VIDEO_FRAME_RATE) 0)
#define VIDEO_FRAME_RATE_10_FPS    ((M3G_VIDEO_FRAME_RATE) 1)
#define VIDEO_FRAME_RATE_15_FPS    ((M3G_VIDEO_FRAME_RATE) 2)

/* structure for specifying/retrieving parameter settings via m3g_SetParm/m3g_GetParm(): */
typedef struct
{
   unsigned int       version;                    /* set to M3G_LIBRARY_VERSION */
   M3G_E_PRM_TYPE     parameterType;
   union
   {
       M3G_BOOL                   boolParam;
       M3G_H245_TERMINAL_TYPE     h245TerminalType;
       M3G_MAX_CCSRL_SEGMENT_SIZE maxCCSRLSegmentSize;
       M3G_AMR_PAYLOAD_FORMAT     amrPayloadFormat;
       M3G_BITMASK                bitmask;        /* for internal use only */
       M3G_OCTET_STRING           octetString;
       M3G_SKEW_ADJUSTMENT        skewAdjustment;
       M3G_VIDEO_BIT_RATE         videoBitRate;
       M3G_VIDEO_FRAME_RATE       videoFrameRate;
       M3G_VFU_TIMER_VAL          vfuTimer;
       M3G_H223_SYNC_TIMER_VAL    h223SyncTimer;
       M3G_IFRAME_NOTIFY_CONTROL_MASK  iframeNotifyControlMask;
   }   parmValue;
} M3G_PARM_INFO;

/* in-line macro used to initialize INIT_M3G_PARM_INFO with defaults of current release.
 * This macro only initializes the version element to a valid value. All other parameter 
 * elements must be explicitly intialized via user code.
 */
static __inline void INIT_M3G_PARM_INFO(M3G_PARM_INFO *pParmInfo)
{
   pParmInfo->version = M3G_LIBRARY_VERSION;
   /* parameterType and parmValue elements must be set via user code: */
   pParmInfo->parameterType = (M3G_E_PRM_TYPE)-1;
   pParmInfo->parmValue.bitmask = 0;
}

/* H.223 multiplex level: */
typedef enum
{
   M3G_E_H223_MUX_LEVEL0 = 0,
   M3G_E_H223_MUX_LEVEL1,
   M3G_E_H223_MUX_LEVEL1_DBL_FLAG,
   M3G_E_H223_MUX_LEVEL2,
   M3G_E_H223_MUX_LEVEL2_OPT_HDR,
   M3G_E_H223_MUX_LEVEL3,
   M3G_E_H223_MUX_LEVEL3_OPT_HDR,
} M3G_E_H223_MUX_LVL_TYPE;

/* H.245 Master/Slave: */
typedef enum
{
   M3G_E_H245_MASTER = 0,
   M3G_E_H245_SLAVE,
   M3G_E_H245_IDENTICAL_MSD_NUMBERS
} M3G_E_H245_MSD_RESULT;

/* capability types: */
typedef enum
{
   M3G_E_H223_CAPABILITY = 0,
   M3G_E_AUDIO_CAPABILITY,
   M3G_E_VIDEO_CAPABILITY,
} M3G_E_CAPABILITY;

/* H.223 multiplex capabilities: */
typedef struct
{
   unsigned int          version;                    /* set to M3G_LIBRARY_VERSION */
   unsigned short        adaptationLayerMedia;
   unsigned short        ALxM_AnnexC_Media;
   unsigned short        maxAL2SDUSize;
   unsigned short        maxAL3SDUSize;
   M3G_BOOL              frameH223AnnexA;
   M3G_BOOL              frameH223DoubleFlag;
   M3G_BOOL              frameAnnexB;
   M3G_BOOL              frameAnnexBWithHead;
   unsigned short        maxAL1MPDUSize;
   unsigned short        maxAL2MPDUSize;
   unsigned short        maxAL3MPDUSize;
   M3G_BOOL              rsCodeCapability;
   M3G_BOOL              mobileOpXmitCap;
   unsigned short        bitRate;
}  M3G_H223_CAPABILITY;

/* values used in adaptationLayerMedia bitmask to indicate adaptation layers 
 * supported per audio and video device:
 */
#define M3G_AUDIO_AL1                      0x001
#define M3G_AUDIO_AL2                      0x002
#define M3G_AUDIO_AL3                      0x004
#define M3G_VIDEO_AL1                      0x008
#define M3G_VIDEO_AL2                      0x010
#define M3G_VIDEO_AL3                      0x020

/* values used in ALxM_AnnexC_Media bitmask to indicate H.223 AnnexC adoption
 * layer media support:
 */
#define M3G_NO_ANNEXC                      0x00
#define M3G_AUDIO_AL1M                     0x01
#define M3G_AUDIO_AL2M                     0x02
#define M3G_AUDIO_AL3M                     0x04
#define M3G_VIDEO_AL1M                     0x08
#define M3G_VIDEO_AL2M                     0x10
#define M3G_VIDEO_AL3M                     0x20

/* direction (in perspective of local endpoint): */
typedef enum
{
  M3G_E_IDLE = 0,
  M3G_E_TX,
  M3G_E_RX,
  M3G_E_TXRX,
} M3G_E_DIRECTION;

/* audio codec type: */
typedef enum
{
  M3G_E_G7231 = 0,
  M3G_E_GSM_AMR_NB,
} M3G_E_AUDIO_TYPE;

/* G.723.1 options: */
typedef struct
{
   M3G_BOOL            silenceSup;
} M3G_G7231_OPTIONS;

/* AMR options: */
typedef struct
{
   M3G_BOOL            amrComfortNoise;  /* H.245 RequestMode only (unsupported) */
} M3G_AMR_OPTIONS;

/* audio options union: */
typedef union
{
   M3G_G7231_OPTIONS         g7231;
   M3G_AMR_OPTIONS           amr;
} M3G_AUDIO_OPTIONS;

/* audio capabilities: */
typedef struct
{
   unsigned int        version;               /* set to M3G_LIBRARY_VERSION */
   unsigned short      tableEntryNumber;
   M3G_E_DIRECTION     direction;
   M3G_E_AUDIO_TYPE    coderType;
   unsigned char       maxFramesPerSDU;
   M3G_AUDIO_OPTIONS   options;
}  M3G_AUDIO_CAPABILITY;

/* video codec type: */
typedef enum
{
  M3G_E_H263 = 0,
  M3G_E_MPEG4,
  M3G_E_H264
} M3G_E_VIDEO_TYPE;

/* H.263 options: */
typedef struct
{
   unsigned short      bppMaxKb;
   unsigned char       sqcifMPI;
   unsigned char       qcifMPI;
   M3G_BOOL            unrestrictedVector;
   M3G_BOOL            arithmeticCoding;
   M3G_BOOL            advancedPrediction;
   M3G_BOOL            pbFrames;
   M3G_BOOL            temporalSpatialTradeoffCap;
   M3G_BOOL            errorCompensation;
} M3G_H263_OPTIONS;

/* values used in MPEG-4 profileAndLevel: */
#define M3G_MPEG4_SIMPLE_PROFILE_LEVEL0          0x08

/* MPEG-4 options: */
typedef struct
{
   unsigned char       profileAndLevel;
   unsigned char       object;
   unsigned char       decoderConfigLength;
   unsigned char       decoderConfigInfo[OCTET_STRING_SIZE];  /* used in local and 
                                                               * remote H.245 OLC 
                                                               * Request messages only;
                                                               * ignored in Terminal
                                                               * Capability Set messages
                                                               */
   M3G_BOOL            visualBackChannel;
} M3G_MPEG4_OPTIONS;


#define M3G_H264_SIMPLE_PROFILE        64
#define M3G_H264_LEVEL_1               15

/* H264 options: */
typedef struct
{
   unsigned char       profile;                                /* H264 Profile */
   unsigned char       level;                                  /* H264 Level   */
   unsigned char       decoderConfigLength;                    /* H264 DCI length */
   unsigned char       decoderConfigInfo[OCTET_STRING_SIZE];   /* used in local and  remote H.245 OLC 
                                                                           * May be present exactly once for Logical Channel Signalling
                                                                           */
   unsigned char       h264SignalingMask;                      /* used to signal parameters described below  in local and  remote H.245 TCS and OLC - optional */
   unsigned char       profileIOP;                             /* used in local and  remote H.245 TCS - optional */
} M3G_H264_OPTIONS;

/* Bits in h264SignalingMask are assigned as follows :*/
#define M3G_H264_ACCEPT_REDUNDANT_SLICES 0x01
#define M3G_H264_NAL_ALIGNED_MODE 0x02



/* video options union: */
typedef union
{
   M3G_H263_OPTIONS         h263;
   M3G_MPEG4_OPTIONS        mpeg4;
   M3G_H264_OPTIONS         h264;
} M3G_VIDEO_OPTIONS;

/* video capabilities: */
typedef struct
{
   unsigned int        version;                    /* set to M3G_LIBRARY_VERSION */
   unsigned short      tableEntryNumber;
   M3G_E_DIRECTION     direction;
   M3G_E_VIDEO_TYPE    coderType;
   unsigned int        maxBitRate;
   M3G_VIDEO_OPTIONS   options;
}  M3G_VIDEO_CAPABILITY;

/* capabilities union: */
typedef union
{
   M3G_H223_CAPABILITY       h223Capability;
   M3G_AUDIO_CAPABILITY      audioCapability;
   M3G_VIDEO_CAPABILITY      videoCapability;
} M3G_CAPABILITY;

/* capability structure: */
typedef struct
{
   unsigned int       version;                    /* set to M3G_LIBRARY_VERSION */
   unsigned short     numCaps;
   M3G_E_CAPABILITY   capabilityType;
   M3G_CAPABILITY     capability[MAX_CAPABILITIES_PER_DEVTYPE];
} M3G_CAPS_LIST;

/* simultaneousCapabilities structure: */
typedef struct
{
   unsigned int                        version;   /* set to M3G_LIBRARY_VERSION */
   M3G_CAPS_LIST *                     pH223Capabilities;
   M3G_CAPS_LIST *                     pAudioCapabilities;
   M3G_CAPS_LIST *                     pVideoCapabilities;
} M3G_SIMULTANEOUS_CAP_SET;

/* H.223 session parameters: */
typedef struct
{
   unsigned int                     version;      /* set to M3G_LIBRARY_VERSION */
   M3G_E_H223_MUX_LVL_TYPE          defaultH223MuxLevel;
   unsigned int                     maxALSDUSize;
   M3G_BOOL                         isWNSRPEnabled;
   M3G_BOOL                         isMultipleMsgsPerPdu;
   M3G_BOOL                         isMONAEnabled;
} M3G_H223_SESSION;


/* Inband DCI attached to an IFRAME Indication */
typedef struct
{
   M3G_E_DIRECTION     direction;                              /* Direction the Iframe was received from */
   unsigned int        iframeSize;                             /* Size of the Iframe in bytes */
   M3G_BOOL            dciChanged;                             /* Indicates whether the DCI changed */
   unsigned char       decoderConfigLength;                    /* H264 DCI length */
   unsigned char       decoderConfigInfo[OCTET_STRING_SIZE];   /* inband H264 DCI*/
} M3G_IFRAME_DATA;

/* in-line macro used to initialize M3G_H223_SESSION to default settings of current release: */
static __inline void INIT_M3G_H223_SESSION(M3G_H223_SESSION *pH223Params)
{
   pH223Params->version = M3G_LIBRARY_VERSION;
   pH223Params->defaultH223MuxLevel = M3G_E_H223_MUX_LEVEL2;
   pH223Params->maxALSDUSize = 200;
   pH223Params->isWNSRPEnabled = M3G_TRUE;
   pH223Params->isMultipleMsgsPerPdu = M3G_TRUE;
   pH223Params->isMONAEnabled = M3G_FALSE;
}

/* adaptation layer type: */
typedef enum
{
   M3G_E_AL1_FRAMED = 0,
   M3G_E_AL1_UNFRAMED,
   M3G_E_AL2_WITH_SEQ_NUMS,
   M3G_E_AL2_WITHOUT_SEQ_NUMS,
   M3G_E_AL3,
   M3G_E_AL1M_FRAMED,
   M3G_E_AL1M_UNFRAMED,
   M3G_E_AL2M,
   M3G_E_AL3M
} M3G_E_ADAPTATION_LYR_TYPE;

/* AL1M, AL2M or AL3M header format: */
typedef enum
{
   M3G_E_ALx_HEADER_SEBCH16 = 0,
   M3G_E_ALx_HEADER_GOLAY24
} M3G_E_ALxM_HEADER_TYPE;

/* AL1M and AL3M CRC lengths: */
typedef enum
{
   M3G_E_AL_CRC_4 = 0,
   M3G_E_AL_CRC_12,
   M3G_E_AL_CRC_20,
   M3G_E_AL_CRC_28,
   M3G_E_AL_CRC_8,
   M3G_E_AL_CRC_16,
   M3G_E_AL_CRC_32,
   M3G_E_AL_CRC_NONE
} M3G_E_ALxM_CRC_TYPE;

/* AL1M and AL3M ARQ mode: */
typedef enum
{
   M3G_E_AL_ARQ_NONE = 0,
   M3G_E_AL_ARQ_TYPEI_FINITE,
   M3G_E_AL_ARQ_TYPEI_INFINITE,
   M3G_E_AL_ARQ_TYPEII_FINITE,
   M3G_E_AL_ARQ_TYPEII_INFINITE,
} M3G_E_ADAPTATION_LYR_ARQ_TYPE;

/* OLC H.223 multiplex parameters structure: */
typedef struct
{
   unsigned int                    version;     /*set to M3G_LIBRARY_VERSION*/
   M3G_E_ADAPTATION_LYR_TYPE       adaptationLayerType;
   M3G_BOOL                        segmentable;
   unsigned char                   AL3_ControlFieldSize;
   unsigned int                    AL3_SendBufferSize;
   M3G_E_ALxM_HEADER_TYPE          ALxM_HeaderFormat;
   M3G_BOOL                        ALxM_ALPDUInterleaving;
   M3G_E_ALxM_CRC_TYPE             ALxM_CRCType;
   M3G_E_ADAPTATION_LYR_ARQ_TYPE   ALxM_ARQType;
   unsigned char                   ALxM_ARQMaxNumRetrans;
   unsigned int                    ALxM_ARQSendBufferSize;
   M3G_BOOL                        AL1M_SplitSDU;
   unsigned char                   ALxM_RCPCCodeRate;
} M3G_H223_LC_PARAMS;

/* in-line macro used to initialize M3G_H223_LC_PARAMS to default settings of current release.
 * The segmentable element is typically updated depending upon media type as audio is 
 * typically non-segmentable whereas video is segmentable.
 */
static __inline void INIT_M3G_H223_LC_PARAMS(M3G_H223_LC_PARAMS *pH223LCParameters)
{
   pH223LCParameters->version = M3G_LIBRARY_VERSION;
   pH223LCParameters->adaptationLayerType = M3G_E_AL2_WITH_SEQ_NUMS;
   pH223LCParameters->segmentable = M3G_FALSE;
   pH223LCParameters->AL3_ControlFieldSize = 1;
   pH223LCParameters->AL3_SendBufferSize = 1024;
   pH223LCParameters->ALxM_HeaderFormat = M3G_E_ALx_HEADER_SEBCH16;
   pH223LCParameters->ALxM_ALPDUInterleaving = M3G_FALSE;
   pH223LCParameters->ALxM_CRCType = M3G_E_AL_CRC_4;
   pH223LCParameters->ALxM_ARQType = M3G_E_AL_ARQ_NONE;
   pH223LCParameters->ALxM_ARQMaxNumRetrans = 0;
   pH223LCParameters->ALxM_ARQSendBufferSize = 0;
   pH223LCParameters->AL1M_SplitSDU = M3G_FALSE;
   pH223LCParameters->ALxM_RCPCCodeRate = 8;
}

typedef unsigned short M3G_LOGICAL_CHANNEL_NUMBER;

/* OpenLogicalChannelAck response from remote: */
typedef struct
{
   unsigned int                 version; /*set to M3G_LIBRARY_VERSION*/
   M3G_LOGICAL_CHANNEL_NUMBER   logicalChannelNumber;
   M3G_E_CAPABILITY             capabilityType;
} M3G_REMOTE_OLCACK_RESP;

/* OpenLogicalChannel request from remote: */
typedef struct
{
   unsigned int                 version; /*set to M3G_LIBRARY_VERSION*/
   M3G_LOGICAL_CHANNEL_NUMBER   logicalChannelNumber;
   M3G_H223_LC_PARAMS           h223MultiplexParams;
   M3G_E_CAPABILITY             capabilityType;
   M3G_CAPABILITY               mediaCapability;
} M3G_REMOTE_OLC_REQ;

/* Issue RequestChannelClose using specified reason code */
typedef enum
{
   M3G_E_REQ_CHAN_CLOSE_UNKNOWN = 0,
   M3G_E_REQ_CHAN_CLOSE_NORMAL,
   M3G_E_REQ_CHAN_CLOSE_REOPEN,
   M3G_E_REQ_CHAN_CLOSE_RESERV_FAIL,
   M3G_E_REQ_CHAN_CLOSE_DISCONNECTED
} M3G_E_REQ_CHAN_CLOSE_REASON;

/* CloseLogicalChannel command or RequestChannelCloseRequest received from remote: */
typedef struct
{
   unsigned int                 version; /*set to M3G_LIBRARY_VERSION*/
   M3G_LOGICAL_CHANNEL_NUMBER   logicalChannelNumber;
   M3G_E_REQ_CHAN_CLOSE_REASON  reason;
} M3G_REMOTE_CLOSED_LC;

/* positively acknowledge OpenLogicalChannelAck or reject via OpenLogicalChannelReject using specified cause code */
typedef enum
{
   M3G_E_OLCACK = 0,              /* OLCAck response                                    */
   M3G_E_OLCREJ_UNSPECIFIED,      /* OLCReject cause unspecified                        */
   M3G_E_OLCREJ_UNS_REV_PARM,     /* OLCReject cause unsuitableReverseParameters        */
   M3G_E_OLCREJ_DATATYP_NOT_SUP,  /* OLCReject cause dataTypeNotSupported               */
   M3G_E_OLCREJ_DATATYP_NOT_AVAIL,/* OLCReject cause dataTypeNotAvailable               */
   M3G_E_OLCREJ_DATATYP_AL_UNSUP, /* OLCReject cause dataTypeALCombintationNotSupported */
   M3G_E_OLCREJ_MC_CHAN_NOT_ALWD, /* OLCReject cause multicatChannelNotAllowed          */
   M3G_E_OLCREJ_INSUFF_BW,        /* OLCReject cause insufficientBandwidth              */
   M3G_E_OLCREJ_STACK_FAILED,     /* OLCReject cause separateStackEstablishmentFailed   */
   M3G_E_OLCREJ_INV_SESSIONID,    /* OLCReject cause invalidSessionID                   */
   M3G_E_OLCREJ_MS_CONFLICT,      /* OLCReject cause masterSlaveConflict                */
   M3G_E_OLCREJ_WAIT_COMM_MODE,   /* OLCReject cause waitForCommunicationMode           */
   M3G_E_OLCREJ_INV_DEP_CHAN,     /* OLCReject cause invalidDependentChannel            */
   M3G_E_OLCREJ_REP_FOR_REJ,      /* OLCReject cause replacementForRejected             */
   M3G_E_OLCREJ_SECURITY_DENIED,  /* OLCReject cause securityDenied                     */
   M3G_E_OLCREJ_DISCONNECTED      /* call disconnected before logical channel is 
                                     established - not to be used in m3g_RespondToOLC() */
} M3G_E_OLC_RESP_TYPE;

/* transmitted/received H.245 UserInputIndication message: */
typedef struct
{
   unsigned int                 version; /*set to M3G_LIBRARY_VERSION*/
   unsigned short               numDigits;
   char                         digitBuffer[MAX_NUM_DIGITS];  /* max = 30 */
} M3G_H245_UII;

/* in-line macro used to initialize M3G_H245_UII to default settings of current release.
 * This macro initializes the version element to a valid value. The numDigits and 
 * digitBuffer elements must be explicitly intialized via user code.
 */
static __inline void INIT_M3G_H245_UII(M3G_H245_UII *pH245UII)
{
   pH245UII->version = M3G_LIBRARY_VERSION;
   /* numDigits element and digitBuffer array must be set via user code: */
   pH245UII->numDigits = 0;
}

/* H.245 MiscellaneousCommand type: */
typedef enum
{
    M3G_E_FAST_UPDATE_PICTURE = 0,
    M3G_E_FAST_UPDATE_GOB,
    M3G_E_FAST_UPDATE_MB,
    M3G_E_TEMP_SPAT_TRDFF,
    M3G_E_VIDEO_FREEZE,
    M3G_E_SYNC_EVERY_GOB,
    M3G_E_NOSYNC_EVERY_GOB
} M3G_E_H245_MISC_CMD_TYPE;

/* videoFastUpdateGOB command: */
typedef struct
{
   unsigned int           numFirstGOB;
   unsigned int           numGOBs;
} M3G_FASTUPDATE_GOB;

/* videoFastUpdateMB parameters: */
typedef struct
{
   unsigned int           numFirstGOB;
   unsigned int           numFirstMB;
   unsigned int           numMBs;
} M3G_FASTUPDATE_MB;

typedef unsigned int M3G_TEMPSPTRDFF;

/* parameters of H.245 MiscellaneousCommand: */
typedef union
{
   void*                  noParams;
   M3G_FASTUPDATE_GOB     fastUpdateGOB;
   M3G_FASTUPDATE_MB      fastUpdateMB;
   M3G_TEMPSPTRDFF        tempSpatialTrdff;
} M3G_H245_MISC_CMD_PARAMS;

/* transmitted/received H.245 MiscellaneousCommand indication message: */
typedef struct
{
   unsigned int                 version; /*set to M3G_LIBRARY_VERSION*/
   M3G_LOGICAL_CHANNEL_NUMBER   logicalChannelNumber;
   M3G_E_H245_MISC_CMD_TYPE     h245MiscCmdType;
   M3G_H245_MISC_CMD_PARAMS     h245MiscCmdParams;
} M3G_H245_MISC_CMD;

/* in-line macro used to initialize M3G_H245_MISC_CMD to default settings of current release.
 * This macro initializes the structure for a MiscellaneousCommand of type videoFastUpdatePicture. 
 * In this case, the logicalChannelNumber element must be explicitly intialized via user code.
 */
static __inline void INIT_M3G_H245_MISC_CMD(M3G_H245_MISC_CMD *pMiscCmd)
{
   pMiscCmd->version = M3G_LIBRARY_VERSION;
   /* logicalChannelNumber element must be set via user code: */
   pMiscCmd->logicalChannelNumber = 0;
   pMiscCmd->h245MiscCmdType = M3G_E_FAST_UPDATE_PICTURE;
   /* h245MiscCmdParams element is ignored in FVU: */
   pMiscCmd->h245MiscCmdParams.noParams = 0;
}

/* Master/Slave Determination event types: */
typedef enum
{
   M3G_E_MSD_RCVD = 0,
   M3G_E_MSD_SENT,
   M3G_E_MSD_ACK_RCVD,
   M3G_E_MSD_ACK_SENT,
   M3G_E_MSD_REL_RCVD,
   M3G_E_MSD_REL_SENT,
   M3G_E_MSD_REJ_RCVD,
   M3G_E_MSD_REJ_SENT,
   M3G_E_MSD_FAILURE
}  M3G_E_MSD_EVT_TYPE;

/* MultiplexEntrySend event types: */
typedef enum
{
    M3G_E_MES_RCVD = 0,
    M3G_E_MES_SENT,
    M3G_E_MES_ACK_RCVD,
    M3G_E_MES_ACK_SENT,
    M3G_E_MES_REL_RCVD,
    M3G_E_MES_REL_SENT,
    M3G_E_MES_REJ_RCVD,
    M3G_E_MES_REJ_SENT,
    M3G_E_MES_FAILURE
}  M3G_E_MES_EVT_TYPE;

/* MultiplexEntrySendReject causes: */
typedef enum
{
   M3G_E_MES_REJECT_UNSPECIFIED_CAUSE = 0,
   M3G_E_MES_REJECT_DECR_TOO_COMPLEX
}  M3G_E_MES_REJECT_CAUSE;

/* MultiplexEntrySend event: */
typedef struct
{
   M3G_E_MES_EVT_TYPE     MESEvtType;
   M3G_E_MES_REJECT_CAUSE MESRejectCause;
} M3G_MES;

/* H.223 SkewIndication: */
typedef struct
{
   M3G_LOGICAL_CHANNEL_NUMBER lcn1;
   M3G_LOGICAL_CHANNEL_NUMBER lcn2;
   unsigned short maxSkew;
}  M3G_SKEW_INDICATION;

/* values used in unsolicited event notification 
 * for m3g_EnableEvents() and m3g_DisableEvents():
 */
#define M3G_H245_UII_EVT_TYP                0x0001
#define M3G_H245_FASTUPDATE_EVT_TYP         0x0002
#define M3G_H245_TEMP_SPAT_TRDFF_EVT_TYP    0x0004
#define M3G_H245_VIDEO_FREEZE_EVT_TYP       0x0008
#define M3G_H245_SYNC_GOB_EVT_TYP           0x0010
#define M3G_LC_INACTIVE_EVT_TYP             0x0020
#define M3G_SKEW_INDICATION_EVT_TYP         0x0040
#define M3G_VERBOSE_MSD_EVT_TYP             0x0080
#define M3G_MES_EVTS_EVT_TYP                0x0100
#define M3G_MONA_PREF_MSG_EVT_TYP           0x0200
#define M3G_REMOTE_VENDORID_EVT_TYP         0x0400
#define M3G_CALL_STATISTICS_EVT_TYP         0x0800
#define M3G_IFRAME_EVT_TYP                  0x1000

/* tracing configuration used in m3g_StartTrace(): */
typedef struct
{
   unsigned short     version;        /* set to M3G_LIBRARY_VERSION */
   const char*        logfile;
   unsigned int       bitmask;
} M3G_TRACE_INFO;

#define M3G_TRACE_H245       0x0001   /* decoded H.245 messages */
#define M3G_TRACE_H223       0x0002   /* raw H.223 multiplex bitstreams */
#define M3G_TRACE_AUDIO      0x0004   /* raw audio bitstreams */
#define M3G_TRACE_VIDEO      0x0008   /* raw audio bitstreams */
#define M3G_TRACE_INTERNALS  0x0010   /* internal 3G module debug tracing */
#define M3G_TRACE_STATISTICS 0x0020   /* 3G session statistics */

/* in-line macro used to initialize M3G_TRACE_INFO to default settings of current release.
 * This macro initializes the structure maximize 3G related tracing to a file name "m3glog.txt".
 */
static __inline void INIT_M3G_TRACE_INFO(M3G_TRACE_INFO *pTraceInfo)
{
   pTraceInfo->version = M3G_LIBRARY_VERSION;
   pTraceInfo->logfile = "m3glog.txt";
   pTraceInfo->bitmask = (M3G_TRACE_H245 | M3G_TRACE_H223 | M3G_TRACE_AUDIO |
                          M3G_TRACE_VIDEO | M3G_TRACE_INTERNALS | M3G_TRACE_STATISTICS);
}

/* nonStandardIdentifier type: */
typedef enum
{
   M3G_E_OID_TYPE = 0,
   M3G_E_H221_ID_TYPE
}  M3G_E_NONSTD_TYPE;

/* ASN.1 object identifiers: */
typedef struct
{
   unsigned char      length;
   unsigned char      objectId[OBJECTID_SIZE];
} M3G_OBJECT_ID;

/* H.221 identifiers: */
typedef struct 
{
   unsigned char      t35CountryCode;
   unsigned char      t35Extension;
   unsigned short     manufacturerCode;
} M3G_H221_NONSTD;

/* nonStandardIdentifier used to identify non-standard parameter type as either 
 * an object identifier or an H.221 type of identifier:
 */
typedef struct
{
   M3G_E_NONSTD_TYPE    oidType;
   union
   {
      M3G_OBJECT_ID   oid;
      M3G_H221_NONSTD h221NonStd;
   } oidValue;
} M3G_NONSTANDARD_ID;

/* vendor and product information used in m3g_SetVendorId(): */
typedef struct
{
   unsigned int       version;       /* set to M3G_LIBRARY_VERSION */
   M3G_NONSTANDARD_ID vendor;
   M3G_OCTET_STRING   productNumber;
   M3G_OCTET_STRING   versionNumber;
} M3G_VENDORID_INFO;

/* in-line macro used to initialize M3G_VENDORID_INFO: 
 * This macro only initializes the vendor identification elements to identify Dialogic. 
 * Users should explicitly intialize the vendor identification via user code.
*/
static __inline void INIT_M3G_VENDORID_INFO(M3G_VENDORID_INFO *pVendorId)
{
   pVendorId->version = M3G_LIBRARY_VERSION;

   /* initialize vendor octet idenfifier to "Dialogic" */
   pVendorId->vendor.oidType = M3G_E_OID_TYPE;
   pVendorId->vendor.oidValue.oid.length = 8;
   pVendorId->vendor.oidValue.oid.objectId[0] = 0x44;
   pVendorId->vendor.oidValue.oid.objectId[1] = 0x69;
   pVendorId->vendor.oidValue.oid.objectId[2] = 0x61;
   pVendorId->vendor.oidValue.oid.objectId[3] = 0x6c;
   pVendorId->vendor.oidValue.oid.objectId[4] = 0x6f;
   pVendorId->vendor.oidValue.oid.objectId[5] = 0x67;
   pVendorId->vendor.oidValue.oid.objectId[6] = 0x69;
   pVendorId->vendor.oidValue.oid.objectId[7] = 0x63;

   /* initialize productNumber octet string to "MMP" */
   pVendorId->productNumber.length = 3;
   pVendorId->productNumber.octet[0] = 0x4d;
   pVendorId->productNumber.octet[1] = 0x4d;
   pVendorId->productNumber.octet[2] = 0x50;

   /* initialize versionNumber octet string to "2.0" */
   pVendorId->versionNumber.length = 3;
   pVendorId->versionNumber.octet[0] = 0x32;
   pVendorId->versionNumber.octet[1] = 0x2e;
   pVendorId->versionNumber.octet[2] = 0x30;
}

/* MONA Media Preconfigured Channel MPC-TX and MPC-RX bits: */
typedef struct
{
   unsigned short rxMPCMask;
   unsigned short txMPCMask;
} M3G_MONA_TXRX_MPC_SUPPORT;

/* MPC-TX and MPC-RX bits as per Table K.15/H.324: */
#define M3G_MPC_AMR_BIT        0x0001
#define M3G_MPC_AMR_WB_BIT     0x0002
#define M3G_MPC_H264_BIT       0x0004
#define M3G_MPC_MPEG4_BIT      0x0008
#define M3G_MPC_H263_BIT       0x0010
#define M3G_MPC_RESERVED_BIT6  0x0020
#define M3G_MPC_RESERVED_BIT7  0x0040
#define M3G_MPC_RESERVED_BIT8  0x0080
#define M3G_MPC_RESERVED_BIT9  0x0100
#define M3G_MPC_RESERVED_BIT10 0x0200
#define M3G_MPC_RESERVED_BIT11 0x0400
#define M3G_MPC_OPERATOR_BIT12 0x0800
#define M3G_MPC_OPERATOR_BIT13 0x1000

/* informatation on established MONA Media Preconfigured Channel forwarded
 * in events M3GEV_TX_MPC_ESTABLISHED and M3GEV_RX_MPC_ESTABLISHED:
 */
typedef struct
{
   unsigned int               version;        /*set to M3G_LIBRARY_VERSION*/
   M3G_LOGICAL_CHANNEL_NUMBER logicalChannelNumber;
   M3G_H223_LC_PARAMS         h223MultiplexParams;
   M3G_E_CAPABILITY           capabilityType;
   M3G_CAPABILITY             mediaCapability;
} M3G_MONA_MPC;

/* call statistics forwarded in event M3GEV_CALL_STATISTICS: */
typedef struct 
{
   unsigned int callDuration;
   unsigned int rxMediaCrcErrs;
   unsigned int rxAudioPacketErrs;
   unsigned int rxVideoPacketErrs;
   unsigned int rxTotalBytes;
   unsigned int rxMediaBytes;
   unsigned int rxAudioBytes;
   unsigned int rxVideoBytes;
   unsigned int rxStuffingBytes;
   unsigned int rxMediaPackets;
   unsigned int rxAudioPackets;
   unsigned int rxVideoPackets;
   unsigned int rxMuxPdus;
   unsigned int rxMuxPduBytes;
   unsigned int txTotalBytes;
   unsigned int txMediaBytes;
   unsigned int txAudioBytes;
   unsigned int txVideoBytes;
   unsigned int txStuffingBytes;
   unsigned int txMediaPackets;
   unsigned int txAudioPackets;
   unsigned int txVideoPackets;
   unsigned int txMuxPdus;
   unsigned int txMuxPduBytes;
} M3G_CALL_STATISTICS;

/*******************************************************************************
 * M3G API methods
 ******************************************************************************/
#ifdef __cplusplus
extern "C" {
#endif

M3GLIB_API M3G_RETVAL m3g_Start(M3G_START_STRUCT* pStartParams);
M3GLIB_API M3G_RETVAL m3g_Stop();
M3GLIB_API SRL_DEVICE_HANDLE m3g_Open(const char *deviceName,
                    M3G_OPEN_INFO* pOpenInfo,
                    void* pUserInfo);
M3GLIB_API SRL_DEVICE_HANDLE m3g_OpenEx(const char *deviceName,
                    M3G_OPEN_INFO* pOpenInfo,
                    void* pUserInfo,
                    unsigned short mode);
M3GLIB_API M3G_RETVAL m3g_Close(SRL_DEVICE_HANDLE deviceHandle);
M3GLIB_API M3G_RETVAL m3g_GetUserInfo(SRL_DEVICE_HANDLE deviceHandle,
                    void ** ppUserInfo);
M3GLIB_API M3G_RETVAL m3g_EnableEvents(SRL_DEVICE_HANDLE deviceHandle,
                    unsigned int eventBitmask);
M3GLIB_API M3G_RETVAL m3g_DisableEvents(SRL_DEVICE_HANDLE deviceHandle,
                    unsigned int eventBitmask);
M3GLIB_API M3G_RETVAL m3g_SetParm(SRL_DEVICE_HANDLE deviceHandle, 
                    M3G_PARM_INFO *pParmInfo);
M3GLIB_API M3G_RETVAL m3g_GetParm(SRL_DEVICE_HANDLE deviceHandle, M3G_E_PRM_TYPE parm);
M3GLIB_API M3G_RETVAL m3g_GetLocalCaps(SRL_DEVICE_HANDLE deviceHandle);
M3GLIB_API M3G_RETVAL m3g_SetTCS(SRL_DEVICE_HANDLE deviceHandle,
                    unsigned short numSimultaneousSets,
                    M3G_SIMULTANEOUS_CAP_SET *pSimultaneousCapList);
M3GLIB_API M3G_RETVAL m3g_StartH245(SRL_DEVICE_HANDLE deviceHandle,
                    M3G_H223_SESSION *pH223Params);
M3GLIB_API M3G_RETVAL m3g_StopH245(SRL_DEVICE_HANDLE deviceHandle);
M3GLIB_API M3G_RETVAL m3g_GetMatchedCaps(SRL_DEVICE_HANDLE deviceHandle, 
                    M3G_CAPS_LIST *pMatchedCapsList);
M3GLIB_API M3G_RETVAL m3g_OpenLC(SRL_DEVICE_HANDLE deviceHandle,
                    M3G_H223_LC_PARAMS *pH223LCParameters,
                    M3G_E_CAPABILITY    capabilityType,
                    M3G_CAPABILITY *pMediaCapability);
M3GLIB_API M3G_RETVAL m3g_RespondToOLC(SRL_DEVICE_HANDLE deviceHandle,
                    M3G_LOGICAL_CHANNEL_NUMBER lcn,
                    M3G_E_OLC_RESP_TYPE olcResponse);
M3GLIB_API M3G_RETVAL m3g_CloseLC(SRL_DEVICE_HANDLE deviceHandle,
                    M3G_LOGICAL_CHANNEL_NUMBER lcn,
                    M3G_E_REQ_CHAN_CLOSE_REASON reason);
M3GLIB_API M3G_RETVAL m3g_StartMedia(SRL_DEVICE_HANDLE deviceHandle);
M3GLIB_API M3G_RETVAL m3g_ModifyMedia(SRL_DEVICE_HANDLE deviceHandle,
                                      M3G_E_DIRECTION   direction);
M3GLIB_API M3G_RETVAL m3g_StopMedia(SRL_DEVICE_HANDLE deviceHandle);
M3GLIB_API M3G_RETVAL m3g_SendH245UII(SRL_DEVICE_HANDLE deviceHandle,
                    M3G_H245_UII *pH245UII);
M3GLIB_API M3G_RETVAL m3g_SendH245MiscCmd(SRL_DEVICE_HANDLE deviceHandle, 
                    M3G_H245_MISC_CMD *pMiscCmd);
M3GLIB_API M3G_RETVAL m3g_Reset(SRL_DEVICE_HANDLE deviceHandle);
M3GLIB_API M3G_RETVAL m3g_StartTrace(SRL_DEVICE_HANDLE deviceHandle, 
                                     M3G_TRACE_INFO *pTraceInfo);
M3GLIB_API M3G_RETVAL m3g_StopTrace(SRL_DEVICE_HANDLE deviceHandle);
M3GLIB_API M3G_RETVAL m3g_SetVendorId(SRL_DEVICE_HANDLE deviceHandle, 
                    M3G_VENDORID_INFO *pVendorIdInfo);

#ifdef __cplusplus
}
#endif  /* __cplusplus */

#pragma pack()

#endif
