/**
 * @file  mp_mtklib.h
 * @brief Media Parser Media Toolkit Library
 * @date  April 4, 2011
 *
 * DIALOGIC CONFIDENTIAL      
 * Copyright (C) 2011 Dialogic Corporation. All Rights Reserved.
 * 
 * The source code contained or described herein and all documents related to
 * the source code ("Material") are owned by Dialogic Corporation or its 
 * suppliers or licensors.  Title to the Material remains with Dialogic 
 * Corporation or its suppliers and licensors.  The Material contains trade
 * secrets and proprietary and confidential information of Dialogic or its 
 * suppliers and licensors.  The Material is protected by worldwide copyright
 * and trade secret laws and treaty provisions.  No part of the Material may be
 * used, copied, reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way without Dialogic's prior express written
 * permission.
 * 
 * No license under any patent, copyright, trade secret or other intellectual 
 * property right is granted to or conferred upon you by disclosure or delivery
 * of the Materials, either expressly, by implication, inducement, estoppel or
 * otherwise.  Any license under such intellectual property rights must be
 * express and approved by Dialogic in writing.
 */

#ifndef _MP_MTKLIB_H_
#define _MP_MTKLIB_H_

#include "mtklib.h"
//#include "stddef.h"



#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */



/*******************************************************************************
 * Media Parser Media Toolkit Data Type's
 ******************************************************************************/




/*******************************************************************************
 * Media Parser Media Toolkit Definitions
 ******************************************************************************/

/* Error return codes */
#define	MPERR_INVALID_FILE_TYPE      0x00000001  //File type not supported
// value 0x00000002 reserved
#define	MPERR_INVALID_FILE           0x00000003  //Invalid file access structure
// value 0x00000004 reserved
#define	MPERR_INVALID_POINTER        0x00000005  //NULL pointer
// values from 0x00000006 to 0x0000000F are reserved
#define	MPERR_FILE_OPEN_FAILED       0x00000010  //File open failed e.g. file not found
#define	MPERR_FILE_ACCESS_DENIED     0x00000011  //Access denied
#define	MPERR_INCOMP_ACCESS_MODE     0x00000012  //File was not opened for read operation
// values from 0x00000013 to 0x00000014 are reserved
#define	MPERR_INTERNAL_RESOURCE      0x00000015  //Error on internal resource (e.g. memory allocation)
#define	MPERR_FILE_FORMAT            0x00000016  //Error in file format
// values from 0x00000017 to 0x00000020 are reserved
#define MPERR_INVALID_STRUCT_VERSION 0x00000021  //Invalid structure version





/*******************************************************************************
 * Media Parser Media Toolkit Data Structures
 ******************************************************************************/


/*!
****************************************************************************
\brief      Multimedia File access structure

                The MP_MMFILE structure is used to access a multimedia file.
                This structure is created by the application, filled in with
                mp_OpenFile() and used by the rest of the mp* API functions.
                The only parameter that needs to be set by the application
                is the structure version.  It can/should completely ignore
                the rest of the structure.
****************************************************************************/
typedef struct tagMP_MMFILE
{
    unsigned int    unVersion;          //Input - Structure version
    unsigned        state;
    unsigned        openMode;
    unsigned        fileType;
    unsigned        verboseLevel;
    void            *mp;
    int             streamCount;
    void            *pMMStream;
    int             track_ids_set;
    unsigned long   video_track_id;
    unsigned long   audio_track_id;

} MP_MMFILE;

#define MP_MMFILE_VER(v)        (sizeof(MP_MMFILE) << 16 | (0xFFFF & (v)))
#define MP_MMFILE_VERSION_0     MP_MMFILE_VER(0)
#define MP_MMFILE_VERSION       MP_MMFILE_VER(0)  //Currently at version 0


/*!
****************************************************************************
\brief      Video track description structure

                The MP_VIDEO_TRACK_DESCRIPTION contains video track
                information
****************************************************************************/
typedef struct tagMP_VIDEO_TRACK_DESCRIPTION
{
    unsigned long track_id;     //!< ID of track
    unsigned long NbSyncPoints; //!< Number of sync points for track

} MP_VIDEO_TRACK_DESCRIPTION;


/*!
****************************************************************************
\brief      Audio track description structure

                The MP_AUDIO_TRACK_DESCRIPTION contains audio track
                information
****************************************************************************/
typedef struct tagMP_AUDIO_TRACK_DESCRIPTION
{
    unsigned long track_id;     //!< ID of track

} MP_AUDIO_TRACK_DESCRIPTION;


/*!
****************************************************************************
\brief      File info descriptor structure

                The MP_FILE_INFO_DESC structure is the descriptor for
                a multimedia file
****************************************************************************/
typedef struct tagMP_FILE_INFO_DESC
{
    unsigned int    unVersion;          //Input - Structure version

    unsigned short  fileInfoSize;       //Input/Output - Total size of the File Info block
                                        //Set by the application to indicate the maximum size of File Info block.
                                        //Reset by mp_GetFileInfo() to indicate the actual size of block used.

    unsigned char   *pFileInfo;         //Input - Pointer to File Info block - set by the application

    unsigned short  fileType;           //Output - Type of file, i.e. type of information block (see #mp_OpenFile)
#define	MP_FILE_TYPE_UNKWN  0           //Unknown/unrecognized/unsupported type of file
#define	MP_FILE_TYPE_ISO    1           //Generic ISO-based format file (applicable for any ISO-based file) 
#define	MP_FILE_TYPE_3GP    2           //3GP format file (ISO-based)
#define	MP_FILE_TYPE_MP4    3           //MP4 format file (ISO-based)
#define	MP_FILE_TYPE_F4V    4           //Flash H.264 file (ISO-based)
#define	MP_FILE_TYPE_3G2    5           //3GPP2 format file (ISO-based)
#define	MP_FILE_TYPE_AVC    6           //AVC format file (ISO-based)
//Values 7-9 reserved for future ISO-based file formats
#define	MP_FILE_TYPE_DMF    10          //Dialogic Media format file (Dialogic-proprietary)

    char            format[8];          //Output - File format standard
                                        //For ISO/3GP, major brand string e.g. "3GP5"

    unsigned long   version;            //Output - Format version
                                        //For 3GP, = (x * 256 + y) which refers to TS 26.244 v5.x.y.
                                        //Ex: v5.1.0 gives 256
    char            compatible_brands[50];  //Output - List of compatible brands
                                            //For ISO/3GP, compatible_brands string e.g. "3gp6 3gp5 3gp4"


    unsigned long   flags;              //Output - see #MP_DESC_FLAG_INSUFF_SPACE
#define	MP_DESC_FLAG_INSUFF_SPACE   1   //Insufficient space in provided File Info block to contain all data

#define MP_NTRACKS      12
    MP_VIDEO_TRACK_DESCRIPTION video_track_descriptor[MP_NTRACKS];  //Output - Array of video track ID's present
    MP_AUDIO_TRACK_DESCRIPTION audio_track_descriptor[MP_NTRACKS];  //Output - Array of video track ID's present

    unsigned long   NbVideoTracks;      //Output - Number of video tracks present in file
    unsigned long   NbAudioTracks;      //Output - Number of audio tracks present in file
    unsigned long   NbVideoHintTracks;  //Output - Number of video hint tracks present in file
    unsigned long   NbAudioHintTracks;  //Output - Number of audio hint tracks present in file

} MP_FILE_INFO_DESC;

#define MP_FILE_INFO_DESC_VER(v)        (sizeof(MP_FILE_INFO_DESC) << 16 | (0xFFFF & (v)))
#define MP_FILE_INFO_DESC_VERSION_0     MP_FILE_INFO_DESC_VER(0)
#define MP_FILE_INFO_DESC_VERSION       MP_FILE_INFO_DESC_VER(0)  //Currently at version 0





//Block Type Identifiers
#define	MP_BLK_TYPE_PRESENTATION        1   //Presentation block
#define	MP_BLK_TYPE_STREAM_UNKNOWN      2   //Unknown stream block
#define	MP_BLK_TYPE_STREAM_AUDIO        3   //Audio stream block
#define	MP_BLK_TYPE_STREAM_VIDEO        4   //Video stream block
#define	MP_BLK_TYPE_STREAM_HINT         5   //Hint stream block
#define	MP_BLK_TYPE_AUDIO_CODEC_AMR     10  //Audio AMR codec block (NB or WB)
#define	MP_BLK_TYPE_AUDIO_CODEC_AAC     11  //Audio AAC codec block
#define	MP_BLK_TYPE_VIDEO_CODEC_H263    20  //Video H.263 codec block
#define	MP_BLK_TYPE_VIDEO_CODEC_MPEG4   21  //Video MPEG-4 codec block
#define	MP_BLK_TYPE_VIDEO_CODEC_H264    22  //Video H.264 codec block
#define	MP_BLK_TYPE_VIDEO_CODEC_VP8     23  //Video VP8 codec block
#define	MP_BLK_TYPE_HINT_PROTOCOL_RTP   30  //Hint RTP protocol block

//Packetization Mode Identifiers
#define	MP_S_PACK_NONE              0   //Non-hinted file, no packetization
#define	MP_S_PACK_UNKNWN            1   //Unknown packetization mode - hinted file but could not be determined
#define	MP_S_PACK_MPEG4             2   //MPEG4 packetization per RFC3016
#define	MP_S_PACK_H263_2190         3   //H.263 packetization per RFC2190
#define	MP_S_PACK_H263_2429         4   //H.263 packetization per RFC2429/4629
#define	MP_S_PACK_H264_MODE0        5   //H.264 packetization-mode 0 (single NAL)
#define	MP_S_PACK_H264_MODE1        6   //H.264 packetization-mode 1 (non-interleaved)
#define	MP_S_PACK_H264_MODE2        7   //H.264 packetization-mode 2 (interleaved)
#define	MP_S_PACK_AMR_OCTET_ALIGN   8   //AMR Octet-Aligned packetization mode
#define	MP_S_PACK_AMR_BW_EFF        9   //AMR Bandwidth Efficient packetization mode

//Other identifiers
#define	MP_S_PTIME_UNKNOWN          0       //Unknown ptime attribute
#define MP_S_CODEC_UNKNOWN          0       //Unsuported/unrecognized codec
#define MP_MAX_DCI_SIZE             512     //Max size for DCI buffers
#define MP_MAX_SDP_SIZE             1024    //Max size for SDP buffers


/*!
****************************************************************************
\brief      Block header

                The MP_BLK_HEADER contains just the 2 fields that are at the
                beginning of any BLK_ structure block.  Used to map to the
                start of a block until it is determined what type of
                block it is (what blkType is).
****************************************************************************/
typedef struct {

    unsigned short  blkSize;    //Size of block
    unsigned short  blkType;    //Type of block (BLK_TYPE_PRESENTATION)

} MP_BLK_HEADER;


/*!
****************************************************************************
\brief      Presentation block

                The MP_BLK_PRESENTATION block contains presentation/movie -level
                information. It is the first block of the file information.
****************************************************************************/
typedef struct {

    unsigned short  blkSize;            //Size of block
    unsigned short  blkType;            //Type of block (MP_BLK_TYPE_PRESENTATION)

    unsigned long   creationTime;       //Presentation creation date.  For ISO/3GP, in seconds since 1-jan-1904
    unsigned long   duration;           //Global presentation play duration, in milliseconds.
    unsigned short  streamCount;        //Number of streams/tracks detected and reported in presentation.
    unsigned long   maxInterleave;      //Max stream interleave depth in presentation, in milliseconds.
    
    //SDP Information (session-level)
    unsigned short  sdpSize;     //Size in bytes of the following SDP buffer
    char            pSdp[MP_MAX_SDP_SIZE];

} MP_BLK_PRESENTATION;

/*!
****************************************************************************
\brief      Unknown stream block

                The MP_BLK_STREAM_UNKNOWN is a placeholder for an unrecognized
                stream.  It is not followed by any MP_BLK_CODEC structure.
****************************************************************************/
typedef struct {

    unsigned short  blkSize;            //Size of block
    unsigned short  blkType;            //Type of block (MP_BLK_TYPE_STREAM_UNKNOWN)

    unsigned long   streamID;           //ID of stream/track in the file

    unsigned short  codec;              //MP_S_CODEC_UNKNOWN

    unsigned long   creationTime;       //Stream/track creation date.  For ISO/3GP, in seconds since 1-jan-1904
    unsigned long   duration;           //Duration of stream in milliseconds

} MP_BLK_STREAM_UNKNOWN;


/*!
****************************************************************************
\brief      Audio stream block

                The MP_BLK_STREAM_AUDIO block contains audio stream/track -level
                information.  It may or may not be followed by a
                MP_BLK_CODEC_AUDIO_xxx structure, depending on whether the given
                audio codec has a CODEC structure defined for it.
****************************************************************************/
typedef struct {

    unsigned short  blkSize;            //Size of block
    unsigned short  blkType;            //Type of block (MP_BLK_TYPE_STREAM_AUDIO)
												
    unsigned long   streamID;           //ID of stream/track in the file

    unsigned short  codec;              //Type of codec

//For 3GP/ISO files, the following definitions are used.
//#define MP_S_CODEC_UNKNOWN         0      //Unsuported/unrecognized codec
#define	MP_ISO_AUD_CODEC_AMR_NB      100       //AMR-NB codec - this struct followed by MP_BLK_CODEC_AUDIO_AMR struct
#define	MP_ISO_AUD_CODEC_AMR_WB      101       //AMR-WB codec - this struct followed by MP_BLK_CODEC_AUDIO_AMR struct
#define	MP_ISO_AUD_CODEC_AAC         102       //AAC codec    - this struct followed by MP_BLK_CODEC_AUDIO_AAC struct
//For Dialogic Proprietary (DMF) files, the MM_DATA_FORMAT_* definitions in mmlib.h are used

    unsigned long   creationTime;       //Stream/track creation date.  For ISO/3GP, in seconds since 1-jan-1904
    unsigned long   duration;           //Duration of stream in milliseconds
    double          bitRate;            //Stream bitrate (kbps)
    double          frameRate;          //Stream frame rate (fps)
    unsigned long   sampleRate;         //Stream sampling rate (Hz) - timescale
    unsigned long   sampleSize;         //Sample size - non-zero if constant sample size. 0 if variable sizes.
    unsigned long   maxSampleSize;      //Size of largest sample in stream. Will equal sampleSize if constant sample size.
    unsigned long   sampleCount;        //Number of samples in stream.
    unsigned long   byteCount;          //Number of total bytes in stream.
    char            handlerName [8];    //Media handler name.  Null-terminated text string.
    unsigned short  channelCount;       //1=mono, 2=stereo, 6=5.1
#define	MP_CHANNEL_COUNT_MONO      1       //Mono sound
#define	MP_CHANNEL_COUNT_STEREO    2       //Stereo sound
#define	MP_CHANNEL_COUNT_51        6       //5.1 sound

    unsigned short  bitsPerSample;      //Number of bits per audio sample

} MP_BLK_STREAM_AUDIO;


/*!
****************************************************************************
\brief      Video stream block

                The MP_BLK_STREAM_VIDEO block contains video stream/track -level
                information.  It may or may not be followed by a
                MP_BLK_CODEC_VIDEO_xxx structure, depending on whether the given
                video codec has a CODEC structure defined for it.
****************************************************************************/
typedef struct {

    unsigned short  blkSize;            //Size of block
    unsigned short  blkType;            //Type of block (MP_BLK_TYPE_STREAM_VIDEO)
												
    unsigned long   streamID;           //ID of stream/track in the file

    unsigned short  codec;              //Type of codec
//#define MP_S_CODEC_UNKNOWN 0          //Unsuported/unrecognized codec
#define	MP_VID_CODEC_H263    1          //H.263 codec - this struct followed by MP_BLK_CODEC_VIDEO_H263 struct
#define	MP_VID_CODEC_MPEG4   2          //MPEG-4 codec - this struct followed by MP_BLK_CODEC_VIDEO_MPEG4 struct
#define	MP_VID_CODEC_H264    3          //H.264 codec - this struct followed by MP_BLK_CODEC_VIDEO_H264 struct
#define	MP_VID_CODEC_VP8     4          //VP8 codec - this struct followed by MP_BLK_CODEC_VIDEO_VP8 struct

    unsigned long   creationTime;       //Stream/track creation date.  For ISO/3GP, in seconds since 1-jan-1904
    unsigned long   duration;           //Duration of stream in milliseconds
    double          bitRate;            //Stream bitrate (kbps)
    double          frameRate;          //Stream frame rate (fps)
    unsigned long   sampleRate;         //Stream sampling rate (Hz) - timescale
    unsigned long   sampleSize;         //Sample size - non-zero if constant sample size. 0 if variable sizes.
    unsigned long   maxSampleSize;      //Size of largest sample in stream. Will equal sampleSize if constant sample size.
    unsigned long   sampleCount;        //Number of samples in stream.
    unsigned long   byteCount;          //Number of total bytes in stream.
    char            handlerName [8];    //Media handler name.  Null-terminated text string.

    unsigned short  width;              //Visual width,  in pixels (e.g. 176 for QCIF)
    unsigned short  height;             //Visual height, in pixels (e.g. 144 for QCIF)
    short           layer;              //Front-to-back ordering. 0 is normal view, negative values are in front
    unsigned short  hResolution;        //Picture horizontal resolution, in pixels/inch.
    unsigned short  vResolution;        //Picture vertical resolution,   in pixels/inch.

} MP_BLK_STREAM_VIDEO;

/*!
****************************************************************************
\brief      Hint stream block

                The MP_BLK_STREAM_HINT block contains hint stream/track -level
                information.  It will be followed by a MP_BLK_PROTOCOL_HINT_RTP
                structure.
****************************************************************************/
typedef struct {

    unsigned short  blkSize;            //Size of block
    unsigned short  blkType;            //Type of block (MP_BLK_TYPE_STREAM_HINT)
												
    unsigned long   streamID;           //ID of stream/track in the file

    unsigned short  codec;              //Type of codec (MP_S_CODEC_UNKNOWN)
#define	MP_HINT_PROTOCOL_RTP   1        //RTP Protocol - this struct followed by MP_BLK_PROTOCOL_HINT_RTP struct

    unsigned long   creationTime;       //Stream/track creation date.  For ISO/3GP, in seconds since 1-jan-1904
    unsigned long   duration;           //Duration of stream in milliseconds
    double          bitRate;            //Stream bitrate (kbps)
    double          frameRate;          //Stream frame rate (fps)
    unsigned long   sampleRate;         //Stream sampling rate (Hz) - timescale
    unsigned long   sampleSize;         //Sample size - non-zero if constant sample size. 0 if variable sizes.
    unsigned long   maxSampleSize;      //Size of largest sample in stream. Will equal sampleSize if constant sample size.
    unsigned long   sampleCount;        //Number of samples in stream.
    unsigned long   byteCount;          //Number of total bytes in stream.
    char            handlerName [8];    //Media handler name.  Null-terminated text string.

    unsigned long   refStreamID [4];    //ID of the 4 first streams referenced by this hint data
    
    //SDP Information (track-level)
    unsigned short  sdpSize;     //Size in bytes of the following SDP buffer
    char            pSdp[MP_MAX_SDP_SIZE];

} MP_BLK_STREAM_HINT;


/*!
****************************************************************************
\brief      Audio AMR Codec block

                The MP_BLK_CODEC_AUDIO_AMR block contains AMR -specific 
                codec information.  It is preceded by a MP_BLK_STREAM_AUDIO
                block and designates the last block of information for the
                given audio stream.
****************************************************************************/
typedef struct {

    unsigned short  blkSize;            //Size of block
    unsigned short  blkType;            //Type of block (MP_BLK_TYPE_AUDIO_CODEC_AMR)

    unsigned short  packetizationMode;  //Packetization mode for AMR (uses one of MP_S_PACK_* definitions)
    unsigned short  ptime;              //Packetization time parameter for audio, same as ptime in RFC2327
                                        // Set to specific value in ms or MP_S_PTIME_UNKNOWN

    unsigned short  modeSet;                //Bitmap of AMR modes possibly present in stream. b0=mode 0 etc.
#define	MP_S_AMR_MODE_475       0x0001      //Mode 0 - 4.75 kb/s
#define	MP_S_AMR_MODE_515       0x0002      //Mode 1 - 5.15 kb/s
#define	MP_S_AMR_MODE_590       0x0004      //Mode 2 - 5.90 kb/s
#define	MP_S_AMR_MODE_670       0x0008      //Mode 3 - 6.70 kb/s
#define	MP_S_AMR_MODE_740       0x0010      //Mode 4 - 7.40 kb/s
#define	MP_S_AMR_MODE_795       0x0020      //Mode 5 - 7.95 kb/s
#define	MP_S_AMR_MODE_102       0x0040      //Mode 6 - 10.2 kb/s
#define	MP_S_AMR_MODE_122       0x0080      //Mode 7 - 12.2 kb/s
#define	MP_S_AMR_MODE_SID       0x0100      //Mode 8 - AMR SID
#define	MP_S_AMR_MODE_NODATA    0x8000      //No data
#define	MP_S_AMR_MODE_ALL       0x81FF      //All modes may be present

#define	MP_S_AMR_WB_MODE_660        0x0001  //Mode 0 - 6.60  kb/s
#define	MP_S_AMR_WB_MODE_885        0x0002  //Mode 1 - 8.85  kb/s
#define	MP_S_AMR_WB_MODE_1265       0x0004  //Mode 2 - 12.65 kb/s
#define	MP_S_AMR_WB_MODE_1425       0x0008  //Mode 3 - 14.25 kb/s
#define	MP_S_AMR_WB_MODE_1585       0x0010  //Mode 4 - 15.85 kb/s
#define	MP_S_AMR_WB_MODE_1825       0x0020  //Mode 5 - 18.25 kb/s
#define	MP_S_AMR_WB_MODE_1985       0x0040  //Mode 6 - 19.85 kb/s
#define	MP_S_AMR_WB_MODE_2305       0x0080  //Mode 7 - 23.05 kb/s
#define	MP_S_AMR_WB_MODE_2385       0x0100  //Mode 8 - 23.85 kb/s
#define	MP_S_AMR_WB_MODE_SID        0x0200  //Mode 9 - AMR SID
#define	MP_S_AMR_WB_MODE_NODATA     0x8000  //No data
#define	MP_S_AMR_WB_MODE_ALL        0x83FF  //All modes may be present

    unsigned char   modeChangePeriod;   //If 0, no restriction.
    unsigned char   framesPerSample;    //Nb of frames per sample (refer to sampleSize in stream header).
    char            vendor [8];         //4-char string of codec's manufacturer.

} MP_BLK_CODEC_AUDIO_AMR;



/*!
****************************************************************************
\brief      Audio AAC Codec block

                The MP_BLK_CODEC_AUDIO_AAC block contains AAC -specific
                codec information.  It is preceded by a MP_BLK_STREAM_AUDIO
                block and designates the last block of information for the
                given audio stream.
****************************************************************************/
typedef struct {

    unsigned short  blkSize;            //Size of block
    unsigned short  blkType;            //Type of block (MP_BLK_TYPE_AUDIO_CODEC_AAC)

    unsigned short  dcStreamType;       //'streamType' field of MPEG-4 DecoderConfig Descriptor.
    unsigned short  dcBufferSize;       //Maximum size of sample, in bytes.
    unsigned long   dcAvgBitrate;       //Average bitrate, in bits/s.
    unsigned long   dcMaxBitrate;       //Maximum bitrate, in bits/s.

    //DCI Information
    unsigned short  dcSpecInfoSize;     //Size in bytes of the following Decoder Specific Information buffer
    unsigned char   pDcSpecInfo[MP_MAX_DCI_SIZE];
                                //Buffer containing Decoder Specific Information of size 'dcSpecInfoSize'

} MP_BLK_CODEC_AUDIO_AAC;	


/*!
****************************************************************************
\brief      Video H.263 Codec block

                The MP_BLK_CODEC_VIDEO_H263 block contains H.263 -specific
                codec information.  It is preceded by a MP_BLK_STREAM_VIDEO
                block and designates the last block of information for the
                given video stream.
****************************************************************************/
typedef struct {

    unsigned short  blkSize;            //Size of block
    unsigned short  blkType;            //Type of block (MP_BLK_TYPE_VIDEO_CODEC_H263)

    unsigned short  packetizationMode;  //Packetization mode for H.263 (uses one of MP_S_PACK_* definitions)

    unsigned char   profile;            //H.263 profile
    unsigned char   level;              //H.263 standard level

    char            vendor [8];         //4-char string of codec's manufacturer.

} MP_BLK_CODEC_VIDEO_H263;	


/*!
****************************************************************************
\brief      Video MPEG-4 Codec block

                The MP_BLK_CODEC_VIDEO_MPEG4 block contains MPEG4 -specific
                codec information.  It is preceded by a MP_BLK_STREAM_VIDEO
                block and designates the last block of information for the
                given video stream.
****************************************************************************/
typedef struct {

    unsigned short  blkSize;            //Size of block
    unsigned short  blkType;            //Type of block (MP_BLK_TYPE_VIDEO_CODEC_MPEG4)

    unsigned short  packetizationMode;  //Packetization mode for MPEG-4 (uses one of MP_S_PACK_* definitions)

    unsigned char   profile_and_level_ind;  //MPEG4 profile/level indication
    unsigned short  dcStreamType;           //'streamType' field of MPEG-4 DecoderConfig Descriptor.
    unsigned short  dcBufferSize;           //Maximum size of sample, in bytes.
    unsigned long   dcAvgBitrate;           //Average bitrate, in bits/s.
    unsigned long   dcMaxBitrate;           //Maximum bitrate, in bits/s.

    //DCI Information
    unsigned short  dcSpecInfoSize;     //Size in bytes of the following Decoder Specific Information buffer
    unsigned char   pDcSpecInfo[MP_MAX_DCI_SIZE];
                                //Buffer containing Decoder Specific Information of size 'dcSpecInfoSize'

} MP_BLK_CODEC_VIDEO_MPEG4;	


/*!
****************************************************************************
\brief      Video H264 Codec block

                The MP_BLK_CODEC_VIDEO_H264 block contains H264 -specific
                codec information.  It is preceded by a MP_BLK_STREAM_VIDEO
                block and designates the last block of information for the
                given video stream.
****************************************************************************/
typedef struct {

    unsigned short  blkSize;            //Size of block
    unsigned short  blkType;            //Type of block (MP_BLK_TYPE_VIDEO_CODEC_H264)

    unsigned short  packetizationMode;  //Packetization mode for H.264 (uses one of MP_S_PACK_* definitions)

    unsigned char   profile;            //H264 profile
    unsigned char   profile_iop;        //H264 profile compatibility
    unsigned char   level;              //H264 level

    //DCI Information (3GPP Format)
    //This is the format that is needed for signaling/transcoding components, etc.
    unsigned short  dcSpecInfoSize;     //Size in bytes of the following Decoder Specific Information buffer
    unsigned char   pDcSpecInfo[MP_MAX_DCI_SIZE];
                                //Buffer containing Decoder Specific Information of size 'dcSpecInfoSize'

    //DCI Information (ISO Format)
    //This is the native format retrieved from the ISO file, but not useful.  Just informational
    unsigned short  dcSpecInfoSizeIsoFormat;   //Size in bytes of the following Decoder Specific Information buffer in ISO format
    unsigned char   pDcSpecInfoIsoFormat[MP_MAX_DCI_SIZE];
                                        //Buffer containing Decoder Specific Information in ISO format of size 'dcSpecInfoSizeIsoFormat'

} MP_BLK_CODEC_VIDEO_H264;	


/*!
****************************************************************************
\brief      Video VP8 Codec block

                The MP_BLK_CODEC_VIDEO_VP8 block contains VP8-specific
                codec information.  It is preceded by a MP_BLK_STREAM_VIDEO
                block and designates the last block of information for the
                given video stream.
****************************************************************************/
typedef struct {

    unsigned short  blkSize;            //Size of block
    unsigned short  blkType;            //Type of block (MP_BLK_TYPE_VIDEO_CODEC_VP8)

} MP_BLK_CODEC_VIDEO_VP8;	


/*!
****************************************************************************
\brief      Hint RTP Protocol block

                The MP_BLK_PROTOCOL_HINT_RTP block contains RTP -specific
                protocol information.  It is preceded by a MP_BLK_STREAM_HINT
                block and designates the last block of information for the
                given hint stream.
****************************************************************************/
typedef struct {

    unsigned short  blkSize;            //Size of block
    unsigned short  blkType;            //Type of block (MP_BLK_TYPE_HINT_PROTOCOL_RTP)

    unsigned short  hintTrackVersion;   //Current version of hint track
    unsigned short  hintVersionComp;    //Oldest version this hint track is compatible with
    unsigned long   maxPacketSize;      //Size of largest RTP packet generated by this track
    unsigned long   timeScale;          //RTP time base
    unsigned long   flags;              //Flags
#define	MP_HINTRTP_FLAG_TIMESTAMP  1       //If set (tsro box present), use timestampOffset field
#define	MP_HINTRTP_FLAG_SEQUENCE   2       //If set (snro box present), use sequenceOffset field

    unsigned long   timestampOffset;    //RTP timestamp offset (first RTP packet timestamp value)
    unsigned long   sequenceOffset;     //RTP sequence number offset (first RTP packet sequence number)

} MP_BLK_PROTOCOL_HINT_RTP;	






/*******************************************************************************
 * Media Parser Media Toolkit API Functions
 ******************************************************************************/


/**
 * @brief Open a multimedia file
 *
 * This API opens a multimedia file of a given type in read mode,
 * and when successful, returns a multimedia file access structure that is then
 * used to access the file.
 *
 * @param [in] pFileName    - pointer to a text string containing the name of file to open.
 * @param [in] fileType     - indicates the type of file (see MP_FILE_TYPE_XXX).
 *                            For ISO file format-based files, MP_FILE_TYPE_ISO can be used, or
 *                            the specific file type can be used if defined (MP_FILE_TYPE_3GP, 
 *                            MP_FILE_TYPE_MP4, MP_FILE_TYPE_F4V, or others defined).
 *                            For Dialogic Media Format (proprietary) files,
 *                            use MP_FILE_TYPE_DMF only.
 * @param[out] pMMFile      - pointer to the multimedia file access structure (MP_MMFILE).
 *
 * @return		SUCCESS (0) if successful, otherwise error codes:
 *              - MPERR_INVALID_FILE_TYPE       (file type not supported)
 *              - MPERR_INVALID_POINTER         (NULL pMMFile or pFileName)
 *              - MPERR_FILE_OPEN_FAILED        (file open failed, e.g. not found)
 *              - MPERR_FILE_ACCESS_DENIED      (access denied)
 *              - MPERR_FILE_FORMAT             (file format error)
 */
extern "C" MTK_API
MTK_RETURN MTK_CONV  mp_OpenFile (char *pFileName, unsigned fileType, MP_MMFILE *pMMFile);



/**
 * @brief Close an opened multimedia file
 *
 * This API closes the file and frees intermediate internal resources.
 *
 * @param [in] pMMFile  - pointer to the multimedia file access structure (MP_MMFILE).
 *
 * @return		SUCCESS (0) if successful, otherwise error codes:
 *              - MPERR_INVALID_FILE        (pMMFile invalid)
 *              - MPERR_INTERNAL_RESOURCE   (error on internal resource)
 */
extern "C" MTK_API
MTK_RETURN MTK_CONV  mp_CloseFile (MP_MMFILE *pMMFile);



/**
 * @brief Get minimum file info size
 *
 * This API obtains the minimum memory buffer size needed prior to calling mp_GetFileInfo()
 * in order to be able to retrieve all of the file information for all of the streams.
 * Calling this API is optional, but if not called, and the file info size used when calling
 * mp_GetFileInfo() is not large enough, not all of the file information will be returned.
 *
 * @param [in] pMMFile          - pointer to the multimedia file access structure (MP_MMFILE).
 * @param[out] pMinFileInfoSize - pointer to the minimum file info size to be returned.
 *
 * @return		SUCCESS (0) if successful, otherwise error codes:
 *              - MPERR_INVALID_FILE        (pMMFile invalid)
 *              - MMERR_INVALID_POINTER     (NULL pMinFileInfoSize)
 */
extern "C" MTK_API
MTK_RETURN MTK_CONV  mp_GetFileInfoSize (MP_MMFILE *pMMFile, unsigned long *pMinFileInfoSize);



/**
 * @brief Get detailed information from an opened multimedia file
 *
 * This API obtains detailed information from the given multimedia file
 *
 * @param [in] pMMFile          - pointer to the multimedia file access structure (MP_MMFILE).
 * @param[out] pFileInfoDesc    - pointer to the file information recipient (see MP_FILE_INFO_DESC descriptor).
 *
 * @return		SUCCESS (0) if successful, otherwise error codes:
 *              - MPERR_INVALID_FILE            (pMMFile invalid)
 *              - MMERR_INVALID_POINTER         (NULL pFileInfoDesc)
 *              - MPERR_INCOMP_ACCESS_MODE      (file not opened for read)
 *              - MPERR_INVALID_STRUCT_VERSION  (invalid version of pFileInfoDesc)
 */
extern "C" MTK_API
MTK_RETURN MTK_CONV  mp_GetFileInfo (MP_MMFILE *pMMFile, MP_FILE_INFO_DESC	*pFileInfoDesc);


                                     
#ifdef __cplusplus
}
#endif  /* __cplusplus */


#endif /* _LB_MTKLIB_H_ */

