/**
 * @file  mtklib.h
 * @brief Media Toolkit Library
 * @date  Apr. 20, 2006
 *
 * DIALOGIC CONFIDENTIAL      
 * Copyright (C) 2007,2008 Dialogic Corporation. All Rights Reserved.
 * 
 * The source code contained or described herein and all documents related to
 * the source code ("Material") are owned by Dialogic Corporation or its 
 * suppliers or licensors.  Title to the Material remains with Dialogic 
 * Corporation or its suppliers and licensors.  The Material contains trade
 * secrets and proprietary and confidential information of Dialogic or its 
 * suppliers and licensors.  The Material is protected by worldwide copyright
 * and trade secret laws and treaty provisions.  No part of the Material may be
 * used, copied, reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way without Dialogic's prior express written
 * permission.
 * 
 * No license under any patent, copyright, trade secret or other intellectual 
 * property right is granted to or conferred upon you by disclosure or delivery
 * of the Materials, either expressly, by implication, inducement, estoppel or
 * otherwise.  Any license under such intellectual property rights must be
 * express and approved by Dialogic in writing.
 */

#ifndef _MTKLIB_H_
#define _MTKLIB_H_

#include "srllib.h"
#include "hrt_target.h"
#include "mtkerrs.h"
#include <string.h>

#define MTK_CONV HRT_CDECL

typedef long MTK_HANDLE;
typedef MTK_HANDLE MTK_FRAME_HANDLE;
typedef int MTK_RETURN;

#if defined(LIBMTK_EXPORTS)
#define MTK_API HRT_EXPORT
#else
#define MTK_API HRT_IMPORT
#endif


/*******************************************************************************
 * Media Toolkit Data Defines
 ******************************************************************************/

#define MTK_SUCCESS  0              ///< Indicates success
#define MTK_ERROR    -1             ///< Indicates failure

#define MTK_INFINITE   0 
#define MTK_VER(v,s) (sizeof(s) << 16 | (0xFFFF & (v)))

#define MTK_MAXMSGSIZE 80           ///< Maximum size of info message string

/**!
 * list of defines for image or text justification.
 */
#define MTK_JUSTIFY_NONE   0x0  ///< No justification performed
#define MTK_JUSTIFY_LEFT   0x1  ///< Left justification
#define MTK_JUSTIFY_RIGHT  0x2  ///< Right justification
#define MTK_JUSTIFY_CENTER 0x3  ///< Center justified, either vertically or horizonatally
#define MTK_JUSTIFY_TOP    0x4  ///< Top justification
#define MTK_JUSTIFY_BOTTOM 0x5  ///< Bottom justification

/*******************************************************************************
 * Media Toolkit Data Types
 ******************************************************************************/
/**
 * @struct tagMTK_ERROR_INFO
 * @struct MTK_ERROR_INFO
 * Error information structure. Structure used to provide error information when
 * a media toolkit API call fails.
 */
typedef struct tagMTK_ERROR_INFO
{
   unsigned int unVersion;          ///< Structure version
   unsigned int unErrorCode;        ///< Error code
   const char * szErrorString;      ///< Error string
   const char * szAdditionalInfo;   ///< Additional error information string
} MTK_ERROR_INFO, *PMTK_ERROR_INFO;
typedef const MTK_ERROR_INFO * CPMTK_ERROR_INFO;

#define MTK_ERROR_INFO_VERSION_0 MTK_VER(0, MTK_ERROR_INFO)
#define MTK_ERROR_INFO_VERSION MTK_ERROR_INFO_VERSION_0

static __inline void INIT_MTK_ERROR_INFO(PMTK_ERROR_INFO pStruct) {
   pStruct->unVersion         = MTK_ERROR_INFO_VERSION;
   pStruct->unErrorCode       = EMTK_NOERROR;
   pStruct->szErrorString     = NULL;
   pStruct->szAdditionalInfo  = NULL;
}

typedef enum {
	eMTK_EVENT_CODE_SUCCESS = 0,       ///< Success
	eMTK_EVENT_CODE_INVALID_DEVICE,    ///< Invalid device
    eMTK_INVALID_HANDLE,               ///< Invalid device handle
    eMTK_INVALID_NAME,                 ///< Invalid device name
    eMTK_INVALID_PARM,                 ///< Invalid parameter in function call
    eMTK_INVALID_ATTR,            	   ///< Invalid attribute
    eMTK_INVALID_EVENT,           	   ///< Invalid event
    eMTK_INVALID_STATE,                ///< Invalid state for current operation
    eMTK_SUBSYSTEM_ERR,                ///< Internal subsystem error
    eMTK_FIRMWARE_ERR,                 ///< Firmware error
    eMTK_LIBRARY_ERR,                  ///< Library error
    eMTK_SYSTEMERR_ERR,                ///< System error
    eMTK_MEMORY_ALLOC_ERR,             ///< Memory allocation error
    eMTK_UNSUPPORTED_API,              ///< API not supported
    eMTK_UNSUPPORTED_TECH,             ///< Technology not supported
    eMTK_UNSUPPORTED_FUNC,             ///< Functionality not supported
	eMTK_EVENT_CODE_IPM_SUBSYSTEM_ERR, ///< ipm library error 
	eMTK_EVENT_CODE_CNF_SUBSYSTEM_ERR, ///< cnf library error
    eMTK_EVENT_CODE_M3G_SUBSYSTEM_ERR, ///< m3g library error
	eMTK_EVENT_CODE_MM_SUBSYSTEM_ERR,  ///< mm library error
    eMTK_EVENT_CODE_DTI_SUBSYSTEM_ERR, ///< dti library error
    eMTK_EVENT_CODE_T38_SUBSYSTEM_ERR, ///< T38 subsystem error
	eMTK_EVENT_CODE_INVALIDSTATE,      ///< Invalid state error
	eMTK_EVENT_CODE_NOTCONNECTED,      ///< Device not connected
    eMTK_MAX_OVERLAYS_EXCEEDED,        ///< sm library - Exceeded maximum number of overlays for a device 
	eMTK_EVENT_CODE_END_OF_LIST        ///< end of list
} eMTK_EVENT_CODE;

/**
 * @struct tagMTK_EVENT_INFO
 * @struct MTK_EVENT_INFO
 * Event information structure for use with mtk_GetResultInfo. This is the
 * data structure used to provide event information when a media toolkit
 * API event occurs. The unValue data member will be set to an event error code
 * for failure events.
 */
typedef struct tagMTK_EVENT_INFO
{
    unsigned int    unVersion;                          ///< structure version
    unsigned int    unValue;                            ///< mtk event value code
    unsigned int    unSubSystemValue;                   ///< subsystem event code
    char            szMsg[MTK_MAXMSGSIZE];              ///< mtk event message string
    char            szSubSystemMsg[MTK_MAXMSGSIZE];     ///< subsystem event message string
    char            szAdditionalInfo[MTK_MAXMSGSIZE];   ///< mtk event additional information
} MTK_EVENT_INFO, *PMTK_EVENT_INFO;
typedef const MTK_EVENT_INFO* CPMTK_EVENT_INFO;

#define MTK_EVENT_INFO_VER(v) (sizeof(MTK_EVENT_INFO) << 16 | (0xFFFF & (v)))
#define MTK_EVENT_INFO_VERSION_0 MTK_EVENT_INFO_VER(0)
#define MTK_EVENT_INFO_VERSION MTK_EVENT_INFO_VERSION_0

static __inline void INIT_MTK_EVENT_INFO(PMTK_EVENT_INFO pStruct){
    pStruct->unVersion        = MTK_EVENT_INFO_VERSION;
    pStruct->unValue          = eMTK_EVENT_CODE_SUCCESS;
    pStruct->unSubSystemValue = eMTK_EVENT_CODE_SUCCESS;
    memset((void*)(pStruct->szMsg), 0, MTK_MAXMSGSIZE);
    memset((void*)(pStruct->szSubSystemMsg), 0, MTK_MAXMSGSIZE);
    memset((void*)(pStruct->szAdditionalInfo), 0, MTK_MAXMSGSIZE);
}

/*!
 * @enum tagEMTK_POSITION_TYPE
 * @brief Enumerated list of position unit types
 */
typedef enum tagEMTK_POSITION_TYPE
{
    eMTK_POSITION_TYPE_MIN,
    eMTK_POSITION_TYPE_PIXEL = eMTK_POSITION_TYPE_MIN,  /*!< pixel based coordinates; (0,0) is upper, left-hand corner*/
    eMTK_POSITION_TYPE_PERCENT,                         /*!< percentage based coordinates; (0,0) is upper, left-hand corner */
    eMTK_POSITION_TYPE_JUSTIFICATION,                   /*!< justification based coordinates, eMTK_JUSTIFY values */
    eMTK_POSITION_TYPE_END_OF_LIST = eMTK_POSITION_TYPE_JUSTIFICATION
} eMTK_POSITION_TYPE;


/*!
 * @enum tagEMTK_SIZE_TYPE
 * @brief Enumerated list of size unit types
 */
typedef enum tagEMTK_SIZE_TYPE
{
    eMTK_SIZE_TYPE_MIN,
    eMTK_SIZE_TYPE_PIXEL = eMTK_SIZE_TYPE_MIN,  /*!< pixel based size */
    eMTK_SIZE_TYPE_PERCENT,                     /*!< percentage based size */
    eMTK_SIZE_TYPE_END_OF_LIST = eMTK_SIZE_TYPE_PERCENT
} eMTK_SIZE_TYPE;


/** 
 * @struct MTK_RECT
 * @struct tagMTK_RECT
 * Rectangle definition structure.
 */
typedef struct tagMTK_RECT
{
   unsigned int   unVersion;  ///< Structure version.
   float          x;          ///< Horizontal offset.
   float          y;          ///< Verticle Offset.
   float          width;      ///< Rectangle width.
   float          height;     ///< Rectangle height.
} MTK_RECT, *PMTK_RECT;
typedef const MTK_RECT * CPMTK_RECT;

#define MTK_RECT_VERSION_0 MTK_VER(0, MTK_RECT)
#define MTK_RECT_VERSION MTK_RECT_VERSION_0
static __inline void INIT_MTK_RECT(PMTK_RECT pStruct) {
   pStruct->unVersion         = MTK_RECT_VERSION;
   pStruct->x = 0.00; 
   pStruct->y = 0.00;
   pStruct->width = 0.00;
   pStruct->height = 0.00;
}


/** 
 * @struct MTK_BITMAP_DATA
 * @struct tagMTK_BITMAP_DATA
 * Bitmap Data definition structure.
 */
typedef struct tagMTK_BITMAP_DATA
{
   unsigned int     unVersion;  ///< Structure version.
   unsigned char *  pucData;    ///< bitmap data
   unsigned int     unLength;   ///< bitmap data length.
} MTK_BITMAP_DATA, *PMTK_BITMAP_DATA;
typedef const MTK_BITMAP_DATA * CPMTK_BITMAP_DATA;

#define MTK_BITMAP_DATA_VERSION_0 MTK_VER(0, MTK_BITMAP_DATA)
#define MTK_BITMAP_DATA_VERSION MTK_BITMAP_DATA_VERSION_0
static __inline void INIT_MTK_BITMAP_DATA(PMTK_BITMAP_DATA pStruct) {
   pStruct->unVersion   = MTK_BITMAP_DATA_VERSION;
   pStruct->pucData     = NULL; 
   pStruct->unLength    = 0;
}


/** 
 * @struct MTK_STRING
 * @struct tagMTK_STRING
 * MTK string definition structure.
 */
typedef struct tagMTK_STRING
{
   unsigned int unVersion;      ///< Structure version.
   char *       szString;       ///< string
   unsigned int unLength;       ///< string length.
} MTK_STRING, *PMTK_STRING;
typedef const MTK_STRING * CPMTK_STRING;

#define MTK_STRING_VERSION_0 MTK_VER(0, MTK_STRING)
#define MTK_STRING_VERSION MTK_STRING_VERSION_0
static __inline void INIT_MTK_STRING(PMTK_STRING pStruct) {
   pStruct->unVersion   = MTK_STRING_VERSION;
   pStruct->szString     = NULL; 
   pStruct->unLength    = 0;
}


/*!
 * @enum tagEMTK_IMAGE_FORMAT
 * @brief Enumerated list of image format.
 */
typedef enum tagEMTK_IMAGE_FORMAT
{
   eMTK_IMAGE_FORMAT_YUV = 0, /*!< Specifies a YUV Image. */
   eMTK_IMAGE_FORMAT_JPEG     /*!< Specifies a JPEG Image. */
} eMTK_IMAGE_FORMAT;


/*!
 * @enum tagEMTK_YUV_IMAGE_FORMAT
 * @brief Enumerated list of YUV image formats. Important Note: Only YUV 4:2:0 is supported.
 */
typedef enum tagEMTK_YUV_IMAGE_FORMAT
{
   eMTK_YUV_IMAGE_FORMAT_MIN = 0,
   eMTK_YUV_IMAGE_FORMAT_411 = eMTK_YUV_IMAGE_FORMAT_MIN, /*!< Specifies a YUV 4:1:1 format. */
   eMTK_YUV_IMAGE_FORMAT_420,     /*!< Specifies a YUV 4:2:0 format. */
   eMTK_YUV_IMAGE_FORMAT_422,     /*!< Specifies a YUV 4:2:2 format. */
   eMTK_YUV_IMAGE_FORMAT_444,     /*!< Specifies a YUV 4:4:4 format. */
   eMTK_YUV_IMAGE_FORMAT_END_OF_LIST
} eMTK_YUV_IMAGE_FORMAT;


/*!
 * @enum tagEMTK_FONT_TYPE
 * @brief Enumerated list of font types.
 */
typedef enum tagEMTK_FONT_TYPE
{
   eMTK_FONT_MIN,
   eMTK_FONT_COURIER_NEW = eMTK_FONT_MIN,  /*!< The specifier for courier new  */
   eMTK_FONT_ARIAL,                        /*!< The specifier for arial  */
   eMTK_FONT_TIMES_NEW_ROMAN,              /*!< The specifier for times new roman  */
   eMTK_FONT_TAHOMA,                       /*!< The specifier for tahoma */
   eMTK_FONT_END_OF_LIST = eMTK_FONT_TAHOMA
} eMTK_FONT_TYPE;


/*!
 * @enum tagEMTK_FONT_STYLE
 * @brief Enumerated list of font styles for bitmask.
 */
typedef enum tagEMTK_FONT_STYLE
{
    eMTK_TEXT_STYLE_NORMAL        = 0x00, ///< normal font style
    eMTK_TEXT_STYLE_ITALIC        = 0x01, ///< italic font style
    eMTK_TEXT_STYLE_BOLD          = 0x02, ///< bold font style
    eMTK_TEXT_STYLE_UNDERLINE     = 0x04, ///< underline font style
    eMTK_TEXT_STYLE_STRIKETHROUGH = 0x08 ///< strikethrough font style
} eMTK_FONT_STYLE;


/*******************************************************************************
 * Media Toolkit API functions
 ******************************************************************************/
#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

/**
 * @brief Get error information for the last error.
 * @param [out] a_pErrorInfo - Pointer to error information structure.
 * @return MTK_SUCCESS if successful, else MTK_ERROR.
 */
MTK_API
MTK_RETURN MTK_CONV mtk_GetErrorInfo(PMTK_ERROR_INFO a_pErrorInfo);


/**
 * @brief Get the event information associated with the given event data pointer.
 * 
 * This function returns the event information associated with the given event 
 * To get this information, the user must first retrieve the event data pointer
 * by making a call to a function suck as sr_getevtdatap() and then pass the
 * void * returned from that function as the to this function.
 * 
 * All MTK events contain a tail of event information in the event data.  As a 
 * result, the event length retrieved from sr_getevtlen is larger than the event
 * data associated with the event.
 *
 * @param [in]  a_hDev        - device handle associated with the event returned from sr_getevtdev
 * @param [in]  a_uneventType - event type of the event returned from sr_getevttype
 * @param [in]  a_peventData  - pointer returned from sr_getevtdatap
 * @param [out] a_peventInfo  - pointer to MTK_EVENT_INFO to fill based upon a_peventData
 *
 * @return MTK_SUCCESS if successful, MTK_ERROR otherwise. Call mtk_GetErrorInfo
 *  for error information upon failure.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  MTK_SUCCESS if successful, MTK_ERROR if invalid data pointer or error occurs.
 *  Call mtk_GetErrorInfo for error information upon failure of this function.
 * @par Asynchronous Mode
 *  Not applicable
 * @par Event Data
 *  Not applicable
 * @par Notes
 *  None
 * @par Cautions
 *  - All MTK events contain a tail of event information in the event data.  As a 
 * result, the event length retrieved from sr_getevtlen is larger than the event
 * data associated with the event.
 *  - ATDV_LASTERR and ATDV_ERRMSGP are not supported for mtk asynchronous event
 *  handling
 * @par Errors
 *  This function will fail if invalid data pointer is passed i.e. a pointer not
 *  associated with the mtk library. Call mtk_GetErrorInfo for error information
 *  upon failure.
 * @sa mtk_GetErrorInfo()
 */
MTK_API
MTK_RETURN MTK_CONV mtk_GetResultInfo(SRL_DEVICE_HANDLE a_hDev,
                                      unsigned long a_uneventType,
                                      void* a_peventData,
                                      PMTK_EVENT_INFO a_peventInfo);


/**
 * @brief Makes a font template
 *
 * This function is currently not supported.
 *
 * This function returns a handle to a font template for use with other APIs
 * when font related specifications are needed.
 *
 * Ensure to call mtk_DestroyMediaTemplate() when the application is done with the template.
 *
 * @param [in] a_eFontType - the type of font
 *
 * @return an MTK_FONT_HANDLE if successful, MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  Returns an MTK_FONT_HANDLE if successful, MTK_ERROR otherwise. 
 * @par Asynchronous Mode
 *  Not supported 
 * @par Event Data
 *  Not applicable
 * @par Notes
 *  None
 * @par Cautions
 * This function is currently not supported.
 *
 * Ensure to call mtk_DestroyMediaTemplate() when the application is done with the template.
 *
 * @par Errors
 *  This function will fail if incorrect parameters specified in the call. Call
 *  mtk_GetErrorInfo for error information upon failure.
 * @sa mtk_DestroyMediaTemplate()
 */
MTK_API
MTK_HANDLE MTK_CONV mtk_CreateFontTemplate(eMTK_FONT_TYPE a_eFontType);

/**
 * @brief Makes a media image template
 * 
 * This function returns a handle to a media image template for use with other APIs
 * when image related specifications are needed.
 *
 * Set and get all attributes of the specific image type via the various 
 * mtk_Set*Image*() and mtk_Get*Image*() functions.
 *
 * Ensure to call mtk_DestroyMediaTemplate() when the application is done with the image template.
 *
 * @param [in] a_eImageFormat - image format
 *
 * @return an MTK_HANDLE if successful, MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  Returns an MTK_HANDLE if successful, MTK_ERROR otherwise. 
 * @par Asynchronous Mode
 *  Not supported 
 * @par Event Data
 *  Not applicable
 * @par Notes
 *  None
 * @par Cautions
 *  Ensure to call mtk_DestroyMediaTemplate() when the application is done with the image template.
 * @par Errors
 *  This function will fail if incorrect parameters specified in the call. Call
 *  mtk_GetErrorInfo for error information upon failure.
 * @sa mtk_SetYUVImageFormat()
 * @sa mtk_GetYUVImageFormat()
 * @sa mtk_SetYUVImageSize()
 * @sa mtk_GetYUVImageSize()
 * @sa mtk_DestroyMediaTemplate()
 */
MTK_API
MTK_HANDLE MTK_CONV mtk_CreateImageTemplate(eMTK_IMAGE_FORMAT a_eImageFormat);


/**
 * @brief Makes a media file template containing the specified media
 * 
 * This function returns a handle to a media file template for use with other APIs
 * when file related specifications are needed.
 *
 * Re-set and get the media file name through the mtk_SetMediaFileName() and
 * mtk_GetMediaFileName() functions.
 *
 * Ensure to call mtk_DestroyMediaTemplate() when the application is done with the media file template.
 *
 * @param [in] a_hMedia - media handle returned by mtk_CreateImageTemplate()
 * @param [in] a_szFileName - file name 
 *
 * @return an MTK_HANDLE if successful, MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  Returns an MTK_HANDLE if successful, MTK_ERROR otherwise. 
 * @par Asynchronous Mode
 *  Not supported 
 * @par Event Data
 *  Not applicable
 * @par Notes
 *  None
 * @par Cautions
 *  Currently only Image handles returned from mtk_CreateImageTemplate() can be used with this function.
 *  Ensure to call mtk_DestroyMediaTemplate() when the application is done with the media file template.
 * @par Errors
 *  This function will fail if incorrect parameters specified in the call. Call
 *  mtk_GetErrorInfo for error information upon failure.
 * @sa mtk_SetMediaFileName()
 * @sa mtk_GetMediaFileName()
 * @sa mtk_DestroyMediaTemplate()
 */
MTK_API
MTK_HANDLE MTK_CONV mtk_CreateMediaFileTemplate(MTK_HANDLE a_hMedia, const char *a_szFileName);


/**
 * @brief Makes a bitmap template containing the specified media 
 * 
 * This function returns a handle to a bitmap template for use with other APIs
 * when bitmap related specifications are needed.
 *
 * The bitmap data and length of data are set and retrieved via the 
 * mtk_SetBitmapData() and mtk_GetBitmapData() functions.
 *
 * Ensure to call mtk_DestroyMediaTemplate() when the application is done with the bitmap
 * template.
 *
 * @param [in]  a_hMedia - media handle returned by mtk_CreateImageTemplate()
 *
 * @return an MTK_HANDLE if successful, MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  Returns an MTK_HANDLE if successful, MTK_ERROR otherwise. 
 * @par Asynchronous Mode
 *  Not supported 
 * @par Event Data
 *  Not applicable
 * @par Notes
 *  None
 * @par Cautions
 *  Ensure to call mtk_DestroyMediaTemplate() when the application is done with the image
 *  template.
 * @par Errors
 *  This function will fail if incorrect parameters specified in the call. Call
 *  mtk_GetErrorInfo for error information upon failure.
 * @sa mtk_SetBitmapData()
 * @sa mtk_GetBitmapData()
 * @sa mtk_DestroyMediaTemplate()
 */
MTK_API
MTK_HANDLE MTK_CONV mtk_CreateBitmapTemplate(MTK_HANDLE a_hMedia);


/**
 * @brief Destroys an mtk library media template
 * 
 * This function releases resources directly associated with the specified
 * a_MediaHandle that was created with any of the mtk_Create*() functions. After
 * this function is called, the associated media handle is no longer valid for
 * use with any other library functionality.
 *
 * @param [in] a_hMedia - handle to an mtk media template to destroy
 *
 * @return MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  returns MTK_SUCCESS if successful, MTK_ERROR otherwise. 
 * @par Asynchronous Mode
 *  Not supported 
 * @par Event Data
 *  Not Applicable
 * @par Notes
 *  None
 * @par Cautions
 *  After this function call, the associated template handle is no longer valid for
 *  use with any other library functionality.  
 * @par Errors
 *  This function will fail if an invalid template handle is specified in the call. Call
 *  mtk_GetErrorInfo for error information upon failure.
 * @sa mtk_CreateFontTemplate()
 * @sa mtk_CreateMediaFileTemplate()
 * @sa mtk_CreateBitampTemplate()
 * @sa mtk_CreateImageTemplate()
 * @sa mtk_DestroyMediaTemplate()
 */
MTK_API
MTK_RETURN MTK_CONV mtk_DestroyMediaTemplate(MTK_HANDLE a_hMedia);


/**
 * @brief Used to set the color format for a YUV image template.
 *
 * This API will set the color format for an MTK YUV image temlpate. The user may
 * specify any valid MTK_HANDLE handle to an MTK YUV Media Image returned from
 * mtk_CreateImageTemplate when calling this function.
 *
 * @param [in]  a_hYUVImage - mtk library handle of the YUV Image template for which
 *  to set the format.
 * @param [in]  a_eFormat - the color format to set for the image.
 * @return MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Asynchronous Mode
 *  Not supported 
 * @par Event Data
 *  Not Applicable
 * @par Notes
 *  none
 * @par Cautions
 *  none
 * @par Errors
 *  This function will fail if an invalid handle is specified in the call.
 *  Call mtk_GetErrorInfo for error information upon failure.
 * @sa mtk_GetYUVImageFormat
 * @sa mtk_CreateImageTemplate
 */
MTK_API
MTK_RETURN MTK_CONV mtk_SetYUVImageFormat(MTK_HANDLE a_hYUVImage, eMTK_YUV_IMAGE_FORMAT a_eFormat);


/**
 * @brief Used to get the color format for a YUV image template.
 *
 * This API will get the color format for an MTK YUV image template. The user may
 * specify any valid MTK_HANDLE handle to an MTK YUV Media Image template returned from
 * mtk_CreateImageTemplate when calling this function.
 *
 * @param [in]  a_hYUVImage - mtk library handle of the YUV Image temlpate for which
 *  to set the format.
 * @param [out]  a_peFormat - a pointer to a eMTK_YUV_IMAGE_FORMAT to set with the
 *  current color format of the image template.
 * @return MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Asynchronous Mode
 *  Not supported 
 * @par Event Data
 *  Not Applicable
 * @par Notes
 *  none
 * @par Cautions
 *  none
 * @par Errors
 *  This function will fail if an invalid handle or NULL pointer is specified
 *  in the call. Call mtk_GetErrorInfo for error information upon failure.
 * @sa mtk_SetYUVImageFormat
 * @sa mtk_CreateImageTemplate
 */
MTK_API
MTK_RETURN MTK_CONV mtk_GetYUVImageFormat(MTK_HANDLE a_hYUVImage, eMTK_YUV_IMAGE_FORMAT* a_peFormat);


/**
 * @brief Set the size for a YUV Image template.
 *
 * This API will set the size for a YUV Image template. The user may specify
 * any valid MTK_HANDLE handle to an MTK YUV Media Image template returned from
 * mtk_CreateImageTemplate when calling this function.
 *
 * @param [in]  a_hYUVImage - mtk library handle of the YUV Image template for which
 *  to set the format.
 * @param [in]  a_unWidth - image width
 * @param [in]  a_unHeight - image height
 * @return MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Asynchronous Mode
 *  Not supported 
 * @par Event Data
 *  Not Applicable
 * @par Notes
 *  none
 * @par Cautions
 *  none
 * @par Errors
 *  This function will fail if an invalid handle or <> is specified in
 *  the call. Call mtk_GetErrorInfo for error information upon failure.
 * @sa mtk_GetYUVImageSize
 * @sa mtk_CreateImageTemplate
 */
MTK_API
MTK_RETURN MTK_CONV mtk_SetYUVImageSize(MTK_HANDLE a_hYUVImage,
                                        unsigned int a_unWidth,
                                        unsigned int a_unHeight);


/**
 * @brief Get the size of a YUV Image template.
 *
 * This API will get the size of YUV Image template. The user may specify
 * any valid MTK_HANDLE handle to an MTK YUV Media Image template returned from
 * mtk_CreateImageTemplate when calling this function.
 *
 * @param [in]  a_hYUVImage - mtk library handle of the YUV Image template for which
 *  to retrieve the current image size.
 * @param [out]  a_punWidth - pointer to an unsigned int in which to set the current image width
 * @param [out]  a_punHeight - pointer to an unsigned int in which to set the current image height
 * @return MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Asynchronous Mode
 *  Not supported 
 * @par Event Data
 *  Not Applicable
 * @par Notes
 *  none
 * @par Cautions
 *  none
 * @par Errors
 *  This function will fail if an invalid handle or null pointer is specified in
 *  the call. Call mtk_GetErrorInfo for error information upon failure.
 * @sa mtk_SetYUVImageSize
 * @sa mtk_CreateImageTemplate
 */
MTK_API
MTK_RETURN MTK_CONV mtk_GetYUVImageSize(MTK_HANDLE a_hYUVImage,
                                        unsigned int* a_punWidth,
                                        unsigned int* a_punHeight);


/**
 * @brief Set the bitmap data for MTK Bitmap template.
 *
 * This API will set the Bitmap data. The user may specify any valid MTK_HANDLE
 * handle to an MTK Bitmap template returned from mtk_CreateBitmapTemplate when
 * calling this function. Since bitmaps can be very large, the data set by this
 * call must be persistent until the user destroys the bitmap and all other items
 * referncing the data through it such as a video overlay. The system will not
 * make a copy of the data but will simply store the value in the pointer.
 *
 * @param [in]  a_hBitmap - mtk library handle of the bitmap template for which
 *  to set the data.
 * @param [in]  a_pData - bitmap data structure
 * @return MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Asynchronous Mode
 *  Not supported 
 * @par Event Data
 *  Not Applicable
 * @par Notes
 *  none
 * @par Cautions
 *  The void pointer passed in to this function must reference persistent data
 *  as stated in the description.
 * @par Errors
 *  This function will fail if an invalid handle or null pointer is specified in
 *  the call. Call mtk_GetErrorInfo for error information upon failure.
 * @sa mtk_GetBitmapData
 * @sa mtk_CreateBitmapTemplate
 */
MTK_API
MTK_RETURN MTK_CONV mtk_SetBitmapData(MTK_HANDLE a_hBitmap,
                                      PMTK_BITMAP_DATA a_pData);


/**
 * @brief Get the bitmap data for a MTK Bitmap template.
 *
 * This API will get the Bitmap data and length of the Bitmap Template. The user
 * may specify any valid MTK_HANDLE handle to an MTK Bitmap Template returned from 
 * mtk_CreateBitmapTemplate when calling this function.
 *
 * @param [in]  a_hBitmap - mtk library handle of the bitmap template for which
 *  to set the data.
 * @param [out]  a_pData - bitmap data structure in which to store the current
 *  bitmap data settings
 * @return MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Asynchronous Mode
 *  Not supported 
 * @par Event Data
 *  Not Applicable
 * @par Notes
 *  none
 * @par Cautions
 *  none
 * @par Errors
 *  This function will fail if an invalid handle or null pointer is specified in
 *  the call. Call mtk_GetErrorInfo for error information upon failure.
 * @sa mtk_SetBitmapData
 * @sa mtk_CreateBitmapTemplate
 */
MTK_API
MTK_RETURN MTK_CONV mtk_GetBitmapData(MTK_HANDLE a_hBitmap,
                                      PMTK_BITMAP_DATA a_pData);


/**
 * @brief Set the MediaFile template filename.
 *
 * This API will set the MediaFile template file name. The user may specify any
 * valid MTK_HANDLE handle to an MTK MediaFile template returned from
 * mtk_CreateMediaFileTemplate when calling this function.
 *
 * @param [in]  a_hMediaFile - mtk library handle of the media file temlate for which
 *  to set the filename.
 * @param [in]  a_cpFileName - MTK String structure containing the file name
 * @return MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Asynchronous Mode
 *  Not supported 
 * @par Event Data
 *  Not Applicable
 * @par Notes
 *  none
 * @par Cautions
 *  none
 * @par Errors
 *  This function will fail if an invalid handle or null pointer is specified in
 *  the call. Call mtk_GetErrorInfo for error information upon failure.
 * @sa mtk_GetMediaFileName
 * @sa mtk_CreateBitmapTemplate
 */
MTK_API
MTK_RETURN MTK_CONV mtk_SetMediaFileName(MTK_HANDLE a_hMediaFile,
                                         CPMTK_STRING a_cpFileName);


/**
 * @brief Get the MediaFile template filename.
 *
 * This API will get the MediaFile template file name. The user may specify any valid
 * MTK_HANDLE handle to an MTK MediaFile template returned from 
 * mtk_CreateMediaFileTemplate when calling this function. The user must allocate
 * a buffer for the system to store the filename data and point the
 * a_pFileName->szString at it. Also the user must specify the length of that
 * buffer in the a_pFileName->unLength. Upon success the returned 
 * a_pFileName->unLength will be set to the length of the current filename.
 *
 * @param [in]  a_hMediaFile - mtk library handle of the media file template for which
 *  to set the filename.
 * @param [out]  a_pFileName - MTK String structure in which to store the
 *  file name data
 * @return MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Asynchronous Mode
 *  Not supported 
 * @par Event Data
 *  Not Applicable
 * @par Notes
 *  none
 * @par Cautions
 *  The function will fail if the length of the allocated buffer is not adequate
 *  to store the current file name plus the null terminator. The user can check
 *  the a_pText->unLength to determine the length of the current text string to
 *  allocate a larger buffer. Note that the returned length does not include
 *  the null terminator.
 * @par Errors
 *  This function will fail if an invalid handle or null pointer is specified in
 *  the call. Call mtk_GetErrorInfo for error information upon failure.
 * @sa mtk_SetMediaFileName
 * @sa mtk_CreateBitmapTemplate
 */
MTK_API
MTK_RETURN MTK_CONV mtk_GetMediaFileName(MTK_HANDLE a_hMediaFile,
                                         PMTK_STRING a_pFileName);


/**
 * @brief Used to set the font type for an MTK Font template.
 *
 * This function is currently not supported.
 *
 * This API will set the font type for an MTK Font template. The user may
 * specify any valid MTK_HANDLE handle to an MTK Font template returned from
 * mtk_CreateFontTemplate when calling this function.
 *
 * @param [in]  a_hFont - mtk library handle of the Font template for which
 *  to set the type.
 * @param [in]  a_eType - the type to set for the Font.
 * @return MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Asynchronous Mode
 *  Not supported 
 * @par Event Data
 *  Not Applicable
 * @par Notes
 *  none
 * @par Cautions
 *  This function is currently not supported.
 * @par Errors
 *  This function will fail if an invalid handle is specified in the call.
 *  Call mtk_GetErrorInfo for error information upon failure.
 * @sa mtk_GetFontType
 * @sa mtk_CreateFontTemplate
 */
MTK_API
MTK_RETURN MTK_CONV mtk_SetFontType(MTK_HANDLE a_hFont, eMTK_FONT_TYPE a_eType);


/**
 * @brief Used to get the font type for an MTK Font template.
 *
 * This function is currently not supported.
 *
 * This API will get the font type for an MTK Font template.
 *
 * @param [in]  a_hFont - mtk library handle of the Font template for which
 *  to get the type.
 * @param [out]  a_peType - pointer to an eMTK_FONT_TYPE in which to set the
 *  current font type of the Font.
 * @return MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Asynchronous Mode
 *  Not supported 
 * @par Event Data
 *  Not Applicable
 * @par Notes
 *  none
 * @par Cautions
 *  This function is currently not supported.
 * @par Errors
 *  This function will fail if an invalid handle or NULL pointer is specified in
 *  the call. Call mtk_GetErrorInfo for error information upon failure.
 * @sa mtk_SetFontType
 * @sa mtk_CreateFontTemplate
 */
MTK_API
MTK_RETURN MTK_CONV mtk_GetFontType(MTK_HANDLE a_hFont, eMTK_FONT_TYPE* a_peType);


/**
 * @brief Used to set the font size for an MTK Font template.
 *
 * This function is currently not supported.
 *
 * This API will set the font size for an MTK Font template. The user may
 * specify any valid MTK_HANDLE handle to an MTK Font template returned from
 * mtk_CreateFontTemplate when calling this function.
 *
 * @param [in]  a_hFont - mtk library handle of the Font template for which
 *  to set the size.
 * @param [in]  a_unSize - the height in pixels of the Font.
 * @return MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Asynchronous Mode
 *  Not supported 
 * @par Event Data
 *  Not Applicable
 * @par Notes
 *  none
 * @par Cautions
 *  This function is currently not supported.
 * @par Errors
 *  This function will fail if an invalid handle is specified in the call.
 *  Call mtk_GetErrorInfo for error information upon failure.
 * @sa mtk_GetFontType
 * @sa mtk_CreateFontTemplate
 */
MTK_API
MTK_RETURN MTK_CONV mtk_SetFontSize(MTK_HANDLE a_hFont, unsigned int a_unSize);


/**
 * @brief Used to get the font size for an MTK Font template.
 *
 * This function is currently not supported.
 *
 * This API will get the font size for an MTK Font template.
 *
 * @param [in]  a_hFont - mtk library handle of the Font template for which
 *  to get the size.
 * @param [out]  a_punSize - pointer to an unsigned int in which to store the
 *  current font size of the Font.
 * @return MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Asynchronous Mode
 *  Not supported 
 * @par Event Data
 *  Not Applicable
 * @par Notes
 *  none
 * @par Cautions
 *  This function is currently not supported.
 * @par Errors
 *  This function will fail if an invalid handle or NULL pointer is specified in
 *  the call. Call mtk_GetErrorInfo for error information upon failure.
 * @sa mtk_SetFontSize
 * @sa mtk_CreateFontTemplate
 */
MTK_API
MTK_RETURN MTK_CONV mtk_GetFontSize(MTK_HANDLE a_hFont, unsigned int * a_punSize);


/**
 * @brief Used to set the font style for an MTK Font template.
 *
 * This function is currently not supported.
 *
 * This API will set the font style for an MTK Font template. The user may
 * specify any valid MTK_HANDLE handle to an MTK Font template returned from
 * mtk_CreateFontTemplate when calling this function.
 *
 * @param [in]  a_hFont - mtk library handle of the Font template for which
 *  to set the font style.
 * @param [in]  a_unStyle - a bitmask of eMTK_FONT_STYLE values.
 * @return MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Asynchronous Mode
 *  Not supported 
 * @par Event Data
 *  Not Applicable
 * @par Notes
 *  none
 * @par Cautions
 *  This function is currently not supported.
 * @par Errors
 *  This function will fail if an invalid handle is specified in the call.
 *  Call mtk_GetErrorInfo for error information upon failure.
 * @sa mtk_GetFontType
 * @sa mtk_CreateFontTemplate
 */
MTK_API
MTK_RETURN MTK_CONV mtk_SetFontStyle(MTK_HANDLE a_hFont, unsigned int a_unStyle);


/**
 * @brief Used to get the font style for an MTK Font template.
 *
 * This function is currently not supported.
 *
 * This API will get the font style for an MTK Font template.
 *
 * @param [in]  a_hFont - mtk library handle of the Font template for which
 *  to get the font style.
 * @param [out]  a_punStyle - pointer to an unsigned int in which to store the
 *  current font style bitmask of the Font.
 * @return MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Asynchronous Mode
 *  Not supported 
 * @par Event Data
 *  Not Applicable
 * @par Notes
 *  none
 * @par Cautions
 *  This function is currently not supported.
 * @par Errors
 *  This function will fail if an invalid handle or NULL pointer is specified in
 *  the call. Call mtk_GetErrorInfo for error information upon failure.
 * @sa mtk_SetFontSize
 * @sa mtk_CreateFontTemplate
 */
MTK_API
MTK_RETURN MTK_CONV mtk_GetFontStyle(MTK_HANDLE a_hFont, unsigned int * a_punStyle);


/**
 * @brief Used to create an MTK frame template.
 *
 * This API creates a frame template to be used to describe a rectangular 
 * region whose size and position are based upon a base frame. There are
 * different ways to designate frame properties relative to a base frame. This
 * is specified when calling the various MTK size and position functions.
 * @return MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  a valid MTK_FRAME_HANDLE if successful, MTK_ERROR otherwise.
 * @par Asynchronous Mode
 *  Not supported
 * @par Event Data
 *  Not Applicable
 * @par Notes
 *  None
 * @par Cautions
 *  None
 * @par Errors
 *  This function will fail if there is an internal library failure. Call 
 *  mtk_GetErrorInfo for error information upon failure.
 * @sa mtk_DestroyFrameTemplate
 * @sa mtk_SetFramePosition
 * @sa mtk_GetFramePosition
 * @sa mtk_SetFrameSize
 * @sa mtk_GetFrameSize
 */
MTK_API
MTK_FRAME_HANDLE MTK_CONV mtk_CreateFrameTemplate();


/**
 * @brief Destroys an MTK Frame template
 * 
 * This function releases all resources associated with the specified a_hFrame handle
 * that was returned from mtk_CreateFrameTemplate(). After this function call,
 * the associated handle is no longer valid for use with any other library
 * functionality.
 *
 * This function is not to be used on handles returned from various 
 * xx_SetFrame() like functions.
 *
 * @param [in] a_hFrame - handle to an MTK Frame template to destroy
 * 
 * @return MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  returns MTK_SUCCESS if successful, MTK_ERROR otherwise. 
 * @par Asynchronous Mode
 *  Not supported 
 * @par Event Data
 *  Not Applicable
 * @par Notes
 *  None
 * @par Cautions
 *  After this function call, the associated handle is no longer valid for
 *  use with any other library functionality. This function is not to be used
 *  on handles returned from ob_SetOverlayBoundingFrame() or other such 
 *  snapshot generating frame functions.
 * @par Errors
 *  This function will fail if an invalid template handle is specified in the 
 *  call. Call mtk_GetErrorInfo for error information upon failure.
 * @sa mtk_CreateFrameTemplate()
 */
MTK_API
MTK_RETURN MTK_CONV mtk_DestroyFrameTemplate(MTK_FRAME_HANDLE a_hFrame);


/**
 * @brief Used to set an MTK frame template position.
 *
 * This API set the position of a sub-frame using a containing frame, such as a
 * video frame, as a coordinate reference. The user specifies the types of 
 * position values in the x and y coordinates by setting the position type. All
 * coordinates use the upper, left-hand corner of the base frame as the origin.
 * @param [in] a_hFrame - handle to an MTK Frame template for which to 
 *                        set the position
 * @param [in] a_x - the horizontal position relative to the base frame.
 * @param [in] a_y - the vertical position relative to the base frame.
 * @param [in] a_ePositionType - the type of value being specified in a_x and a_y
 * @return MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Asynchronous Mode
 *  Not supported
 * @par Event Data
 *  Not Applicable
 * @par Notes
 *  None
 * @par Cautions
 *  None
 * @par Errors
 *  This function will fail if an invalid handle is specified in the call. Call
 *  mtk_GetErrorInfo for error information upon failure.
 * @sa mtk_CreateFrameTemplate
 * @sa mtk_GetFramePosition
 */
MTK_API
MTK_RETURN MTK_CONV mtk_SetFramePosition(MTK_FRAME_HANDLE a_hFrame,
                                         float a_x,
                                         float a_y,
                                         eMTK_POSITION_TYPE a_ePositionType);


/**
 * @brief Used to get an MTK frame template position.
 *
 * This API gets the position of frame using a containing frame, such as a
 * video frame, as a coordinate reference.
 * @param [in] a_hFrame - handle to an MTK Frame template for which to 
 *                        set the position
 * @param [out] a_px - pointer to a float to be filled in with the current 
 *                     horizontal position relative to the base frame.
 * @param [out] a_py - pointer to a float to be filled in with the current 
 *                     vertical position relative to the base frame.
 * @param [out] a_pePositionType - pointer to an eMTK_POSITION_TYPE to be filled
 *                     with the current type of value stored in a_px and a_py
 * @return MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Asynchronous Mode
 *  Not supported
 * @par Event Data
 *  Not Applicable
 * @par Notes
 *  None
 * @par Cautions
 *  None
 * @par Errors
 *  This function will fail if an invalid handle or a NULL pointer is specified
 *  in the call. Call mtk_GetErrorInfo for error information upon failure.
 * @sa mtk_CreateFrameTemplate
 * @sa mtk_GetFramePosition
 */
MTK_API
MTK_RETURN MTK_CONV mtk_GetFramePosition(MTK_FRAME_HANDLE a_hFrame,
                                         float* a_px,
                                         float* a_py,
                                         eMTK_POSITION_TYPE* a_pePositionType);


/**
 * @brief Used to set an MTK frame template size.
 *
 * This API set the size of a sub-frame using a containing frame, such as a
 * video frame, as a coordinate reference. The user specifies the types of 
 * size values in the w and h values by setting a_eSizeType.
 * @param [in] a_hFrame - handle to an MTK Frame template for which to 
 *                        set the position
 * @param [in] a_w - the horizontal size
 * @param [in] a_h - the vertical size
 * @param [in] a_eSizeType - the type of value being specified in a_w and a_h
 * @return MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Asynchronous Mode
 *  Not supported
 * @par Event Data
 *  Not Applicable
 * @par Notes
 *  None
 * @par Cautions
 *  None
 * @par Errors
 *  This function will fail if an invalid handle is specified in the call. Call
 *  mtk_GetErrorInfo for error information upon failure.
 * @sa mtk_CreateFrameTemplate
 * @sa mtk_GetFramePosition
 */
MTK_API
MTK_RETURN MTK_CONV mtk_SetFrameSize(MTK_FRAME_HANDLE a_hFrame,
                                     float a_w,
                                     float a_h,
                                     eMTK_SIZE_TYPE a_eSizeType);


/**
 * @brief Used to get an MTK frame template size.
 *
 * This API gets the size of a frame using a containing frame, such as a
 * video frame, as a coordinate reference.
 * @param [in] a_hFrame - handle to an MTK Frame template for which to 
 *                        get the size
 * @param [out] a_pw - pointer to a float to be filled in with the current 
 *                     horizontal size
 * @param [out] a_ph - pointer to a float to be filled in with the current 
 *                     vertical size
 * @param [out] a_peSizeType - pointer to an eMTK_SIZE_TYPE to be filled with
 *                     the current type of value stored in a_pw and a_ph
 * @return MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Asynchronous Mode
 *  Not supported
 * @par Event Data
 *  Not Applicable
 * @par Notes
 *  None
 * @par Cautions
 *  None
 * @par Errors
 *  This function will fail if an invalid handle or a NULL pointer is specified
 *  in the call. Call mtk_GetErrorInfo for error information upon failure.
 * @sa mtk_CreateFrameTemplate
 * @sa mtk_GetFramePosition
 */
MTK_API
MTK_RETURN MTK_CONV mtk_GetFrameSize(MTK_FRAME_HANDLE a_hFrame,
                                     float* a_pw,
                                     float* a_ph,
                                     eMTK_SIZE_TYPE* a_peSizeType);


#ifdef __cplusplus
}
#endif  /* __cplusplus */

#endif /* _MTKLIB_H_ */

