/**
 * @file  ob_mtklib.h
 * @date  May 23, 2007
 * @brief Overlay Builder Media Toolkit Library
 *
 * DIALOGIC CONFIDENTIAL      
 * Copyright (C) 2007 Dialogic Corporation. All Rights Reserved.
 * 
 * The source code contained or described herein and all documents related to
 * the source code ("Material") are owned by Dialogic Corporation or its 
 * suppliers or licensors.  Title to the Material remains with Dialogic 
 * Corporation or its suppliers and licensors.  The Material contains trade
 * secrets and proprietary and confidential information of Dialogic or its 
 * suppliers and licensors.  The Material is protected by worldwide copyright
 * and trade secret laws and treaty provisions.  No part of the Material may be
 * used, copied, reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way without Dialogic's prior express written
 * permission.
 * 
 * No license under any patent, copyright, trade secret or other intellectual 
 * property right is granted to or conferred upon you by disclosure or delivery
 * of the Materials, either expressly, by implication, inducement, estoppel or
 * otherwise.  Any license under such intellectual property rights must be
 * express and approved by Dialogic in writing.
 */


#ifndef _OBMTKLIB_H_
#define _OBMTKLIB_H_

#include "mtklib.h"
#include "mtkerrs.h"

#define OB_VER(v,s) (sizeof(s) << 16 | (0xFFFF & (v)))
typedef MTK_HANDLE OB_OVERLAY_HANDLE;

/*******************************************************************************
 * Media Toolkit Overlay Builder API functions
 ******************************************************************************/
#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */


/*!
 * @enum tagEOB_FILL_STYLE
 * @brief Enumerated list of fill types for an overlay that express how a frame
 * is filled with an overlay.
 */
typedef enum tagEOB_FILL_STYLE
{
    eOB_FILL_STYLE_MIN,
    eOB_FILL_STYLE_RESIZE_TO_FIT = eOB_FILL_STYLE_MIN,  /*!< overlay will be re-sized to fit the frame */
    eOB_FILL_STYLE_NO_CHANGE_IN_SIZE,                   /*!< overlay will be as presented by user with no re-sizing */
    eOB_FILL_STYLE_MAINTAIN_ASPECT_RATIO,               /*!< overlay will be stretched to maintain aspect ratio of 
                                                        /*   original; resizing will occur until one side of the overlay
                                                        /*   hits either sides of the frame */
    eOB_FILL_STYLE_END_OF_LIST = eOB_FILL_STYLE_MAINTAIN_ASPECT_RATIO
} eOB_FILL_STYLE;


/**
 * @brief Makes a text overlay template
 * 
 * This function currently is not supported.
 * 
 * This function returns a handle to a text overlay template that can then be added to a
 * streaming device.
 *
 * <b>Important: Text overlays currently are not supported.</b>
 * 
 * Ensure to call ob_DestroyOverlayTemplate() when the application is done with the overlay template.
 *
 * @param [in] a_cpTextString - text to display in overlay
  *
 * @return an OB_OVERLAY_HANDLE if successful, MTK_ERROR otherwise.
 * @par Include
 *  ob_mtklib.h
 * @par Synchronous Mode
 *  Returns an OB_OVERLAY_HANDLE if successful, MTK_ERROR otherwise. 
 * @par Asynchronous Mode
 *  Not supported 
 * @par Event Data
 *  Not applicable
 * @par Notes
 *  Changes to a given overlay template through various ob_Set*() functions do
 *  not affect those added, or re-added, to a device. In other words the device
 *  keeps a snapshot of the overlay from the time it was added.
 * @par Cautions
 *  This function currently is not supported.
 *  Ensure to call ob_DestroyOverlayTemplate() when the application is done with
 *  the overlay template.
 * @par Errors
 *  This function will fail if an incorrect parameter or NULL pointer is 
 *  specified in the call. Call mtk_GetErrorInfo for error information upon
 *  failure.
 * @sa ob_DestroyOverlayTemplate()
 */
MTK_API
OB_OVERLAY_HANDLE MTK_CONV ob_CreateTextOverlayTemplate(CPMTK_STRING a_cpTextString);


/**
 * @brief Makes an Image Overlay Template
 * 
 * This function returns a handle to an Image Overlay Template that can then be added to a
 * streaming device.
 *
 * Set and get the overlay template attributes via the various ob_SetOverlay*() and 
 * ob_GetOverlay*() functions.
 *
 * Ensure to call ob_DestroyOverlayTemplate() when the application is done with
 * the overlay template.
 *
 * @param [in] a_hImage - handle returned from mtk_CreateMediaFileTemplate() or
 * mtk_CreateBitmapTemplate()
 *   
 * @return an OB_OVERLAY_HANDLE if successful, MTK_ERROR otherwise.
 * @par Include
 *  ob_mtklib.h
 * @par Synchronous Mode
 *  Returns an OB_OVERLAY_HANDLE if successful, MTK_ERROR otherwise. 
 * @par Asynchronous Mode
 *  Not supported 
 * @par Event Data
 *  Not applicable
 * @par Notes
 *  Changes to a given overlay template through various ob_Set*() functions do
 *  not affect those added, or re-added, to a device. In other words the device
 *  keeps a snapshot of the overlay from the time it was added.
 * @par Cautions
 *  Ensure to call ob_DestroyOverlayTemplate() when the application is done with the overlay.
 * @par Errors
 *  This function will fail if incorrect parameters specified in the call. Call
 *  mtk_GetErrorInfo for error information upon failure.
 * @sa ob_SetOverlayBoundingFrame() 
 * @sa ob_GetOverlayBoundingFrame()
 * @sa ob_SetOverlayDuration() 
 * @sa ob_GetOverlayDuration()
 * @sa ob_SetOverlayTransparency() 
 * @sa ob_GetOverlayTransparency()
 * @sa ob_SetOverlayJustification()
 * @sa ob_GetOverlayJustification()
 * @sa ob_DestroyOverlayTemplate()
 * @sa mtk_CreateMediaFileTemplate()
 * @sa mtk_CreateBitmapTemplate()
 */
MTK_API
OB_OVERLAY_HANDLE MTK_CONV ob_CreateImageOverlayTemplate(MTK_HANDLE a_hImage);


/**
 * @brief Destroys an overlay template
 * 
 * This function releases all resources associated with the specified a_OverlayHandle
 * that was created with ob_CreateImageOverlayTemplate() or ob_CreateTextOverlayTemplate(). After
 * this function call, the associated handle is no longer valid for use with any
 * other library functionality.
 *
 * This function is not to be used on handles returned in the SMEV_ADD_OVERLAY
 * completion event data.
 *
 * @param [in] a_hOverlay - handle to an ob overlay template to destroy
 * 
 * @return MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  returns MTK_SUCCESS if successful, MTK_ERROR otherwise. 
 * @par Asynchronous Mode
 *  Not supported 
 * @par Event Data
 *  Not Applicable
 * @par Notes
 *  None
 * @par Cautions
 *  After this function call, the associated template handle is no longer valid for
 *  use with any other library functionality.
 *  This function should not be called on overlay template handles returned in the event data
 *  from a call to sm_AddOverlay().  
 * @par Errors
 *  This function will fail if an invalid template handle is specified in the call. Call
 *  mtk_GetErrorInfo for error information upon failure.
 * @sa ob_CreateImageOverlayTemplate()
 * @sa ob_CreateTextOverlayTemplate()
 */
MTK_API
MTK_RETURN MTK_CONV ob_DestroyOverlayTemplate(OB_OVERLAY_HANDLE a_hOverlay);


/**
 * @brief Used to set the bounding frame for an OB Overlay Template.
 *
 * This API will set the bounding frame for an OB Overlay Template. This is the 
 * space and location in which the overlay will be sized to fit on the screen and how
 * it will relate to that area of the screen. A snapshot of the frame will be taken and 
 * stored to be referenced by the handle returned by this function. This handle to the 
 * shapshot can be used to change the values of the frame associated with the overlay
 * without changing the original template.
 *
 * The user may specify any valid OB_OVERLAY_HANDLE handle to an overlay template returned
 * from ob_CreateImageOverlayTemplate, ob_CreateTextOverlayTemplate or one returned from 
 * sm_AddOverlay when calling this function.
 *
 * @param [in]  a_hOverlay - ob library handle of the overlay template for which to set 
 *  the bounding frame.
 * @param [in]  a_hFrame - a handle to a frame template created by a call to
 *  mtk_CreateFrameTemplate()
 * @return Upon success, a handle to a snapshot frame containing the parameters
 *  of the frame template; MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Asynchronous Mode
 *  Not supported
 * @par Event Data
 *  Not Applicable
 * @par Notes
 *  none
 * @par Cautions
 *  Do not call mtk_DestroyFrame() on the handle returned by this function
 * @par Errors
 *  This function will fail if an invalid handle is specified in the call. Call
 *  mtk_GetErrorInfo for error information upon failure.
 * @sa ob_GetOverlayBoundingFrame
 * @sa mtk_CreateFrameTemplate
 */
MTK_API
MTK_FRAME_HANDLE MTK_CONV ob_SetOverlayBoundingFrame(
    OB_OVERLAY_HANDLE a_hOverlay,
    MTK_FRAME_HANDLE a_hFrame);


/**
 * @brief Used to get the bounding frame set on an OB Overlay Template.
 *
 * This API will get a handle to the bounding frame snapshot for an OB Overlay 
 * Template. The user may specify any valid OB_OVERLAY_HANDLE handle to an 
 * overlay template returned from ob_CreateImageOverlayTemplate,
 * ob_CreateTextOverlayTemplate or sm_AddOverlay in a_hOverlay when calling
 * this function.
 * 
 * a_phFrame will be set to (-1) if the bounding frame has not been set for the
 * given overlay.
 *
 * @param [in]  a_hOverlay - ob library handle of the overlay template for which to get 
 *  the bounding frame.
 * @param [out] a_phFrame - pointer to an MTK_FRAME_HANDLE to be filled with the
 *                          handle of the current bounding frame for the 
 *                          specified overlay
 * @return MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Asynchronous Mode
 *  Not supported
 * @par Event Data
 *  Not Applicable
 * @par Notes
 *  none
 * @par Cautions
 *  none
 * @par Errors
 *  This function will fail if an invalid handle or a NULL pointer is specified
 *  in the call. It will Call mtk_GetErrorInfo for error information upon failure.
 * @sa ob_SetOverlayBoundingFrame
 */
MTK_API
MTK_RETURN MTK_CONV ob_GetOverlayBoundingFrame(OB_OVERLAY_HANDLE a_hOverlay,
                                               MTK_FRAME_HANDLE* a_phFrame);


/**
 * @brief Used to set the overlay fill style for an OB Overlay Template.
 *
 * This API will set the overlay fill style for an OB Overlay Template. This is
 * the manner in which the bounding frame will be filled by the media set in the 
 * overlay.
 *
 * Currently only eOB_FILL_STYLE_RESIZE_TO_FIT is supported. This means that the
 * image or text will be resized in both x and y directions to fill the entire 
 * bounding frame potentially causing distortion in the original image.
 *
 * The user may specify any valid OB_OVERLAY_HANDLE handle to an overlay template returned
 * from ob_CreateImageOverlayTemplate, ob_CreateTextOverlayTemplate or one returned from 
 * sm_AddOverlay when calling this function.
 *
 * @param [in]  a_hOverlay - ob library handle of the overlay template for which to set 
 *  the fill style.
 * @param [in]  a_eFillStyle - the fill style to be used for the overlay
 * @return
 *  MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  Returns MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Asynchronous Mode
 *  Not supported
 * @par Event Data
 *  Not Applicable
 * @par Notes
 *  none
 * @par Cautions
 *  Currently only eOB_FILL_STYLE_RESIZE_TO_FIT is supported.
 * @par Errors
 *  This function will fail if an invalid handle is specified in the call. Call
 *  mtk_GetErrorInfo for error information upon failure.
 * @sa ob_GetOverlayBoundingFrame
 * @sa mtk_CreateFrameTemplate
 */
MTK_API
MTK_RETURN MTK_CONV ob_SetOverlayFillStyle(OB_OVERLAY_HANDLE a_hOverlay,
                                           eOB_FILL_STYLE a_eFillStyle);


/**
 * @brief Used to get the current overlay fill style for an OB Overlay Template.
 *
 * This API will get the overlay fill style for an OB Overlay Template. This is
 * the manner in which the bounding frame will be filled by the media set in the 
 * overlay.
 *
 * The user may specify any valid OB_OVERLAY_HANDLE handle to an overlay 
 * template returned from ob_CreateImageOverlayTemplate, 
 * ob_CreateTextOverlayTemplate or one returned from sm_AddOverlay when calling
 * this function.
 *
 * @param [in]  a_hOverlay - ob library handle of the overlay template for which to set 
 *                           the fill style.
 * @param [in]  a_peFillStyle - pointer to be filled with the fill style set on
 *                              the overlay
 * @return
 *  MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  Returns MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Asynchronous Mode
 *  Not supported
 * @par Event Data
 *  Not Applicable
 * @par Notes
 *  none
 * @par Cautions
 *  Currently only eOB_FILL_STYLE_RESIZE_TO_FIT is supported.
 * @par Errors
 *  This function will fail if an invalid handle is specified in the call. Call
 *  mtk_GetErrorInfo for error information upon failure.
 * @sa ob_GetOverlayBoundingFrame
 * @sa mtk_CreateFrameTemplate
 */
MTK_API
MTK_RETURN MTK_CONV ob_GetOverlayFillStyle(OB_OVERLAY_HANDLE a_hOverlay,
                                           eOB_FILL_STYLE* a_peFillStyle);


/**
 * @brief Used to set the duration for an OB Overlay Template.
 *
 * This API will set the duration, in milliseconds, for an OB Overlay Template. This is
 * the time for which the overlay will play on top of the starting of a stream
 * from the device with which it is associated. The value may be MTK_INFINITE
 * for the overlay to last the lifetime of the stream. The user may specify any
 * valid OB_OVERLAY_HANDLE handle to an overlay template returned from
 * ob_CreateImageOverlayTemplate, ob_CreateTextOverlayTemplate or one returned
 * from sm_AddOverlay when calling this function.
 *
 * @param [in]  a_hOverlay - ob library template handle of the overlay for which to set
 *  the duration.
 * @param [in]  a_unDuration - length time in milliseconds that overlay will be
 *  seen for each streaming occurance or MTK_INFINITE.
 * @return MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Asynchronous Mode
 *  Not supported
 * @par Event Data
 *  Not Applicable
 * @par Notes
 *  none
 * @par Cautions
 *  none
 * @par Errors
 *  This function will fail if an invalid handle is specified in the call.
 *  Call mtk_GetErrorInfo for error information upon failure.
 * @sa ob_GetOverlayDuration
 * @sa ob_CreateTextOverlayTemplate
 * @sa ob_CreateImageOverlayTemplate
 */
MTK_API
MTK_RETURN MTK_CONV ob_SetOverlayDuration(OB_OVERLAY_HANDLE a_hOverlay,
                                          unsigned int a_unDuration);


/**
 * @brief Used to get the duration for an OB Overlay Template.
 *
 * This API will get the duration, in milliseconds, for an OB Overlay Template. This is
 * the time for which the overlay will play on top of the beginning of a stream
 * from the device with which it is associated. The user may specify any
 * valid OB_OVERLAY_HANDLE handle to an overlay returned from
 * ob_CreateImageOverlayTemplate, ob_CreateTextOverlayTemplate or one returned from sm_AddOverlay
 * when calling this function.
 *
 * @param [in]  a_hOverlay - ob library handle of the overlay template for which
 *  to get the duration.
 * @param [out]  a_punDuration - pointer to unsigned int in which to store the 
 *  current length of time the overlay is to be overlayed on the associated
 *  stream
 * @return MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Asynchronous Mode
 *  Not supported
 * @par Event Data
 *  Not Applicable
 * @par Notes
 *  none
 * @par Cautions
 *  none
 * @par Errors
 *  This function will fail if an invalid handle or NULL pointer is specified
 *  in the call. Call mtk_GetErrorInfo for error information upon failure.
 * @sa ob_GetOverlayDuration
 * @sa ob_CreateTextOverlayTemplate
 * @sa ob_CreateImageOverlayTemplate
 */
MTK_API
MTK_RETURN MTK_CONV ob_GetOverlayDuration(OB_OVERLAY_HANDLE a_hOverlay,
                                          unsigned int* a_punDuration);


/**
 * @brief Used to set the transparency for an OB Overlay Template.
 *
 * This function is currently not supported.
 *
 * This API will set the transparency, as a percentage between 0 and 100, for
 * an OB Overlay Template. The user may specify any valid OB_OVERLAY_HANDLE handle to
 * an overlay template returned from ob_CreateImageOverlayTemplate,
 * ob_CreateTextOverlayTemplate or one returned from sm_AddOverlay when calling
 * this function.
 *
 * @param [in]  a_hOverlay - ob library handle of the overlay template for which to set 
 *  the transparency.
 * @param [in]  a_unTransparency - percentage transparency the overlay will be
 *  on top of the associated stream.
 * @return MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Asynchronous Mode
 *  Not supported
 * @par Event Data
 *  Not Applicable
 * @par Notes
 *  none
 * @par Cautions
 *  none
 * @par Errors
 *  This function will fail if an invalid handle is specified in the call.
 *  Call mtk_GetErrorInfo for error information upon failure.
 * @sa ob_GetOverlayTransparency
 * @sa ob_CreateTextOverlayTemplate
 * @sa ob_CreateImageOverlayTemplate
 */
MTK_API
MTK_RETURN MTK_CONV ob_SetOverlayTransparency(OB_OVERLAY_HANDLE a_hOverlay,
                                              unsigned int a_unTransparency);

/**
 * @brief Used to get the transparency for an OB Overlay Template.
 *
 * This function is currently not supported.
 *
 * This API will get the transparency, as a percentage between 0 and 100, for
 * an OB Overlay Template. The user may specify any valid OB_OVERLAY_HANDLE handle to
 * an overlay returned from ob_CreateImageOverlayTemplate, ob_CreateTextOverlayTemplate or one
 * returned from sm_AddOverlay when calling this function.
 *
 * @param [in]  a_hOverlay - ob library handle of the overlay template for which to get
 *  the transparency.
 * @param [out]  a_punTransparency - a pointer to an unsigned int in which to
 *  store the current percentage transparency for the overlay.
 * @return MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Asynchronous Mode
 *  Not supported
 * @par Event Data
 *  Not Applicable
 * @par Notes
 *  none
 * @par Cautions
 *  none
 * @par Errors
 *  This function will fail if an invalid handle or NULL pointer is specified
 *  in the call. Call mtk_GetErrorInfo for error information upon failure.
 * @sa ob_SetOverlayTransparency
 * @sa ob_CreateTextOverlayTemplate
 * @sa ob_CreateImageOverlayTemplate
 */
MTK_API
MTK_RETURN MTK_CONV ob_GetOverlayTransparency(OB_OVERLAY_HANDLE a_hOverlay,
                                              unsigned int* a_punTransparency);


/**
 * @brief Used to set the text string for an OB Text Overlay Template.
 *
 * This function is currently not supported.
 *
 * This API will set the string for an OB Text Overlay Template. The user may specify
 * any valid OB_OVERLAY_HANDLE handle to an overlay returned from
 * ob_CreateTextOverlayTemplate or one returned from sm_AddOverlay when called with a 
 * handle to a Text Overlay Template as the template.
 *
 * @param [in] a_hTextOverlay - ob library handle of the Text Overlay Template for which
 *  to set the text background transparency.
 * @param [in] a_cpText - an MTK_STRING structure containing the string to be
 *  used for the Text Overlay Template
 * @return MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Asynchronous Mode
 *  Not supported
 * @par Event Data
 *  Not Applicable
 * @par Notes
 *  none
 * @par Cautions
 *  none
 * @par Errors
 *  This function will fail if an invalid handle or NULL pointer is specified
 *  in the call. Call mtk_GetErrorInfo for error information upon failure.
 * @sa ob_GetTextOverlayString
 * @sa ob_CreateTextOverlayTemplate
 * @sa ob_CreateImageOverlayTemplate
 */
MTK_API
MTK_RETURN MTK_CONV ob_SetTextOverlayString(OB_OVERLAY_HANDLE a_hTextOverlay,
                                            CPMTK_STRING a_cpText);


/**
 * @brief Used to get the text string for an OB Text Overlay Template.
 *
 * This function is currently not supported.
 *
 * This API will get the string for an OB Text Overlay Template. The user may specify
 * any valid OB_OVERLAY_HANDLE handle to an overlay returned from
 * ob_CreateTextOverlayTemplate or one returned from sm_AddOverlay when called with a 
 * handle to a Text Overlay Template as the template. Upon return, the 
 * a_pFileName->unLength will be set to the length of the current filename, not
 * including the null terminator.
 *
 * @param [in] a_hTextOverlay - ob library handle of the Text Overlay Template for which
 *  to set the text background transparency.
 * @param [in] a_pText - an MTK_STRING structure in which the current text string
 *  will be stored
 * @return MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Asynchronous Mode
 *  Not supported
 * @par Event Data
 *  Not Applicable
 * @par Notes
 *  none
 * @par Cautions
 *  The function will fail if the length of the allocated buffer is not adequate
 *  to store the current text string. The user can check the a_pText->unLength
 *  to determine the length of the current text string to allocate a larger
 *  buffer. Note that this length does not include the null terminator.
 * @par Errors
 *  This function will fail if an invalid handle or NULL pointer is specified
 *  in the call. Call mtk_GetErrorInfo for error information upon failure.
 * @sa ob_SetTextOverlayString
 * @sa ob_CreateTextOverlayTemplate
 * @sa ob_CreateImageOverlayTemplate
 */
MTK_API
MTK_RETURN MTK_CONV ob_GetTextOverlayString(OB_OVERLAY_HANDLE a_hTextOverlay,
                                            PMTK_STRING a_pText);


/**
 * @brief Used to set the text color for an OB Text Overlay Template.
 *
 * This function is currently not supported.
 *
 * This API will set the text color for an OB Text Overlay Template. The user may specify
 * any valid OB_OVERLAY_HANDLE handle to an overlay returned from
 * ob_CreateTextOverlayTemplate or one returned from sm_AddOverlay when called with a 
 * handle to a Text Overlay Template as the template.
 *
 * @param [in] a_hTextOverlay - ob library handle of the Text Overlay Template for which
 *  to set the text background transparency.
 * @param [in] a_unTextColor - the text color as an RGB bitmask to be used for
 *  the overlay
 * @return MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Asynchronous Mode
 *  Not supported
 * @par Event Data
 *  Not Applicable
 * @par Notes
 *  none
 * @par Cautions
 *  none
 * @par Errors
 *  This function will fail if an invalid handle is specified in the call.
 *  Call mtk_GetErrorInfo for error information upon failure.
 * @sa ob_GetTextOverlayTextColor
 * @sa ob_CreateTextOverlayTemplate
 * @sa ob_CreateImageOverlayTemplate
 */
MTK_API
MTK_RETURN MTK_CONV ob_SetTextOverlayTextColor(OB_OVERLAY_HANDLE a_hTextOverlay,
                                               unsigned int a_unTextColor);


/**
 * @brief Used to set the text color for an OB Text Overlay Template.
 *
 * This function is currently not supported.
 *
 * This API will set the text color for an OB Text Overlay Template. The user may specify
 * any valid OB_OVERLAY_HANDLE handle to an overlay returned from
 * ob_CreateTextOverlayTemplate or one returned from sm_AddOverlay when called with a 
 * handle to a Text Overlay Template as the template.
 *
 * @param [in] a_hTextOverlay - ob library handle of the Text Overlay Template for which
 *  to set the text background transparency.
 * @param [in] a_punTextColor - a pointer to a unsigned int in which the current
 *  text color for the overlay will be stored as an RGB bitmask
 * @return MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Asynchronous Mode
 *  Not supported
 * @par Event Data
 *  Not Applicable
 * @par Notes
 *  none
 * @par Cautions
 *  none
 * @par Errors
 *  This function will fail if an invalid handle is specified in the call.
 *  Call mtk_GetErrorInfo for error information upon failure.
 * @sa ob_SetTextOverlayTextColor
 * @sa ob_CreateTextOverlayTemplate
 * @sa ob_CreateImageOverlayTemplate
 */
MTK_API
MTK_RETURN MTK_CONV ob_GetTextOverlayTextColor(OB_OVERLAY_HANDLE a_hTextOverlay,
                                               unsigned int* a_punTextColor);


/**
 * @brief Used to set the text back ground color for an OB Text Overlay Template.
 *
 * This function is currently not supported.
 *
 * This API will set text back ground color for an OB Text Overlay Template as an RGB
 * bitmask. The user may specify any valid OB_OVERLAY_HANDLE
 * handle to an overlay returned from ob_CreateTextOverlayTemplate or one
 * returned from sm_AddOverlay when called with a handle to a Text Overlay Template as the 
 * template.
 *
 * @param [in] a_hTextOverlay - ob library handle of the Text Overlay Template for which
 *  to set the text background color.
 * @param [in] a_unColor - RGB bitmask for the background of the Text Overlay Template.
 * @return MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Asynchronous Mode
 *  Not supported
 * @par Event Data
 *  Not Applicable
 * @par Notes
 *  none
 * @par Cautions
 *  none
 * @par Errors
 *  This function will fail if an invalid handle is specified in the call.
 *  Call mtk_GetErrorInfo for error information upon failure.
 * @sa ob_GetTextOverlayBackgroundColor
 * @sa ob_CreateTextOverlayTemplate
 * @sa ob_CreateImageOverlayTemplate
 */
MTK_API
MTK_RETURN MTK_CONV ob_SetTextOverlayBackgroundColor(OB_OVERLAY_HANDLE a_hTextOverlay,
                                                     unsigned int a_unColor);

                                                        
/**
 * @brief Used to get the text back ground color for an OB Text Overlay Template.
 *
 * This function is currently not supported.
 *
 * This API will get the text back ground color for an OB Text Overlay Template as an RGB
 * bitmask. The user may specify any valid OB_OVERLAY_HANDLE handle to an overlay
 * returned from ob_CreateTextOverlayTemplate or one returned from sm_AddOverlay when 
 * called with a handle to a Text Overlay Template as the template.
 *
 * @param [in] a_hTextOverlay - ob library handle of the Text Overlay Template for which
 *  to get the text background transparency.
 * @param [out] a_punColor - pointer to an unsigned int in which to store the 
 *  current text background color for the overlay.
 * @return MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Include
 *  mtklib.h
 * @par Synchronous Mode
 *  MTK_SUCCESS if successful, MTK_ERROR otherwise.
 * @par Asynchronous Mode
 *  Not supported
 * @par Event Data
 *  Not Applicable
 * @par Notes
 *  none
 * @par Cautions
 *  none
 * @par Errors
 *  This function will fail if an invalid handle or NULL pointer is specified 
 *  in the call. Call mtk_GetErrorInfo for error information upon failure.
 * @sa ob_SetTextOverlayBackgroundColor
 * @sa ob_CreateTextOverlayTemplate
 * @sa ob_CreateImageOverlayTemplate
 */
MTK_API
MTK_RETURN MTK_CONV ob_GetTextOverlayBackgroundColor(OB_OVERLAY_HANDLE a_hTextOverlay,
                                                     unsigned int *a_punColor);


#ifdef __cplusplus
}
#endif  /* __cplusplus */

#endif /* _OBMTKLIB_H_ */
