/****************************************************************************
 *		Copyright (c) 2000 Dialogic Corp.
 *		All Rights Reserved.
 *
 *		THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF Dialogic Corp.
 *		The copyright notice above does not evidence any actual or
 *		intended publication of such source code.
 ***************************************************************************/

/****************************************************************************
 * FILE:			tsptrace.h
 * AUTHOR:		DM3 TSP Team
 * ORG:			Dialogic Corporation
 * ORIG-DATE:	01-Aug-00
 * LAST-MOD:	01-Aug-00 at 17:50:46 by Darran Hunt
 * PURPOSE:		This module contains debug level definitions and macros
 *					for always-on tracing.
 *
 * HISTORY:
 * Date			Who	Description
 * 01-Aug-00	DNH	Initial Creation.
 ***************************************************************************/

#ifndef __TSPTRACE_H__
#define __TSPTRACE_H__

/*
 * Kernel Include Files
 */
#include "qtypes.h"

#ifdef __cplusplus
extern "C" {
#endif

/* Circular Trace buffer record definitions */

/* CAS Signal trace record */
typedef struct {
	Uint8 type;			/* Trace type					*/
	Uint8 direction;	/* Signal received or sent	*/
	Uint32 timestamp;	/* in milliseconds			*/
	Uint32 signalId;	/* CAS Signal Id				*/
} t_tspCasSignalTrace;

/* CAS ABCD bit transition trace record */
typedef struct {
	Uint8 type;			/* Trace type						*/
	Uint8 direction;	/* Signal received or sent		*/
	Uint32 timestamp;	/* in milliseconds				*/
	Uint8 preCode;		/* Bit state prior to change	*/
	Uint8 postCode;	/* Bit state after change		*/
} t_tspCasBitTrace ;

/* Types of trace records  */
enum {
	tspCasBitTrace = 0x01,
	tspCasSignalTrace = 0x02
} e_tspTraceType;

/* Note: type field is overlayed in all trace records
*  in order to reduce overall record size
*/
typedef union {
	Uint8 type;	
	t_tspCasBitTrace casBit;
	t_tspCasSignalTrace casSignal;
	Uint8 data[16];		/* Pad */
} t_tspTraceRecord;

/*
 * Memory use is 16*TSP_CBufCount*TSP_TraceCBufSize
 * E.g. 16*120*12 = 23040 bytes.
 */

/* Number of records in a circular trace buffer */
#define TSP_TraceCBufSize	12

/* Number of trace buffers (should match number of instances) */
#define TSP_CBufCount		120

/*
* The trace buffers are written so that the oldest record
* is overwritten by the new record.  Use the timestamps
* to determine order.
* A type of 0 indicates an unused entry in the buffer 
*/
/* Head index for each trace buffer */
extern Uint8 tspTraceCBufHead[TSP_CBufCount+1];

/* One trace buffer per instance */
extern t_tspTraceRecord tspTraceCBuf[TSP_CBufCount+1][TSP_TraceCBufSize];

extern Uint32 tspDebugChannel;

/*
 * Exported functions
 */


/* Initialize CAS circular trace buffers */
void traceCasInit();

/* External trace function hook */

/* This function pointer is used to enable CAS transition
*  (bit change) tracing.  When set to point to a function
*  of the appropriate type the function will be called 
*  whenever a change in the CAS signaling bits occurs
*  (i.e. when bits are transmitted or received).
*/
void (*tspCASTransTraceHook)(
	Uint8 instance,
	Uint32 lineId,
	Uint32 chanId,
	Uint32 timestamp,
	Uint32 direction,
	Uint8 preCode,
	Uint8 postCode);

/* This function pointer is used to enable CAS signal
*  tracing.  When set to point to a function of the 
*  appropriate type the function will be called 
*  whenever a CAS signal is transmitted are received.
*  It will be called immediately prior to a CAS signal
*  being transmitted, and immediately after a CAS signal
*  has been completely detected.
*/
void (*tspCASSignalTraceHook)(
	Uint8 instance,
	Uint32 lineId,
	Uint32 chanId,
	Uint32 timestamp,
	Uint32 direction,
	Uint32 signalId);


/**********************************************************************
* NAME:			casBitTrace(instance, lineId, chanId, timestamp,
*										direction, preCode, postCode)
* DESCRIPTION:	This function implements a circular trace buffer for CAS
*					signaling bit changes.  One buffer is maintained for 
*					each instance.
*					The trace is enabled by setting the tspCASTransTraceHook function
*					pointer to the casBitTrace function.  The CAS component will
*					call the trace function whenever ABCD signaling bits are
*					sent or received.
*					This trace function when used in conjuction with the
*					tspCASTransTraceHook function pointer will record all
*					CAS bit changes in a ciruclar trace buffer.  Each TSC
*					instance will have its own trace buffer.
*					(Note the same buffer is used for both signal and bit traces
*					so the trace records will be mixed)
* INPUTS:		instance		- CAS component instance (note this typically
*									  matches the TSC component instance number)
*					lineId		- line that the instance is running on
*					chanId		- channel (timeslot) that the instance is running on
*					timestamp	- time of bit transition (in msecs)
*					direction	- bits were received or transmitted
*					preCode		- bit states before change
*					postCode		- bit states after change
**********************************************************************/
void traceCasBits(
	Uint8 instance,
	Uint32 lineId,
	Uint32 chanId,
	Uint32 timestamp,
	Uint32 direction,
	Uint8 preCode,
	Uint8 postCode);


/**********************************************************************
* NAME:			casSignalTrace(instance, lineId, chanId, timestamp,
*										direction, signalId)
* DESCRIPTION:	This function implements a circular trace buffer for CAS
*					signals.  One buffer is maintained for each instance.
*					The trace is enabled by set the tspCASSignalTraceHook function
*					pointer to the casSignalTrace function.  The CAS component will
*					call the trace function whenever a CAS signal is sent or
*					received.
*					This trace function when used in conjuction with the
*					tspCASSignalTraceHook function pointer will record all
*					CAS signals sent and received in a ciruclar trace buffer.
*					Each TSC instance will have its own trace buffer.
*					(Note the same buffer is used for both signal and bit traces
*					so the trace records will be mixed)
* INPUTS:		instance		- CAS component instance (note this typically
*									  matches the TSC component instance number)
*					lineId		- line that the instance is running on
*					chanId		- channel (timeslot) that the instance is running on
*					timestamp	- time of bit transition (in msecs)
*					direction	- Signal was sent or received.
*					signalId		- CAS signal that was sent or received.
**********************************************************************/
void traceCasSignals(
	Uint8 instance,
	Uint32 lineId,
	Uint32 chanId,
	Uint32 timestamp,
	Uint32 direction,
	Uint32 signalId);


/**********************************************************************
* NAME:			traceClientEnable(clientDesc, transaction, label)
* DESCRIPTION: Enables trace event reporting for specified client.
*					Every time the board level trace buffer fills up it
*					will be sent to the client as a TSC_EvtTraceBoard_Type_CAS
*					event.
* INPUTS:		clientDesc	- send the event to this client
*					transaction	- transaction id to report in the event
*					label			- RTC label to report in the event
* RETURNS:		TSC_ErrNone			- successfully enable event for client
*					TSC_ErrExhausted	- event already enable for another client
*											  (supports only one client).
**********************************************************************/
int traceClientEnable(
	QCompDesc clientDesc,
	Uint32 transaction,
	Uint32 label);


/**********************************************************************
* NAME:			traceClientDisable(clientDesc)
* DESCRIPTION: Disables trace event reporting for specified client.
* INPUTS:		clientDesc	- cancel event reporting for this client.
* RETURNS:		TSC_ErrNone			- successfully disabled event for client
*					Std_ErrNotEnabled	- event not enabled, or not enabled for
*											  the client.
**********************************************************************/
int traceClientDisable(QCompDesc clientDesc);


/**********************************************************************
* NAME:			traceReportCas(instance, lineId, chanId) 
* DESCRIPTION: Add the instance's circular trace buffer to the board
*					event buffer.  The board event buffer can hold a number of
*					instance buffers, and once filled up it is sent to the
*					trace client (if one is registered).
* INPUTS:		instance	- capture this instance's trace buffer
*					lineId	- line that the instance is tracing
*					chanId	- channel that the instance is tracing
**********************************************************************/
void traceReportCas(
	Uint8 instance,
	Uint32 lineId,
	Uint32 chanId);

/**********************************************************************
* NAME:			traceGetCmplt(msgp)
* DESCRIPTION: Builds a TSC_MsgTraceGetCmplt message with the contents
*					of the CAS circular trace buffer for the instance that the
*					request was sent to.
* INPUTS:		msgp	- the TSC_MsgTraceGet request message.
* RETURNS:		the TSC_MsgTraceGetCmplt message ready to send.
**********************************************************************/
QMsgRef traceGetCmplt(QMsgRef msgp);

#ifdef __cplusplus
}
#endif

#endif /* __TSPTRACE_H__ */
